/**************************************************************************
 * 
 * Copyright 2007 VMware, Inc.
 * Copyright (c) 2008-2010 VMware, Inc.
 * All Rights Reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL VMWARE AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 **************************************************************************/


/**
 * Mesa / Gallium format conversion and format selection code.
 * \author Brian Paul
 */

#include "main/imports.h"
#include "main/context.h"
#include "main/enums.h"
#include "main/formats.h"
#include "main/glformats.h"
#include "main/texcompress.h"
#include "main/texgetimage.h"
#include "main/teximage.h"
#include "main/texstore.h"
#include "main/image.h"
#include "main/macros.h"
#include "main/formatquery.h"

#include "pipe/p_context.h"
#include "pipe/p_defines.h"
#include "pipe/p_screen.h"
#include "util/u_format.h"
#include "st_cb_texture.h"
#include "st_context.h"
#include "st_format.h"


/**
 * Translate Mesa format to Gallium format.
 */
enum pipe_format
st_mesa_format_to_pipe_format(struct st_context *st, mesa_format mesaFormat)
{
   switch (mesaFormat) {
   case MESA_FORMAT_A8B8G8R8_UNORM:
      return PIPE_FORMAT_ABGR8888_UNORM;
   case MESA_FORMAT_R8G8B8A8_UNORM:
      return PIPE_FORMAT_RGBA8888_UNORM;
   case MESA_FORMAT_B8G8R8A8_UNORM:
      return PIPE_FORMAT_BGRA8888_UNORM;
   case MESA_FORMAT_A8R8G8B8_UNORM:
      return PIPE_FORMAT_ARGB8888_UNORM;
   case MESA_FORMAT_X8B8G8R8_UNORM:
      return PIPE_FORMAT_XBGR8888_UNORM;
   case MESA_FORMAT_R8G8B8X8_UNORM:
      return PIPE_FORMAT_RGBX8888_UNORM;
   case MESA_FORMAT_B8G8R8X8_UNORM:
      return PIPE_FORMAT_BGRX8888_UNORM;
   case MESA_FORMAT_X8R8G8B8_UNORM:
      return PIPE_FORMAT_XRGB8888_UNORM;
   case MESA_FORMAT_B5G5R5A1_UNORM:
      return PIPE_FORMAT_B5G5R5A1_UNORM;
   case MESA_FORMAT_B4G4R4A4_UNORM:
      return PIPE_FORMAT_B4G4R4A4_UNORM;
   case MESA_FORMAT_B5G6R5_UNORM:
      return PIPE_FORMAT_B5G6R5_UNORM;
   case MESA_FORMAT_B2G3R3_UNORM:
      return PIPE_FORMAT_B2G3R3_UNORM;
   case MESA_FORMAT_B10G10R10A2_UNORM:
      return PIPE_FORMAT_B10G10R10A2_UNORM;
   case MESA_FORMAT_R10G10B10A2_UNORM:
      return PIPE_FORMAT_R10G10B10A2_UNORM;
   case MESA_FORMAT_L4A4_UNORM:
      return PIPE_FORMAT_L4A4_UNORM;
   case MESA_FORMAT_L8A8_UNORM:
      return PIPE_FORMAT_LA88_UNORM;
   case MESA_FORMAT_A8L8_UNORM:
      return PIPE_FORMAT_AL88_UNORM;
   case MESA_FORMAT_L16A16_UNORM:
      return PIPE_FORMAT_LA1616_UNORM;
   case MESA_FORMAT_A16L16_UNORM:
      return PIPE_FORMAT_AL1616_UNORM;
   case MESA_FORMAT_A_UNORM8:
      return PIPE_FORMAT_A8_UNORM;
   case MESA_FORMAT_A_UNORM16:
      return PIPE_FORMAT_A16_UNORM;
   case MESA_FORMAT_L_UNORM8:
      return PIPE_FORMAT_L8_UNORM;
   case MESA_FORMAT_L_UNORM16:
      return PIPE_FORMAT_L16_UNORM;
   case MESA_FORMAT_I_UNORM8:
      return PIPE_FORMAT_I8_UNORM;
   case MESA_FORMAT_I_UNORM16:
      return PIPE_FORMAT_I16_UNORM;
   case MESA_FORMAT_Z_UNORM16:
      return PIPE_FORMAT_Z16_UNORM;
   case MESA_FORMAT_Z_UNORM32:
      return PIPE_FORMAT_Z32_UNORM;
   case MESA_FORMAT_S8_UINT_Z24_UNORM:
      return PIPE_FORMAT_S8_UINT_Z24_UNORM;
   case MESA_FORMAT_Z24_UNORM_S8_UINT:
      return PIPE_FORMAT_Z24_UNORM_S8_UINT;
   case MESA_FORMAT_X8_UINT_Z24_UNORM:
      return PIPE_FORMAT_X8Z24_UNORM;
   case MESA_FORMAT_Z24_UNORM_X8_UINT:
      return PIPE_FORMAT_Z24X8_UNORM;
   case MESA_FORMAT_S_UINT8:
      return PIPE_FORMAT_S8_UINT;
   case MESA_FORMAT_Z_FLOAT32:
      return PIPE_FORMAT_Z32_FLOAT;
   case MESA_FORMAT_Z32_FLOAT_S8X24_UINT:
      return PIPE_FORMAT_Z32_FLOAT_S8X24_UINT;
   case MESA_FORMAT_YCBCR:
      return PIPE_FORMAT_UYVY;
   case MESA_FORMAT_YCBCR_REV:
      return PIPE_FORMAT_YUYV;
   case MESA_FORMAT_RGB_DXT1:
      return PIPE_FORMAT_DXT1_RGB;
   case MESA_FORMAT_RGBA_DXT1:
      return PIPE_FORMAT_DXT1_RGBA;
   case MESA_FORMAT_RGBA_DXT3:
      return PIPE_FORMAT_DXT3_RGBA;
   case MESA_FORMAT_RGBA_DXT5:
      return PIPE_FORMAT_DXT5_RGBA;
   case MESA_FORMAT_SRGB_DXT1:
      return PIPE_FORMAT_DXT1_SRGB;
   case MESA_FORMAT_SRGBA_DXT1:
      return PIPE_FORMAT_DXT1_SRGBA;
   case MESA_FORMAT_SRGBA_DXT3:
      return PIPE_FORMAT_DXT3_SRGBA;
   case MESA_FORMAT_SRGBA_DXT5:
      return PIPE_FORMAT_DXT5_SRGBA;
   case MESA_FORMAT_L8A8_SRGB:
      return PIPE_FORMAT_LA88_SRGB;
   case MESA_FORMAT_A8L8_SRGB:
      return PIPE_FORMAT_AL88_SRGB;
   case MESA_FORMAT_L_SRGB8:
      return PIPE_FORMAT_L8_SRGB;
   case MESA_FORMAT_BGR_SRGB8:
      return PIPE_FORMAT_R8G8B8_SRGB;
   case MESA_FORMAT_A8B8G8R8_SRGB:
      return PIPE_FORMAT_ABGR8888_SRGB;
   case MESA_FORMAT_R8G8B8A8_SRGB:
      return PIPE_FORMAT_RGBA8888_SRGB;
   case MESA_FORMAT_B8G8R8A8_SRGB:
      return PIPE_FORMAT_BGRA8888_SRGB;
   case MESA_FORMAT_A8R8G8B8_SRGB:
      return PIPE_FORMAT_ARGB8888_SRGB;
   case MESA_FORMAT_RGBA_FLOAT32:
      return PIPE_FORMAT_R32G32B32A32_FLOAT;
   case MESA_FORMAT_RGBA_FLOAT16:
      return PIPE_FORMAT_R16G16B16A16_FLOAT;
   case MESA_FORMAT_RGB_FLOAT32:
      return PIPE_FORMAT_R32G32B32_FLOAT;
   case MESA_FORMAT_RGB_FLOAT16:
      return PIPE_FORMAT_R16G16B16_FLOAT;
   case MESA_FORMAT_LA_FLOAT32:
      return PIPE_FORMAT_L32A32_FLOAT;
   case MESA_FORMAT_LA_FLOAT16:
      return PIPE_FORMAT_L16A16_FLOAT;
   case MESA_FORMAT_L_FLOAT32:
      return PIPE_FORMAT_L32_FLOAT;
   case MESA_FORMAT_L_FLOAT16:
      return PIPE_FORMAT_L16_FLOAT;
   case MESA_FORMAT_A_FLOAT32:
      return PIPE_FORMAT_A32_FLOAT;
   case MESA_FORMAT_A_FLOAT16:
      return PIPE_FORMAT_A16_FLOAT;
   case MESA_FORMAT_I_FLOAT32:
      return PIPE_FORMAT_I32_FLOAT;
   case MESA_FORMAT_I_FLOAT16:
      return PIPE_FORMAT_I16_FLOAT;
   case MESA_FORMAT_R_FLOAT32:
      return PIPE_FORMAT_R32_FLOAT;
   case MESA_FORMAT_R_FLOAT16:
      return PIPE_FORMAT_R16_FLOAT;
   case MESA_FORMAT_RG_FLOAT32:
      return PIPE_FORMAT_R32G32_FLOAT;
   case MESA_FORMAT_RG_FLOAT16:
      return PIPE_FORMAT_R16G16_FLOAT;

   case MESA_FORMAT_R_UNORM8:
      return PIPE_FORMAT_R8_UNORM;
   case MESA_FORMAT_R_UNORM16:
      return PIPE_FORMAT_R16_UNORM;
   case MESA_FORMAT_R8G8_UNORM:
      return PIPE_FORMAT_RG88_UNORM;
   case MESA_FORMAT_G8R8_UNORM:
      return PIPE_FORMAT_GR88_UNORM;
   case MESA_FORMAT_R16G16_UNORM:
      return PIPE_FORMAT_RG1616_UNORM;
   case MESA_FORMAT_G16R16_UNORM:
      return PIPE_FORMAT_GR1616_UNORM;
   case MESA_FORMAT_RGBA_UNORM16:
      return PIPE_FORMAT_R16G16B16A16_UNORM;

   /* signed int formats */
   case MESA_FORMAT_A_UINT8:
      return PIPE_FORMAT_A8_UINT;
   case MESA_FORMAT_A_UINT16:
      return PIPE_FORMAT_A16_UINT;
   case MESA_FORMAT_A_UINT32:
      return PIPE_FORMAT_A32_UINT;

   case MESA_FORMAT_A_SINT8:
      return PIPE_FORMAT_A8_SINT;
   case MESA_FORMAT_A_SINT16:
      return PIPE_FORMAT_A16_SINT;
   case MESA_FORMAT_A_SINT32:
      return PIPE_FORMAT_A32_SINT;

   case MESA_FORMAT_I_UINT8:
      return PIPE_FORMAT_I8_UINT;
   case MESA_FORMAT_I_UINT16:
      return PIPE_FORMAT_I16_UINT;
   case MESA_FORMAT_I_UINT32:
      return PIPE_FORMAT_I32_UINT;

   case MESA_FORMAT_I_SINT8:
      return PIPE_FORMAT_I8_SINT;
   case MESA_FORMAT_I_SINT16:
      return PIPE_FORMAT_I16_SINT;
   case MESA_FORMAT_I_SINT32:
      return PIPE_FORMAT_I32_SINT;

   case MESA_FORMAT_L_UINT8:
      return PIPE_FORMAT_L8_UINT;
   case MESA_FORMAT_L_UINT16:
      return PIPE_FORMAT_L16_UINT;
   case MESA_FORMAT_L_UINT32:
      return PIPE_FORMAT_L32_UINT;

   case MESA_FORMAT_L_SINT8:
      return PIPE_FORMAT_L8_SINT;
   case MESA_FORMAT_L_SINT16:
      return PIPE_FORMAT_L16_SINT;
   case MESA_FORMAT_L_SINT32:
      return PIPE_FORMAT_L32_SINT;

   case MESA_FORMAT_LA_UINT8:
      return PIPE_FORMAT_L8A8_UINT;
   case MESA_FORMAT_LA_UINT16:
      return PIPE_FORMAT_L16A16_UINT;
   case MESA_FORMAT_LA_UINT32:
      return PIPE_FORMAT_L32A32_UINT;

   case MESA_FORMAT_LA_SINT8:
      return PIPE_FORMAT_L8A8_SINT;
   case MESA_FORMAT_LA_SINT16:
      return PIPE_FORMAT_L16A16_SINT;
   case MESA_FORMAT_LA_SINT32:
      return PIPE_FORMAT_L32A32_SINT;

   case MESA_FORMAT_R_SINT8:
      return PIPE_FORMAT_R8_SINT;
   case MESA_FORMAT_RG_SINT8:
      return PIPE_FORMAT_R8G8_SINT;
   case MESA_FORMAT_RGB_SINT8:
      return PIPE_FORMAT_R8G8B8_SINT;
   case MESA_FORMAT_RGBA_SINT8:
      return PIPE_FORMAT_R8G8B8A8_SINT;
   case MESA_FORMAT_R_SINT16:
      return PIPE_FORMAT_R16_SINT;
   case MESA_FORMAT_RG_SINT16:
      return PIPE_FORMAT_R16G16_SINT;
   case MESA_FORMAT_RGB_SINT16:
      return PIPE_FORMAT_R16G16B16_SINT;
   case MESA_FORMAT_RGBA_SINT16:
      return PIPE_FORMAT_R16G16B16A16_SINT;
   case MESA_FORMAT_R_SINT32:
      return PIPE_FORMAT_R32_SINT;
   case MESA_FORMAT_RG_SINT32:
      return PIPE_FORMAT_R32G32_SINT;
   case MESA_FORMAT_RGB_SINT32:
      return PIPE_FORMAT_R32G32B32_SINT;
   case MESA_FORMAT_RGBA_SINT32:
      return PIPE_FORMAT_R32G32B32A32_SINT;

   /* unsigned int formats */
   case MESA_FORMAT_R_UINT8:
      return PIPE_FORMAT_R8_UINT;
   case MESA_FORMAT_RG_UINT8:
      return PIPE_FORMAT_R8G8_UINT;
   case MESA_FORMAT_RGB_UINT8:
      return PIPE_FORMAT_R8G8B8_UINT;
   case MESA_FORMAT_RGBA_UINT8:
      return PIPE_FORMAT_R8G8B8A8_UINT;
   case MESA_FORMAT_R_UINT16:
      return PIPE_FORMAT_R16_UINT;
   case MESA_FORMAT_RG_UINT16:
      return PIPE_FORMAT_R16G16_UINT;
   case MESA_FORMAT_RGB_UINT16:
      return PIPE_FORMAT_R16G16B16_UINT;
   case MESA_FORMAT_RGBA_UINT16:
      return PIPE_FORMAT_R16G16B16A16_UINT;
   case MESA_FORMAT_R_UINT32:
      return PIPE_FORMAT_R32_UINT;
   case MESA_FORMAT_RG_UINT32:
      return PIPE_FORMAT_R32G32_UINT;
   case MESA_FORMAT_RGB_UINT32:
      return PIPE_FORMAT_R32G32B32_UINT;
   case MESA_FORMAT_RGBA_UINT32:
      return PIPE_FORMAT_R32G32B32A32_UINT;

   case MESA_FORMAT_R_RGTC1_UNORM:
      return PIPE_FORMAT_RGTC1_UNORM;
   case MESA_FORMAT_R_RGTC1_SNORM:
      return PIPE_FORMAT_RGTC1_SNORM;
   case MESA_FORMAT_RG_RGTC2_UNORM:
      return PIPE_FORMAT_RGTC2_UNORM;
   case MESA_FORMAT_RG_RGTC2_SNORM:
      return PIPE_FORMAT_RGTC2_SNORM;

   case MESA_FORMAT_L_LATC1_UNORM:
      return PIPE_FORMAT_LATC1_UNORM;
   case MESA_FORMAT_L_LATC1_SNORM:
      return PIPE_FORMAT_LATC1_SNORM;
   case MESA_FORMAT_LA_LATC2_UNORM:
      return PIPE_FORMAT_LATC2_UNORM;
   case MESA_FORMAT_LA_LATC2_SNORM:
      return PIPE_FORMAT_LATC2_SNORM;

   /* The destination RGBA format mustn't be changed, because it's also
    * a destination format of the unpack/decompression function. */
   case MESA_FORMAT_ETC1_RGB8:
      return st->has_etc1 ? PIPE_FORMAT_ETC1_RGB8 : PIPE_FORMAT_R8G8B8A8_UNORM;

   case MESA_FORMAT_BPTC_RGBA_UNORM:
      return PIPE_FORMAT_BPTC_RGBA_UNORM;
   case MESA_FORMAT_BPTC_SRGB_ALPHA_UNORM:
      return PIPE_FORMAT_BPTC_SRGBA;
   case MESA_FORMAT_BPTC_RGB_SIGNED_FLOAT:
      return PIPE_FORMAT_BPTC_RGB_FLOAT;
   case MESA_FORMAT_BPTC_RGB_UNSIGNED_FLOAT:
      return PIPE_FORMAT_BPTC_RGB_UFLOAT;

   /* signed normalized formats */
   case MESA_FORMAT_R_SNORM8:
      return PIPE_FORMAT_R8_SNORM;
   case MESA_FORMAT_R8G8_SNORM:
      return PIPE_FORMAT_RG88_SNORM;
   case MESA_FORMAT_G8R8_SNORM:
      return PIPE_FORMAT_GR88_SNORM;
   case MESA_FORMAT_R8G8B8A8_SNORM:
      return PIPE_FORMAT_RGBA8888_SNORM;
   case MESA_FORMAT_A8B8G8R8_SNORM:
      return PIPE_FORMAT_ABGR8888_SNORM;

   case MESA_FORMAT_A_SNORM8:
      return PIPE_FORMAT_A8_SNORM;
   case MESA_FORMAT_L_SNORM8:
      return PIPE_FORMAT_L8_SNORM;
   case MESA_FORMAT_L8A8_SNORM:
      return PIPE_FORMAT_LA88_SNORM;
   case MESA_FORMAT_A8L8_SNORM:
      return PIPE_FORMAT_AL88_SNORM;
   case MESA_FORMAT_I_SNORM8:
      return PIPE_FORMAT_I8_SNORM;

   case MESA_FORMAT_R_SNORM16:
      return PIPE_FORMAT_R16_SNORM;
   case MESA_FORMAT_R16G16_SNORM:
      return PIPE_FORMAT_RG1616_SNORM;
   case MESA_FORMAT_G16R16_SNORM:
      return PIPE_FORMAT_GR1616_SNORM;
   case MESA_FORMAT_RGBA_SNORM16:
      return PIPE_FORMAT_R16G16B16A16_SNORM;

   case MESA_FORMAT_A_SNORM16:
      return PIPE_FORMAT_A16_SNORM;
   case MESA_FORMAT_L_SNORM16:
      return PIPE_FORMAT_L16_SNORM;
   case MESA_FORMAT_LA_SNORM16:
      return PIPE_FORMAT_L16A16_SNORM;
   case MESA_FORMAT_I_SNORM16:
      return PIPE_FORMAT_I16_SNORM;

   case MESA_FORMAT_R9G9B9E5_FLOAT:
      return PIPE_FORMAT_R9G9B9E5_FLOAT;
   case MESA_FORMAT_R11G11B10_FLOAT:
      return PIPE_FORMAT_R11G11B10_FLOAT;
   case MESA_FORMAT_B10G10R10A2_UINT:
      return PIPE_FORMAT_B10G10R10A2_UINT;
   case MESA_FORMAT_R10G10B10A2_UINT:
      return PIPE_FORMAT_R10G10B10A2_UINT;

   case MESA_FORMAT_B4G4R4X4_UNORM:
      return PIPE_FORMAT_B4G4R4X4_UNORM;
   case MESA_FORMAT_B5G5R5X1_UNORM:
      return PIPE_FORMAT_B5G5R5X1_UNORM;
   case MESA_FORMAT_R8G8B8X8_SNORM:
      return PIPE_FORMAT_RGBX8888_SNORM;
   case MESA_FORMAT_X8B8G8R8_SNORM:
      return PIPE_FORMAT_XBGR8888_SNORM;
   case MESA_FORMAT_R8G8B8X8_SRGB:
      return PIPE_FORMAT_RGBX8888_SRGB;
   case MESA_FORMAT_X8B8G8R8_SRGB:
      return PIPE_FORMAT_XBGR8888_SRGB;
   case MESA_FORMAT_RGBX_UINT8:
      return PIPE_FORMAT_R8G8B8X8_UINT;
   case MESA_FORMAT_RGBX_SINT8:
      return PIPE_FORMAT_R8G8B8X8_SINT;
   case MESA_FORMAT_B10G10R10X2_UNORM:
      return PIPE_FORMAT_B10G10R10X2_UNORM;
   case MESA_FORMAT_RGBX_UNORM16:
      return PIPE_FORMAT_R16G16B16X16_UNORM;
   case MESA_FORMAT_RGBX_SNORM16:
      return PIPE_FORMAT_R16G16B16X16_SNORM;
   case MESA_FORMAT_RGBX_FLOAT16:
      return PIPE_FORMAT_R16G16B16X16_FLOAT;
   case MESA_FORMAT_RGBX_UINT16:
      return PIPE_FORMAT_R16G16B16X16_UINT;
   case MESA_FORMAT_RGBX_SINT16:
      return PIPE_FORMAT_R16G16B16X16_SINT;
   case MESA_FORMAT_RGBX_FLOAT32:
      return PIPE_FORMAT_R32G32B32X32_FLOAT;
   case MESA_FORMAT_RGBX_UINT32:
      return PIPE_FORMAT_R32G32B32X32_UINT;
   case MESA_FORMAT_RGBX_SINT32:
      return PIPE_FORMAT_R32G32B32X32_SINT;

   case MESA_FORMAT_B8G8R8X8_SRGB:
      return PIPE_FORMAT_BGRX8888_SRGB;
   case MESA_FORMAT_X8R8G8B8_SRGB:
      return PIPE_FORMAT_XRGB8888_SRGB;

   /* ETC2 formats are emulated as uncompressed ones.
    * The destination formats mustn't be changed, because they are also
    * destination formats of the unpack/decompression function. */
   case MESA_FORMAT_ETC2_RGB8:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_RGB8 : PIPE_FORMAT_R8G8B8A8_UNORM;
   case MESA_FORMAT_ETC2_SRGB8:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_SRGB8 : PIPE_FORMAT_B8G8R8A8_SRGB;
   case MESA_FORMAT_ETC2_RGBA8_EAC:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_RGBA8 : PIPE_FORMAT_R8G8B8A8_UNORM;
   case MESA_FORMAT_ETC2_SRGB8_ALPHA8_EAC:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_SRGBA8 : PIPE_FORMAT_B8G8R8A8_SRGB;
   case MESA_FORMAT_ETC2_R11_EAC:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_R11_UNORM : PIPE_FORMAT_R16_UNORM;
   case MESA_FORMAT_ETC2_RG11_EAC:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_RG11_UNORM : PIPE_FORMAT_R16G16_UNORM;
   case MESA_FORMAT_ETC2_SIGNED_R11_EAC:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_R11_SNORM : PIPE_FORMAT_R16_SNORM;
   case MESA_FORMAT_ETC2_SIGNED_RG11_EAC:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_RG11_SNORM : PIPE_FORMAT_R16G16_SNORM;
   case MESA_FORMAT_ETC2_RGB8_PUNCHTHROUGH_ALPHA1:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_RGB8A1 : PIPE_FORMAT_R8G8B8A8_UNORM;
   case MESA_FORMAT_ETC2_SRGB8_PUNCHTHROUGH_ALPHA1:
      return st->has_etc2 ? PIPE_FORMAT_ETC2_SRGB8A1 : PIPE_FORMAT_B8G8R8A8_SRGB;

   case MESA_FORMAT_RGBA_ASTC_4x4:
      return PIPE_FORMAT_ASTC_4x4;
   case MESA_FORMAT_RGBA_ASTC_5x4:
      return PIPE_FORMAT_ASTC_5x4;
   case MESA_FORMAT_RGBA_ASTC_5x5:
      return PIPE_FORMAT_ASTC_5x5;
   case MESA_FORMAT_RGBA_ASTC_6x5:
      return PIPE_FORMAT_ASTC_6x5;
   case MESA_FORMAT_RGBA_ASTC_6x6:
      return PIPE_FORMAT_ASTC_6x6;
   case MESA_FORMAT_RGBA_ASTC_8x5:
      return PIPE_FORMAT_ASTC_8x5;
   case MESA_FORMAT_RGBA_ASTC_8x6:
      return PIPE_FORMAT_ASTC_8x6;
   case MESA_FORMAT_RGBA_ASTC_8x8:
      return PIPE_FORMAT_ASTC_8x8;
   case MESA_FORMAT_RGBA_ASTC_10x5:
      return PIPE_FORMAT_ASTC_10x5;
   case MESA_FORMAT_RGBA_ASTC_10x6:
      return PIPE_FORMAT_ASTC_10x6;
   case MESA_FORMAT_RGBA_ASTC_10x8:
      return PIPE_FORMAT_ASTC_10x8;
   case MESA_FORMAT_RGBA_ASTC_10x10:
      return PIPE_FORMAT_ASTC_10x10;
   case MESA_FORMAT_RGBA_ASTC_12x10:
      return PIPE_FORMAT_ASTC_12x10;
   case MESA_FORMAT_RGBA_ASTC_12x12:
      return PIPE_FORMAT_ASTC_12x12;

   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_4x4:
      return PIPE_FORMAT_ASTC_4x4_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_5x4:
      return PIPE_FORMAT_ASTC_5x4_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_5x5:
      return PIPE_FORMAT_ASTC_5x5_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_6x5:
      return PIPE_FORMAT_ASTC_6x5_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_6x6:
      return PIPE_FORMAT_ASTC_6x6_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_8x5:
      return PIPE_FORMAT_ASTC_8x5_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_8x6:
      return PIPE_FORMAT_ASTC_8x6_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_8x8:
      return PIPE_FORMAT_ASTC_8x8_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x5:
      return PIPE_FORMAT_ASTC_10x5_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x6:
      return PIPE_FORMAT_ASTC_10x6_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x8:
      return PIPE_FORMAT_ASTC_10x8_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x10:
      return PIPE_FORMAT_ASTC_10x10_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_12x10:
      return PIPE_FORMAT_ASTC_12x10_SRGB;
   case MESA_FORMAT_SRGB8_ALPHA8_ASTC_12x12:
      return PIPE_FORMAT_ASTC_12x12_SRGB;

   default:
      return PIPE_FORMAT_NONE;
   }
}


/**
 * Translate Gallium format to Mesa format.
 */
mesa_format
st_pipe_format_to_mesa_format(enum pipe_format format)
{
   switch (format) {
   case PIPE_FORMAT_ABGR8888_UNORM:
      return MESA_FORMAT_A8B8G8R8_UNORM;
   case PIPE_FORMAT_RGBA8888_UNORM:
      return MESA_FORMAT_R8G8B8A8_UNORM;
   case PIPE_FORMAT_BGRA8888_UNORM:
      return MESA_FORMAT_B8G8R8A8_UNORM;
   case PIPE_FORMAT_ARGB8888_UNORM:
      return MESA_FORMAT_A8R8G8B8_UNORM;
   case PIPE_FORMAT_XBGR8888_UNORM:
      return MESA_FORMAT_X8B8G8R8_UNORM;
   case PIPE_FORMAT_RGBX8888_UNORM:
      return MESA_FORMAT_R8G8B8X8_UNORM;
   case PIPE_FORMAT_BGRX8888_UNORM:
      return MESA_FORMAT_B8G8R8X8_UNORM;
   case PIPE_FORMAT_XRGB8888_UNORM:
      return MESA_FORMAT_X8R8G8B8_UNORM;
   case PIPE_FORMAT_B5G5R5A1_UNORM:
      return MESA_FORMAT_B5G5R5A1_UNORM;
   case PIPE_FORMAT_B4G4R4A4_UNORM:
      return MESA_FORMAT_B4G4R4A4_UNORM;
   case PIPE_FORMAT_B5G6R5_UNORM:
      return MESA_FORMAT_B5G6R5_UNORM;
   case PIPE_FORMAT_B2G3R3_UNORM:
      return MESA_FORMAT_B2G3R3_UNORM;
   case PIPE_FORMAT_B10G10R10A2_UNORM:
      return MESA_FORMAT_B10G10R10A2_UNORM;
   case PIPE_FORMAT_R10G10B10A2_UNORM:
      return MESA_FORMAT_R10G10B10A2_UNORM;
   case PIPE_FORMAT_L4A4_UNORM:
      return MESA_FORMAT_L4A4_UNORM;
   case PIPE_FORMAT_LA88_UNORM:
      return MESA_FORMAT_L8A8_UNORM;
   case PIPE_FORMAT_AL88_UNORM:
      return MESA_FORMAT_A8L8_UNORM;
   case PIPE_FORMAT_LA1616_UNORM:
      return MESA_FORMAT_L16A16_UNORM;
   case PIPE_FORMAT_AL1616_UNORM:
      return MESA_FORMAT_A16L16_UNORM;
   case PIPE_FORMAT_A8_UNORM:
      return MESA_FORMAT_A_UNORM8;
   case PIPE_FORMAT_A16_UNORM:
      return MESA_FORMAT_A_UNORM16;
   case PIPE_FORMAT_L8_UNORM:
      return MESA_FORMAT_L_UNORM8;
   case PIPE_FORMAT_L16_UNORM:
      return MESA_FORMAT_L_UNORM16;
   case PIPE_FORMAT_I8_UNORM:
      return MESA_FORMAT_I_UNORM8;
   case PIPE_FORMAT_I16_UNORM:
      return MESA_FORMAT_I_UNORM16;
   case PIPE_FORMAT_S8_UINT:
      return MESA_FORMAT_S_UINT8;

   case PIPE_FORMAT_R16G16B16A16_UNORM:
      return MESA_FORMAT_RGBA_UNORM16;

   case PIPE_FORMAT_Z16_UNORM:
      return MESA_FORMAT_Z_UNORM16;
   case PIPE_FORMAT_Z32_UNORM:
      return MESA_FORMAT_Z_UNORM32;
   case PIPE_FORMAT_S8_UINT_Z24_UNORM:
      return MESA_FORMAT_S8_UINT_Z24_UNORM;
   case PIPE_FORMAT_X8Z24_UNORM:
      return MESA_FORMAT_X8_UINT_Z24_UNORM;
   case PIPE_FORMAT_Z24X8_UNORM:
      return MESA_FORMAT_Z24_UNORM_X8_UINT;
   case PIPE_FORMAT_Z24_UNORM_S8_UINT:
      return MESA_FORMAT_Z24_UNORM_S8_UINT;
   case PIPE_FORMAT_Z32_FLOAT:
      return MESA_FORMAT_Z_FLOAT32;
   case PIPE_FORMAT_Z32_FLOAT_S8X24_UINT:
      return MESA_FORMAT_Z32_FLOAT_S8X24_UINT;

   case PIPE_FORMAT_UYVY:
      return MESA_FORMAT_YCBCR;
   case PIPE_FORMAT_YUYV:
      return MESA_FORMAT_YCBCR_REV;

   case PIPE_FORMAT_DXT1_RGB:
      return MESA_FORMAT_RGB_DXT1;
   case PIPE_FORMAT_DXT1_RGBA:
      return MESA_FORMAT_RGBA_DXT1;
   case PIPE_FORMAT_DXT3_RGBA:
      return MESA_FORMAT_RGBA_DXT3;
   case PIPE_FORMAT_DXT5_RGBA:
      return MESA_FORMAT_RGBA_DXT5;
   case PIPE_FORMAT_DXT1_SRGB:
      return MESA_FORMAT_SRGB_DXT1;
   case PIPE_FORMAT_DXT1_SRGBA:
      return MESA_FORMAT_SRGBA_DXT1;
   case PIPE_FORMAT_DXT3_SRGBA:
      return MESA_FORMAT_SRGBA_DXT3;
   case PIPE_FORMAT_DXT5_SRGBA:
      return MESA_FORMAT_SRGBA_DXT5;
   case PIPE_FORMAT_LA88_SRGB:
      return MESA_FORMAT_L8A8_SRGB;
   case PIPE_FORMAT_AL88_SRGB:
      return MESA_FORMAT_A8L8_SRGB;
   case PIPE_FORMAT_L8_SRGB:
      return MESA_FORMAT_L_SRGB8;
   case PIPE_FORMAT_R8G8B8_SRGB:
      return MESA_FORMAT_BGR_SRGB8;
   case PIPE_FORMAT_ABGR8888_SRGB:
      return MESA_FORMAT_A8B8G8R8_SRGB;
   case PIPE_FORMAT_RGBA8888_SRGB:
      return MESA_FORMAT_R8G8B8A8_SRGB;
   case PIPE_FORMAT_BGRA8888_SRGB:
      return MESA_FORMAT_B8G8R8A8_SRGB;
   case PIPE_FORMAT_ARGB8888_SRGB:
      return MESA_FORMAT_A8R8G8B8_SRGB;
   case PIPE_FORMAT_R32G32B32A32_FLOAT:
      return MESA_FORMAT_RGBA_FLOAT32;
   case PIPE_FORMAT_R16G16B16A16_FLOAT:
      return MESA_FORMAT_RGBA_FLOAT16;
   case PIPE_FORMAT_R32G32B32_FLOAT:
      return MESA_FORMAT_RGB_FLOAT32;
   case PIPE_FORMAT_R16G16B16_FLOAT:
      return MESA_FORMAT_RGB_FLOAT16;
   case PIPE_FORMAT_L32A32_FLOAT:
      return MESA_FORMAT_LA_FLOAT32;
   case PIPE_FORMAT_L16A16_FLOAT:
      return MESA_FORMAT_LA_FLOAT16;
   case PIPE_FORMAT_L32_FLOAT:
      return MESA_FORMAT_L_FLOAT32;
   case PIPE_FORMAT_L16_FLOAT:
      return MESA_FORMAT_L_FLOAT16;
   case PIPE_FORMAT_A32_FLOAT:
      return MESA_FORMAT_A_FLOAT32;
   case PIPE_FORMAT_A16_FLOAT:
      return MESA_FORMAT_A_FLOAT16;
   case PIPE_FORMAT_I32_FLOAT:
      return MESA_FORMAT_I_FLOAT32;
   case PIPE_FORMAT_I16_FLOAT:
      return MESA_FORMAT_I_FLOAT16;
   case PIPE_FORMAT_R32_FLOAT:
      return MESA_FORMAT_R_FLOAT32;
   case PIPE_FORMAT_R16_FLOAT:
      return MESA_FORMAT_R_FLOAT16;
   case PIPE_FORMAT_R32G32_FLOAT:
      return MESA_FORMAT_RG_FLOAT32;
   case PIPE_FORMAT_R16G16_FLOAT:
      return MESA_FORMAT_RG_FLOAT16;

   case PIPE_FORMAT_R8_UNORM:
      return MESA_FORMAT_R_UNORM8;
   case PIPE_FORMAT_R16_UNORM:
      return MESA_FORMAT_R_UNORM16;
   case PIPE_FORMAT_RG88_UNORM:
      return MESA_FORMAT_R8G8_UNORM;
   case PIPE_FORMAT_GR88_UNORM:
      return MESA_FORMAT_G8R8_UNORM;
   case PIPE_FORMAT_RG1616_UNORM:
      return MESA_FORMAT_R16G16_UNORM;
   case PIPE_FORMAT_GR1616_UNORM:
      return MESA_FORMAT_G16R16_UNORM;

   case PIPE_FORMAT_A8_UINT:
      return MESA_FORMAT_A_UINT8;
   case PIPE_FORMAT_A16_UINT:
      return MESA_FORMAT_A_UINT16;
   case PIPE_FORMAT_A32_UINT:
      return MESA_FORMAT_A_UINT32;
   case PIPE_FORMAT_A8_SINT:
      return MESA_FORMAT_A_SINT8;
   case PIPE_FORMAT_A16_SINT:
      return MESA_FORMAT_A_SINT16;
   case PIPE_FORMAT_A32_SINT:
      return MESA_FORMAT_A_SINT32;

   case PIPE_FORMAT_I8_UINT:
      return MESA_FORMAT_I_UINT8;
   case PIPE_FORMAT_I16_UINT:
      return MESA_FORMAT_I_UINT16;
   case PIPE_FORMAT_I32_UINT:
      return MESA_FORMAT_I_UINT32;
   case PIPE_FORMAT_I8_SINT:
      return MESA_FORMAT_I_SINT8;
   case PIPE_FORMAT_I16_SINT:
      return MESA_FORMAT_I_SINT16;
   case PIPE_FORMAT_I32_SINT:
      return MESA_FORMAT_I_SINT32;

  case PIPE_FORMAT_L8_UINT:
      return MESA_FORMAT_L_UINT8;
   case PIPE_FORMAT_L16_UINT:
      return MESA_FORMAT_L_UINT16;
   case PIPE_FORMAT_L32_UINT:
      return MESA_FORMAT_L_UINT32;
   case PIPE_FORMAT_L8_SINT:
      return MESA_FORMAT_L_SINT8;
   case PIPE_FORMAT_L16_SINT:
      return MESA_FORMAT_L_SINT16;
   case PIPE_FORMAT_L32_SINT:
      return MESA_FORMAT_L_SINT32;

   case PIPE_FORMAT_L8A8_UINT:
      return MESA_FORMAT_LA_UINT8;
   case PIPE_FORMAT_L16A16_UINT:
      return MESA_FORMAT_LA_UINT16;
   case PIPE_FORMAT_L32A32_UINT:
      return MESA_FORMAT_LA_UINT32;
   case PIPE_FORMAT_L8A8_SINT:
      return MESA_FORMAT_LA_SINT8;
   case PIPE_FORMAT_L16A16_SINT:
      return MESA_FORMAT_LA_SINT16;
   case PIPE_FORMAT_L32A32_SINT:
      return MESA_FORMAT_LA_SINT32;

   case PIPE_FORMAT_R8_SINT:
      return MESA_FORMAT_R_SINT8;
   case PIPE_FORMAT_R8G8_SINT:
      return MESA_FORMAT_RG_SINT8;
   case PIPE_FORMAT_R8G8B8_SINT:
      return MESA_FORMAT_RGB_SINT8;
   case PIPE_FORMAT_R8G8B8A8_SINT:
      return MESA_FORMAT_RGBA_SINT8;

   case PIPE_FORMAT_R16_SINT:
      return MESA_FORMAT_R_SINT16;
   case PIPE_FORMAT_R16G16_SINT:
      return MESA_FORMAT_RG_SINT16;
   case PIPE_FORMAT_R16G16B16_SINT:
      return MESA_FORMAT_RGB_SINT16;
   case PIPE_FORMAT_R16G16B16A16_SINT:
      return MESA_FORMAT_RGBA_SINT16;

   case PIPE_FORMAT_R32_SINT:
      return MESA_FORMAT_R_SINT32;
   case PIPE_FORMAT_R32G32_SINT:
      return MESA_FORMAT_RG_SINT32;
   case PIPE_FORMAT_R32G32B32_SINT:
      return MESA_FORMAT_RGB_SINT32;
   case PIPE_FORMAT_R32G32B32A32_SINT:
      return MESA_FORMAT_RGBA_SINT32;

   /* unsigned int formats */
   case PIPE_FORMAT_R8_UINT:
      return MESA_FORMAT_R_UINT8;
   case PIPE_FORMAT_R8G8_UINT:
      return MESA_FORMAT_RG_UINT8;
   case PIPE_FORMAT_R8G8B8_UINT:
      return MESA_FORMAT_RGB_UINT8;
   case PIPE_FORMAT_R8G8B8A8_UINT:
      return MESA_FORMAT_RGBA_UINT8;

   case PIPE_FORMAT_R16_UINT:
      return MESA_FORMAT_R_UINT16;
   case PIPE_FORMAT_R16G16_UINT:
      return MESA_FORMAT_RG_UINT16;
   case PIPE_FORMAT_R16G16B16_UINT:
      return MESA_FORMAT_RGB_UINT16;
   case PIPE_FORMAT_R16G16B16A16_UINT:
      return MESA_FORMAT_RGBA_UINT16;

   case PIPE_FORMAT_R32_UINT:
      return MESA_FORMAT_R_UINT32;
   case PIPE_FORMAT_R32G32_UINT:
      return MESA_FORMAT_RG_UINT32;
   case PIPE_FORMAT_R32G32B32_UINT:
      return MESA_FORMAT_RGB_UINT32;
   case PIPE_FORMAT_R32G32B32A32_UINT:
      return MESA_FORMAT_RGBA_UINT32;

   case PIPE_FORMAT_RGTC1_UNORM:
      return MESA_FORMAT_R_RGTC1_UNORM;
   case PIPE_FORMAT_RGTC1_SNORM:
      return MESA_FORMAT_R_RGTC1_SNORM;
   case PIPE_FORMAT_RGTC2_UNORM:
      return MESA_FORMAT_RG_RGTC2_UNORM;
   case PIPE_FORMAT_RGTC2_SNORM:
      return MESA_FORMAT_RG_RGTC2_SNORM;

   case PIPE_FORMAT_LATC1_UNORM:
      return MESA_FORMAT_L_LATC1_UNORM;
   case PIPE_FORMAT_LATC1_SNORM:
      return MESA_FORMAT_L_LATC1_SNORM;
   case PIPE_FORMAT_LATC2_UNORM:
      return MESA_FORMAT_LA_LATC2_UNORM;
   case PIPE_FORMAT_LATC2_SNORM:
      return MESA_FORMAT_LA_LATC2_SNORM;

   case PIPE_FORMAT_ETC1_RGB8:
      return MESA_FORMAT_ETC1_RGB8;

   case PIPE_FORMAT_BPTC_RGBA_UNORM:
      return MESA_FORMAT_BPTC_RGBA_UNORM;
   case PIPE_FORMAT_BPTC_SRGBA:
      return MESA_FORMAT_BPTC_SRGB_ALPHA_UNORM;
   case PIPE_FORMAT_BPTC_RGB_FLOAT:
      return MESA_FORMAT_BPTC_RGB_SIGNED_FLOAT;
   case PIPE_FORMAT_BPTC_RGB_UFLOAT:
      return MESA_FORMAT_BPTC_RGB_UNSIGNED_FLOAT;

   /* signed normalized formats */
   case PIPE_FORMAT_R8_SNORM:
      return MESA_FORMAT_R_SNORM8;
   case PIPE_FORMAT_RG88_SNORM:
      return MESA_FORMAT_R8G8_SNORM;
   case PIPE_FORMAT_GR88_SNORM:
      return MESA_FORMAT_G8R8_SNORM;
   case PIPE_FORMAT_RGBA8888_SNORM:
      return MESA_FORMAT_R8G8B8A8_SNORM;
   case PIPE_FORMAT_ABGR8888_SNORM:
      return MESA_FORMAT_A8B8G8R8_SNORM;

   case PIPE_FORMAT_A8_SNORM:
      return MESA_FORMAT_A_SNORM8;
   case PIPE_FORMAT_L8_SNORM:
      return MESA_FORMAT_L_SNORM8;
   case PIPE_FORMAT_LA88_SNORM:
      return MESA_FORMAT_L8A8_SNORM;
   case PIPE_FORMAT_AL88_SNORM:
      return MESA_FORMAT_A8L8_SNORM;
   case PIPE_FORMAT_I8_SNORM:
      return MESA_FORMAT_I_SNORM8;

   case PIPE_FORMAT_R16_SNORM:
      return MESA_FORMAT_R_SNORM16;
   case PIPE_FORMAT_RG1616_SNORM:
      return MESA_FORMAT_R16G16_SNORM;
   case PIPE_FORMAT_GR1616_SNORM:
      return MESA_FORMAT_G16R16_SNORM;
   case PIPE_FORMAT_R16G16B16A16_SNORM:
      return MESA_FORMAT_RGBA_SNORM16;

   case PIPE_FORMAT_A16_SNORM:
      return MESA_FORMAT_A_SNORM16;
   case PIPE_FORMAT_L16_SNORM:
      return MESA_FORMAT_L_SNORM16;
   case PIPE_FORMAT_L16A16_SNORM:
      return MESA_FORMAT_LA_SNORM16;
   case PIPE_FORMAT_I16_SNORM:
      return MESA_FORMAT_I_SNORM16;

   case PIPE_FORMAT_R9G9B9E5_FLOAT:
      return MESA_FORMAT_R9G9B9E5_FLOAT;
   case PIPE_FORMAT_R11G11B10_FLOAT:
      return MESA_FORMAT_R11G11B10_FLOAT;

   case PIPE_FORMAT_B10G10R10A2_UINT:
      return MESA_FORMAT_B10G10R10A2_UINT;
   case PIPE_FORMAT_R10G10B10A2_UINT:
      return MESA_FORMAT_R10G10B10A2_UINT;

   case PIPE_FORMAT_B4G4R4X4_UNORM:
      return MESA_FORMAT_B4G4R4X4_UNORM;
   case PIPE_FORMAT_B5G5R5X1_UNORM:
      return MESA_FORMAT_B5G5R5X1_UNORM;
   case PIPE_FORMAT_RGBX8888_SNORM:
      return MESA_FORMAT_R8G8B8X8_SNORM;
   case PIPE_FORMAT_XBGR8888_SNORM:
      return MESA_FORMAT_X8B8G8R8_SNORM;
   case PIPE_FORMAT_RGBX8888_SRGB:
      return MESA_FORMAT_R8G8B8X8_SRGB;
   case PIPE_FORMAT_XBGR8888_SRGB:
      return MESA_FORMAT_X8B8G8R8_SRGB;
   case PIPE_FORMAT_R8G8B8X8_UINT:
      return MESA_FORMAT_RGBX_UINT8;
   case PIPE_FORMAT_R8G8B8X8_SINT:
      return MESA_FORMAT_RGBX_SINT8;
   case PIPE_FORMAT_B10G10R10X2_UNORM:
      return MESA_FORMAT_B10G10R10X2_UNORM;
   case PIPE_FORMAT_R16G16B16X16_UNORM:
      return MESA_FORMAT_RGBX_UNORM16;
   case PIPE_FORMAT_R16G16B16X16_SNORM:
      return MESA_FORMAT_RGBX_SNORM16;
   case PIPE_FORMAT_R16G16B16X16_FLOAT:
      return MESA_FORMAT_RGBX_FLOAT16;
   case PIPE_FORMAT_R16G16B16X16_UINT:
      return MESA_FORMAT_RGBX_UINT16;
   case PIPE_FORMAT_R16G16B16X16_SINT:
      return MESA_FORMAT_RGBX_SINT16;
   case PIPE_FORMAT_R32G32B32X32_FLOAT:
      return MESA_FORMAT_RGBX_FLOAT32;
   case PIPE_FORMAT_R32G32B32X32_UINT:
      return MESA_FORMAT_RGBX_UINT32;
   case PIPE_FORMAT_R32G32B32X32_SINT:
      return MESA_FORMAT_RGBX_SINT32;

   case PIPE_FORMAT_BGRX8888_SRGB:
      return MESA_FORMAT_B8G8R8X8_SRGB;
   case PIPE_FORMAT_XRGB8888_SRGB:
      return MESA_FORMAT_X8R8G8B8_SRGB;

   case PIPE_FORMAT_ETC2_RGB8:
      return MESA_FORMAT_ETC2_RGB8;
   case PIPE_FORMAT_ETC2_SRGB8:
      return MESA_FORMAT_ETC2_SRGB8;
   case PIPE_FORMAT_ETC2_RGB8A1:
      return MESA_FORMAT_ETC2_RGB8_PUNCHTHROUGH_ALPHA1;
   case PIPE_FORMAT_ETC2_SRGB8A1:
      return MESA_FORMAT_ETC2_SRGB8_PUNCHTHROUGH_ALPHA1;
   case PIPE_FORMAT_ETC2_RGBA8:
      return MESA_FORMAT_ETC2_RGBA8_EAC;
   case PIPE_FORMAT_ETC2_SRGBA8:
      return MESA_FORMAT_ETC2_SRGB8_ALPHA8_EAC;
   case PIPE_FORMAT_ETC2_R11_UNORM:
      return MESA_FORMAT_ETC2_R11_EAC;
   case PIPE_FORMAT_ETC2_R11_SNORM:
      return MESA_FORMAT_ETC2_SIGNED_R11_EAC;
   case PIPE_FORMAT_ETC2_RG11_UNORM:
      return MESA_FORMAT_ETC2_RG11_EAC;
   case PIPE_FORMAT_ETC2_RG11_SNORM:
      return MESA_FORMAT_ETC2_SIGNED_RG11_EAC;

   case PIPE_FORMAT_ASTC_4x4:
      return MESA_FORMAT_RGBA_ASTC_4x4;
   case PIPE_FORMAT_ASTC_5x4:
      return MESA_FORMAT_RGBA_ASTC_5x4;
   case PIPE_FORMAT_ASTC_5x5:
      return MESA_FORMAT_RGBA_ASTC_5x5;
   case PIPE_FORMAT_ASTC_6x5:
      return MESA_FORMAT_RGBA_ASTC_6x5;
   case PIPE_FORMAT_ASTC_6x6:
      return MESA_FORMAT_RGBA_ASTC_6x6;
   case PIPE_FORMAT_ASTC_8x5:
      return MESA_FORMAT_RGBA_ASTC_8x5;
   case PIPE_FORMAT_ASTC_8x6:
      return MESA_FORMAT_RGBA_ASTC_8x6;
   case PIPE_FORMAT_ASTC_8x8:
      return MESA_FORMAT_RGBA_ASTC_8x8;
   case PIPE_FORMAT_ASTC_10x5:
      return MESA_FORMAT_RGBA_ASTC_10x5;
   case PIPE_FORMAT_ASTC_10x6:
      return MESA_FORMAT_RGBA_ASTC_10x6;
   case PIPE_FORMAT_ASTC_10x8:
      return MESA_FORMAT_RGBA_ASTC_10x8;
   case PIPE_FORMAT_ASTC_10x10:
      return MESA_FORMAT_RGBA_ASTC_10x10;
   case PIPE_FORMAT_ASTC_12x10:
      return MESA_FORMAT_RGBA_ASTC_12x10;
   case PIPE_FORMAT_ASTC_12x12:
      return MESA_FORMAT_RGBA_ASTC_12x12;

   case PIPE_FORMAT_ASTC_4x4_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_4x4;
   case PIPE_FORMAT_ASTC_5x4_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_5x4;
   case PIPE_FORMAT_ASTC_5x5_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_5x5;
   case PIPE_FORMAT_ASTC_6x5_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_6x5;
   case PIPE_FORMAT_ASTC_6x6_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_6x6;
   case PIPE_FORMAT_ASTC_8x5_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_8x5;
   case PIPE_FORMAT_ASTC_8x6_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_8x6;
   case PIPE_FORMAT_ASTC_8x8_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_8x8;
   case PIPE_FORMAT_ASTC_10x5_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x5;
   case PIPE_FORMAT_ASTC_10x6_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x6;
   case PIPE_FORMAT_ASTC_10x8_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x8;
   case PIPE_FORMAT_ASTC_10x10_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_10x10;
   case PIPE_FORMAT_ASTC_12x10_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_12x10;
   case PIPE_FORMAT_ASTC_12x12_SRGB:
      return MESA_FORMAT_SRGB8_ALPHA8_ASTC_12x12;

   default:
      return MESA_FORMAT_NONE;
   }
}


/**
 * Debug only: check that the two functions above correctly map
 * Mesa formats to Gallium formats and back again.
 */
static void
test_format_conversion(struct st_context *st)
{
   GLuint i;

   /* test all Mesa formats */
   for (i = 1; i < MESA_FORMAT_COUNT; i++) {
      enum pipe_format pf;

      /* ETC formats are translated differently, skip them. */
      if (_mesa_is_format_etc2(i))
         continue;
      if (i == MESA_FORMAT_ETC1_RGB8 && !st->has_etc1)
         continue;

      pf = st_mesa_format_to_pipe_format(st, i);
      if (pf != PIPE_FORMAT_NONE) {
         mesa_format mf = st_pipe_format_to_mesa_format(pf);
         assert(mf == i);
      }
   }

   /* Test all Gallium formats */
   for (i = 1; i < PIPE_FORMAT_COUNT; i++) {
      mesa_format mf = st_pipe_format_to_mesa_format(i);

      /* ETC formats are translated differently, skip them. */
      if (i == PIPE_FORMAT_ETC1_RGB8 && !st->has_etc1)
         continue;

      if (_mesa_is_format_etc2(mf) && !st->has_etc2)
         continue;

      if (mf != MESA_FORMAT_NONE) {
         enum pipe_format pf = st_mesa_format_to_pipe_format(st, mf);
         assert(pf == i);
      }
   }
}


/**
 * Map GL texture formats to Gallium pipe formats.
 */
struct format_mapping
{
   GLenum glFormats[18];       /**< list of GLenum formats, 0-terminated */
   enum pipe_format pipeFormats[13]; /**< list of pipe formats, 0-terminated */
};


#define DEFAULT_RGBA_FORMATS \
      PIPE_FORMAT_R8G8B8A8_UNORM, \
      PIPE_FORMAT_B8G8R8A8_UNORM, \
      PIPE_FORMAT_A8R8G8B8_UNORM, \
      PIPE_FORMAT_A8B8G8R8_UNORM, \
      0

#define DEFAULT_RGB_FORMATS \
      PIPE_FORMAT_R8G8B8X8_UNORM, \
      PIPE_FORMAT_B8G8R8X8_UNORM, \
      PIPE_FORMAT_X8R8G8B8_UNORM, \
      PIPE_FORMAT_X8B8G8R8_UNORM, \
      PIPE_FORMAT_B5G6R5_UNORM, \
      DEFAULT_RGBA_FORMATS

#define DEFAULT_SRGBA_FORMATS \
      PIPE_FORMAT_R8G8B8A8_SRGB, \
      PIPE_FORMAT_B8G8R8A8_SRGB, \
      PIPE_FORMAT_A8R8G8B8_SRGB, \
      PIPE_FORMAT_A8B8G8R8_SRGB, \
      0

#define DEFAULT_DEPTH_FORMATS \
      PIPE_FORMAT_Z24X8_UNORM, \
      PIPE_FORMAT_X8Z24_UNORM, \
      PIPE_FORMAT_Z16_UNORM, \
      PIPE_FORMAT_Z24_UNORM_S8_UINT, \
      PIPE_FORMAT_S8_UINT_Z24_UNORM, \
      0

#define DEFAULT_SNORM8_RGBA_FORMATS \
      PIPE_FORMAT_R8G8B8A8_SNORM, \
      0

#define DEFAULT_UNORM16_RGBA_FORMATS \
      PIPE_FORMAT_R16G16B16A16_UNORM, \
      DEFAULT_RGBA_FORMATS


/**
 * This table maps OpenGL texture format enums to Gallium pipe_format enums.
 * Multiple GL enums might map to multiple pipe_formats.
 * The first pipe format in the list that's supported is the one that's chosen.
 */
static const struct format_mapping format_map[] = {
   /* Basic RGB, RGBA formats */
   {
      { GL_RGB10, 0 },
      { PIPE_FORMAT_B10G10R10X2_UNORM, PIPE_FORMAT_B10G10R10A2_UNORM,
        PIPE_FORMAT_R10G10B10A2_UNORM, DEFAULT_RGB_FORMATS }
   },
   {
      { GL_RGB10_A2, 0 },
      { PIPE_FORMAT_R10G10B10A2_UNORM, PIPE_FORMAT_B10G10R10A2_UNORM,
        DEFAULT_RGBA_FORMATS }
   },
   {
      { 4, GL_RGBA, GL_RGBA8, 0 },
      { PIPE_FORMAT_R8G8B8A8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_BGRA, 0 },
      { DEFAULT_RGBA_FORMATS }
   },
   {
      { 3, GL_RGB, GL_RGB8, 0 },
      { PIPE_FORMAT_R8G8B8X8_UNORM, DEFAULT_RGB_FORMATS }
   },
   {
      { GL_RGB12, GL_RGB16, 0 },
      { PIPE_FORMAT_R16G16B16X16_UNORM, PIPE_FORMAT_R16G16B16A16_UNORM,
        DEFAULT_RGB_FORMATS }
   },
   {
      { GL_RGBA12, GL_RGBA16, 0 },
      { PIPE_FORMAT_R16G16B16A16_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_RGBA4, GL_RGBA2, 0 },
      { PIPE_FORMAT_B4G4R4A4_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_RGB5_A1, 0 },
      { PIPE_FORMAT_B5G5R5A1_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_R3_G3_B2, 0 },
      { PIPE_FORMAT_B2G3R3_UNORM, PIPE_FORMAT_B5G6R5_UNORM,
        PIPE_FORMAT_B5G5R5A1_UNORM, DEFAULT_RGB_FORMATS }
   },
   {
      { GL_RGB4 },
      { PIPE_FORMAT_B4G4R4X4_UNORM, PIPE_FORMAT_B4G4R4A4_UNORM,
        DEFAULT_RGB_FORMATS }
   },
   {
      { GL_RGB5 },
      { PIPE_FORMAT_B5G5R5X1_UNORM, PIPE_FORMAT_B5G5R5A1_UNORM,
        DEFAULT_RGB_FORMATS }
   },
   {
      { GL_RGB565 },
      { PIPE_FORMAT_B5G6R5_UNORM, DEFAULT_RGB_FORMATS }
   },

   /* basic Alpha formats */
   {
      { GL_ALPHA12, GL_ALPHA16, 0 },
      { PIPE_FORMAT_A16_UNORM, PIPE_FORMAT_R16G16B16A16_UNORM,
        PIPE_FORMAT_A8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_ALPHA, GL_ALPHA4, GL_ALPHA8, GL_COMPRESSED_ALPHA, 0 },
      { PIPE_FORMAT_A8_UNORM, DEFAULT_RGBA_FORMATS }
   },

   /* basic Luminance formats */
   {
      { GL_LUMINANCE12, GL_LUMINANCE16, 0 },
      { PIPE_FORMAT_L16_UNORM, PIPE_FORMAT_R16G16B16A16_UNORM,
        PIPE_FORMAT_L8_UNORM, DEFAULT_RGB_FORMATS }
   },
   {
      { 1, GL_LUMINANCE, GL_LUMINANCE4, GL_LUMINANCE8, 0 },
      { PIPE_FORMAT_L8_UNORM, DEFAULT_RGB_FORMATS }
   },

   /* basic Luminance/Alpha formats */
   {
      { GL_LUMINANCE12_ALPHA4, GL_LUMINANCE12_ALPHA12,
        GL_LUMINANCE16_ALPHA16, 0},
      { PIPE_FORMAT_L16A16_UNORM, PIPE_FORMAT_R16G16B16A16_UNORM,
        PIPE_FORMAT_L8A8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { 2, GL_LUMINANCE_ALPHA, GL_LUMINANCE6_ALPHA2, GL_LUMINANCE8_ALPHA8, 0 },
      { PIPE_FORMAT_L8A8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_LUMINANCE4_ALPHA4, 0 },
      { PIPE_FORMAT_L4A4_UNORM, PIPE_FORMAT_L8A8_UNORM,
        DEFAULT_RGBA_FORMATS }
   },

   /* basic Intensity formats */
   {
      { GL_INTENSITY12, GL_INTENSITY16, 0 },
      { PIPE_FORMAT_I16_UNORM, PIPE_FORMAT_R16G16B16A16_UNORM,
        PIPE_FORMAT_I8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_INTENSITY, GL_INTENSITY4, GL_INTENSITY8,
        GL_COMPRESSED_INTENSITY, 0 },
      { PIPE_FORMAT_I8_UNORM, DEFAULT_RGBA_FORMATS }
   },

   /* YCbCr */
   {
      { GL_YCBCR_MESA, 0 },
      { PIPE_FORMAT_UYVY, PIPE_FORMAT_YUYV, 0 }
   },

   /* compressed formats */ /* XXX PIPE_BIND_SAMPLER_VIEW only */
   {
      { GL_COMPRESSED_RGB, 0 },
      { PIPE_FORMAT_DXT1_RGB, DEFAULT_RGB_FORMATS }
   },
   {
      { GL_COMPRESSED_RGBA, 0 },
      { PIPE_FORMAT_DXT5_RGBA, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_RGB_S3TC, GL_RGB4_S3TC, GL_COMPRESSED_RGB_S3TC_DXT1_EXT, 0 },
      { PIPE_FORMAT_DXT1_RGB, 0 }
   },
   {
      { GL_COMPRESSED_RGBA_S3TC_DXT1_EXT, 0 },
      { PIPE_FORMAT_DXT1_RGBA, 0 }
   },
   {
      { GL_RGBA_S3TC, GL_RGBA4_S3TC, GL_COMPRESSED_RGBA_S3TC_DXT3_EXT, 0 },
      { PIPE_FORMAT_DXT3_RGBA, 0 }
   },
   {
      { GL_COMPRESSED_RGBA_S3TC_DXT5_EXT, 0 },
      { PIPE_FORMAT_DXT5_RGBA, 0 }
   },

#if 0
   {
      { GL_COMPRESSED_RGB_FXT1_3DFX, 0 },
      { PIPE_FORMAT_RGB_FXT1, 0 }
   },
   {
      { GL_COMPRESSED_RGBA_FXT1_3DFX, 0 },
      { PIPE_FORMAT_RGBA_FXT1, 0 }
   },
#endif

   /* Depth formats */
   {
      { GL_DEPTH_COMPONENT16, 0 },
      { PIPE_FORMAT_Z16_UNORM, DEFAULT_DEPTH_FORMATS }
   },
   {
      { GL_DEPTH_COMPONENT24, 0 },
      { PIPE_FORMAT_Z24X8_UNORM, PIPE_FORMAT_X8Z24_UNORM,
        DEFAULT_DEPTH_FORMATS }
   },
   {
      { GL_DEPTH_COMPONENT32, 0 },
      { PIPE_FORMAT_Z32_UNORM, DEFAULT_DEPTH_FORMATS }
   },
   {
      { GL_DEPTH_COMPONENT, 0 },
      { DEFAULT_DEPTH_FORMATS }
   },
   {
      { GL_DEPTH_COMPONENT32F, 0 },
      { PIPE_FORMAT_Z32_FLOAT, 0 }
   },

   /* stencil formats */
   {
      { GL_STENCIL_INDEX, GL_STENCIL_INDEX1_EXT, GL_STENCIL_INDEX4_EXT,
        GL_STENCIL_INDEX8_EXT, GL_STENCIL_INDEX16_EXT, 0 },
      {
         PIPE_FORMAT_S8_UINT, PIPE_FORMAT_Z24_UNORM_S8_UINT,
         PIPE_FORMAT_S8_UINT_Z24_UNORM, 0
      }
   },

   /* Depth / Stencil formats */
   {
      { GL_DEPTH_STENCIL_EXT, GL_DEPTH24_STENCIL8_EXT, 0 },
      { PIPE_FORMAT_Z24_UNORM_S8_UINT, PIPE_FORMAT_S8_UINT_Z24_UNORM, 0 }
   },
   {
      { GL_DEPTH32F_STENCIL8, 0 },
      { PIPE_FORMAT_Z32_FLOAT_S8X24_UINT, 0 }
   },

   /* sRGB formats */
   {
      { GL_SRGB_EXT, GL_SRGB8_EXT, 0 },
      { PIPE_FORMAT_R8G8B8X8_SRGB, PIPE_FORMAT_B8G8R8X8_SRGB,
        DEFAULT_SRGBA_FORMATS }
   },
   {
      { GL_SRGB_ALPHA_EXT, GL_SRGB8_ALPHA8_EXT, 0 },
      { PIPE_FORMAT_R8G8B8A8_SRGB, DEFAULT_SRGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_SRGB_EXT, GL_COMPRESSED_SRGB_S3TC_DXT1_EXT, 0 },
      { PIPE_FORMAT_DXT1_SRGB, PIPE_FORMAT_R8G8B8X8_SRGB,
        PIPE_FORMAT_B8G8R8X8_SRGB, DEFAULT_SRGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT, 0 },
      { PIPE_FORMAT_DXT1_SRGBA, 0 }
   },
   {
      { GL_COMPRESSED_SRGB_ALPHA_EXT,
        GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT, 0 },
      { PIPE_FORMAT_DXT3_SRGBA, DEFAULT_SRGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT, 0 },
      { PIPE_FORMAT_DXT5_SRGBA, 0 }
   },
   {
      { GL_SLUMINANCE_ALPHA_EXT, GL_SLUMINANCE8_ALPHA8_EXT,
        GL_COMPRESSED_SLUMINANCE_ALPHA_EXT, 0 },
      { PIPE_FORMAT_L8A8_SRGB, DEFAULT_SRGBA_FORMATS }
   },
   {
      { GL_SLUMINANCE_EXT, GL_SLUMINANCE8_EXT, GL_COMPRESSED_SLUMINANCE_EXT,
        0 },
      { PIPE_FORMAT_L8_SRGB, DEFAULT_SRGBA_FORMATS }
   },

   /* 16-bit float formats */
   {
      { GL_RGBA16F_ARB, 0 },
      { PIPE_FORMAT_R16G16B16A16_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_RGB16F_ARB, 0 },
      { PIPE_FORMAT_R16G16B16_FLOAT, PIPE_FORMAT_R16G16B16X16_FLOAT,
        PIPE_FORMAT_R16G16B16A16_FLOAT,
        PIPE_FORMAT_R32G32B32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA16F_ARB, 0 },
      { PIPE_FORMAT_L16A16_FLOAT, PIPE_FORMAT_R16G16B16A16_FLOAT,
        PIPE_FORMAT_L32A32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_ALPHA16F_ARB, 0 },
      { PIPE_FORMAT_A16_FLOAT, PIPE_FORMAT_L16A16_FLOAT,
        PIPE_FORMAT_A32_FLOAT, PIPE_FORMAT_R16G16B16A16_FLOAT,
        PIPE_FORMAT_L32A32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_INTENSITY16F_ARB, 0 },
      { PIPE_FORMAT_I16_FLOAT, PIPE_FORMAT_L16A16_FLOAT,
        PIPE_FORMAT_I32_FLOAT, PIPE_FORMAT_R16G16B16A16_FLOAT,
        PIPE_FORMAT_L32A32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_LUMINANCE16F_ARB, 0 },
      { PIPE_FORMAT_L16_FLOAT, PIPE_FORMAT_L16A16_FLOAT,
        PIPE_FORMAT_L32_FLOAT, PIPE_FORMAT_R16G16B16A16_FLOAT,
        PIPE_FORMAT_L32A32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_R16F, 0 },
      { PIPE_FORMAT_R16_FLOAT, PIPE_FORMAT_R16G16_FLOAT,
        PIPE_FORMAT_R32_FLOAT, PIPE_FORMAT_R16G16B16A16_FLOAT,
        PIPE_FORMAT_R32G32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_RG16F, 0 },
      { PIPE_FORMAT_R16G16_FLOAT, PIPE_FORMAT_R16G16B16A16_FLOAT,
        PIPE_FORMAT_R32G32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },

   /* 32-bit float formats */
   {
      { GL_RGBA32F_ARB, 0 },
      { PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_RGB32F_ARB, 0 },
      { PIPE_FORMAT_R32G32B32_FLOAT, PIPE_FORMAT_R32G32B32X32_FLOAT,
        PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA32F_ARB, 0 },
      { PIPE_FORMAT_L32A32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_ALPHA32F_ARB, 0 },
      { PIPE_FORMAT_A32_FLOAT, PIPE_FORMAT_L32A32_FLOAT,
        PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_INTENSITY32F_ARB, 0 },
      { PIPE_FORMAT_I32_FLOAT, PIPE_FORMAT_L32A32_FLOAT,
        PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_LUMINANCE32F_ARB, 0 },
      { PIPE_FORMAT_L32_FLOAT, PIPE_FORMAT_L32A32_FLOAT,
        PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_R32F, 0 },
      { PIPE_FORMAT_R32_FLOAT, PIPE_FORMAT_R32G32_FLOAT,
        PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },
   {
      { GL_RG32F, 0 },
      { PIPE_FORMAT_R32G32_FLOAT, PIPE_FORMAT_R32G32B32A32_FLOAT, 0 }
   },

   /* R, RG formats */
   {
      { GL_RED, GL_R8, 0 },
      { PIPE_FORMAT_R8_UNORM, PIPE_FORMAT_R8G8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_RG, GL_RG8, 0 },
      { PIPE_FORMAT_R8G8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_R16, 0 },
      { PIPE_FORMAT_R16_UNORM, PIPE_FORMAT_R16G16_UNORM,
        DEFAULT_UNORM16_RGBA_FORMATS }
   },
   {
      { GL_RG16, 0 },
      { PIPE_FORMAT_R16G16_UNORM, DEFAULT_UNORM16_RGBA_FORMATS }
   },

   /* compressed R, RG formats */
   {
      { GL_COMPRESSED_RED, GL_COMPRESSED_RED_RGTC1, 0 },
      { PIPE_FORMAT_RGTC1_UNORM, PIPE_FORMAT_R8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_SIGNED_RED_RGTC1, 0 },
      { PIPE_FORMAT_RGTC1_SNORM, DEFAULT_SNORM8_RGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_RG, GL_COMPRESSED_RG_RGTC2, 0 },
      { PIPE_FORMAT_RGTC2_UNORM, PIPE_FORMAT_R8G8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_SIGNED_RG_RGTC2, 0 },
      { PIPE_FORMAT_RGTC2_SNORM, DEFAULT_SNORM8_RGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_LUMINANCE, GL_COMPRESSED_LUMINANCE_LATC1_EXT, 0 },
      { PIPE_FORMAT_LATC1_UNORM, PIPE_FORMAT_L8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_SIGNED_LUMINANCE_LATC1_EXT, 0 },
      { PIPE_FORMAT_LATC1_SNORM, DEFAULT_SNORM8_RGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_LUMINANCE_ALPHA, GL_COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT,
        GL_COMPRESSED_LUMINANCE_ALPHA_3DC_ATI, 0 },
      { PIPE_FORMAT_LATC2_UNORM, PIPE_FORMAT_L8A8_UNORM, DEFAULT_RGBA_FORMATS }
   },
   {
      { GL_COMPRESSED_SIGNED_LUMINANCE_ALPHA_LATC2_EXT, 0 },
      { PIPE_FORMAT_LATC2_SNORM, DEFAULT_SNORM8_RGBA_FORMATS }
   },

   /* ETC1 */
   {
      { GL_ETC1_RGB8_OES, 0 },
      { PIPE_FORMAT_ETC1_RGB8, 0 }
   },

   /* ETC2 */
   {
      { GL_COMPRESSED_RGB8_ETC2, 0 },
      { PIPE_FORMAT_ETC2_RGB8, 0 }
   },
   {
      { GL_COMPRESSED_SRGB8_ETC2, 0 },
      { PIPE_FORMAT_ETC2_SRGB8, 0 }
   },
   {
      { GL_COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2, 0 },
      { PIPE_FORMAT_ETC2_RGB8A1, 0 }
   },
   {
      { GL_COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2, 0 },
      { PIPE_FORMAT_ETC2_SRGB8A1, 0 }
   },
   {
      { GL_COMPRESSED_RGBA8_ETC2_EAC, 0 },
      { PIPE_FORMAT_ETC2_RGBA8, 0 }
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ETC2_EAC, 0 },
      { PIPE_FORMAT_ETC2_SRGBA8, 0 }
   },
   {
      { GL_COMPRESSED_R11_EAC, 0 },
      { PIPE_FORMAT_ETC2_R11_UNORM, 0 }
   },
   {
      { GL_COMPRESSED_SIGNED_R11_EAC, 0 },
      { PIPE_FORMAT_ETC2_R11_SNORM, 0 }
   },
   {
      { GL_COMPRESSED_RG11_EAC, 0 },
      { PIPE_FORMAT_ETC2_RG11_UNORM, 0 }
   },
   {
      { GL_COMPRESSED_SIGNED_RG11_EAC, 0 },
      { PIPE_FORMAT_ETC2_RG11_SNORM, 0 }
   },

   /* BPTC */
   {
      { GL_COMPRESSED_RGBA_BPTC_UNORM, 0 },
      { PIPE_FORMAT_BPTC_RGBA_UNORM, 0 },
   },
   {
      { GL_COMPRESSED_SRGB_ALPHA_BPTC_UNORM, 0 },
      { PIPE_FORMAT_BPTC_SRGBA, 0 },
   },
   {
      { GL_COMPRESSED_RGB_BPTC_SIGNED_FLOAT, 0 },
      { PIPE_FORMAT_BPTC_RGB_FLOAT, 0 },
   },
   {
      { GL_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT, 0 },
      { PIPE_FORMAT_BPTC_RGB_UFLOAT, 0 },
   },

   /* ASTC */
   {
      { GL_COMPRESSED_RGBA_ASTC_4x4_KHR, 0 },
      { PIPE_FORMAT_ASTC_4x4, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_5x4_KHR, 0 },
      { PIPE_FORMAT_ASTC_5x4, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_5x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_5x5, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_6x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_6x5, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_6x6_KHR, 0 },
      { PIPE_FORMAT_ASTC_6x6, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_8x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_8x5, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_8x6_KHR, 0 },
      { PIPE_FORMAT_ASTC_8x6, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_8x8_KHR, 0 },
      { PIPE_FORMAT_ASTC_8x8, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_10x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x5, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_10x6_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x6, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_10x8_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x8, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_10x10_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x10, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_12x10_KHR, 0 },
      { PIPE_FORMAT_ASTC_12x10, 0},
   },
   {
      { GL_COMPRESSED_RGBA_ASTC_12x12_KHR, 0 },
      { PIPE_FORMAT_ASTC_12x12, 0},
   },

   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_4x4_KHR, 0 },
      { PIPE_FORMAT_ASTC_4x4_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x4_KHR, 0 },
      { PIPE_FORMAT_ASTC_5x4_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_5x5_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_6x5_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x6_KHR, 0 },
      { PIPE_FORMAT_ASTC_6x6_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_8x5_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x6_KHR, 0 },
      { PIPE_FORMAT_ASTC_8x6_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x8_KHR, 0 },
      { PIPE_FORMAT_ASTC_8x8_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x5_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x5_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x6_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x6_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x8_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x8_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x10_KHR, 0 },
      { PIPE_FORMAT_ASTC_10x10_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x10_KHR, 0 },
      { PIPE_FORMAT_ASTC_12x10_SRGB, 0},
   },
   {
      { GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x12_KHR, 0 },
      { PIPE_FORMAT_ASTC_12x12_SRGB, 0},
   },

   /* signed/unsigned integer formats.
    */
   {
      { GL_RGBA_INTEGER_EXT,
        GL_BGRA_INTEGER_EXT,
        GL_RGBA8I_EXT, 0 },
      { PIPE_FORMAT_R8G8B8A8_SINT, 0 }
   },
   {
      { GL_RGB_INTEGER_EXT, 
        GL_BGR_INTEGER_EXT,
        GL_RGB8I_EXT,
        GL_BLUE_INTEGER_EXT, 0 },
      { PIPE_FORMAT_R8G8B8_SINT, PIPE_FORMAT_R8G8B8X8_SINT,
        PIPE_FORMAT_R8G8B8A8_SINT, 0 }
   },
   {
      { GL_ALPHA_INTEGER_EXT,
        GL_ALPHA8I_EXT, 0 },
      { PIPE_FORMAT_A8_SINT, 0 }
   },
   {
      { GL_ALPHA16I_EXT, 0 },
      { PIPE_FORMAT_A16_SINT, 0 }
   },
   {
      { GL_ALPHA32I_EXT, 0 },
      { PIPE_FORMAT_A32_SINT, 0 }
   },
   {
      { GL_ALPHA8UI_EXT, 0 },
      { PIPE_FORMAT_A8_UINT, 0 }
   },
   {
      { GL_ALPHA16UI_EXT, 0 },
      { PIPE_FORMAT_A16_UINT, 0 }
   },
   {
      { GL_ALPHA32UI_EXT, 0 },
      { PIPE_FORMAT_A32_UINT, 0 }
   },
   {
      { GL_INTENSITY8I_EXT, 0 },
      { PIPE_FORMAT_I8_SINT, 0 }
   },
   {
      { GL_INTENSITY16I_EXT, 0 },
      { PIPE_FORMAT_I16_SINT, 0 }
   },
   {
      { GL_INTENSITY32I_EXT, 0 },
      { PIPE_FORMAT_I32_SINT, 0 }
   },
   {
      { GL_INTENSITY8UI_EXT, 0 },
      { PIPE_FORMAT_I8_UINT, 0 }
   },
   {
      { GL_INTENSITY16UI_EXT, 0 },
      { PIPE_FORMAT_I16_UINT, 0 }
   },
   {
      { GL_INTENSITY32UI_EXT, 0 },
      { PIPE_FORMAT_I32_UINT, 0 }
   },
   {
      { GL_LUMINANCE8I_EXT, 0 },
      { PIPE_FORMAT_L8_SINT, 0 }
   },
   {
      { GL_LUMINANCE16I_EXT, 0 },
      { PIPE_FORMAT_L16_SINT, 0 }
   },
   {
      { GL_LUMINANCE32I_EXT, 0 },
      { PIPE_FORMAT_L32_SINT, 0 }
   },
   {
      { GL_LUMINANCE_INTEGER_EXT,
        GL_LUMINANCE8UI_EXT, 0 },
      { PIPE_FORMAT_L8_UINT, 0 }
   },
   {
      { GL_LUMINANCE16UI_EXT, 0 },
      { PIPE_FORMAT_L16_UINT, 0 }
   },
   {
      { GL_LUMINANCE32UI_EXT, 0 },
      { PIPE_FORMAT_L32_UINT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA_INTEGER_EXT,
        GL_LUMINANCE_ALPHA8I_EXT, 0 },
      { PIPE_FORMAT_L8A8_SINT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA16I_EXT, 0 },
      { PIPE_FORMAT_L16A16_SINT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA32I_EXT, 0 },
      { PIPE_FORMAT_L32A32_SINT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA8UI_EXT, 0 },
      { PIPE_FORMAT_L8A8_UINT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA16UI_EXT, 0 },
      { PIPE_FORMAT_L16A16_UINT, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA32UI_EXT, 0 },
      { PIPE_FORMAT_L32A32_UINT, 0 }
   },
   {
      { GL_RGB16I_EXT, 0 },
      { PIPE_FORMAT_R16G16B16_SINT, PIPE_FORMAT_R16G16B16X16_SINT,
        PIPE_FORMAT_R16G16B16A16_SINT, 0 },
   },
   {
      { GL_RGBA16I_EXT, 0 },
      { PIPE_FORMAT_R16G16B16A16_SINT, 0 },
   },
   {
      { GL_RGB32I_EXT, 0 },
      { PIPE_FORMAT_R32G32B32_SINT, PIPE_FORMAT_R32G32B32X32_SINT,
        PIPE_FORMAT_R32G32B32A32_SINT, 0 },
   },
   {
      { GL_RGBA32I_EXT, 0 },
      { PIPE_FORMAT_R32G32B32A32_SINT, 0 }
   },
   {
      { GL_RGBA8UI_EXT, 0 },
      { PIPE_FORMAT_R8G8B8A8_UINT, 0 }
   },
   {
      { GL_RGB8UI_EXT, 0 },
      { PIPE_FORMAT_R8G8B8_UINT, PIPE_FORMAT_R8G8B8X8_UINT,
        PIPE_FORMAT_R8G8B8A8_UINT, 0 }
   },
   {
      { GL_RGB16UI_EXT, 0 },
      { PIPE_FORMAT_R16G16B16_UINT, PIPE_FORMAT_R16G16B16X16_UINT,
        PIPE_FORMAT_R16G16B16A16_UINT, 0 }
   },
   {
      { GL_RGBA16UI_EXT, 0 },
      { PIPE_FORMAT_R16G16B16A16_UINT, 0 }
   },
   {
      { GL_RGB32UI_EXT, 0},
      { PIPE_FORMAT_R32G32B32_UINT, PIPE_FORMAT_R32G32B32X32_UINT,
        PIPE_FORMAT_R32G32B32A32_UINT, 0 }
   },
   {
      { GL_RGBA32UI_EXT, 0},
      { PIPE_FORMAT_R32G32B32A32_UINT, 0 }
   },
   {
     { GL_R8I, GL_RED_INTEGER_EXT, 0},
     { PIPE_FORMAT_R8_SINT, 0},
   },
   {
     { GL_R16I, 0},
     { PIPE_FORMAT_R16_SINT, 0},
   },
   {
     { GL_R32I, 0},
     { PIPE_FORMAT_R32_SINT, 0},
   },
  {
     { GL_R8UI, 0},
     { PIPE_FORMAT_R8_UINT, 0},
   },
   {
     { GL_R16UI, 0},
     { PIPE_FORMAT_R16_UINT, 0},
   },
   {
     { GL_R32UI, 0},
     { PIPE_FORMAT_R32_UINT, 0},
   },
   {
     { GL_RG8I, GL_GREEN_INTEGER_EXT, 0},
     { PIPE_FORMAT_R8G8_SINT, 0},
   },
   {
     { GL_RG16I, 0},
     { PIPE_FORMAT_R16G16_SINT, 0},
   },
   {
     { GL_RG32I, 0},
     { PIPE_FORMAT_R32G32_SINT, 0},
   },
  {
     { GL_RG8UI, 0},
     { PIPE_FORMAT_R8G8_UINT, 0},
   },
   {
     { GL_RG16UI, 0},
     { PIPE_FORMAT_R16G16_UINT, 0},
   },
   {
     { GL_RG32UI, 0},
     { PIPE_FORMAT_R32G32_UINT, 0},
   },
   /* signed normalized formats */
   {
      { GL_RED_SNORM, GL_R8_SNORM, 0 },
      { PIPE_FORMAT_R8_SNORM, PIPE_FORMAT_R8G8_SNORM,
        PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_R16_SNORM, 0 },
      { PIPE_FORMAT_R16_SNORM,
        PIPE_FORMAT_R16G16_SNORM,
        PIPE_FORMAT_R16G16B16A16_SNORM,
        PIPE_FORMAT_R8_SNORM,
        PIPE_FORMAT_R8G8_SNORM,
        PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_RG_SNORM, GL_RG8_SNORM, 0 },
      { PIPE_FORMAT_R8G8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_RG16_SNORM, 0 },
      { PIPE_FORMAT_R16G16_SNORM, PIPE_FORMAT_R16G16B16A16_SNORM,
        PIPE_FORMAT_R8G8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_RGB_SNORM, GL_RGB8_SNORM, 0 },
      { PIPE_FORMAT_R8G8B8X8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_RGBA_SNORM, GL_RGBA8_SNORM, 0 },
      { PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_RGB16_SNORM, 0 },
      { PIPE_FORMAT_R16G16B16X16_SNORM, PIPE_FORMAT_R16G16B16A16_SNORM,
        PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_RGBA16_SNORM, 0 },
      { PIPE_FORMAT_R16G16B16A16_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_ALPHA_SNORM, GL_ALPHA8_SNORM, 0 },
      { PIPE_FORMAT_A8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_ALPHA16_SNORM, 0 },
      { PIPE_FORMAT_A16_SNORM, PIPE_FORMAT_R16G16B16A16_SNORM,
        PIPE_FORMAT_A8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_LUMINANCE_SNORM, GL_LUMINANCE8_SNORM, 0 },
      { PIPE_FORMAT_L8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_LUMINANCE16_SNORM, 0 },
      { PIPE_FORMAT_L16_SNORM, PIPE_FORMAT_R16G16B16A16_SNORM,
        PIPE_FORMAT_L8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_LUMINANCE_ALPHA_SNORM, GL_LUMINANCE8_ALPHA8_SNORM, 0 },
      { PIPE_FORMAT_L8A8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_LUMINANCE16_ALPHA16_SNORM, 0 },
      { PIPE_FORMAT_L16A16_SNORM, PIPE_FORMAT_R16G16B16A16_SNORM,
        PIPE_FORMAT_L8A8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_INTENSITY_SNORM, GL_INTENSITY8_SNORM, 0 },
      { PIPE_FORMAT_I8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_INTENSITY16_SNORM, 0 },
      { PIPE_FORMAT_I16_SNORM, PIPE_FORMAT_R16G16B16A16_SNORM,
        PIPE_FORMAT_I8_SNORM, PIPE_FORMAT_R8G8B8A8_SNORM, 0 }
   },
   {
      { GL_RGB9_E5, 0 },
      { PIPE_FORMAT_R9G9B9E5_FLOAT, 0 }
   },
   {
      { GL_R11F_G11F_B10F, 0 },
      { PIPE_FORMAT_R11G11B10_FLOAT, 0 }
   },
   {
      { GL_RGB10_A2UI, 0 },
      { PIPE_FORMAT_R10G10B10A2_UINT, PIPE_FORMAT_B10G10R10A2_UINT, 0 }
   },
};


/**
 * Return first supported format from the given list.
 * \param allow_dxt  indicates whether it's OK to return a DXT format.
 */
static enum pipe_format
find_supported_format(struct pipe_screen *screen,
                      const enum pipe_format formats[],
                      enum pipe_texture_target target,
                      unsigned sample_count,
                      unsigned tex_usage,
                      boolean allow_dxt)
{
   uint i;
   for (i = 0; formats[i]; i++) {
      if (screen->is_format_supported(screen, formats[i], target,
                                      sample_count, tex_usage)) {
         if (!allow_dxt && util_format_is_s3tc(formats[i])) {
            /* we can't return a dxt format, continue searching */
            continue;
         }

         return formats[i];
      }
   }
   return PIPE_FORMAT_NONE;
}

struct exact_format_mapping
{
   GLenum format;
   GLenum type;
   enum pipe_format pformat;
};

static const struct exact_format_mapping rgba8888_tbl[] =
{
   { GL_RGBA,     GL_UNSIGNED_INT_8_8_8_8,        PIPE_FORMAT_ABGR8888_UNORM },
   { GL_ABGR_EXT, GL_UNSIGNED_INT_8_8_8_8_REV,    PIPE_FORMAT_ABGR8888_UNORM },
   { GL_RGBA,     GL_UNSIGNED_INT_8_8_8_8_REV,    PIPE_FORMAT_RGBA8888_UNORM },
   { GL_ABGR_EXT, GL_UNSIGNED_INT_8_8_8_8,        PIPE_FORMAT_RGBA8888_UNORM },
   { GL_BGRA,     GL_UNSIGNED_INT_8_8_8_8,        PIPE_FORMAT_ARGB8888_UNORM },
   { GL_BGRA,     GL_UNSIGNED_INT_8_8_8_8_REV,    PIPE_FORMAT_BGRA8888_UNORM },
   { GL_RGBA,     GL_UNSIGNED_BYTE,               PIPE_FORMAT_R8G8B8A8_UNORM },
   { GL_ABGR_EXT, GL_UNSIGNED_BYTE,               PIPE_FORMAT_A8B8G8R8_UNORM },
   { GL_BGRA,     GL_UNSIGNED_BYTE,               PIPE_FORMAT_B8G8R8A8_UNORM },
   { 0,           0,                              0                          }
};

static const struct exact_format_mapping rgbx8888_tbl[] =
{
   { GL_RGBA,     GL_UNSIGNED_INT_8_8_8_8,        PIPE_FORMAT_XBGR8888_UNORM },
   { GL_ABGR_EXT, GL_UNSIGNED_INT_8_8_8_8_REV,    PIPE_FORMAT_XBGR8888_UNORM },
   { GL_RGBA,     GL_UNSIGNED_INT_8_8_8_8_REV,    PIPE_FORMAT_RGBX8888_UNORM },
   { GL_ABGR_EXT, GL_UNSIGNED_INT_8_8_8_8,        PIPE_FORMAT_RGBX8888_UNORM },
   { GL_BGRA,     GL_UNSIGNED_INT_8_8_8_8,        PIPE_FORMAT_XRGB8888_UNORM },
   { GL_BGRA,     GL_UNSIGNED_INT_8_8_8_8_REV,    PIPE_FORMAT_BGRX8888_UNORM },
   { GL_RGBA,     GL_UNSIGNED_BYTE,               PIPE_FORMAT_R8G8B8X8_UNORM },
   { GL_ABGR_EXT, GL_UNSIGNED_BYTE,               PIPE_FORMAT_X8B8G8R8_UNORM },
   { GL_BGRA,     GL_UNSIGNED_BYTE,               PIPE_FORMAT_B8G8R8X8_UNORM },
   { 0,           0,                              0                          }
};

/**
 * For unsized/base internal formats, we may choose a convenient effective
 * internal format for {format, type}. If one exists, return that, otherwise
 * return PIPE_FORMAT_NONE.
 */
static enum pipe_format
find_exact_format(GLint internalFormat, GLenum format, GLenum type)
{
   uint i;
   const struct exact_format_mapping* tbl;

   if (format == GL_NONE || type == GL_NONE)
      return PIPE_FORMAT_NONE;

   switch (internalFormat) {
   case 4:
   case GL_RGBA:
      tbl = rgba8888_tbl;
      break;
   case 3:
   case GL_RGB:
      tbl = rgbx8888_tbl;
      break;
   default:
      return PIPE_FORMAT_NONE;
   }

   for (i = 0; tbl[i].format; i++)
      if (tbl[i].format == format && tbl[i].type == type)
         return tbl[i].pformat;

   return PIPE_FORMAT_NONE;
}

/**
 * Given an OpenGL internalFormat value for a texture or surface, return
 * the best matching PIPE_FORMAT_x, or PIPE_FORMAT_NONE if there's no match.
 * This is called during glTexImage2D, for example.
 *
 * The bindings parameter typically has PIPE_BIND_SAMPLER_VIEW set, plus
 * either PIPE_BINDING_RENDER_TARGET or PIPE_BINDING_DEPTH_STENCIL if
 * we want render-to-texture ability.
 *
 * \param internalFormat  the user value passed to glTexImage2D
 * \param target  one of PIPE_TEXTURE_x
 * \param bindings  bitmask of PIPE_BIND_x flags.
 * \param allow_dxt  indicates whether it's OK to return a DXT format.  This
 *                   only matters when internalFormat names a generic or
 *                   specific compressed format.  And that should only happen
 *                   when we're getting called from gl[Copy]TexImage().
 */
enum pipe_format
st_choose_format(struct st_context *st, GLenum internalFormat,
                 GLenum format, GLenum type,
                 enum pipe_texture_target target, unsigned sample_count,
                 unsigned bindings, boolean allow_dxt)
{
   struct pipe_screen *screen = st->pipe->screen;
   unsigned i;
   int j;
   enum pipe_format pf;

#ifdef DEBUG
   {
      static boolean firstCall = TRUE;
      if (firstCall) {
         test_format_conversion(st);
         firstCall = FALSE;
      }
   }
#else
   (void) test_format_conversion;
#endif

   /* can't render to compressed formats at this time */
   if (_mesa_is_compressed_format(st->ctx, internalFormat)
       && (bindings & ~PIPE_BIND_SAMPLER_VIEW)) {
      return PIPE_FORMAT_NONE;
   }

   /* search for exact matches */
   pf = find_exact_format(internalFormat, format, type);
   if (pf != PIPE_FORMAT_NONE &&
       screen->is_format_supported(screen, pf,
                                   target, sample_count, bindings))
      return pf;

   /* search table for internalFormat */
   for (i = 0; i < ARRAY_SIZE(format_map); i++) {
      const struct format_mapping *mapping = &format_map[i];
      for (j = 0; mapping->glFormats[j]; j++) {
         if (mapping->glFormats[j] == internalFormat) {
            /* Found the desired internal format.  Find first pipe format
             * which is supported by the driver.
             */
            return find_supported_format(screen, mapping->pipeFormats,
                                         target, sample_count, bindings,
                                         allow_dxt);
         }
      }
   }

   _mesa_problem(NULL, "unhandled format!\n");
   return PIPE_FORMAT_NONE;
}


/**
 * Called by FBO code to choose a PIPE_FORMAT_ for drawing surfaces.
 */
enum pipe_format
st_choose_renderbuffer_format(struct st_context *st,
                              GLenum internalFormat, unsigned sample_count)
{
   uint usage;
   if (_mesa_is_depth_or_stencil_format(internalFormat))
      usage = PIPE_BIND_DEPTH_STENCIL;
   else
      usage = PIPE_BIND_RENDER_TARGET;
   return st_choose_format(st, internalFormat, GL_NONE, GL_NONE,
                           PIPE_TEXTURE_2D, sample_count, usage, FALSE);
}


/**
 * Given an OpenGL user-requested format and type, and swapBytes state,
 * return the format which exactly matches those parameters, so that
 * a memcpy-based transfer can be done.
 *
 * If no format is supported, return PIPE_FORMAT_NONE.
 */
enum pipe_format
st_choose_matching_format(struct st_context *st, unsigned bind,
			  GLenum format, GLenum type, GLboolean swapBytes)
{
   struct pipe_screen *screen = st->pipe->screen;
   mesa_format mesa_format;

   for (mesa_format = 1; mesa_format < MESA_FORMAT_COUNT; mesa_format++) {
      if (_mesa_get_format_color_encoding(mesa_format) == GL_SRGB) {
         continue;
      }
      if (_mesa_get_format_bits(mesa_format, GL_TEXTURE_INTENSITY_SIZE) > 0) {
         /* If `format` is GL_RED/GL_RED_INTEGER, then we might match some
          * intensity formats, which we don't want.
          */
         continue;
      }

      if (_mesa_format_matches_format_and_type(mesa_format, format, type,
                                               swapBytes, NULL)) {
         enum pipe_format format =
            st_mesa_format_to_pipe_format(st, mesa_format);

         if (format &&
             screen->is_format_supported(screen, format, PIPE_TEXTURE_2D, 0,
                                         bind)) {
            return format;
         }
         /* It's unlikely to find 2 matching Mesa formats. */
         break;
      }
   }
   return PIPE_FORMAT_NONE;
}


/**
 * Called via ctx->Driver.ChooseTextureFormat().
 */
mesa_format
st_ChooseTextureFormat(struct gl_context *ctx, GLenum target,
                       GLint internalFormat,
                       GLenum format, GLenum type)
{
   struct st_context *st = st_context(ctx);
   enum pipe_format pFormat;
   mesa_format mFormat;
   unsigned bindings;
   bool is_renderbuffer = false;
   enum pipe_texture_target pTarget;

   if (target == GL_RENDERBUFFER) {
      pTarget = PIPE_TEXTURE_2D;
      is_renderbuffer = true;
   } else {
      pTarget = gl_target_to_pipe(target);
   }

   if (target == GL_TEXTURE_1D || target == GL_TEXTURE_1D_ARRAY) {
      /* We don't do compression for these texture targets because of
       * difficulty with sub-texture updates on non-block boundaries, etc.
       * So change the internal format request to an uncompressed format.
       */
      internalFormat =
        _mesa_generic_compressed_format_to_uncompressed_format(internalFormat);
   }

   /* GL textures may wind up being render targets, but we don't know
    * that in advance.  Specify potential render target flags now for formats
    * that we know should always be renderable.
    */
   bindings = PIPE_BIND_SAMPLER_VIEW;
   if (_mesa_is_depth_or_stencil_format(internalFormat))
      bindings |= PIPE_BIND_DEPTH_STENCIL;
   else if (is_renderbuffer || internalFormat == 3 || internalFormat == 4 ||
            internalFormat == GL_RGB || internalFormat == GL_RGBA ||
            internalFormat == GL_RGB8 || internalFormat == GL_RGBA8 ||
            internalFormat == GL_BGRA ||
            internalFormat == GL_RGB16F ||
            internalFormat == GL_RGBA16F ||
            internalFormat == GL_RGB32F ||
            internalFormat == GL_RGBA32F)
	 bindings |= PIPE_BIND_RENDER_TARGET;

   /* GLES allows the driver to choose any format which matches
    * the format+type combo, because GLES only supports unsized internal
    * formats and expects the driver to choose whatever suits it.
    */
   if (_mesa_is_gles(ctx)) {
      GLenum baseFormat = _mesa_base_tex_format(ctx, internalFormat);
      GLenum basePackFormat = _mesa_base_pack_format(format);
      GLenum iformat = internalFormat;

      /* Treat GL_BGRA as GL_RGBA. */
      if (iformat == GL_BGRA)
         iformat = GL_RGBA;

      /* Check if the internalformat is unsized and compatible
       * with the "format".
       */
      if (iformat == baseFormat && iformat == basePackFormat) {
         pFormat = st_choose_matching_format(st, bindings, format, type,
                                             ctx->Unpack.SwapBytes);

         if (pFormat != PIPE_FORMAT_NONE)
            return st_pipe_format_to_mesa_format(pFormat);

         if (!is_renderbuffer) {
            /* try choosing format again, this time without render target bindings */
            pFormat = st_choose_matching_format(st, PIPE_BIND_SAMPLER_VIEW,
                                                format, type,
                                                ctx->Unpack.SwapBytes);
            if (pFormat != PIPE_FORMAT_NONE)
               return st_pipe_format_to_mesa_format(pFormat);
         }
      }
   }

   pFormat = st_choose_format(st, internalFormat, format, type,
                              pTarget, 0, bindings, ctx->Mesa_DXTn);

   if (pFormat == PIPE_FORMAT_NONE && !is_renderbuffer) {
      /* try choosing format again, this time without render target bindings */
      pFormat = st_choose_format(st, internalFormat, format, type,
                                 pTarget, 0, PIPE_BIND_SAMPLER_VIEW,
                                 ctx->Mesa_DXTn);
   }

   if (pFormat == PIPE_FORMAT_NONE) {
      /* lie about using etc1/etc2 natively if we do decoding tricks */
      mFormat = _mesa_glenum_to_compressed_format(internalFormat);
      if ((mFormat == MESA_FORMAT_ETC1_RGB8 && !st->has_etc1) ||
          (_mesa_is_format_etc2(mFormat) && !st->has_etc2))
          return mFormat;

      /* no luck at all */
      return MESA_FORMAT_NONE;
   }

   mFormat = st_pipe_format_to_mesa_format(pFormat);

   /* Debugging aid */
   if (0) {
      debug_printf("%s(intFormat=%s, format=%s, type=%s) -> %s, %s\n",
                   __func__,
                   _mesa_enum_to_string(internalFormat),
                   _mesa_enum_to_string(format),
                   _mesa_enum_to_string(type),
                   util_format_name(pFormat),
                   _mesa_get_format_name(mFormat));
   }

   return mFormat;
}


/**
 * Called via ctx->Driver.QueryInternalFormat().
 */
static size_t
st_QuerySamplesForFormat(struct gl_context *ctx, GLenum target,
                         GLenum internalFormat, int samples[16])
{
   struct st_context *st = st_context(ctx);
   enum pipe_format format;
   unsigned i, bind, num_sample_counts = 0;

   (void) target;

   if (_mesa_is_depth_or_stencil_format(internalFormat))
      bind = PIPE_BIND_DEPTH_STENCIL;
   else
      bind = PIPE_BIND_RENDER_TARGET;

   /* If an sRGB framebuffer is unsupported, sRGB formats behave like linear
    * formats.
    */
   if (!ctx->Extensions.EXT_framebuffer_sRGB) {
      internalFormat = _mesa_get_linear_internalformat(internalFormat);
   }

   /* Set sample counts in descending order. */
   for (i = 16; i > 1; i--) {
      format = st_choose_format(st, internalFormat, GL_NONE, GL_NONE,
                                PIPE_TEXTURE_2D, i, bind, FALSE);

      if (format != PIPE_FORMAT_NONE) {
         samples[num_sample_counts++] = i;
      }
   }

   if (!num_sample_counts) {
      samples[num_sample_counts++] = 1;
   }

   return num_sample_counts;
}

/**
 * ARB_internalformat_query2 driver hook.
 */
void
st_QueryInternalFormat(struct gl_context *ctx, GLenum target,
                       GLenum internalFormat, GLenum pname, GLint *params)
{
   struct st_context *st = st_context(ctx);
   /* The API entry-point gives us a temporary params buffer that is non-NULL
    * and guaranteed to have at least 16 elements.
    */
   assert(params != NULL);

   switch (pname) {
   case GL_SAMPLES:
      st_QuerySamplesForFormat(ctx, target, internalFormat, params);
      break;

   case GL_NUM_SAMPLE_COUNTS: {
      size_t num_samples;
      num_samples = st_QuerySamplesForFormat(ctx, target, internalFormat,
                                             params);
      params[0] = (GLint) num_samples;
      break;
   }
   case GL_INTERNALFORMAT_PREFERRED: {
      params[0] = GL_NONE;

      /* We need to resolve an internal format that is compatible with
       * the passed internal format, and optimal to the driver. By now,
       * we just validate that the passed internal format is supported by
       * the driver, and if so return the same internal format, otherwise
       * return GL_NONE.
       */
      uint usage;
      if (_mesa_is_depth_or_stencil_format(internalFormat))
         usage = PIPE_BIND_DEPTH_STENCIL;
      else
         usage = PIPE_BIND_RENDER_TARGET;
      enum pipe_format pformat = st_choose_format(st,
                                                  internalFormat,
                                                  GL_NONE,
                                                  GL_NONE,
                                                  PIPE_TEXTURE_2D, 1,
                                                  usage, FALSE);
      if (pformat)
         params[0] = internalFormat;
      break;
   }
   default:
      /* For the rest of the pnames, we call back the Mesa's default
       * function for drivers that don't implement ARB_internalformat_query2.
       */
      _mesa_query_internal_format_default(ctx, target, internalFormat, pname,
                                          params);
   }
}

/**
 * This is used for translating texture border color and the clear
 * color.  For example, the clear color is interpreted according to
 * the renderbuffer's base format.  For example, if clearing a
 * GL_LUMINANCE buffer, we'll return colorOut[0] = colorOut[1] =
 * colorOut[2] = colorIn[0].
 * Similarly for texture border colors.
 */
void
st_translate_color(const union gl_color_union *colorIn,
                   union pipe_color_union *colorOut,
                   GLenum baseFormat, GLboolean is_integer)
{
   if (is_integer) {
      const int *in = colorIn->i;
      int *out = colorOut->i;

      switch (baseFormat) {
      case GL_RED:
         out[0] = in[0];
         out[1] = 0;
         out[2] = 0;
         out[3] = 1;
         break;
      case GL_RG:
         out[0] = in[0];
         out[1] = in[1];
         out[2] = 0;
         out[3] = 1;
         break;
      case GL_RGB:
         out[0] = in[0];
         out[1] = in[1];
         out[2] = in[2];
         out[3] = 1;
         break;
      case GL_ALPHA:
         out[0] = out[1] = out[2] = 0;
         out[3] = in[3];
         break;
      case GL_LUMINANCE:
         out[0] = out[1] = out[2] = in[0];
         out[3] = 1;
         break;
      case GL_LUMINANCE_ALPHA:
         out[0] = out[1] = out[2] = in[0];
         out[3] = in[3];
         break;
      case GL_INTENSITY:
         out[0] = out[1] = out[2] = out[3] = in[0];
         break;
      default:
         COPY_4V(out, in);
      }
   }
   else {
      const float *in = colorIn->f;
      float *out = colorOut->f;

      switch (baseFormat) {
      case GL_RED:
         out[0] = in[0];
         out[1] = 0.0F;
         out[2] = 0.0F;
         out[3] = 1.0F;
         break;
      case GL_RG:
         out[0] = in[0];
         out[1] = in[1];
         out[2] = 0.0F;
         out[3] = 1.0F;
         break;
      case GL_RGB:
         out[0] = in[0];
         out[1] = in[1];
         out[2] = in[2];
         out[3] = 1.0F;
         break;
      case GL_ALPHA:
         out[0] = out[1] = out[2] = 0.0F;
         out[3] = in[3];
         break;
      case GL_LUMINANCE:
         out[0] = out[1] = out[2] = in[0];
         out[3] = 1.0F;
         break;
      case GL_LUMINANCE_ALPHA:
         out[0] = out[1] = out[2] = in[0];
         out[3] = in[3];
         break;
      case GL_INTENSITY:
         out[0] = out[1] = out[2] = out[3] = in[0];
         break;
      default:
         COPY_4V(out, in);
      }
   }
}
