/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma;

import java.util.List;
import java.util.Set;

import org.videolan.vlma.model.Media;
import org.videolan.vlma.model.Order;
import org.videolan.vlma.model.Satellite;
import org.videolan.vlma.model.Server;

/**
 * VLMaService interface.
 * This interface provides a simple way to control the application data (servers,
 * adapters, media, programs, etc.).
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 * @author Adrien Grand <jpountz at videolan.org>
 */
public interface VLMaService {

    /**
     * @see org.videolan.vlma.Data#getSatellites()
     */
    List<Satellite> getSatellites();

    /**
     * @see org.videolan.vlma.Data#getServers()
     */
    List<Server> getServers();

    /**
     * @see org.videolan.vlma.Data#getMedias()
     */
    List<Media> getMedias();

    /**
     * Returns the set of orders that should be executed.
     */
    Set<Order> getOrders();

    /**
     * Returns the orders to cancel.
     *
     * @return the orders to cancel
     */
    Set<Order> getOrdersToCancel();

    /**
     * Cancel an order.
     *
     * @param order the order to cancel
     */
    void cancelOrder(Order order);

    /**
     * @see org.videolan.vlma.Data#giveOrders()
     */
    void giveOrders();

    /**
     * @see org.videolan.vlma.Data#startOrderMonitoring()
     */
    void startOrderMonitoring();

    /**
     * @see org.videolan.vlma.Data#startCheckAllVLCs()
     */
    void startCheckAllVLCs();

    /**
     * Sends a notification
     *
     * @param message the message to send
     */
    void sendNotification(String message);

    /**
     * Updates data using SNMP.
     *
     * @param server the server to query
     */
    void updateSnmpData(Server server);

    /**
     * Updates the RRD graph.
     *
     * @param server the server whose graph must be updated
     */
    void updateRrdGraph(Server server);

    /**
     * Checks if the VLM telnet interface is available.
     *
     * @param server the server to check
     * @return true if and only if the server is up
     */
    boolean checkVLC(Server server);

}
