/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.model;

import java.io.Serializable;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * VLMa streaming strategy.
 *
 * @author Adrien Grand <jpountz at videolan.org>
 */
public class StreamingStrategy implements Serializable {

    private static final long serialVersionUID = -75919931677731393L;

    public static enum Protocol {
        UDP_MULTICAST, HTTP;
    }

    public static enum Announcement {
        SAP, PODCAST, M3U;
    }

    public static enum Encapsulation {
        TS, PS, MPEG1, OGG, ASF, ASFH, AVI, MPJPEG;
    }

    private static final Protocol DEFAULT_PROTOCOL = Protocol.UDP_MULTICAST;
    private static final Encapsulation DEFAULT_ENCAPSULATION = Encapsulation.TS;

    private Protocol protocol = DEFAULT_PROTOCOL;
    private Set<Announcement> announcements = new HashSet<Announcement>();
    private Encapsulation encapsulation = DEFAULT_ENCAPSULATION;

    /**
     * Gets the announcement list.
     *
     * @return the announcements
     */
    public Set<Announcement> getAnnouncements() {
        return announcements;
    }

    /**
     * Adds an announcement.
     *
     * @param announcement the announcement to add
     */
    public void addAnnouncement(Announcement announcement) {
        this.announcements.add(announcement);
    }

    /**
     * Verifies whether or not an announcement is enabled.
     *
     * @return true if and only if the announcement is enabled
     */
    public boolean isEnabled(Announcement announcement) {
        return announcements.contains(announcement);
    }

    /**
     * Sets the announcements.
     *
     * @param announcements the announcements to set
     */
    public void setAnnouncements(Collection<Announcement> announcements) {
        this.announcements = new HashSet<Announcement>();
        this.announcements.addAll(announcements);
    }

    /**
     * Gets the protocol.
     *
     * @return the protocol
     */
    public Protocol getProtocol() {
        return protocol;
    }

    /**
     * Sets the protocol.
     *
     * @param protocol the protocol to set
     */
    public void setProtocol(Protocol protocol) {
        this.protocol = protocol;
    }

    /**
     * Gets the encapsulation.
     *
     * @return the encapsulation
     */
    public Encapsulation getEncapsulation() {
        return encapsulation;
    }

    /**
     * Sets the encapsulation.
     *
     * @param encapsulation the encapsulation to set
     */
    public void setEncapsulation(Encapsulation encapsulation) {
        this.encapsulation = encapsulation;
    }

}
