#pragma once
#include "Utils/TypeInfo.h"
#include "Utils/Lock.h"
#include "PtrThrowable.h"
#include "InlineList.h"

// In Visual Studio 2008, we do not have exception_ptr, so we implement our own
// solution in that case!
#if defined(VISUAL_STUDIO) && VISUAL_STUDIO <= 2008
#define CUSTOM_EXCEPTION_PTR
#else
#include <stdexcept>
#endif

namespace os {
	class UThreadData;

#ifdef CUSTOM_EXCEPTION_PTR
	struct CppExceptionType;
#endif

	/**
	 * Mode for futures:
	 */
	struct FutureMode {
		// There are two modes in which the Future can be used in respect to how they interact with
		// other UThreads on the current OS thread.
		enum Mode {
			// The normal mode, when this future is waiting, any other UThread may be scheduled.
			normal,
			// The exclusive mode. This means that when this future is waiting for a result, no
			// other UThreads are allowed to run on the current OS thread. The exception is other
			// threads that are spawned to post a result to an exclusive future, or their
			// descendants. This is used in the lazy compilation to not break the threading
			// guarantees provided by Storm (i.e. function calls do not cause a thread switch by
			// themselves).
			exclusive,
		};
	};


	/**
	 * Future that stores the result in some unrelated memory. This class is designed to be possible
	 * to use with minimal type information and to be easy to use from machine code or general
	 * non-templated C++ code. Therefore it is not very type-safe at all. For a more type-safe and
	 * user-friendly version, see the Future<> class below.
	 */
	class FutureBase : NoCopy {
	public:
		// Create the future. Indicate a priority level.
		FutureBase(FutureMode::Mode mode = FutureMode::normal);

		// Detect unhandled exceptions and print them.
		~FutureBase();

		// Wait for the result. This function will either return normally, indicating the result is
		// written to 'target', or throw an exception posted.
		void result();

		// Same as above, but provides an opportunity to modify any pointer-based exception before
		// it is thrown.
		typedef PtrThrowable *(*InterceptFn)(PtrThrowable *original, void *env);
		void result(InterceptFn fn, void *env);

		// Detach this exception, ie. don't complain about uncaught errors.
		void detach();

		// Tell the waiting thread we have posted a result.
		void posted();

		// Post an error. This function must be called from within a throw-catch block. Otherwise
		// the runtime crashes.
		void error();

		// Has any result been posted?
		bool dataPosted();

		// Any result (either data or error) posted?
		bool anyPosted();

		// Check if the future is in exclusive mode.
		inline bool exclusive() const { return mode == FutureMode::exclusive; }

		// Pointer-based exceptions are stored here so that they can be garbage collected properly.
		PtrThrowable *ptrException;

	private:

		// Throw the error captured earlier.
		void throwError();

		// Throw the error captured earlier, knowing it is a pointer.
		void throwPtrError(InterceptFn fn, void *env);

		// Save the current exception.
		void saveError();

		// Save the current exception, knowing it is a pointer.
		void savePtrError(const PtrThrowable *exception);

		// Perform any cleanup needed.
		void cleanError();

#ifdef CUSTOM_EXCEPTION_PTR
		class ExceptionPtr {
		public:
			ExceptionPtr();
			~ExceptionPtr();

			void *data;
			const CppExceptionType *type;

			// Only used on 64-bit systems.
			HINSTANCE instance;

			// Rethrow this exception.
			void rethrow();
			void rethrowPtr(PtrThrowable *&ptr);

			// Clear the data here.
			void clear();

			// Copy from an exception.
			void set(void *from, const CppExceptionType *type, HINSTANCE instance);

			// Copy a pointer from an exception.
			void setPtr(void *from, const CppExceptionType *type, HINSTANCE instance, PtrThrowable *&store);
		};

		// The exception.
		ExceptionPtr exceptionData;

		// Filter expression function.
		int filter(EXCEPTION_POINTERS *ptrs, bool pointer);

#elif defined(POSIX)

		// Exception data. Either an std::exception ptr (if it was a C++ exception), or a
		// std::type_info * if it was a pointer exception.
		size_t exceptionData[sizeof(std::exception_ptr) / sizeof(size_t)];

#else
		int filter(EXCEPTION_POINTERS *ptrs, InterceptFn fn, void *env);

		// Regular implementation, we use filter functions to modify the data.
		std::exception_ptr exceptionData;

#endif

		// Any error?
		bool hasError() const {
			nat r = atomicRead(resultPosted);
			return r == resultError || r == resultErrorPtr;
		}

		// What kind of result was posted?
		enum {
			// No result.
			resultEmpty,

			// A regular value.
			resultValue,

			// A generic C++ exception.
			resultError,

			// A pointer to PtrThrowable.
			resultErrorPtr,
		};

		// Anything posted? Any of 'result*'
		nat resultPosted;

		// Did we read the result?
		enum {
			// Not read.
			readNone,

			// Read at least once.
			readOnce,

			// Detached. i.e. will not be read.
			readDetached,
		};

		// Anything read?
		nat resultRead;

		// Mode.
		FutureMode::Mode mode;

		// Warn about errors from detached threads.
		void warnDetachedError();

	protected:
		// Notify/wait operations. These were previously in a separate subclass to allow swapping
		// which semaphore to use. Now, we implement our own event-like structure to allow custom
		// waiting behavior and avoid circular dependencies (sema depends on UThread, UThread
		// depends on us). We still need to let other classes intercept these operations though.
		virtual void notify();
		virtual void wait();

	private:
		// Threads waiting for this future.
		InlineList<UThreadData> waitingThreads;

		// Has 'notify' been called?
		bool notified;

		// Lock for 'waitingThreads' and 'notified'.
		util::Lock lock;
	};


	/**
	 * A real C++ friendly implementation of the future. Defaults to use the
	 * semaphore with support for background threads. This class does not
	 * inherit from the FutureBase<> class to provide a cleaner interface
	 * to the user. The underlying FutureBase<> object may still be queried
	 * if neccessary.
	 * This class assumes that you will call 'result' at least once.
	 */
	template <class T>
	class Future : NoCopy {
	public:
		// Create, optionally specify a mode.
		Future() {}
		Future(FutureMode::Mode mode) : future(mode) {}

		// Destroy.
		~Future() {
			if (future.dataPosted()) {
				void *v = value;
				((T *)v)->~T();
			}
		}

		// Get the result or throw the error, when the thread is ready.
		T result() {
			future.result();
			void *v = value;
			return *(T *)v;
		}

		// Get the result, specify a intercept function.
		T result(FutureBase::InterceptFn fn, void *env) {
			future.result(fn, env);
			void *v = value;
			return *(T *)v;
		}

		// Post the result.
		void post(const T &result) {
			new (value) T(result);
			future.posted();
		}

		// Post an error. Only call from a catch-block.
		void error() {
			future.error();
		}

		// Get the underlying Future object.
		FutureBase &impl() {
			return future;
		}

		// Get our data.
		void *data() {
			return value;
		}

	private:
		// Underlying object.
		FutureBase future;

		// Result storage. Note that we do not want to initialize it before
		// 'future' tries to return something into this.
		byte value[sizeof(T)];
	};

	/**
	 * Special case for void values (exceptions and syncronization may still be interesting!).
	 */
	template <>
	class Future<void> : NoCopy {
	public:
		// Create, optionally specify a mode.
		Future() {}
		Future(FutureMode::Mode mode) : future(mode) {}

		// Destroy.
		~Future() {}

		// Get the result or throw the error, when the thread is ready.
		void result() {
			future.result();
		}

		// Get the result, specify a intercept function.
		void result(FutureBase::InterceptFn fn, void *env) {
			future.result(fn, env);
		}

		// Post the result.
		void post() {
			future.posted();
		}

		// Post an error. Only call from a catch-block.
		void error() {
			future.error();
		}

		// Get the underlying Future object.
		FutureBase &impl() {
			return future;
		}

		// Get our data.
		void *data() {
			return null;
		}

	private:
		// Underlying object.
		FutureBase future;
	};

}
