/************************************* */
/* Rule Set Based Access Control       */
/* Author and (c) 1999-2008:           */
/*   Amon Ott <ao@rsbac.org>           */
/* Syscall wrapper functions for all   */
/* admin tools                         */
/* Last modified: 03/Mar/2008          */
/************************************* */

#include <rsbac/types.h>
#include <rsbac/getname.h>
#include <rsbac/helpers.h>
#include <rsbac/error.h>
#include <rsbac/syscall_rsbac.h>
#include <rsbac/syscalls.h>

#include "nls.h"

int rsbac_version(void)
{
	return sys_rsbac(RSBAC_VERSION_NR, RSYS_version, NULL);
}

int rsbac_stats(void)
  {
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_stats, NULL);
  }

int rsbac_check(int correct, int check_inode)
  {
    union rsbac_syscall_arg_t arg;

    arg.check.correct = correct;
    arg.check.check_inode = check_inode;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_check, &arg);
  }

int rsbac_write(void)
  {
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_write, NULL);
  }

int rsbac_get_attr(
                        rsbac_list_ta_number_t ta_number,
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   union rsbac_target_id_t * tid,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value,
                   int inherit)
  { 
    union rsbac_syscall_arg_t arg;

    arg.get_attr.ta_number = ta_number;
    arg.get_attr.module = module;
    arg.get_attr.target = target;
    arg.get_attr.tid = tid;
    arg.get_attr.attr = attr;
    arg.get_attr.value = value;
    arg.get_attr.inherit = inherit;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_get_attr, &arg);
  }

int rsbac_get_attr_n(
                        rsbac_list_ta_number_t ta_number,
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   char * t_name,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value,
                   int inherit)
  { 
    union rsbac_syscall_arg_t arg;

    arg.get_attr_n.ta_number = ta_number;
    arg.get_attr_n.module = module;
    arg.get_attr_n.target = target;
    arg.get_attr_n.t_name = t_name;
    arg.get_attr_n.attr = attr;
    arg.get_attr_n.value = value;
    arg.get_attr_n.inherit = inherit;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_get_attr_n, &arg);
  }

int rsbac_set_attr(
                        rsbac_list_ta_number_t ta_number,
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   union rsbac_target_id_t * tid,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value)

  { 
    union rsbac_syscall_arg_t arg;

    arg.set_attr.ta_number = ta_number;
    arg.set_attr.module = module;
    arg.set_attr.target = target;
    arg.set_attr.tid = tid;
    arg.set_attr.attr = attr;
    arg.set_attr.value = value;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_set_attr, &arg);
  }

int rsbac_set_attr_n(
                        rsbac_list_ta_number_t ta_number,
                   enum rsbac_switch_target_t module,
                   enum rsbac_target_t target,
                   char * t_name,
                   enum rsbac_attribute_t attr,
                   union rsbac_attribute_value_t * value)

  {
    union rsbac_syscall_arg_t arg;

    arg.set_attr_n.ta_number = ta_number;
    arg.set_attr_n.module = module;
    arg.set_attr_n.target = target;
    arg.set_attr_n.t_name = t_name;
    arg.set_attr_n.attr = attr;
    arg.set_attr_n.value = value;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_set_attr_n, &arg);
  }

int rsbac_remove_target(
              rsbac_list_ta_number_t ta_number,
        enum  rsbac_target_t target,
        union rsbac_target_id_t * tid)
  { 
    union rsbac_syscall_arg_t arg;

    arg.remove_target.ta_number = ta_number;
    arg.remove_target.target = target;
    arg.remove_target.tid = tid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_remove_target, &arg);
  }

int rsbac_remove_target_n(
       rsbac_list_ta_number_t ta_number,
  enum rsbac_target_t target,
       char * t_name)
  { 
    union rsbac_syscall_arg_t arg;

    arg.remove_target_n.ta_number = ta_number;
    arg.remove_target_n.target = target;
    arg.remove_target_n.t_name = t_name;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_remove_target_n, &arg);
  }

int rsbac_net_list_all_netdev(
  rsbac_list_ta_number_t ta_number,
  rsbac_netdev_id_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.net_list_all_netdev.ta_number = ta_number;
    arg.net_list_all_netdev.id_p = id_p;
    arg.net_list_all_netdev.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_net_list_all_netdev, &arg);
  }

int rsbac_net_template(
  rsbac_list_ta_number_t ta_number,
  enum rsbac_net_temp_syscall_t call,
  rsbac_net_temp_id_t id,
  union rsbac_net_temp_syscall_data_t * data_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.net_template.ta_number = ta_number;
    arg.net_template.call = call;
    arg.net_template.id = id;
    arg.net_template.data_p = data_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_net_template, &arg);
  }

int rsbac_net_list_all_template(
  rsbac_list_ta_number_t ta_number,
  rsbac_net_temp_id_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.net_list_all_template.ta_number = ta_number;
    arg.net_list_all_template.id_p = id_p;
    arg.net_list_all_template.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_net_list_all_template, &arg);
  }

int rsbac_switch(enum rsbac_switch_target_t module, int value)
  {
    union rsbac_syscall_arg_t arg;

    arg.switch_module.module = module;
    arg.switch_module.value = value;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_switch, &arg);
  }

int rsbac_get_switch(enum rsbac_switch_target_t module, int * value_p,
		int * switchable_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.get_switch_module.module = module;
    arg.get_switch_module.value_p = value_p;
    arg.get_switch_module.switchable_p = switchable_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_get_switch, &arg);
  }

/************** MAC ***************/

int rsbac_mac_set_curr_level(rsbac_security_level_t level,
                             rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_set_curr_level.level = level;
    arg.mac_set_curr_level.categories_p = categories_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_set_curr_level, &arg);
  }

int rsbac_mac_get_curr_level(rsbac_security_level_t      * level_p,
                             rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_curr_level.level_p = level_p;
    arg.mac_get_curr_level.categories_p = categories_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_get_curr_level, &arg);
  }

int rsbac_mac_get_max_level(rsbac_security_level_t      * level_p,
                            rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_max_level.level_p = level_p;
    arg.mac_get_max_level.categories_p = categories_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_get_max_level, &arg);
  }

int rsbac_mac_get_min_level(rsbac_security_level_t      * level_p,
                            rsbac_mac_category_vector_t * categories_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_curr_level.level_p = level_p;
    arg.mac_get_curr_level.categories_p = categories_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_get_min_level, &arg);
  }

int rsbac_mac_add_p_tru(
  rsbac_list_ta_number_t ta_number,
  rsbac_pid_t pid,
  rsbac_uid_t uid,
  rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_add_p_tru.ta_number = ta_number;
    arg.mac_add_p_tru.pid = pid;
    arg.mac_add_p_tru.uid = uid;
    arg.mac_add_p_tru.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_add_p_tru, &arg);
  }

int rsbac_mac_remove_p_tru(
  rsbac_list_ta_number_t ta_number,
  rsbac_pid_t pid,
  rsbac_uid_t uid)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_remove_p_tru.ta_number = ta_number;
    arg.mac_remove_p_tru.pid = pid;
    arg.mac_remove_p_tru.uid = uid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_remove_p_tru, &arg);
  }

int rsbac_mac_add_f_tru(
  rsbac_list_ta_number_t ta_number,
  char * filename,
  rsbac_uid_t uid,
  rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_add_f_tru.ta_number = ta_number;
    arg.mac_add_f_tru.filename = filename;
    arg.mac_add_f_tru.uid = uid;
    arg.mac_add_f_tru.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_add_f_tru, &arg);
  }

int rsbac_mac_remove_f_tru(
  rsbac_list_ta_number_t ta_number,
  char * filename,
  rsbac_uid_t uid)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_remove_f_tru.ta_number = ta_number;
    arg.mac_remove_f_tru.filename = filename;
    arg.mac_remove_f_tru.uid = uid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_remove_f_tru, &arg);
  }

/* trulist must have space for maxnum tru_range entries - first and last each! */
int rsbac_mac_get_f_trulist(
  rsbac_list_ta_number_t ta_number,
  char * filename,
  rsbac_uid_t trulist[],
  rsbac_time_t ttllist[],
  u_int maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_f_trulist.ta_number = ta_number;
    arg.mac_get_f_trulist.filename = filename;
    arg.mac_get_f_trulist.trulist = trulist;
    arg.mac_get_f_trulist.ttllist = ttllist;
    arg.mac_get_f_trulist.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_get_f_trulist, &arg);
  }

int rsbac_mac_get_p_trulist(
  rsbac_list_ta_number_t ta_number,
  rsbac_pid_t pid,
  rsbac_uid_t trulist[],
  rsbac_time_t ttllist[],
  u_int maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.mac_get_p_trulist.ta_number = ta_number;
    arg.mac_get_p_trulist.pid = pid;
    arg.mac_get_p_trulist.trulist = trulist;
    arg.mac_get_p_trulist.ttllist = ttllist;
    arg.mac_get_p_trulist.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_mac_get_p_trulist, &arg);
  }

/************** PM ***************/

int rsbac_stats_pm(void)
  {
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_stats_pm, NULL);
  }

int rsbac_pm(
        rsbac_list_ta_number_t ta_number,
  enum  rsbac_pm_function_type_t    function,
  union rsbac_pm_function_param_t * param_p,
        rsbac_pm_tkt_id_t           ticket)
  {
    union rsbac_syscall_arg_t arg;

    arg.pm.ta_number = ta_number;
    arg.pm.function = function;
    arg.pm.param_p = param_p;
    arg.pm.ticket = ticket;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_pm, &arg);
  }

int rsbac_pm_change_current_task(rsbac_pm_task_id_t task)
  {
    union rsbac_syscall_arg_t arg;

    arg.pm_change_current_task.task = task;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_pm_change_current_task, &arg);
  }

int rsbac_pm_create_file(const char * filename,
                             int mode,
                             rsbac_pm_object_class_id_t object_class)
  {
    union rsbac_syscall_arg_t arg;

    arg.pm_create_file.filename = filename;
    arg.pm_create_file.mode = mode;
    arg.pm_create_file.object_class = object_class;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_pm_create_file, &arg);
  }

/************** DAZ **************/

int rsbac_daz_flush_cache(void)
  {
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_daz_flush_cache, NULL);
  }

/************** RC ***************/

int rsbac_rc_copy_role(
  rsbac_list_ta_number_t ta_number,
  rsbac_rc_role_id_t from_role,
  rsbac_rc_role_id_t to_role)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_copy_role.ta_number = ta_number;
    arg.rc_copy_role.from_role = from_role;
    arg.rc_copy_role.to_role = to_role;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_copy_role, &arg);
  }

int rsbac_rc_copy_type(
       rsbac_list_ta_number_t ta_number,
  enum rsbac_target_t         target,
       rsbac_rc_type_id_t     from_type,
       rsbac_rc_type_id_t     to_type)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_copy_type.ta_number = ta_number;
    arg.rc_copy_type.target = target;
    arg.rc_copy_type.from_type = from_type;
    arg.rc_copy_type.to_type = to_type;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_copy_type, &arg);
  }

int rsbac_rc_get_item(
        rsbac_list_ta_number_t  ta_number,
  enum  rsbac_rc_target_t       target,
  union rsbac_rc_target_id_t  * tid_p,
  union rsbac_rc_target_id_t  * subtid_p,
  enum  rsbac_rc_item_t         item,
  union rsbac_rc_item_value_t * value_p,
        rsbac_time_t          * ttl_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_get_item.ta_number = ta_number;
    arg.rc_get_item.target = target;
    arg.rc_get_item.tid_p = tid_p;
    arg.rc_get_item.subtid_p = subtid_p;
    arg.rc_get_item.item = item;
    arg.rc_get_item.value_p = value_p;
    arg.rc_get_item.ttl_p = ttl_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_get_item, &arg);
  }

/* Setting values */
int rsbac_rc_set_item(
        rsbac_list_ta_number_t  ta_number,
  enum  rsbac_rc_target_t       target,
  union rsbac_rc_target_id_t  * tid_p,
  union rsbac_rc_target_id_t  * subtid_p,
  enum  rsbac_rc_item_t         item,
  union rsbac_rc_item_value_t * value_p,
        rsbac_time_t            ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_set_item.ta_number = ta_number;
    arg.rc_set_item.target = target;
    arg.rc_set_item.tid_p = tid_p;
    arg.rc_set_item.subtid_p = subtid_p;
    arg.rc_set_item.item = item;
    arg.rc_set_item.value_p = value_p;
    arg.rc_set_item.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_set_item, &arg);
  }

int rsbac_rc_get_list(
        rsbac_list_ta_number_t  ta_number,
  enum  rsbac_rc_target_t       target,
  union rsbac_rc_target_id_t  * tid_p,
  enum  rsbac_rc_item_t         item,
        u_int maxnum,
        __u32  * array_p,
        rsbac_time_t * ttl_array_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_get_list.ta_number = ta_number;
    arg.rc_get_list.target = target;
    arg.rc_get_list.tid_p = tid_p;
    arg.rc_get_list.item = item;
    arg.rc_get_list.maxnum = maxnum;
    arg.rc_get_list.array_p = array_p;
    arg.rc_get_list.ttl_array_p = ttl_array_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_get_list, &arg);
  }

int rsbac_rc_change_role (rsbac_rc_role_id_t role, char * pass)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_change_role.role = role;
    arg.rc_change_role.pass = pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_change_role, &arg);
  }

int rsbac_rc_get_eff_rights_n(
        rsbac_list_ta_number_t ta_number,
  enum  rsbac_target_t   target,
        char           * t_name,
        rsbac_rc_request_vector_t * request_vector_p,
        rsbac_time_t          * ttl_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_get_eff_rights_n.ta_number = ta_number;
    arg.rc_get_eff_rights_n.target = target;
    arg.rc_get_eff_rights_n.t_name = t_name;
    arg.rc_get_eff_rights_n.request_vector_p = request_vector_p;
    arg.rc_get_eff_rights_n.ttl_p = ttl_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_get_eff_rights_n, &arg);
  }

int rsbac_rc_get_current_role (rsbac_rc_role_id_t * role_p)
  {
    union rsbac_syscall_arg_t arg;

    arg.rc_get_current_role.role_p = role_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_get_current_role, &arg);
  }

int rsbac_rc_select_fd_create_type(rsbac_rc_type_id_t type)
{
	union rsbac_syscall_arg_t arg;

	arg.rc_select_fd_create_type.type = type;

	return sys_rsbac(RSBAC_VERSION_NR, RSYS_rc_select_fd_create_type, &arg);
}


/************** AUTH ***************/

/* Provide means for adding and removing of capabilities */
int rsbac_auth_add_p_cap(
  rsbac_list_ta_number_t ta_number,
  rsbac_pid_t pid,
  enum rsbac_auth_cap_type_t cap_type,
  struct rsbac_auth_cap_range_t cap_range,
  rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_add_p_cap.ta_number = ta_number;
    arg.auth_add_p_cap.pid = pid;
    arg.auth_add_p_cap.cap_type = cap_type;
    arg.auth_add_p_cap.cap_range = cap_range;
    arg.auth_add_p_cap.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_auth_add_p_cap, &arg);
  }

int rsbac_auth_remove_p_cap(
  rsbac_list_ta_number_t ta_number,
  rsbac_pid_t pid,
  enum rsbac_auth_cap_type_t cap_type,
  struct rsbac_auth_cap_range_t cap_range)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_remove_p_cap.ta_number = ta_number;
    arg.auth_remove_p_cap.pid = pid;
    arg.auth_remove_p_cap.cap_type = cap_type;
    arg.auth_remove_p_cap.cap_range = cap_range;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_auth_remove_p_cap, &arg);
  }

int rsbac_auth_add_f_cap(
  rsbac_list_ta_number_t ta_number,
  char * filename,
  enum rsbac_auth_cap_type_t cap_type,
  struct rsbac_auth_cap_range_t cap_range,
  rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_add_f_cap.ta_number = ta_number;
    arg.auth_add_f_cap.filename = filename;
    arg.auth_add_f_cap.cap_type = cap_type;
    arg.auth_add_f_cap.cap_range = cap_range;
    arg.auth_add_f_cap.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_auth_add_f_cap, &arg);
  }

int rsbac_auth_remove_f_cap(
  rsbac_list_ta_number_t ta_number,
  char * filename,
  enum rsbac_auth_cap_type_t cap_type,
  struct rsbac_auth_cap_range_t cap_range)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_remove_f_cap.ta_number = ta_number;
    arg.auth_remove_f_cap.filename = filename;
    arg.auth_remove_f_cap.cap_type = cap_type;
    arg.auth_remove_f_cap.cap_range = cap_range;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_auth_remove_f_cap, &arg);
  }

/* caplist must have space for maxnum cap_range entries - first and last each! */
int rsbac_auth_get_f_caplist(
  rsbac_list_ta_number_t ta_number,
  char * filename,
  enum rsbac_auth_cap_type_t cap_type,
  struct rsbac_auth_cap_range_t caplist[],
  rsbac_time_t ttllist[],
  u_int maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_get_f_caplist.ta_number = ta_number;
    arg.auth_get_f_caplist.filename = filename;
    arg.auth_get_f_caplist.cap_type = cap_type;
    arg.auth_get_f_caplist.caplist = caplist;
    arg.auth_get_f_caplist.ttllist = ttllist;
    arg.auth_get_f_caplist.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_auth_get_f_caplist, &arg);
  }

int rsbac_auth_get_p_caplist(
  rsbac_list_ta_number_t ta_number,
  rsbac_pid_t pid,
  enum rsbac_auth_cap_type_t cap_type,
  struct rsbac_auth_cap_range_t caplist[],
  rsbac_time_t ttllist[],
  u_int maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.auth_get_p_caplist.ta_number = ta_number;
    arg.auth_get_p_caplist.pid = pid;
    arg.auth_get_p_caplist.cap_type = cap_type;
    arg.auth_get_p_caplist.caplist = caplist;
    arg.auth_get_p_caplist.ttllist = ttllist;
    arg.auth_get_p_caplist.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_auth_get_p_caplist, &arg);
  }

/**********************************/
/************** REG ***************/

int rsbac_reg(rsbac_reg_handle_t handle,
                  void * arg)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.reg.handle = handle;
    s_arg.reg.arg = arg;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_reg, &s_arg);
  }


/**********************************/
/************** ACL ***************/

int rsbac_acl(
  rsbac_list_ta_number_t ta_number,
  enum   rsbac_acl_syscall_type_t call,
  struct rsbac_acl_syscall_arg_t * arg)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl.ta_number = ta_number;
    s_arg.acl.call = call;
    s_arg.acl.arg = arg;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl, &s_arg);
  }

int rsbac_acl_n(
  rsbac_list_ta_number_t ta_number,
  enum   rsbac_acl_syscall_type_t call,
  struct rsbac_acl_syscall_n_arg_t * arg)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_n.ta_number = ta_number;
    s_arg.acl_n.call = call;
    s_arg.acl_n.arg = arg;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_n, &s_arg);
  }

int rsbac_acl_get_rights(
  rsbac_list_ta_number_t ta_number,
  struct rsbac_acl_syscall_arg_t   * arg,
  rsbac_acl_rights_vector_t * rights_p,
  u_int                     effective)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_rights.ta_number = ta_number;
    s_arg.acl_get_rights.arg = arg;
    s_arg.acl_get_rights.rights_p = rights_p;
    s_arg.acl_get_rights.effective = effective;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_get_rights, &s_arg);
  }

int rsbac_acl_get_rights_n(
  rsbac_list_ta_number_t ta_number,
  struct rsbac_acl_syscall_n_arg_t * arg,
  rsbac_acl_rights_vector_t * rights_p,
  u_int                     effective)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_rights_n.ta_number = ta_number;
    s_arg.acl_get_rights_n.arg = arg;
    s_arg.acl_get_rights_n.rights_p = rights_p;
    s_arg.acl_get_rights_n.effective = effective;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_get_rights_n, &s_arg);
  }

int rsbac_acl_get_tlist (
  rsbac_list_ta_number_t     ta_number,
  enum   rsbac_target_t      target,
  union  rsbac_target_id_t * tid,
  struct rsbac_acl_entry_t   entry_array[],
         rsbac_time_t        ttl_array[],
         u_int               maxnum)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_tlist.ta_number = ta_number;
    s_arg.acl_get_tlist.target = target;
    s_arg.acl_get_tlist.tid = tid;
    s_arg.acl_get_tlist.entry_array = entry_array;
    s_arg.acl_get_tlist.ttl_array = ttl_array;
    s_arg.acl_get_tlist.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_get_tlist, &s_arg);
  }

int rsbac_acl_get_tlist_n(
  rsbac_list_ta_number_t     ta_number,
  enum   rsbac_target_t      target,
         char              * t_name,
  struct rsbac_acl_entry_t   entry_array[],
         rsbac_time_t        ttl_array[],
         u_int               maxnum)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_tlist_n.ta_number = ta_number;
    s_arg.acl_get_tlist_n.target = target;
    s_arg.acl_get_tlist_n.t_name = t_name;
    s_arg.acl_get_tlist_n.entry_array = entry_array;
    s_arg.acl_get_tlist_n.ttl_array = ttl_array;
    s_arg.acl_get_tlist_n.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_get_tlist_n, &s_arg);
  }

int rsbac_acl_get_mask (
  rsbac_list_ta_number_t     ta_number,
  enum   rsbac_target_t              target,
  union  rsbac_target_id_t         * tid,
         rsbac_acl_rights_vector_t * mask_p)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_mask.ta_number = ta_number;
    s_arg.acl_get_mask.target = target;
    s_arg.acl_get_mask.tid = tid;
    s_arg.acl_get_mask.mask_p = mask_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_get_mask, &s_arg);
  }

int rsbac_acl_get_mask_n(
       rsbac_list_ta_number_t      ta_number,
  enum rsbac_target_t              target,
       char                      * t_name,
       rsbac_acl_rights_vector_t * mask_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_get_mask_n.ta_number = ta_number;
    s_arg.acl_get_mask_n.target = target;
    s_arg.acl_get_mask_n.t_name = t_name;
    s_arg.acl_get_mask_n.mask_p = mask_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_get_mask_n, &s_arg);
  }

/********  ACL groups *********/

int rsbac_acl_group(
        rsbac_list_ta_number_t           ta_number,
  enum  rsbac_acl_group_syscall_type_t   call,
  union rsbac_acl_group_syscall_arg_t  * arg_p)
  { 
    union rsbac_syscall_arg_t s_arg;

    s_arg.acl_group.ta_number = ta_number;
    s_arg.acl_group.call = call;
    s_arg.acl_group.arg_p = arg_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_group, &s_arg);
  }

/**********************************/
/************** JAIL **************/

int rsbac_jail(rsbac_version_t version,
               char * path,
               rsbac_jail_ip_t ip,
               rsbac_jail_flags_t flags,
               rsbac_cap_vector_t max_caps,
               rsbac_jail_scd_vector_t scd_get,
               rsbac_jail_scd_vector_t scd_modify)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.jail.version = version;
    s_arg.jail.path = path;
    s_arg.jail.ip = ip;
    s_arg.jail.flags = flags;
    s_arg.jail.max_caps = max_caps;
    s_arg.jail.scd_get = scd_get;
    s_arg.jail.scd_modify = scd_modify;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_jail, &s_arg);
  }

/**********************************/
/**************  UM  **************/

int rsbac_um_auth_name(char * name,
                       char * pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_auth_name.name = name;
    s_arg.um_auth_name.pass = pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_auth_name, &s_arg);
  }

int rsbac_um_auth_uid(rsbac_uid_t uid,
                      char * pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_auth_uid.uid = uid;
    s_arg.um_auth_name.pass = pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_auth_uid, &s_arg);
  }

int rsbac_um_add_user(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t uid,
  struct rsbac_um_user_entry_t * entry_p,
  char * pass,
  rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_add_user.ta_number = ta_number;
    s_arg.um_add_user.uid = uid;
    s_arg.um_add_user.entry_p = entry_p;
    s_arg.um_add_user.pass = pass;
    s_arg.um_add_user.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_add_user, &s_arg);
  }

int rsbac_um_add_group(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t gid,
  struct rsbac_um_group_entry_t * entry_p,
  char * pass,
  rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_add_group.ta_number = ta_number;
    s_arg.um_add_group.gid = gid;
    s_arg.um_add_group.entry_p = entry_p;
    s_arg.um_add_group.pass = pass;
    s_arg.um_add_group.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_add_group, &s_arg);
  }

int rsbac_um_add_gm(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t uid,
  rsbac_gid_num_t gid,
  rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_add_gm.ta_number = ta_number;
    s_arg.um_add_gm.uid = uid;
    s_arg.um_add_gm.gid = gid;
    s_arg.um_add_gm.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_add_gm, &s_arg);
  }

int rsbac_um_mod_user(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t uid,
  enum rsbac_um_mod_t mod,
  union rsbac_um_mod_data_t * data_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_mod_user.ta_number = ta_number;
    s_arg.um_mod_user.uid = uid;
    s_arg.um_mod_user.mod = mod;
    s_arg.um_mod_user.data_p = data_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_mod_user, &s_arg);
  }

int rsbac_um_mod_group(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t gid,
  enum rsbac_um_mod_t mod,
  union rsbac_um_mod_data_t * data_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_mod_group.ta_number = ta_number;
    s_arg.um_mod_group.gid = gid;
    s_arg.um_mod_group.mod = mod;
    s_arg.um_mod_group.data_p = data_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_mod_group, &s_arg);
  }

int rsbac_um_get_user_item(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t uid,
  enum rsbac_um_mod_t mod,
  union rsbac_um_mod_data_t * data_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_user_item.ta_number = ta_number;
    s_arg.um_get_user_item.uid = uid;
    s_arg.um_get_user_item.mod = mod;
    s_arg.um_get_user_item.data_p = data_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_user_item, &s_arg);
  }

int rsbac_um_get_group_item(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t gid,
  enum rsbac_um_mod_t mod,
  union rsbac_um_mod_data_t * data_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_group_item.ta_number = ta_number;
    s_arg.um_get_group_item.gid = gid;
    s_arg.um_get_group_item.mod = mod;
    s_arg.um_get_group_item.data_p = data_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_group_item, &s_arg);
  }

int rsbac_um_remove_user(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t uid)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_remove_user.ta_number = ta_number;
    s_arg.um_remove_user.uid = uid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_remove_user, &s_arg);
  }

int rsbac_um_remove_group(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t gid)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_remove_group.ta_number = ta_number;
    s_arg.um_remove_group.gid = gid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_remove_group, &s_arg);
  }

int rsbac_um_remove_gm(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t uid,
  rsbac_gid_num_t gid)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_remove_gm.ta_number = ta_number;
    s_arg.um_remove_gm.uid = uid;
    s_arg.um_remove_gm.gid = gid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_remove_gm, &s_arg);
  }

int rsbac_um_user_exists(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t uid)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_user_exists.ta_number = ta_number;
    s_arg.um_user_exists.uid = uid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_user_exists, &s_arg);
  }

int rsbac_um_group_exists(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t gid)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_group_exists.ta_number = ta_number;
    s_arg.um_group_exists.gid = gid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_group_exists, &s_arg);
  }

int rsbac_um_get_next_user(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t old_user,
  rsbac_uid_t * next_user_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_next_user.ta_number = ta_number;
    s_arg.um_get_next_user.old_user = old_user;
    s_arg.um_get_next_user.next_user_p = next_user_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_next_user, &s_arg);
  }

int rsbac_um_get_user_list(
  rsbac_list_ta_number_t ta_number,
  rsbac_um_set_t vset,
  rsbac_uid_t user_array[],
  u_int       maxnum)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_user_list.ta_number = ta_number;
    s_arg.um_get_user_list.vset = vset;
    s_arg.um_get_user_list.user_array = user_array;
    s_arg.um_get_user_list.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_user_list, &s_arg);
  }

int rsbac_um_get_gm_list(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t user,
  rsbac_gid_num_t group_array[],
  u_int       maxnum)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_gm_list.ta_number = ta_number;
    s_arg.um_get_gm_list.user = user;
    s_arg.um_get_gm_list.group_array = group_array;
    s_arg.um_get_gm_list.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_gm_list, &s_arg);
  }

int rsbac_um_get_gm_user_list(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t group,
  rsbac_uid_num_t user_array[],
  u_int       maxnum)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_gm_user_list.ta_number = ta_number;
    s_arg.um_get_gm_user_list.group = group;
    s_arg.um_get_gm_user_list.user_array = user_array;
    s_arg.um_get_gm_user_list.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_gm_user_list, &s_arg);
  }

int rsbac_um_get_group_list(
  rsbac_list_ta_number_t ta_number,
  rsbac_um_set_t vset,
  rsbac_gid_t group_array[],
  u_int       maxnum)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_group_list.ta_number = ta_number;
    s_arg.um_get_group_list.vset = vset;
    s_arg.um_get_group_list.group_array = group_array;
    s_arg.um_get_group_list.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_group_list, &s_arg);
  }

int rsbac_um_get_uid(
  rsbac_list_ta_number_t ta_number,
  char * name,
  rsbac_uid_t * uid_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_uid.ta_number = ta_number;
    s_arg.um_get_uid.name = name;
    s_arg.um_get_uid.uid_p = uid_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_uid, &s_arg);
  }

int rsbac_um_get_gid(
  rsbac_list_ta_number_t ta_number,
  char * name,
  rsbac_gid_t * gid_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_get_gid.ta_number = ta_number;
    s_arg.um_get_gid.name = name;
    s_arg.um_get_gid.gid_p = gid_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_get_gid, &s_arg);
  }

int rsbac_um_set_pass(rsbac_uid_t uid,
                      char * old_pass,
                      char * new_pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_set_pass.uid = uid;
    s_arg.um_set_pass.old_pass = old_pass;
    s_arg.um_set_pass.new_pass = new_pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_set_pass, &s_arg);
  }

int rsbac_um_set_pass_name(char * name,
                           char * old_pass,
                           char * new_pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_set_pass_name.name = name;
    s_arg.um_set_pass_name.old_pass = old_pass;
    s_arg.um_set_pass_name.new_pass = new_pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_set_pass_name, &s_arg);
  }

int rsbac_um_add_onetime(rsbac_uid_t uid,
                      char * old_pass,
                      char * new_pass,
                      rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_add_onetime.uid = uid;
    s_arg.um_add_onetime.old_pass = old_pass;
    s_arg.um_add_onetime.new_pass = new_pass;
    s_arg.um_add_onetime.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_add_onetime, &s_arg);
  }

int rsbac_um_add_onetime_name(char * name,
                           char * old_pass,
                           char * new_pass,
                           rsbac_time_t ttl)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_add_onetime_name.name = name;
    s_arg.um_add_onetime_name.old_pass = old_pass;
    s_arg.um_add_onetime_name.new_pass = new_pass;
    s_arg.um_add_onetime_name.ttl = ttl;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_add_onetime_name, &s_arg);
  }

int rsbac_um_remove_all_onetime(rsbac_uid_t uid,
                      char * old_pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_remove_all_onetime.uid = uid;
    s_arg.um_remove_all_onetime.old_pass = old_pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_remove_all_onetime, &s_arg);
  }

int rsbac_um_remove_all_onetime_name(char * name,
                           char * old_pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_remove_all_onetime_name.name = name;
    s_arg.um_remove_all_onetime_name.old_pass = old_pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_remove_all_onetime_name, &s_arg);
  }

int rsbac_um_count_onetime(rsbac_uid_t uid,
                      char * old_pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_count_onetime.uid = uid;
    s_arg.um_count_onetime.old_pass = old_pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_count_onetime, &s_arg);
  }

int rsbac_um_count_onetime_name(char * name,
                           char * old_pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_count_onetime_name.name = name;
    s_arg.um_count_onetime_name.old_pass = old_pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_count_onetime_name, &s_arg);
  }

int rsbac_um_set_group_pass(rsbac_gid_t gid,
                            char * new_pass)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_set_group_pass.gid = gid;
    s_arg.um_set_group_pass.new_pass = new_pass;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_set_group_pass, &s_arg);
  }

int rsbac_um_check_account(rsbac_uid_t uid)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_check_account.uid = uid;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_check_account, &s_arg);
  }

int rsbac_um_check_account_name(char * name)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_check_account_name.name = name;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_check_account_name, &s_arg);
  }

int rsbac_um_select_vset(rsbac_um_set_t vset)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.um_select_vset.vset = vset;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_um_select_vset, &s_arg);
  }

int rsbac_list_ta_begin(rsbac_time_t ttl,
                        rsbac_list_ta_number_t * ta_number_p,
                        rsbac_uid_t commit_uid,
                        char * password)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.list_ta_begin.ttl = ttl;
    s_arg.list_ta_begin.ta_number_p = ta_number_p;
    s_arg.list_ta_begin.commit_uid = commit_uid;
    s_arg.list_ta_begin.password = password;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_ta_begin, &s_arg);
  }

int rsbac_list_ta_refresh(rsbac_time_t ttl,
                          rsbac_list_ta_number_t ta_number,
                          char * password)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.list_ta_refresh.ttl = ttl;
    s_arg.list_ta_refresh.ta_number = ta_number;
    s_arg.list_ta_refresh.password = password;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_ta_refresh, &s_arg);
  }

int rsbac_list_ta_commit(rsbac_list_ta_number_t ta_number,
                         char * password)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.list_ta_commit.ta_number = ta_number;
    s_arg.list_ta_commit.password = password;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_ta_commit, &s_arg);
  }

int rsbac_list_ta_forget(rsbac_list_ta_number_t ta_number,
                         char * password)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.list_ta_forget.ta_number = ta_number;
    s_arg.list_ta_forget.password = password;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_ta_forget, &s_arg);
  }

int rsbac_list_all_dev(
  rsbac_list_ta_number_t ta_number,
  struct rsbac_dev_desc_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.list_all_dev.ta_number = ta_number;
    arg.list_all_dev.id_p = id_p;
    arg.list_all_dev.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_all_dev, &arg);
  }

int rsbac_acl_list_all_dev(
  rsbac_list_ta_number_t ta_number,
  struct rsbac_dev_desc_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.acl_list_all_dev.ta_number = ta_number;
    arg.acl_list_all_dev.id_p = id_p;
    arg.acl_list_all_dev.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_list_all_dev, &arg);
  }

int rsbac_list_all_user(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.list_all_user.ta_number = ta_number;
    arg.list_all_user.id_p = id_p;
    arg.list_all_user.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_all_user, &arg);
  }

int rsbac_acl_list_all_user(
  rsbac_list_ta_number_t ta_number,
  rsbac_uid_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.acl_list_all_user.ta_number = ta_number;
    arg.acl_list_all_user.id_p = id_p;
    arg.acl_list_all_user.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_list_all_user, &arg);
  }

int rsbac_list_all_group(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.list_all_group.ta_number = ta_number;
    arg.list_all_group.id_p = id_p;
    arg.list_all_group.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_all_group, &arg);
  }

int rsbac_acl_list_all_group(
  rsbac_list_ta_number_t ta_number,
  rsbac_gid_t * id_p,
  u_long maxnum)
  {
    union rsbac_syscall_arg_t arg;

    arg.acl_list_all_group.ta_number = ta_number;
    arg.acl_list_all_group.id_p = id_p;
    arg.acl_list_all_group.maxnum = maxnum;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_acl_list_all_group, &arg);
  }

int rsbac_list_all_ipc(rsbac_list_ta_number_t ta_number,
               struct rsbac_ipc_t * id_p, u_long maxnum)
{
       union rsbac_syscall_arg_t arg;

       arg.list_all_ipc.ta_number = ta_number;
       arg.list_all_ipc.id_p = id_p;
       arg.list_all_ipc.maxnum = maxnum;
       return sys_rsbac(RSBAC_VERSION_NR, RSYS_list_all_ipc, &arg);
}

/************************************************* */
/*             DEBUG/LOG functions                 */
/************************************************* */

int rsbac_adf_log_switch(enum rsbac_adf_request_t request,
                         enum rsbac_target_t      target,
                         u_int value)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.adf_log_switch.request = request;
    s_arg.adf_log_switch.target = target;
    s_arg.adf_log_switch.value = value;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_adf_log_switch, &s_arg);
  }

int rsbac_get_adf_log(enum rsbac_adf_request_t request,
                      enum rsbac_target_t      target,
                      u_int * value_p)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.get_adf_log.request = request;
    s_arg.get_adf_log.target = target;
    s_arg.get_adf_log.value_p = value_p;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_get_adf_log, &s_arg);
  }

/*
 * Commands to rsbac_log:
 *
 * 	0 -- Close the log.  Currently a NOP.
 * 	1 -- Open the log. Currently a NOP.
 * 	2 -- Read from the log.
 * 	3 -- Read up to the last 4k of messages in the ring buffer.
 * 	4 -- Read and clear last 4k of messages in the ring buffer
 * 	5 -- Clear ring buffer.
 */
int rsbac_log(int type,
                  char * buf,
                  int len)
  {
    union rsbac_syscall_arg_t s_arg;

    s_arg.log.type = type;
    s_arg.log.buf = buf;
    s_arg.log.len = len;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_log, &s_arg);
  }

int rsbac_init(char * root_dev)
  {
    union rsbac_syscall_arg_t arg;

    arg.init.root_dev = root_dev;
    return sys_rsbac(RSBAC_VERSION_NR, RSYS_init, &arg);
  }


void locale_init(){
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
}
