# ifndef _SKIT_AVEC_H
# define _SKIT_AVEC_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
/*Class:avec
NAME: @code{avec} - Associative sparse vector 
DESCRIPTION:       
 
 The class implements an associative sparse vector.
 The constructor is be invocated whith a size parameter:
 @example
          avec<float> x(n);
 @end example

 This size is a declarative size, for matrix-sparse vector multiplication
 (see @ref{csr class}),
 or conversion to dense vector
 (see @ref{vec class}).
 The sparse vector is initialy null, i.e.  empty.
 Incremental building writes: 
 @example
          x.entry(i) += 3.14;
 @end example

 where the i-th entry is created, if not yet exists.
 The read access is as:
 @example
          cout << x(i) << endl;
 @end example

AUTHOR: 
    
    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LMC-IMAG, 38041 Grenoble cedex 9, France
DATE:
    
    11 march 1997

METHODS: @avec

End:

*/
// =============================[ AVEC CLASS ]===================================
# include "rheolef/avecrep.h"
namespace rheolef { 

#ifdef _RHEOLEF_HAVE_EXPRESSION_TEMPLATE

    template <class SX> class SVExpr;

    template <class T, class SX>
    inline
    avec<T>&
    assign (avec<T>& sy, const SVExpr<SX>& sx)
    { 
        Index sx_n = sx.n();
	if (sy.n() == 0 && sx_n != 0) sy.resize(sx_n);
	check_length (sx, sy); 
	szassignopsx (sy, my_ident<SVExpr<SX>::element_type, T>(),
		      sx.begin(), sx.end());
	return sy;     
    }   
#endif // _RHEOLEF_HAVE_EXPRESSION_TEMPLATE

//<avec:
template <class T>
class avec : public smart_pointer<avecrep<T> > 
{
  public:
    typedef          T                          element_type;
    typedef typename avecrep<T>::value_type     value_type;
    typedef typename avecrep<T>::iterator       iterator;
    typedef typename avecrep<T>::const_iterator const_iterator;

    // cstors, conversion and assign
	explicit avec (Index n) : smart_pointer<avecrep<T> >(new_macro(avecrep<T>(n))) {}
#ifdef _RHEOLEF_HAVE_EXPRESSION_TEMPLATE

	template <class SX>
	avec(const SVExpr<SX>& sx) 
			: Array<T>(sx.size()) { assign (*this, sx); }
	template <class SX>
	avec<T>& operator = (const SVExpr<SX>& sx) 
			{ reset(); return assign (*this, sx); }
#endif // _RHEOLEF_HAVE_EXPRESSION_TEMPLATE

    // accessors
	Index n () const { return data().n(); }
	Index nnz () const { return data().nnz(); }
	T operator() (Index i) const { return data().operator() (i); }
	T& entry (Index i) { return data().entry(i); }
   
	iterator begin() { return data().begin(); }
	iterator end() { return data().end(); }
	const_iterator begin() const { return data().begin(); }
	const_iterator end() const { return data().end(); }
	T operator[](Index i) const {   return data().operator[](i); }
	T& operator[](Index i) { return data().operator[](i); }
	Index size() const { return data().size(); }
	void resize (Index n) { /* for compatibility with sparse mat-row */ }
	iterator insert (iterator position, const value_type& pair_i_ix)
		{ 
		  warning_macro("insert...");
		  iterator i = data().insert(position, pair_i_ix);
		  warning_macro("insert done.");
		  return i;
		}
	void erase (iterator first, iterator last) { data().erase(first,last); }
	void erase (iterator position) { data().erase(position); }
	void reset () { data().reset(); }

    // properties
	bool any_element_is_negative() const 
		{ return data().any_element_is_negative (); }
	bool any_element_is_inf_or_nan() const 
		{ return data().any_element_is_inf_or_nan (); }
	bool all_elements_are_int_or_inf_or_nan() const 
		{ return data().all_elements_are_int_or_inf_or_nan (); }
	T max() const { return data().max(); }
	T min() const { return data().min(); }
	T max_abs() const { return data().max_abs(); }
	T min_abs() const { return data().min_abs(); }
protected:
	const avecrep<T>& data() const {
		return smart_pointer<avecrep<T> >::data();
	}
	avecrep<T>& data() {
		return smart_pointer<avecrep<T> >::data();
	}
};
template <class T>
std::ostream& operator << (std::ostream& s, const avec<T>& sx);
//>avec:
}// namespace rheolef
#endif // _SKIT_AVEC_H
