package org.red5.server.jetty;

/*
 * RED5 Open Source Flash Server - http://www.osflash.org/red5
 *
 * Copyright (c) 2006-2009 by respective authors (see below). All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free Software
 * Foundation; either version 2.1 of the License, or (at your option) any later
 * version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along
 * with this library; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

import java.io.File;

import org.eclipse.jetty.deploy.WebAppDeployer;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.server.handler.ContextHandlerCollection;
import org.eclipse.jetty.server.handler.HandlerCollection;
import org.eclipse.jetty.util.thread.ThreadPool;
import org.red5.server.LoaderBase;
import org.red5.server.api.IApplicationContext;
import org.red5.server.jmx.JMXAgent;
import org.red5.server.jmx.mxbeans.LoaderMXBean;
import org.red5.server.util.FileUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Class that loads Red5 applications using Jetty.
 */
public class JettyLoader extends LoaderBase implements LoaderMXBean {

	/**
	 *  Logger
	 */
	protected static Logger log = LoggerFactory.getLogger(JettyLoader.class);

	/**
	 *  Default web config filename
	 */
	protected String defaultWebConfig = "plugins/jetty-web.xml";

	/**
	 *  IServer implementation
	 */
	protected Server jetty;

	protected Connector[] connectors;
	protected ThreadPool threadPool;
	protected HandlerCollection handlers;
	
	/**
	 * Remove context from the current host.
	 * 
	 * @param path		Path
	 */
	@Override
	public void removeContext(String path) {
		Handler[] handlers = jetty.getHandlers();
		for (Handler handler : handlers) {
			if (handler instanceof ContextHandler && ((ContextHandler) handler).getContextPath().equals(path)) {
				try {
					((ContextHandler) handler).stop();
					jetty.removeBean(handler);
					break;
				} catch (Exception e) {
					log.error("Could not remove context: {}", path, e);
				}
			}
		}
		IApplicationContext ctx = LoaderBase.removeRed5ApplicationContext(path);
		if (ctx != null) {
			ctx.stop();
		} else {
			log.warn("Red5 application context could not be stopped, it was null for path: {}", path);
		}
	}

	/**
	 *
	 */
	@SuppressWarnings("all")
	public void init() {
		log.info("Loading Jetty context");
		// So this class is left just starting jetty
		try {
			if (webappFolder == null) {
				// Use default webapps directory
				webappFolder = FileUtil.formatPath(System.getProperty("red5.root"), "/webapps");
			}
			System.setProperty("red5.webapp.root", webappFolder);
			log.info("Application root: {}", webappFolder);

			// root location for servlet container
			String serverRoot = System.getProperty("red5.root");
			log.debug("Server root: {}", serverRoot);

			// set in the system for tomcat classes
			System.setProperty("jetty.home", serverRoot);
			System.setProperty("jetty.class.path", String.format("%s/lib%s%s/plugins", serverRoot, File.separator, serverRoot));

			String[] handlersArr = new String[] { "org.eclipse.jetty.webapp.WebInfConfiguration", "org.eclipse.jetty.webapp.WebXmlConfiguration",
					"org.eclipse.jetty.webapp.JettyWebXmlConfiguration", "org.eclipse.jetty.webapp.TagLibConfiguration", "org.red5.server.jetty.Red5WebPropertiesConfiguration" };

			// instance a new org.mortbay.jetty.Server
			log.info("Starting jetty servlet engine");
			jetty = new Server();			
			jetty.setConnectors(connectors);
			jetty.setHandler(handlers);
			jetty.setThreadPool(threadPool);
			jetty.setStopAtShutdown(true);

			LoaderBase.setApplicationLoader(new JettyApplicationLoader(jetty, applicationContext));

			try {
				// Add web applications from web app root with web config
				HandlerCollection contexts = (HandlerCollection) jetty.getChildHandlerByClass(ContextHandlerCollection.class);
				if (contexts == null) {
					contexts = (HandlerCollection) jetty.getChildHandlerByClass(HandlerCollection.class);
				}
				WebAppDeployer deployer = new WebAppDeployer();
				deployer.setContexts(contexts);
				deployer.setWebAppDir(webappFolder);
				deployer.setDefaultsDescriptor(defaultWebConfig);
				deployer.setConfigurationClasses(handlersArr);
				deployer.setExtract(true);
				deployer.setParentLoaderPriority(true);
				deployer.start();
			} catch (Exception e) {
				log.error("Error deploying web applications", e);
			}

			// Start Jetty
			jetty.start();

		} catch (Exception e) {
			log.error("Error loading jetty", e);
		} finally {
			registerJMX();
		}

	}

	//TODO: Implement this for those who want to use Jetty
	public boolean startWebApplication(String applicationName) {
		return false;
	}

	public void registerJMX() {
		JMXAgent.registerMBean(this, this.getClass().getName(), LoaderMXBean.class);
	}

	public Connector[] getConnectors() {
		return connectors;
	}

	public void setConnectors(Connector[] connectors) {
		this.connectors = connectors;
	}

	public void setThreadPool(ThreadPool threadPool) {
		this.threadPool = threadPool;
	}

	public void setHandlers(HandlerCollection handlers) {
		this.handlers = handlers;
	}

	/**
	 * Shut server down
	 */
	public void shutdown() {
		log.info("Shutting down Jetty context");
		JMXAgent.shutdown();
		try {
			jetty.stop();
			System.exit(0);
		} catch (Exception e) {
			log.warn("Jetty could not be stopped", e);
			System.exit(1);
		}
	}

}
