# coding=utf-8
# --------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for
# license information.
#
# Code generated by Microsoft (R) AutoRest Code Generator.
# Changes may cause incorrect behavior and will be lost if the code is
# regenerated.
# --------------------------------------------------------------------------

from msrest.service_client import ServiceClient
from msrest import Serializer, Deserializer
from msrestazure import AzureConfiguration
from .version import VERSION
from msrest.pipeline import ClientRawResponse
import uuid
from . import models


class KeyVaultClientConfiguration(AzureConfiguration):
    """Configuration for KeyVaultClient
    Note that all parameters used to create this instance are saved as instance
    attributes.

    :param credentials: Credentials needed for the client to connect to Azure.
    :type credentials: :mod:`A msrestazure Credentials
     object<msrestazure.azure_active_directory>`
    """

    def __init__(
            self, credentials):

        if credentials is None:
            raise ValueError("Parameter 'credentials' must not be None.")
        base_url = '{vaultBaseUrl}'

        super(KeyVaultClientConfiguration, self).__init__(base_url)

        self.add_user_agent('keyvaultclient/{}'.format(VERSION))
        self.add_user_agent('Azure-SDK-For-Python')

        self.credentials = credentials


class KeyVaultClient(object):
    """The key vault client performs cryptographic key operations and vault operations against the Key Vault service.

    :ivar config: Configuration for client.
    :vartype config: KeyVaultClientConfiguration

    :param credentials: Credentials needed for the client to connect to Azure.
    :type credentials: :mod:`A msrestazure Credentials
     object<msrestazure.azure_active_directory>`
    """

    def __init__(
            self, credentials):

        self.config = KeyVaultClientConfiguration(credentials)
        self._client = ServiceClient(self.config.credentials, self.config)

        client_models = {k: v for k, v in models.__dict__.items() if isinstance(v, type)}
        self.api_version = '2016-10-01'
        self._serialize = Serializer(client_models)
        self._deserialize = Deserializer(client_models)


    def create_key(
            self, vault_base_url, key_name, kty, key_size=None, key_ops=None, key_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Creates a new key, stores it, then returns key parameters and
        attributes to the client.

        The create key operation can be used to create any key type in Azure
        Key Vault. If the named key already exists, Azure Key Vault creates a
        new version of the key.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name for the new key. The system will generate
         the version name for the new key.
        :type key_name: str
        :param kty: The type of key to create. For valid key types, see
         JsonWebKeyType. Supported JsonWebKey key types (kty) for Elliptic
         Curve, RSA, HSM, Octet. Possible values include: 'EC', 'RSA',
         'RSA-HSM', 'oct'
        :type kty: str or :class:`JsonWebKeyType
         <azure.keyvault.models.JsonWebKeyType>`
        :param key_size: The key size in bytes. For example, 1024 or 2048.
        :type key_size: int
        :param key_ops:
        :type key_ops: list of str or :class:`JsonWebKeyOperation
         <azure.keyvault.models.JsonWebKeyOperation>`
        :param key_attributes:
        :type key_attributes: :class:`KeyAttributes
         <azure.keyvault.models.KeyAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyBundle <azure.keyvault.models.KeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyCreateParameters(kty=kty, key_size=key_size, key_ops=key_ops, key_attributes=key_attributes, tags=tags)

        # Construct URL
        url = '/keys/{key-name}/create'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str', pattern='^[0-9a-zA-Z-]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyCreateParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def import_key(
            self, vault_base_url, key_name, key, hsm=None, key_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Imports an externally created key, stores it, and returns key
        parameters and attributes to the client.

        The import key operation may be used to import any key type into an
        Azure Key Vault. If the named key already exists, Azure Key Vault
        creates a new version of the key.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: Name for the imported key.
        :type key_name: str
        :param key: The Json web key
        :type key: :class:`JsonWebKey <azure.keyvault.models.JsonWebKey>`
        :param hsm: Whether to import as a hardware key (HSM) or software key.
        :type hsm: bool
        :param key_attributes: The key management attributes.
        :type key_attributes: :class:`KeyAttributes
         <azure.keyvault.models.KeyAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyBundle <azure.keyvault.models.KeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyImportParameters(hsm=hsm, key=key, key_attributes=key_attributes, tags=tags)

        # Construct URL
        url = '/keys/{key-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str', pattern='^[0-9a-zA-Z-]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyImportParameters')

        # Construct and send request
        request = self._client.put(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def delete_key(
            self, vault_base_url, key_name, custom_headers=None, raw=False, **operation_config):
        """Deletes a key of any type from storage in Azure Key Vault.

        The delete key operation cannot be used to remove individual versions
        of a key. This operation removes the cryptographic material associated
        with the key, which means the key is not usable for Sign/Verify,
        Wrap/Unwrap or Encrypt/Decrypt operations.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key to delete.
        :type key_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedKeyBundle
         <azure.keyvault.models.DeletedKeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/keys/{key-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('DeletedKeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_key(
            self, vault_base_url, key_name, key_version, key_ops=None, key_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """The update key operation changes specified attributes of a stored key
        and can be applied to any key type and key version stored in Azure Key
        Vault.

        In order to perform this operation, the key must already exist in the
        Key Vault. Note: The cryptographic material of a key itself cannot be
        changed.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of key to update.
        :type key_name: str
        :param key_version: The version of the key to update.
        :type key_version: str
        :param key_ops: Json web key operations. For more information on
         possible key operations, see JsonWebKeyOperation.
        :type key_ops: list of str or :class:`JsonWebKeyOperation
         <azure.keyvault.models.JsonWebKeyOperation>`
        :param key_attributes:
        :type key_attributes: :class:`KeyAttributes
         <azure.keyvault.models.KeyAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyBundle <azure.keyvault.models.KeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyUpdateParameters(key_ops=key_ops, key_attributes=key_attributes, tags=tags)

        # Construct URL
        url = '/keys/{key-name}/{key-version}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyUpdateParameters')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_key(
            self, vault_base_url, key_name, key_version, custom_headers=None, raw=False, **operation_config):
        """Gets the public part of a stored key.

        The get key operation is applicable to all key types. If the requested
        key is symmetric, then no key material is released in the response.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key to get.
        :type key_name: str
        :param key_version: Adding the version parameter retrieves a specific
         version of a key.
        :type key_version: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyBundle <azure.keyvault.models.KeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/keys/{key-name}/{key-version}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_key_versions(
            self, vault_base_url, key_name, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """Retrieves a list of individual key versions with the same key name.

        The full key identifier, attributes, and tags are provided in the
        response.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyItemPaged <azure.keyvault.models.KeyItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/keys/{key-name}/versions'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
                    'key-name': self._serialize.url("key_name", key_name, 'str')
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.KeyItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.KeyItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def get_keys(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List keys in the specified vault.

        Retrieves a list of the keys in the Key Vault as JSON Web Key
        structures that contain the public part of a stored key. The LIST
        operation is applicable to all key types, however only the base key
        identifier,attributes, and tags are provided in the response.
        Individual versions of a key are not listed in the response.
        Authorization: Requires the keys/list permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyItemPaged <azure.keyvault.models.KeyItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/keys'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.KeyItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.KeyItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def backup_key(
            self, vault_base_url, key_name, custom_headers=None, raw=False, **operation_config):
        """Requests that a backup of the specified key be downloaded to the
        client.

        The Key Backup operation exports a key from Azure Key Vault in a
        protected form. Note that this operation does NOT return key material
        in a form that can be used outside the Azure Key Vault system, the
        returned key material is either protected to a Azure Key Vault HSM or
        to Azure Key Vault itself. The intent of this operation is to allow a
        client to GENERATE a key in one Azure Key Vault instance, BACKUP the
        key, and then RESTORE it into another Azure Key Vault instance. The
        BACKUP operation may be used to export, in protected form, any key type
        from Azure Key Vault. Individual versions of a key cannot be backed up.
        BACKUP / RESTORE can be performed within geographical boundaries only;
        meaning that a BACKUP from one geographical area cannot be restored to
        another geographical area. For example, a backup from the US
        geographical area cannot be restored in an EU geographical area.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`BackupKeyResult
         <azure.keyvault.models.BackupKeyResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/keys/{key-name}/backup'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('BackupKeyResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def restore_key(
            self, vault_base_url, key_bundle_backup, custom_headers=None, raw=False, **operation_config):
        """Restores a backed up key to a vault.

        Imports a previously backed up key into Azure Key Vault, restoring the
        key, its key identifier, attributes and access control policies. The
        RESTORE operation may be used to import a previously backed up key.
        Individual versions of a key cannot be restored. The key is restored in
        its entirety with the same key name as it had when it was backed up. If
        the key name is not available in the target Key Vault, the RESTORE
        operation will be rejected. While the key name is retained during
        restore, the final key identifier will change if the key is restored to
        a different vault. Restore will restore all versions and preserve
        version identifiers. The RESTORE operation is subject to security
        constraints: The target Key Vault must be owned by the same Microsoft
        Azure Subscription as the source Key Vault The user must have RESTORE
        permission in the target Key Vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_bundle_backup: The backup blob associated with a key
         bundle.
        :type key_bundle_backup: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyBundle <azure.keyvault.models.KeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyRestoreParameters(key_bundle_backup=key_bundle_backup)

        # Construct URL
        url = '/keys/restore'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyRestoreParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def encrypt(
            self, vault_base_url, key_name, key_version, algorithm, value, custom_headers=None, raw=False, **operation_config):
        """Encrypts an arbitrary sequence of bytes using an encryption key that is
        stored in a key vault.

        The ENCRYPT operation encrypts an arbitrary sequence of bytes using an
        encryption key that is stored in Azure Key Vault. Note that the ENCRYPT
        operation only supports a single block of data, the size of which is
        dependent on the target key and the encryption algorithm to be used.
        The ENCRYPT operation is only strictly necessary for symmetric keys
        stored in Azure Key Vault since protection with an asymmetric key can
        be performed using public portion of the key. This operation is
        supported for asymmetric keys as a convenience for callers that have a
        key-reference but do not have access to the public key material.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param key_version: The version of the key.
        :type key_version: str
        :param algorithm: algorithm identifier. Possible values include:
         'RSA-OAEP', 'RSA-OAEP-256', 'RSA1_5'
        :type algorithm: str or :class:`JsonWebKeyEncryptionAlgorithm
         <azure.keyvault.models.JsonWebKeyEncryptionAlgorithm>`
        :param value:
        :type value: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyOperationResult
         <azure.keyvault.models.KeyOperationResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyOperationsParameters(algorithm=algorithm, value=value)

        # Construct URL
        url = '/keys/{key-name}/{key-version}/encrypt'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyOperationsParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyOperationResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def decrypt(
            self, vault_base_url, key_name, key_version, algorithm, value, custom_headers=None, raw=False, **operation_config):
        """Decrypts a single block of encrypted data.

        The DECRYPT operation decrypts a well-formed block of ciphertext using
        the target encryption key and specified algorithm. This operation is
        the reverse of the ENCRYPT operation; only a single block of data may
        be decrypted, the size of this block is dependent on the target key and
        the algorithm to be used. The DECRYPT operation applies to asymmetric
        and symmetric keys stored in Azure Key Vault since it uses the private
        portion of the key.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param key_version: The version of the key.
        :type key_version: str
        :param algorithm: algorithm identifier. Possible values include:
         'RSA-OAEP', 'RSA-OAEP-256', 'RSA1_5'
        :type algorithm: str or :class:`JsonWebKeyEncryptionAlgorithm
         <azure.keyvault.models.JsonWebKeyEncryptionAlgorithm>`
        :param value:
        :type value: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyOperationResult
         <azure.keyvault.models.KeyOperationResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyOperationsParameters(algorithm=algorithm, value=value)

        # Construct URL
        url = '/keys/{key-name}/{key-version}/decrypt'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyOperationsParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyOperationResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def sign(
            self, vault_base_url, key_name, key_version, algorithm, value, custom_headers=None, raw=False, **operation_config):
        """Creates a signature from a digest using the specified key.

        The SIGN operation is applicable to asymmetric and symmetric keys
        stored in Azure Key Vault since this operation uses the private portion
        of the key.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param key_version: The version of the key.
        :type key_version: str
        :param algorithm: The signing/verification algorithm identifier. For
         more information on possible algorithm types, see
         JsonWebKeySignatureAlgorithm. Possible values include: 'PS256',
         'PS384', 'PS512', 'RS256', 'RS384', 'RS512', 'RSNULL'
        :type algorithm: str or :class:`JsonWebKeySignatureAlgorithm
         <azure.keyvault.models.JsonWebKeySignatureAlgorithm>`
        :param value:
        :type value: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyOperationResult
         <azure.keyvault.models.KeyOperationResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeySignParameters(algorithm=algorithm, value=value)

        # Construct URL
        url = '/keys/{key-name}/{key-version}/sign'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeySignParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyOperationResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def verify(
            self, vault_base_url, key_name, key_version, algorithm, digest, signature, custom_headers=None, raw=False, **operation_config):
        """Verifies a signature using a specified key.

        The VERIFY operation is applicable to symmetric keys stored in Azure
        Key Vault. VERIFY is not strictly necessary for asymmetric keys stored
        in Azure Key Vault since signature verification can be performed using
        the public portion of the key but this operation is supported as a
        convenience for callers that only have a key-reference and not the
        public portion of the key.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param key_version: The version of the key.
        :type key_version: str
        :param algorithm: The signing/verification algorithm. For more
         information on possible algorithm types, see
         JsonWebKeySignatureAlgorithm. Possible values include: 'PS256',
         'PS384', 'PS512', 'RS256', 'RS384', 'RS512', 'RSNULL'
        :type algorithm: str or :class:`JsonWebKeySignatureAlgorithm
         <azure.keyvault.models.JsonWebKeySignatureAlgorithm>`
        :param digest: The digest used for signing.
        :type digest: bytes
        :param signature: The signature to be verified.
        :type signature: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyVerifyResult
         <azure.keyvault.models.KeyVerifyResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyVerifyParameters(algorithm=algorithm, digest=digest, signature=signature)

        # Construct URL
        url = '/keys/{key-name}/{key-version}/verify'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyVerifyParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyVerifyResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def wrap_key(
            self, vault_base_url, key_name, key_version, algorithm, value, custom_headers=None, raw=False, **operation_config):
        """Wraps a symmetric key using a specified key.

        The WRAP operation supports encryption of a symmetric key using a key
        encryption key that has previously been stored in an Azure Key Vault.
        The WRAP operation is only strictly necessary for symmetric keys stored
        in Azure Key Vault since protection with an asymmetric key can be
        performed using the public portion of the key. This operation is
        supported for asymmetric keys as a convenience for callers that have a
        key-reference but do not have access to the public key material.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param key_version: The version of the key.
        :type key_version: str
        :param algorithm: algorithm identifier. Possible values include:
         'RSA-OAEP', 'RSA-OAEP-256', 'RSA1_5'
        :type algorithm: str or :class:`JsonWebKeyEncryptionAlgorithm
         <azure.keyvault.models.JsonWebKeyEncryptionAlgorithm>`
        :param value:
        :type value: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyOperationResult
         <azure.keyvault.models.KeyOperationResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyOperationsParameters(algorithm=algorithm, value=value)

        # Construct URL
        url = '/keys/{key-name}/{key-version}/wrapkey'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyOperationsParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyOperationResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def unwrap_key(
            self, vault_base_url, key_name, key_version, algorithm, value, custom_headers=None, raw=False, **operation_config):
        """Unwraps a symmetric key using the specified key that was initially used
        for wrapping that key.

        The UNWRAP operation supports decryption of a symmetric key using the
        target key encryption key. This operation is the reverse of the WRAP
        operation. The UNWRAP operation applies to asymmetric and symmetric
        keys stored in Azure Key Vault since it uses the private portion of the
        key.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key.
        :type key_name: str
        :param key_version: The version of the key.
        :type key_version: str
        :param algorithm: algorithm identifier. Possible values include:
         'RSA-OAEP', 'RSA-OAEP-256', 'RSA1_5'
        :type algorithm: str or :class:`JsonWebKeyEncryptionAlgorithm
         <azure.keyvault.models.JsonWebKeyEncryptionAlgorithm>`
        :param value:
        :type value: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyOperationResult
         <azure.keyvault.models.KeyOperationResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.KeyOperationsParameters(algorithm=algorithm, value=value)

        # Construct URL
        url = '/keys/{key-name}/{key-version}/unwrapkey'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str'),
            'key-version': self._serialize.url("key_version", key_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'KeyOperationsParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyOperationResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_deleted_keys(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List deleted keys in the specified vault. Authorization: Requires the
        keys/list permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedKeyItemPaged
         <azure.keyvault.models.DeletedKeyItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/deletedkeys'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.DeletedKeyItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.DeletedKeyItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def get_deleted_key(
            self, vault_base_url, key_name, custom_headers=None, raw=False, **operation_config):
        """Retrieves the deleted key information plus its attributes.
        Authorization: Requires the keys/get permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key
        :type key_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedKeyBundle
         <azure.keyvault.models.DeletedKeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedkeys/{key-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('DeletedKeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def purge_deleted_key(
            self, vault_base_url, key_name, custom_headers=None, raw=False, **operation_config):
        """Permanently deletes the specified key. aka purges the key.
        Authorization: Requires the keys/purge permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the key
        :type key_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: None
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedkeys/{key-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [204]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        if raw:
            client_raw_response = ClientRawResponse(None, response)
            return client_raw_response

    def recover_deleted_key(
            self, vault_base_url, key_name, custom_headers=None, raw=False, **operation_config):
        """Recovers the deleted key back to its current version under /keys.
        Authorization: Requires the keys/recover permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param key_name: The name of the deleted key
        :type key_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`KeyBundle <azure.keyvault.models.KeyBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedkeys/{key-name}/recover'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'key-name': self._serialize.url("key_name", key_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('KeyBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def set_secret(
            self, vault_base_url, secret_name, value, tags=None, content_type=None, secret_attributes=None, custom_headers=None, raw=False, **operation_config):
        """Sets a secret in a specified key vault.

        The SET operation adds a secret to the Azure Key Vault. If the named
        secret already exists, Azure Key Vault creates a new version of that
        secret.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret.
        :type secret_name: str
        :param value: The value of the secret.
        :type value: str
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param content_type: Type of the secret value such as a password.
        :type content_type: str
        :param secret_attributes: The secret management attributes.
        :type secret_attributes: :class:`SecretAttributes
         <azure.keyvault.models.SecretAttributes>`
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SecretBundle <azure.keyvault.models.SecretBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.SecretSetParameters(value=value, tags=tags, content_type=content_type, secret_attributes=secret_attributes)

        # Construct URL
        url = '/secrets/{secret-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str', pattern='^[0-9a-zA-Z-]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'SecretSetParameters')

        # Construct and send request
        request = self._client.put(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SecretBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def delete_secret(
            self, vault_base_url, secret_name, custom_headers=None, raw=False, **operation_config):
        """Deletes a secret from a specified key vault.

        The DELETE operation applies to any secret stored in Azure Key Vault.
        DELETE cannot be applied to an individual version of a secret.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret.
        :type secret_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedSecretBundle
         <azure.keyvault.models.DeletedSecretBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/secrets/{secret-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('DeletedSecretBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_secret(
            self, vault_base_url, secret_name, secret_version, content_type=None, secret_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Updates the attributes associated with a specified secret in a given
        key vault.

        The UPDATE operation changes specified attributes of an existing stored
        secret. Attributes that are not specified in the request are left
        unchanged. The value of a secret itself cannot be changed.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret.
        :type secret_name: str
        :param secret_version: The version of the secret.
        :type secret_version: str
        :param content_type: Type of the secret value such as a password.
        :type content_type: str
        :param secret_attributes: The secret management attributes.
        :type secret_attributes: :class:`SecretAttributes
         <azure.keyvault.models.SecretAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SecretBundle <azure.keyvault.models.SecretBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.SecretUpdateParameters(content_type=content_type, secret_attributes=secret_attributes, tags=tags)

        # Construct URL
        url = '/secrets/{secret-name}/{secret-version}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str'),
            'secret-version': self._serialize.url("secret_version", secret_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'SecretUpdateParameters')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SecretBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_secret(
            self, vault_base_url, secret_name, secret_version, custom_headers=None, raw=False, **operation_config):
        """Get a specified secret from a given key vault.

        The GET operation is applicable to any secret stored in Azure Key
        Vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret.
        :type secret_name: str
        :param secret_version: The version of the secret.
        :type secret_version: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SecretBundle <azure.keyvault.models.SecretBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/secrets/{secret-name}/{secret-version}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str'),
            'secret-version': self._serialize.url("secret_version", secret_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SecretBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_secrets(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List secrets in a specified key vault.

        The LIST operation is applicable to the entire vault, however only the
        base secret identifier and attributes are provided in the response.
        Individual secret versions are not listed in the response.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SecretItemPaged
         <azure.keyvault.models.SecretItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/secrets'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.SecretItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.SecretItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def get_secret_versions(
            self, vault_base_url, secret_name, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List the versions of the specified secret.

        The LIST VERSIONS operation can be applied to all versions having the
        same secret name in the same key vault. The full secret identifier and
        attributes are provided in the response. No values are returned for the
        secrets and only current versions of a secret are listed.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret.
        :type secret_name: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SecretItemPaged
         <azure.keyvault.models.SecretItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/secrets/{secret-name}/versions'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
                    'secret-name': self._serialize.url("secret_name", secret_name, 'str')
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.SecretItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.SecretItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def get_deleted_secrets(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List deleted secrets in the specified vault. Authorization: requires
        the secrets/list permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedSecretItemPaged
         <azure.keyvault.models.DeletedSecretItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/deletedsecrets'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.DeletedSecretItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.DeletedSecretItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def get_deleted_secret(
            self, vault_base_url, secret_name, custom_headers=None, raw=False, **operation_config):
        """Retrieves the deleted secret information plus its attributes.
        Authorization: requires the secrets/get permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret
        :type secret_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedSecretBundle
         <azure.keyvault.models.DeletedSecretBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedsecrets/{secret-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('DeletedSecretBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def purge_deleted_secret(
            self, vault_base_url, secret_name, custom_headers=None, raw=False, **operation_config):
        """Permanently deletes the specified secret. aka purges the secret.
        Authorization: requires the secrets/purge permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret
        :type secret_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: None
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedsecrets/{secret-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [204]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        if raw:
            client_raw_response = ClientRawResponse(None, response)
            return client_raw_response

    def recover_deleted_secret(
            self, vault_base_url, secret_name, custom_headers=None, raw=False, **operation_config):
        """Recovers the deleted secret back to its current version under /secrets.
        Authorization: requires the secrets/recover permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the deleted secret
        :type secret_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SecretBundle <azure.keyvault.models.SecretBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedsecrets/{secret-name}/recover'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SecretBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def backup_secret(
            self, vault_base_url, secret_name, custom_headers=None, raw=False, **operation_config):
        """Requests that a backup of the specified secret be downloaded to the
        client. Authorization: requires the secrets/backup permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_name: The name of the secret.
        :type secret_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`BackupSecretResult
         <azure.keyvault.models.BackupSecretResult>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/secrets/{secret-name}/backup'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'secret-name': self._serialize.url("secret_name", secret_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('BackupSecretResult', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def restore_secret(
            self, vault_base_url, secret_bundle_backup, custom_headers=None, raw=False, **operation_config):
        """Restores a backed up secret to a vault. Authorization: requires the
        secrets/restore permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param secret_bundle_backup: The backup blob associated with a secret
         bundle.
        :type secret_bundle_backup: bytes
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SecretBundle <azure.keyvault.models.SecretBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.SecretRestoreParameters(secret_bundle_backup=secret_bundle_backup)

        # Construct URL
        url = '/secrets/restore'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'SecretRestoreParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SecretBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_certificates(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List certificates in a specified key vault.

        The GetCertificates operation returns the set of certificates resources
        in the specified key vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateItemPaged
         <azure.keyvault.models.CertificateItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/certificates'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.CertificateItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.CertificateItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def delete_certificate(
            self, vault_base_url, certificate_name, custom_headers=None, raw=False, **operation_config):
        """Deletes a certificate from a specified key vault.

        Deletes all versions of a certificate object along with its associated
        policy. Delete certificate cannot be used to remove individual versions
        of a certificate object.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedCertificateBundle
         <azure.keyvault.models.DeletedCertificateBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/{certificate-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('DeletedCertificateBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def set_certificate_contacts(
            self, vault_base_url, contact_list=None, custom_headers=None, raw=False, **operation_config):
        """Sets the certificate contacts for the specified key vault.

        Sets the certificate contacts for the specified key vault.
        Authorization: requires the certificates/managecontacts permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param contact_list: The contact list for the vault certificates.
        :type contact_list: list of :class:`Contact
         <azure.keyvault.models.Contact>`
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`Contacts <azure.keyvault.models.Contacts>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        contacts = models.Contacts(contact_list=contact_list)

        # Construct URL
        url = '/certificates/contacts'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(contacts, 'Contacts')

        # Construct and send request
        request = self._client.put(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('Contacts', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_certificate_contacts(
            self, vault_base_url, custom_headers=None, raw=False, **operation_config):
        """Lists the certificate contacts for a specified key vault.

        The GetCertificateContacts operation returns the set of certificate
        contact resources in the specified key vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`Contacts <azure.keyvault.models.Contacts>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/contacts'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('Contacts', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def delete_certificate_contacts(
            self, vault_base_url, custom_headers=None, raw=False, **operation_config):
        """Deletes the certificate contacts for a specified key vault.

        Deletes the certificate contacts for a specified key vault certificate.
        Authorization: requires the certificates/managecontacts permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`Contacts <azure.keyvault.models.Contacts>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/contacts'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('Contacts', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_certificate_issuers(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List certificate issuers for a specified key vault.

        The GetCertificateIssuers operation returns the set of certificate
        issuer resources in the specified key vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateIssuerItemPaged
         <azure.keyvault.models.CertificateIssuerItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/certificates/issuers'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.CertificateIssuerItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.CertificateIssuerItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def set_certificate_issuer(
            self, vault_base_url, issuer_name, provider, credentials=None, organization_details=None, attributes=None, custom_headers=None, raw=False, **operation_config):
        """Sets the specified certificate issuer.

        The SetCertificateIssuer operation adds or updates the specified
        certificate issuer.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param issuer_name: The name of the issuer.
        :type issuer_name: str
        :param provider: The issuer provider.
        :type provider: str
        :param credentials: The credentials to be used for the issuer.
        :type credentials: :class:`IssuerCredentials
         <azure.keyvault.models.IssuerCredentials>`
        :param organization_details: Details of the organization as provided
         to the issuer.
        :type organization_details: :class:`OrganizationDetails
         <azure.keyvault.models.OrganizationDetails>`
        :param attributes: Attributes of the issuer object.
        :type attributes: :class:`IssuerAttributes
         <azure.keyvault.models.IssuerAttributes>`
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`IssuerBundle <azure.keyvault.models.IssuerBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameter = models.CertificateIssuerSetParameters(provider=provider, credentials=credentials, organization_details=organization_details, attributes=attributes)

        # Construct URL
        url = '/certificates/issuers/{issuer-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'issuer-name': self._serialize.url("issuer_name", issuer_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameter, 'CertificateIssuerSetParameters')

        # Construct and send request
        request = self._client.put(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('IssuerBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_certificate_issuer(
            self, vault_base_url, issuer_name, provider=None, credentials=None, organization_details=None, attributes=None, custom_headers=None, raw=False, **operation_config):
        """Updates the specified certificate issuer.

        The UpdateCertificateIssuer operation performs an update on the
        specified certificate issuer entity.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param issuer_name: The name of the issuer.
        :type issuer_name: str
        :param provider: The issuer provider.
        :type provider: str
        :param credentials: The credentials to be used for the issuer.
        :type credentials: :class:`IssuerCredentials
         <azure.keyvault.models.IssuerCredentials>`
        :param organization_details: Details of the organization as provided
         to the issuer.
        :type organization_details: :class:`OrganizationDetails
         <azure.keyvault.models.OrganizationDetails>`
        :param attributes: Attributes of the issuer object.
        :type attributes: :class:`IssuerAttributes
         <azure.keyvault.models.IssuerAttributes>`
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`IssuerBundle <azure.keyvault.models.IssuerBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameter = models.CertificateIssuerUpdateParameters(provider=provider, credentials=credentials, organization_details=organization_details, attributes=attributes)

        # Construct URL
        url = '/certificates/issuers/{issuer-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'issuer-name': self._serialize.url("issuer_name", issuer_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameter, 'CertificateIssuerUpdateParameters')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('IssuerBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_certificate_issuer(
            self, vault_base_url, issuer_name, custom_headers=None, raw=False, **operation_config):
        """Lists the specified certificate issuer.

        The GetCertificateIssuer operation returns the specified certificate
        issuer resources in the specified key vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param issuer_name: The name of the issuer.
        :type issuer_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`IssuerBundle <azure.keyvault.models.IssuerBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/issuers/{issuer-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'issuer-name': self._serialize.url("issuer_name", issuer_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('IssuerBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def delete_certificate_issuer(
            self, vault_base_url, issuer_name, custom_headers=None, raw=False, **operation_config):
        """Deletes the specified certificate issuer.

        The DeleteCertificateIssuer operation permanently removes the specified
        certificate issuer from the vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param issuer_name: The name of the issuer.
        :type issuer_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`IssuerBundle <azure.keyvault.models.IssuerBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/issuers/{issuer-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'issuer-name': self._serialize.url("issuer_name", issuer_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('IssuerBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def create_certificate(
            self, vault_base_url, certificate_name, certificate_policy=None, certificate_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Creates a new certificate. .

        If this is the first version, the certificate resource is created.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param certificate_policy: The management policy for the certificate.
        :type certificate_policy: :class:`CertificatePolicy
         <azure.keyvault.models.CertificatePolicy>`
        :param certificate_attributes: The attributes of the certificate
         (optional).
        :type certificate_attributes: :class:`CertificateAttributes
         <azure.keyvault.models.CertificateAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateOperation
         <azure.keyvault.models.CertificateOperation>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.CertificateCreateParameters(certificate_policy=certificate_policy, certificate_attributes=certificate_attributes, tags=tags)

        # Construct URL
        url = '/certificates/{certificate-name}/create'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str', pattern='^[0-9a-zA-Z-]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'CertificateCreateParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [202]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 202:
            deserialized = self._deserialize('CertificateOperation', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def import_certificate(
            self, vault_base_url, certificate_name, base64_encoded_certificate, password=None, certificate_policy=None, certificate_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Imports a certificate into a specified key vault.

        Imports an existing valid certificate, containing a private key, into
        Azure Key Vault. The certificate to be imported can be in either PFX or
        PEM format. If the certificate is in PEM format the PEM file must
        contain the key as well as x509 certificates.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param base64_encoded_certificate: Base64 encoded representation of
         the certificate object to import. This certificate needs to contain
         the private key.
        :type base64_encoded_certificate: str
        :param password: If the private key in base64EncodedCertificate is
         encrypted, the password used for encryption.
        :type password: str
        :param certificate_policy: The management policy for the certificate.
        :type certificate_policy: :class:`CertificatePolicy
         <azure.keyvault.models.CertificatePolicy>`
        :param certificate_attributes: The attributes of the certificate
         (optional).
        :type certificate_attributes: :class:`CertificateAttributes
         <azure.keyvault.models.CertificateAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateBundle
         <azure.keyvault.models.CertificateBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.CertificateImportParameters(base64_encoded_certificate=base64_encoded_certificate, password=password, certificate_policy=certificate_policy, certificate_attributes=certificate_attributes, tags=tags)

        # Construct URL
        url = '/certificates/{certificate-name}/import'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str', pattern='^[0-9a-zA-Z-]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'CertificateImportParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificateBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_certificate_versions(
            self, vault_base_url, certificate_name, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List the versions of a certificate.

        The GetCertificateVersions operation returns the versions of a
        certificate in the specified key vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateItemPaged
         <azure.keyvault.models.CertificateItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/certificates/{certificate-name}/versions'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
                    'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.CertificateItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.CertificateItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def get_certificate_policy(
            self, vault_base_url, certificate_name, custom_headers=None, raw=False, **operation_config):
        """Lists the policy for a certificate.

        The GetCertificatePolicy operation returns the specified certificate
        policy resources in the specified key vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate in a given key
         vault.
        :type certificate_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificatePolicy
         <azure.keyvault.models.CertificatePolicy>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/{certificate-name}/policy'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificatePolicy', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_certificate_policy(
            self, vault_base_url, certificate_name, certificate_policy, custom_headers=None, raw=False, **operation_config):
        """Updates the policy for a certificate.

        Set specified members in the certificate policy. Leave others as null.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate in the given
         vault.
        :type certificate_name: str
        :param certificate_policy: The policy for the certificate.
        :type certificate_policy: :class:`CertificatePolicy
         <azure.keyvault.models.CertificatePolicy>`
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificatePolicy
         <azure.keyvault.models.CertificatePolicy>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/{certificate-name}/policy'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(certificate_policy, 'CertificatePolicy')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificatePolicy', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_certificate(
            self, vault_base_url, certificate_name, certificate_version, certificate_policy=None, certificate_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Updates the specified attributes associated with the given certificate.

        The UpdateCertificate operation applies the specified update on the
        given certificate; note the only elements being updated are the
        certificate's attributes.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate in the given key
         vault.
        :type certificate_name: str
        :param certificate_version: The version of the certificate.
        :type certificate_version: str
        :param certificate_policy: The management policy for the certificate.
        :type certificate_policy: :class:`CertificatePolicy
         <azure.keyvault.models.CertificatePolicy>`
        :param certificate_attributes: The attributes of the certificate
         (optional).
        :type certificate_attributes: :class:`CertificateAttributes
         <azure.keyvault.models.CertificateAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateBundle
         <azure.keyvault.models.CertificateBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.CertificateUpdateParameters(certificate_policy=certificate_policy, certificate_attributes=certificate_attributes, tags=tags)

        # Construct URL
        url = '/certificates/{certificate-name}/{certificate-version}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str'),
            'certificate-version': self._serialize.url("certificate_version", certificate_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'CertificateUpdateParameters')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificateBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_certificate(
            self, vault_base_url, certificate_name, certificate_version, custom_headers=None, raw=False, **operation_config):
        """Gets information about a specified certificate. Authorization: requires
        the certificates/get permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate in the given
         vault.
        :type certificate_name: str
        :param certificate_version: The version of the certificate.
        :type certificate_version: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateBundle
         <azure.keyvault.models.CertificateBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/{certificate-name}/{certificate-version}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str'),
            'certificate-version': self._serialize.url("certificate_version", certificate_version, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificateBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_certificate_operation(
            self, vault_base_url, certificate_name, cancellation_requested, custom_headers=None, raw=False, **operation_config):
        """Updates a certificate operation. Authorization: requires the
        certificates/update permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param cancellation_requested: Indicates if cancellation was requested
         on the certificate operation.
        :type cancellation_requested: bool
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateOperation
         <azure.keyvault.models.CertificateOperation>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        certificate_operation = models.CertificateOperationUpdateParameter(cancellation_requested=cancellation_requested)

        # Construct URL
        url = '/certificates/{certificate-name}/pending'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(certificate_operation, 'CertificateOperationUpdateParameter')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificateOperation', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_certificate_operation(
            self, vault_base_url, certificate_name, custom_headers=None, raw=False, **operation_config):
        """Gets the operation associated with a specified certificate.
        Authorization: requires the certificates/get permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateOperation
         <azure.keyvault.models.CertificateOperation>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/{certificate-name}/pending'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificateOperation', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def delete_certificate_operation(
            self, vault_base_url, certificate_name, custom_headers=None, raw=False, **operation_config):
        """Deletes the operation for a specified certificate. Authorization:
        requires the certificates/update permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateOperation
         <azure.keyvault.models.CertificateOperation>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/certificates/{certificate-name}/pending'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificateOperation', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def merge_certificate(
            self, vault_base_url, certificate_name, x509_certificates, certificate_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Merges a certificate or a certificate chain with a key pair existing on
        the server.

        The MergeCertificate operation performs the merging of a certificate or
        certificate chain with a key pair currently available in the service.
        Authorization: requires the certificates/update permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate.
        :type certificate_name: str
        :param x509_certificates: The certificate or the certificate chain to
         merge.
        :type x509_certificates: list of bytearray
        :param certificate_attributes: The attributes of the certificate
         (optional).
        :type certificate_attributes: :class:`CertificateAttributes
         <azure.keyvault.models.CertificateAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateBundle
         <azure.keyvault.models.CertificateBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.CertificateMergeParameters(x509_certificates=x509_certificates, certificate_attributes=certificate_attributes, tags=tags)

        # Construct URL
        url = '/certificates/{certificate-name}/pending/merge'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'CertificateMergeParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [201]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 201:
            deserialized = self._deserialize('CertificateBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_deleted_certificates(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """Lists the deleted certificates in the specified vault, currently
        available for recovery.

        The GetDeletedCertificates operation retrieves the certificates in the
        current vault which are in a deleted state and ready for recovery or
        purging.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedCertificateItemPaged
         <azure.keyvault.models.DeletedCertificateItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/deletedcertificates'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.DeletedCertificateItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.DeletedCertificateItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def get_deleted_certificate(
            self, vault_base_url, certificate_name, custom_headers=None, raw=False, **operation_config):
        """Retrieves information about the specified deleted certificate.

        The GetDeletedCertificate operation retrieves the deleted certificate
        information plus its attributes, such as retention interval, scheduled
        permanent deletion and the current deletion recovery level.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate
        :type certificate_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`DeletedCertificateBundle
         <azure.keyvault.models.DeletedCertificateBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedcertificates/{certificate-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('DeletedCertificateBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def purge_deleted_certificate(
            self, vault_base_url, certificate_name, custom_headers=None, raw=False, **operation_config):
        """Permanently deletes the specified deleted certificate.

        The PurgeDeletedCertificate operation performs an irreversible deletion
        of the specified certificate, without possibility for recovery. The
        operation is not available if the recovery level does not specify
        'Purgeable'. Requires the explicit granting of the 'purge' permission.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the certificate
        :type certificate_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: None
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedcertificates/{certificate-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [204]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        if raw:
            client_raw_response = ClientRawResponse(None, response)
            return client_raw_response

    def recover_deleted_certificate(
            self, vault_base_url, certificate_name, custom_headers=None, raw=False, **operation_config):
        """Recovers the deleted certificate back to its current version under
        /certificates.

        The RecoverDeletedCertificate operation performs the reversal of the
        Delete operation. The operation is applicable in vaults enabled for
        soft-delete, and must be issued during the retention interval
        (available in the deleted certificate's attributes).

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param certificate_name: The name of the deleted certificate
        :type certificate_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`CertificateBundle
         <azure.keyvault.models.CertificateBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/deletedcertificates/{certificate-name}/recover'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'certificate-name': self._serialize.url("certificate_name", certificate_name, 'str')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('CertificateBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_storage_accounts(
            self, vault_base_url, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List storage accounts managed by specified key vault.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`StorageAccountItemPaged
         <azure.keyvault.models.StorageAccountItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/storage'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True)
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.StorageAccountItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.StorageAccountItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def delete_storage_account(
            self, vault_base_url, storage_account_name, custom_headers=None, raw=False, **operation_config):
        """Deletes a storage account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`StorageBundle <azure.keyvault.models.StorageBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/storage/{storage-account-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('StorageBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_storage_account(
            self, vault_base_url, storage_account_name, custom_headers=None, raw=False, **operation_config):
        """Gets information about a specified storage account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`StorageBundle <azure.keyvault.models.StorageBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/storage/{storage-account-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('StorageBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def set_storage_account(
            self, vault_base_url, storage_account_name, resource_id, active_key_name, auto_regenerate_key, regeneration_period=None, storage_account_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Creates or updates a new storage account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param resource_id: Storage account resource id.
        :type resource_id: str
        :param active_key_name: Current active storage account key name.
        :type active_key_name: str
        :param auto_regenerate_key: whether keyvault should manage the storage
         account for the user.
        :type auto_regenerate_key: bool
        :param regeneration_period: The key regeneration time duration
         specified in ISO-8601 format.
        :type regeneration_period: str
        :param storage_account_attributes: The attributes of the storage
         account.
        :type storage_account_attributes: :class:`StorageAccountAttributes
         <azure.keyvault.models.StorageAccountAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`StorageBundle <azure.keyvault.models.StorageBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.StorageAccountCreateParameters(resource_id=resource_id, active_key_name=active_key_name, auto_regenerate_key=auto_regenerate_key, regeneration_period=regeneration_period, storage_account_attributes=storage_account_attributes, tags=tags)

        # Construct URL
        url = '/storage/{storage-account-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'StorageAccountCreateParameters')

        # Construct and send request
        request = self._client.put(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('StorageBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_storage_account(
            self, vault_base_url, storage_account_name, active_key_name=None, auto_regenerate_key=None, regeneration_period=None, storage_account_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Updates the specified attributes associated with the given storage
        account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param active_key_name: The current active storage account key name.
        :type active_key_name: str
        :param auto_regenerate_key: whether keyvault should manage the storage
         account for the user.
        :type auto_regenerate_key: bool
        :param regeneration_period: The key regeneration time duration
         specified in ISO-8601 format.
        :type regeneration_period: str
        :param storage_account_attributes: The attributes of the storage
         account.
        :type storage_account_attributes: :class:`StorageAccountAttributes
         <azure.keyvault.models.StorageAccountAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`StorageBundle <azure.keyvault.models.StorageBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.StorageAccountUpdateParameters(active_key_name=active_key_name, auto_regenerate_key=auto_regenerate_key, regeneration_period=regeneration_period, storage_account_attributes=storage_account_attributes, tags=tags)

        # Construct URL
        url = '/storage/{storage-account-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'StorageAccountUpdateParameters')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('StorageBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def regenerate_storage_account_key(
            self, vault_base_url, storage_account_name, key_name, custom_headers=None, raw=False, **operation_config):
        """Regenerates the specified key value for the given storage account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param key_name: The storage account key name.

        :type key_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`StorageBundle <azure.keyvault.models.StorageBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters = models.StorageAccountRegenerteKeyParameters(key_name=key_name)

        # Construct URL
        url = '/storage/{storage-account-name}/regeneratekey'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters, 'StorageAccountRegenerteKeyParameters')

        # Construct and send request
        request = self._client.post(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('StorageBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_sas_definitions(
            self, vault_base_url, storage_account_name, maxresults=None, custom_headers=None, raw=False, **operation_config):
        """List storage SAS definitions for the given storage account.
        
        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param maxresults: Maximum number of results to return in a page. If
         not specified the service will return up to 25 results.
        :type maxresults: int
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SasDefinitionItemPaged
         <azure.keyvault.models.SasDefinitionItemPaged>`
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        def internal_paging(next_link=None, raw=False):

            if not next_link:
                # Construct URL
                url = '/storage/{storage-account-name}/sas'
                path_format_arguments = {
                    'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
                    'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$')
                }
                url = self._client.format_url(url, **path_format_arguments)

                # Construct parameters
                query_parameters = {}
                if maxresults is not None:
                    query_parameters['maxresults'] = self._serialize.query("maxresults", maxresults, 'int', maximum=25, minimum=1)
                query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

            else:
                url = next_link
                query_parameters = {}

            # Construct headers
            header_parameters = {}
            header_parameters['Content-Type'] = 'application/json; charset=utf-8'
            if self.config.generate_client_request_id:
                header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
            if custom_headers:
                header_parameters.update(custom_headers)
            if self.config.accept_language is not None:
                header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

            # Construct and send request
            request = self._client.get(url, query_parameters)
            response = self._client.send(
                request, header_parameters, **operation_config)

            if response.status_code not in [200]:
                raise models.KeyVaultErrorException(self._deserialize, response)

            return response

        # Deserialize response
        deserialized = models.SasDefinitionItemPaged(internal_paging, self._deserialize.dependencies)

        if raw:
            header_dict = {}
            client_raw_response = models.SasDefinitionItemPaged(internal_paging, self._deserialize.dependencies, header_dict)
            return client_raw_response

        return deserialized

    def delete_sas_definition(
            self, vault_base_url, storage_account_name, sas_definition_name, custom_headers=None, raw=False, **operation_config):

        """Deletes a SAS definition from a specified storage account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param sas_definition_name: The name of the SAS definition.
        :type sas_definition_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SasDefinitionBundle
         <azure.keyvault.models.SasDefinitionBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/storage/{storage-account-name}/sas/{sas-definition-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$'),
            'sas-definition-name': self._serialize.url("sas_definition_name", sas_definition_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.delete(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SasDefinitionBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def get_sas_definition(
            self, vault_base_url, storage_account_name, sas_definition_name, custom_headers=None, raw=False, **operation_config):

        """Gets information about a SAS definition for the specified storage
        account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param sas_definition_name: The name of the SAS definition.
        :type sas_definition_name: str
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SasDefinitionBundle
         <azure.keyvault.models.SasDefinitionBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        # Construct URL
        url = '/storage/{storage-account-name}/sas/{sas-definition-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$'),
            'sas-definition-name': self._serialize.url("sas_definition_name", sas_definition_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct and send request
        request = self._client.get(url, query_parameters)
        response = self._client.send(request, header_parameters, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SasDefinitionBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def set_sas_definition(
            self, vault_base_url, storage_account_name, sas_definition_name, parameters, sas_definition_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Creates or updates a new SAS definition for the specified storage
        account.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param sas_definition_name: The name of the SAS definition.
        :type sas_definition_name: str
        :param parameters: Sas definition creation metadata in the form of
         key-value pairs.
        :type parameters: dict
        :param sas_definition_attributes: The attributes of the SAS
         definition.
        :type sas_definition_attributes: :class:`SasDefinitionAttributes
         <azure.keyvault.models.SasDefinitionAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SasDefinitionBundle
         <azure.keyvault.models.SasDefinitionBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters1 = models.SasDefinitionCreateParameters(parameters=parameters, sas_definition_attributes=sas_definition_attributes, tags=tags)

        # Construct URL
        url = '/storage/{storage-account-name}/sas/{sas-definition-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$'),
            'sas-definition-name': self._serialize.url("sas_definition_name", sas_definition_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters1, 'SasDefinitionCreateParameters')

        # Construct and send request
        request = self._client.put(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SasDefinitionBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized

    def update_sas_definition(
            self, vault_base_url, storage_account_name, sas_definition_name, parameters=None, sas_definition_attributes=None, tags=None, custom_headers=None, raw=False, **operation_config):
        """Updates the specified attributes associated with the given SAS
        definition.

        :param vault_base_url: The vault name, for example
         https://myvault.vault.azure.net.
        :type vault_base_url: str
        :param storage_account_name: The name of the storage account.
        :type storage_account_name: str
        :param sas_definition_name: The name of the SAS definition.
        :type sas_definition_name: str
        :param parameters: Sas definition update metadata in the form of
         key-value pairs.
        :type parameters: dict
        :param sas_definition_attributes: The attributes of the SAS
         definition.
        :type sas_definition_attributes: :class:`SasDefinitionAttributes
         <azure.keyvault.models.SasDefinitionAttributes>`
        :param tags: Application specific metadata in the form of key-value
         pairs.
        :type tags: dict
        :param dict custom_headers: headers that will be added to the request
        :param bool raw: returns the direct response alongside the
         deserialized response
        :param operation_config: :ref:`Operation configuration
         overrides<msrest:optionsforoperations>`.
        :rtype: :class:`SasDefinitionBundle
         <azure.keyvault.models.SasDefinitionBundle>`
        :rtype: :class:`ClientRawResponse<msrest.pipeline.ClientRawResponse>`
         if raw=true
        :raises:
         :class:`KeyVaultErrorException<azure.keyvault.models.KeyVaultErrorException>`
        """
        parameters1 = models.SasDefinitionUpdateParameters(parameters=parameters, sas_definition_attributes=sas_definition_attributes, tags=tags)

        # Construct URL
        url = '/storage/{storage-account-name}/sas/{sas-definition-name}'
        path_format_arguments = {
            'vaultBaseUrl': self._serialize.url("vault_base_url", vault_base_url, 'str', skip_quote=True),
            'storage-account-name': self._serialize.url("storage_account_name", storage_account_name, 'str', pattern='^[0-9a-zA-Z]+$'),
            'sas-definition-name': self._serialize.url("sas_definition_name", sas_definition_name, 'str', pattern='^[0-9a-zA-Z]+$')
        }
        url = self._client.format_url(url, **path_format_arguments)

        # Construct parameters
        query_parameters = {}
        query_parameters['api-version'] = self._serialize.query("self.api_version", self.api_version, 'str')

        # Construct headers
        header_parameters = {}
        header_parameters['Content-Type'] = 'application/json; charset=utf-8'
        if self.config.generate_client_request_id:
            header_parameters['x-ms-client-request-id'] = str(uuid.uuid1())
        if custom_headers:
            header_parameters.update(custom_headers)
        if self.config.accept_language is not None:
            header_parameters['accept-language'] = self._serialize.header("self.config.accept_language", self.config.accept_language, 'str')

        # Construct body
        body_content = self._serialize.body(parameters1, 'SasDefinitionUpdateParameters')

        # Construct and send request
        request = self._client.patch(url, query_parameters)
        response = self._client.send(
            request, header_parameters, body_content, **operation_config)

        if response.status_code not in [200]:
            raise models.KeyVaultErrorException(self._deserialize, response)

        deserialized = None

        if response.status_code == 200:
            deserialized = self._deserialize('SasDefinitionBundle', response)

        if raw:
            client_raw_response = ClientRawResponse(deserialized, response)
            return client_raw_response

        return deserialized
