/*	Projection_Viewer

PIRL CVS ID: Projection_Viewer.java,v 1.5 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	PIRL.Image_Tools.Projection;
import	PIRL.PVL.Parameter;
import	PIRL.PVL.Parser;
import	PIRL.PVL.PVL_Exception;
import	PIRL.Utilities.Streams;

import	javax.swing.JFrame;
import	javax.swing.JPanel;
import	javax.swing.JLabel;
import	javax.swing.JButton;
import	javax.swing.UIManager;
import	java.awt.Container;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.Insets;
import	java.awt.Point;
import	java.awt.geom.Point2D;
import	java.awt.Dimension;
import	java.awt.event.ActionListener;
import	java.awt.event.ActionEvent;
import	java.io.File;
import	java.io.InputStream;
import	java.io.IOException;
import	java.net.URL;
import	java.net.MalformedURLException;

/**	A <i>Projection_Viewer</i> provides a view of coordinate projection
	mappings.
<p>
	A JFrame containing a Projection_Pane and optionally a Parameter_Pane.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.5
	@see		PIRL.PVL.Parameter
	@see		PIRL.Viewers.Projection_Pane
	@see		PIRL.Viewers.Parameter_Pane
*/
public class Projection_Viewer
	extends JFrame
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Image_Tools.Projection_Viewer (1.5 2012/04/16 06:22:59)";

private Parameter
	Parameters					= null;

private Projection_Pane
	Projection_Panel;

private boolean
	Showing_Parameters;
private JButton
	Show_Parameters_Button;
private Dimension
	Show_Parameters_Button_Dimension;

private Parameter_Pane
	Parameters_Panel;

private int
	Location_Panel_Height;

/**	Exit status on success.
*/
public static final int
	EXIT_SUCCESS				= 0;

/**	Exit status when a command line syntax problem occurs.
*/
public static final int
	EXIT_COMMAND_LINE_SYNTAX	= 1;

/**	Exit status when no parameters can be obtained.
*/
public static final int
	EXIT_NO_PARAMETERS			= 2;

/**	Exit status when no projection can be determined.
*/
public static final int
	EXIT_NO_PROJECTION			= 3;
	

//  DEBUG control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTORS		= 1 << 0,
	DEBUG_ACCESSORS			= 1 << 1,
	DEBUG_ACTIONS			= 1 << 2,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Projection_Viewer from an Aggregate Parameter.
<p>
	@param	parameters	The Aggregate Parameter that contains the
		projection parameters.
	@throws	IllegalArgumentException	If a Projection can not be
		constructed from the parameters.
*/
Projection_Viewer
	(
	Parameter	parameters
	)
{
super ("Projection_Viewer");

if (System.getProperty ("os.name").equals ("Mac OS X"))
	{
	//	Broken OS X TreeTable handles workaround.
	try {UIManager.setLookAndFeel
		(UIManager.getCrossPlatformLookAndFeelClassName ());}
	catch (Exception exception) {/* Just leave the existing LAF. */}
	}

Parameters = parameters;

try {Projection_Panel = new Projection_Pane (Projection.Create (parameters));}
catch (Throwable throwable)
	{
	IllegalArgumentException
		exception = new IllegalArgumentException (ID + '\n'
			+ "Unable to create a Projection\n"
			+ "from the \"" + ((parameters == null) ?
				"null" : parameters.Name ()) + "\" parameters.");
	exception.initCause (throwable);
	throw exception;
	}

setContentPane (Panels ());
pack ();
}

/*==============================================================================
	GUI
*/
private JPanel Panels ()
{
JPanel
	panel = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();

//	Parameters name.
location.anchor		= GridBagConstraints.WEST;
location.fill		= GridBagConstraints.NONE;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.insets		= new Insets (5, 5, 0, 5);
panel.add (new JLabel
	("Parameters: " + ((Parameters == null) ?
		"none" : File_Name (Parameters.Name ()))),
	location);

//	Projection.
location.anchor		= GridBagConstraints.CENTER;
location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.insets		= new Insets (0, 0, 0, 0);
panel.add (Projection_Panel, location);

//	Parameters.
location.anchor		= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
location.insets		= new Insets (0, 5, 5, 5);
panel.add (Show_Parameters_Button = new JButton ("Show Parameters"), location);
Showing_Parameters = false;
Show_Parameters_Button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{Show_Parameters (! Showing_Parameters);}});
Show_Parameters_Button_Dimension = Show_Parameters_Button.getPreferredSize ();

String
	name = null;
if (Parameters.Name ().equals ("-"))
	{
	Parameters.Name (Parser.CONTAINER_NAME);
	name = "-";
	}
//	Don't show the parameters yet.
Parameters_Panel = new Parameter_Pane (Parameters);
if (name != null)
	Parameters.Name (name);

//	Size the panel to the width of the Parameters_Panel; retain the height.
Dimension
	dimension = panel.getPreferredSize ();
Location_Panel_Height = dimension.height;
dimension.width = Parameters_Panel.getPreferredSize ().width;
panel.setPreferredSize (dimension);

return panel;
}

/*==============================================================================
	Actions
*/
/**	Select whether the parameters used to construct the Projection
	are shown.
<p>
	@param	show	If true, a Parameter_Pane will be shown at the
		botton of the viewer window. If false, the Parameter_Pane
		will be removed from the display.
*/
public void Show_Parameters
	(
	boolean	show
	)
{
if ((DEBUG & DEBUG_ACTIONS) != 0)
	System.out.println
		(">>> Show_Parameters: " + show);
Showing_Parameters = show;
Container
	container = getContentPane ();
Dimension
	dimension = new Dimension
		(container.getSize ().width, Location_Panel_Height);
if (Showing_Parameters)
	{
	Show_Parameters_Button.setText ("Hide Parameters");
	GridBagConstraints
		location = new GridBagConstraints ();
	location.anchor		= GridBagConstraints.CENTER;
	location.fill		= GridBagConstraints.BOTH;
	location.weightx	= 1.0;
	location.weighty	= 1.0;
	location.gridwidth	= GridBagConstraints.REMAINDER;
	container.add (Parameters_Panel, location);
	dimension.height += Parameters_Panel.getPreferredSize ().height;
	}
else
	{
	Show_Parameters_Button.setText ("Show Parameters");
	container.remove (Parameters_Panel);
	}
Show_Parameters_Button.setPreferredSize (Show_Parameters_Button_Dimension);
container.setPreferredSize (dimension);

//	Without this pack the display will not be redrawn.
pack ();
if ((DEBUG & DEBUG_ACTIONS) != 0)
	System.out.println
		("<<< Show_Parameters");
}

/*==============================================================================
	main
*/
/**	Launches an interactive Projection_Viewer or performs a non-interactive
	coordinate projection.
<p>
	Exit status values:
<p><dl>
	<dt>{@link #EXIT_SUCCESS} (0) - Success.
	<li>{@link #EXIT_COMMAND_LINE_SYNTAX} (1) - A command line syntax problem occured.
	<li>{@link #EXIT_NO_PARAMETERS} (2) - No parameters could be obtained.
	<li>{@link #EXIT_NO_PROJECTION} (3) - No projection could be determined.
</ul><p>
	@param	args	The {@link #Usage command line} arguments.
	@see	#Usage()
*/
public static void main
	(
	String[]	args
	)
{
if (args.length == 0)
	Usage ();

String
	source = null;
boolean
	image_coordinates = false,
	spherical = false;
Point2D.Double
	coordinate = null;

for (int
		count = 0;
		count < args.length;
		count++)
	{
	if (args[count].length () == 0)
		continue;
	if (args[count].charAt (0) == '-' &&
		args[count].length () > 1)
		{
		switch (args[count].charAt (1))
			{
			case 'S':
			case 's':
				spherical = true;
				break;

			case 'I':	//	Image x,y
			case 'i':
				image_coordinates = true;

			case 'W':	//	World lon,lat
			case 'w':
				if ((count + 1) == args.length ||
					coordinate != null)
					Usage ();
				coordinate = Coordinate (args[++count]);
				break;

			default:
				System.out.println ("Unknown option: " + args[count]);
				Usage ();
			}
		}
	else
		{
		if (source != null)
			{
			System.out.println
				("Mulitple parameters source -\n"
				+"  " + source + '\n'
				+"  " + args[count] + '\n');
			Usage ();
			}
		source = args[count];
		}
	}
if (source == null)
	{
	System.out.println
		("No parameters source specified.\n");
	Usage ();
	}

//	Assemble the source parameters.
Parameter
	parameters = Named_Parameters (source);
if (parameters == null)
	System.exit (EXIT_NO_PARAMETERS);

//	Construct a Projection from the parameters.
Projection
	projection = null;
try {projection = Projection.Create (parameters);}
catch (Throwable exception)
	{
	System.out.println (exception.getMessage ());
	System.exit (EXIT_NO_PROJECTION);
	}
//	Set the elliptical/spherical mode for polar projection.
projection.Use_Spherical (spherical);

if (coordinate == null)
	{
	//	Interactive.
	Projection_Viewer
		viewer = new Projection_Viewer (parameters);
	viewer.setDefaultCloseOperation (JFrame.EXIT_ON_CLOSE);
	viewer.setVisible (true);
	}
else
	{
	//	Non-interactive coordinate projection.
	System.out.println ("      Projection: " + projection.Name ());
	if (image_coordinates)
		{
		System.out.println ("Image coordinate: "
			+ (int)coordinate.x + " x, " + (int)coordinate.y + " y");
		coordinate = projection.to_World (coordinate);
		System.out.println ("World coordinate: "
			+ coordinate.x + " lon, " + coordinate.y + " lat");
		}
	else
		{
		System.out.println ("World coordinate: "
			+ coordinate.x + " lon, " + coordinate.y + " lat");
		Point
			image_coordinate = projection.to_Image (coordinate);
		System.out.println ("Image coordinate: "
			+ image_coordinate.x + " x, " + image_coordinate.y + " y");
		}
	System.exit (EXIT_SUCCESS);
	}
}


private static Point2D.Double Coordinate
	(
	String	coordinate
	)
{
int
	index = coordinate.indexOf (',');
if (index < 0)
	Usage ();
Point2D.Double
	point = null;
try
	{
	point = new Point2D.Double
		(
		Double.parseDouble (coordinate.substring (0, index)),
		Double.parseDouble (coordinate.substring (index + 1))
		);
	}
catch (NumberFormatException exception)
	{
	System.out.println
		("Not a valid coordinate: " + coordinate);
	Usage ();
	}
return point;
}

/**	Assemble a Paremeter from a source.
<p>
	If an InputStream can not be constructed on the source a message
	that the source could not be accessed is printed to stderr. If the
	source is the dash ('-') character stdin will be used. If a
	Parameter can not be constructed from the InputStream a message
	describing the problem is printed to stderr.
<p>
	The name of the returned Parameter will be the source String. If
	a URL can not be constructed from the source String, or it a URL can
	be formed and it is found to have the "file" protocol,
	the full canonical pathname of the referenced file is set as the
	comment for the returned Parameter. If, however, the source String is
	the dash character no comment is applied.
<p>
	@param	source	The parameter source reference String. This may be
		a file pathname, a URL, or the dash ('-') character to indicate
		that the source is stdin.
	@return	A Parameter. This will be null if no Parameter could be
		constructed from the source.
*/
public static Parameter Named_Parameters
	(
	String	source
	)
{
String
	canonical_pathname = null;
InputStream
	source_stream;
if (source.equals ("-"))
	source_stream = System.in;
else if ((source_stream = Streams.Get_Stream (source))
		!= null)
	{
	//	Identify the source file.
	try
		{
		URL
			url = new URL (source);
		if (url.getProtocol ().equals ("file"))
			{
			try {canonical_pathname =
					new File (url.getPath ()).getCanonicalPath ();}
			catch (Exception e) {}
			}
		}
	catch (MalformedURLException except)
		{
		try {canonical_pathname =
				new File (source).getCanonicalPath ();}
		catch (Exception e) {}
		}
	}
else
	{
	System.err.println
		("Unable to access source: " + source);
	return null;
	}

//	Load the parameters.
Parameter
	parameters = null;
try {parameters = new Parameter (source, new Parser (source_stream))
		.Comments (canonical_pathname);}
catch (PVL_Exception exception)
	{
	System.err.println
		("Unable to parse the PVL in "
			+ source + '\n'
		+ exception.getMessage () + '\n');
	}
return parameters;
}

private static String File_Name
	(
	String	source
	)
{
if (source == null ||
	source.trim ().length () == 0 ||
	source.equals (Parser.CONTAINER_NAME))
	return "";
if (source.equals ("-"))
	return "stdin";
File
	file = null;
try {file = new File (new URL (source).getPath ());}
catch (MalformedURLException except)
	{file = new File (source);}
return file.getName ();
}

/**	Print command line usage syntax.
<p>
<blockquote><pre>
Usage: <b>Projection_Viewer</b> &lt;<i>Switches</i>&gt; &lt;<i>PVL source</i>&gt;
&nbsp;&nbsp;Switches -
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>S</u></b>pherical
&nbsp;&nbsp;&nbsp;&nbsp;<b>-<u>I</u></b>mage &lt;<i>x</i>&gt;,&lt;<i>y</i>&gt; | -<b>W</b>orld &lt;<i>longitude</i>&gt;,&lt;<i>latitude</i>&gt;]
</pre></blockquote>
<p>
<h4>
	PVL source:
</h4><p>
	The source of PVL parameters containing the map projection information.
	This may be a file pathname, a URL, or the dash ('-') character to
	specify that stdin is the source.
<p>
<h4>
	Spherical:
</h4><p>
	When a polar stereographic projection is selected the faster but
	slightly less accurate spherical form of the projection will be used
	when this option is specified. Otherwise the elliptical form will be
	used.
<h4>
	Non-interactive coordinate projection:
</h4><p>
	When either an image x,y or real world longitude,latitude coordinate
	is specified the projection determined by the source parameters is
	applied to map the coordinate to the other reference system. The name
	of the selected projection and both sets of coordinates are printed.
<p>
	<b>N.B.</b>This method always results in a System.exit with a status
	of {@link #EXIT_COMMAND_LINE_SYNTAX}.
*/
public static void Usage ()
{
System.out.println
	(
	"Usage: Projection_Viewer"
	+ " [-Spherical]"
	+ " [-Image <x>,<y> | -World <longitude>,<latitude>]"
	+ " <PVL source>"
	);
System.exit (EXIT_COMMAND_LINE_SYNTAX);
}

}
