# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""maas-test utilities."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

__metaclass__ = type
__all__ = [
    "BINARY",
    "binary_content",
    "DEFAULT_STATE_DIR",
    "determine_vm_series",
    "determine_vm_architecture",
    "get_uri",
    "DEFAULT_PIDFILE_DIR",
    "read_file",
    "retries",
    "run_command",
    "CasesLoader",
    ]


import inspect
from io import BytesIO
import logging
from pipes import quote
import platform
import re
from subprocess import (
    PIPE,
    Popen,
    )
from time import (
    sleep,
    time,
    )
import unittest

import distro_info
from lxml import etree
from six.moves import input
from testtools.content import Content
from testtools.content_type import ContentType


# Default location for maas-test state, such as SSH keys for the virtual
# machine, and the http proxy cache.
DEFAULT_STATE_DIR = '/var/cache/maas-test'

# Default location for pidfiles.
DEFAULT_PIDFILE_DIR = '/run/maas-test'

# Default location for log files.
DEFAULT_LOG_DIR = '/var/log/maas-test'


def read_file(path):
    """Return a given file's contents, as `bytes`."""
    with open(path, "rb") as f:
        return f.read()


def make_exception(args, retcode, stdout, stderr):
    """Create an exception from the information about a failed command."""
    cmd = " ".join(quote(arg) for arg in args)
    return Exception(
        "Command '%s' failed (%d):\n%s\n%s" % (
            cmd, retcode,
            stdout.decode('utf8', errors='replace'),
            stderr.decode('utf8', errors='replace')))


def run_command(args, input=None, check_call=False):
    """A wrapper to Popen to run commands in the command-line."""
    process = Popen(args, stdout=PIPE, stderr=PIPE, stdin=PIPE, shell=False)
    stdout, stderr = process.communicate(input)
    retcode = process.returncode
    if check_call and retcode != 0:
        raise make_exception(args, retcode, stdout, stderr)
    return retcode, stdout, stderr


def get_uri(path):
    """GET an API V1 URI.

    :return: The API URI.
    """
    api_root = '/api/1.0/'
    return api_root + path


BINARY = ContentType("application", "octet-stream")


def binary_content(data):
    """Create a `Content` object from a byte string.

    This is useful for adding details which are blobs of binary data.
    """
    return Content(BINARY, lambda: [data])


def retries(timeout=30, delay=1):
    """Helper for retrying something, sleeping between attempts.

    Yields ``(elapsed, remaining)`` tuples, giving times in seconds.

    @param timeout: From now, how long to keep iterating, in seconds.
    @param delay: The sleep between each iteration, in seconds.
    """
    start = time()
    end = start + timeout
    for now in iter(time, None):
        if now < end:
            yield now - start, end - now
            sleep(min(delay, end - now))
        else:
            break


class UnknownCPUArchitecture(RuntimeError):
    """Could not determine CPU architecture."""


def determine_vm_architecture():
    """Figure out CPU architecture for the virtual machine that will run MAAS.

    :return: Architecture string, e.g. 'i386'.
    :raise UnknownCPUArchitecture: if the CPU architecture could not be
        determined.
    """
    # Ubuntu sometimes uses different architecture names than we get from
    # the platform module.  This dict translates for those cases.
    ubuntu_names = {
        'i686': 'i386',
        'x86_64': 'amd64',
    }
    raw_arch = platform.machine()
    if raw_arch == '':
        raise UnknownCPUArchitecture(
            "Could not determine system CPU architecture.  "
            "Please report a bug against http://launchpad.net/maas-test "
            "with this system's details.  To work around the problem, "
            "try the --vm-arch option.")
    return ubuntu_names.get(raw_arch, raw_arch)


def determine_vm_series():
    """Figure out the Ubuntu release series to run MAAS on.

    :return: Series codename, e.g. "precise" for 12.04 Precise Pangolin.
    """
    distro, version, series = platform.linux_distribution()
    return series


def get_supported_node_series():
    """List of MAAS-supported Ubuntu series.

    These are the series that MAAS can deploy on the nodes
    it manages.
    """
    # Return all series above and including Precise.
    return _get_supported_series('precise')


def get_supported_maas_series():
    """List of Ubuntu series for MAAS.

    These are the series on which maas-test can install a
    a MAAS instance.
    """
    # Return all series above and including Trusty.
    return _get_supported_series('trusty')


def _get_supported_series(cut_off_series):
    """Return the list of supported series, starting from the given
    `cut_off_series`.
    """
    udi = distro_info.UbuntuDistroInfo()
    supported = udi.supported(result="codename")
    cut_off_index = (
        supported.index(cut_off_series) if cut_off_series in supported else 0)
    return supported[cut_off_index:]


class CasesLoader(unittest.TestLoader):
    """A test loader specialised for loading maas-test cases."""

    sortTestMethodsUsing = property(
        doc="Unused; sorting is implemented in getTestCaseNames()")

    def getTestCaseNames(self, testCaseClass):
        """Return test case names in order of line number.

        This assumes that all test methods have been defined in the
        *same file*, and in the order they should be run.
        """
        def getlineno(method):
            _, lineno = inspect.findsource(method)
            return lineno

        method_starts = {
            name: getlineno(method) for name, method in
            inspect.getmembers(testCaseClass, inspect.ismethod)
            if name.startswith(self.testMethodPrefix)
        }

        return sorted(method_starts, key=method_starts.get)


def extract_mac_ip_mapping(nmap_xml_scan):
    """Extract the IP->MAC address mapping from the result of a nmap scan.

    The returned mapping uses uppercase MAC addresses.

    :param nmap_xml_scan: The XML result of scanning a network using nmap.
        Scanning a network using nmap is typically done by running:
        `nmap -sP <network_definition> -oX -`.
    :type nmap_xml_scan: string
    """
    parser = etree.XMLParser(remove_blank_text=True)
    xml_doc = etree.fromstring(nmap_xml_scan, parser)
    hosts = xml_doc.xpath("/nmaprun/host")
    mapping = {}
    for host in hosts:
        ips = host.xpath("address[@addrtype='ipv4']/@addr")
        macs = host.xpath("address[@addrtype='mac']/@addr")
        if len(ips) == 1 and len(macs) == 1:
            mapping[macs[0].upper()] = ips[0]
    return mapping


def mipf_arch_list(architecture):
    """Return the architectures that must be downloaded by m-i-p-f.

    This converts an architecture name (e.g. 'amd64' or 'armhf/generic') into
    the list of architectures that m-i-p-f must download in a format that
    this script understands.
    More precisely, this means doing two things:
    - default the subarchitecture 'generic' if not specified.
    - work around bug 1181334 by including 'i386/generic' in the list of
      architectures to be downloaded if it's not explicitely specified.
    """
    # If subarchitecture is not provided, assume 'generic'.
    if '/' not in architecture:
        architecture = '%s/generic' % architecture
    arch_list = [architecture]
    # Include i386, see
    # https://bugs.launchpad.net/ubuntu/+source/maas/+bug/1181334
    if architecture != "i386/generic":
        arch_list.append("i386/generic")
    return arch_list


def check_kvm_ok():
    """Check that this machine is capable of running KVM.

    Uses ``sudo`` to run ``kvm-ok`` as root. From kvm_ok(1):

      If running as root, it will check your CPU's MSRs to see if VT is
      disabled in the BIOS.

    :return: True if KVM extensions are found, False otherwise.

    """
    logging.info("Checking for KVM extensions.")
    retcode, stdout, _ = run_command(['sudo', 'kvm-ok'], check_call=True)
    if retcode != 0:
        logging.debug(stdout.strip())
        return False
    else:
        return True


def extract_package_version(policy):
    """Get the version of the package to be installed from policy settings.

    The given policy is assumed to be of the form returned by running:
    `apt-cache policy <packagename>`.  This method returns None if the
    given policy string cannot be parsed.
    """
    match = re.search('Candidate: (.*)\n', policy)
    if match is not None:
        return match.group(1)
    else:
        return None


def virtualization_type():
    """Returns the name of the virtualization type of this machine.

    :return: the name of the virtualization type if the machine
        is running on virtual hardware.  None otherwise.
    """
    logging.info("Checking for virtualised hardware...")
    retcode, stdout, _ = run_command(['sudo', 'virt-what'], check_call=True)
    if stdout == '':
        # The hardware is physical.
        return None
    return stdout.strip()


def get_user_input(message):
    """A wrapper around raw_input() that allows us to sanely mock it.
    """
    return input(message)


class CachingOutputStream:
    """An object that caches output sent to a stream."""

    def __init__(self, stream):
        """Create a CachingOutputStream.

        :param stream: A file-like object to which content should be
            written after caching.
        """
        self.cache = BytesIO()
        self.stream = stream

    def __getattr__(self, attr):
        """Pass-through any attribute ther than write().

        :param attr: The attribute to return. This will be looked up on
            and returned from the output stream to which this
            CachingOutputStream writes.
        :type attr: string
        """
        return getattr(self.stream, attr)

    def write(self, output):
        """Write to the internal cache and to the output stream.

        :param output: The content to write to store in the
            CachingOutputStream's cache and then write to the output
            stream.
        :type output: string
        """
        self.cache.write(output.encode('utf-8'))
        self.stream.write(output)


def compose_filter(key, values):
    """Create a simplestreams filter string.

    The filter string that is returned performs a regex match of `key` against
    given literal `values`, e.g. "arch~(i386|amd64|armhf)" (for key 'arch'
    with values 'i386' etc.).

    :param key: The simplestreams key that is to be filtered.
    :param values: Iterable of strings.  Any of these literal values for `key`
        will match the simplestream filter; nothing else will.
    :return: A regex string suitable for passing to simplestreams.
    """
    return "%s~(%s)" % (
        key,
        '|'.join(re.escape(literal) for literal in values),
        )
