/*
 * Hydrogen
 * Copyright(c) 2002-2008 by Alex >Comix< Cominu [comix@users.sourceforge.net]
 * Copyright(c) 2008-2021 The hydrogen development team [hydrogen-devel@lists.sourceforge.net]
 *
 * http://www.hydrogen-music.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see https://www.gnu.org/licenses
 *
 */

#ifndef H2C_EXPONENTIAL_TABLES_H
#define H2C_EXPONENTIAL_TABLES_H

static inline float compute_exponant( const float input, const float* table, const int table_size )
{
	int idx = ( int )( input * table_size );
	if ( idx < 0 ) {
		idx = 0;
	} else if ( idx >= table_size ) {
		idx = table_size-1;
	}
	return ( table[idx] * input ) / ( ( float )(idx+1) / ( float )table_size );
};

static int concave_exponant_table_size = 4096;
static float concave_exponant_table[4096] = {
	1.5156104658808E-10, 9.9740596827554E-10, 3.0028776133939E-09, 6.5638149639822E-09,
	1.2038851488192E-08, 1.9761595218725E-08, 3.0047077767325E-08, 4.3195717944104E-08,
	5.9495986363365E-08, 7.9226309106588E-08, 1.02656489298E-07, 1.3004880513507E-07,
	1.6165888092569E-07, 1.9773639315002E-07, 2.3852565311928E-07, 2.8426609509032E-07,
	3.3519269046365E-07, 3.9153630318046E-07, 4.5352399763331E-07, 5.213793077196E-07,
	5.9532247372657E-07, 6.7557065230827E-07, 7.6233810374495E-07, 8.5583635986203E-07,
	9.5627437535876E-07, 1.0638586648075E-06, 1.1787934271983E-06, 1.3012806595954E-06,
	1.4315202612247E-06, 1.5697101291111E-06, 1.7160462462169E-06, 1.8707227629012E-06,
	2.0339320724036E-06, 2.2058648809638E-06, 2.3867102731093E-06, 2.5766557725752E-06,
	2.7758873992656E-06, 2.9845897226155E-06, 3.2029459116722E-06, 3.4311377821785E-06,
	3.6693458409091E-06, 3.9177493274853E-06, 4.1765262538692E-06, 4.4458534417183E-06,
	4.7259065577629E-06, 5.0168601473543E-06, 5.3188876663161E-06, 5.6321615112198E-06,
	5.956853048194E-06, 6.2931326403679E-06, 6.64116967404E-06, 7.0011325836549E-06,
	7.3731888756662E-06, 7.7575051513535E-06, 8.154247128661E-06, 8.5635796631137E-06,
	8.9856667678688E-06, 9.420671632952E-06, 9.8687566437245E-06, 1.0330083398626E-05,
	1.0804812726233E-05, 1.1293104701668E-05, 1.1795118662398E-05, 1.2311013223455E-05,
	1.2840946292098E-05, 1.3385075081963E-05, 1.3943556126712E-05, 1.4516545293213E-05,
	1.5104197794269E-05, 1.5706668200924E-05, 1.6324110454361E-05, 1.6956677877416E-05,
	1.7604523185716E-05, 1.826779849847E-05, 1.8946655348926E-05, 1.9641244694497E-05,
	2.0351716926584E-05, 2.1078221880112E-05, 2.182090884277E-05, 2.2579926563991E-05,
	2.3355423263671E-05, 2.4147546640639E-05, 2.4956443880884E-05, 2.5782261665568E-05,
	2.6625146178798E-05, 2.7485243115207E-05, 2.8362697687318E-05, 2.9257654632723E-05,
	3.0170258221064E-05, 3.1100652260842E-05, 3.2048980106047E-05, 3.301538466262E-05,
	3.4000008394759E-05, 3.5002993331058E-05, 3.6024481070506E-05, 3.7064612788337E-05,
	3.8123529241735E-05, 3.9201370775412E-05, 4.029827732705E-05, 4.1414388432615E-05,
	4.2549843231552E-05, 4.3704780471863E-05, 4.4879338515063E-05, 4.6073655341034E-05,
	4.7287868552767E-05, 4.8522115381001E-05, 4.9776532688759E-05, 5.1051256975792E-05,
	5.2346424382923E-05, 5.3662170696303E-05, 5.4998631351575E-05, 5.6355941437951E-05,
	5.7734235702213E-05, 5.9133648552623E-05, 6.0554314062759E-05, 6.1996365975273E-05,
	6.3459937705577E-05, 6.4945162345451E-05, 6.6452172666588E-05, 6.7981101124065E-05,
	6.9532079859749E-05, 7.1105240705638E-05, 7.2700715187144E-05, 7.4318634526304E-05,
	7.5959129644943E-05, 7.7622331167772E-05, 7.9308369425429E-05, 8.1017374457472E-05,
	8.2749476015308E-05, 8.450480356508E-05, 8.6283486290497E-05, 8.8085653095613E-05,
	8.9911432607564E-05, 9.1760953179253E-05, 9.363434289199E-05, 9.5531729558084E-05,
	9.7453240723402E-05, 9.9399003669869E-05, 0.00010136914541794, 0.00010336379272903,
	0.00010538307210788, 0.00010742710980493, 0.00010949603181864, 0.00011158996389769,
	0.00011370903154331, 0.00011585336001143, 0.00011802307431483, 0.00012021829922532,
	0.00012243915927582, 0.00012468577876239, 0.00012695828174634, 0.00012925679205617,
	0.00013158143328955, 0.00013393232881532, 0.00013630960177533, 0.0001387133750864,
	0.0001411437714421, 0.00014360091331466, 0.00014608492295674, 0.00014859592240318,
	0.00015113403347283, 0.00015369937777019, 0.00015629207668719, 0.00015891225140483,
	0.00016156002289484, 0.00016423551192134, 0.0001669388390424, 0.00016967012461169,
	0.00017242948878002, 0.00017521705149686, 0.00017803293251191, 0.00018087725137658,
	0.0001837501274455, 0.00018665167987795, 0.00018958202763931, 0.00019254128950253,
	0.00019552958404948, 0.00019854702967239, 0.0002015937445752, 0.00020466984677489,
	0.00020777545410288, 0.00021091068420629, 0.00021407565454928, 0.00021727048241431,
	0.00022049528490343, 0.00022375017893956, 0.00022703528126767, 0.00023035070845605,
	0.00023369657689752, 0.00023707300281058, 0.00024048010224068, 0.00024391799106127,
	0.00024738678497506, 0.00025088659951509, 0.0002544175500459, 0.00025797975176461,
	0.00026157331970202, 0.00026519836872374, 0.0002688550135312, 0.00027254336866277,
	0.00027626354849477, 0.00028001566724253, 0.00028379983896141, 0.0002876161775478,
	0.00029146479674014, 0.00029534581011993, 0.00029925933111265, 0.0003032054729888,
	0.0003071843488648, 0.000311196071704, 0.00031524075431755, 0.00031931850936538,
	0.0003234294493571, 0.0003275736866529, 0.00033175133346446, 0.00033596250185584,
	0.00034020730374436, 0.00034448585090146, 0.00034879825495356, 0.00035314462738292,
	0.00035752507952849, 0.00036193972258674, 0.00036638866761246, 0.00037087202551964,
	0.00037538990708219, 0.00037994242293482, 0.0003845296835738, 0.00038915179935773,
	0.00039380888050836, 0.00039850103711131, 0.00040322837911686, 0.0004079910163407,
	0.00041278905846467, 0.0004176226150375, 0.00042249179547557, 0.00042739670906357,
	0.00043233746495528, 0.00043731417217426, 0.00044232693961454, 0.00044737587604133,
	0.00045246109009173, 0.00045758269027536, 0.00046274078497509, 0.00046793548244769,
	0.00047316689082449, 0.00047843511811206, 0.00048374027219282, 0.00048908246082574,
	0.00049446179164693, 0.00049987837217032, 0.00050533230978826, 0.00051082371177214,
	0.00051635268527304, 0.00052191933732228, 0.00052752377483211, 0.00053316610459623,
	0.00053884643329044, 0.00054456486747321, 0.00055032151358626, 0.00055611647795516,
	0.00056194986678986, 0.00056782178618532, 0.00057373234212203, 0.00057968164046658,
	0.00058566978697224, 0.00059169688727947, 0.00059776304691649, 0.0006038683712998,
	0.00061001296573476, 0.00061619693541606, 0.00062242038542831, 0.00062868342074651,
	0.00063498614623659, 0.00064132866665593, 0.00064771108665387, 0.00065413351077218,
	0.00066059604344562, 0.00066709878900238, 0.00067364185166462, 0.00068022533554893,
	0.0006868493446668, 0.00069351398292516, 0.00070021935412679, 0.00070696556197084,
	0.00071375271005326, 0.00072058090186731, 0.00072745024080399, 0.0007343608301525,
	0.00074131277310071, 0.0007483061727356, 0.0007553411320437, 0.00076241775391155,
	0.00076953614112613, 0.00077669639637532, 0.00078389862224827, 0.00079114292123592,
	0.00079842939573135, 0.00080575814803024, 0.00081312928033128, 0.0008205428947366,
	0.00082799909325218, 0.00083549797778822, 0.00084303965015963, 0.00085062421208635,
	0.0008582517651938, 0.00086592241101328, 0.00087363625098232, 0.00088139338644514,
	0.00088919391865298, 0.00089703794876453, 0.00090492557784626, 0.00091285690687288,
	0.00092083203672763, 0.00092885106820274, 0.00093691410199974, 0.00094502123872984,
	0.00095317257891433, 0.00096136822298491, 0.00096960827128409, 0.00097789282406548,
	0.00098622198149424, 0.00099459584364735, 0.001003014510514, 0.001011478081996,
	0.001019986657908, 0.0010285403379778, 0.0010371392218471, 0.0010457834090712,
	0.0010544729991198, 0.0010632080913772, 0.0010719887851427, 0.0010808151796308,
	0.0010896873739715, 0.0010986054672109, 0.0011075695583113, 0.0011165797461514,
	0.0011256361295269, 0.0011347388071508, 0.0011438878776535, 0.0011530834395831,
	0.0011623255914062, 0.0011716144315074, 0.0011809500581905, 0.001190332569678,
	0.0011997620641119, 0.0012092386395539, 0.0012187623939855, 0.0012283334253088,
	0.0012379518313459, 0.0012476177098403, 0.0012573311584563, 0.0012670922747797,
	0.001276901156318, 0.0012867579005008, 0.0012966626046798, 0.0013066153661293,
	0.0013166162820465, 0.0013266654495518, 0.0013367629656887, 0.0013469089274247,
	0.001357103431651, 0.0013673465751831, 0.0013776384547609, 0.0013879791670492,
	0.0013983688086377, 0.0014088074760413, 0.0014192952657007, 0.001429832273982,
	0.0014404185971777, 0.0014510543315065, 0.0014617395731136, 0.0014724744180711,
	0.0014832589623782, 0.0014940933019612, 0.0015049775326744, 0.0015159117502995,
	0.0015268960505465, 0.0015379305290537, 0.0015490152813879, 0.0015601504030447,
	0.0015713359894489, 0.0015825721359542, 0.0015938589378443, 0.0016051964903323,
	0.0016165848885615, 0.0016280242276053, 0.0016395146024676, 0.001651056108083,
	0.001662648839317, 0.0016742928909662, 0.0016859883577586, 0.001697735334354,
	0.0017095339153436, 0.0017213841952511, 0.0017332862685321, 0.0017452402295748,
	0.0017572461727003, 0.0017693041921624, 0.0017814143821481, 0.0017935768367776,
	0.0018057916501051, 0.0018180589161181, 0.0018303787287384, 0.0018427511818219,
	0.0018551763691589, 0.0018676543844743, 0.0018801853214278, 0.0018927692736144,
	0.001905406334564, 0.0019180965977421, 0.0019308401565498, 0.0019436371043241,
	0.0019564875343379, 0.0019693915398006, 0.0019823492138579, 0.0019953606495919,
	0.002008425940022, 0.0020215451781044, 0.0020347184567323, 0.0020479458687367,
	0.0020612275068859, 0.0020745634638863, 0.002087953832382, 0.0021013987049554,
	0.0021148981741272, 0.0021284523323568, 0.0021420612720421, 0.0021557250855201,
	0.0021694438650669, 0.0021832177028978, 0.0021970466911675, 0.0022109309219705,
	0.002224870487341, 0.0022388654792533, 0.0022529159896218, 0.0022670221103013,
	0.0022811839330872, 0.0022954015497153, 0.0023096750518628, 0.0023240045311475,
	0.0023383900791285, 0.0023528317873066, 0.0023673297471238, 0.002381884049964,
	0.0023964947871529, 0.0024111620499586, 0.002425885929591, 0.0024406665172026,
	0.0024555039038886, 0.0024703981806867, 0.0024853494385776, 0.0025003577684852,
	0.0025154232612763, 0.0025305460077614, 0.0025457260986943, 0.0025609636247727,
	0.0025762586766381, 0.0025916113448759, 0.0026070217200159, 0.0026224898925319,
	0.0026380159528426, 0.002653599991311, 0.0026692420982451, 0.0026849423638977,
	0.0027007008784668, 0.0027165177320956, 0.0027323930148728, 0.0027483268168326,
	0.0027643192279548, 0.0027803703381652, 0.0027964802373355, 0.0028126490152836,
	0.0028288767617737, 0.0028451635665164, 0.0028615095191689, 0.0028779147093352,
	0.002894379226566, 0.0029109031603591, 0.0029274866001596, 0.0029441296353598,
	0.0029608323552992, 0.0029775948492654, 0.0029944172064931, 0.0030112995161654,
	0.0030282418674131, 0.0030452443493152, 0.003062307050899, 0.0030794300611402,
	0.0030966134689631, 0.0031138573632404, 0.0031311618327941, 0.0031485269663946,
	0.0031659528527619, 0.0031834395805649, 0.0032009872384219, 0.0032185959149007,
	0.0032362656985186, 0.0032539966777429, 0.0032717889409905, 0.0032896425766285,
	0.003307557672974, 0.0033255343182943, 0.0033435726008072, 0.0033616726086811,
	0.0033798344300347, 0.0033980581529378, 0.0034163438654109, 0.0034346916554256,
	0.0034531016109044, 0.0034715738197215, 0.0034901083697021, 0.0035087053486229,
	0.0035273648442125, 0.003546086944151, 0.0035648717360705, 0.003583719307555,
	0.0036026297461406, 0.0036216031393156, 0.0036406395745208, 0.0036597391391493,
	0.0036789019205468, 0.0036981280060118, 0.0037174174827953, 0.0037367704381017,
	0.0037561869590881, 0.0037756671328647, 0.0037952110464953, 0.0038148187869968,
	0.0038344904413397, 0.0038542260964481, 0.0038740258391998, 0.0038938897564264,
	0.0039138179349135, 0.0039338104614009, 0.0039538674225822, 0.0039739889051057,
	0.0039941749955737, 0.0040144257805432, 0.0040347413465258, 0.0040551217799877,
	0.0040755671673502, 0.0040960775949892, 0.0041166531492357, 0.0041372939163759,
	0.0041579999826514, 0.0041787714342588, 0.0041996083573505, 0.0042205108380342,
	0.0042414789623735, 0.0042625128163874, 0.0042836124860512, 0.004304778057296,
	0.0043260096160088, 0.0043473072480331, 0.0043686710391684, 0.0043901010751708,
	0.0044115974417526, 0.0044331602245831, 0.0044547895092878, 0.0044764853814494,
	0.0044982479266072, 0.0045200772302576, 0.0045419733778541, 0.0045639364548072,
	0.004585966546485, 0.0046080637382126, 0.0046302281152729, 0.004652459762906,
	0.00467475876631, 0.0046971252106406, 0.0047195591810113, 0.0047420607624937,
	0.0047646300401172, 0.0047872670988696, 0.0048099720236968, 0.0048327448995029,
	0.0048555858111506, 0.0048784948434611, 0.0049014720812141, 0.0049245176091479,
	0.0049476315119599, 0.004970813874306, 0.0049940647808013, 0.0050173843160198,
	0.0050407725644949, 0.005064229610719, 0.0050877555391437, 0.0051113504341804,
	0.0051350143801997, 0.0051587474615319, 0.0051825497624669, 0.0052064213672545,
	0.0052303623601041, 0.0052543728251853, 0.0052784528466277, 0.0053026025085207,
	0.0053268218949143, 0.0053511110898186, 0.005375470177204, 0.0053998992410014,
	0.0054243983651023, 0.0054489676333588, 0.0054736071295836, 0.0054983169375503,
	0.0055230971409934, 0.0055479478236082, 0.0055728690690511, 0.0055978609609397,
	0.0056229235828528, 0.0056480570183303, 0.0056732613508737, 0.0056985366639457,
	0.0057238830409708, 0.005749300565335, 0.0057747893203858, 0.0058003493894327,
	0.0058259808557469, 0.0058516838025617, 0.0058774583130723, 0.0059033044704359,
	0.005929222357772, 0.0059552120581623, 0.0059812736546507, 0.0060074072302437,
	0.0060336128679101, 0.0060598906505814, 0.0060862406611516, 0.0061126629824775,
	0.0061391576973785, 0.0061657248886372, 0.0061923646389988, 0.0062190770311716,
	0.0062458621478271, 0.0062727200715997, 0.0062996508850874, 0.0063266546708512,
	0.0063537315114156, 0.0063808814892685, 0.0064081046868615, 0.0064354011866095,
	0.0064627710708913, 0.0064902144220494, 0.00651773132239, 0.0065453218541835,
	0.0065729860996639, 0.0066007241410294, 0.0066285360604423, 0.0066564219400292,
	0.0066843818618806, 0.0067124159080517, 0.0067405241605619, 0.0067687067013951,
	0.0067969636124997, 0.0068252949757887, 0.0068537008731398, 0.0068821813863954,
	0.0069107365973627, 0.006939366587814, 0.0069680714394861, 0.0069968512340812,
	0.0070257060532664, 0.007054635978674, 0.0070836410919015, 0.0071127214745117,
	0.0071418772080328, 0.0071711083739583, 0.0072004150537472, 0.0072297973288241,
	0.0072592552805793, 0.0072887889903685, 0.0073183985395136, 0.0073480840093018,
	0.0073778454809866, 0.0074076830357872, 0.0074375967548889, 0.007467586719443,
	0.0074976530105671, 0.0075277957093448, 0.0075580148968261, 0.0075883106540273,
	0.0076186830619311, 0.0076491322014865, 0.0076796581536093, 0.0077102609991818,
	0.0077409408190527, 0.0077716976940377, 0.0078025317049192, 0.0078334429324463,
	0.0078644314573351, 0.0078954973602688, 0.0079266407218972, 0.0079578616228377,
	0.0079891601436744, 0.0080205363649589, 0.0080519903672099, 0.0080835222309134,
	0.008115132036523, 0.0081468198644594, 0.0081785857951111, 0.008210429908834,
	0.0082423522859517, 0.0082743530067554, 0.0083064321515042, 0.0083385898004248,
	0.0083708260337119, 0.0084031409315279, 0.0084355345740035, 0.0084680070412373,
	0.0085005584132957, 0.0085331887702138, 0.0085658981919944, 0.0085986867586089,
	0.0086315545499967, 0.008664501646066, 0.008697528126693, 0.0087306340717227,
	0.0087638195609685, 0.0087970846742124, 0.0088304294912051, 0.0088638540916661,
	0.0088973585552834, 0.0089309429617142, 0.0089646073905843, 0.0089983519214885,
	0.0090321766339906, 0.0090660816076236, 0.0091000669218893, 0.0091341326562588,
	0.0091682788901726, 0.0092025057030402, 0.0092368131742405, 0.0092712013831217,
	0.0093056704090017, 0.0093402203311676, 0.0093748512288761, 0.0094095631813535,
	0.0094443562677958, 0.0094792305673686, 0.0095141861592073, 0.0095492231224171,
	0.009584341536073, 0.0096195414792199, 0.0096548230308728, 0.0096901862700164,
	0.0097256312756059, 0.0097611581265662, 0.0097967669017925, 0.0098324576801504,
	0.0098682305404754, 0.0099040855615735, 0.0099400228222213, 0.0099760424011654,
	0.010012144377123, 0.010048328828782, 0.010084595834801, 0.010120945473808,
	0.010157377824404, 0.010193892965158, 0.010230490974612, 0.010267171931278,
	0.010303935913638, 0.010340783000146, 0.010377713269227, 0.010414726799276,
	0.010451823668661, 0.010489003955718, 0.010526267738758, 0.010563615096059,
	0.010601046105874, 0.010638560846426, 0.010676159395908, 0.010713841832486,
	0.010751608234296, 0.010789458679448, 0.010827393246022, 0.010865412012068,
	0.01090351505561, 0.010941702454643, 0.010979974287134, 0.01101833063102,
	0.011056771564213, 0.011095297164593, 0.011133907510016, 0.011172602678306,
	0.011211382747263, 0.011250247794655, 0.011289197898225, 0.011328233135688,
	0.011367353584728, 0.011406559323006, 0.011445850428153, 0.01148522697777,
	0.011524689049434, 0.011564236720693, 0.011603870069068, 0.011643589172051,
	0.011683394107107, 0.011723284951676, 0.011763261783167, 0.011803324678964,
	0.011843473716424, 0.011883708972874, 0.011924030525617, 0.011964438451927,
	0.012004932829052, 0.012045513734212, 0.0120861812446, 0.012126935437383,
	0.012167776389701, 0.012208704178665, 0.012249718881362, 0.01229082057485,
	0.012332009336161, 0.012373285242302, 0.01241464837025, 0.012456098796958,
	0.012497636599352, 0.012539261854329, 0.012580974638763, 0.0126227750295,
	0.012664663103359, 0.012706638937133, 0.01274870260759, 0.012790854191469,
	0.012833093765485, 0.012875421406326, 0.012917837190653, 0.012960341195103,
	0.013002933496285, 0.013045614170783, 0.013088383295153, 0.013131240945928,
	0.013174187199613, 0.013217222132688, 0.013260345821607, 0.013303558342797,
	0.013346859772661, 0.013390250187575, 0.013433729663891, 0.013477298277932,
	0.013520956106, 0.013564703224366, 0.013608539709281, 0.013652465636967,
	0.013696481083621, 0.013740586125415, 0.013784780838495, 0.013829065298984,
	0.013873439582976, 0.013917903766543, 0.01396245792573, 0.014007102136556,
	0.014051836475018, 0.014096661017084, 0.0141415758387, 0.014186581015785,
	0.014231676624235, 0.014276862739919, 0.014322139438681, 0.014367506796342,
	0.014412964888697, 0.014458513791516, 0.014504153580545, 0.014549884331504,
	0.014595706120089, 0.014641619021971, 0.014687623112798, 0.014733718468191,
	0.014779905163748, 0.014826183275042, 0.014872552877621, 0.01491901404701,
	0.014965566858707, 0.015012211388188, 0.015058947710905, 0.015105775902283,
	0.015152696037724, 0.015199708192608, 0.015246812442286, 0.015294008862091,
	0.015341297527325, 0.015388678513272, 0.015436151895189, 0.015483717748308,
	0.015531376147839, 0.015579127168967, 0.015626970886854, 0.015674907376637,
	0.01572293671343, 0.015771058972322, 0.01581927422838, 0.015867582556647,
	0.015915984032139, 0.015964478729853, 0.016013066724759, 0.016061748091805,
	0.016110522905915, 0.01615939124199, 0.016208353174906, 0.016257408779517,
	0.016306558130653, 0.01635580130312, 0.016405138371701, 0.016454569411157,
	0.016504094496223, 0.016553713701614, 0.016603427102018, 0.016653234772103,
	0.016703136786512, 0.016753133219866, 0.016803224146762, 0.016853409641773,
	0.016903689779452, 0.016954064634326, 0.0170045342809, 0.017055098793657,
	0.017105758247056, 0.017156512715532, 0.0172073622735, 0.01725830699535,
	0.017309346955449, 0.017360482228144, 0.017411712887755, 0.017463039008584,
	0.017514460664906, 0.017565977930976, 0.017617590881026, 0.017669299589265,
	0.017721104129879, 0.017773004577032, 0.017825001004866, 0.0178770934875,
	0.01792928209903, 0.017981566913532, 0.018033948005056, 0.018086425447632,
	0.018138999315268, 0.018191669681949, 0.018244436621637, 0.018297300208273,
	0.018350260515775, 0.018403317618039, 0.01845647158894, 0.01850972250233,
	0.018563070432038, 0.018616515451873, 0.018670057635619, 0.018723697057042,
	0.018777433789883, 0.018831267907863, 0.018885199484679, 0.018939228594008,
	0.018993355309505, 0.019047579704801, 0.019101901853509, 0.019156321829217,
	0.019210839705493, 0.019265455555883, 0.019320169453911, 0.01937498147308,
	0.01942989168687, 0.019484900168741, 0.019540006992132, 0.019595212230457,
	0.019650515957114, 0.019705918245474, 0.019761419168889, 0.019817018800692,
	0.01987271721419, 0.019928514482673, 0.019984410679406, 0.020040405877635,
	0.020096500150584, 0.020152693571457, 0.020208986213434, 0.020265378149677,
	0.020321869453325, 0.020378460197496, 0.020435150455288, 0.020491940299777,
	0.020548829804018, 0.020605819041044, 0.020662908083871, 0.020720097005489,
	0.02077738587887, 0.020834774776964, 0.020892263772702, 0.020949852938992,
	0.021007542348721, 0.021065332074758, 0.021123222189948, 0.021181212767117,
	0.021239303879071, 0.021297495598593, 0.021355787998447, 0.021414181151377,
	0.021472675130105, 0.021531270007332, 0.02158996585574, 0.021648762747991,
	0.021707660756723, 0.021766659954558, 0.021825760414095, 0.021884962207912,
	0.021944265408569, 0.022003670088604, 0.022063176320534, 0.022122784176858,
	0.022182493730053, 0.022242305052575, 0.022302218216863, 0.022362233295333,
	0.02242235036038, 0.022482569484383, 0.022542890739696, 0.022603314198656,
	0.02266383993358, 0.022724468016762, 0.02278519852048, 0.02284603151699,
	0.022906967078526, 0.022968005277307, 0.023029146185526, 0.023090389875362,
	0.02315173641897, 0.023213185888487, 0.02327473835603, 0.023336393893695,
	0.02339815257356, 0.023460014467682, 0.023521979648099, 0.023584048186829,
	0.02364622015587, 0.023708495627201, 0.023770874672781, 0.023833357364549,
	0.023895943774425, 0.02395863397431, 0.024021428036083, 0.024084326031608,
	0.024147328032724, 0.024210434111256, 0.024273644339004, 0.024336958787754,
	0.024400377529269, 0.024463900635293, 0.024527528177553, 0.024591260227754,
	0.024655096857582, 0.024719038138707, 0.024783084142775, 0.024847234941416,
	0.02491149060624, 0.024975851208838, 0.02504031682078, 0.025104887513619,
	0.02516956335889, 0.025234344428105, 0.02529923079276, 0.025364222524331,
	0.025429319694275, 0.025494522374031, 0.025559830635017, 0.025625244548635,
	0.025690764186264, 0.025756389619267, 0.025822120918989, 0.025887958156753,
	0.025953901403865, 0.026019950731614, 0.026086106211265, 0.026152367914071,
	0.02621873591126, 0.026285210274045, 0.026351791073619, 0.026418478381158,
	0.026485272267816, 0.026552172804732, 0.026619180063024, 0.026686294113793,
	0.02675351502812, 0.026820842877068, 0.026888277731681, 0.026955819662987,
	0.027023468741991, 0.027091225039685, 0.027159088627038, 0.027227059575002,
	0.027295137954512, 0.027363323836483, 0.027431617291813, 0.02750001839138,
	0.027568527206045, 0.02763714380665, 0.027705868264019, 0.027774700648959,
	0.027843641032256, 0.027912689484681, 0.027981846076984, 0.028051110879898,
	0.028120483964139, 0.028189965400404, 0.028259555259371, 0.028329253611701,
	0.028399060528036, 0.028468976079001, 0.028539000335204, 0.028609133367232,
	0.028679375245656, 0.028749726041029, 0.028820185823886, 0.028890754664744,
	0.028961432634102, 0.02903221980244, 0.029103116240224, 0.029174122017897,
	0.029245237205887, 0.029316461874606, 0.029387796094444, 0.029459239935777,
	0.029530793468961, 0.029602456764335, 0.02967422989222, 0.029746112922921,
	0.029818105926724, 0.029890208973896, 0.029962422134689, 0.030034745479337,
	0.030107179078054, 0.03017972300104, 0.030252377318474, 0.030325142100521,
	0.030398017417326, 0.030471003339016, 0.030544099935704, 0.030617307277483,
	0.030690625434428, 0.030764054476598, 0.030837594474034, 0.030911245496762,
	0.030985007614786, 0.031058880898097, 0.031132865416667, 0.03120696124045,
	0.031281168439385, 0.03135548708339, 0.031429917242371, 0.031504458986212,
	0.031579112384782, 0.031653877507934, 0.031728754425501, 0.031803743207302,
	0.031878843923136, 0.031954056642787, 0.03202938143602, 0.032104818372586,
	0.032180367522217, 0.032256028954627, 0.032331802739515, 0.032407688946563,
	0.032483687645434, 0.032559798905776, 0.032636022797219, 0.032712359389378,
	0.032788808751848, 0.032865370954211, 0.032942046066029, 0.033018834156848,
	0.033095735296198, 0.033172749553592, 0.033249876998526, 0.033327117700479,
	0.033404471728913, 0.033481939153275, 0.033559520042993, 0.033637214467481,
	0.033715022496134, 0.033792944198332, 0.033870979643437, 0.033949128900795,
	0.034027392039735, 0.034105769129572, 0.034184260239601, 0.034262865439101,
	0.034341584797338, 0.034420418383557, 0.034499366266989, 0.034578428516847,
	0.034657605202331, 0.034736896392621, 0.034816302156881, 0.034895822564261,
	0.034975457683891, 0.035055207584889, 0.035135072336354, 0.035215052007368,
	0.035295146666999, 0.035375356384296, 0.035455681228296, 0.035536121268014,
	0.035616676572455, 0.035697347210602, 0.035778133251427, 0.035859034763881,
	0.035940051816903, 0.036021184479413, 0.036102432820316, 0.036183796908501,
	0.036265276812842, 0.036346872602194, 0.036428584345399, 0.036510412111281,
	0.036592355968649, 0.036674415986295, 0.036756592232997, 0.036838884777514,
	0.036921293688592, 0.03700381903496, 0.03708646088533, 0.0371692193084,
	0.037252094372851, 0.037335086147348, 0.037418194700541, 0.037501420101062,
	0.037584762417532, 0.03766822171855, 0.037751798072704, 0.037835491548564,
	0.037919302214686, 0.038003230139607, 0.038087275391851, 0.038171438039927,
	0.038255718152326, 0.038340115797524, 0.038424631043982, 0.038509263960145,
	0.038594014614443, 0.038678883075289, 0.038763869411083, 0.038848973690205,
	0.038934195981024, 0.039019536351892, 0.039104994871144, 0.039190571607101,
	0.039276266628068, 0.039362080002335, 0.039448011798176, 0.03953406208385,
	0.039620230927601, 0.039706518397655, 0.039792924562226, 0.039879449489512,
	0.039966093247693, 0.040052855904936, 0.040139737529393, 0.0402267381892,
	0.040313857952476, 0.040401096887328, 0.040488455061846, 0.040575932544103,
	0.04066352940216, 0.040751245704062, 0.040839081517837, 0.040927036911498,
	0.041015111953046, 0.041103306710464, 0.04119162125172, 0.041280055644768,
	0.041368609957545, 0.041457284257975, 0.041546078613967, 0.041634993093413,
	0.041724027764191, 0.041813182694164, 0.04190245795118, 0.041991853603073,
	0.042081369717659, 0.042171006362743, 0.042260763606112, 0.042350641515538,
	0.042440640158782, 0.042530759603584, 0.042620999917675, 0.042711361168767,
	0.042801843424559, 0.042892446752735, 0.042983171220964, 0.0430740168969,
	0.043164983848181, 0.043256072142434, 0.043347281847266, 0.043438613030274,
	0.043530065759038, 0.043621640101122, 0.043713336124078, 0.043805153895442,
	0.043897093482735, 0.043989154953464, 0.044081338375121, 0.044173643815184,
	0.044266071341114, 0.04435862102036, 0.044451292920356, 0.04454408710852,
	0.044637003652258, 0.044730042618958, 0.044823204075996, 0.044916488090733,
	0.045009894730514, 0.045103424062673, 0.045197076154525, 0.045290851073374,
	0.045384748886507, 0.045478769661199, 0.045572913464709, 0.045667180364283,
	0.045761570427149, 0.045856083720526, 0.045950720311614, 0.046045480267601,
	0.04614036365566, 0.046235370542951, 0.046330500996616, 0.046425755083787,
	0.046521132871579, 0.046616634427094, 0.046712259817419, 0.046808009109627,
	0.046903882370777, 0.046999879667913, 0.047096001068067, 0.047192246638253,
	0.047288616445475, 0.047385110556719, 0.04748172903896, 0.047578471959157,
	0.047675339384256, 0.047772331381187, 0.047869448016868, 0.047966689358202,
	0.048064055472078, 0.048161546425371, 0.048259162284942, 0.048356903117637,
	0.04845476899029, 0.04855275996972, 0.04865087612273, 0.048749117516113,
	0.048847484216645, 0.048945976291089, 0.049044593806193, 0.049143336828694,
	0.049242205425312, 0.049341199662754, 0.049440319607715, 0.049539565326872,
	0.049638936886893, 0.049738434354428, 0.049838057796116, 0.04993780727858,
	0.050037682868432, 0.050137684632268, 0.05023781263667, 0.050338066948207,
	0.050438447633435, 0.050538954758894, 0.050639588391114, 0.050740348596607,
	0.050841235441875, 0.050942248993403, 0.051043389317664, 0.051144656481119,
	0.051246050550212, 0.051347571591376, 0.051449219671028, 0.051550994855574,
	0.051652897211405, 0.051754926804897, 0.051857083702416, 0.051959367970312,
	0.05206177967492, 0.052164318882566, 0.052266985659557, 0.052369780072192,
	0.052472702186751, 0.052575752069506, 0.052678929786711, 0.052782235404608,
	0.052885668989427, 0.052989230607383, 0.053092920324678, 0.0531967382075,
	0.053300684322025, 0.053404758734415, 0.053508961510817, 0.053613292717367,
	0.053717752420187, 0.053822340685385, 0.053927057579055, 0.05403190316728,
	0.054136877516128, 0.054241980691654, 0.0543472127599, 0.054452573786894,
	0.054558063838652, 0.054663682981175, 0.054769431280453, 0.054875308802461,
	0.054981315613161, 0.055087451778502, 0.055193717364422, 0.055300112436841,
	0.055406637061671, 0.055513291304807, 0.055620075232133, 0.055726988909519,
	0.055834032402823, 0.055941205777888, 0.056048509100544, 0.056155942436611,
	0.056263505851893, 0.056371199412182, 0.056479023183256, 0.05658697723088,
	0.056695061620809, 0.05680327641878, 0.056911621690522, 0.057020097501747,
	0.057128703918157, 0.057237441005438, 0.057346308829267, 0.057455307455304,
	0.057564436949198, 0.057673697376586, 0.057783088803091, 0.057892611294323,
	0.058002264915879, 0.058112049733343, 0.058221965812288, 0.058332013218272,
	0.058442192016841, 0.058552502273528, 0.058662944053854, 0.058773517423325,
	0.058884222447437, 0.058995059191672, 0.059106027721498, 0.059217128102373,
	0.059328360399739, 0.059439724679028, 0.059551221005658, 0.059662849445035,
	0.059774610062551, 0.059886502923587, 0.059998528093509, 0.060110685637673,
	0.060222975621421, 0.060335398110082, 0.060447953168974, 0.0605606408634,
	0.060673461258652, 0.060786414420009, 0.060899500412738, 0.061012719302092,
	0.061126071153313, 0.061239556031629, 0.061353174002258, 0.061466925130402,
	0.061580809481252, 0.061694827119988, 0.061808978111775, 0.061923262521767,
	0.062037680415106, 0.06215223185692, 0.062266916912325, 0.062381735646425,
	0.062496688124312, 0.062611774411064, 0.062726994571748, 0.062842348671419,
	0.062957836775118, 0.063073458947874, 0.063189215254704, 0.063305105760614,
	0.063421130530595, 0.063537289629628, 0.063653583122679, 0.063770011074705,
	0.063886573550649, 0.064003270615441, 0.064120102333999, 0.06423706877123,
	0.064354169992028, 0.064471406061274, 0.064588777043839, 0.064706283004578,
	0.064823924008337, 0.064941700119949, 0.065059611404235, 0.065177657926002,
	0.065295839750047, 0.065414156941154, 0.065532609564095, 0.06565119768363,
	0.065769921364506, 0.065888780671458, 0.06600777566921, 0.066126906422474,
	0.066246172995948, 0.066365575454319, 0.066485113862263, 0.066604788284441,
	0.066724598785507, 0.066844545430097, 0.066964628282838, 0.067084847408347,
	0.067205202871224, 0.067325694736062, 0.067446323067438, 0.06756708792992,
	0.067687989388062, 0.067809027506407, 0.067930202349486, 0.068051513981819,
	0.068172962467911, 0.068294547872258, 0.068416270259343, 0.068538129693637,
	0.0686601262396, 0.06878225996168, 0.068904530924311, 0.069026939191918,
	0.069149484828912, 0.069272167899694, 0.069394988468652, 0.069517946600161,
	0.069641042358588, 0.069764275808283, 0.06988764701359, 0.070011156038835,
	0.070134802948338, 0.070258587806404, 0.070382510677326, 0.070506571625386,
	0.070630770714856, 0.070755108009994, 0.070879583575047, 0.071004197474249,
	0.071128949771825, 0.071253840531987, 0.071378869818934, 0.071504037696856,
	0.071629344229929, 0.071754789482318, 0.071880373518177, 0.072006096401648,
	0.072131958196862, 0.072257958967936, 0.072384098778979, 0.072510377694085,
	0.072636795777339, 0.072763353092813, 0.072890049704567, 0.073016885676652,
	0.073143861073105, 0.073270975957952, 0.073398230395208, 0.073525624448875,
	0.073653158182946, 0.073780831661401, 0.073908644948208, 0.074036598107325,
	0.074164691202697, 0.074292924298259, 0.074421297457933, 0.074549810745631,
	0.074678464225253, 0.074807257960687, 0.07493619201581, 0.075065266454488,
	0.075194481340576, 0.075323836737916, 0.07545333271034, 0.075582969321668,
	0.075712746635709, 0.07584266471626, 0.075972723627108, 0.076102923432027,
	0.076233264194781, 0.076363745979122, 0.076494368848791, 0.076625132867517,
	0.07675603809902, 0.076887084607005, 0.077018272455169, 0.077149601707196,
	0.077281072426759, 0.077412684677521, 0.077544438523132, 0.077676334027232,
	0.077808371253449, 0.0779405502654, 0.078072871126692, 0.078205333900919,
	0.078337938651665, 0.078470685442502, 0.078603574336991, 0.078736605398683,
	0.078869778691116, 0.079003094277819, 0.079136552222308, 0.079270152588089,
	0.079403895438656, 0.079537780837494, 0.079671808848073, 0.079805979533856,
	0.079940292958292, 0.080074749184821, 0.080209348276871, 0.080344090297859,
	0.080478975311191, 0.080614003380261, 0.080749174568455, 0.080884488939144,
	0.081019946555691, 0.081155547481447, 0.081291291779752, 0.081427179513934,
	0.081563210747312, 0.081699385543193, 0.081835703964873, 0.081972166075638,
	0.08210877193876, 0.082245521617505, 0.082382415175124, 0.082519452674859,
	0.082656634179941, 0.082793959753588, 0.082931429459011, 0.083069043359407,
	0.083206801517964, 0.083344703997857, 0.083482750862252, 0.083620942174304,
	0.083759277997156, 0.083897758393941, 0.084036383427782, 0.08417515316179,
	0.084314067659066, 0.084453126982699, 0.084592331195767, 0.084731680361341,
	0.084871174542476, 0.08501081380222, 0.085150598203609, 0.085290527809667,
	0.08543060268341, 0.08557082288784, 0.085711188485952, 0.085851699540728,
	0.085992356115139, 0.086133158272145, 0.086274106074699, 0.086415199585739,
	0.086556438868193, 0.086697823984982, 0.086839354999011, 0.086981031973179,
	0.087122854970371, 0.087264824053464, 0.087406939285322, 0.087549200728801,
	0.087691608446744, 0.087834162501984, 0.087976862957345, 0.088119709875638,
	0.088262703319666, 0.088405843352219, 0.088549130036079, 0.088692563434014,
	0.088836143608785, 0.08897987062314, 0.089123744539819, 0.089267765421549,
	0.089411933331047, 0.08955624833102, 0.089700710484166, 0.089845319853169,
	0.089990076500706, 0.090134980489442, 0.090280031882031, 0.090425230741117,
	0.090570577129335, 0.090716071109307, 0.090861712743646, 0.091007502094956,
	0.091153439225827, 0.091299524198842, 0.091445757076573, 0.091592137921579,
	0.091738666796412, 0.091885343763611, 0.092032168885707, 0.092179142225219,
	0.092326263844657, 0.092473533806518, 0.092620952173292, 0.092768519007456,
	0.092916234371479, 0.093064098327818, 0.093212110938921, 0.093360272267223,
	0.093508582375153, 0.093657041325126, 0.093805649179549, 0.093954406000817,
	0.094103311851316, 0.094252366793422, 0.0944015708895, 0.094550924201904,
	0.09470042679298, 0.094850078725061, 0.094999880060473, 0.095149830861529,
	0.095299931190534, 0.09545018110978, 0.095600580681553, 0.095751129968124,
	0.095901829031758, 0.096052677934707, 0.096203676739214, 0.096354825507513,
	0.096506124301826, 0.096657573184365, 0.096809172217334, 0.096960921462923,
	0.097112820983317, 0.097264870840686, 0.097417071097193, 0.09756942181499,
	0.097721923056219, 0.097874574883012, 0.09802737735749, 0.098180330541766,
	0.098333434497941, 0.098486689288107, 0.098640094974346, 0.098793651618729,
	0.098947359283318, 0.099101218030164, 0.09925522792131, 0.099409389018787,
	0.099563701384617, 0.099718165080812, 0.099872780169372, 0.10002754671229,
	0.10018246477155, 0.10033753440912, 0.10049275568697, 0.10064812866704,
	0.10080365341127, 0.10095932998161, 0.10111515843998, 0.10127113884827,
	0.10142727126841, 0.10158355576227, 0.10173999239175, 0.10189658121871,
	0.10205332230503, 0.10221021571254, 0.1023672615031, 0.10252445973854,
	0.10268181048068, 0.10283931379134, 0.10299696973232, 0.10315477836541,
	0.10331273975241, 0.10347085395507, 0.10362912103518, 0.10378754105447,
	0.10394611407471, 0.10410484015762, 0.10426371936493, 0.10442275175836,
	0.1045819373996, 0.10474127635037, 0.10490076867234, 0.10506041442719,
	0.10522021367659, 0.1053801664822, 0.10554027290566, 0.10570053300862,
	0.1058609468527, 0.10602151449951, 0.10618223601068, 0.10634311144779,
	0.10650414087244, 0.10666532434621, 0.10682666193067, 0.10698815368738,
	0.10714979967788, 0.10731159996373, 0.10747355460646, 0.10763566366757,
	0.1077979272086, 0.10796034529104, 0.10812291797638, 0.10828564532611,
	0.10844852740171, 0.10861156426463, 0.10877475597634, 0.10893810259827,
	0.10910160419187, 0.10926526081856, 0.10942907253975, 0.10959303941687,
	0.10975716151129, 0.10992143888441, 0.11008587159762, 0.11025045971227,
	0.11041520328974, 0.11058010239136, 0.11074515707848, 0.11091036741243,
	0.11107573345454, 0.11124125526611, 0.11140693290845, 0.11157276644285,
	0.1117387559306, 0.11190490143296, 0.11207120301121, 0.11223766072661,
	0.11240427464038, 0.11257104481379, 0.11273797130804, 0.11290505418437,
	0.11307229350397, 0.11323968932804, 0.11340724171779, 0.11357495073438,
	0.11374281643899, 0.11391083889278, 0.11407901815691, 0.11424735429251,
	0.11441584736071, 0.11458449742266, 0.11475330453945, 0.11492226877219,
	0.11509139018198, 0.11526066882991, 0.11543010477706, 0.11559969808449,
	0.11576944881326, 0.11593935702443, 0.11610942277903, 0.11627964613809,
	0.11645002716264, 0.11662056591369, 0.11679126245224, 0.11696211683929,
	0.11713312913581, 0.1173042994028, 0.11747562770121, 0.117647114092,
	0.11781875863613, 0.11799056139452, 0.11816252242811, 0.11833464179781,
	0.11850691956455, 0.11867935578922, 0.11885195053271, 0.11902470385591,
	0.1191976158197, 0.11937068648493, 0.11954391591247, 0.11971730416316,
	0.11989085129785, 0.12006455737735, 0.12023842246248, 0.12041244661407,
	0.12058662989291, 0.12076097235979, 0.1209354740755, 0.12111013510081,
	0.12128495549648, 0.12145993532328, 0.12163507464194, 0.12181037351321,
	0.12198583199782, 0.12216145015648, 0.1223372280499, 0.1225131657388,
	0.12268926328385, 0.12286552074574, 0.12304193818516, 0.12321851566275,
	0.12339525323919, 0.12357215097511, 0.12374920893116, 0.12392642716796,
	0.12410380574614, 0.1242813447263, 0.12445904416906, 0.12463690413499,
	0.1248149246847, 0.12499310587874, 0.1251714477777, 0.12534995044212,
	0.12552861393256, 0.12570743830956, 0.12588642363364, 0.12606556996533,
	0.12624487736514, 0.12642434589357, 0.12660397561113, 0.12678376657829,
	0.12696371885554, 0.12714383250334, 0.12732410758215, 0.12750454415242,
	0.1276851422746, 0.12786590200912, 0.1280468234164, 0.12822790655685,
	0.12840915149089, 0.12859055827891, 0.1287721269813, 0.12895385765844,
	0.1291357503707, 0.12931780517844, 0.12950002214202, 0.12968240132178,
	0.12986494277806, 0.13004764657118, 0.13023051276146, 0.13041354140921,
	0.13059673257474, 0.13078008631833, 0.13096360270026, 0.13114728178082,
	0.13133112362026, 0.13151512827885, 0.13169929581683, 0.13188362629444,
	0.13206811977191, 0.13225277630947, 0.13243759596732, 0.13262257880568,
	0.13280772488473, 0.13299303426466, 0.13317850700566, 0.13336414316789,
	0.13354994281151, 0.13373590599668, 0.13392203278354, 0.13410832323222,
	0.13429477740285, 0.13448139535555, 0.13466817715042, 0.13485512284758,
	0.1350422325071, 0.13522950618907, 0.13541694395357, 0.13560454586066,
	0.1357923119704, 0.13598024234284, 0.13616833703801, 0.13635659611595,
	0.13654501963669, 0.13673360766023, 0.13692236024658, 0.13711127745574,
	0.13730035934769, 0.13748960598243, 0.13767901741991, 0.1378685937201,
	0.13805833494295, 0.13824824114841, 0.13843831239643, 0.13862854874691,
	0.1388189502598, 0.13900951699499, 0.13920024901239, 0.13939114637189,
	0.13958220913339, 0.13977343735675, 0.13996483110185, 0.14015639042855,
	0.14034811539669, 0.14054000606613, 0.1407320624967, 0.14092428474822,
	0.14111667288051, 0.14130922695338, 0.14150194702664, 0.14169483316008,
	0.14188788541347, 0.1420811038466, 0.14227448851923, 0.14246803949112,
	0.14266175682203, 0.14285564057169, 0.14304969079984, 0.1432439075662,
	0.1434382909305, 0.14363284095243, 0.1438275576917, 0.14402244120801,
	0.14421749156102, 0.14441270881043, 0.14460809301589, 0.14480364423707,
	0.14499936253361, 0.14519524796516, 0.14539130059135, 0.1455875204718,
	0.14578390766613, 0.14598046223395, 0.14617718423485, 0.14637407372844,
	0.14657113077429, 0.14676835543198, 0.14696574776107, 0.14716330782113,
	0.1473610356717, 0.14755893137232, 0.14775699498253, 0.14795522656186,
	0.14815362616982, 0.14835219386591, 0.14855092970965, 0.14874983376051,
	0.14894890607799, 0.14914814672157, 0.1493475557507, 0.14954713322485,
	0.14974687920346, 0.14994679374599, 0.15014687691187, 0.15034712876051,
	0.15054754935135, 0.15074813874378, 0.15094889699722, 0.15114982417105,
	0.15135092032467, 0.15155218551743, 0.15175361980873, 0.15195522325791,
	0.15215699592433, 0.15235893786734, 0.15256104914627, 0.15276332982044,
	0.15296577994918, 0.1531683995918, 0.15337118880761, 0.15357414765589,
	0.15377727619594, 0.15398057448704, 0.15418404258845, 0.15438768055945,
	0.15459148845928, 0.15479546634718, 0.15499961428241, 0.1552039323242,
	0.15540842053175, 0.15561307896429, 0.15581790768103, 0.15602290674116,
	0.15622807620387, 0.15643341612835, 0.15663892657376, 0.15684460759928,
	0.15705045926407, 0.15725648162726, 0.15746267474802, 0.15766903868546,
	0.15787557349872, 0.15808227924691, 0.15828915598914, 0.15849620378452,
	0.15870342269213, 0.15891081277107, 0.15911837408041, 0.15932610667922,
	0.15953401062657, 0.1597420859815, 0.15995033280307, 0.16015875115031,
	0.16036734108225, 0.16057610265791, 0.16078503593631, 0.16099414097645,
	0.16120341783734, 0.16141286657795, 0.16162248725728, 0.1618322799343,
	0.16204224466797, 0.16225238151725, 0.1624626905411, 0.16267317179845,
	0.16288382534825, 0.16309465124941, 0.16330564956085, 0.1635168203415,
	0.16372816365024, 0.16393967954598, 0.1641513680876, 0.16436322933399,
	0.164575263344, 0.16478747017652, 0.16499984989039, 0.16521240254446,
	0.16542512819757, 0.16563802690855, 0.16585109873624, 0.16606434373943,
	0.16627776197696, 0.1664913535076, 0.16670511839016, 0.16691905668343,
	0.16713316844618, 0.16734745373717, 0.16756191261518, 0.16777654513895,
	0.16799135136724, 0.16820633135878, 0.1684214851723, 0.16863681286652,
	0.16885231450016, 0.16906799013193, 0.16928383982052, 0.16949986362464,
	0.16971606160295, 0.16993243381414, 0.17014898031688, 0.17036570116983,
	0.17058259643164, 0.17079966616096, 0.17101691041642, 0.17123432925665,
	0.17145192274028, 0.17166969092592, 0.17188763387218, 0.17210575163765,
	0.17232404428093, 0.17254251186061, 0.17276115443525, 0.17297997206342,
	0.17319896480369, 0.17341813271461, 0.17363747585473, 0.17385699428257,
	0.17407668805668, 0.17429655723557, 0.17451660187776, 0.17473682204175,
	0.17495721778605, 0.17517778916914, 0.17539853624951, 0.17561945908564,
	0.175840557736, 0.17606183225904, 0.17628328271322, 0.17650490915699,
	0.17672671164878, 0.17694869024702, 0.17717084501014, 0.17739317599655,
	0.17761568326467, 0.17783836687288, 0.17806122687959, 0.17828426334318,
	0.17850747632202, 0.17873086587449, 0.17895443205894, 0.17917817493374,
	0.17940209455723, 0.17962619098775, 0.17985046428363, 0.1800749145032,
	0.18029954170477, 0.18052434594666, 0.18074932728716, 0.18097448578457,
	0.18119982149717, 0.18142533448326, 0.18165102480109, 0.18187689250893,
	0.18210293766504, 0.18232916032767, 0.18255556055506, 0.18278213840545,
	0.18300889393706, 0.18323582720811, 0.18346293827681, 0.18369022720137,
	0.18391769403998, 0.18414533885084, 0.18437316169213, 0.18460116262201,
	0.18482934169867, 0.18505769898025, 0.18528623452491, 0.1855149483908,
	0.18574384063605, 0.18597291131879, 0.18620216049715, 0.18643158822923,
	0.18666119457316, 0.18689097958702, 0.18712094332891, 0.18735108585692,
	0.18758140722913, 0.1878119075036, 0.1880425867384, 0.18827344499158,
	0.18850448232119, 0.18873569878528, 0.18896709444188, 0.18919866934901,
	0.1894304235647, 0.18966235714695, 0.18989447015377, 0.19012676264315,
	0.19035923467309, 0.19059188630157, 0.19082471758656, 0.19105772858603,
	0.19129091935795, 0.19152428996025, 0.1917578404509, 0.19199157088782,
	0.19222548132896, 0.19245957183222, 0.19269384245553, 0.1929282932568,
	0.19316292429393, 0.19339773562482, 0.19363272730734, 0.19386789939939,
	0.19410325195883, 0.19433878504353, 0.19457449871134, 0.19481039302012,
	0.19504646802772, 0.19528272379196, 0.19551916037067, 0.19575577782169,
	0.19599257620282, 0.19622955557186, 0.19646671598663, 0.19670405750491,
	0.19694158018449, 0.19717928408314, 0.19741716925864, 0.19765523576876,
	0.19789348367124, 0.19813191302384, 0.19837052388429, 0.19860931631035,
	0.19884829035972, 0.19908744609014, 0.19932678355931, 0.19956630282495,
	0.19980600394475, 0.2000458869764, 0.20028595197759, 0.20052619900599,
	0.20076662811929, 0.20100723937513, 0.20124803283118, 0.20148900854508,
	0.20173016657447, 0.201971506977, 0.20221302981029, 0.20245473513195,
	0.20269662299961, 0.20293869347087, 0.20318094660332, 0.20342338245456,
	0.20366600108217, 0.20390880254373, 0.20415178689681, 0.20439495419898,
	0.20463830450779, 0.20488183788079, 0.20512555437552, 0.20536945404952,
	0.20561353696031, 0.20585780316542, 0.20610225272236, 0.20634688568863,
	0.20659170212174, 0.20683670207917, 0.20708188561842, 0.20732725279696,
	0.20757280367226, 0.2078185383018, 0.20806445674301, 0.20831055905336,
	0.20855684529029, 0.20880331551123, 0.20904996977361, 0.20929680813486,
	0.20954383065239, 0.20979103738361, 0.21003842838592, 0.21028600371671,
	0.21053376343337, 0.21078170759328, 0.21102983625381, 0.21127814947233,
	0.2115266473062, 0.21177532981277, 0.21202419704938, 0.21227324907337,
	0.21252248594208, 0.21277190771283, 0.21302151444293, 0.2132713061897,
	0.21352128301044, 0.21377144496244, 0.21402179210299, 0.21427232448938,
	0.21452304217888, 0.21477394522876, 0.21502503369628, 0.2152763076387,
	0.21552776711326, 0.2157794121772, 0.21603124288777, 0.21628325930217,
	0.21653546147765, 0.2167878494714, 0.21704042334063, 0.21729318314255,
	0.21754612893434, 0.21779926077319, 0.21805257871628, 0.21830608282078,
	0.21855977314386, 0.21881364974267, 0.21906771267436, 0.21932196199608,
	0.21957639776497, 0.21983102003816, 0.22008582887276, 0.2203408243259,
	0.22059600645469, 0.22085137531622, 0.2211069309676, 0.22136267346591,
	0.22161860286824, 0.22187471923167, 0.22213102261325, 0.22238751307005,
	0.22264419065914, 0.22290105543755, 0.22315810746232, 0.2234153467905,
	0.22367277347911, 0.22393038758517, 0.22418818916569, 0.22444617827769,
	0.22470435497816, 0.2249627193241, 0.22522127137249, 0.22548001118031,
	0.22573893880455, 0.22599805430215, 0.22625735773009, 0.22651684914532,
	0.22677652860479, 0.22703639616542, 0.22729645188416, 0.22755669581794,
	0.22781712802366, 0.22807774855825, 0.22833855747861, 0.22859955484164,
	0.22886074070423, 0.22912211512327, 0.22938367815563, 0.22964542985819,
	0.22990737028781, 0.23016949950136, 0.23043181755568, 0.23069432450762,
	0.23095702041402, 0.23121990533171, 0.23148297931752, 0.23174624242825,
	0.23200969472074, 0.23227333625177, 0.23253716707816, 0.23280118725668,
	0.23306539684413, 0.23332979589729, 0.23359438447292, 0.23385916262779,
	0.23412413041866, 0.23438928790228, 0.2346546351354, 0.23492017217476,
	0.23518589907707, 0.23545181589909, 0.23571792269751, 0.23598421952905,
	0.23625070645042, 0.23651738351831, 0.23678425078942, 0.23705130832044,
	0.23731855616803, 0.23758599438888, 0.23785362303965, 0.23812144217699,
	0.23838945185757, 0.23865765213802, 0.23892604307498, 0.23919462472509,
	0.23946339714498, 0.23973236039125, 0.24000151452053, 0.24027085958942,
	0.24054039565452, 0.24081012277242, 0.24108004099971, 0.24135015039297,
	0.24162045100878, 0.24189094290369, 0.24216162613427, 0.24243250075707,
	0.24270356682864, 0.24297482440552, 0.24324627354424, 0.24351791430133,
	0.24378974673332, 0.2440617708967, 0.24433398684801, 0.24460639464372,
	0.24487899434035, 0.24515178599437, 0.24542476966227, 0.24569794540053,
	0.24597131326561, 0.24624487331397, 0.24651862560207, 0.24679257018636,
	0.24706670712328, 0.24734103646927, 0.24761555828076, 0.24789027261416,
	0.24816517952591, 0.2484402790724, 0.24871557131004, 0.24899105629522,
	0.24926673408435, 0.2495426047338, 0.24981866829995, 0.25009492483917,
	0.25037137440782, 0.25064801706228, 0.25092485285887, 0.25120188185396,
	0.25147910410388, 0.25175651966497, 0.25203412859354, 0.25231193094593,
	0.25258992677844, 0.25286811614737, 0.25314649910904, 0.25342507571973,
	0.25370384603573, 0.25398281011333, 0.25426196800879, 0.25454131977839,
	0.25482086547839, 0.25510060516504, 0.2553805388946, 0.2556606667233,
	0.25594098870738, 0.25622150490307, 0.2565022153666, 0.25678312015418,
	0.25706421932203, 0.25734551292634, 0.25762700102332, 0.25790868366916,
	0.25819056092004, 0.25847263283214, 0.25875489946164, 0.25903736086469,
	0.25932001709747, 0.25960286821613, 0.2598859142768, 0.26016915533564,
	0.26045259144878, 0.26073622267234, 0.26102004906245, 0.26130407067522,
	0.26158828756677, 0.26187269979319, 0.26215730741058, 0.26244211047503,
	0.26272710904263, 0.26301230316946, 0.26329769291158, 0.26358327832505,
	0.26386905946595, 0.26415503639032, 0.26444120915421, 0.26472757781365,
	0.26501414242469, 0.26530090304334, 0.26558785972563, 0.26587501252758,
	0.26616236150519, 0.26644990671446, 0.26673764821139, 0.26702558605198,
	0.26731372029219, 0.26760205098802, 0.26789057819543, 0.26817930197039,
	0.26846822236885, 0.26875733944677, 0.26904665326009, 0.26933616386475,
	0.26962587131669, 0.26991577567184, 0.2702058769861, 0.27049617531541,
	0.27078667071566, 0.27107736324276, 0.27136825295261, 0.27165933990109,
	0.27195062414409, 0.27224210573749, 0.27253378473716, 0.27282566119896,
	0.27311773517875, 0.27341000673238, 0.27370247591571, 0.27399514278457,
	0.27428800739479, 0.27458106980221, 0.27487433006264, 0.2751677882319,
	0.2754614443658, 0.27575529852014, 0.27604935075072, 0.27634360111334,
	0.27663804966377, 0.27693269645779, 0.27722754155118, 0.2775225849997,
	0.27781782685912, 0.27811326718518, 0.27840890603364, 0.27870474346023,
	0.27900077952069, 0.27929701427076, 0.27959344776615, 0.27989008006258,
	0.28018691121576, 0.2804839412814, 0.2807811703152, 0.28107859837285,
	0.28137622551003, 0.28167405178242, 0.28197207724571, 0.28227030195556,
	0.28256872596762, 0.28286734933757, 0.28316617212104, 0.28346519437368,
	0.28376441615112, 0.28406383750901, 0.28436345850296, 0.2846632791886,
	0.28496329962154, 0.28526351985738, 0.28556393995173, 0.28586455996018,
	0.28616537993832, 0.28646639994174, 0.28676762002601, 0.2870690402467,
	0.28737066065938, 0.2876724813196, 0.28797450228293, 0.2882767236049,
	0.28857914534106, 0.28888176754694, 0.28918459027807, 0.28948761358997,
	0.28979083753816, 0.29009426217816, 0.29039788756545, 0.29070171375556,
	0.29100574080395, 0.29130996876613, 0.29161439769758, 0.29191902765376,
	0.29222385869015, 0.2925288908622, 0.29283412422539, 0.29313955883514,
	0.29344519474692, 0.29375103201616, 0.29405707069829, 0.29436331084873,
	0.29466975252292, 0.29497639577626, 0.29528324066416, 0.29559028724203,
	0.29589753556525, 0.29620498568923, 0.29651263766935, 0.29682049156098,
	0.29712854741951, 0.29743680530028, 0.29774526525868, 0.29805392735004,
	0.29836279162973, 0.29867185815308, 0.29898112697543, 0.29929059815212,
	0.29960027173846, 0.29991014778978, 0.30022022636139, 0.3005305075086,
	0.3008409912867, 0.30115167775101, 0.3014625669568, 0.30177365895935,
	0.30208495381396, 0.30239645157588, 0.30270815230039, 0.30302005604275,
	0.3033321628582, 0.30364447280201, 0.3039569859294, 0.30426970229563,
	0.30458262195591, 0.30489574496547, 0.30520907137954, 0.30552260125332,
	0.30583633464203, 0.30615027160086, 0.30646441218501, 0.30677875644966,
	0.30709330445002, 0.30740805624124, 0.30772301187851, 0.30803817141698,
	0.30835353491183, 0.3086691024182, 0.30898487399124, 0.3093008496861,
	0.30961702955791, 0.30993341366181, 0.31025000205291, 0.31056679478634,
	0.31088379191722, 0.31120099350065, 0.31151839959172, 0.31183601024555,
	0.31215382551721, 0.3124718454618, 0.31279007013439, 0.31310849959006,
	0.31342713388386, 0.31374597307087, 0.31406501720614, 0.31438426634472,
	0.31470372054165, 0.31502337985197, 0.31534324433071, 0.3156633140329,
	0.31598358901356, 0.31630406932771, 0.31662475503035, 0.31694564617648,
	0.31726674282111, 0.31758804501923, 0.31790955282582, 0.31823126629587,
	0.31855318548434, 0.31887531044621, 0.31919764123644, 0.31952017790998,
	0.3198429205218, 0.32016586912682, 0.32048902378001, 0.32081238453628,
	0.32113595145056, 0.32145972457779, 0.32178370397287, 0.32210788969072,
	0.32243228178624, 0.32275688031434, 0.32308168532991, 0.32340669688783,
	0.32373191504299, 0.32405733985026, 0.32438297136452, 0.32470880964064,
	0.32503485473347, 0.32536110669786, 0.32568756558867, 0.32601423146073,
	0.32634110436889, 0.32666818436797, 0.32699547151281, 0.32732296585822,
	0.32765066745901, 0.32797857637, 0.32830669264598, 0.32863501634176,
	0.32896354751213, 0.32929228621186, 0.32962123249575, 0.32995038641856,
	0.33027974803507, 0.33060931740004, 0.33093909456822, 0.33126907959438,
	0.33159927253324, 0.33192967343956, 0.33226028236807, 0.3325910993735,
	0.33292212451057, 0.33325335783399, 0.3335847993985, 0.33391644925878,
	0.33424830746954, 0.33458037408548, 0.33491264916128, 0.33524513275163,
	0.33557782491121, 0.33591072569469, 0.33624383515674, 0.33657715335202,
	0.33691068033519, 0.33724441616089, 0.33757836088377, 0.33791251455847,
	0.33824687723963, 0.33858144898187, 0.33891622983982, 0.33925121986808,
	0.33958641912128, 0.33992182765402, 0.3402574455209, 0.34059327277651,
	0.34092930947544, 0.34126555567227, 0.34160201142159, 0.34193867677797,
	0.34227555179597, 0.34261263653014, 0.34294993103506, 0.34328743536527,
	0.34362514957531, 0.34396307371972, 0.34430120785303, 0.34463955202978,
	0.34497810630448, 0.34531687073165, 0.3456558453658, 0.34599503026144,
	0.34633442547307, 0.34667403105518, 0.34701384706226, 0.34735387354879,
	0.34769411056926, 0.34803455817813, 0.34837521642987, 0.34871608537894,
	0.3490571650798, 0.3493984555869, 0.34973995695468, 0.35008166923759,
	0.35042359249004, 0.35076572676648, 0.35110807212133, 0.351450628609,
	0.35179339628391, 0.35213637520045, 0.35247956541304, 0.35282296697607,
	0.35316657994392, 0.35351040437098, 0.35385444031163, 0.35419868782024,
	0.35454314695118, 0.35488781775882, 0.35523270029751, 0.35557779462159,
	0.35592310078542, 0.35626861884334, 0.35661434884968, 0.35696029085877,
	0.35730644492494, 0.35765281110249, 0.35799938944576, 0.35834618000904,
	0.35869318284664, 0.35904039801284, 0.35938782556196, 0.35973546554826,
	0.36008331802603, 0.36043138304955, 0.36077966067308, 0.36112815095089,
	0.36147685393724, 0.36182576968638, 0.36217489825256, 0.36252423969002,
	0.362873794053, 0.36322356139573, 0.36357354177243, 0.36392373523733,
	0.36427414184463, 0.36462476164857, 0.36497559470332, 0.3653266410631,
	0.3656779007821, 0.36602937391451, 0.3663810605145, 0.36673296063627,
	0.36708507433397, 0.36743740166178, 0.36778994267385, 0.36814269742435,
	0.36849566596742, 0.36884884835721, 0.36920224464786, 0.3695558548935,
	0.36990967914827, 0.37026371746628, 0.37061796990165, 0.3709724365085,
	0.37132711734094, 0.37168201245306, 0.37203712189897, 0.37239244573275,
	0.37274798400849, 0.37310373678028, 0.37345970410218, 0.37381588602827,
	0.37417228261261, 0.37452889390927, 0.37488571997229, 0.37524276085572,
	0.37560001661362, 0.37595748730001, 0.37631517296893, 0.3766730736744,
	0.37703118947046, 0.37738952041111, 0.37774806655036, 0.37810682794223,
	0.37846580464072, 0.37882499669981, 0.37918440417351, 0.37954402711578,
	0.37990386558063, 0.38026391962201, 0.38062418929389, 0.38098467465025,
	0.38134537574504, 0.38170629263222, 0.38206742536572, 0.38242877399949,
	0.38279033858748, 0.38315211918361, 0.38351411584181, 0.383876328616,
	0.3842387575601, 0.38460140272801, 0.38496426417365, 0.38532734195091,
	0.38569063611369, 0.38605414671587, 0.38641787381135, 0.38678181745401,
	0.38714597769771, 0.38751035459632, 0.38787494820372, 0.38823975857375,
	0.38860478576027, 0.38897002981712, 0.38933549079816, 0.38970116875722,
	0.39006706374812, 0.39043317582471, 0.39079950504079, 0.39116605145018,
	0.3915328151067, 0.39189979606415, 0.39226699437633, 0.39263441009704,
	0.39300204328007, 0.3933698939792, 0.39373796224821, 0.39410624814088,
	0.39447475171098, 0.39484347301227, 0.3952124120985, 0.39558156902344,
	0.39595094384084, 0.39632053660442, 0.39669034736794, 0.39706037618513,
	0.39743062310971, 0.39780108819541, 0.39817177149594, 0.39854267306501,
	0.39891379295634, 0.39928513122363, 0.39965668792057, 0.40002846310085,
	0.40040045681815, 0.40077266912617, 0.40114510007858, 0.40151774972904,
	0.40189061813123, 0.4022637053388, 0.40263701140541, 0.40301053638471,
	0.40338428033034, 0.40375824329594, 0.40413242533516, 0.40450682650161,
	0.40488144684892, 0.40525628643071, 0.40563134530061, 0.4060066235122,
	0.4063821211191, 0.40675783817491, 0.40713377473322, 0.40750993084762,
	0.4078863065717, 0.40826290195902, 0.40863971706316, 0.4090167519377,
	0.40939400663619, 0.40977148121219, 0.41014917571926, 0.41052709021094,
	0.41090522474077, 0.4112835793623, 0.41166215412905, 0.41204094909454,
	0.41241996431232, 0.41279919983588, 0.41317865571874, 0.4135583320144,
	0.41393822877638, 0.41431834605816, 0.41469868391324, 0.41507924239511,
	0.41546002155723, 0.4158410214531, 0.41622224213618, 0.41660368365993,
	0.41698534607783, 0.41736722944331, 0.41774933380984, 0.41813165923087,
	0.41851420575982, 0.41889697345014, 0.41927996235525, 0.41966317252859,
	0.42004660402357, 0.4204302568936, 0.42081413119211, 0.42119822697248,
	0.42158254428813, 0.42196708319245, 0.42235184373882, 0.42273682598064,
	0.42312202997128, 0.42350745576412, 0.42389310341253, 0.42427897296987,
	0.4246650644895, 0.42505137802478, 0.42543791362905, 0.42582467135567,
	0.42621165125797, 0.42659885338928, 0.42698627780293, 0.42737392455226,
	0.42776179369058, 0.4281498852712, 0.42853819934743, 0.42892673597259,
	0.42931549519996, 0.42970447708284, 0.43009368167452, 0.43048310902829,
	0.43087275919743, 0.4312626322352, 0.43165272819489, 0.43204304712975,
	0.43243358909304, 0.43282435413803, 0.43321534231795, 0.43360655368605,
	0.43399798829558, 0.43438964619976, 0.43478152745183, 0.43517363210501,
	0.43556596021252, 0.43595851182758, 0.4363512870034, 0.43674428579317,
	0.43713750825011, 0.4375309544274, 0.43792462437823, 0.4383185181558,
	0.43871263581327, 0.43910697740383, 0.43950154298065, 0.43989633259688,
	0.4402913463057, 0.44068658416025, 0.44108204621369, 0.44147773251916,
	0.44187364312981, 0.44226977809876, 0.44266613747915, 0.4430627213241,
	0.44345952968674, 0.44385656262017, 0.44425382017752, 0.44465130241189,
	0.44504900937637, 0.44544694112407, 0.44584509770807, 0.44624347918146,
	0.44664208559733, 0.44704091700875, 0.44743997346879, 0.44783925503051,
	0.44823876174698, 0.44863849367126, 0.4490384508564, 0.44943863335544,
	0.44983904122143, 0.4502396745074, 0.45064053326639, 0.45104161755143,
	0.45144292741553, 0.45184446291171, 0.45224622409299, 0.45264821101237,
	0.45305042372286, 0.45345286227746, 0.45385552672916, 0.45425841713094,
	0.45466153353579, 0.4550648759967, 0.45546844456662, 0.45587223929854,
	0.45627626024541, 0.4566805074602, 0.45708498099586, 0.45748968090533,
	0.45789460724157, 0.45829976005752, 0.4587051394061, 0.45911074534024,
	0.45951657791288, 0.45992263717694, 0.46032892318532, 0.46073543599094,
	0.46114217564671, 0.46154914220552, 0.46195633572028, 0.46236375624386,
	0.46277140382917, 0.46317927852908, 0.46358738039647, 0.46399570948421,
	0.46440426584516, 0.4648130495322, 0.46522206059817, 0.46563129909593,
	0.46604076507833, 0.46645045859821, 0.46686037970841, 0.46727052846176,
	0.46768090491109, 0.46809150910922, 0.46850234110898, 0.46891340096318,
	0.46932468872462, 0.46973620444612, 0.47014794818046, 0.47055991998046,
	0.47097211989889, 0.47138454798854, 0.47179720430219, 0.47221008889263,
	0.47262320181261, 0.47303654311491, 0.47345011285228, 0.47386391107749,
	0.47427793784329, 0.47469219320242, 0.47510667720762, 0.47552138991164,
	0.4759363313672, 0.47635150162704, 0.47676690074387, 0.47718252877042,
	0.4775983857594, 0.47801447176352, 0.47843078683548, 0.47884733102798,
	0.47926410439372, 0.47968110698539, 0.48009833885567, 0.48051580005724,
	0.48093349064278, 0.48135141066496, 0.48176956017644, 0.4821879392299,
	0.48260654787797, 0.48302538617333, 0.4834444541686, 0.48386375191645,
	0.48428327946949, 0.48470303688038, 0.48512302420173, 0.48554324148617,
	0.48596368878631, 0.48638436615478, 0.48680527364418, 0.48722641130712,
	0.4876477791962, 0.488069377364, 0.48849120586313, 0.48891326474617,
	0.4893355540657, 0.48975807387429, 0.49018082422452, 0.49060380516896,
	0.49102701676017, 0.4914504590507, 0.49187413209311, 0.49229803593995,
	0.49272217064376, 0.49314653625707, 0.49357113283243, 0.49399596042237,
	0.4944210190794, 0.49484630885605, 0.49527182980484, 0.49569758197827,
	0.49612356542885, 0.49654978020909, 0.49697622637147, 0.4974029039685,
	0.49782981305266, 0.49825695367642, 0.49868432589228, 0.4991119297527,
	0.49953976531016, 0.49996783261711, 0.50039613172602, 0.50082466268933,
	0.50125342555951, 0.501682420389, 0.50211164723024, 0.50254110613565,
	0.50297079715769, 0.50340072034877, 0.50383087576131, 0.50426126344773,
	0.50469188346045, 0.50512273585188, 0.50555382067441, 0.50598513798045,
	0.50641668782239, 0.50684847025262, 0.50728048532353, 0.5077127330875,
	0.5081452135969, 0.5085779269041, 0.50901087306148, 0.50944405212138,
	0.50987746413618, 0.51031110915823, 0.51074498723987, 0.51117909843344,
	0.51161344279129, 0.51204802036574, 0.51248283120914, 0.5129178753738,
	0.51335315291205, 0.5137886638762, 0.51422440831856, 0.51466038629144,
	0.51509659784714, 0.51553304303796, 0.5159697219162, 0.51640663453413,
	0.51684378094405, 0.51728116119824, 0.51771877534896, 0.5181566234485,
	0.51859470554911, 0.51903302170305, 0.51947157196259, 0.51991035637997,
	0.52034937500744, 0.52078862789725, 0.52122811510162, 0.52166783667281,
	0.52210779266303, 0.52254798312451, 0.52298840810946, 0.52342906767012,
	0.52386996185867, 0.52431109072734, 0.52475245432832, 0.52519405271381,
	0.52563588593601, 0.52607795404709, 0.52652025709925, 0.52696279514466,
	0.5274055682355, 0.52784857642394, 0.52829181976214, 0.52873529830226,
	0.52917901209646, 0.52962296119689, 0.5300671456557, 0.53051156552503,
	0.53095622085702, 0.5314011117038, 0.53184623811751, 0.53229160015026,
	0.53273719785417, 0.53318303128137, 0.53362910048396, 0.53407540551404,
	0.53452194642373, 0.53496872326512, 0.53541573609029, 0.53586298495135,
	0.53631046990037, 0.53675819098943, 0.53720614827061, 0.53765434179597,
	0.53810277161759, 0.53855143778752, 0.53900034035783, 0.53944947938056,
	0.53989885490776, 0.54034846699147, 0.54079831568374, 0.5412484010366,
	0.54169872310208, 0.5421492819322, 0.54260007757898, 0.54305111009445,
	0.54350237953061, 0.54395388593946, 0.54440562937303, 0.54485760988329,
	0.54530982752225, 0.5457622823419, 0.54621497439422, 0.54666790373118,
	0.54712107040478, 0.54757447446698, 0.54802811596974, 0.54848199496503,
	0.54893611150481, 0.54939046564103, 0.54984505742564, 0.55029988691059,
	0.55075495414782, 0.55121025918926, 0.55166580208684, 0.55212158289249,
	0.55257760165814, 0.5530338584357, 0.55349035327709, 0.55394708623421,
	0.55440405735898, 0.55486126670328, 0.55531871431903, 0.5557764002581,
	0.55623432457239, 0.55669248731377, 0.55715088853414, 0.55760952828536,
	0.55806840661929, 0.55852752358782, 0.55898687924279, 0.55944647363606,
	0.55990630681949, 0.56036637884492, 0.5608266897642, 0.56128723962916,
	0.56174802849164, 0.56220905640347, 0.56267032341647, 0.56313182958246,
	0.56359357495327, 0.5640555595807, 0.56451778351656, 0.56498024681266,
	0.56544294952078, 0.56590589169274, 0.56636907338031, 0.56683249463529,
	0.56729615550945, 0.56776005605458, 0.56822419632244, 0.5686885763648,
	0.56915319623344, 0.56961805598009, 0.57008315565653, 0.5705484953145,
	0.57101407500575, 0.57147989478202, 0.57194595469505, 0.57241225479657,
	0.57287879513832, 0.573345575772, 0.57381259674935, 0.57427985812208,
	0.57474735994191, 0.57521510226053, 0.57568308512966, 0.57615130860098,
	0.5766197727262, 0.577088477557, 0.57755742314508, 0.5780266095421,
	0.57849603679975, 0.57896570496969, 0.57943561410361, 0.57990576425315,
	0.58037615546998, 0.58084678780576, 0.58131766131213, 0.58178877604074,
	0.58226013204324, 0.58273172937125, 0.58320356807642, 0.58367564821037,
	0.58414796982473, 0.58462053297111, 0.58509333770113, 0.58556638406641,
	0.58603967211855, 0.58651320190916, 0.58698697348982, 0.58746098691214,
	0.58793524222771, 0.58840973948811, 0.58888447874493, 0.58935946004973,
	0.5898346834541, 0.5903101490096, 0.5907858567678, 0.59126180678025,
	0.59173799909851, 0.59221443377413, 0.59269111085866, 0.59316803040364,
	0.59364519246061, 0.59412259708111, 0.59460024431666, 0.59507813421878,
	0.59555626683901, 0.59603464222885, 0.59651326043982, 0.59699212152342,
	0.59747122553117, 0.59795057251456, 0.59843016252508, 0.59890999561423,
	0.59939007183349, 0.59987039123435, 0.60035095386828, 0.60083175978676,
	0.60131280904126, 0.60179410168324, 0.60227563776417, 0.6027574173355,
	0.60323944044868, 0.60372170715517, 0.6042042175064, 0.60468697155381,
	0.60516996934885, 0.60565321094295, 0.60613669638752, 0.60662042573399,
	0.60710439903379, 0.60758861633832, 0.608073077699, 0.60855778316724,
	0.60904273279442, 0.60952792663196, 0.61001336473124, 0.61049904714365,
	0.61098497392058, 0.6114711451134, 0.61195756077351, 0.61244422095225,
	0.61293112570101, 0.61341827507115, 0.61390566911402, 0.61439330788099,
	0.61488119142339, 0.61536931979259, 0.61585769303991, 0.61634631121671,
	0.61683517437431, 0.61732428256403, 0.61781363583722, 0.61830323424519,
	0.61879307783925, 0.61928316667072, 0.61977350079091, 0.62026408025111,
	0.62075490510265, 0.62124597539679, 0.62173729118485, 0.62222885251811,
	0.62272065944784, 0.62321271202534, 0.62370501030187, 0.6241975543287,
	0.62469034415711, 0.62518337983835, 0.62567666142368, 0.62617018896436,
	0.62666396251164, 0.62715798211676, 0.62765224783096, 0.62814675970548,
	0.62864151779155, 0.62913652214041, 0.62963177280327, 0.63012726983136,
	0.6306230132759, 0.63111900318809, 0.63161523961915, 0.63211172262027,
	0.63260845224266, 0.63310542853751, 0.63360265155602, 0.63410012134937,
	0.63459783796874, 0.63509580146532, 0.63559401189028, 0.63609246929478,
	0.63659117373, 0.63709012524709, 0.63758932389722, 0.63808876973154,
	0.6385884628012, 0.63908840315734, 0.63958859085111, 0.64008902593364,
	0.64058970845606, 0.64109063846951, 0.64159181602511, 0.64209324117399,
	0.64259491396724, 0.643096834456, 0.64359900269137, 0.64410141872445,
	0.64460408260634, 0.64510699438813, 0.64561015412093, 0.64611356185582,
	0.64661721764387, 0.64712112153618, 0.64762527358381, 0.64812967383784,
	0.64863432234933, 0.64913921916935, 0.64964436434895, 0.6501497579392,
	0.65065539999113, 0.6511612905558, 0.65166742968425, 0.65217381742752,
	0.65268045383664, 0.65318733896264, 0.65369447285655, 0.65420185556939,
	0.65470948715217, 0.65521736765592, 0.65572549713163, 0.65623387563033,
	0.65674250320299, 0.65725137990063, 0.65776050577424, 0.65826988087481,
	0.65877950525331, 0.65928937896074, 0.65979950204806, 0.66030987456626,
	0.6608204965663, 0.66133136809914, 0.66184248921574, 0.66235385996707,
	0.66286548040406, 0.66337735057768, 0.66388947053887, 0.66440184033856,
	0.66491446002769, 0.66542732965719, 0.665940449278, 0.66645381894103,
	0.6669674386972, 0.66748130859743, 0.66799542869264, 0.66850979903372,
	0.66902441967159, 0.66953929065713, 0.67005441204126, 0.67056978387485,
	0.6710854062088, 0.67160127909398, 0.67211740258129, 0.67263377672158,
	0.67315040156574, 0.67366727716463, 0.67418440356911, 0.67470178083004,
	0.67521940899828, 0.67573728812468, 0.67625541826008, 0.67677379945533,
	0.67729243176127, 0.67781131522872, 0.67833044990852, 0.67884983585151,
	0.67936947310849, 0.67988936173029, 0.68040950176772, 0.68092989327159,
	0.68145053629271, 0.68197143088189, 0.68249257708991, 0.68301397496758,
	0.68353562456568, 0.68405752593501, 0.68457967912633, 0.68510208419044,
	0.6856247411781, 0.68614765014009, 0.68667081112717, 0.6871942241901,
	0.68771788937965, 0.68824180674656, 0.68876597634159, 0.68929039821548,
	0.68981507241897, 0.69033999900281, 0.69086517801772, 0.69139060951444,
	0.69191629354369, 0.6924422301562, 0.69296841940269, 0.69349486133386,
	0.69402155600042, 0.69454850345309, 0.69507570374257, 0.69560315691955,
	0.69613086303473, 0.6966588221388, 0.69718703428244, 0.69771549951635,
	0.69824421789118, 0.69877318945763, 0.69930241426635, 0.69983189236803,
	0.70036162381331, 0.70089160865286, 0.70142184693733, 0.70195233871737,
	0.70248308404363, 0.70301408296675, 0.70354533553737, 0.70407684180613,
	0.70460860182365, 0.70514061564056, 0.70567288330749, 0.70620540487505,
	0.70673818039385, 0.70727120991452, 0.70780449348765, 0.70833803116385,
	0.70887182299372, 0.70940586902785, 0.70994016931683, 0.71047472391125,
	0.7110095328617, 0.71154459621875, 0.71207991403298, 0.71261548635496,
	0.71315131323526, 0.71368739472443, 0.71422373087305, 0.71476032173166,
	0.71529716735082, 0.71583426778107, 0.71637162307296, 0.71690923327704,
	0.71744709844382, 0.71798521862386, 0.71852359386766, 0.71906222422577,
	0.71960110974869, 0.72014025048695, 0.72067964649106, 0.72121929781153,
	0.72175920449885, 0.72229936660354, 0.72283978417608, 0.72338045726698,
	0.72392138592671, 0.72446257020577, 0.72500401015464, 0.72554570582379,
	0.72608765726369, 0.72662986452482, 0.72717232765764, 0.72771504671262,
	0.7282580217402, 0.72880125279085, 0.72934473991501, 0.72988848316313,
	0.73043248258565, 0.73097673823301, 0.73152125015565, 0.73206601840399,
	0.73261104302846, 0.73315632407948, 0.73370186160748, 0.73424765566286,
	0.73479370629604, 0.73534001355743, 0.73588657749742, 0.73643339816642,
	0.73698047561482, 0.73752780989302, 0.73807540105139, 0.73862324914034,
	0.73917135421023, 0.73971971631144, 0.74026833549434, 0.7408172118093,
	0.74136634530669, 0.74191573603687, 0.74246538405019, 0.74301528939701,
	0.74356545212767, 0.74411587229252, 0.7446665499419, 0.74521748512615,
	0.7457686778956, 0.74632012830059, 0.74687183639143, 0.74742380221845,
	0.74797602583196, 0.74852850728229, 0.74908124661974, 0.74963424389462,
	0.75018749915723, 0.75074101245787, 0.75129478384684, 0.75184881337442,
	0.7524031010909, 0.75295764704657, 0.75351245129171, 0.75406751387658,
	0.75462283485148, 0.75517841426665, 0.75573425217237, 0.7562903486189,
	0.75684670365648, 0.75740331733539, 0.75796018970586, 0.75851732081814,
	0.75907471072247, 0.75963235946909, 0.76019026710824, 0.76074843369013,
	0.761306859265, 0.76186554388308, 0.76242448759457, 0.76298369044969,
	0.76354315249866, 0.76410287379168, 0.76466285437895, 0.76522309431068,
	0.76578359363705, 0.76634435240827, 0.76690537067451, 0.76746664848597,
	0.76802818589281, 0.76858998294523, 0.76915203969339, 0.76971435618745,
	0.7702769324776, 0.77083976861397, 0.77140286464675, 0.77196622062607,
	0.77252983660208, 0.77309371262494, 0.77365784874479, 0.77422224501176,
	0.77478690147599, 0.77535181818761, 0.77591699519674, 0.77648243255352,
	0.77704813030805, 0.77761408851046, 0.77818030721085, 0.77874678645935,
	0.77931352630604, 0.77988052680103, 0.78044778799442, 0.78101530993629,
	0.78158309267675, 0.78215113626588, 0.78271944075374, 0.78328800619044,
	0.78385683262603, 0.7844259201106, 0.78499526869419, 0.78556487842689,
	0.78613474935875, 0.78670488153982, 0.78727527502015, 0.7878459298498,
	0.7884168460788, 0.7889880237572, 0.78955946293504, 0.79013116366234,
	0.79070312598914, 0.79127534996546, 0.79184783564132, 0.79242058306674,
	0.79299359229174, 0.79356686336632, 0.79414039634049, 0.79471419126426,
	0.79528824818762, 0.79586256716057, 0.7964371482331, 0.7970119914552,
	0.79758709687685, 0.79816246454804, 0.79873809451873, 0.79931398683891,
	0.79989014155853, 0.80046655872758, 0.801043238396, 0.80162018061376,
	0.80219738543082, 0.80277485289711, 0.80335258306259, 0.8039305759772,
	0.80450883169089, 0.80508735025358, 0.80566613171521, 0.8062451761257,
	0.80682448353499, 0.80740405399299, 0.80798388754962, 0.80856398425479,
	0.80914434415841, 0.80972496731039, 0.81030585376063, 0.81088700355903,
	0.81146841675549, 0.81205009339988, 0.81263203354211, 0.81321423723206,
	0.8137967045196, 0.81437943545461, 0.81496243008697, 0.81554568846654,
	0.81612921064319, 0.81671299666678, 0.81729704658716, 0.8178813604542,
	0.81846593831774, 0.81905078022763, 0.81963588623371, 0.82022125638583,
	0.82080689073381, 0.82139278932749, 0.82197895221669, 0.82256537945125,
	0.82315207108098, 0.8237390271557, 0.82432624772523, 0.82491373283936,
	0.82550148254792, 0.8260894969007, 0.8266777759475, 0.82726631973811,
	0.82785512832233, 0.82844420174995, 0.82903354007075, 0.82962314333451,
	0.830213011591, 0.83080314489, 0.83139354328128, 0.83198420681461,
	0.83257513553975, 0.83316632950645, 0.83375778876447, 0.83434951336356,
	0.83494150335347, 0.83553375878394, 0.83612627970472, 0.83671906616554,
	0.83731211821613, 0.83790543590622, 0.83849901928554, 0.83909286840381,
	0.83968698331074, 0.84028136405606, 0.84087601068947, 0.84147092326068,
	0.8420661018194, 0.84266154641532, 0.84325725709814, 0.84385323391755,
	0.84444947692325, 0.84504598616491, 0.84564276169222, 0.84623980355487,
	0.84683711180251, 0.84743468648483, 0.84803252765149, 0.84863063535215,
	0.84922900963648, 0.84982765055413, 0.85042655815475, 0.851025732488,
	0.85162517360351, 0.85222488155094, 0.85282485637992, 0.85342509814007,
	0.85402560688105, 0.85462638265246, 0.85522742550394, 0.85582873548511,
	0.85643031264558, 0.85703215703497, 0.85763426870288, 0.85823664769892,
	0.8588392940727, 0.8594422078738, 0.86004538915184, 0.86064883795639,
	0.86125255433704, 0.86185653834339, 0.86246079002501, 0.86306530943147,
	0.86367009661236, 0.86427515161723, 0.86488047449566, 0.86548606529721,
	0.86609192407144, 0.8666980508679, 0.86730444573615, 0.86791110872573,
	0.86851803988619, 0.86912523926706, 0.86973270691789, 0.87034044288821,
	0.87094844722756, 0.87155671998544, 0.8721652612114, 0.87277407095495,
	0.8733831492656, 0.87399249619287, 0.87460211178626, 0.87521199609529,
	0.87582214916945, 0.87643257105824, 0.87704326181115, 0.87765422147768,
	0.87826545010732, 0.87887694774954, 0.87948871445382, 0.88010075026965,
	0.8807130552465, 0.88132562943383, 0.88193847288112, 0.88255158563782,
	0.88316496775339, 0.88377861927729, 0.88439254025897, 0.88500673074788,
	0.88562119079346, 0.88623592044515, 0.88685091975239, 0.88746618876462,
	0.88808172753126, 0.88869753610175, 0.88931361452549, 0.88992996285193,
	0.89054658113046, 0.8911634694105, 0.89178062774147, 0.89239805617276,
	0.89301575475378, 0.89363372353393, 0.8942519625626, 0.89487047188919,
	0.89548925156308, 0.89610830163365, 0.89672762215028, 0.89734721316236,
	0.89796707471926, 0.89858720687034, 0.89920760966498, 0.89982828315253,
	0.90044922738236, 0.90107044240382, 0.90169192826626, 0.90231368501904,
	0.90293571271149, 0.90355801139296, 0.9041805811128, 0.90480342192032,
	0.90542653386487, 0.90604991699577, 0.90667357136235, 0.90729749701392,
	0.90792169399981, 0.90854616236932, 0.90917090217177, 0.90979591345647,
	0.91042119627271, 0.91104675066981, 0.91167257669704, 0.91229867440371,
	0.9129250438391, 0.91355168505251, 0.91417859809321, 0.91480578301048,
	0.91543323985359, 0.91606096867182, 0.91668896951444, 0.91731724243071,
	0.91794578746989, 0.91857460468124, 0.91920369411401, 0.91983305581745,
	0.92046268984082, 0.92109259623335, 0.92172277504428, 0.92235322632286,
	0.92298395011831, 0.92361494647986, 0.92424621545674, 0.92487775709817,
	0.92550957145338, 0.92614165857157, 0.92677401850196, 0.92740665129376,
	0.92803955699618, 0.92867273565841, 0.92930618732965, 0.9299399120591,
	0.93057390989595, 0.93120818088939, 0.9318427250886, 0.93247754254276,
	0.93311263330105, 0.93374799741265, 0.93438363492672, 0.93501954589243,
	0.93565573035895, 0.93629218837543, 0.93692891999104, 0.93756592525492,
	0.93820320421622, 0.9388407569241, 0.93947858342768, 0.94011668377613,
	0.94075505801856, 0.94139370620411, 0.94203262838191, 0.94267182460109,
	0.94331129491077, 0.94395103936006, 0.94459105799809, 0.94523135087395,
	0.94587191803677, 0.94651275953565, 0.94715387541968, 0.94779526573797,
	0.94843693053961, 0.94907886987369, 0.9497210837893, 0.95036357233552,
	0.95100633556143, 0.95164937351612, 0.95229268624865, 0.95293627380809,
	0.95358013624352, 0.95422427360399, 0.95486868593856, 0.9555133732963,
	0.95615833572626, 0.95680357327747, 0.957449085999, 0.95809487393989,
	0.95874093714916, 0.95938727567587, 0.96003388956904, 0.9606807788777,
	0.96132794365088, 0.9619753839376, 0.96262309978687, 0.96327109124772,
	0.96391935836916, 0.96456790120019, 0.96521671978982, 0.96586581418706,
	0.96651518444089, 0.96716483060032, 0.96781475271434, 0.96846495083193,
	0.96911542500208, 0.96976617527377, 0.97041720169599, 0.97106850431769,
	0.97172008318786, 0.97237193835547, 0.97302406986947, 0.97367647777883,
	0.97432916213251, 0.97498212297945, 0.97563536036862, 0.97628887434896,
	0.9769426649694, 0.9775967322789, 0.97825107632639, 0.9789056971608,
	0.97956059483107, 0.98021576938612, 0.98087122087487, 0.98152694934625,
	0.98218295484916, 0.98283923743254, 0.98349579714528, 0.98415263403628,
	0.98480974815447, 0.98546713954873, 0.98612480826796, 0.98678275436105,
	0.9874409778769, 0.98809947886439, 0.98875825737241, 0.98941731344982,
	0.99007664714552, 0.99073625850836, 0.99139614758723, 0.99205631443099,
	0.9927167590885, 0.99337748160862, 0.9940384820402, 0.9946997604321,
	0.99536131683317, 0.99602315129226, 0.9966852638582, 0.99734765457984,
	0.998010323506, 0.99867327068553, 0.99933649616726, 1,
};
inline float concave_exponant( float value )
{
	return compute_exponant( value, concave_exponant_table, concave_exponant_table_size );
}

static int convex_exponant_table_size = 4096;
static float convex_exponant_table[4096] = {
	0.046890735084968, 0.060510367089532, 0.070244146759565, 0.078085884528682,
	0.084766399299554, 0.090646885757071, 0.095935918022615, 0.10076629271815,
	0.10522846666912, 0.10938719406252, 0.11329063214116, 0.11697569515052,
	0.12047138381695, 0.12380095142677, 0.12698336637053, 0.13003433090946,
	0.13296700988896, 0.13579256402953, 0.13852054804987, 0.14115921312863,
	0.1437157402818, 0.14619642294463, 0.14860681160019, 0.15095182963725,
	0.15323586711087, 0.1554628573286, 0.15763633994212, 0.15975951333014,
	0.16183527840461, 0.16386627548964, 0.16585491556127, 0.16780340686311,
	0.16971377770443, 0.17158789608648, 0.17342748667809, 0.17523414556343,
	0.17700935310772, 0.17875448522512, 0.18047082328378, 0.18215956284339,
	0.18382182138835, 0.18545864519372, 0.18707101543914, 0.18865985366887,
	0.19022602668079, 0.19177035091555, 0.19329359640666, 0.19479649034366,
	0.19627972029366, 0.1977439371202, 0.19918975763326, 0.20061776700013,
	0.20202852094263, 0.20342254774346, 0.20480035008141, 0.20616240671285,
	0.20750917401484, 0.20884108740361, 0.2101585626402, 0.21146199703427,
	0.21275177055533, 0.21402824686007, 0.21529177424328, 0.21654268651924,
	0.21778130383952, 0.21900793345289, 0.2202228704121, 0.22142639823195,
	0.22261878950286, 0.22380030646332, 0.22497120153468, 0.22613171782122,
	0.2272820895781, 0.22842254264984, 0.2295532948814, 0.23067455650401,
	0.23178653049755, 0.23288941293122, 0.23398339328412, 0.23506865474701,
	0.23614537450672, 0.23721372401435, 0.23827386923838, 0.23932597090372,
	0.24037018471754, 0.24140666158298, 0.24243554780127, 0.24345698526318,
	0.24447111163043, 0.24547806050768, 0.24647796160566, 0.24747094089613,
	0.24845712075893, 0.24943662012184, 0.2504095545935, 0.25137603658992,
	0.25233617545488, 0.25329007757457, 0.2542378464869, 0.25517958298564,
	0.25611538521981, 0.25704534878847, 0.25796956683127, 0.25888813011489,
	0.2598011271157, 0.26070864409871, 0.26161076519316, 0.26250757246479,
	0.26339914598504, 0.26428556389732, 0.26516690248049, 0.26604323620968,
	0.26691463781467, 0.26778117833578, 0.26864292717761, 0.26949995216049,
	0.27035231957004, 0.27120009420461, 0.27204333942098, 0.27288211717822,
	0.27371648807992, 0.27454651141474, 0.27537224519548, 0.27619374619664,
	0.27701106999062, 0.27782427098256, 0.27863340244388, 0.27943851654466,
	0.2802396643848, 0.28103689602406, 0.28183026051108, 0.28261980591135,
	0.28340557933416, 0.28418762695871, 0.28496599405919, 0.28574072502914,
	0.28651186340489, 0.28727945188822, 0.28804353236828, 0.28880414594282,
	0.28956133293859, 0.29031513293125, 0.29106558476448, 0.29181272656855,
	0.29255659577828, 0.2932972291504, 0.29403466278042, 0.29476893211889,
	0.29550007198725, 0.29622811659308, 0.29695309954495, 0.29767505386682,
	0.29839401201195, 0.29911000587644, 0.29982306681234, 0.30053322564037,
	0.30124051266227, 0.30194495767279, 0.30264658997128, 0.30334543837304,
	0.30404153122028, 0.30473489639272, 0.305425561318, 0.30611355298169,
	0.30679889793711, 0.30748162231476, 0.30816175183163, 0.3088393118001,
	0.30951432713675, 0.31018682237079, 0.31085682165234, 0.31152434876047,
	0.31218942711103, 0.31285207976424, 0.31351232943213, 0.31417019848572,
	0.31482570896207, 0.3154788825711, 0.3161297407023, 0.31677830443114,
	0.31742459452547, 0.31806863145166, 0.31871043538058, 0.31935002619349,
	0.31998742348773, 0.32062264658228, 0.3212557145232, 0.32188664608892,
	0.32251545979539, 0.32314217390113, 0.32376680641214, 0.32438937508668,
	0.32500989743996, 0.32562839074873, 0.32624487205568, 0.32685935817386,
	0.32747186569086, 0.32808241097303, 0.32869101016948, 0.32929767921605,
	0.3299024338392, 0.33050528955976, 0.33110626169664, 0.33170536537043,
	0.33230261550691, 0.33289802684053, 0.33349161391775, 0.33408339110036,
	0.33467337256866, 0.33526157232465, 0.33584800419508, 0.3364326818345,
	0.33701561872814, 0.33759682819487, 0.33817632338997, 0.33875411730788,
	0.33933022278497, 0.3399046525021, 0.34047741898725, 0.34104853461804,
	0.34161801162422, 0.3421858620901, 0.3427520979569, 0.34331673102509,
	0.34387977295669, 0.34444123527747, 0.34500112937915, 0.34555946652156,
	0.3461162578347, 0.34667151432084, 0.34722524685648, 0.34777746619439,
	0.34832818296549, 0.34887740768077, 0.34942515073316, 0.34997142239933,
	0.35051623284148, 0.35105959210911, 0.35160151014072, 0.35214199676549,
	0.35268106170497, 0.35321871457465, 0.35375496488559, 0.35428982204593,
	0.35482329536248, 0.35535539404219, 0.35588612719361, 0.35641550382837,
	0.35694353286255, 0.35747022311812, 0.35799558332429, 0.35851962211881,
	0.35904234804937, 0.3595637695748, 0.36008389506641, 0.3606027328092,
	0.36112029100309, 0.36163657776411, 0.36215160112561, 0.36266536903939,
	0.36317788937685, 0.36368916993009, 0.36419921841305, 0.36470804246253,
	0.36521564963931, 0.36572204742913, 0.36622724324378, 0.36673124442203,
	0.3672340582307, 0.36773569186555, 0.36823615245229, 0.36873544704751,
	0.36923358263959, 0.36973056614959, 0.37022640443218, 0.37072110427648,
	0.37121467240697, 0.37170711548426, 0.37219844010601, 0.37268865280768,
	0.37317776006336, 0.3736657682866, 0.37415268383112, 0.37463851299161,
	0.37512326200452, 0.37560693704872, 0.37608954424632, 0.37657108966331,
	0.37705157931033, 0.37753101914332, 0.37800941506421, 0.37848677292164,
	0.37896309851155, 0.37943839757789, 0.37991267581323, 0.38038593885941,
	0.38085819230815, 0.38132944170167, 0.38179969253328, 0.382268950248,
	0.38273722024311, 0.38320450786875, 0.38367081842848, 0.38413615717984,
	0.38460052933487, 0.38506394006072, 0.3855263944801, 0.38598789767187,
	0.38644845467151, 0.38690807047168, 0.38736675002267, 0.38782449823291,
	0.38828131996948, 0.38873722005856, 0.38919220328593, 0.3896462743974,
	0.39009943809931, 0.39055169905894, 0.39100306190499, 0.39145353122801,
	0.39190311158081, 0.39235180747892, 0.39279962340099, 0.39324656378921,
	0.39369263304972, 0.394137835553, 0.39458217563428, 0.39502565759394,
	0.39546828569788, 0.39591006417789, 0.39635099723204, 0.39679108902508,
	0.39723034368872, 0.3976687653221, 0.39810635799203, 0.39854312573345,
	0.39897907254969, 0.39941420241284, 0.3998485192641, 0.40028202701409,
	0.40071472954319, 0.40114663070185, 0.40157773431091, 0.40200804416193,
	0.40243756401747, 0.40286629761141, 0.40329424864928, 0.40372142080848,
	0.40414781773867, 0.40457344306197, 0.4049983003733, 0.40542239324066,
	0.40584572520538, 0.40626829978239, 0.40669012046055, 0.40711119070284,
	0.40753151394667, 0.40795109360413, 0.40836993306225, 0.40878803568324,
	0.40920540480476, 0.40962204374014, 0.41003795577865, 0.41045314418572,
	0.41086761220321, 0.4112813630496, 0.41169439992025, 0.41210672598763,
	0.41251834440153, 0.4129292582893, 0.41333947075604, 0.41374898488486,
	0.41415780373708, 0.41456593035241, 0.41497336774921, 0.41538011892465,
	0.41578618685496, 0.41619157449558, 0.41659628478141, 0.41700032062698,
	0.41740368492665, 0.41780638055478, 0.41820841036598, 0.41860977719522,
	0.41901048385807, 0.41941053315088, 0.41980992785092, 0.42020867071661,
	0.42060676448765, 0.42100421188524, 0.42140101561222, 0.42179717835324,
	0.42219270277495, 0.42258759152616, 0.42298184723798, 0.42337547252401,
	0.4237684699805, 0.4241608421865, 0.42455259170399, 0.4249437210781,
	0.42533423283721, 0.42572412949311, 0.42611341354115, 0.42650208746043,
	0.42689015371388, 0.42727761474844, 0.42766447299521, 0.42805073086956,
	0.42843639077132, 0.42882145508485, 0.42920592617925, 0.42958980640844,
	0.42997309811132, 0.43035580361188, 0.43073792521938, 0.4311194652284,
	0.43150042591905, 0.43188080955705, 0.43226061839384, 0.43263985466676,
	0.43301852059911, 0.43339661840033, 0.43377415026607, 0.43415111837833,
	0.43452752490558, 0.43490337200286, 0.43527866181192, 0.4356533964613,
	0.43602757806647, 0.43640120872993, 0.43677429054132, 0.43714682557752,
	0.43751881590278, 0.43789026356879, 0.43826117061483, 0.43863153906783,
	0.43900137094251, 0.43937066824143, 0.43973943295517, 0.44010766706235,
	0.44047537252978, 0.44084255131254, 0.44120920535405, 0.44157533658622,
	0.4419409469295, 0.442306038293, 0.44267061257457, 0.44303467166088,
	0.44339821742753, 0.44376125173914, 0.44412377644942, 0.44448579340128,
	0.44484730442689, 0.44520831134779, 0.44556881597498, 0.44592882010897,
	0.4462883255399, 0.44664733404759, 0.44700584740166, 0.44736386736156,
	0.44772139567671, 0.44807843408653, 0.44843498432055, 0.44879104809845,
	0.44914662713019, 0.44950172311605, 0.44985633774669, 0.45021047270328,
	0.45056412965752, 0.45091731027174, 0.45127001619897, 0.45162224908301,
	0.45197401055849, 0.45232530225096, 0.45267612577693, 0.45302648274399,
	0.45337637475082, 0.4537258033873, 0.45407477023455, 0.45442327686502,
	0.45477132484252, 0.45511891572233, 0.45546605105124, 0.4558127323676,
	0.45615896120143, 0.45650473907442, 0.45685006750004, 0.45719494798358,
	0.45753938202222, 0.45788337110509, 0.45822691671331, 0.45857002032009,
	0.45891268339075, 0.45925490738277, 0.45959669374592, 0.45993804392221,
	0.46027895934605, 0.46061944144423, 0.460959491636, 0.46129911133316,
	0.46163830194005, 0.46197706485365, 0.46231540146362, 0.46265331315235,
	0.46299080129501, 0.46332786725963, 0.4636645124071, 0.46400073809126,
	0.46433654565894, 0.46467193645002, 0.46500691179744, 0.46534147302732,
	0.46567562145893, 0.46600935840479, 0.46634268517071, 0.46667560305582,
	0.46700811335262, 0.46734021734706, 0.46767191631853, 0.46800321153995,
	0.4683341042778, 0.46866459579217, 0.46899468733679, 0.46932438015909,
	0.46965367550024, 0.46998257459518, 0.47031107867269, 0.47063918895541,
	0.47096690665987, 0.47129423299659, 0.47162116917004, 0.47194771637876,
	0.47227387581533, 0.47259964866646, 0.47292503611303, 0.47325003933009,
	0.47357465948692, 0.4738988977471, 0.4742227552685, 0.47454623320333,
	0.47486933269822, 0.47519205489418, 0.47551440092673, 0.47583637192585,
	0.47615796901608, 0.47647919331652, 0.47680004594088, 0.47712052799752,
	0.47744064058947, 0.47776038481448, 0.47807976176506, 0.47839877252847,
	0.47871741818683, 0.4790356998171, 0.4793536184911, 0.4796711752756,
	0.47998837123232, 0.48030520741795, 0.4806216848842, 0.48093780467785,
	0.48125356784075, 0.48156897540986, 0.48188402841729, 0.48219872789033,
	0.48251307485147, 0.48282707031846, 0.4831407153043, 0.48345401081729,
	0.48376695786107, 0.48407955743463, 0.48439181053237, 0.48470371814407,
	0.48501528125501, 0.4853265008459, 0.48563737789297, 0.485947913368,
	0.48625810823832, 0.48656796346684, 0.48687748001209, 0.48718665882827,
	0.48749550086522, 0.48780400706849, 0.48811217837937, 0.48842001573489,
	0.48872752006785, 0.48903469230687, 0.48934153337639, 0.48964804419673,
	0.48995422568406, 0.49026007875049, 0.49056560430403, 0.49087080324867,
	0.49117567648439, 0.49148022490715, 0.49178444940896, 0.49208835087789,
	0.49239193019808, 0.49269518824979, 0.49299812590937, 0.49330074404936,
	0.49360304353847, 0.49390502524158, 0.49420669001982, 0.49450803873055,
	0.4948090722274, 0.49510979136027, 0.49541019697541, 0.49571028991536,
	0.49601007101904, 0.49630954112173, 0.49660870105513, 0.49690755164733,
	0.4972060937229, 0.49750432810282, 0.49780225560459, 0.4980998770422,
	0.49839719322618, 0.49869420496357, 0.498990913058, 0.49928731830969,
	0.49958342151544, 0.4998792234687, 0.50017472495954, 0.50046992677472,
	0.50076482969765, 0.50105943450848, 0.50135374198405, 0.50164775289795,
	0.50194146802055, 0.50223488811897, 0.50252801395713, 0.50282084629578,
	0.5031133858925, 0.5034056335017, 0.5036975898747, 0.50398925575966,
	0.50428063190168, 0.50457171904277, 0.50486251792189, 0.50515302927493,
	0.50544325383479, 0.50573319233134, 0.50602284549147, 0.50631221403909,
	0.50660129869514, 0.50689010017766, 0.50717861920172, 0.5074668564795,
	0.50775481272031, 0.50804248863054, 0.50832988491376, 0.50861700227068,
	0.50890384139919, 0.50919040299435, 0.50947668774845, 0.50976269635099,
	0.51004842948868, 0.51033388784553, 0.51061907210276, 0.51090398293892,
	0.51118862102982, 0.51147298704859, 0.5117570816657, 0.51204090554894,
	0.51232445936345, 0.51260774377177, 0.51289075943379, 0.51317350700682,
	0.51345598714556, 0.51373820050215, 0.51402014772616, 0.51430182946463,
	0.51458324636205, 0.5148643990604, 0.51514528819914, 0.51542591441526,
	0.51570627834325, 0.51598638061514, 0.51626622186053, 0.51654580270654,
	0.51682512377789, 0.51710418569688, 0.51738298908342, 0.51766153455502,
	0.51793982272681, 0.51821785421157, 0.51849562961973, 0.51877314955937,
	0.51905041463625, 0.51932742545382, 0.51960418261324, 0.51988068671335,
	0.52015693835075, 0.52043293811974, 0.5207086866124, 0.52098418441855,
	0.52125943212578, 0.52153443031947, 0.52180917958278, 0.52208368049669,
	0.52235793363999, 0.5226319395893, 0.52290569891908, 0.52317921220163,
	0.52345248000712, 0.5237255029036, 0.52399828145699, 0.5242708162311,
	0.52454310778766, 0.5248151566863, 0.5250869634846, 0.52535852873806,
	0.52562985300011, 0.52590093682218, 0.52617178075363, 0.52644238534181,
	0.52671275113208, 0.52698287866777, 0.52725276849023, 0.52752242113882,
	0.52779183715096, 0.52806101706207, 0.52832996140564, 0.52859867071322,
	0.52886714551442, 0.52913538633692, 0.52940339370652, 0.52967116814708,
	0.52993871018059, 0.53020602032715, 0.53047309910498, 0.53073994703044,
	0.53100656461803, 0.53127295238041, 0.53153911082841, 0.531805040471,
	0.53207074181537, 0.53233621536687, 0.53260146162906, 0.53286648110371,
	0.5331312742908, 0.53339584168853, 0.53366018379335, 0.53392430109994,
	0.53418819410123, 0.53445186328842, 0.53471530915096, 0.53497853217659,
	0.53524153285134, 0.5355043116595, 0.53576686908371, 0.53602920560487,
	0.53629132170224, 0.53655321785338, 0.53681489453419, 0.5370763522189,
	0.53733759138012, 0.53759861248879, 0.53785941601422, 0.5381200024241,
	0.53838037218451, 0.53864052575988, 0.53890046361309, 0.53916018620538,
	0.53941969399642, 0.53967898744431, 0.53993806700554, 0.54019693313509,
	0.54045558628632, 0.54071402691109, 0.54097225545968, 0.54123027238085,
	0.54148807812184, 0.54174567312834, 0.54200305784455, 0.54226023271316,
	0.54251719817534, 0.54277395467077, 0.54303050263768, 0.54328684251276,
	0.54354297473128, 0.54379889972702, 0.54405461793229, 0.54431012977796,
	0.54456543569347, 0.54482053610679, 0.54507543144448, 0.54533012213166,
	0.54558460859203, 0.5458388912479, 0.54609297052015, 0.54634684682827,
	0.54660052059034, 0.54685399222308, 0.5471072621418, 0.54736033076047,
	0.54761319849166, 0.5478658657466, 0.54811833293515, 0.54837060046582,
	0.54862266874578, 0.54887453818088, 0.54912620917561, 0.54937768213314,
	0.54962895745535, 0.54988003554276, 0.55013091679463, 0.55038160160888,
	0.55063209038215, 0.55088238350979, 0.55113248138587, 0.55138238440318,
	0.55163209295323, 0.55188160742626, 0.55213092821126, 0.55238005569596,
	0.55262899026685, 0.55287773230915, 0.55312628220686, 0.55337464034275,
	0.55362280709834, 0.55387078285396, 0.55411856798868, 0.55436616288041,
	0.5546135679058, 0.55486078344034, 0.5551078098583, 0.55535464753278,
	0.55560129683566, 0.55584775813768, 0.55609403180839, 0.55634011821614,
	0.55658601772817, 0.55683173071051, 0.55707725752808, 0.5573225985446,
	0.55756775412269, 0.5578127246238, 0.55805751040827, 0.55830211183529,
	0.55854652926293, 0.55879076304815, 0.55903481354678, 0.55927868111355,
	0.55952236610207, 0.55976586886487, 0.56000918975337, 0.56025232911789,
	0.56049528730769, 0.56073806467093, 0.56098066155468, 0.56122307830496,
	0.56146531526671, 0.56170737278381, 0.56194925119908, 0.56219095085428,
	0.56243247209012, 0.56267381524627, 0.56291498066136, 0.56315596867298,
	0.56339677961767, 0.56363741383097, 0.56387787164739, 0.56411815340039,
	0.56435825942246, 0.56459819004504, 0.56483794559858, 0.56507752641254,
	0.56531693281535, 0.56555616513448, 0.56579522369638, 0.56603410882653,
	0.56627282084942, 0.56651136008858, 0.56674972686654, 0.56698792150488,
	0.56722594432419, 0.56746379564414, 0.5677014757834, 0.56793898505971,
	0.56817632378984, 0.56841349228965, 0.56865049087401, 0.56888731985689,
	0.56912397955132, 0.56936047026937, 0.56959679232221, 0.5698329460201,
	0.57006893167234, 0.57030474958734, 0.57054040007261, 0.57077588343472,
	0.57101119997936, 0.5712463500113, 0.57148133383445, 0.57171615175177,
	0.57195080406539, 0.57218529107651, 0.57241961308546, 0.5726537703917,
	0.57288776329381, 0.57312159208949, 0.57335525707559, 0.57358875854806,
	0.57382209680204, 0.57405527213176, 0.57428828483063, 0.57452113519119,
	0.57475382350515, 0.57498635006337, 0.57521871515585, 0.57545091907177,
	0.57568296209947, 0.57591484452648, 0.57614656663946, 0.57637812872429,
	0.576609531066, 0.5768407739488, 0.57707185765611, 0.57730278247053,
	0.57753354867383, 0.57776415654699, 0.57799460637021, 0.57822489842285,
	0.57845503298351, 0.57868501032997, 0.57891483073924, 0.57914449448754,
	0.5793740018503, 0.57960335310217, 0.57983254851703, 0.58006158836799,
	0.58029047292737, 0.58051920246673, 0.58074777725689, 0.58097619756787,
	0.58120446366895, 0.58143257582865, 0.58166053431474, 0.58188833939424,
	0.58211599133342, 0.5823434903978, 0.58257083685217, 0.58279803096057,
	0.58302507298631, 0.58325196319197, 0.58347870183939, 0.58370528918969,
	0.58393172550327, 0.5841580110398, 0.58438414605822, 0.58461013081678,
	0.584835965573, 0.58506165058368, 0.58528718610493, 0.58551257239215,
	0.58573780970002, 0.58596289828256, 0.58618783839304, 0.58641263028408,
	0.58663727420757, 0.58686177041475, 0.58708611915613, 0.58731032068158,
	0.58753437524025, 0.58775828308063, 0.58798204445053, 0.58820565959709,
	0.58842912876676, 0.58865245220535, 0.58887563015797, 0.5890986628691,
	0.58932155058253, 0.5895442935414, 0.5897668919882, 0.58998934616477,
	0.59021165631227, 0.59043382267124, 0.59065584548155, 0.59087772498246,
	0.59109946141254, 0.59132105500977, 0.59154250601144, 0.59176381465425,
	0.59198498117425, 0.59220600580685, 0.59242688878684, 0.59264763034839,
	0.59286823072503, 0.5930886901497, 0.59330900885468, 0.59352918707167,
	0.59374922503172, 0.5939691229653, 0.59418888110226, 0.59440849967183,
	0.59462797890264, 0.59484731902273, 0.59506652025952, 0.59528558283985,
	0.59550450698994, 0.59572329293543, 0.59594194090138, 0.59616045111223,
	0.59637882379185, 0.59659705916352, 0.59681515744994, 0.59703311887322,
	0.5972509436549, 0.59746863201593, 0.59768618417671, 0.59790360035702,
	0.59812088077612, 0.59833802565267, 0.59855503520476, 0.59877190964993,
	0.59898864920516, 0.59920525408685, 0.59942172451085, 0.59963806069245,
	0.5998542628464, 0.60007033118687, 0.6002862659275, 0.60050206728137,
	0.60071773546102, 0.60093327067843, 0.60114867314505, 0.60136394307179,
	0.60157908066901, 0.60179408614653, 0.60200895971366, 0.60222370157913,
	0.60243831195119, 0.60265279103751, 0.60286713904528, 0.60308135618111,
	0.60329544265114, 0.60350939866094, 0.60372322441559, 0.60393692011965,
	0.60415048597713, 0.60436392219156, 0.60457722896594, 0.60479040650276,
	0.605003455004, 0.60521637467114, 0.60542916570514, 0.60564182830645,
	0.60585436267504, 0.60606676901036, 0.60627904751136, 0.60649119837651,
	0.60670322180376, 0.60691511799058, 0.60712688713394, 0.60733852943032,
	0.60755004507572, 0.60776143426563, 0.60797269719509, 0.60818383405862,
	0.60839484505026, 0.6086057303636, 0.60881649019172, 0.60902712472723,
	0.60923763416227, 0.6094480186885, 0.6096582784971, 0.6098684137788,
	0.61007842472383, 0.61028831152197, 0.61049807436255, 0.61070771343439,
	0.61091722892589, 0.61112662102497, 0.61133588991908, 0.61154503579523,
	0.61175405883996, 0.61196295923936, 0.61217173717906, 0.61238039284424,
	0.61258892641964, 0.61279733808953, 0.61300562803775, 0.61321379644768,
	0.61342184350227, 0.613629769384, 0.61383757427494, 0.61404525835669,
	0.61425282181043, 0.61446026481691, 0.61466758755641, 0.61487479020881,
	0.61508187295354, 0.61528883596959, 0.61549567943554, 0.61570240352953,
	0.61590900842926, 0.61611549431204, 0.61632186135471, 0.61652810973372,
	0.61673423962509, 0.6169402512044, 0.61714614464684, 0.61735192012716,
	0.61755757781971, 0.61776311789841, 0.61796854053677, 0.6181738459079,
	0.61837903418448, 0.6185841055388, 0.61878906014272, 0.61899389816771,
	0.61919861978482, 0.61940322516472, 0.61960771447765, 0.61981208789347,
	0.62001634558161, 0.62022048771113, 0.62042451445068, 0.62062842596851,
	0.62083222243249, 0.62103590401009, 0.62123947086836, 0.621442923174,
	0.62164626109329, 0.62184948479215, 0.62205259443607, 0.62225559019019,
	0.62245847221926, 0.62266124068763, 0.62286389575928, 0.6230664375978,
	0.62326886636641, 0.62347118222793, 0.62367338534484, 0.62387547587921,
	0.62407745399274, 0.62427931984677, 0.62448107360226, 0.62468271541978,
	0.62488424545956, 0.62508566388144, 0.62528697084491, 0.62548816650906,
	0.62568925103265, 0.62589022457405, 0.62609108729128, 0.626291839342,
	0.62649248088349, 0.62669301207269, 0.62689343306617, 0.62709374402015,
	0.62729394509048, 0.62749403643267, 0.62769401820186, 0.62789389055286,
	0.62809365364009, 0.62829330761765, 0.62849285263929, 0.62869228885839,
	0.62889161642799, 0.6290908355008, 0.62928994622917, 0.62948894876509,
	0.62968784326024, 0.62988662986593, 0.63008530873314, 0.63028388001252,
	0.63048234385435, 0.6306807004086, 0.63087894982489, 0.63107709225251,
	0.63127512784041, 0.63147305673721, 0.6316708790912, 0.63186859505031,
	0.63206620476219, 0.63226370837411, 0.63246110603304, 0.63265839788561,
	0.63285558407814, 0.63305266475659, 0.63324964006664, 0.63344651015361,
	0.6336432751625, 0.63383993523802, 0.63403649052451, 0.63423294116604,
	0.63442928730632, 0.63462552908877, 0.63482166665648, 0.63501770015222,
	0.63521362971845, 0.63540945549733, 0.63560517763068, 0.63580079626002,
	0.63599631152656, 0.63619172357121, 0.63638703253455, 0.63658223855687,
	0.63677734177813, 0.636972342338, 0.63716724037585, 0.63736203603072,
	0.63755672944138, 0.63775132074627, 0.63794581008353, 0.63814019759103,
	0.63833448340628, 0.63852866766656, 0.6387227505088, 0.63891673206965,
	0.63911061248547, 0.63930439189231, 0.63949807042594, 0.63969164822183,
	0.63988512541515, 0.64007850214078, 0.64027177853332, 0.64046495472706,
	0.64065803085603, 0.64085100705394, 0.64104388345422, 0.64123666019003,
	0.64142933739423, 0.64162191519939, 0.6418143937378, 0.64200677314148,
	0.64219905354214, 0.64239123507123, 0.64258331785991, 0.64277530203907,
	0.6429671877393, 0.64315897509092, 0.64335066422399, 0.64354225526827,
	0.64373374835326, 0.64392514360816, 0.64411644116192, 0.64430764114321,
	0.64449874368043, 0.64468974890169, 0.64488065693484, 0.64507146790748,
	0.6452621819469, 0.64545279918014, 0.645643319734, 0.64583374373496,
	0.64602407130926, 0.64621430258289, 0.64640443768154, 0.64659447673066,
	0.64678441985543, 0.64697426718077, 0.64716401883133, 0.6473536749315,
	0.64754323560542, 0.64773270097697, 0.64792207116974, 0.6481113463071,
	0.64830052651215, 0.64848961190773, 0.64867860261642, 0.64886749876055,
	0.64905630046219, 0.64924500784317, 0.64943362102505, 0.64962214012916,
	0.64981056527654, 0.64999889658802, 0.65018713418415, 0.65037527818525,
	0.65056332871138, 0.65075128588235, 0.65093914981774, 0.65112692063686,
	0.65131459845879, 0.65150218340236, 0.65168967558616, 0.65187707512851,
	0.65206438214753, 0.65225159676106, 0.65243871908672, 0.65262574924189,
	0.65281268734368, 0.652999533509, 0.65318628785449, 0.65337295049657,
	0.65355952155141, 0.65374600113496, 0.65393238936291, 0.65411868635073,
	0.65430489221366, 0.65449100706668, 0.65467703102456, 0.65486296420184,
	0.6550488067128, 0.65523455867151, 0.65542022019182, 0.65560579138731,
	0.65579127237136, 0.65597666325712, 0.6561619641575, 0.65634717518519,
	0.65653229645265, 0.65671732807211, 0.65690227015558, 0.65708712281484,
	0.65727188616145, 0.65745656030674, 0.65764114536181, 0.65782564143755,
	0.65801004864464, 0.65819436709349, 0.65837859689435, 0.6585627381572,
	0.65874679099183, 0.6589307555078, 0.65911463181445, 0.65929842002089,
	0.65948212023605, 0.6596657325686, 0.65984925712702, 0.66003269401957,
	0.66021604335429, 0.660399305239, 0.66058247978131, 0.66076556708864,
	0.66094856726816, 0.66113148042685, 0.66131430667147, 0.66149704610858,
	0.66167969884451, 0.6618622649854, 0.66204474463717, 0.66222713790553,
	0.66240944489599, 0.66259166571384, 0.66277380046418, 0.66295584925189,
	0.66313781218164, 0.66331968935791, 0.66350148088497, 0.66368318686688,
	0.66386480740751, 0.66404634261049, 0.6642277925793, 0.66440915741718,
	0.66459043722718, 0.66477163211215, 0.66495274217473, 0.66513376751739,
	0.66531470824236, 0.6654955644517, 0.66567633624726, 0.66585702373069,
	0.66603762700345, 0.6662181461668, 0.6663985813218, 0.66657893256933,
	0.66675920001005, 0.66693938374444, 0.66711948387279, 0.66729950049519,
	0.66747943371154, 0.66765928362153, 0.66783905032469, 0.66801873392034,
	0.6681983345076, 0.66837785218541, 0.66855728705253, 0.66873663920752,
	0.66891590874874, 0.66909509577437, 0.66927420038242, 0.66945322267069,
	0.6696321627368, 0.66981102067818, 0.66998979659209, 0.67016849057557,
	0.67034710272551, 0.6705256331386, 0.67070408191136, 0.67088244914009,
	0.67106073492095, 0.67123893934989, 0.67141706252268, 0.67159510453493,
	0.67177306548205, 0.67195094545927, 0.67212874456163, 0.67230646288402,
	0.67248410052113, 0.67266165756747, 0.67283913411737, 0.673016530265,
	0.67319384610433, 0.67337108172917, 0.67354823723315, 0.67372531270971,
	0.67390230825213, 0.67407922395351, 0.67425605990677, 0.67443281620467,
	0.67460949293979, 0.67478609020452, 0.67496260809109, 0.67513904669157,
	0.67531540609784, 0.67549168640162, 0.67566788769444, 0.67584401006768,
	0.67602005361255, 0.67619601842007, 0.67637190458111, 0.67654771218636,
	0.67672344132635, 0.67689909209143, 0.6770746645718, 0.67725015885747,
	0.67742557503831, 0.67760091320399, 0.67777617344405, 0.67795135584784,
	0.67812646050455, 0.67830148750322, 0.6784764369327, 0.6786513088817,
	0.67882610343874, 0.67900082069221, 0.67917546073032, 0.6793500236411,
	0.67952450951246, 0.67969891843211, 0.67987325048761, 0.68004750576638,
	0.68022168435565, 0.68039578634251, 0.68056981181388, 0.68074376085653,
	0.68091763355707, 0.68109143000193, 0.68126515027742, 0.68143879446967,
	0.68161236266464, 0.68178585494817, 0.68195927140592, 0.68213261212339,
	0.68230587718594, 0.68247906667876, 0.6826521806869, 0.68282521929524,
	0.68299818258852, 0.68317107065132, 0.68334388356808, 0.68351662142305,
	0.68368928430038, 0.68386187228402, 0.6840343854578, 0.68420682390539,
	0.68437918771031, 0.68455147695592, 0.68472369172543, 0.68489583210192,
	0.68506789816831, 0.68523989000736, 0.6854118077017, 0.68558365133379,
	0.68575542098597, 0.6859271167404, 0.68609873867912, 0.68627028688402,
	0.68644176143682, 0.68661316241913, 0.68678448991238, 0.68695574399787,
	0.68712692475676, 0.68729803227007, 0.68746906661865, 0.68764002788323,
	0.68781091614438, 0.68798173148255, 0.68815247397802, 0.68832314371095,
	0.68849374076134, 0.68866426520907, 0.68883471713384, 0.68900509661526,
	0.68917540373276, 0.68934563856564, 0.68951580119307, 0.68968589169407,
	0.68985591014752, 0.69002585663216, 0.69019573122662, 0.69036553400934,
	0.69053526505866, 0.69070492445277, 0.69087451226973, 0.69104402858745,
	0.69121347348372, 0.69138284703617, 0.69155214932232, 0.69172138041953,
	0.69189054040506, 0.69205962935598, 0.69222864734929, 0.6923975944618,
	0.69256647077021, 0.6927352763511, 0.69290401128089, 0.69307267563589,
	0.69324126949224, 0.693409792926, 0.69357824601306, 0.69374662882919,
	0.69391494145003, 0.69408318395109, 0.69425135640773, 0.69441945889521,
	0.69458749148864, 0.694755454263, 0.69492334729315, 0.69509117065382,
	0.6952589244196, 0.69542660866496, 0.69559422346424, 0.69576176889166,
	0.69592924502129, 0.69609665192708, 0.69626398968288, 0.69643125836238,
	0.69659845803916, 0.69676558878667, 0.69693265067823, 0.69709964378703,
	0.69726656818615, 0.69743342394854, 0.69760021114702, 0.69776692985428,
	0.69793358014291, 0.69810016208534, 0.69826667575391, 0.69843312122081,
	0.69859949855814, 0.69876580783783, 0.69893204913173, 0.69909822251155,
	0.69926432804888, 0.69943036581519, 0.69959633588183, 0.69976223832001,
	0.69992807320085, 0.70009384059533, 0.70025954057432, 0.70042517320856,
	0.70059073856867, 0.70075623672516, 0.70092166774842, 0.70108703170871,
	0.70125232867619, 0.70141755872088, 0.7015827219127, 0.70174781832144,
	0.70191284801678, 0.70207781106829, 0.7022427075454, 0.70240753751744,
	0.70257230105362, 0.70273699822304, 0.70290162909468, 0.7030661937374,
	0.70323069221994, 0.70339512461095, 0.70355949097893, 0.7037237913923,
	0.70388802591934, 0.70405219462822, 0.70421629758702, 0.70438033486367,
	0.70454430652602, 0.70470821264178, 0.70487205327857, 0.70503582850388,
	0.7051995383851, 0.7053631829895, 0.70552676238424, 0.70569027663637,
	0.70585372581284, 0.70601710998047, 0.70618042920598, 0.70634368355597,
	0.70650687309695, 0.70666999789529, 0.70683305801729, 0.7069960535291,
	0.70715898449678, 0.7073218509863, 0.70748465306348, 0.70764739079407,
	0.70781006424368, 0.70797267347784, 0.70813521856195, 0.70829769956132,
	0.70846011654114, 0.7086224695665, 0.70878475870239, 0.70894698401368,
	0.70910914556513, 0.70927124342142, 0.70943327764709, 0.70959524830661,
	0.70975715546431, 0.70991899918444, 0.71008077953115, 0.71024249656845,
	0.71040415036028, 0.71056574097046, 0.71072726846272, 0.71088873290066,
	0.71105013434781, 0.71121147286758, 0.71137274852327, 0.71153396137808,
	0.71169511149513, 0.71185619893741, 0.71201722376781, 0.71217818604914,
	0.7123390858441, 0.71249992321527, 0.71266069822514, 0.71282141093612,
	0.71298206141048, 0.71314264971043, 0.71330317589806, 0.71346364003534,
	0.71362404218419, 0.71378438240638, 0.71394466076361, 0.71410487731747,
	0.71426503212946, 0.71442512526098, 0.71458515677331, 0.71474512672767,
	0.71490503518515, 0.71506488220675, 0.71522466785339, 0.71538439218587,
	0.7155440552649, 0.71570365715109, 0.71586319790498, 0.71602267758697,
	0.71618209625739, 0.71634145397648, 0.71650075080436, 0.71665998680107,
	0.71681916202656, 0.71697827654067, 0.71713733040316, 0.71729632367369,
	0.71745525641181, 0.71761412867699, 0.71777294052862, 0.71793169202596,
	0.71809038322821, 0.71824901419446, 0.71840758498371, 0.71856609565487,
	0.71872454626675, 0.71888293687807, 0.71904126754746, 0.71919953833345,
	0.7193577492945, 0.71951590048895, 0.71967399197506, 0.719832023811,
	0.71998999605485, 0.7201479087646, 0.72030576199814, 0.72046355581327,
	0.72062129026771, 0.72077896541909, 0.72093658132493, 0.72109413804269,
	0.72125163562971, 0.72140907414326, 0.72156645364051, 0.72172377417855,
	0.72188103581438, 0.7220382386049, 0.72219538260694, 0.72235246787722,
	0.72250949447239, 0.722666462449, 0.72282337186352, 0.72298022277233,
	0.72313701523172, 0.72329374929789, 0.72345042502697, 0.72360704247497,
	0.72376360169785, 0.72392010275146, 0.72407654569158, 0.72423293057388,
	0.72438925745397, 0.72454552638735, 0.72470173742946, 0.72485789063564,
	0.72501398606114, 0.72517002376114, 0.72532600379071, 0.72548192620487,
	0.72563779105853, 0.72579359840652, 0.72594934830359, 0.7261050408044,
	0.72626067596354, 0.7264162538355, 0.7265717744747, 0.72672723793547,
	0.72688264427204, 0.7270379935386, 0.72719328578921, 0.72734852107788,
	0.72750369945853, 0.72765882098498, 0.727813885711, 0.72796889369024,
	0.7281238449763, 0.72827873962269, 0.72843357768282, 0.72858835921005,
	0.72874308425764, 0.72889775287877, 0.72905236512654, 0.72920692105397,
	0.72936142071401, 0.72951586415951, 0.72967025144325, 0.72982458261793,
	0.72997885773619, 0.73013307685054, 0.73028724001347, 0.73044134727734,
	0.73059539869447, 0.73074939431708, 0.73090333419731, 0.73105721838724,
	0.73121104693885, 0.73136481990405, 0.73151853733467, 0.73167219928247,
	0.73182580579913, 0.73197935693625, 0.73213285274535, 0.73228629327787,
	0.73243967858518, 0.73259300871857, 0.73274628372926, 0.73289950366838,
	0.733052668587, 0.7332057785361, 0.73335883356658, 0.73351183372928,
	0.73366477907497, 0.73381766965431, 0.73397050551791, 0.7341232867163,
	0.73427601329995, 0.73442868531922, 0.73458130282443, 0.7347338658658,
	0.73488637449349, 0.73503882875758, 0.73519122870807, 0.73534357439491,
	0.73549586586794, 0.73564810317696, 0.73580028637167, 0.73595241550172,
	0.73610449061666, 0.73625651176599, 0.73640847899912, 0.7365603923654,
	0.73671225191411, 0.73686405769444, 0.73701580975552, 0.7371675081464,
	0.73731915291607, 0.73747074411343, 0.73762228178734, 0.73777376598654,
	0.73792519675974, 0.73807657415556, 0.73822789822256, 0.73837916900921,
	0.73853038656393, 0.73868155093505, 0.73883266217084, 0.73898372031951,
	0.73913472542917, 0.73928567754789, 0.73943657672364, 0.73958742300436,
	0.73973821643789, 0.739888957072, 0.7400396449544, 0.74019028013272,
	0.74034086265455, 0.74049139256737, 0.74064186991862, 0.74079229475567,
	0.74094266712579, 0.74109298707622, 0.74124325465411, 0.74139346990655,
	0.74154363288056, 0.74169374362309, 0.74184380218102, 0.74199380860117,
	0.74214376293029, 0.74229366521505, 0.74244351550207, 0.74259331383789,
	0.74274306026899, 0.74289275484178, 0.74304239760261, 0.74319198859775,
	0.74334152787341, 0.74349101547575, 0.74364045145083, 0.74378983584466,
	0.7439391687032, 0.74408845007233, 0.74423767999785, 0.74438685852551,
	0.74453598570101, 0.74468506156994, 0.74483408617788, 0.74498305957029,
	0.74513198179261, 0.74528085289019, 0.74542967290832, 0.74557844189224,
	0.74572715988709, 0.74587582693798, 0.74602444308995, 0.74617300838797,
	0.74632152287693, 0.74646998660168, 0.746618399607, 0.74676676193761,
	0.74691507363815, 0.74706333475321, 0.74721154532731, 0.74735970540492,
	0.74750781503044, 0.74765587424819, 0.74780388310246, 0.74795184163745,
	0.7480997498973, 0.7482476079261, 0.74839541576788, 0.7485431734666,
	0.74869088106614, 0.74883853861035, 0.74898614614301, 0.74913370370781,
	0.74928121134843, 0.74942866910843, 0.74957607703136, 0.74972343516068,
	0.7498707435398, 0.75001800221205, 0.75016521122073, 0.75031237060905,
	0.75045948042018, 0.75060654069723, 0.75075355148322, 0.75090051282115,
	0.75104742475394, 0.75119428732444, 0.75134110057545, 0.75148786454973,
	0.75163457928994, 0.75178124483872, 0.75192786123862, 0.75207442853215,
	0.75222094676175, 0.75236741596981, 0.75251383619865, 0.75266020749053,
	0.75280652988768, 0.75295280343223, 0.75309902816628, 0.75324520413185,
	0.75339133137094, 0.75353740992544, 0.75368343983721, 0.75382942114807,
	0.75397535389974, 0.75412123813391, 0.75426707389222, 0.75441286121622,
	0.75455860014743, 0.7547042907273, 0.75484993299723, 0.75499552699856,
	0.75514107277258, 0.7552865703605, 0.75543201980349, 0.75557742114268,
	0.7557227744191, 0.75586807967378, 0.75601333694763, 0.75615854628156,
	0.75630370771639, 0.7564488212929, 0.7565938870518, 0.75673890503376,
	0.75688387527939, 0.75702879782923, 0.75717367272378, 0.75731850000348,
	0.75746327970872, 0.75760801187982, 0.75775269655706, 0.75789733378067,
	0.75804192359079, 0.75818646602755, 0.758330961131, 0.75847540894114,
	0.75861980949791, 0.75876416284122, 0.75890846901088, 0.75905272804669,
	0.75919693998839, 0.75934110487563, 0.75948522274804, 0.7596292936452,
	0.75977331760661, 0.75991729467174, 0.76006122487999, 0.76020510827071,
	0.76034894488321, 0.76049273475673, 0.76063647793046, 0.76078017444356,
	0.76092382433509, 0.76106742764411, 0.76121098440959, 0.76135449467047,
	0.76149795846561, 0.76164137583385, 0.76178474681396, 0.76192807144465,
	0.7620713497646, 0.76221458181243, 0.76235776762669, 0.76250090724591,
	0.76264400070854, 0.76278704805298, 0.76293004931761, 0.76307300454073,
	0.76321591376058, 0.76335877701539, 0.76350159434328, 0.76364436578238,
	0.76378709137073, 0.76392977114633, 0.76407240514714, 0.76421499341103,
	0.76435753597588, 0.76450003287947, 0.76464248415955, 0.76478488985382,
	0.76492724999992, 0.76506956463545, 0.76521183379796, 0.76535405752495,
	0.76549623585385, 0.76563836882207, 0.76578045646696, 0.76592249882581,
	0.76606449593587, 0.76620644783434, 0.76634835455837, 0.76649021614505,
	0.76663203263145, 0.76677380405456, 0.76691553045134, 0.76705721185869,
	0.76719884831346, 0.76734043985247, 0.76748198651247, 0.76762348833017,
	0.76776494534223, 0.76790635758527, 0.76804772509585, 0.76818904791049,
	0.76833032606566, 0.76847155959778, 0.76861274854321, 0.7687538929383,
	0.7688949928193, 0.76903604822246, 0.76917705918396, 0.76931802573992,
	0.76945894792645, 0.76959982577957, 0.76974065933529, 0.76988144862954,
	0.77002219369823, 0.77016289457721, 0.77030355130228, 0.77044416390921,
	0.77058473243369, 0.77072525691141, 0.77086573737798, 0.77100617386896,
	0.77114656641989, 0.77128691506625, 0.77142721984346, 0.77156748078693,
	0.77170769793197, 0.77184787131391, 0.77198800096797, 0.77212808692937,
	0.77226812923326, 0.77240812791476, 0.77254808300894, 0.77268799455081,
	0.77282786257535, 0.77296768711749, 0.77310746821212, 0.77324720589408,
	0.77338690019817, 0.77352655115912, 0.77366615881166, 0.77380572319044,
	0.77394524433008, 0.77408472226514, 0.77422415703017, 0.77436354865963,
	0.77450289718796, 0.77464220264957, 0.7747814650788, 0.77492068450995,
	0.77505986097729, 0.77519899451503, 0.77533808515736, 0.77547713293838,
	0.77561613789221, 0.77575510005286, 0.77589401945435, 0.77603289613063,
	0.77617173011561, 0.77631052144316, 0.7764492701471, 0.77658797626121,
	0.77672663981923, 0.77686526085487, 0.77700383940176, 0.77714237549353,
	0.77728086916373, 0.77741932044589, 0.77755772937349, 0.77769609597997,
	0.77783442029872, 0.77797270236311, 0.77811094220643, 0.77824913986196,
	0.77838729536293, 0.77852540874251, 0.77866348003386, 0.77880150927006,
	0.77893949648419, 0.77907744170925, 0.77921534497822, 0.77935320632403,
	0.77949102577958, 0.77962880337771, 0.77976653915123, 0.7799042331329,
	0.78004188535546, 0.78017949585158, 0.78031706465392, 0.78045459179506,
	0.78059207730757, 0.78072952122397, 0.78086692357674, 0.78100428439831,
	0.78114160372108, 0.78127888157741, 0.78141611799961, 0.78155331301996,
	0.7816904666707, 0.781827578984, 0.78196464999204, 0.78210167972691,
	0.7822386682207, 0.78237561550543, 0.7825125216131, 0.78264938657567,
	0.78278621042503, 0.78292299319308, 0.78305973491163, 0.78319643561248,
	0.78333309532739, 0.78346971408806, 0.78360629192618, 0.78374282887337,
	0.78387932496124, 0.78401578022133, 0.78415219468517, 0.78428856838423,
	0.78442490134995, 0.78456119361372, 0.78469744520692, 0.78483365616085,
	0.78496982650681, 0.78510595627603, 0.78524204549973, 0.78537809420906,
	0.78551410243515, 0.7856500702091, 0.78578599756194, 0.7859218845247,
	0.78605773112835, 0.78619353740383, 0.78632930338202, 0.78646502909379,
	0.78660071456996, 0.78673635984132, 0.7868719649386, 0.78700752989252,
	0.78714305473374, 0.78727853949289, 0.78741398420058, 0.78754938888734,
	0.78768475358371, 0.78782007832017, 0.78795536312715, 0.78809060803506,
	0.78822581307428, 0.78836097827512, 0.7884961036679, 0.78863118928286,
	0.78876623515022, 0.78890124130017, 0.78903620776286, 0.78917113456839,
	0.78930602174683, 0.78944086932823, 0.78957567734257, 0.78971044581982,
	0.78984517478992, 0.78997986428274, 0.79011451432814, 0.79024912495593,
	0.79038369619591, 0.7905182280778, 0.79065272063132, 0.79078717388613,
	0.79092158787189, 0.79105596261817, 0.79119029815455, 0.79132459451056,
	0.79145885171568, 0.79159306979938, 0.79172724879106, 0.79186138872012,
	0.79199548961591, 0.79212955150774, 0.79226357442488, 0.79239755839658,
	0.79253150345204, 0.79266540962044, 0.79279927693092, 0.79293310541257,
	0.79306689509446, 0.79320064600562, 0.79333435817505, 0.79346803163171,
	0.79360166640453, 0.79373526252239, 0.79386882001416, 0.79400233890865,
	0.79413581923465, 0.79426926102091, 0.79440266429616, 0.79453602908908,
	0.79466935542831, 0.79480264334247, 0.79493589286014, 0.79506910400987,
	0.79520227682016, 0.7953354113195, 0.79546850753634, 0.79560156549907,
	0.79573458523607, 0.79586756677569, 0.79600051014624, 0.79613341537598,
	0.79626628249317, 0.79639911152599, 0.79653190250264, 0.79666465545124,
	0.79679737039991, 0.79693004737671, 0.79706268640969, 0.79719528752684,
	0.79732785075615, 0.79746037612554, 0.79759286366294, 0.7977253133962,
	0.79785772535317, 0.79799009956165, 0.79812243604942, 0.79825473484422,
	0.79838699597376, 0.79851921946571, 0.7986514053477, 0.79878355364736,
	0.79891566439226, 0.79904773760994, 0.79917977332792, 0.79931177157366,
	0.79944373237462, 0.79957565575822, 0.79970754175183, 0.79983939038281,
	0.79997120167846, 0.80010297566608, 0.80023471237292, 0.80036641182619,
	0.8004980740531, 0.80062969908078, 0.80076128693638, 0.80089283764698,
	0.80102435123964, 0.80115582774139, 0.80128726717923, 0.80141866958013,
	0.80155003497101, 0.80168136337879, 0.80181265483032, 0.80194390935247,
	0.80207512697202, 0.80220630771576, 0.80233745161043, 0.80246855868274,
	0.80259962895939, 0.80273066246702, 0.80286165923225, 0.80299261928168,
	0.80312354264185, 0.8032544293393, 0.80338527940053, 0.803516092852,
	0.80364686972014, 0.80377761003137, 0.80390831381204, 0.80403898108852,
	0.8041696118871, 0.80430020623408, 0.80443076415569, 0.80456128567817,
	0.8046917708277, 0.80482221963044, 0.80495263211253, 0.80508300830006,
	0.80521334821909, 0.80534365189568, 0.80547391935582, 0.8056041506255,
	0.80573434573067, 0.80586450469724, 0.8059946275511, 0.80612471431811,
	0.80625476502411, 0.80638477969488, 0.8065147583562, 0.80664470103381,
	0.80677460775342, 0.8069044785407, 0.80703431342132, 0.80716411242088,
	0.80729387556499, 0.80742360287921, 0.80755329438906, 0.80768295012005,
	0.80781257009766, 0.80794215434733, 0.80807170289448, 0.80820121576449,
	0.80833069298273, 0.80846013457451, 0.80858954056515, 0.80871891097991,
	0.80884824584402, 0.80897754518272, 0.80910680902117, 0.80923603738454,
	0.80936523029795, 0.80949438778651, 0.80962350987527, 0.80975259658928,
	0.80988164795355, 0.81001066399306, 0.81013964473278, 0.81026859019762,
	0.81039750041249, 0.81052637540226, 0.81065521519176, 0.81078401980582,
	0.8109127892692, 0.81104152360669, 0.81117022284299, 0.81129888700282,
	0.81142751611084, 0.81155611019171, 0.81168466927003, 0.8118131933704,
	0.81194168251738, 0.8120701367355, 0.81219855604927, 0.81232694048317,
	0.81245529006164, 0.81258360480911, 0.81271188474998, 0.8128401299086,
	0.81296834030933, 0.81309651597647, 0.81322465693431, 0.81335276320711,
	0.8134808348191, 0.81360887179448, 0.81373687415743, 0.81386484193209,
	0.81399277514259, 0.81412067381303, 0.81424853796746, 0.81437636762993,
	0.81450416282446, 0.81463192357502, 0.81475964990559, 0.81488734184009,
	0.81501499940242, 0.81514262261648, 0.8152702115061, 0.81539776609512,
	0.81552528640733, 0.81565277246651, 0.8157802242964, 0.81590764192072,
	0.81603502536317, 0.81616237464741, 0.81628968979708, 0.81641697083579,
	0.81654421778713, 0.81667143067467, 0.81679860952193, 0.81692575435243,
	0.81705286518965, 0.81717994205703, 0.81730698497803, 0.81743399397603,
	0.81756096907441, 0.81768791029653, 0.81781481766572, 0.81794169120527,
	0.81806853093846, 0.81819533688853, 0.81832210907872, 0.81844884753221,
	0.81857555227218, 0.81870222332177, 0.8188288607041, 0.81895546444227,
	0.81908203455935, 0.81920857107838, 0.81933507402238, 0.81946154341434,
	0.81958797927722, 0.81971438163398, 0.81984075050752, 0.81996708592074,
	0.8200933878965, 0.82021965645765, 0.82034589162699, 0.82047209342733,
	0.82059826188142, 0.82072439701201, 0.82085049884181, 0.82097656739352,
	0.82110260268979, 0.82122860475327, 0.82135457360657, 0.82148050927229,
	0.82160641177299, 0.82173228113122, 0.82185811736948, 0.82198392051028,
	0.82210969057608, 0.82223542758932, 0.82236113157242, 0.82248680254777,
	0.82261244053775, 0.8227380455647, 0.82286361765094, 0.82298915681877,
	0.82311466309045, 0.82324013648824, 0.82336557703436, 0.82349098475101,
	0.82361635966036, 0.82374170178456, 0.82386701114575, 0.82399228776601,
	0.82411753166744, 0.82424274287209, 0.82436792140198, 0.82449306727913,
	0.82461818052551, 0.82474326116309, 0.8248683092138, 0.82499332469955,
	0.82511830764223, 0.8252432580637, 0.8253681759858, 0.82549306143036,
	0.82561791441916, 0.82574273497397, 0.82586752311654, 0.82599227886859,
	0.82611700225181, 0.8262416932879, 0.82636635199848, 0.82649097840521,
	0.82661557252967, 0.82674013439346, 0.82686466401813, 0.82698916142521,
	0.82711362663623, 0.82723805967266, 0.82736246055598, 0.82748682930762,
	0.82761116594902, 0.82773547050155, 0.82785974298661, 0.82798398342553,
	0.82810819183965, 0.82823236825028, 0.82835651267868, 0.82848062514614,
	0.82860470567387, 0.8287287542831, 0.82885277099501, 0.82897675583078,
	0.82910070881155, 0.82922462995845, 0.82934851929257, 0.82947237683499,
	0.82959620260677, 0.82971999662894, 0.82984375892252, 0.82996748950849,
	0.83009118840781, 0.83021485564144, 0.83033849123029, 0.83046209519525,
	0.83058566755722, 0.83070920833704, 0.83083271755554, 0.83095619523354,
	0.83107964139182, 0.83120305605115, 0.83132643923228, 0.83144979095591,
	0.83157311124277, 0.83169640011351, 0.83181965758881, 0.83194288368929,
	0.83206607843556, 0.83218924184822, 0.83231237394784, 0.83243547475495,
	0.8325585442901, 0.83268158257377, 0.83280458962646, 0.83292756546861,
	0.83305051012068, 0.83317342360308, 0.8332963059362, 0.83341915714041,
	0.83354197723608, 0.83366476624353, 0.83378752418306, 0.83391025107498,
	0.83403294693954, 0.83415561179699, 0.83427824566755, 0.83440084857143,
	0.83452342052882, 0.83464596155986, 0.8347684716847, 0.83489095092346,
	0.83501339929623, 0.83513581682309, 0.8352582035241, 0.83538055941929,
	0.83550288452867, 0.83562517887224, 0.83574744246996, 0.83586967534178,
	0.83599187750764, 0.83611404898745, 0.83623618980108, 0.83635829996842,
	0.83648037950929, 0.83660242844354, 0.83672444679096, 0.83684643457133,
	0.83696839180443, 0.83709031850999, 0.83721221470774, 0.83733408041738,
	0.83745591565859, 0.83757772045102, 0.83769949481433, 0.83782123876812,
	0.83794295233201, 0.83806463552557, 0.83818628836835, 0.8383079108799,
	0.83842950307973, 0.83855106498734, 0.83867259662221, 0.8387940980038,
	0.83891556915155, 0.83903701008487, 0.83915842082315, 0.83927980138578,
	0.83940115179212, 0.83952247206149, 0.83964376221322, 0.8397650222666,
	0.83988625224092, 0.84000745215542, 0.84012862202935, 0.84024976188192,
	0.84037087173233, 0.84049195159976, 0.84061300150337, 0.84073402146229,
	0.84085501149565, 0.84097597162254, 0.84109690186204, 0.84121780223321,
	0.84133867275509, 0.8414595134467, 0.84158032432705, 0.84170110541511,
	0.84182185672985, 0.84194257829021, 0.84206327011511, 0.84218393222346,
	0.84230456463414, 0.84242516736601, 0.84254574043793, 0.84266628386871,
	0.84278679767717, 0.84290728188209, 0.84302773650224, 0.84314816155637,
	0.84326855706321, 0.84338892304148, 0.84350925950986, 0.84362956648702,
	0.84374984399163, 0.84387009204232, 0.8439903106577, 0.84411049985637,
	0.8442306596569, 0.84435079007786, 0.84447089113779, 0.84459096285521,
	0.84471100524862, 0.8448310183365, 0.84495100213733, 0.84507095666954,
	0.84519088195156, 0.84531077800181, 0.84543064483867, 0.84555048248051,
	0.84567029094569, 0.84579007025254, 0.84590982041937, 0.8460295414645,
	0.84614923340618, 0.84626889626268, 0.84638853005225, 0.84650813479311,
	0.84662771050346, 0.84674725720149, 0.84686677490536, 0.84698626363324,
	0.84710572340323, 0.84722515423347, 0.84734455614205, 0.84746392914704,
	0.84758327326649, 0.84770258851846, 0.84782187492096, 0.847941132492,
	0.84806036124956, 0.84817956121161, 0.8482987323961, 0.84841787482096,
	0.84853698850411, 0.84865607346343, 0.84877512971681, 0.84889415728211,
	0.84901315617717, 0.84913212641982, 0.84925106802785, 0.84936998101906,
	0.84948886541122, 0.84960772122208, 0.84972654846937, 0.84984534717081,
	0.84996411734411, 0.85008285900694, 0.85020157217697, 0.85032025687184,
	0.85043891310918, 0.8505575409066, 0.85067614028171, 0.85079471125206,
	0.85091325383523, 0.85103176804875, 0.85115025391014, 0.85126871143692,
	0.85138714064657, 0.85150554155656, 0.85162391418434, 0.85174225854736,
	0.85186057466303, 0.85197886254875, 0.85209712222191, 0.85221535369987,
	0.85233355699999, 0.8524517321396, 0.85256987913602, 0.85268799800653,
	0.85280608876843, 0.85292415143898, 0.85304218603542, 0.85316019257499,
	0.8532781710749, 0.85339612155234, 0.85351404402449, 0.85363193850852,
	0.85374980502157, 0.85386764358077, 0.85398545420323, 0.85410323690605,
	0.8542209917063, 0.85433871862104, 0.85445641766732, 0.85457408886216,
	0.85469173222257, 0.85480934776555, 0.85492693550808, 0.85504449546711,
	0.85516202765959, 0.85527953210245, 0.85539700881259, 0.8555144578069,
	0.85563187910228, 0.85574927271557, 0.85586663866362, 0.85598397696325,
	0.85610128763129, 0.85621857068451, 0.85633582613971, 0.85645305401364,
	0.85657025432304, 0.85668742708465, 0.85680457231518, 0.85692169003132,
	0.85703878024975, 0.85715584298714, 0.85727287826013, 0.85738988608536,
	0.85750686647944, 0.85762381945896, 0.85774074504051, 0.85785764324066,
	0.85797451407595, 0.85809135756292, 0.85820817371809, 0.85832496255796,
	0.858441724099, 0.85855845835771, 0.85867516535051, 0.85879184509386,
	0.85890849760418, 0.85902512289786, 0.85914172099131, 0.85925829190089,
	0.85937483564296, 0.85949135223386, 0.85960784168992, 0.85972430402746,
	0.85984073926275, 0.85995714741209, 0.86007352849173, 0.86018988251793,
	0.86030620950691, 0.86042250947489, 0.86053878243806, 0.86065502841262,
	0.86077124741474, 0.86088743946056, 0.86100360456622, 0.86111974274785,
	0.86123585402154, 0.8613519384034, 0.8614679959095, 0.8615840265559,
	0.86170003035863, 0.86181600733374, 0.86193195749723, 0.8620478808651,
	0.86216377745334, 0.86227964727791, 0.86239549035477, 0.86251130669984,
	0.86262709632906, 0.86274285925832, 0.86285859550353, 0.86297430508055,
	0.86308998800524, 0.86320564429345, 0.86332127396102, 0.86343687702374,
	0.86355245349743, 0.86366800339787, 0.86378352674083, 0.86389902354206,
	0.8640144938173, 0.86412993758228, 0.8642453548527, 0.86436074564426,
	0.86447610997264, 0.8645914478535, 0.86470675930249, 0.86482204433525,
	0.8649373029674, 0.86505253521453, 0.86516774109224, 0.86528292061611,
	0.86539807380169, 0.86551320066453, 0.86562830122016, 0.86574337548409,
	0.86585842347183, 0.86597344519887, 0.86608844068066, 0.86620340993268,
	0.86631835297036, 0.86643326980913, 0.8665481604644, 0.86666302495157,
	0.86677786328602, 0.86689267548313, 0.86700746155825, 0.86712222152671,
	0.86723695540384, 0.86735166320496, 0.86746634494536, 0.86758100064032,
	0.86769563030511, 0.86781023395498, 0.86792481160517, 0.8680393632709,
	0.86815388896739, 0.86826838870983, 0.8683828625134, 0.86849731039326,
	0.86861173236458, 0.86872612844249, 0.86884049864211, 0.86895484297855,
	0.86906916146691, 0.86918345412226, 0.86929772095969, 0.86941196199423,
	0.86952617724093, 0.86964036671481, 0.86975453043089, 0.86986866840416,
	0.86998278064959, 0.87009686718217, 0.87021092801685, 0.87032496316855,
	0.87043897265222, 0.87055295648277, 0.87066691467508, 0.87078084724405,
	0.87089475420455, 0.87100863557143, 0.87112249135954, 0.8712363215837,
	0.87135012625873, 0.87146390539943, 0.87157765902059, 0.87169138713698,
	0.87180508976336, 0.87191876691448, 0.87203241860507, 0.87214604484984,
	0.87225964566351, 0.87237322106075, 0.87248677105626, 0.8726002956647,
	0.8727137949007, 0.87282726877892, 0.87294071731397, 0.87305414052046,
	0.87316753841299, 0.87328091100614, 0.87339425831447, 0.87350758035255,
	0.87362087713492, 0.87373414867609, 0.8738473949906, 0.87396061609293,
	0.87407381199758, 0.87418698271901, 0.8743001282717, 0.87441324867008,
	0.8745263439286, 0.87463941406167, 0.8747524590837, 0.87486547900908,
	0.87497847385219, 0.87509144362741, 0.87520438834908, 0.87531730803156,
	0.87543020268915, 0.87554307233618, 0.87565591698696, 0.87576873665576,
	0.87588153135687, 0.87599430110454, 0.87610704591302, 0.87621976579655,
	0.87633246076936, 0.87644513084564, 0.8765577760396, 0.87667039636542,
	0.87678299183727, 0.87689556246931, 0.87700810827568, 0.87712062927052,
	0.87723312546793, 0.87734559688203, 0.87745804352691, 0.87757046541664,
	0.8776828625653, 0.87779523498694, 0.8779075826956, 0.8780199057053,
	0.87813220403006, 0.87824447768389, 0.87835672668077, 0.87846895103468,
	0.87858115075959, 0.87869332586943, 0.87880547637817, 0.87891760229971,
	0.87902970364797, 0.87914178043685, 0.87925383268024, 0.87936586039202,
	0.87947786358604, 0.87958984227616, 0.87970179647622, 0.87981372620003,
	0.87992563146141, 0.88003751227415, 0.88014936865206, 0.88026120060889,
	0.88037300815841, 0.88048479131438, 0.88059655009052, 0.88070828450056,
	0.88081999455822, 0.88093168027718, 0.88104334167115, 0.88115497875379,
	0.88126659153877, 0.88137818003973, 0.88148974427031, 0.88160128424414,
	0.88171279997484, 0.88182429147599, 0.88193575876119, 0.88204720184402,
	0.88215862073803, 0.88227001545679, 0.88238138601382, 0.88249273242266,
	0.88260405469681, 0.88271535284979, 0.88282662689509, 0.88293787684617,
	0.88304910271651, 0.88316030451955, 0.88327148226875, 0.88338263597754,
	0.88349376565931, 0.8836048713275, 0.88371595299548, 0.88382701067664,
	0.88393804438435, 0.88404905413196, 0.88416003993282, 0.88427100180027,
	0.88438193974762, 0.88449285378818, 0.88460374393526, 0.88471461020214,
	0.88482545260208, 0.88493627114836, 0.88504706585422, 0.88515783673289,
	0.88526858379762, 0.8853793070616, 0.88549000653805, 0.88560068224015,
	0.88571133418109, 0.88582196237402, 0.88593256683211, 0.88604314756849,
	0.88615370459631, 0.88626423792867, 0.88637474757869, 0.88648523355947,
	0.88659569588409, 0.88670613456562, 0.88681654961713, 0.88692694105166,
	0.88703730888225, 0.88714765312194, 0.88725797378373, 0.88736827088062,
	0.88747854442563, 0.88758879443171, 0.88769902091184, 0.88780922387898,
	0.88791940334608, 0.88802955932606, 0.88813969183186, 0.88824980087638,
	0.88835988647252, 0.88846994863318, 0.88857998737123, 0.88869000269953,
	0.88879999463094, 0.8889099631783, 0.88901990835445, 0.8891298301722,
	0.88923972864437, 0.88934960378375, 0.88945945560313, 0.88956928411528,
	0.88967908933297, 0.88978887126895, 0.88989862993595, 0.89000836534672,
	0.89011807751397, 0.89022776645041, 0.89033743216873, 0.89044707468162,
	0.89055669400175, 0.89066629014179, 0.89077586311439, 0.89088541293218,
	0.89099493960781, 0.89110444315387, 0.891213923583, 0.89132338090777,
	0.89143281514078, 0.89154222629459, 0.89165161438178, 0.89176097941489,
	0.89187032140647, 0.89197964036904, 0.89208893631513, 0.89219820925724,
	0.89230745920786, 0.8924166861795, 0.89252589018461, 0.89263507123567,
	0.89274422934513, 0.89285336452543, 0.892962476789, 0.89307156614826,
	0.89318063261563, 0.8932896762035, 0.89339869692426, 0.89350769479029,
	0.89361666981395, 0.8937256220076, 0.89383455138359, 0.89394345795425,
	0.89405234173191, 0.89416120272887, 0.89427004095743, 0.8943788564299,
	0.89448764915855, 0.89459641915565, 0.89470516643345, 0.89481389100422,
	0.89492259288018, 0.89503127207357, 0.8951399285966, 0.89524856246147,
	0.89535717368039, 0.89546576226553, 0.89557432822908, 0.89568287158319,
	0.89579139234002, 0.89589989051172, 0.8960083661104, 0.89611681914821,
	0.89622524963724, 0.89633365758959, 0.89644204301737, 0.89655040593265,
	0.89665874634749, 0.89676706427396, 0.8968753597241, 0.89698363270996,
	0.89709188324355, 0.89720011133691, 0.89730831700203, 0.89741650025092,
	0.89752466109555, 0.8976327995479, 0.89774091561995, 0.89784900932364,
	0.89795708067092, 0.89806512967373, 0.89817315634399, 0.89828116069361,
	0.8983891427345, 0.89849710247855, 0.89860503993764, 0.89871295512366,
	0.89882084804845, 0.89892871872388, 0.89903656716179, 0.89914439337401,
	0.89925219737236, 0.89935997916865, 0.89946773877469, 0.89957547620227,
	0.89968319146317, 0.89979088456917, 0.89989855553201, 0.90000620436347,
	0.90011383107527, 0.90022143567915, 0.90032901818684, 0.90043657861003,
	0.90054411696044, 0.90065163324976, 0.90075912748966, 0.90086659969183,
	0.90097404986791, 0.90108147802956, 0.90118888418844, 0.90129626835615,
	0.90140363054434, 0.9015109707646, 0.90161828902855, 0.90172558534777,
	0.90183285973385, 0.90194011219836, 0.90204734275286, 0.9021545514089,
	0.90226173817804, 0.90236890307179, 0.90247604610169, 0.90258316727925,
	0.90269026661597, 0.90279734412334, 0.90290439981286, 0.90301143369599,
	0.90311844578421, 0.90322543608895, 0.90333240462168, 0.90343935139383,
	0.90354627641682, 0.90365317970207, 0.90376006126098, 0.90386692110496,
	0.90397375924538, 0.90408057569364, 0.90418737046109, 0.9042941435591,
	0.90440089499901, 0.90450762479217, 0.9046143329499, 0.90472101948352,
	0.90482768440435, 0.90493432772368, 0.90504094945281, 0.90514754960302,
	0.90525412818559, 0.90536068521176, 0.90546722069281, 0.90557373463997,
	0.90568022706448, 0.90578669797757, 0.90589314739044, 0.90599957531431,
	0.90610598176037, 0.90621236673981, 0.90631873026381, 0.90642507234354,
	0.90653139299016, 0.90663769221481, 0.90674397002864, 0.90685022644278,
	0.90695646146835, 0.90706267511647, 0.90716886739823, 0.90727503832474,
	0.90738118790707, 0.90748731615631, 0.90759342308351, 0.90769950869974,
	0.90780557301604, 0.90791161604346, 0.90801763779301, 0.90812363827573,
	0.90822961750262, 0.90833557548469, 0.90844151223292, 0.90854742775831,
	0.90865332207181, 0.90875919518441, 0.90886504710706, 0.90897087785069,
	0.90907668742626, 0.90918247584469, 0.90928824311689, 0.90939398925379,
	0.90949971426627, 0.90960541816524, 0.90971110096157, 0.90981676266615,
	0.90992240328983, 0.91002802284347, 0.91013362133792, 0.91023919878401,
	0.91034475519258, 0.91045029057445, 0.91055580494043, 0.91066129830131,
	0.9107667706679, 0.91087222205097, 0.91097765246131, 0.91108306190968,
	0.91118845040683, 0.91129381796352, 0.91139916459048, 0.91150449029845,
	0.91160979509815, 0.91171507900029, 0.91182034201557, 0.91192558415469,
	0.91203080542834, 0.9121360058472, 0.91224118542193, 0.9123463441632,
	0.91245148208165, 0.91255659918793, 0.91266169549267, 0.91276677100651,
	0.91287182574004, 0.91297685970389, 0.91308187290864, 0.9131868653649,
	0.91329183708324, 0.91339678807424, 0.91350171834845, 0.91360662791644,
	0.91371151678875, 0.91381638497592, 0.91392123248848, 0.91402605933694,
	0.91413086553183, 0.91423565108365, 0.91434041600288, 0.91444516030002,
	0.91454988398555, 0.91465458706992, 0.91475926956361, 0.91486393147707,
	0.91496857282074, 0.91507319360505, 0.91517779384043, 0.9152823735373,
	0.91538693270607, 0.91549147135713, 0.91559598950089, 0.91570048714772,
	0.915804964308, 0.9159094209921, 0.91601385721037, 0.91611827297317,
	0.91622266829084, 0.9163270431737, 0.91643139763209, 0.91653573167633,
	0.91664004531671, 0.91674433856353, 0.9168486114271, 0.91695286391769,
	0.91705709604558, 0.91716130782103, 0.9172654992543, 0.91736967035564,
	0.91747382113529, 0.91757795160349, 0.91768206177045, 0.9177861516464,
	0.91789022124154, 0.91799427056606, 0.91809829963018, 0.91820230844405,
	0.91830629701787, 0.9184102653618, 0.91851421348599, 0.91861814140059,
	0.91872204911576, 0.91882593664161, 0.91892980398828, 0.91903365116588,
	0.91913747818453, 0.91924128505432, 0.91934507178534, 0.91944883838769,
	0.91955258487143, 0.91965631124664, 0.91976001752337, 0.91986370371169,
	0.91996736982162, 0.92007101586321, 0.92017464184649, 0.92027824778147,
	0.92038183367817, 0.92048539954659, 0.92058894539672, 0.92069247123856,
	0.92079597708208, 0.92089946293726, 0.92100292881405, 0.92110637472242,
	0.92120980067231, 0.92131320667366, 0.9214165927364, 0.92151995887045,
	0.92162330508573, 0.92172663139216, 0.92182993779962, 0.92193322431801,
	0.92203649095721, 0.9221397377271, 0.92224296463755, 0.92234617169842,
	0.92244935891955, 0.9225525263108, 0.922655673882, 0.92275880164297,
	0.92286190960355, 0.92296499777353, 0.92306806616273, 0.92317111478095,
	0.92327414363797, 0.92337715274357, 0.92348014210753, 0.92358311173962,
	0.92368606164958, 0.92378899184718, 0.92389190234216, 0.92399479314424,
	0.92409766426317, 0.92420051570864, 0.92430334749039, 0.9244061596181,
	0.92450895210149, 0.92461172495023, 0.924714478174, 0.92481721178249,
	0.92491992578535, 0.92502262019224, 0.92512529501282, 0.92522795025672,
	0.92533058593358, 0.92543320205302, 0.92553579862467, 0.92563837565814,
	0.92574093316303, 0.92584347114894, 0.92594598962546, 0.92604848860216,
	0.92615096808863, 0.92625342809442, 0.9263558686291, 0.92645828970222,
	0.92656069132331, 0.92666307350193, 0.92676543624759, 0.92686777956982,
	0.92697010347813, 0.92707240798202, 0.92717469309099, 0.92727695881454,
	0.92737920516215, 0.92748143214329, 0.92758363976743, 0.92768582804404,
	0.92778799698256, 0.92789014659245, 0.92799227688313, 0.92809438786405,
	0.92819647954462, 0.92829855193427, 0.92840060504239, 0.9285026388784,
	0.92860465345169, 0.92870664877163, 0.92880862484762, 0.92891058168903,
	0.92901251930521, 0.92911443770553, 0.92921633689934, 0.92931821689598,
	0.92942007770478, 0.92952191933507, 0.92962374179617, 0.92972554509741,
	0.92982732924807, 0.92992909425746, 0.93003084013488, 0.9301325668896,
	0.93023427453091, 0.93033596306807, 0.93043763251034, 0.93053928286699,
	0.93064091414726, 0.93074252636038, 0.9308441195156, 0.93094569362213,
	0.93104724868921, 0.93114878472604, 0.93125030174182, 0.93135179974575,
	0.93145327874703, 0.93155473875483, 0.93165617977833, 0.93175760182671,
	0.93185900490912, 0.93196038903471, 0.93206175421265, 0.93216310045205,
	0.93226442776207, 0.93236573615182, 0.93246702563043, 0.932568296207,
	0.93266954789064, 0.93277078069045, 0.93287199461552, 0.93297318967494,
	0.93307436587777, 0.9331755232331, 0.93327666174997, 0.93337778143746,
	0.9334788823046, 0.93357996436044, 0.93368102761401, 0.93378207207435,
	0.93388309775046, 0.93398410465137, 0.93408509278608, 0.93418606216359,
	0.93428701279289, 0.93438794468298, 0.93448885784282, 0.93458975228139,
	0.93469062800766, 0.93479148503057, 0.9348923233591, 0.93499314300217,
	0.93509394396873, 0.9351947262677, 0.93529548990801, 0.93539623489858,
	0.93549696124832, 0.93559766896612, 0.93569835806089, 0.93579902854151,
	0.93589968041687, 0.93600031369584, 0.9361009283873, 0.93620152450009,
	0.93630210204308, 0.93640266102512, 0.93650320145505, 0.9366037233417,
	0.9367042266939, 0.93680471152047, 0.93690517783022, 0.93700562563197,
	0.93710605493451, 0.93720646574664, 0.93730685807715, 0.93740723193481,
	0.93750758732841, 0.9376079242667, 0.93770824275844, 0.9378085428124,
	0.93790882443732, 0.93800908764194, 0.93810933243499, 0.9382095588252,
	0.93830976682128, 0.93840995643197, 0.93851012766595, 0.93861028053194,
	0.93871041503862, 0.93881053119468, 0.9389106290088, 0.93901070848966,
	0.93911076964593, 0.93921081248626, 0.9393108370193, 0.93941084325372,
	0.93951083119813, 0.9396108008612, 0.93971075225152, 0.93981068537774,
	0.93991060024847, 0.9400104968723, 0.94011037525785, 0.94021023541371,
	0.94031007734847, 0.9404099010707, 0.94050970658899, 0.9406094939119,
	0.940709263048, 0.94080901400584, 0.94090874679398, 0.94100846142094,
	0.94110815789528, 0.94120783622552, 0.94130749642018, 0.94140713848778,
	0.94150676243684, 0.94160636827585, 0.94170595601331, 0.94180552565772,
	0.94190507721756, 0.94200461070131, 0.94210412611743, 0.94220362347441,
	0.94230310278068, 0.94240256404472, 0.94250200727496, 0.94260143247985,
	0.94270083966781, 0.94280022884728, 0.94289960002668, 0.94299895321441,
	0.9430982884189, 0.94319760564854, 0.94329690491173, 0.94339618621685,
	0.9434954495723, 0.94359469498644, 0.94369392246764, 0.94379313202428,
	0.94389232366471, 0.94399149739727, 0.94409065323033, 0.94418979117221,
	0.94428891123124, 0.94438801341576, 0.94448709773409, 0.94458616419453,
	0.9446852128054, 0.94478424357499, 0.94488325651161, 0.94498225162354,
	0.94508122891906, 0.94518018840646, 0.945279130094, 0.94537805398994,
	0.94547696010255, 0.94557584844007, 0.94567471901076, 0.94577357182285,
	0.94587240688457, 0.94597122420415, 0.94607002378982, 0.94616880564979,
	0.94626756979226, 0.94636631622544, 0.94646504495753, 0.94656375599672,
	0.94666244935118, 0.94676112502911, 0.94685978303866, 0.94695842338802,
	0.94705704608533, 0.94715565113875, 0.94725423855643, 0.94735280834651,
	0.94745136051713, 0.94754989507642, 0.94764841203249, 0.94774691139347,
	0.94784539316747, 0.9479438573626, 0.94804230398695, 0.94814073304861,
	0.94823914455569, 0.94833753851624, 0.94843591493836, 0.94853427383011,
	0.94863261519955, 0.94873093905475, 0.94882924540375, 0.94892753425459,
	0.94902580561533, 0.94912405949399, 0.94922229589859, 0.94932051483717,
	0.94941871631774, 0.9495169003483, 0.94961506693687, 0.94971321609143,
	0.94981134781998, 0.94990946213052, 0.950007559031, 0.95010563852943,
	0.95020370063375, 0.95030174535193, 0.95039977269194, 0.95049778266171,
	0.9505957752692, 0.95069375052235, 0.95079170842908, 0.95088964899733,
	0.95098757223502, 0.95108547815006, 0.95118336675036, 0.95128123804383,
	0.95137909203837, 0.95147692874186, 0.9515747481622, 0.95167255030726,
	0.95177033518493, 0.95186810280306, 0.95196585316953, 0.95206358629219,
	0.95216130217889, 0.95225900083748, 0.9523566822758, 0.95245434650169,
	0.95255199352296, 0.95264962334746, 0.95274723598299, 0.95284483143737,
	0.95294240971839, 0.95303997083387, 0.9531375147916, 0.95323504159936,
	0.95333255126494, 0.95343004379612, 0.95352751920067, 0.95362497748635,
	0.95372241866092, 0.95381984273215, 0.95391724970777, 0.95401463959553,
	0.95411201240318, 0.95420936813844, 0.95430670680903, 0.95440402842269,
	0.95450133298712, 0.95459862051003, 0.95469589099913, 0.95479314446212,
	0.95489038090669, 0.95498760034053, 0.95508480277131, 0.95518198820672,
	0.95527915665442, 0.95537630812208, 0.95547344261736, 0.95557056014791,
	0.95566766072138, 0.95576474434541, 0.95586181102765, 0.95595886077571,
	0.95605589359723, 0.95615290949983, 0.95624990849112, 0.95634689057871,
	0.95644385577021, 0.95654080407321, 0.95663773549531, 0.95673465004409,
	0.95683154772713, 0.95692842855202, 0.95702529252632, 0.9571221396576,
	0.95721896995341, 0.95731578342131, 0.95741258006886, 0.95750935990359,
	0.95760612293304, 0.95770286916475, 0.95779959860623, 0.95789631126502,
	0.95799300714863, 0.95808968626457, 0.95818634862035, 0.95828299422346,
	0.9583796230814, 0.95847623520165, 0.95857283059172, 0.95866940925906,
	0.95876597121115, 0.95886251645547, 0.95895904499947, 0.95905555685061,
	0.95915205201634, 0.95924853050412, 0.95934499232136, 0.95944143747553,
	0.95953786597403, 0.95963427782431, 0.95973067303377, 0.95982705160984,
	0.95992341355991, 0.9600197588914, 0.96011608761171, 0.96021239972821,
	0.96030869524831, 0.96040497417938, 0.96050123652881, 0.96059748230395,
	0.96069371151219, 0.96078992416087, 0.96088612025736, 0.96098229980901,
	0.96107846282315, 0.96117460930714, 0.96127073926831, 0.96136685271398,
	0.96146294965147, 0.96155903008812, 0.96165509403123, 0.96175114148811,
	0.96184717246606, 0.96194318697238, 0.96203918501437, 0.96213516659931,
	0.96223113173449, 0.96232708042718, 0.96242301268465, 0.96251892851417,
	0.962614827923, 0.96271071091841, 0.96280657750763, 0.96290242769792,
	0.96299826149652, 0.96309407891066, 0.96318987994757, 0.96328566461449,
	0.96338143291862, 0.96347718486719, 0.9635729204674, 0.96366863972647,
	0.96376434265158, 0.96386002924994, 0.96395569952874, 0.96405135349515,
	0.96414699115636, 0.96424261251955, 0.96433821759188, 0.96443380638051,
	0.96452937889261, 0.96462493513533, 0.96472047511582, 0.96481599884123,
	0.96491150631868, 0.96500699755532, 0.96510247255827, 0.96519793133466,
	0.96529337389161, 0.96538880023623, 0.96548421037563, 0.96557960431691,
	0.96567498206717, 0.9657703436335, 0.965865689023, 0.96596101824274,
	0.96605633129981, 0.96615162820128, 0.96624690895422, 0.96634217356568,
	0.96643742204274, 0.96653265439244, 0.96662787062183, 0.96672307073795,
	0.96681825474785, 0.96691342265855, 0.9670085744771, 0.9671037102105,
	0.96719882986578, 0.96729393344996, 0.96738902097003, 0.96748409243302,
	0.96757914784591, 0.96767418721569, 0.96776921054937, 0.96786421785392,
	0.96795920913633, 0.96805418440356, 0.96814914366259, 0.96824408692038,
	0.96833901418389, 0.96843392546008, 0.9685288207559, 0.96862370007829,
	0.96871856343419, 0.96881341083054, 0.96890824227427, 0.96900305777231,
	0.96909785733157, 0.96919264095898, 0.96928740866143, 0.96938216044585,
	0.96947689631914, 0.96957161628818, 0.96966632035987, 0.9697610085411,
	0.96985568083875, 0.9699503372597, 0.97004497781082, 0.97013960249898,
	0.97023421133104, 0.97032880431387, 0.9704233814543, 0.9705179427592,
	0.97061248823541, 0.97070701788977, 0.97080153172911, 0.97089602976025,
	0.97099051199004, 0.97108497842528, 0.97117942907279, 0.97127386393939,
	0.97136828303187, 0.97146268635705, 0.97155707392171, 0.97165144573265,
	0.97174580179666, 0.97184014212052, 0.971934466711, 0.97202877557488,
	0.97212306871893, 0.97221734614991, 0.97231160787458, 0.9724058538997,
	0.97250008423201, 0.97259429887825, 0.97268849784518, 0.97278268113953,
	0.97287684876801, 0.97297100073737, 0.97306513705433, 0.97315925772559,
	0.97325336275788, 0.9733474521579, 0.97344152593235, 0.97353558408794,
	0.97362962663134, 0.97372365356927, 0.97381766490839, 0.97391166065539,
	0.97400564081695, 0.97409960539973, 0.9741935544104, 0.97428748785563,
	0.97438140574206, 0.97447530807636, 0.97456919486516, 0.97466306611512,
	0.97475692183287, 0.97485076202504, 0.97494458669827, 0.97503839585917,
	0.97513218951438, 0.9752259676705, 0.97531973033415, 0.97541347751193,
	0.97550720921044, 0.97560092543629, 0.97569462619606, 0.97578831149635,
	0.97588198134374, 0.9759756357448, 0.97606927470611, 0.97616289823425,
	0.97625650633577, 0.97635009901725, 0.97644367628523, 0.97653723814627,
	0.97663078460692, 0.97672431567372, 0.9768178313532, 0.97691133165191,
	0.97700481657638, 0.97709828613312, 0.97719174032867, 0.97728517916953,
	0.97737860266222, 0.97747201081325, 0.97756540362911, 0.97765878111632,
	0.97775214328136, 0.97784549013072, 0.97793882167089, 0.97803213790835,
	0.97812543884957, 0.97821872450103, 0.9783119948692, 0.97840524996054,
	0.97849848978151, 0.97859171433856, 0.97868492363814, 0.9787781176867,
	0.97887129649068, 0.97896446005652, 0.97905760839065, 0.9791507414995,
	0.97924385938948, 0.97933696206703, 0.97943004953856, 0.97952312181047,
	0.97961617888918, 0.97970922078108, 0.97980224749258, 0.97989525903006,
	0.97998825539992, 0.98008123660853, 0.98017420266229, 0.98026715356757,
	0.98036008933073, 0.98045300995815, 0.98054591545619, 0.9806388058312,
	0.98073168108955, 0.98082454123758, 0.98091738628164, 0.98101021622807,
	0.9811030310832, 0.98119583085337, 0.98128861554492, 0.98138138516415,
	0.9814741397174, 0.98156687921097, 0.98165960365119, 0.98175231304435,
	0.98184500739677, 0.98193768671473, 0.98203035100453, 0.98212300027247,
	0.98221563452484, 0.9823082537679, 0.98240085800794, 0.98249344725123,
	0.98258602150405, 0.98267858077265, 0.9827711250633, 0.98286365438225,
	0.98295616873576, 0.98304866813008, 0.98314115257144, 0.98323362206609,
	0.98332607662026, 0.98341851624018, 0.98351094093209, 0.9836033507022,
	0.98369574555673, 0.9837881255019, 0.98388049054392, 0.98397284068898,
	0.98406517594331, 0.98415749631309, 0.98424980180451, 0.98434209242377,
	0.98443436817705, 0.98452662907054, 0.9846188751104, 0.98471110630281,
	0.98480332265395, 0.98489552416998, 0.98498771085705, 0.98507988272132,
	0.98517203976895, 0.98526418200608, 0.98535630943885, 0.98544842207342,
	0.98554051991591, 0.98563260297245, 0.98572467124917, 0.9858167247522,
	0.98590876348766, 0.98600078746165, 0.9860927966803, 0.98618479114971,
	0.98627677087598, 0.98636873586521, 0.98646068612349, 0.98655262165692,
	0.98664454247159, 0.98673644857357, 0.98682833996895, 0.9869202166638,
	0.98701207866418, 0.98710392597618, 0.98719575860584, 0.98728757655923,
	0.98737937984241, 0.98747116846142, 0.98756294242231, 0.98765470173112,
	0.98774644639389, 0.98783817641665, 0.98792989180544, 0.98802159256628,
	0.9881132787052, 0.9882049502282, 0.98829660714132, 0.98838824945055,
	0.9884798771619, 0.98857149028138, 0.98866308881498, 0.98875467276871,
	0.98884624214854, 0.98893779696046, 0.98902933721047, 0.98912086290453,
	0.98921237404863, 0.98930387064872, 0.98939535271079, 0.98948682024078,
	0.98957827324467, 0.9896697117284, 0.98976113569793, 0.9898525451592,
	0.98994394011816, 0.99003532058075, 0.99012668655289, 0.99021803804053,
	0.99030937504959, 0.99040069758599, 0.99049200565565, 0.99058329926449,
	0.99067457841843, 0.99076584312336, 0.99085709338519, 0.99094832920983,
	0.99103955060316, 0.99113075757109, 0.99122195011949, 0.99131312825426,
	0.99140429198128, 0.99149544130641, 0.99158657623554, 0.99167769677454,
	0.99176880292926, 0.99185989470558, 0.99195097210934, 0.9920420351464,
	0.99213308382262, 0.99222411814383, 0.99231513811589, 0.99240614374462,
	0.99249713503587, 0.99258811199546, 0.99267907462922, 0.99277002294297,
	0.99286095694254, 0.99295187663373, 0.99304278202236, 0.99313367311425,
	0.99322454991518, 0.99331541243097, 0.9934062606674, 0.99349709463028,
	0.99358791432538, 0.9936787197585, 0.99376951093542, 0.99386028786191,
	0.99395105054375, 0.99404179898671, 0.99413253319655, 0.99422325317903,
	0.99431395893992, 0.99440465048497, 0.99449532781994, 0.99458599095056,
	0.99467663988258, 0.99476727462174, 0.99485789517378, 0.99494850154443,
	0.99503909373943, 0.99512967176448, 0.99522023562532, 0.99531078532766,
	0.99540132087722, 0.99549184227971, 0.99558234954083, 0.99567284266629,
	0.99576332166178, 0.99585378653299, 0.99594423728564, 0.99603467392539,
	0.99612509645793, 0.99621550488895, 0.99630589922412, 0.99639627946911,
	0.99648664562959, 0.99657699771124, 0.9966673357197, 0.99675765966065,
	0.99684796953972, 0.99693826536259, 0.99702854713488, 0.99711881486225,
	0.99720906855034, 0.99729930820478, 0.99738953383121, 0.99747974543525,
	0.99756994302253, 0.99766012659867, 0.99775029616929, 0.99784045174,
	0.99793059331643, 0.99802072090416, 0.99811083450881, 0.99820093413598,
	0.99829101979126, 0.99838109148024, 0.99847114920852, 0.99856119298169,
	0.99865122280531, 0.99874123868498, 0.99883124062626, 0.99892122863473,
	0.99901120271595, 0.99910116287549, 0.99919110911892, 0.99928104145177,
	0.99937095987962, 0.99946086440801, 0.99955075504248, 0.99964063178859,
	0.99973049465185, 0.99982034363783, 0.99991017875203, 1,
};
inline float convex_exponant( float value )
{
	return compute_exponant( value, convex_exponant_table, convex_exponant_table_size );
}

#endif //H2C_EXPONENTIAL_TABLES_H

/* vim: set softtabstop=4 noexpandtab: */
