/* --------------------------------------------------------------------------
 * driver_mixer_oss.c
 * This module contains the low level driver to control the OSS mixer. The
 * low level driver overloads methods of class mixer.
 *
 * Copyright 2002-2006 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>

#include <glib.h>
#include <pbb.h>

#include "gettext_macros.h"
#include "class_config.h"
#include "class_mixer.h"
#include "driver_mixer_oss.h"
#include "support.h"
#include "debug.h"

#define SECTION "MODULE MIXER OSS"

struct driver_ossmixer {
	char *device;	 /** OSS mixer device, allocated string */
	GString *channels;
	struct {
		unsigned int init_complete:1;
		unsigned int mute:1;
		unsigned int:0;
	} flags;
	int mixerdevmask;		/* supported channels from mixer */
	int userdevmask;        /* desired channels from user */
	int volume;		/* last volume set */
	int master;		/* master sound channel */
} moddata;

char *mixerchannels[] = {"volume", "bass", "treble", "synth", "pcm", "speaker", \
        "line", "mic", "cd", "imix", "altpcm", "reclev", "igain", "ogain", \
		"line1", "line2", "line3", "digital1", "digital2", "digital3", "phonein", \
		"phoneout", "video", "radio", "monitor"};

void
driver_mixer_oss_exit ()
{
	struct driver_ossmixer *base = &moddata;

	g_free (base->device);
	g_string_free (base->channels, TRUE);
}

/**
 * @brief Sets the volume of a given channel
 *
 * This function sets a channel's volume to a new level
 * It returns the average volume of the left and right channel
 *
 * @param  fd       filehandle of an open OSS mixer device
 * @param  channel  channel number that should get a new volume level
 * @param  volume   new volume levelto set
 * @return new volume level or -1 on error
 */
int
ossmixer_set_channel_volume (int fd, int channel, int volume)
{
	volume += 256*volume;
	if (ioctl (fd, MIXER_WRITE(channel), &volume) >= 0)
		return (volume & 255);
	return -1;
}

/** 
 * @brief Reads the volume of a given channel
 *
 * This function asks the mixer for a channel's volume level
 * It returns the average volume of the left and right channel
 *
 * @param  fd       filehandle of an open OSS mixer device
 * @param  channel  channel number to read out
 * @return volume level of the given channel or -1 on error
 */
int
ossmixer_get_channel_volume(int fd, int channel)
{
	int volume;
	if (ioctl (fd, MIXER_READ(channel), &volume) >= 0)
		return ((volume & 255) + ((volume >> 8) & 255)) >> 1;
	return -1;
}

/**
 * @brief Retrieves the channels the sound hardware supports
 *
 * This function asks the mixer which sound channels it
 * supports
 *
 * @param  fd   filehandle of an open OSS mixer device
 * @return channel mask or -1 on error
 */
int
ossmixer_get_channel_mask (int fd)
{
	int devmask;
	if (ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devmask) >= 0)
		return devmask;
	return -1;
}

/**
 * @brief Set masters volume level to all slave channels
 *
 * This function sets the volume level of every slave to the same
 * level as the master. If the mixer device couldn't be opened,
 * nothing would be done.
 */
void
ossmixer_set_slaves_volume ()
{
	struct driver_ossmixer *base = &moddata;
	int volume = (base->flags.mute ? 0 : base->volume);
	unsigned int n, devmask;
	int fd;

	devmask = base->userdevmask & base->mixerdevmask;
	if ((fd = open (base->device, O_RDWR)) >= 0) {     /* open mixer device */
		for (n=0; n < sizeof(mixerchannels) / sizeof(char *); n++)
			if ((devmask >> n & 1) == 1)
				ossmixer_set_channel_volume (fd, n, volume);
		close(fd);
	}
}

void
ossmixer_set_channel_mask (char *channels[])
{
	struct driver_ossmixer *base = &moddata;
	int n;

	base->userdevmask = 0;
	g_string_truncate (base->channels, 0);

	while (*channels) {
		for (n=sizeof(mixerchannels) / sizeof(char *) -1; n >= 0; n--)
			if (!strcasecmp (mixerchannels[n], *channels)) {
				if (base->userdevmask == 0)
					base->master = n;   /* first channel in list becomes master */
				else
					base->channels = g_string_append (base->channels, ", ");
				base->userdevmask |= 1 << n;
				base->channels = g_string_append (base->channels, mixerchannels[n]);
				break;
			}
		channels++;
	}
}

int
ossmixer_finish_init ()
{
	struct driver_ossmixer *base = &moddata;
	int fd, devmask;

	if ((fd = open (base->device, O_RDWR)) >= 0) {     /* open mixer device */
		if ((base->volume = ossmixer_get_channel_volume (fd, base->master)) != -1) {
			if ((devmask = ossmixer_get_channel_mask (fd)) != -1)
				base->mixerdevmask = devmask;
			else {
				base->mixerdevmask = 1 << SOUND_MIXER_VOLUME;
				print_msg (PBB_WARN, _("Can't get devmask from mixer [%s]; using default.\n"), base->device);
			}
			base->flags.init_complete = 1;        /* mixer setup completed */
			close (fd);
			return 0;
		}
		print_msg (PBB_ERR, _("Can't get volume of master channel [%s].\n"), mixerchannels[base->master]);
		close(fd);
	} else
		print_msg (PBB_WARN, _("Can't open mixer device [%s]. %s\n"), base->device, strerror(errno));
	return -1;
}

/**
 * @brief Increment or decrease the master channel volume level
 *
 * This function increases, decreases or mute/unmute sound channels.
 * It opens the mixer devices as long as it is needed. The volume level
 * is tracked internal and only syncronized time by time with the hardware
 * volume level to take external mixer adjustments into account. If
 * everything was alright the new volume level would be returned. If the
 * mixer device could not be opened the function will return -1 without
 * changing the internal volume level. If the mixer isn't already set up,
 * this function will do it.
 *
 * @param  increment  how much should the masters colume change
 * @return current volume level or -1 on error
 */
int
ossmixer_increment_master_volume (int increment)
{
	struct driver_ossmixer *base = &moddata;
	int volume = base->volume;
	int fd;

	if (base->flags.init_complete == 0)       /* is mixer setup already completed? */
		if ((ossmixer_finish_init ()) != 0)    /* no, then do it now */
			return volume;                     /* Oops, try it again later */

	if ((fd = open (base->device, O_RDWR)) >= 0) {     /* open mixer device */
		if ((volume = ossmixer_get_channel_volume (fd, base->master)) != -1) {
			if ((volume != 0) && (base->flags.mute))   /* someone else increased volume? */
				base->flags.mute = 0;                       /* so we can't stay muted. */
			if (base->flags.mute == 0)
			    if ((base->volume < volume - VOLUME_FR)
				|| (base->volume > volume + VOLUME_FR))  /* someone else changed volume */
					base->volume = volume;               /* so we must sychonize our volume level */
		}
		volume = base->volume + increment;
		if (volume > VOLUME_MAX)
			volume = VOLUME_MAX;
		if (volume < VOLUME_MIN)
			volume = VOLUME_MIN;
		if (increment != 0) {                       /* volume should be increased/decreased? */
			base->volume = volume;       /* store new volume level */
			base->flags.mute = 0;            /* unmute channels */
		} else {
			if (base->flags.mute == 0) {    /* channels not already muted? */
				volume = 0;
				base->flags.mute = 1;        /* channels are muted. */
			} else {
				volume = base->volume;    /* set former volume level again */
				base->flags.mute = 0;        /* unmute channels */
			}
		}
		ossmixer_set_channel_volume(fd, base->master, volume);
		close(fd);
	} else
		volume = -1;
	return volume;
}

gboolean
ossmixer_is_muted (void)
{
	struct driver_ossmixer *base = &moddata;
	return base->flags.mute == 1 ? TRUE : FALSE;
}

int
ossmixer_get_volume (enum valueinc type)
{
	struct driver_ossmixer *base = &moddata;

	switch (type) {
		case VALUE_ABS:
		case VALUE_ABS_PERCENT:
			return base->volume;
		case VALUE_REL:
		case VALUE_REL_PERCENT:
		default:
			return -1;   /* no relative values available */
	}
}

void
ossmixer_set_volume (enum valueinc type, int volume)
{
	struct driver_ossmixer *base = &moddata;
	int inc;

	switch (type) {
		case VALUE_REL:
		case VALUE_REL_PERCENT:
			inc = volume;
			break;
		case VALUE_ABS:
		case VALUE_ABS_PERCENT:
			inc = volume - base->volume;
			if (inc == 0) return;
	}

	if ((ossmixer_increment_master_volume (inc)) != -1)
		ossmixer_set_slaves_volume ();
}

void
ossmixer_set_mixer (char *mixer)
{
	struct driver_ossmixer *base = &moddata;
	GString *tmp;

	if ((check_path (mixer, TYPE_CHARDEV, 0)) == 0) {
		if (base->device)
			g_free (base->device);

		/* set the new mixer device */
		tmp = g_string_new (mixer);
		base->device = tmp->str;
		g_string_free (tmp, FALSE);

		/* save new mixer device to configuration file */
		config_set_string (SECTION, "Device", base->device);

		/* re-initialize the mixer with next usage */
		base->flags.init_complete = 0;
	}
}

const char *
ossmixer_get_mixer (void)
{
	struct driver_ossmixer *base = &moddata;
	return base->device;
}

void
ossmixer_set_channels (char *channels)
{
	struct driver_ossmixer *base = &moddata;
	char **tmpstrlst;
	int n;

	tmpstrlst = g_strsplit (channels, ",", 0);
	for (n=0; tmpstrlst[n] != NULL; n++)
		g_strstrip (tmpstrlst[n]);
	
	ossmixer_set_channel_mask (tmpstrlst);
	g_strfreev (tmpstrlst);
	
	/* save new mixer channels to configuration file */
	config_set_string (SECTION, "Channels", base->channels->str);
}

const char *
ossmixer_get_channels (void)
{
	struct driver_ossmixer *base = &moddata;
	return base->channels->str;
}

static struct driver_mixer driver_mixer_oss = {
	.name         = N_("OSS Mixer"),
	.get_mixer    = ossmixer_get_mixer,
	.set_mixer    = ossmixer_set_mixer,
	.get_channels = ossmixer_get_channels,
	.set_channels = ossmixer_set_channels,
	.get_volume   = ossmixer_get_volume,
	.set_volume   = ossmixer_set_volume,
	.is_muted     = ossmixer_is_muted,
	.driver_exit  = driver_mixer_oss_exit
};

struct driver_mixer *
driver_mixer_oss_init ()
{
	struct driver_ossmixer *base = &moddata;
	char **channels;

	base->master		      = SOUND_MIXER_VOLUME;
	base->mixerdevmask	      = 1 << SOUND_MIXER_VOLUME;
	base->userdevmask	      = 0;
	base->volume		      = -1;
	base->flags.mute	      = 0;
	base->flags.init_complete = 0;
	base->channels            = g_string_new (NULL);

	base->device = config_get_string (SECTION, "Device", "/dev/mixer");
	if ((check_path (base->device, TYPE_CHARDEV, 0)) == 0) {
		channels = config_get_strlist (SECTION, "Channels", "volume");
		ossmixer_set_channel_mask (channels);
		g_strfreev (channels);

		return &driver_mixer_oss;
	}

	driver_mixer_oss_exit ();
	return NULL;
}

