/**
 * WinPR: Windows Portable Runtime
 * WinPR Logger
 *
 * Copyright 2014 Armin Novak <armin.novak@thincast.com>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <winpr/config.h>

#include "CallbackAppender.h"

typedef struct
{
	wLogAppender common;

	wLogCallbacks* callbacks;
} wLogCallbackAppender;

static BOOL WLog_CallbackAppender_Open(WINPR_ATTR_UNUSED wLog* log,
                                       WINPR_ATTR_UNUSED wLogAppender* appender)
{
	return TRUE;
}

static BOOL WLog_CallbackAppender_Close(WINPR_ATTR_UNUSED wLog* log,
                                        WINPR_ATTR_UNUSED wLogAppender* appender)
{
	return TRUE;
}

static BOOL WLog_CallbackAppender_WriteMessage(wLog* log, wLogAppender* appender,
                                               const wLogMessage* cmessage)
{
	WINPR_ASSERT(cmessage);
	if (!appender)
		return FALSE;

	char prefix[WLOG_MAX_PREFIX_SIZE] = { 0 };
	WLog_Layout_GetMessagePrefix(log, appender->Layout, cmessage, prefix, sizeof(prefix));

	wLogCallbackAppender* callbackAppender = (wLogCallbackAppender*)appender;

	if (callbackAppender->callbacks && callbackAppender->callbacks->message)
	{
		wLogMessage message = *cmessage;
		message.PrefixString = prefix;
		return callbackAppender->callbacks->message(&message);
	}
	else
		return FALSE;
}

static BOOL WLog_CallbackAppender_WriteDataMessage(wLog* log, wLogAppender* appender,
                                                   const wLogMessage* cmessage)
{
	if (!appender)
		return FALSE;

	char prefix[WLOG_MAX_PREFIX_SIZE] = { 0 };
	WLog_Layout_GetMessagePrefix(log, appender->Layout, cmessage, prefix, sizeof(prefix));

	wLogCallbackAppender* callbackAppender = (wLogCallbackAppender*)appender;
	if (callbackAppender->callbacks && callbackAppender->callbacks->data)
	{
		wLogMessage message = *cmessage;
		message.PrefixString = prefix;
		return callbackAppender->callbacks->data(&message);
	}
	else
		return FALSE;
}

static BOOL WLog_CallbackAppender_WriteImageMessage(wLog* log, wLogAppender* appender,
                                                    const wLogMessage* cmessage)
{
	WINPR_ASSERT(cmessage);
	if (!appender)
		return FALSE;

	char prefix[WLOG_MAX_PREFIX_SIZE] = { 0 };
	WLog_Layout_GetMessagePrefix(log, appender->Layout, cmessage, prefix, sizeof(prefix));

	wLogCallbackAppender* callbackAppender = (wLogCallbackAppender*)appender;
	if (callbackAppender->callbacks && callbackAppender->callbacks->image)
	{
		wLogMessage message = *cmessage;
		message.PrefixString = prefix;
		return callbackAppender->callbacks->image(&message);
	}
	else
		return FALSE;
}

static BOOL WLog_CallbackAppender_WritePacketMessage(wLog* log, wLogAppender* appender,
                                                     const wLogMessage* cmessage)
{
	WINPR_ASSERT(cmessage);
	if (!appender)
		return FALSE;

	char prefix[WLOG_MAX_PREFIX_SIZE] = { 0 };
	WLog_Layout_GetMessagePrefix(log, appender->Layout, cmessage, prefix, sizeof(prefix));

	wLogCallbackAppender* callbackAppender = (wLogCallbackAppender*)appender;
	if (callbackAppender->callbacks && callbackAppender->callbacks->package)
	{
		wLogMessage message = *cmessage;
		message.PrefixString = prefix;
		return callbackAppender->callbacks->package(&message);
	}
	else
		return FALSE;
}

static BOOL WLog_CallbackAppender_Set(wLogAppender* appender, const char* setting, void* value)
{
	wLogCallbackAppender* callbackAppender = (wLogCallbackAppender*)appender;

	if (!value || (strcmp(setting, "callbacks") != 0))
		return FALSE;

	if (!(callbackAppender->callbacks = calloc(1, sizeof(wLogCallbacks))))
	{
		return FALSE;
	}

	callbackAppender->callbacks = memcpy(callbackAppender->callbacks, value, sizeof(wLogCallbacks));
	return TRUE;
}

static void WLog_CallbackAppender_Free(wLogAppender* appender)
{
	wLogCallbackAppender* callbackAppender = NULL;
	if (!appender)
	{
		return;
	}

	callbackAppender = (wLogCallbackAppender*)appender;

	free(callbackAppender->callbacks);
	free(appender);
}

wLogAppender* WLog_CallbackAppender_New(WINPR_ATTR_UNUSED wLog* log)
{
	wLogCallbackAppender* CallbackAppender =
	    (wLogCallbackAppender*)calloc(1, sizeof(wLogCallbackAppender));
	if (!CallbackAppender)
		return NULL;

	CallbackAppender->common.Type = WLOG_APPENDER_CALLBACK;
	CallbackAppender->common.Open = WLog_CallbackAppender_Open;
	CallbackAppender->common.Close = WLog_CallbackAppender_Close;
	CallbackAppender->common.WriteMessage = WLog_CallbackAppender_WriteMessage;
	CallbackAppender->common.WriteDataMessage = WLog_CallbackAppender_WriteDataMessage;
	CallbackAppender->common.WriteImageMessage = WLog_CallbackAppender_WriteImageMessage;
	CallbackAppender->common.WritePacketMessage = WLog_CallbackAppender_WritePacketMessage;
	CallbackAppender->common.Free = WLog_CallbackAppender_Free;
	CallbackAppender->common.Set = WLog_CallbackAppender_Set;

	return &CallbackAppender->common;
}
