/*
 * Copyright (C) 2015 Glyptodon LLC
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * A service for setting and retrieving browser-local preferences. Preferences
 * may be any JSON-serializable type.
 */
angular.module('settings').provider('preferenceService', function preferenceServiceProvider() {

    /**
     * Reference to the provider itself.
     *
     * @type preferenceServiceProvider
     */
    var provider = this;

    /**
     * The storage key of Guacamole preferences within local storage.
     *
     * @type String
     */
    var GUAC_PREFERENCES_STORAGE_KEY = "GUAC_PREFERENCES";

    /**
     * All valid input method type names.
     *
     * @type Object.<String, String>
     */
    var inputMethods = {

        /**
         * No input method is used. Keyboard events are generated from a
         * physical keyboard.
         *
         * @constant
         * @type String
         */
        NONE : 'none',

        /**
         * Keyboard events will be generated from the Guacamole on-screen
         * keyboard.
         *
         * @constant
         * @type String
         */
        OSK : 'osk',

        /**
         * Keyboard events will be generated by inferring the keys necessary to
         * produce typed text from an IME (Input Method Editor) such as the
         * native on-screen keyboard of a mobile device.
         *
         * @constant
         * @type String
         */
        TEXT : 'text'

    };

    /**
     * Returns the key of the language currently in use within the browser.
     * This is not necessarily the user's desired language, but is rather the
     * language user by the browser's interface.
     *
     * @returns {String}
     *     The key of the language currently in use within the browser.
     */
    var getDefaultLanguageKey = function getDefaultLanguageKey() {

        // Pull browser language, falling back to US English
        var language = (navigator.languages && navigator.languages[0])
                     || navigator.language
                     || navigator.browserLanguage
                     || 'en';

        // Convert to format used internally
        return language.replace(/-/g, '_');

    };

    /**
     * All currently-set preferences, as name/value pairs. Each property name
     * corresponds to the name of a preference.
     *
     * @type Object.<String, Object>
     */
    this.preferences = {

        /**
         * Whether translation of touch to mouse events should emulate an
         * absolute pointer device, or a relative pointer device.
         * 
         * @type Boolean
         */
        emulateAbsoluteMouse : true,

        /**
         * The default input method. This may be any of the values defined
         * within preferenceService.inputMethods.
         *
         * @type String
         */
        inputMethod : inputMethods.NONE,
        
        /**
         * The key of the desired display language.
         * 
         * @type String
         */
        language : getDefaultLanguageKey()

    };

    // Get stored preferences, ignore inability to use localStorage
    try {

        if (localStorage) {
            var preferencesJSON = localStorage.getItem(GUAC_PREFERENCES_STORAGE_KEY);
            if (preferencesJSON)
                angular.extend(provider.preferences, JSON.parse(preferencesJSON));
        }

    }
    catch (ignore) {}

    // Factory method required by provider
    this.$get = ['$injector', function preferenceServiceFactory($injector) {

        // Required services
        var $rootScope = $injector.get('$rootScope');
        var $window    = $injector.get('$window');

        var service = {};

        /**
         * All valid input method type names.
         *
         * @type Object.<String, String>
         */
        service.inputMethods = inputMethods;

        /**
         * All currently-set preferences, as name/value pairs. Each property name
         * corresponds to the name of a preference.
         *
         * @type Object.<String, Object>
         */
        service.preferences = provider.preferences;

        /**
         * Persists the current values of all preferences, if possible.
         */
        service.save = function save() {

            // Save updated preferences, ignore inability to use localStorage
            try {
                if (localStorage)
                    localStorage.setItem(GUAC_PREFERENCES_STORAGE_KEY, JSON.stringify(service.preferences));
            }
            catch (ignore) {}

        };

        // Persist settings when window is unloaded
        $window.addEventListener('unload', service.save);

        // Persist settings upon navigation 
        $rootScope.$on('$routeChangeSuccess', function handleNavigate() {
            service.save();
        });

        // Persist settings upon logout
        $rootScope.$on('guacLogout', function handleLogout() {
            service.save();
        });

        return service;

    }];

});
