#!/usr/bin/env python

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, os, sys

from gquilt_pkg import icons
from gquilt_pkg import config, dialogue, ifce, ws_event, diff
from gquilt_pkg import console, file_tree, patch_list
from gquilt_pkg import actions, gutils
from gquilt_pkg import cmd_result

_GQUILT_UI_DESCR = \
'''
<ui>
  <menubar name="gquilt_menubar">
    <menu name="playground_menu" action="gquilt_playground_actions">
      <menuitem action="new_playground"/>
      <menuitem action="change_working_directory"/>
      <menuitem action="init_playground"/>
      <menuitem action="gquilt_quit"/>
    </menu>
    <menu name="actions_menu" action="gquilt_patch_actions">
      <menuitem action="pm_pop_all"/>
      <menuitem action="pm_push_all"/>
      <menuitem action="pm_combined_diff"/>
      <menuitem action="pm_import_patch_series"/>
    </menu>
    <menu name="views_menu" action="gquilt_view_actions">
      <menuitem action="views_update_all"/>
      <menuitem action="views_update_playground"/>
      <menuitem action="views_update_files"/>
      <menuitem action="views_update_patches"/>
    </menu>
  </menubar>
  <menubar name="gquilt_right_side_menubar">
    <menu name="gquilt_config" action="gquilt_configuration">
      <menuitem action="gquilt_config_editors"/>
      <menuitem action="gquilt_config_perusers"/>
    </menu>
  </menubar>
</ui>
'''

class GQuilt(gtk.Window, actions.AGandUIManager):
    def _create_tool_bar(self):
        tbar = self.patches.ui_manager.get_widget('/patches_toolbar')
        tbar.set_orientation(gtk.ORIENTATION_HORIZONTAL)
        tbar.set_style(gtk.TOOLBAR_BOTH)
        tbcl = gtk.ToolItem()
        tbcl.set_tooltip(self.tooltips, "Type in a back end command for execution")
        tbcl.set_expand(True)
        tbcl.add(self.cmd_line)
        tbar.insert(tbcl, -1)
        return tbar
    def __init__(self, dir_specified=False):
        open_dialog = None # we need this later
        if not dir_specified:
            if ifce.playground_type() is None:
                open_dialog = config.WSOpenDialog()
                if open_dialog.run() == gtk.RESPONSE_OK:
                    wspath = open_dialog.get_path()
                    if wspath:
                        open_dialog.show_busy()
                        result = ifce.chdir(wspath)
                        open_dialog.unshow_busy()
                        dialogue.report_any_problems(result)
                else:
                    sys.exit()
                open_dialog.show_busy()
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        dialogue.init(self)
        actions.AGandUIManager.__init__(self)
        actions.add_class_indep_actions(actions.ON_PGND_INDEP,
            [
                ("gquilt_playground_actions", None, "_Playground"),
                ("gquilt_configuration", None, "_Configuration"),
                ("change_working_directory", gtk.STOCK_OPEN, "_Open", "",
                 "Change current working directory", self._change_wd_acb),
                ("new_playground", icons.STOCK_NEW_PLAYGROUND, "_New", "",
                 "Create a new intitialized playground", self._new_playground_acb),
                ("gquilt_config_editors", gtk.STOCK_PREFERENCES, "_Editor Allocation", "",
                 "Allocate editors to file types", self._config_editors_acb),
                ("gquilt_config_perusers", gtk.STOCK_PREFERENCES, "_Peruser Allocation", "",
                 "Allocate perusers to file types", self._config_perusers_acb),
                ("gquilt_quit", gtk.STOCK_QUIT, "_Quit", "",
                 "Quit", self._quit),
            ])
        actions.add_class_indep_actions(actions.ON_NOT_IN_PGND,
            [
                ("init_playground", icons.STOCK_INIT, "_Initialise", "",
                 "Initialise the current working directory as a playground", self._init_pgnd_acb),
            ])
        actions.add_class_indep_actions(actions.ON_IN_PGND,
            [
                ("gquilt_patch_actions", None, "_Actions"),
                ("gquilt_view_actions", None, "_Views"),
                ("views_update_all", icons.STOCK_SYNCH, "Update All", "",
                 "Update all views. Useful after external actions change workspace/playground state.",
                 self._update_views_acb),
                ("views_update_playground", icons.STOCK_SYNCH, "Update Playground", "",
                 "Update the working directory file tree display.",
                 self._update_files_view_acb),
                ("views_update_files", icons.STOCK_SYNCH, "Update Files", "",
                 "Update the top patch file tree display.",
                 self._update_files_view_acb),
                ("views_update_patches", icons.STOCK_SYNCH, "Update Patches", "",
                 "Update the patch list display.",
                 self._update_patches_view_acb),
                ("pm_push_all", icons.PUSH, "Push All", "",
                 "Push all unapplied patches",
                 self._push_all_patches_acb),
                ("pm_import_patch_series", icons.PUSH, "Import Patch Series", "",
                 "Import a series of quilt/mq patches",
                 self._import_patch_series_acb),
            ])
        actions.add_class_indep_actions(actions.ON_IN_PGND_PMIC,
            [
                ("pm_combined_diff", icons.STOCK_DIFF, "Combined Diff", "",
                 "View the combined diff for all currently applied patches",
                 self._combined_diff_acb),
                ("pm_pop_all", icons.POP, "Pop All", "",
                 "Pop all applied patches",
                 self._pop_all_patches_acb),
            ])
        self.ui_manager.add_ui_from_string(_GQUILT_UI_DESCR)
        self.set_icon_from_file(icons.APP_ICON_FILE)
        self.last_import_dir = None
        self.connect("destroy", self._quit)
        self.set_title("gquilt: " + os.getcwd())
        self.tooltips = gtk.Tooltips()
        console.LOG.set_size_request(720, 160)
        self.playground_files = file_tree.WSTree()
        self.playground_files.set_size_request(240, 320)
        self.patch_files = file_tree.MutablePatchTree()
        self.patch_files.set_size_request(240, 320)
        self.patch_files.get_conditional_action('menu_files').set_label('Files (top patch)')
        self.patches = patch_list.List()
        self.patches.set_size_request(240, 320)
        self.cmd_line = gutils.LabelledEntryWithHistory('quilt ')
        self.cmd_line.entry.connect("activate", self._exec_typed_cmd)
        vbox = gtk.VBox()
        self.add(vbox)
        hbox = gtk.HBox()
        hbox.pack_start(self.ui_manager.get_widget("/gquilt_menubar"), expand=False)
        hbox.pack_end(self.ui_manager.get_widget("/gquilt_right_side_menubar"), expand=False)
        vbox.pack_start(hbox, expand=False)
        vbox.pack_start(self._create_tool_bar(), False)
        vpane = gtk.VPaned()
        vbox.pack_start(vpane)
        hpane_outer = gtk.HPaned()
        hpane_inner = gtk.HPaned()
        hpane_outer.add1(self.playground_files)
        hpane_outer.add2(hpane_inner)
        hpane_inner.add1(self.patch_files)
        hpane_inner.add2(self.patches)
        vpane.add1(hpane_outer)
        vpane.add2(console.LOG)
        self.show_all()
        self.show()
        self._set_playground(os.getcwd())
        self.set_focus(self.cmd_line.entry)
        if open_dialog:
            open_dialog.unshow_busy()
            open_dialog.destroy()
    def _quit(self, _arg):
        gtk.main_quit()
    def _add_playground_to_title(self):
        self.set_title("gquilt: " + os.getcwd())
    def show_busy(self):
        self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
        while gtk.events_pending():
            gtk.main_iteration()
    def unshow_busy(self):
        self.window.set_cursor(None)
    def _update_patches_view_acb(self, _arg=None):
        ws_event.notify_events(ws_event.PATCH_CHANGES)
    def _update_files_view_acb(self, _arg=None):
        ws_event.notify_events(ws_event.FILE_CHANGES)
    def _update_views_acb(self, _arg=None):
        ws_event.notify_events(ws_event.ALL_BUT_CHANGE_WD)
    def _combined_diff_acb(self, _arg):
        self.show_busy()
        dialog = diff.CombinedDiffTextDialog(parent=dialogue.main_window)
        self.unshow_busy()
        dialog.show()
    def _set_playground(self, newpg):
        result = ifce.chdir(newpg)
        dialogue.report_any_problems(result)
        newpg = ifce.ifce.get_playground_root()
        if ifce.playground_type() is None:
            msg = os.linesep.join(["Directory %s is not repository." % newpg,
                                   "Do you wish to create one there?"])
            if dialogue.ask_yes_no(msg, self):
                is_ok, req_backend = ifce.choose_backend()
                if is_ok:
                    res, sout, serr = ifce.new_playground(newpg, req_backend)
                    if res != cmd_result.OK:
                        dialogue.inform_user(os.linesep.join([sout, serr]))
        self.show_busy()
        self._add_playground_to_title()
        self.cmd_line.set_label(ifce.ifce.cmd_label)
        console.LOG.log_entry_bold("playground: " + newpg)
        self.unshow_busy()
    def _init_pgnd_acb(self, _arg):
        is_ok, req_backend = ifce.choose_backend()
        if not is_ok:
            return
        result = ifce.new_playground(os.getcwd(), req_backend)
        dialogue.report_any_problems(result)
        self._set_playground(os.getcwd())
    def _new_playground_acb(self, _arg):
        is_ok, req_backend = ifce.choose_backend()
        if not is_ok:
            return
        newpg = dialogue.ask_dir_name("Select/create playground ..")
        if newpg is not None:
            res, sout, serr = ifce.new_playground(newpg, req_backend)
            if res != cmd_result.OK:
                dialogue.inform_user(os.linesep.join([sout, serr]))
                return
            self._set_playground(newpg)
    def _change_wd_acb(self, _arg):
        open_dialog = config.WSOpenDialog(parent=self)
        if open_dialog.run() == gtk.RESPONSE_OK:
            newpg = open_dialog.get_path()
            if newpg:
                self._set_playground(newpg)
        open_dialog.destroy()
    def _config_editors_acb(self, _arg):
        config.EditorAllocationDialog(parent=self).show()
    def _config_perusers_acb(self, _arg):
        config.PeruserAllocationDialog(parent=self).show()
    def _import_patch_series_acb(self, _arg=None):
        dirname = dialogue.ask_dir_name("Select directory to import patch series from ..", suggestion=self.last_import_dir)
        if dirname is not None:
            self.last_import_dir = dirname
            sfname = os.path.join(dirname, "series")
            if not os.path.exists(sfname):
                dialogue.inform_user("Series file not found")
                return
            fobj = open(sfname, 'r')
            series = fobj.readlines()
            series.reverse()
            fobj.close()
            for name_raw in series:
                name = name_raw.strip()
                if name == "" or name[0] == "#":
                    continue
                pfname = os.path.join(dirname, name)
                self.show_busy()
                res, sout, serr = ifce.ifce.do_import_patch(pfname)
                self.unshow_busy()
                if res == cmd_result.ERROR_SUGGEST_FORCE:
                    ans = dialogue.ask_force_skip_or_cancel((res, sout, serr), parent=gutils.get_gtk_window(self))
                    if ans == dialogue.RESPONSE_FORCE:
                        res, sout, serr = ifce.ifce.do_import_patch(pfname, force=True)
                    elif ans == dialogue.RESPONSE_SKIP:
                        continue
                    else:
                        return
                if res != cmd_result.OK:
                    dialogue.inform_user(os.linesep.join([sout, serr]))
    def _pop_all_patches_acb(self, arg):
        self.patches.do_pop_all(arg)
    def _push_all_patches_acb(self, arg):
        self.patches.do_push_all(arg)
    def _exec_typed_cmd(self, entry):
        self.show_busy()
        text = entry.get_text_and_clear_to_history()
        if text:
            res, sout, serr = ifce.ifce.do_exec_tool_cmd(text)
        else:
            res = cmd_result.OK
        self.unshow_busy()
        if res != cmd_result.OK:
            dialogue.inform_user(os.linesep.join([sout, serr]))
        self._update_views()
