/* 
 * gpsk31  - PSK31 for Linux with a GTK+ Interface
 * 
 * Copyright (C) 2000 Luc Langehegermann, LX2GT
 * Copyright (C) 2005,2006,2007,2008 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * The main author can be reached at pg4i@amsat.org or by smail-mail:
 * Joop Stakenborg, Bramengaarde 24, 3992KG Houten, The Netherlands.
 * 
 */

 /*
  * menu.C Functions for the menus
  */


#include <gtk/gtk.h>
#include <stdio.h>
#include "main_screen.h"
#include "callback.h"
#include "call_dialog.h"
#include "misc.h"
#include "globals.h"
#include "menu.h"
#include <unistd.h>
#include <stdlib.h>
#include "file.h"
#include "server.h"
#include "text.h"
#include <string.h>
#include "cfg.h"
#include "phasescope.h"
#include "spectrum.h"

extern char configfile[];	/* explicit config file to be loaded */
GtkUIManager *ui_manager;
const char *trx_mode_names[] = {
        "BPSK31",
        "QPSK31"
};

#define BPSK31 0
#define QPSK31 1
int mode;

static void send_call ();
static void send_name ();
static void send_qth ();
static void send_rsts ();
static void send_rstr ();
static void send_frequency ();
static void send_mycall ();
static void send_myqth ();
static void send_myname ();

static void mode_qpsk_toggled ();
static void mode_afc_toggled ();
static void mode_net_toggled ();
static void mode_cwid_toggled ();

/* Normal items */
static GtkActionEntry entries[] =
{
  { "FileMenu", NULL, "_File" },
  { "SendMenu", NULL, "S_end" },
  { "ModeMenu", NULL, "_Mode" },
  { "WindowMenu", NULL, "_Window" },
  { "HelpMenu", NULL, "_Help" },

    {"Send", GTK_STOCK_OPEN, "Send File", "<control>S", "", G_CALLBACK(file_send_dialog)},
    {"Log", GTK_STOCK_NEW, "Log to _File", "<control>F", "", G_CALLBACK(save_log_dialog)},
    {"Close", GTK_STOCK_CLOSE, "Close Lo_gfile", "<control>G", "", G_CALLBACK(close_logfile)},
    {"Quit", GTK_STOCK_QUIT, "_Quit GPSK31", "<control>Q", "", G_CALLBACK(exit_dialog)},

    {"Call", NULL, "_Call", "<control>C", "", G_CALLBACK(send_call)},
    {"Name", NULL, "_Name", "<control>N", "", G_CALLBACK(send_name)},
    {"QTH", NULL, "QT_H", "<control>H", "", G_CALLBACK(send_qth)},
    {"RST-S", NULL, "RS_T-S", "<control>T", "", G_CALLBACK(send_rsts)},
    {"Frequency", NULL, "Fr_equency", "<control>E", "", G_CALLBACK(send_frequency)},
    {"Mycall", NULL, "MyCa_ll", "<control>L", "", G_CALLBACK(send_mycall)},
    {"Myname", NULL, "MyNa_me", "<control>M", "", G_CALLBACK(send_myname)},
    {"MyQTH", NULL, "M_yQTH", "<control>Y", "", G_CALLBACK(send_myqth)},
    {"RST-R", NULL, "RST-_R", "<control>R", "", G_CALLBACK(send_rstr)},

    {"Configuration", GTK_STOCK_PREFERENCES, "Preferences", "<control>P", "", G_CALLBACK(config_dialog)},

    {"Manual", GTK_STOCK_HELP, "Manual", "<control>I", "", G_CALLBACK(help_dialog)},
    {"ChangeLog", GTK_STOCK_HELP, "ChangeLog", "<control>O", "", G_CALLBACK(changelog_dialog)},
    {"About", GTK_STOCK_DIALOG_INFO, "About GPSK31", "<control><shift>H", "", G_CALLBACK(about_dialog)},

};


static GtkToggleActionEntry toggle_entries[] =
{
    {"QPSK", NULL, "_QPSK", "<alt>Q", "", G_CALLBACK(mode_qpsk_toggled)},
    {"CW_ID", NULL, "_CW-ID", "<alt>C", "", G_CALLBACK(mode_cwid_toggled)},
    {"AFC", NULL, "_AFC", "<alt>A", "", G_CALLBACK(mode_afc_toggled)},
    {"NET", NULL, "_NET", "<alt>N", "", G_CALLBACK(mode_net_toggled)},

    {"Details", NULL, "QSO Details", "<control>W", "", G_CALLBACK(toggle_call_dialog)},
};

static const char *ui_description =
"<ui>"
"  <menubar name='MainMenu'>"
"	 <menu action='FileMenu'>"
"		<menuitem action='Send'/>"
"		<separator name='sep1'/>"
"		<menuitem action='Log'/>"
"		<menuitem action='Close'/>"
"		<separator name='sep2'/>"
"		<menuitem action='Quit'/>"
"	 </menu>"
"	 <menu action='SendMenu'>"
"		<menuitem action='Call'/>"
"		<menuitem action='Name'/>"
"		<menuitem action='QTH'/>"
"		<menuitem action='RST-S'/>"
"		<menuitem action='Frequency'/>"
"		<separator name='sep3'/>"
"		<menuitem action='Mycall'/>"
"		<menuitem action='Myname'/>"
"		<menuitem action='MyQTH'/>"
"		<menuitem action='RST-R'/>"
"	 </menu>"
"	 <menu action='ModeMenu'>"
"		<menuitem action='QPSK'/>"
"		<menuitem action='CW_ID'/>"
"		<menuitem action='AFC'/>"
"		<menuitem action='NET'/>"
"	 </menu>"
"	 <menu action='WindowMenu'>"
"		<menuitem action='Details'/>"
"		<separator name='sep3'/>"
"		<menuitem action='Configuration'/>"
"	 </menu>"
"	 <menu action='HelpMenu'>"
"		<menuitem action='ChangeLog'/>"
"		<menuitem action='Manual'/>"
"		<menuitem action='About'/>"
"	 </menu>"
"  </menubar>"
"</ui>";

void
init_menu ()
{
	GtkActionGroup *action_group = gtk_action_group_new ("MenuActions");
	gtk_action_group_add_actions (action_group, entries,
		G_N_ELEMENTS (entries), main_screen.window);
	gtk_action_group_add_toggle_actions (action_group,
		toggle_entries, G_N_ELEMENTS (toggle_entries), main_screen.window);

	ui_manager = gtk_ui_manager_new ();
	gtk_ui_manager_insert_action_group (ui_manager, action_group, 0);

	GtkAccelGroup *accel_group = gtk_ui_manager_get_accel_group (ui_manager);
	gtk_window_add_accel_group (GTK_WINDOW (main_screen.window), accel_group);

	gtk_ui_manager_add_ui_from_string (ui_manager, ui_description, -1, NULL);
	main_screen.menu = gtk_ui_manager_get_widget (ui_manager, "/MainMenu");
}

void about_dialog()
{
   static GtkWidget *window;
   static GtkWidget *label; 
   static GtkWidget *ok_button;

   window = gtk_dialog_new_with_buttons ("GPSK31 - About",
      GTK_WINDOW(main_screen.window), GTK_DIALOG_DESTROY_WITH_PARENT, NULL);
   label = gtk_label_new ("\nGPSK31 - Linux GTK+ PSK31 Terminal\n" \
		          "Version "VERSION" - "DATE"\n\n" \
			  "DSP Routines by Hansi Reiser, DL9RDZ\n" \
			  "GTK+ GUI by Luc Langehegermann, LX2GT\n" \
			  "Contributions by Thomas Ries, HB9XAR\n" \
			  "GTK+-2.0 version/maintenance by Joop Stakenborg, PG4I\n" \
			  );
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(window)->vbox), label);
   gtk_widget_show (label);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
   ok_button = gtk_dialog_add_button (GTK_DIALOG(window), GTK_STOCK_OK, GTK_RESPONSE_OK);
   gtk_dialog_run (GTK_DIALOG(window));
   gtk_widget_destroy (window);
}

static void display_file (const gchar *title, const gchar *filename)
{
   static GtkWidget *window;
   static GtkWidget *ok_button, *tv, *scrolled;
   gchar *helpfile, buf[80], *b;
   FILE *in;
   GtkTextBuffer *buffer;
   GtkTextIter start, end;
   PangoFontDescription *font_desc;

   gchar *temp = g_strdup_printf ("GPSK31 - %s", title);
   window = gtk_dialog_new_with_buttons (temp,
      GTK_WINDOW(main_screen.window), GTK_DIALOG_DESTROY_WITH_PARENT, NULL);
   g_free (temp);
   gtk_widget_set_size_request (window, 650, 400);
   scrolled = gtk_scrolled_window_new (NULL, NULL);
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(window)->vbox), scrolled);
   gtk_scrolled_window_set_policy
    (GTK_SCROLLED_WINDOW (scrolled), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
   tv = gtk_text_view_new ();
   gtk_container_add (GTK_CONTAINER (scrolled), tv);
   helpfile = g_strdup_printf
     ("%s%s%s", DATADIR, G_DIR_SEPARATOR_S, filename);
   buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(tv));
   gtk_text_buffer_get_bounds (buffer, &start, &end);
   b = g_new0 (gchar, 100);
   in = fopen (helpfile, "r");
   if (in)
   {
      do
      {
        if (fgets (buf, 80, in) == NULL)
           break;
        else
        {
           b = g_locale_to_utf8 (buf, -1, NULL, NULL, NULL);
           gtk_text_buffer_insert (buffer, &start, b, -1);
         }
       }
       while (!feof (in));
       fclose (in);
   }
   g_free (b);
   g_free (helpfile);
   font_desc = pango_font_description_from_string ("mono");
   gtk_widget_modify_font (tv, font_desc);
   pango_font_description_free (font_desc);
   gtk_widget_show_all (scrolled);

   ok_button = gtk_dialog_add_button (GTK_DIALOG(window), GTK_STOCK_OK, GTK_RESPONSE_OK);
   gtk_widget_grab_focus (scrolled);
   gtk_dialog_run (GTK_DIALOG(window));
   gtk_widget_destroy (window);

}

void changelog_dialog()
{
	display_file ("History of changes", "ChangeLog");
}

void help_dialog()
{
	display_file ("Hints", "README");
}

static void exit_destroy (GtkButton *button,  gpointer user_data)
{
	gtk_widget_destroy (GTK_WIDGET(user_data));
}

static void real_exit (GtkButton *button,  gpointer user_data)
{
  char *home = getenv ("HOME");
  char conf[strlen (home) + 60];

  if (strlen(configfile) == 0)
  {
    strcpy (conf, home);
    strcat (conf, "/.gpsk31/gpsk31.cfg");
  }
  else
  {
    /* first try full path as specified */
    strcpy (conf, configfile);
    if (!g_file_test(conf, G_FILE_TEST_EXISTS))
    {
      /* not found, try again in $HOME/.gpsk31 */
      strcpy (conf, home);
      strcat (conf, "/.gpsk31/");
      strcat (conf, configfile);
    }
  }
  savepreferences (conf);
  gtk_main_quit ();
}

void exit_dialog()
{
   static GtkWidget    *window = NULL;
   static GtkWidget    *label;
   static GtkWidget    *yes_button;
   static GtkWidget    *no_button;

   window = gtk_dialog_new_with_buttons ("GPSK31 - Exit",
      GTK_WINDOW(main_screen.window), GTK_DIALOG_DESTROY_WITH_PARENT, NULL);
   gtk_window_set_default_size (GTK_WINDOW(window), 300, 100);
   g_signal_connect (G_OBJECT (window), "destroy",
                          G_CALLBACK (gtk_widget_destroyed),
                          &window);
   label = gtk_label_new ("Do you really want to quit?"); 
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(window)->vbox), label);
   gtk_widget_show (label);

   yes_button = gtk_dialog_add_button (GTK_DIALOG(window), GTK_STOCK_OK, GTK_RESPONSE_OK);
   no_button = gtk_dialog_add_button (GTK_DIALOG(window), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);

   g_signal_connect (G_OBJECT (no_button), "clicked",
                       G_CALLBACK (exit_destroy), G_OBJECT (window));

   g_signal_connect (G_OBJECT (yes_button), "clicked",
			G_CALLBACK (real_exit), NULL);
   gtk_widget_show_all (window);
}

void warning_dialog (const gchar *str, gboolean exit_program)
{
   static GtkWidget     *window;
   static GtkWidget     *label;

   window = gtk_dialog_new_with_buttons ("GPSK31 - Warning",
      GTK_WINDOW(main_screen.window), GTK_DIALOG_DESTROY_WITH_PARENT, NULL);
   gtk_window_set_default_size (GTK_WINDOW(window), 400, 100);
   gtk_dialog_add_button (GTK_DIALOG(window), GTK_STOCK_OK, GTK_RESPONSE_OK);
   label = gtk_label_new (NULL);
   gtk_label_set_markup (GTK_LABEL (label), str);
   gtk_container_add (GTK_CONTAINER (GTK_DIALOG(window)->vbox), label);
   gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
   gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_CENTER);
   gtk_widget_show (label);

   gtk_dialog_run (GTK_DIALOG (window));
   if (exit_program)
     exit (0);
   else
     gtk_widget_destroy (window);
}

void
on_config_page_select (GtkTreeSelection *selection, gpointer user_data)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	    
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gchar *tmp = gtk_tree_model_get_string_from_iter (model, &iter);
		gtk_notebook_set_current_page (GTK_NOTEBOOK(user_data), atoi((const char *)tmp));
	}
}

void config_dialog()
{
	GdkColor color;
	GdkColormap *cmap = gdk_colormap_get_system ();

	gchar *title;
	if (strlen(configfile) == 0)
		title = g_strdup ("GPSK31 - Preferences");
	else
		title = g_strdup_printf ("GPSK31 - Preferences - %s", configfile);
	GtkWidget *window = gtk_dialog_new_with_buttons (title,
		GTK_WINDOW(main_screen.window), GTK_DIALOG_DESTROY_WITH_PARENT,
		GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
		GTK_STOCK_OK, GTK_RESPONSE_OK, NULL);
	g_free (title);
	GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (GTK_DIALOG(window)->vbox), hbox);

	GtkListStore *store = gtk_list_store_new (1, G_TYPE_STRING);
	GtkWidget *treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	g_object_unref (G_OBJECT (store));
	gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview), FALSE);

	GtkCellRenderer *renderer = gtk_cell_renderer_text_new ();
	GtkTreeViewColumn *column =
		gtk_tree_view_column_new_with_attributes ("", renderer, "text", 0, NULL);
	gtk_tree_view_column_set_sizing
		(GTK_TREE_VIEW_COLUMN(column), GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_resizable
		(GTK_TREE_VIEW_COLUMN(column), TRUE);
	gtk_tree_view_column_set_fixed_width
		(GTK_TREE_VIEW_COLUMN(column), 130);
	gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

	GtkTreeIter iter;
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "Modem", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "Options", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "TX window", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "RX window", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "Phase scope", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "Spectrum/Waterfall", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "Devices", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "Personal", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "F1 - F8", -1);
	gtk_list_store_append (store, &iter);
	gtk_list_store_set (store, &iter, 0, "Shift F1 - F8", -1);

	gtk_container_add (GTK_CONTAINER (hbox), treeview);
	GtkWidget *vsep  = gtk_vseparator_new ();
	gtk_container_add (GTK_CONTAINER(hbox), vsep);
	GtkWidget *notebook = gtk_notebook_new ();
	gtk_container_add (GTK_CONTAINER(hbox), notebook);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (notebook), FALSE);
	
	/* add callback for selecting a row */
	GObject *selection = G_OBJECT (gtk_tree_view_get_selection(GTK_TREE_VIEW (treeview)));
	gtk_tree_selection_set_mode (GTK_TREE_SELECTION (selection), GTK_SELECTION_SINGLE);
	g_signal_connect (selection, "changed", G_CALLBACK (on_config_page_select), notebook);

	/* Modem */
	GtkWidget *vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	GtkWidget *label = gtk_label_new ("The settings on this page require a restart");
	gtk_container_add (GTK_CONTAINER (vbox), label);
	GtkWidget *hsep = gtk_hseparator_new ();
	gtk_container_add (GTK_CONTAINER (vbox), hsep);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Modem frequency");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *freqspin = gtk_spin_button_new_with_range (800, 1200, 1);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (freqspin), ini_settings.freq);
	gtk_container_add (GTK_CONTAINER (hbox), freqspin);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("DCD level");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *dcdspin = gtk_spin_button_new_with_range (0, 30, 1);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (dcdspin), ini_settings.dcd_level);
	gtk_container_add (GTK_CONTAINER (hbox), dcdspin);
	GtkWidget *lsbcheck = gtk_check_button_new_with_label ("Use LSB for qpsk");
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	gtk_box_pack_start (GTK_BOX (hbox), lsbcheck, TRUE, FALSE, 100);
	if (ini_settings.lsb == 1)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (lsbcheck), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (lsbcheck), FALSE);

	/* Options */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	label = gtk_label_new ("The settings on this page require a restart");
	gtk_container_add (GTK_CONTAINER (vbox), label);
	hsep = gtk_hseparator_new ();
	gtk_container_add (GTK_CONTAINER (vbox), hsep);
	GtkWidget *newrxcheck = gtk_check_button_new_with_label ("Use new demodulator/rx filter");
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	gtk_box_pack_start (GTK_BOX (hbox), newrxcheck, TRUE, FALSE, 100);
	if (ini_settings.new_rx == 1)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (newrxcheck), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (newrxcheck), FALSE);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Filename for the log");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *logentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), logentry);
	gtk_entry_set_text (GTK_ENTRY(logentry), ini_settings.log_filename);
	GtkWidget *xlogautocheck = gtk_check_button_new_with_label ("Ask xlog to fill in the current\nfrequency when logging");
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	gtk_box_pack_start (GTK_BOX (hbox), xlogautocheck, TRUE, FALSE, 100);
	if (ini_settings.xlog_auto_freq == 1)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (xlogautocheck), TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (xlogautocheck), FALSE);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("TCP port to listen for rx/tx data\n(set to zero to disable)");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *tcpportentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), tcpportentry);
	gchar *portstr = g_strdup_printf ("%d", ini_settings.tcp_listen_port);
	gtk_entry_set_text (GTK_ENTRY(tcpportentry), portstr);
	g_free (portstr);

	/* TX window */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	label = gtk_label_new ("Text color to use for the TX window");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.tx_window_color, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *txwindowcolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), txwindowcolorbutton, TRUE, FALSE, 0);
	label = gtk_label_new ("Font to use for the TX window");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	GtkWidget *txwindowfontbutton = gtk_font_button_new_with_font (ini_settings.tx_window_font);
	gtk_box_pack_start (GTK_BOX (vbox), txwindowfontbutton, TRUE, FALSE, 0);

	/* RX window */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	label = gtk_label_new ("Color to use for received text in the RX window");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.rx_window_color_rx, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *rxwindowrxcolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), rxwindowrxcolorbutton, TRUE, FALSE, 0);
	label = gtk_label_new ("Color to use for transmitted text in the RX window");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.rx_window_color_tx, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *rxwindowtxcolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), rxwindowtxcolorbutton, TRUE, FALSE, 0);
	label = gtk_label_new ("Font to use for the RX window");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	GtkWidget *rxwindowfontbutton = gtk_font_button_new_with_font (ini_settings.rx_window_font);
	gtk_box_pack_start (GTK_BOX (vbox), rxwindowfontbutton, TRUE, FALSE, 0);

	/* Phase scope */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	label = gtk_label_new ("Color to use for DCD on");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.phase_scope_dcd_on_color, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *dcd_oncolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), dcd_oncolorbutton, TRUE, FALSE, 0);
	label = gtk_label_new ("Color to use for DCD off");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.phase_scope_dcd_off_color, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *dcd_offcolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), dcd_offcolorbutton, TRUE, FALSE, 0);
	label = gtk_label_new ("Color to use for phase scope arc");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.phase_scope_arc_color, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *arccolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), arccolorbutton, TRUE, FALSE, 0);

	/* Spectrum/Waterfall */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	label = gtk_label_new ("Color of the spectrum display");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.spectrum_color, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *spectrumcolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), spectrumcolorbutton, TRUE, FALSE, 0);
	label = gtk_label_new ("Color of the spectrum and waterfall tune line");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.spectrum_tune_line_color, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *spectrumtunelinecolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), spectrumtunelinecolorbutton, TRUE, FALSE, 0);
	label = gtk_label_new ("Color of the spectrum and waterfall center frequency");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, FALSE, 0);
	gdk_color_parse (ini_settings.spectrum_center_line_color, &color);
	if (!gdk_colormap_alloc_color (cmap, &color, TRUE, TRUE))
	{
		g_error ("Couldn't allocate color");
	}
	GtkWidget *spectrumcenterlinecolorbutton = gtk_color_button_new_with_color (&color);
	gtk_box_pack_start (GTK_BOX (vbox), spectrumcenterlinecolorbutton, TRUE, FALSE, 0);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Spectrum/Waterfall bandwidth\n(requires restart)");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *bwcombo = gtk_combo_box_new_text ();
	gtk_combo_box_append_text (GTK_COMBO_BOX(bwcombo), "Small");
	gtk_combo_box_append_text (GTK_COMBO_BOX(bwcombo), "Normal");
	gtk_combo_box_append_text (GTK_COMBO_BOX(bwcombo), "Wide");
	gtk_combo_box_set_active (GTK_COMBO_BOX(bwcombo), ini_settings.spectrum_width);
	gtk_container_add (GTK_CONTAINER (hbox), bwcombo);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Spectrum/Waterfall speed\n(requires restart)");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *speedspin = gtk_spin_button_new_with_range (1, 500, 1);
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (speedspin), ini_settings.spectrum_speed);
	gtk_container_add (GTK_CONTAINER (hbox), speedspin);

	/* Devices */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	label = gtk_label_new ("The settings on this page require a restart");
	gtk_container_add (GTK_CONTAINER (vbox), label);
	hsep = gtk_hseparator_new ();
	gtk_container_add (GTK_CONTAINER (vbox), hsep);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Device for PTT\n(leave empty when\nyou don't use PTT)");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *pttentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), pttentry);
	gtk_entry_set_text (GTK_ENTRY(pttentry), ini_settings.ptt_device);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Audio device");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *audioentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), audioentry);
	gtk_entry_set_text (GTK_ENTRY(audioentry), ini_settings.audio_device);

	/* Personal */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Name");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *nameentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), nameentry);
	gtk_entry_set_text (GTK_ENTRY(nameentry), ini_settings.name);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("QTH");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *qthentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), qthentry);
	gtk_entry_set_text (GTK_ENTRY(qthentry), ini_settings.qth);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Call");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *callentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), callentry);
	gtk_entry_set_text (GTK_ENTRY(callentry), ini_settings.call);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("String for CW ID");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	GtkWidget *cwentry = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), cwentry);
	gtk_entry_set_text (GTK_ENTRY(cwentry), ini_settings.cw_string);

	/* F1-F8 */
	vbox = gtk_vbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Label");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	label = gtk_label_new ("Text");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	hsep = gtk_hseparator_new ();
	gtk_container_add (GTK_CONTAINER (vbox), hsep);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F1label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F1label);
	gtk_entry_set_text (GTK_ENTRY(F1label), ini_settings.label[0]);
	GtkWidget *F1text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F1text);
	gtk_entry_set_text (GTK_ENTRY(F1text), ini_settings.text[0]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F2label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F2label);
	gtk_entry_set_text (GTK_ENTRY(F2label), ini_settings.label[1]);
	GtkWidget *F2text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F2text);
	gtk_entry_set_text (GTK_ENTRY(F2text), ini_settings.text[1]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F3label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F3label);
	gtk_entry_set_text (GTK_ENTRY(F3label), ini_settings.label[2]);
	GtkWidget *F3text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F3text);
	gtk_entry_set_text (GTK_ENTRY(F3text), ini_settings.text[2]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F4label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F4label);
	gtk_entry_set_text (GTK_ENTRY(F4label), ini_settings.label[3]);
	GtkWidget *F4text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F4text);
	gtk_entry_set_text (GTK_ENTRY(F4text), ini_settings.text[3]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F5label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F5label);
	gtk_entry_set_text (GTK_ENTRY(F5label), ini_settings.label[4]);
	GtkWidget *F5text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F5text);
	gtk_entry_set_text (GTK_ENTRY(F5text), ini_settings.text[4]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F6label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F6label);
	gtk_entry_set_text (GTK_ENTRY(F6label), ini_settings.label[5]);
	GtkWidget *F6text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F6text);
	gtk_entry_set_text (GTK_ENTRY(F6text), ini_settings.text[5]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F7label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F7label);
	gtk_entry_set_text (GTK_ENTRY(F7label), ini_settings.label[6]);
	GtkWidget *F7text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F7text);
	gtk_entry_set_text (GTK_ENTRY(F7text), ini_settings.text[6]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *F8label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F8label);
	gtk_entry_set_text (GTK_ENTRY(F8label), ini_settings.label[7]);
	GtkWidget *F8text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), F8text);
	gtk_entry_set_text (GTK_ENTRY(F8text), ini_settings.text[7]);

	/* SHIFT F1-F8 */
	vbox = gtk_vbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (notebook), vbox);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	label = gtk_label_new ("Label");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	label = gtk_label_new ("Text");
	gtk_container_add (GTK_CONTAINER (hbox), label);
	hsep = gtk_hseparator_new ();
	gtk_container_add (GTK_CONTAINER (vbox), hsep);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF1label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF1label);
	gtk_entry_set_text (GTK_ENTRY(shiftF1label), ini_settings.label[8]);
	GtkWidget *shiftF1text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF1text);
	gtk_entry_set_text (GTK_ENTRY(shiftF1text), ini_settings.text[8]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF2label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF2label);
	gtk_entry_set_text (GTK_ENTRY(shiftF2label), ini_settings.label[9]);
	GtkWidget *shiftF2text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF2text);
	gtk_entry_set_text (GTK_ENTRY(shiftF2text), ini_settings.text[9]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF3label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF3label);
	gtk_entry_set_text (GTK_ENTRY(shiftF3label), ini_settings.label[10]);
	GtkWidget *shiftF3text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF3text);
	gtk_entry_set_text (GTK_ENTRY(shiftF3text), ini_settings.text[10]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF4label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF4label);
	gtk_entry_set_text (GTK_ENTRY(shiftF4label), ini_settings.label[11]);
	GtkWidget *shiftF4text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF4text);
	gtk_entry_set_text (GTK_ENTRY(shiftF4text), ini_settings.text[11]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF5label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF5label);
	gtk_entry_set_text (GTK_ENTRY(shiftF5label), ini_settings.label[12]);
	GtkWidget *shiftF5text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF5text);
	gtk_entry_set_text (GTK_ENTRY(shiftF5text), ini_settings.text[12]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF6label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF6label);
	gtk_entry_set_text (GTK_ENTRY(shiftF6label), ini_settings.label[13]);
	GtkWidget *shiftF6text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF6text);
	gtk_entry_set_text (GTK_ENTRY(shiftF6text), ini_settings.text[13]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF7label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF7label);
	gtk_entry_set_text (GTK_ENTRY(shiftF7label), ini_settings.label[14]);
	GtkWidget *shiftF7text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF7text);
	gtk_entry_set_text (GTK_ENTRY(shiftF7text), ini_settings.text[14]);
	hbox = gtk_hbox_new (TRUE, 0);
	gtk_container_add (GTK_CONTAINER (vbox), hbox);
	GtkWidget *shiftF8label = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF8label);
	gtk_entry_set_text (GTK_ENTRY(shiftF8label), ini_settings.label[15]);
	GtkWidget *shiftF8text = gtk_entry_new ();
	gtk_container_add (GTK_CONTAINER (hbox), shiftF8text);
	gtk_entry_set_text (GTK_ENTRY(shiftF8text), ini_settings.text[15]);

	gtk_widget_show_all (window);
	gint response = gtk_dialog_run (GTK_DIALOG(window));
	if (response == GTK_RESPONSE_OK)
	{
		/* Modem */
		ini_settings.freq = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(freqspin));
		ini_settings.dcd_level = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(dcdspin));
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (lsbcheck)))
			ini_settings.lsb = 1;
		else
			ini_settings.lsb = 0;
		
		/* Options */
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (newrxcheck)))
			ini_settings.new_rx = 1;
		else
			ini_settings.new_rx = 0;
		const gchar *logname = gtk_entry_get_text (GTK_ENTRY(logentry));
		if (strlen(logname) > 0)
			ini_settings.log_filename = g_strdup (logname);
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (xlogautocheck)))
			ini_settings.xlog_auto_freq = 1;
		else
			ini_settings.xlog_auto_freq = 0;
		const gchar *tcpport = gtk_entry_get_text (GTK_ENTRY(tcpportentry));
		if (strlen(tcpport) > 0)
 		ini_settings.tcp_listen_port = atoi (tcpport);
 
		/* TX window */
		gtk_color_button_get_color (GTK_COLOR_BUTTON(txwindowcolorbutton), &color);
		ini_settings.tx_window_color = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		const gchar *temp = gtk_font_button_get_font_name (GTK_FONT_BUTTON(txwindowfontbutton));
		ini_settings.tx_window_font = g_strdup (temp);
		/* tx colors and font */
		GtkTextBuffer *txbuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (main_screen.txwindow));
		highlight.tx = g_strdup_printf ("%d", rand ());
		gtk_text_buffer_create_tag (txbuffer, highlight.tx, "foreground", ini_settings.tx_window_color, NULL);
		
		PangoFontDescription *txfont = pango_font_description_from_string (ini_settings.tx_window_font);
		gtk_widget_modify_font (main_screen.txwindow, txfont);
		pango_font_description_free(txfont);

		/* RX window */
		gtk_color_button_get_color (GTK_COLOR_BUTTON(rxwindowrxcolorbutton), &color);
		ini_settings.rx_window_color_rx = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		gtk_color_button_get_color (GTK_COLOR_BUTTON(rxwindowtxcolorbutton), &color);
		ini_settings.rx_window_color_tx = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		temp = gtk_font_button_get_font_name (GTK_FONT_BUTTON(rxwindowfontbutton));
		ini_settings.rx_window_font = g_strdup (temp);
		/* rx colors and font */
		GtkTextBuffer *rxbuffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (main_screen.rxwindow));
		highlight.rxrx = g_strdup_printf ("%d", rand ());
		gtk_text_buffer_create_tag (rxbuffer, highlight.rxrx, "foreground", ini_settings.rx_window_color_rx, NULL);
		highlight.rxtx = g_strdup_printf ("%d", rand ());
		gtk_text_buffer_create_tag (rxbuffer, highlight.rxtx, "foreground", ini_settings.rx_window_color_rx, NULL);
		PangoFontDescription *rxfont = pango_font_description_from_string (ini_settings.rx_window_font);
		gtk_widget_modify_font (main_screen.rxwindow, rxfont);
		pango_font_description_free(rxfont);

		/* Phase scope */
		gtk_color_button_get_color (GTK_COLOR_BUTTON(dcd_oncolorbutton), &color);
		ini_settings.phase_scope_dcd_on_color = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		gtk_color_button_get_color (GTK_COLOR_BUTTON(dcd_offcolorbutton), &color);
		ini_settings.phase_scope_dcd_off_color = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		gtk_color_button_get_color (GTK_COLOR_BUTTON(arccolorbutton), &color);
		ini_settings.phase_scope_arc_color = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		phase_scope_init ();
		draw_arc ();

		/* Spectrum */
		gtk_color_button_get_color (GTK_COLOR_BUTTON(spectrumcolorbutton), &color);
		ini_settings.spectrum_color = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		gtk_color_button_get_color (GTK_COLOR_BUTTON(spectrumtunelinecolorbutton), &color);
		ini_settings.spectrum_tune_line_color = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		gtk_color_button_get_color (GTK_COLOR_BUTTON(spectrumcenterlinecolorbutton), &color);
		ini_settings.spectrum_center_line_color = g_strdup_printf
			("#%02x%02x%02x", color.red >> 8, color.green >> 8, color.blue >> 8);
		ini_settings.spectrum_width = gtk_combo_box_get_active (GTK_COMBO_BOX(bwcombo));
		ini_settings.spectrum_speed = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(speedspin));
		spectrum_setup ();
		
		/* Devices */
		const gchar *pttname = gtk_entry_get_text (GTK_ENTRY(pttentry));
		ini_settings.ptt_device = g_strdup (pttname);
		const gchar *audioname = gtk_entry_get_text (GTK_ENTRY(audioentry));
		ini_settings.audio_device = g_strdup (audioname);

		/* Personal */
		const gchar *name = gtk_entry_get_text (GTK_ENTRY(nameentry));
		ini_settings.name = g_strdup (name);
		const gchar *qth = gtk_entry_get_text (GTK_ENTRY(qthentry));
		ini_settings.qth = g_strdup (qth);
		const gchar *call = gtk_entry_get_text (GTK_ENTRY(callentry));
		ini_settings.call = g_strdup (call);
		const gchar *cw_string = gtk_entry_get_text (GTK_ENTRY(cwentry));
		ini_settings.cw_string = g_strdup (cw_string);
		
		/* F1-F8 */
		const char *str = gtk_entry_get_text (GTK_ENTRY(F1label));
		ini_settings.label[0] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F1text));
		ini_settings.text[0] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F2label));
		ini_settings.label[1] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F2text));
		ini_settings.text[1] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F3label));
		ini_settings.label[2] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F3text));
		ini_settings.text[2] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F4label));
		ini_settings.label[3] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F4text));
		ini_settings.text[3] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F5label));
		ini_settings.label[4] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F5text));
		ini_settings.text[4] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F6label));
		ini_settings.label[5] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F6text));
		ini_settings.text[5] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F7label));
		ini_settings.label[6] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F7text));
		ini_settings.text[6] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F8label));
		ini_settings.label[7] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(F8text));
		ini_settings.text[7] = g_strdup (str);

		/* SHIFT F1-F8 */
		str = gtk_entry_get_text (GTK_ENTRY(shiftF1label));
		ini_settings.label[8] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF1text));
		ini_settings.text[8] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF2label));
		ini_settings.label[9] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF2text));
		ini_settings.text[9] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF3label));
		ini_settings.label[10] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF3text));
		ini_settings.text[10] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF4label));
		ini_settings.label[11] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF4text));
		ini_settings.text[11] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF5label));
		ini_settings.label[12] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF5text));
		ini_settings.text[12] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF6label));
		ini_settings.label[13] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF6text));
		ini_settings.text[13] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF7label));
		ini_settings.label[14] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF7text));
		ini_settings.text[14] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF8label));
		ini_settings.label[15] = g_strdup (str);
		str = gtk_entry_get_text (GTK_ENTRY(shiftF8text));
		ini_settings.text[15] = g_strdup (str);
	}
	gtk_widget_destroy (window);
}


/* This unchecks the qso_details_menu */
void check_menu(const char *path, int state)
{
	GtkWidget *item =  gtk_ui_manager_get_widget (ui_manager, path);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(item), state);
}

/*
 * This functions sends the info entered in the qso details dialog
 */

void send_call ()
{
	commPutData (qsodata.call, -1);
	put_tx_window (qsodata.call);
}


void send_name ()
{
        commPutData (qsodata.name, -1);
	put_tx_window (qsodata.name);
}

void send_qth ()
{
        commPutData (qsodata.qth, -1);
	put_tx_window (qsodata.qth);
}

void send_rsts ()
{
        commPutData (qsodata.rst_s, -1);
	put_tx_window (qsodata.rst_s);
}

void send_rstr ()
{
        commPutData (qsodata.rst_r, -1);
	put_tx_window (qsodata.rst_r);
}

void send_frequency ()
{
        commPutData (qsodata.freq, -1);
	put_tx_window (qsodata.freq);
}

void send_mycall ()
{
        commPutData (ini_settings.call, -1);
	put_tx_window (ini_settings.call);
}

void send_myqth ()
{
        commPutData (ini_settings.qth, -1);
	put_tx_window (ini_settings.qth);
}

void send_myname ()
{
        commPutData (ini_settings.name, -1);
	put_tx_window (ini_settings.name);
}

const char *trx_get_mode_name(void)
{
  return trx_mode_names[mode];
}


void mode_qpsk_toggled ()
{
	if (GTK_CHECK_MENU_ITEM (gtk_ui_manager_get_widget (ui_manager, "/MainMenu/ModeMenu/QPSK"))->active)
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.mode_qpsk), TRUE);
		mode = QPSK31;
	}
	else
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.mode_qpsk), FALSE);
		mode = BPSK31;
	}
}
                                                        
void mode_afc_toggled ()
{
	if (GTK_CHECK_MENU_ITEM (gtk_ui_manager_get_widget (ui_manager, "/MainMenu/ModeMenu/AFC"))->active)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.afc), TRUE);
	else    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.afc), FALSE);
}

void mode_net_toggled ()
{
        if (GTK_CHECK_MENU_ITEM (gtk_ui_manager_get_widget (ui_manager, "/MainMenu/ModeMenu/NET"))->active)
                gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.net), TRUE);
        else    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.net), FALSE);
}

void mode_cwid_toggled ()
{
	if (GTK_CHECK_MENU_ITEM (gtk_ui_manager_get_widget (ui_manager, "/MainMenu/ModeMenu/CW_ID"))->active)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.cwid), TRUE);
	else 
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (main_screen.cwid), FALSE);
}
