-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/esql/driver.adb,v $
--  Description     : Main driver for the embedded SQL translator
--  Author          : Michael Erdmann                                        --
--  Created         : 18.12.2000                                             --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2007/06/23 15:56:18 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This package contains the main driver of the translator, which means     --
--  the command line is processed and for each file, the esql translator     --
--  is invoked.                                                              --
--  If an error happens in the ESQL translator, the processed file is        --
--  closed and the next file is invoked for processing.                      --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------

--* Ada
with Ada.Exceptions;                    use Ada.Exceptions;
with Ada.Strings;                       use Ada.Strings;
with Ada.Strings.Unbounded;             use Ada.Strings.Unbounded;
with Ada.Strings.Fixed;                 use Ada.Strings.Fixed;
with Ada.Calendar;                      use Ada.Calendar;

with Ada.Unchecked_Conversion;
with Ada.Text_IO;                       use Ada.Text_IO;
with Ada.Command_Line;                  use Ada.Command_Line;

--* Translator packages
with Scanner;                           use Scanner;
with ESQL_Parser;                       use ESQL_Parser;
with ESQL_Variables;                    use ESQL_Variables;
with Options;                           use Options;

package body Driver is

   Version : constant String :=
       "$Id: driver.adb,v 1.41 2007/06/23 15:56:18 merdmann Exp $";

   Supported_Compilers : array( Target_Compiler_Type ) of Unbounded_String := (
      GNU_Ada => To_Unbounded_String("gnat"),
      Aonix   => To_Unbounded_String("aonix"),
      Unknown => To_Unbounded_String("unknown")
   );

   Copyright_Shown      : Boolean := False;
   Show_Help_Info       : Boolean := False;

   Nbr_Of_Errors        : Natural := 0;
   Nbr_Of_Warnings      : Natural := 0;
   Nbr_Of_Files         : Natural := 0;
   Nbr_Of_Files_Success : Natural := 0;

   ------------------
   -- Process_File --
   ------------------
   procedure Process_File(
      F      : in out Scanner.File_Reader;
      Result : out  Exit_Status ) is
      --
      -- Process a file by reading in the source file and
      -- checking for ISO/92 invocation key words.
      --
      P      : ESQL_Reader := New_ESQL_Parser(f);
   begin
      loop
         Embedded_SQL_Statement( P );   --  ISO/92 Section 19.1
      end loop;

   exception
      when End_of_File_Exception =>
         Nbr_Of_Errors   := Nbr_Of_Errors   + Number_Of_Errors( P );
         Nbr_Of_Warnings := Nbr_Of_Warnings + Number_Of_Warnings( P );

         if Number_Of_Errors( P ) > 0 then
            Result := Failure;
         else
            Result := Success;
         end if;
      when The_Error : Others =>
         Message( f,
            "Fatal error, uncatched exception " &
            Exception_Name( The_Error ) &
            " occured" );
         Message( f, "*** Processing of file aborted ***" );
         Result := Failure;
   end Process_File;

   -----------
   -- Error --
   -----------
   procedure Error(
      s : String ) is
      -- print out an error message
   begin
      Put_Line( Standard_Error, "Error : " & s );
   end Error;

   ----------------
   -- Assginment --
   ----------------
   procedure Handle_Variable_Definition(
      S : in String) is
      N : Unbounded_String := Null_Unbounded_String;
      V : Unbounded_String := Null_Unbounded_String;
      -- Breakup assginments like <name>'='<value> by storing
      -- the name into N and the value into V
      I : Natural := S'First;
   begin
      while I in S'Range loop
         exit when S(I) = '=' ;

         N := N & S(I);
         I := I + 1;
      end loop;

      if S(I) = '=' then
         I := I + 1;
         while I in S'Range loop
            exit when S(I) <= ' ';

            V := V & S(I);
            I := I + 1;
         end loop;
      end if;

      ESQL_Variables.Add( To_String(N), To_String(V) );
   end Handle_Variable_Definition;

   ------------------
   -- Option_Flags --
   ------------------
   procedure Handle_Option_Flag(
      S       : in String;
      Current : in out Natural ) is
      -- Handle the option flags of the esql translator
   begin
      Current := Current + 1;
      if S = "-pedantic" then
         Option_Pedantic        := True;
      elsif s = "-v" then
         Option_Verbose         := True;
      elsif s = "-debugcode" then
         Option_Debug_Code      := True;
      elsif S = "-iso92"  then
         Option_ISO92_Exception := True;
      elsif S = "-nosqlstate" then
         Option_No_SQLSTATE     := True;
      elsif S = "-debug" then
         Option_DEBUG           := True;
      elsif S = "-odbc" then
         Option_DBCS := To_Unbounded_String("ODBC");
      elsif S = "-s" then
         Option_Silent          := True;
      elsif S = "-connectpkg" then
         Option_Connect_Pkg := To_Unbounded_String(Argument(Current));
         Current := Current + 1;
      elsif S = "-schema" then
         Create( Schema_File,
             Mode => Out_File,
             Name => Argument(Current) );
         Option_SCHEMA := True;
         Current := Current + 1;
      elsif  S = "-compiler" then
         Target_Compiler := Unknown;

         for I  in Target_Compiler_Type loop
            if Argument(Current) = Supported_Compilers(I) then
               Target_Compiler := I;
               exit;
            end if;
         end loop;

         if Target_Compiler = Unknown then
            Put_Line("Warning: compiler set to unknown");
         end if;

         Current := Current + 1;
      elsif S = "-limit" then
         Option_Error_Limit := Integer'Value(Argument(Current));
         Current := Current + 1;
      elsif S = "-gnatnosref" then
         Option_GNAT_Sref := False;
      elsif S = "-h" or S = "-help" or S = "--help" then
         Show_Help_Info := True;
      else
         if S'Length > 2 and then S(1..2) = "-D" then
            Handle_Variable_Definition( S(3..S'Length) );
         else
            Error("unknown switch " & S & " ignored ");
         end if;
      end if;


      -- turn of compiler specific flags for each supported compiler.
      case Target_Compiler is
         when GNU_Ada =>
            null;

         when others =>
            Option_GNAT_SREF := False;
      end case;

   end Handle_Option_Flag;

   ----------
   -- Name --
   ----------

   function Name(
      s   : String ) return String is
      --
      -- Return the part for the name after the extentsion has been
      -- removed from the name, e.g. test.help.esql returns test.help.
      --
      pos : Integer;
   begin
      pos := Index( s, ".", Backward );
      if pos = 0 then
         return s;
      else
         return s(1..pos-1);
      end if;
    end Name;

    ---------------
    -- Copyright --
    ---------------

    procedure Copyright(
       Version : in String ) is
       -- the copyright will be shown only once
    begin
       if not Option_SILENT and not Copyright_Shown then
          Put_Line("");
          Put_Line("GNADE Embedded SQL (GESQL) Ada 95 Translator Version " & Version);
          Put_Line("Copyright (C) 2000-2007 Michael Erdmann");
          Put_Line("");

          Copyright_Shown := True;
       end if;
    end Copyright;

    -------------
    -- Summary --
    -------------
    procedure Summary is
    begin
       Put_Line("");

       if Nbr_Of_Files_Success /= Nbr_Of_Files then
          Put( "Processed" &
             Natural'Image(Nbr_Of_Files-Nbr_Of_Files_Success) & " of" &
             Natural'Image(Nbr_Of_Files) & " file(s) not successfully" );
          Put("," & Natural'Image( Nbr_Of_Errors ) & " error(s)");
       else
          Put( "Processed" & Natural'Image(Nbr_Of_Files) & " file(s) successfully" );
       end if;

       if Nbr_Of_Warnings > 0 then
          Put( "," & Natural'Image( Nbr_Of_Warnings ) & " warning(s)");
       end if;
       Put_Line( "" );
       Put_Line( "" );

    end Summary;

    ---------------
    -- Help_Text --
    ---------------
    procedure Help_Text is
    begin
      Put_Line( "" );
      Put_Line( "Usage: gesql [<options>] file(s) ");
      Put_Line("" );
      Put_Line( "options : " );
      Put_Line( "   -pedantic        Pedantic checking against ISO/92 SQL");
      Put_Line( "   -debugcode       Generate inline debug code" );
      Put_Line( "   -v               Print out processing information" );
      Put_Line( "   -s               Do not print the copyright notice" );
      Put_Line( "   -iso92           Conditions will sustain till next condition");
      Put_Line( "   -limit <nbr>     Max. number of errors" );
      Put_Line( "   -nosqlstate      Don't insert automatically SQLSTATE");
--      Put_Line( "   -odbc            Generate code for the ODBC interface");
      Put_Line( "   -schema <file>   Generate the schema into a file");
      Put_Line( "   -compiler <name> Define the target compiler as listed below:");
      Put_Line( "   -gnatnosref      Add no Source_Reference Pragma");
      Put_Line( "   -D<name>=<value> Define a substitution variable");
      Put     ( "                      " );
--      for I in Target_Compiler_Type loop
--         Put( " " & To_String( Supported_Compilers(I)) );
--      end loop;
      Put_Line( "" );
      Put_Line( "" );
      Put_Line( "   -h, --help       This message ");

      Put_Line( "" );
      Put_Line( "file(s) is any Ada 95 source file contining embedded SQL code. ");
      Put_Line( "" );
   end Help_Text;

   ----------
   -- Main --
   ----------
   procedure Main(
      Version : in String ) is
      -- Process the command line. Files and options are
      -- processed in the same order as they are specified.
      -- In case of fatal error in a file, the next files
      -- will be invoked for translation.
      Arg     : Natural     := 1;
      Result  : Exit_Status := Success;
      --
   begin
      if Argument_Count = 0 then
         Copyright( Version );
         Help_Text;
         Set_Exit_Status( Failure );
      end if;

      while Arg in 1..Argument_Count loop
         if Argument(Arg)(1) = '-' then
            Handle_Option_Flag(Argument(Arg), Arg);
            if Show_Help_Info then
               Copyright( Version );
               Help_Text;
               exit;
            end if;
         else
            declare
               F : File_Reader;
            begin
               Copyright( Version );

               if Option_Verbose then
                  Put_Line("Processing file " & Argument(Arg) & " : " );
               end if;

               if ( Index( Argument(Arg), ".adb" ) > 0 ) or
                  ( Index( Argument(Arg), ".ads" ) > 0 )
               then
                  Error( "Ada 95 specifications or body files are no input of gesql" );
                  Nbr_Of_Files := 1;
                  exit;
               end if;
               F := Open( Argument(Arg), Name(Argument(Arg)) & ".adb" );

               Nbr_Of_Files := Nbr_Of_Files + 1;

               Mark_Source_Line( F );
               Process_File( F, Result );

               Comment(F, "***************************************************");
               Comment(F, "        !!!  D O   N O T   E D I T   !!! ");
               Comment(F, " This file is gnerated by GNADE ESQL translator ");
               Comment(F, "" );
               Comment(F, " ESQL VERSION : " & Version );
               Comment(F, " SOURCE FILE  : " & Argument(Arg));
               Comment(F, " OPTIONS      : " );
               for I in 1..Argument_Count loop
                  Comment( F, "     " & Argument(I));
               end loop;
               Comment(F, "***************************************************");

               if Result /= Success then
                  Delete( F );
               else
                  Nbr_Of_Files_Success := Nbr_Of_Files_Success + 1;
                  Close(F);
               end if;

               Option_SILENT := True;

            exception
              when NAME_ERROR =>
                 Error("can't open input file : " & Argument(Arg));
                 Nbr_Of_Errors := Nbr_Of_Errors + 1;
           end;
           Arg := Arg + 1;
         end if;
      end loop;

      if Option_Schema then
         Close( Schema_File );
      end if;

      if Nbr_Of_Files > 0  then
         Summary;
      else
         if Nbr_Of_Errors = 0 and Argument_Count > 0 then
            Help_Text;
         end if;
      end if;

      if Nbr_Of_Errors > 0 then
         Result := Failure;
      end if;

      Set_Exit_Status( Result );
   end Main;

end Driver;


