-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/contrib/gsql/contents_manager.adb,v $
--  Description     : Handle the contents of the loaded document             --
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: me $
--  Last Modified On: $Date: 2001/10/03 19:56:03 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2000 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  The contents handler maintains on class level two lists                  --
--                                                                           --
--  * list of all document types                                             --
--  * list of all document components                                        --
--                                                                           --
--  This package provides additionaly the model for a so called contents     --
--  handler. A contents handler is an implementation which handels data      --
--  of one kind, e.g. all query definitions.                                 --
--                                                                           --
--  The package copies by means of the Initialize procedure elements of the  --
--  same class from the project file into the handler space.                 --
--                                                                           --
--  The method Finalize may be used to copy back elements of the same class  --
--  into the project file.                                                   --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;                    use Ada.Text_IO;
with Project_File;                   use Project_File;
with GSQL_Types;                     use GSQL_Types;

package body Contents_Manager is

   --- ********************************************************************* --
   --- ***              L O C A L   D A T A    TYPES                     *** --
   --- ********************************************************************* --

   -----------------
   -- Class Table --
   -----------------
   type Class_Record_Type is record
         Name        : Unbounded_String   := Null_Unbounded_String;
         Constructor : Constructor_Access := null;
      end record;

   CT : array( 1..50 ) of Class_Record_Type;

   ------------------
   -- Element List --
   ------------------
   type Element_Record_Type is  record
         Class_Name : Unbounded_String        := Null_Unbounded_String;
         Kind       : Contents_Handler.Handle := null;
         Parent     : Element_ID              := Null_Element_ID;
      end record;

   ST : array( 1..Element_ID'Last ) of Element_Record_Type;

   --- ********************************************************************* --
   --- ***              L O C A L   P R O C E D U R E S                  *** --
   --- ********************************************************************* --

   -------------
   -- Matches --
   -------------
   function Matches(
      I    : in Element_ID;
      Cls  : in Unbounded_String;
      Name : in Unbounded_String ) return Boolean is
      X    : Unbounded_String := Key(ST(I).Kind.all);
   begin
      return ST(I).Class_Name = Cls and Name = X ;
   end Matches;

   -----------------
   -- Instanciate --
   -----------------
   function Constructor(
      Class_Name : in String ) return Constructor_Access is
      -- instanciate an instance of the given class name
      Result     : Constructor_Access := null;
   begin
      for I in CT'Range loop
         if CT(I).Constructor /= null and then
            To_String(CT(I).Name) = Class_Name
         then
            Result := CT(I).Constructor;
            exit;
         end if;
      end loop;
      return Result;
   end Constructor;

   --- ********************************************************************* --
   --- ***             P U B L I C   P R O C E D U R E S                 *** --
   --- **                  C L A S S   M E T H O D S                     *** --
   --- ********************************************************************* --

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      Class_Name  : in String;
      Parent      : in Element_ID ) is
      -- read out the elements of a given class
      Cat         : Text_Buffer(1..Max_Catalog_Length);
      Item        : Handle;
      Id          : Element_ID;
      Instanciate : Constructor_Access  := Constructor( Class_Name );
   begin
      if Instanciate = null then
         return;
      end if;
      Project_File.Catalog( Class_Name, Cat );

      for I in Cat'Range loop
         if Cat(I) /= Null_Unbounded_String then
            Item := Instanciate( To_String(Cat(I)) );
            if Item = null then
               raise Instanciation_Error;
            end if;

            Contents_Handler.Key( Item.all,  Cat(I) );

            Id := Add_Contents_Element(Cat(I), item, Parent );
            Initialize( Item.all );
         end if;
      end loop;
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      Class_Name : in String := "" ) is
      Data       : Text_Buffer( 1..Max_Section_Length );
      Last       : Positive := 1;
   begin
      for I in ST'Range loop
         if ST(I).Kind /= null  and then
            To_String(ST(I).Class_Name) = Class_Name then
               Value( ST(I).Kind.all, Data, Last );

               Project_File.Update(
                  To_String(ST(I).Class_Name),               -- class name
                  Contents_Handler.Key( St(I).Kind.all ),    -- keyword
                  Data(1..Last)                              -- value
               );

         end if;
      end loop;
   end Finalize;

   --------------
   -- Register --
   --------------
   procedure Register(
      Class_Name  : in String;
      Constructor : in Constructor_Access ) is
      -- register a contents class name
   begin
      for I in CT'Range loop
         if CT(I).Constructor = null then
            CT(I).Constructor := Constructor;
            CT(I).Name        := To_Unbounded_String( Class_Name );
            return;
         end if;
      end loop;
   end Register;

   ------------
   -- Create --
   ------------
   function  Create(
      Class_Name  : in Unbounded_String;
      Name        : in Unbounded_String ) return Contents_Handler.Handle is
   begin
      return Create( To_String(Class_Name), To_String(Name) );
   end Create;

   ------------
   -- Create --
   ------------
   function  Create(
      Class_Name  : in String;
      Name        : in String ) return Contents_Handler.Handle is
      -- create an instance based upon the document type
      Result      : Contents_Handler.Handle;
   begin
      Result := Constructor(Class_Name)(Name);

      Contents_Handler.Key( Result.all, Name );
      Initialize( Result.all );
      return Result;
   end Create;

   --- ********************************************************************* --
   --- ***             P U B L I C   P R O C E D U R E S                 *** --
   --- **               I N S T A N C E   M E T H O D S                  *** --
   --- ********************************************************************* --

   ------------------
   -- Element_Name --
   ------------------
   function Element_Name(
      Id : in Element_ID ) return Unbounded_String is
   begin
      return To_Unbounded_String(Contents_Handler.Key(ST(Id).Kind.all));
   end Element_Name;

   --------------
   -- Contents --
   --------------
   function Contents(
      Id : in Element_ID ) return Contents_Handler.Handle is
   begin
      return ST(Id).Kind;
   end Contents;

   --------------
   -- Contents --
   --------------
   function Contents(
      Cls  : in Unbounded_String;
      Name : in Unbounded_String ) return Contents_Handler.Handle is
   begin
      for I in ST'Range loop
         if ST(I).Kind /= null and then Matches( Element_Id(I), Cls, Name )
         then
            return ST(I).Kind;
         end if;
      end loop;
      raise Element_Not_Found;
   end Contents;

   -------------
   -- Exisits --
   -------------
   function Get_Element_ID(
      Name   : in String;
      Parent : in Element_ID) return Element_ID is
   begin
      for I in ST'Range loop
         if ST(I).Kind /= null and then ST(I).Parent = Parent then
            declare
               S : constant Unbounded_String :=  Key( ST(I).Kind.all );
            begin
               if S = Name then
                  return I;
               end if;
            end;
         end if;
      end loop;

      return Null_Element_ID;
   end Get_Element_ID;

   --------------------------
   -- Add_Contents_Element --
   --------------------------
   function Add_Contents_Element(
      Name    : in Unbounded_String;
      Handler : in Contents_Handler.Handle;
      Parent  : in Element_ID := Null_Element_ID ) return Element_ID is
   begin
      Contents_Handler.Key( Handler.all, Name );

      for I in ST'Range loop
         if ST(I).Kind = null then
            ST(I).Kind   := Handler;
            ST(I).Parent := Parent;
            return I;
         end if;
      end loop;
      raise No_More_Elements;
   end Add_Contents_Element;

   -----------------------------
   -- Remove_Contents_Element --
   -----------------------------
   procedure Remove_Contents_Element(
      Id : in Element_ID ) is
      -- remove the element from the contents manager memory
   begin
      ST(Id).Kind := null;
   end Remove_Contents_Element;

   --------------
   -- Selected --
   --------------
   procedure Selected(
      Id : in Element_ID ) is
      -- to be called if the object has been selected
   begin
      Selected( ST(Id).Kind.all );
   end Selected;


end Contents_Manager;
