# Gmsh - Copyright (C) 1997-2019 C. Geuzaine, J.-F. Remacle
#
# See the LICENSE.txt file for license information. Please report all
# issues on https://gitlab.onelab.info/gmsh/gmsh/issues.

# This file defines the Gmsh Julia API (v4.4).
#
# Do not edit it directly: it is automatically generated by `api/gen.py'.
#
# By design, the Gmsh Julia API is purely functional, and only uses elementary
# types. See `demos/api' for examples.

"""
    module gmsh

Top-level functions
"""
module gmsh

const GMSH_API_VERSION = "4.4"
const GMSH_API_VERSION_MAJOR = 4
const GMSH_API_VERSION_MINOR = 4
const libdir = dirname(@__FILE__)
const libname = Sys.iswindows() ? "gmsh-4.4" : "libgmsh"
import Libdl
const lib = Libdl.find_library([libname], [libdir])

"""
    gmsh.initialize(argv = Vector{String}(), readConfigFiles = true)

Initialize Gmsh. This must be called before any call to the other functions in
the API. If `argc` and `argv` (or just `argv` in Python or Julia) are provided,
they will be handled in the same way as the command line arguments in the Gmsh
app. If `readConfigFiles` is set, read system Gmsh configuration files (gmshrc
and gmsh-options).
"""
function initialize(argv = Vector{String}(), readConfigFiles = true)
    ierr = Ref{Cint}()
    ccall((:gmshInitialize, lib), Cvoid,
          (Cint, Ptr{Ptr{Cchar}}, Cint, Ptr{Cint}),
          length(argv), argv, readConfigFiles, ierr)
    ierr[] != 0 && error("gmshInitialize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.finalize()

Finalize Gmsh. This must be called when you are done using the Gmsh API.
"""
function finalize()
    ierr = Ref{Cint}()
    ccall((:gmshFinalize, lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshFinalize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.open(fileName)

Open a file. Equivalent to the `File->Open` menu in the Gmsh app. Handling of
the file depends on its extension and/or its contents: opening a file with model
data will create a new model.
"""
function open(fileName)
    ierr = Ref{Cint}()
    ccall((:gmshOpen, lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          fileName, ierr)
    ierr[] != 0 && error("gmshOpen returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.merge(fileName)

Merge a file. Equivalent to the `File->Merge` menu in the Gmsh app. Handling of
the file depends on its extension and/or its contents. Merging a file with model
data will add the data to the current model.
"""
function merge(fileName)
    ierr = Ref{Cint}()
    ccall((:gmshMerge, lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          fileName, ierr)
    ierr[] != 0 && error("gmshMerge returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.write(fileName)

Write a file. The export format is determined by the file extension.
"""
function write(fileName)
    ierr = Ref{Cint}()
    ccall((:gmshWrite, lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          fileName, ierr)
    ierr[] != 0 && error("gmshWrite returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.clear()

Clear all loaded models and post-processing data, and add a new empty model.
"""
function clear()
    ierr = Ref{Cint}()
    ccall((:gmshClear, lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshClear returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    module gmsh.option

Option handling functions
"""
module option

import ..gmsh

"""
    gmsh.option.setNumber(name, value)

Set a numerical option to `value`. `name` is of the form "category.option" or
"category[num].option". Available categories and options are listed in the Gmsh
reference manual.
"""
function setNumber(name, value)
    ierr = Ref{Cint}()
    ccall((:gmshOptionSetNumber, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Cdouble, Ptr{Cint}),
          name, value, ierr)
    ierr[] != 0 && error("gmshOptionSetNumber returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.option.getNumber(name)

Get the `value` of a numerical option. `name` is of the form "category.option"
or "category[num].option". Available categories and options are listed in the
Gmsh reference manual.

Return `value`.
"""
function getNumber(name)
    api_value_ = Ref{Cdouble}()
    ierr = Ref{Cint}()
    ccall((:gmshOptionGetNumber, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cdouble}, Ptr{Cint}),
          name, api_value_, ierr)
    ierr[] != 0 && error("gmshOptionGetNumber returned non-zero error code: $(ierr[])")
    return api_value_[]
end

"""
    gmsh.option.setString(name, value)

Set a string option to `value`. `name` is of the form "category.option" or
"category[num].option". Available categories and options are listed in the Gmsh
reference manual.
"""
function setString(name, value)
    ierr = Ref{Cint}()
    ccall((:gmshOptionSetString, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cchar}, Ptr{Cint}),
          name, value, ierr)
    ierr[] != 0 && error("gmshOptionSetString returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.option.getString(name)

Get the `value` of a string option. `name` is of the form "category.option" or
"category[num].option". Available categories and options are listed in the Gmsh
reference manual.

Return `value`.
"""
function getString(name)
    api_value_ = Ref{Ptr{Cchar}}()
    ierr = Ref{Cint}()
    ccall((:gmshOptionGetString, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Ptr{Cchar}}, Ptr{Cint}),
          name, api_value_, ierr)
    ierr[] != 0 && error("gmshOptionGetString returned non-zero error code: $(ierr[])")
    value = unsafe_string(api_value_[])
    return value
end

"""
    gmsh.option.setColor(name, r, g, b, a = 0)

Set a color option to the RGBA value (`r`, `g`, `b`, `a`), where where `r`, `g`,
`b` and `a` should be integers between 0 and 255. `name` is of the form
"category.option" or "category[num].option". Available categories and options
are listed in the Gmsh reference manual, with the "Color." middle string
removed.
"""
function setColor(name, r, g, b, a = 0)
    ierr = Ref{Cint}()
    ccall((:gmshOptionSetColor, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Cint, Cint, Cint, Cint, Ptr{Cint}),
          name, r, g, b, a, ierr)
    ierr[] != 0 && error("gmshOptionSetColor returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.option.getColor(name)

Get the `r`, `g`, `b`, `a` value of a color option. `name` is of the form
"category.option" or "category[num].option". Available categories and options
are listed in the Gmsh reference manual, with the "Color." middle string
removed.

Return `r`, `g`, `b`, `a`.
"""
function getColor(name)
    api_r_ = Ref{Cint}()
    api_g_ = Ref{Cint}()
    api_b_ = Ref{Cint}()
    api_a_ = Ref{Cint}()
    ierr = Ref{Cint}()
    ccall((:gmshOptionGetColor, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}),
          name, api_r_, api_g_, api_b_, api_a_, ierr)
    ierr[] != 0 && error("gmshOptionGetColor returned non-zero error code: $(ierr[])")
    return api_r_[], api_g_[], api_b_[], api_a_[]
end

end # end of module option

"""
    module gmsh.model

Model functions
"""
module model

import ..gmsh

"""
    gmsh.model.add(name)

Add a new model, with name `name`, and set it as the current model.
"""
function add(name)
    ierr = Ref{Cint}()
    ccall((:gmshModelAdd, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          name, ierr)
    ierr[] != 0 && error("gmshModelAdd returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.remove()

Remove the current model.
"""
function remove()
    ierr = Ref{Cint}()
    ccall((:gmshModelRemove, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelRemove returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.list()

List the names of all models.

Return `names`.
"""
function list()
    api_names_ = Ref{Ptr{Ptr{Cchar}}}()
    api_names_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelList, gmsh.lib), Cvoid,
          (Ptr{Ptr{Ptr{Cchar}}}, Ptr{Csize_t}, Ptr{Cint}),
          api_names_, api_names_n_, ierr)
    ierr[] != 0 && error("gmshModelList returned non-zero error code: $(ierr[])")
    tmp_api_names_ = unsafe_wrap(Array, api_names_[], api_names_n_[], own=true)
    names = [unsafe_string(tmp_api_names_[i]) for i in 1:length(tmp_api_names_) ]
    return names
end

"""
    gmsh.model.setCurrent(name)

Set the current model to the model with name `name`. If several models have the
same name, select the one that was added first.
"""
function setCurrent(name)
    ierr = Ref{Cint}()
    ccall((:gmshModelSetCurrent, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          name, ierr)
    ierr[] != 0 && error("gmshModelSetCurrent returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.getEntities(dim = -1)

Get all the entities in the current model. If `dim` is >= 0, return only the
entities of the specified dimension (e.g. points if `dim` == 0). The entities
are returned as a vector of (dim, tag) integer pairs.

Return `dimTags`.
"""
function getEntities(dim = -1)
    api_dimTags_ = Ref{Ptr{Cint}}()
    api_dimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetEntities, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          api_dimTags_, api_dimTags_n_, dim, ierr)
    ierr[] != 0 && error("gmshModelGetEntities returned non-zero error code: $(ierr[])")
    tmp_api_dimTags_ = unsafe_wrap(Array, api_dimTags_[], api_dimTags_n_[], own=true)
    dimTags = [ (tmp_api_dimTags_[i], tmp_api_dimTags_[i+1]) for i in 1:2:length(tmp_api_dimTags_) ]
    return dimTags
end

"""
    gmsh.model.setEntityName(dim, tag, name)

Set the name of the entity of dimension `dim` and tag `tag`.
"""
function setEntityName(dim, tag, name)
    ierr = Ref{Cint}()
    ccall((:gmshModelSetEntityName, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cchar}, Ptr{Cint}),
          dim, tag, name, ierr)
    ierr[] != 0 && error("gmshModelSetEntityName returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.getEntityName(dim, tag)

Get the name of the entity of dimension `dim` and tag `tag`.

Return `name`.
"""
function getEntityName(dim, tag)
    api_name_ = Ref{Ptr{Cchar}}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetEntityName, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cchar}}, Ptr{Cint}),
          dim, tag, api_name_, ierr)
    ierr[] != 0 && error("gmshModelGetEntityName returned non-zero error code: $(ierr[])")
    name = unsafe_string(api_name_[])
    return name
end

"""
    gmsh.model.getPhysicalGroups(dim = -1)

Get all the physical groups in the current model. If `dim` is >= 0, return only
the entities of the specified dimension (e.g. physical points if `dim` == 0).
The entities are returned as a vector of (dim, tag) integer pairs.

Return `dimTags`.
"""
function getPhysicalGroups(dim = -1)
    api_dimTags_ = Ref{Ptr{Cint}}()
    api_dimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetPhysicalGroups, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          api_dimTags_, api_dimTags_n_, dim, ierr)
    ierr[] != 0 && error("gmshModelGetPhysicalGroups returned non-zero error code: $(ierr[])")
    tmp_api_dimTags_ = unsafe_wrap(Array, api_dimTags_[], api_dimTags_n_[], own=true)
    dimTags = [ (tmp_api_dimTags_[i], tmp_api_dimTags_[i+1]) for i in 1:2:length(tmp_api_dimTags_) ]
    return dimTags
end

"""
    gmsh.model.getEntitiesForPhysicalGroup(dim, tag)

Get the tags of the model entities making up the physical group of dimension
`dim` and tag `tag`.

Return `tags`.
"""
function getEntitiesForPhysicalGroup(dim, tag)
    api_tags_ = Ref{Ptr{Cint}}()
    api_tags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetEntitiesForPhysicalGroup, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, api_tags_, api_tags_n_, ierr)
    ierr[] != 0 && error("gmshModelGetEntitiesForPhysicalGroup returned non-zero error code: $(ierr[])")
    tags = unsafe_wrap(Array, api_tags_[], api_tags_n_[], own=true)
    return tags
end

"""
    gmsh.model.getPhysicalGroupsForEntity(dim, tag)

Get the tags of the physical groups (if any) to which the model entity of
dimension `dim` and tag `tag` belongs.

Return `physicalTags`.
"""
function getPhysicalGroupsForEntity(dim, tag)
    api_physicalTags_ = Ref{Ptr{Cint}}()
    api_physicalTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetPhysicalGroupsForEntity, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, api_physicalTags_, api_physicalTags_n_, ierr)
    ierr[] != 0 && error("gmshModelGetPhysicalGroupsForEntity returned non-zero error code: $(ierr[])")
    physicalTags = unsafe_wrap(Array, api_physicalTags_[], api_physicalTags_n_[], own=true)
    return physicalTags
end

"""
    gmsh.model.addPhysicalGroup(dim, tags, tag = -1)

Add a physical group of dimension `dim`, grouping the model entities with tags
`tags`. Return the tag of the physical group, equal to `tag` if `tag` is
positive, or a new tag if `tag` < 0.

Return an integer value.
"""
function addPhysicalGroup(dim, tags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelAddPhysicalGroup, gmsh.lib), Cint,
          (Cint, Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          dim, convert(Vector{Cint}, tags), length(tags), tag, ierr)
    ierr[] != 0 && error("gmshModelAddPhysicalGroup returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.setPhysicalName(dim, tag, name)

Set the name of the physical group of dimension `dim` and tag `tag`.
"""
function setPhysicalName(dim, tag, name)
    ierr = Ref{Cint}()
    ccall((:gmshModelSetPhysicalName, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cchar}, Ptr{Cint}),
          dim, tag, name, ierr)
    ierr[] != 0 && error("gmshModelSetPhysicalName returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.getPhysicalName(dim, tag)

Get the name of the physical group of dimension `dim` and tag `tag`.

Return `name`.
"""
function getPhysicalName(dim, tag)
    api_name_ = Ref{Ptr{Cchar}}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetPhysicalName, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cchar}}, Ptr{Cint}),
          dim, tag, api_name_, ierr)
    ierr[] != 0 && error("gmshModelGetPhysicalName returned non-zero error code: $(ierr[])")
    name = unsafe_string(api_name_[])
    return name
end

"""
    gmsh.model.getBoundary(dimTags, combined = true, oriented = true, recursive = false)

Get the boundary of the model entities `dimTags`. Return in `outDimTags` the
boundary of the individual entities (if `combined` is false) or the boundary of
the combined geometrical shape formed by all input entities (if `combined` is
true). Return tags multiplied by the sign of the boundary entity if `oriented`
is true. Apply the boundary operator recursively down to dimension 0 (i.e. to
points) if `recursive` is true.

Return `outDimTags`.
"""
function getBoundary(dimTags, combined = true, oriented = true, recursive = false)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetBoundary, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), api_outDimTags_, api_outDimTags_n_, combined, oriented, recursive, ierr)
    ierr[] != 0 && error("gmshModelGetBoundary returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim = -1)

Get the model entities in the bounding box defined by the two points (`xmin`,
`ymin`, `zmin`) and (`xmax`, `ymax`, `zmax`). If `dim` is >= 0, return only the
entities of the specified dimension (e.g. points if `dim` == 0).

Return `tags`.
"""
function getEntitiesInBoundingBox(xmin, ymin, zmin, xmax, ymax, zmax, dim = -1)
    api_tags_ = Ref{Ptr{Cint}}()
    api_tags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetEntitiesInBoundingBox, gmsh.lib), Cvoid,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          xmin, ymin, zmin, xmax, ymax, zmax, api_tags_, api_tags_n_, dim, ierr)
    ierr[] != 0 && error("gmshModelGetEntitiesInBoundingBox returned non-zero error code: $(ierr[])")
    tmp_api_tags_ = unsafe_wrap(Array, api_tags_[], api_tags_n_[], own=true)
    tags = [ (tmp_api_tags_[i], tmp_api_tags_[i+1]) for i in 1:2:length(tmp_api_tags_) ]
    return tags
end

"""
    gmsh.model.getBoundingBox(dim, tag)

Get the bounding box (`xmin`, `ymin`, `zmin`), (`xmax`, `ymax`, `zmax`) of the
model entity of dimension `dim` and tag `tag`. If `dim` and `tag` are negative,
get the bounding box of the whole model.

Return `xmin`, `ymin`, `zmin`, `xmax`, `ymax`, `zmax`.
"""
function getBoundingBox(dim, tag)
    api_xmin_ = Ref{Cdouble}()
    api_ymin_ = Ref{Cdouble}()
    api_zmin_ = Ref{Cdouble}()
    api_xmax_ = Ref{Cdouble}()
    api_ymax_ = Ref{Cdouble}()
    api_zmax_ = Ref{Cdouble}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetBoundingBox, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cint}),
          dim, tag, api_xmin_, api_ymin_, api_zmin_, api_xmax_, api_ymax_, api_zmax_, ierr)
    ierr[] != 0 && error("gmshModelGetBoundingBox returned non-zero error code: $(ierr[])")
    return api_xmin_[], api_ymin_[], api_zmin_[], api_xmax_[], api_ymax_[], api_zmax_[]
end

"""
    gmsh.model.getDimension()

Get the geometrical dimension of the current model.

Return an integer value.
"""
function getDimension()
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGetDimension, gmsh.lib), Cint,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelGetDimension returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.addDiscreteEntity(dim, tag = -1, boundary = Cint[])

Add a discrete model entity (defined by a mesh) of dimension `dim` in the
current model. Return the tag of the new discrete entity, equal to `tag` if
`tag` is positive, or a new tag if `tag` < 0. `boundary` specifies the tags of
the entities on the boundary of the discrete entity, if any. Specifying
`boundary` allows Gmsh to construct the topology of the overall model.

Return an integer value.
"""
function addDiscreteEntity(dim, tag = -1, boundary = Cint[])
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelAddDiscreteEntity, gmsh.lib), Cint,
          (Cint, Cint, Ptr{Cint}, Csize_t, Ptr{Cint}),
          dim, tag, convert(Vector{Cint}, boundary), length(boundary), ierr)
    ierr[] != 0 && error("gmshModelAddDiscreteEntity returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.removeEntities(dimTags, recursive = false)

Remove the entities `dimTags` of the current model. If `recursive` is true,
remove all the entities on their boundaries, down to dimension 0.
"""
function removeEntities(dimTags, recursive = false)
    ierr = Ref{Cint}()
    ccall((:gmshModelRemoveEntities, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), recursive, ierr)
    ierr[] != 0 && error("gmshModelRemoveEntities returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.removeEntityName(name)

Remove the entity name `name` from the current model.
"""
function removeEntityName(name)
    ierr = Ref{Cint}()
    ccall((:gmshModelRemoveEntityName, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          name, ierr)
    ierr[] != 0 && error("gmshModelRemoveEntityName returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.removePhysicalGroups(dimTags = Tuple{Cint,Cint}[])

Remove the physical groups `dimTags` of the current model. If `dimTags` is
empty, remove all groups.
"""
function removePhysicalGroups(dimTags = Tuple{Cint,Cint}[])
    ierr = Ref{Cint}()
    ccall((:gmshModelRemovePhysicalGroups, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), ierr)
    ierr[] != 0 && error("gmshModelRemovePhysicalGroups returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.removePhysicalName(name)

Remove the physical name `name` from the current model.
"""
function removePhysicalName(name)
    ierr = Ref{Cint}()
    ccall((:gmshModelRemovePhysicalName, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          name, ierr)
    ierr[] != 0 && error("gmshModelRemovePhysicalName returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.getType(dim, tag)

Get the type of the entity of dimension `dim` and tag `tag`.

Return `entityType`.
"""
function getType(dim, tag)
    api_entityType_ = Ref{Ptr{Cchar}}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetType, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cchar}}, Ptr{Cint}),
          dim, tag, api_entityType_, ierr)
    ierr[] != 0 && error("gmshModelGetType returned non-zero error code: $(ierr[])")
    entityType = unsafe_string(api_entityType_[])
    return entityType
end

"""
    gmsh.model.getParent(dim, tag)

In a partitioned model, get the parent of the entity of dimension `dim` and tag
`tag`, i.e. from which the entity is a part of, if any. `parentDim` and
`parentTag` are set to -1 if the entity has no parent.

Return `parentDim`, `parentTag`.
"""
function getParent(dim, tag)
    api_parentDim_ = Ref{Cint}()
    api_parentTag_ = Ref{Cint}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetParent, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}),
          dim, tag, api_parentDim_, api_parentTag_, ierr)
    ierr[] != 0 && error("gmshModelGetParent returned non-zero error code: $(ierr[])")
    return api_parentDim_[], api_parentTag_[]
end

"""
    gmsh.model.getPartitions(dim, tag)

In a partitioned model, return the tags of the partition(s) to which the entity
belongs.

Return `partitions`.
"""
function getPartitions(dim, tag)
    api_partitions_ = Ref{Ptr{Cint}}()
    api_partitions_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetPartitions, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, api_partitions_, api_partitions_n_, ierr)
    ierr[] != 0 && error("gmshModelGetPartitions returned non-zero error code: $(ierr[])")
    partitions = unsafe_wrap(Array, api_partitions_[], api_partitions_n_[], own=true)
    return partitions
end

"""
    gmsh.model.getValue(dim, tag, parametricCoord)

Evaluate the parametrization of the entity of dimension `dim` and tag `tag` at
the parametric coordinates `parametricCoord`. Only valid for `dim` equal to 0
(with empty `parametricCoord`), 1 (with `parametricCoord` containing parametric
coordinates on the curve) or 2 (with `parametricCoord` containing pairs of u, v
parametric coordinates on the surface, concatenated: [p1u, p1v, p2u, ...]).
Return triplets of x, y, z coordinates in `points`, concatenated: [p1x, p1y,
p1z, p2x, ...].

Return `points`.
"""
function getValue(dim, tag, parametricCoord)
    api_points_ = Ref{Ptr{Cdouble}}()
    api_points_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetValue, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, convert(Vector{Cdouble}, parametricCoord), length(parametricCoord), api_points_, api_points_n_, ierr)
    ierr[] != 0 && error("gmshModelGetValue returned non-zero error code: $(ierr[])")
    points = unsafe_wrap(Array, api_points_[], api_points_n_[], own=true)
    return points
end

"""
    gmsh.model.getDerivative(dim, tag, parametricCoord)

Evaluate the derivative of the parametrization of the entity of dimension `dim`
and tag `tag` at the parametric coordinates `parametricCoord`. Only valid for
`dim` equal to 1 (with `parametricCoord` containing parametric coordinates on
the curve) or 2 (with `parametricCoord` containing pairs of u, v parametric
coordinates on the surface, concatenated: [p1u, p1v, p2u, ...]). For `dim` equal
to 1 return the x, y, z components of the derivative with respect to u [d1ux,
d1uy, d1uz, d2ux, ...]; for `dim` equal to 2 return the x, y, z components of
the derivate with respect to u and v: [d1ux, d1uy, d1uz, d1vx, d1vy, d1vz, d2ux,
...].

Return `derivatives`.
"""
function getDerivative(dim, tag, parametricCoord)
    api_derivatives_ = Ref{Ptr{Cdouble}}()
    api_derivatives_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetDerivative, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, convert(Vector{Cdouble}, parametricCoord), length(parametricCoord), api_derivatives_, api_derivatives_n_, ierr)
    ierr[] != 0 && error("gmshModelGetDerivative returned non-zero error code: $(ierr[])")
    derivatives = unsafe_wrap(Array, api_derivatives_[], api_derivatives_n_[], own=true)
    return derivatives
end

"""
    gmsh.model.getCurvature(dim, tag, parametricCoord)

Evaluate the (maximum) curvature of the entity of dimension `dim` and tag `tag`
at the parametric coordinates `parametricCoord`. Only valid for `dim` equal to 1
(with `parametricCoord` containing parametric coordinates on the curve) or 2
(with `parametricCoord` containing pairs of u, v parametric coordinates on the
surface, concatenated: [p1u, p1v, p2u, ...]).

Return `curvatures`.
"""
function getCurvature(dim, tag, parametricCoord)
    api_curvatures_ = Ref{Ptr{Cdouble}}()
    api_curvatures_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetCurvature, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, convert(Vector{Cdouble}, parametricCoord), length(parametricCoord), api_curvatures_, api_curvatures_n_, ierr)
    ierr[] != 0 && error("gmshModelGetCurvature returned non-zero error code: $(ierr[])")
    curvatures = unsafe_wrap(Array, api_curvatures_[], api_curvatures_n_[], own=true)
    return curvatures
end

"""
    gmsh.model.getPrincipalCurvatures(tag, parametricCoord)

Evaluate the principal curvatures of the surface with tag `tag` at the
parametric coordinates `parametricCoord`, as well as their respective
directions. `parametricCoord` are given by pair of u and v coordinates,
concatenated: [p1u, p1v, p2u, ...].

Return `curvatureMax`, `curvatureMin`, `directionMax`, `directionMin`.
"""
function getPrincipalCurvatures(tag, parametricCoord)
    api_curvatureMax_ = Ref{Ptr{Cdouble}}()
    api_curvatureMax_n_ = Ref{Csize_t}()
    api_curvatureMin_ = Ref{Ptr{Cdouble}}()
    api_curvatureMin_n_ = Ref{Csize_t}()
    api_directionMax_ = Ref{Ptr{Cdouble}}()
    api_directionMax_n_ = Ref{Csize_t}()
    api_directionMin_ = Ref{Ptr{Cdouble}}()
    api_directionMin_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetPrincipalCurvatures, gmsh.lib), Cvoid,
          (Cint, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          tag, convert(Vector{Cdouble}, parametricCoord), length(parametricCoord), api_curvatureMax_, api_curvatureMax_n_, api_curvatureMin_, api_curvatureMin_n_, api_directionMax_, api_directionMax_n_, api_directionMin_, api_directionMin_n_, ierr)
    ierr[] != 0 && error("gmshModelGetPrincipalCurvatures returned non-zero error code: $(ierr[])")
    curvatureMax = unsafe_wrap(Array, api_curvatureMax_[], api_curvatureMax_n_[], own=true)
    curvatureMin = unsafe_wrap(Array, api_curvatureMin_[], api_curvatureMin_n_[], own=true)
    directionMax = unsafe_wrap(Array, api_directionMax_[], api_directionMax_n_[], own=true)
    directionMin = unsafe_wrap(Array, api_directionMin_[], api_directionMin_n_[], own=true)
    return curvatureMax, curvatureMin, directionMax, directionMin
end

"""
    gmsh.model.getNormal(tag, parametricCoord)

Get the normal to the surface with tag `tag` at the parametric coordinates
`parametricCoord`. `parametricCoord` are given by pairs of u and v coordinates,
concatenated: [p1u, p1v, p2u, ...]. `normals` are returned as triplets of x, y,
z components, concatenated: [n1x, n1y, n1z, n2x, ...].

Return `normals`.
"""
function getNormal(tag, parametricCoord)
    api_normals_ = Ref{Ptr{Cdouble}}()
    api_normals_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetNormal, gmsh.lib), Cvoid,
          (Cint, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          tag, convert(Vector{Cdouble}, parametricCoord), length(parametricCoord), api_normals_, api_normals_n_, ierr)
    ierr[] != 0 && error("gmshModelGetNormal returned non-zero error code: $(ierr[])")
    normals = unsafe_wrap(Array, api_normals_[], api_normals_n_[], own=true)
    return normals
end

"""
    gmsh.model.setVisibility(dimTags, value, recursive = false)

Set the visibility of the model entities `dimTags` to `value`. Apply the
visibility setting recursively if `recursive` is true.
"""
function setVisibility(dimTags, value, recursive = false)
    ierr = Ref{Cint}()
    ccall((:gmshModelSetVisibility, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), value, recursive, ierr)
    ierr[] != 0 && error("gmshModelSetVisibility returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.getVisibility(dim, tag)

Get the visibility of the model entity of dimension `dim` and tag `tag`.

Return `value`.
"""
function getVisibility(dim, tag)
    api_value_ = Ref{Cint}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetVisibility, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}, Ptr{Cint}),
          dim, tag, api_value_, ierr)
    ierr[] != 0 && error("gmshModelGetVisibility returned non-zero error code: $(ierr[])")
    return api_value_[]
end

"""
    gmsh.model.setColor(dimTags, r, g, b, a = 0, recursive = false)

Set the color of the model entities `dimTags` to the RGBA value (`r`, `g`, `b`,
`a`), where `r`, `g`, `b` and `a` should be integers between 0 and 255. Apply
the color setting recursively if `recursive` is true.
"""
function setColor(dimTags, r, g, b, a = 0, recursive = false)
    ierr = Ref{Cint}()
    ccall((:gmshModelSetColor, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Cint, Cint, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), r, g, b, a, recursive, ierr)
    ierr[] != 0 && error("gmshModelSetColor returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.getColor(dim, tag)

Get the color of the model entity of dimension `dim` and tag `tag`.

Return `r`, `g`, `b`, `a`.
"""
function getColor(dim, tag)
    api_r_ = Ref{Cint}()
    api_g_ = Ref{Cint}()
    api_b_ = Ref{Cint}()
    api_a_ = Ref{Cint}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGetColor, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}),
          dim, tag, api_r_, api_g_, api_b_, api_a_, ierr)
    ierr[] != 0 && error("gmshModelGetColor returned non-zero error code: $(ierr[])")
    return api_r_[], api_g_[], api_b_[], api_a_[]
end

"""
    gmsh.model.setCoordinates(tag, x, y, z)

Set the `x`, `y`, `z` coordinates of a geometrical point.
"""
function setCoordinates(tag, x, y, z)
    ierr = Ref{Cint}()
    ccall((:gmshModelSetCoordinates, gmsh.lib), Cvoid,
          (Cint, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          tag, x, y, z, ierr)
    ierr[] != 0 && error("gmshModelSetCoordinates returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    module gmsh.model.mesh

Mesh functions
"""
module mesh

import ...gmsh

"""
    gmsh.model.mesh.generate(dim = 3)

Generate a mesh of the current model, up to dimension `dim` (0, 1, 2 or 3).
"""
function generate(dim = 3)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGenerate, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          dim, ierr)
    ierr[] != 0 && error("gmshModelMeshGenerate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.partition(numPart)

Partition the mesh of the current model into `numPart` partitions.
"""
function partition(numPart)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshPartition, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          numPart, ierr)
    ierr[] != 0 && error("gmshModelMeshPartition returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.unpartition()

Unpartition the mesh of the current model.
"""
function unpartition()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshUnpartition, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshUnpartition returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.optimize(method, force = false)

Optimize the mesh of the current model using `method` (empty for default
tetrahedral mesh optimizer, "Netgen" for Netgen optimizer, "HighOrder" for
direct high-order mesh optimizer, "HighOrderElastic" for high-order elastic
smoother). If `force` is set apply the optimization also to discrete entities.
"""
function optimize(method, force = false)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshOptimize, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Cint, Ptr{Cint}),
          method, force, ierr)
    ierr[] != 0 && error("gmshModelMeshOptimize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.recombine()

Recombine the mesh of the current model.
"""
function recombine()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRecombine, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshRecombine returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.refine()

Refine the mesh of the current model by uniformly splitting the elements.
"""
function refine()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRefine, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshRefine returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.smooth()

Smooth the mesh of the current model.
"""
function smooth()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSmooth, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshSmooth returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setOrder(order)

Set the order of the elements in the mesh of the current model to `order`.
"""
function setOrder(order)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetOrder, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          order, ierr)
    ierr[] != 0 && error("gmshModelMeshSetOrder returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.getLastEntityError()

Get the last entities (if any) where a meshing error occurred. Currently only
populated by the new 3D meshing algorithms.

Return `dimTags`.
"""
function getLastEntityError()
    api_dimTags_ = Ref{Ptr{Cint}}()
    api_dimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetLastEntityError, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          api_dimTags_, api_dimTags_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetLastEntityError returned non-zero error code: $(ierr[])")
    tmp_api_dimTags_ = unsafe_wrap(Array, api_dimTags_[], api_dimTags_n_[], own=true)
    dimTags = [ (tmp_api_dimTags_[i], tmp_api_dimTags_[i+1]) for i in 1:2:length(tmp_api_dimTags_) ]
    return dimTags
end

"""
    gmsh.model.mesh.getLastNodeError()

Get the last nodes (if any) where a meshing error occurred. Currently only
populated by the new 3D meshing algorithms.

Return `nodeTags`.
"""
function getLastNodeError()
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetLastNodeError, gmsh.lib), Cvoid,
          (Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Cint}),
          api_nodeTags_, api_nodeTags_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetLastNodeError returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    return nodeTags
end

"""
    gmsh.model.mesh.clear()

Clear the mesh, i.e. delete all the nodes and elements.
"""
function clear()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshClear, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshClear returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.getNodes(dim = -1, tag = -1, includeBoundary = false, returnParametricCoord = true)

Get the nodes classified on the entity of dimension `dim` and tag `tag`. If
`tag` < 0, get the nodes for all entities of dimension `dim`. If `dim` and `tag`
are negative, get all the nodes in the mesh. `nodeTags` contains the node tags
(their unique, strictly positive identification numbers). `coord` is a vector of
length 3 times the length of `nodeTags` that contains the x, y, z coordinates of
the nodes, concatenated: [n1x, n1y, n1z, n2x, ...]. If `dim` >= 0 and
`returnParamtricCoord` is set, `parametricCoord` contains the parametric
coordinates ([u1, u2, ...] or [u1, v1, u2, ...]) of the nodes, if available. The
length of `parametricCoord` can be 0 or `dim` times the length of `nodeTags`. If
`includeBoundary` is set, also return the nodes classified on the boundary of
the entity (which will be reparametrized on the entity if `dim` >= 0 in order to
compute their parametric coordinates).

Return `nodeTags`, `coord`, `parametricCoord`.
"""
function getNodes(dim = -1, tag = -1, includeBoundary = false, returnParametricCoord = true)
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    api_coord_ = Ref{Ptr{Cdouble}}()
    api_coord_n_ = Ref{Csize_t}()
    api_parametricCoord_ = Ref{Ptr{Cdouble}}()
    api_parametricCoord_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetNodes, gmsh.lib), Cvoid,
          (Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Cint, Cint, Cint, Cint, Ptr{Cint}),
          api_nodeTags_, api_nodeTags_n_, api_coord_, api_coord_n_, api_parametricCoord_, api_parametricCoord_n_, dim, tag, includeBoundary, returnParametricCoord, ierr)
    ierr[] != 0 && error("gmshModelMeshGetNodes returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    coord = unsafe_wrap(Array, api_coord_[], api_coord_n_[], own=true)
    parametricCoord = unsafe_wrap(Array, api_parametricCoord_[], api_parametricCoord_n_[], own=true)
    return nodeTags, coord, parametricCoord
end

"""
    gmsh.model.mesh.getNodesByElementType(elementType, tag = -1, returnParametricCoord = true)

Get the nodes classified on the entity of tag `tag`, for all the elements of
type `elementType`. The other arguments are treated as in `getNodes`.

Return `nodeTags`, `coord`, `parametricCoord`.
"""
function getNodesByElementType(elementType, tag = -1, returnParametricCoord = true)
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    api_coord_ = Ref{Ptr{Cdouble}}()
    api_coord_n_ = Ref{Csize_t}()
    api_parametricCoord_ = Ref{Ptr{Cdouble}}()
    api_parametricCoord_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetNodesByElementType, gmsh.lib), Cvoid,
          (Cint, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Cint, Cint, Ptr{Cint}),
          elementType, api_nodeTags_, api_nodeTags_n_, api_coord_, api_coord_n_, api_parametricCoord_, api_parametricCoord_n_, tag, returnParametricCoord, ierr)
    ierr[] != 0 && error("gmshModelMeshGetNodesByElementType returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    coord = unsafe_wrap(Array, api_coord_[], api_coord_n_[], own=true)
    parametricCoord = unsafe_wrap(Array, api_parametricCoord_[], api_parametricCoord_n_[], own=true)
    return nodeTags, coord, parametricCoord
end

"""
    gmsh.model.mesh.getNode(nodeTag)

Get the coordinates and the parametric coordinates (if any) of the node with tag
`tag`. This is a sometimes useful but inefficient way of accessing nodes, as it
relies on a cache stored in the model. For large meshes all the nodes in the
model should be numbered in a continuous sequence of tags from 1 to N to
maintain reasonable performance (in this case the internal cache is based on a
vector; otherwise it uses a map).

Return `coord`, `parametricCoord`.
"""
function getNode(nodeTag)
    api_coord_ = Ref{Ptr{Cdouble}}()
    api_coord_n_ = Ref{Csize_t}()
    api_parametricCoord_ = Ref{Ptr{Cdouble}}()
    api_parametricCoord_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetNode, gmsh.lib), Cvoid,
          (Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          nodeTag, api_coord_, api_coord_n_, api_parametricCoord_, api_parametricCoord_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetNode returned non-zero error code: $(ierr[])")
    coord = unsafe_wrap(Array, api_coord_[], api_coord_n_[], own=true)
    parametricCoord = unsafe_wrap(Array, api_parametricCoord_[], api_parametricCoord_n_[], own=true)
    return coord, parametricCoord
end

"""
    gmsh.model.mesh.rebuildNodeCache(onlyIfNecessary = true)

Rebuild the node cache.
"""
function rebuildNodeCache(onlyIfNecessary = true)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRebuildNodeCache, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          onlyIfNecessary, ierr)
    ierr[] != 0 && error("gmshModelMeshRebuildNodeCache returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.getNodesForPhysicalGroup(dim, tag)

Get the nodes from all the elements belonging to the physical group of dimension
`dim` and tag `tag`. `nodeTags` contains the node tags; `coord` is a vector of
length 3 times the length of `nodeTags` that contains the x, y, z coordinates of
the nodes, concatenated: [n1x, n1y, n1z, n2x, ...].

Return `nodeTags`, `coord`.
"""
function getNodesForPhysicalGroup(dim, tag)
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    api_coord_ = Ref{Ptr{Cdouble}}()
    api_coord_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetNodesForPhysicalGroup, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, api_nodeTags_, api_nodeTags_n_, api_coord_, api_coord_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetNodesForPhysicalGroup returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    coord = unsafe_wrap(Array, api_coord_[], api_coord_n_[], own=true)
    return nodeTags, coord
end

"""
    gmsh.model.mesh.addNodes(dim, tag, nodeTags, coord, parametricCoord = Cdouble[])

Add nodes classified on the model entity of dimension `dim` and tag `tag`.
`nodeTags` contains the node tags (their unique, strictly positive
identification numbers). `coord` is a vector of length 3 times the length of
`nodeTags` that contains the x, y, z coordinates of the nodes, concatenated:
[n1x, n1y, n1z, n2x, ...]. The optional `parametricCoord` vector contains the
parametric coordinates of the nodes, if any. The length of `parametricCoord` can
be 0 or `dim` times the length of `nodeTags`. If the `nodeTags` vector is empty,
new tags are automatically assigned to the nodes.
"""
function addNodes(dim, tag, nodeTags, coord, parametricCoord = Cdouble[])
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshAddNodes, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Csize_t}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Cint}),
          dim, tag, convert(Vector{Csize_t}, nodeTags), length(nodeTags), convert(Vector{Cdouble}, coord), length(coord), convert(Vector{Cdouble}, parametricCoord), length(parametricCoord), ierr)
    ierr[] != 0 && error("gmshModelMeshAddNodes returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.reclassifyNodes()

Reclassify all nodes on their associated model entity, based on the elements.
Can be used when importing nodes in bulk (e.g. by associating them all to a
single volume), to reclassify them correctly on model surfaces, curves, etc.
after the elements have been set.
"""
function reclassifyNodes()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshReclassifyNodes, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshReclassifyNodes returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.relocateNodes(dim = -1, tag = -1)

Relocate the nodes classified on the entity of dimension `dim` and tag `tag`
using their parametric coordinates. If `tag` < 0, relocate the nodes for all
entities of dimension `dim`. If `dim` and `tag` are negative, relocate all the
nodes in the mesh.
"""
function relocateNodes(dim = -1, tag = -1)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRelocateNodes, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}),
          dim, tag, ierr)
    ierr[] != 0 && error("gmshModelMeshRelocateNodes returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.getElements(dim = -1, tag = -1)

Get the elements classified on the entity of dimension `dim` and tag `tag`. If
`tag` < 0, get the elements for all entities of dimension `dim`. If `dim` and
`tag` are negative, get all the elements in the mesh. `elementTypes` contains
the MSH types of the elements (e.g. `2` for 3-node triangles: see
`getElementProperties` to obtain the properties for a given element type).
`elementTags` is a vector of the same length as `elementTypes`; each entry is a
vector containing the tags (unique, strictly positive identifiers) of the
elements of the corresponding type. `nodeTags` is also a vector of the same
length as `elementTypes`; each entry is a vector of length equal to the number
of elements of the given type times the number N of nodes for this type of
element, that contains the node tags of all the elements of the given type,
concatenated: [e1n1, e1n2, ..., e1nN, e2n1, ...].

Return `elementTypes`, `elementTags`, `nodeTags`.
"""
function getElements(dim = -1, tag = -1)
    api_elementTypes_ = Ref{Ptr{Cint}}()
    api_elementTypes_n_ = Ref{Csize_t}()
    api_elementTags_ = Ref{Ptr{Ptr{Csize_t}}}()
    api_elementTags_n_ = Ref{Ptr{Csize_t}}()
    api_elementTags_nn_ = Ref{Csize_t}()
    api_nodeTags_ = Ref{Ptr{Ptr{Csize_t}}}()
    api_nodeTags_n_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_nn_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElements, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Csize_t}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Csize_t}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Cint, Ptr{Cint}),
          api_elementTypes_, api_elementTypes_n_, api_elementTags_, api_elementTags_n_, api_elementTags_nn_, api_nodeTags_, api_nodeTags_n_, api_nodeTags_nn_, dim, tag, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElements returned non-zero error code: $(ierr[])")
    elementTypes = unsafe_wrap(Array, api_elementTypes_[], api_elementTypes_n_[], own=true)
    tmp_api_elementTags_ = unsafe_wrap(Array, api_elementTags_[], api_elementTags_nn_[], own=true)
    tmp_api_elementTags_n_ = unsafe_wrap(Array, api_elementTags_n_[], api_elementTags_nn_[], own=true)
    elementTags = [ unsafe_wrap(Array, tmp_api_elementTags_[i], tmp_api_elementTags_n_[i], own=true) for i in 1:api_elementTags_nn_[] ]
    tmp_api_nodeTags_ = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_nn_[], own=true)
    tmp_api_nodeTags_n_ = unsafe_wrap(Array, api_nodeTags_n_[], api_nodeTags_nn_[], own=true)
    nodeTags = [ unsafe_wrap(Array, tmp_api_nodeTags_[i], tmp_api_nodeTags_n_[i], own=true) for i in 1:api_nodeTags_nn_[] ]
    return elementTypes, elementTags, nodeTags
end

"""
    gmsh.model.mesh.getElement(elementTag)

Get the type and node tags of the element with tag `tag`. This is a sometimes
useful but inefficient way of accessing elements, as it relies on a cache stored
in the model. For large meshes all the elements in the model should be numbered
in a continuous sequence of tags from 1 to N to maintain reasonable performance
(in this case the internal cache is based on a vector; otherwise it uses a map).

Return `elementType`, `nodeTags`.
"""
function getElement(elementTag)
    api_elementType_ = Ref{Cint}()
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElement, gmsh.lib), Cvoid,
          (Csize_t, Ptr{Cint}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Cint}),
          elementTag, api_elementType_, api_nodeTags_, api_nodeTags_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElement returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    return api_elementType_[], nodeTags
end

"""
    gmsh.model.mesh.getElementByCoordinates(x, y, z, dim = -1, strict = false)

Search the mesh for an element located at coordinates (`x`, `y`, `z`). This is a
sometimes useful but inefficient way of accessing elements, as it relies on a
search in a spatial octree. If an element is found, return its tag, type and
node tags, as well as the local coordinates (`u`, `v`, `w`) within the element
corresponding to search location. If `dim` is >= 0, only search for elements of
the given dimension. If `strict` is not set, use a tolerance to find elements
near the search location.

Return `elementTag`, `elementType`, `nodeTags`, `u`, `v`, `w`.
"""
function getElementByCoordinates(x, y, z, dim = -1, strict = false)
    api_elementTag_ = Ref{Csize_t}()
    api_elementType_ = Ref{Cint}()
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    api_u_ = Ref{Cdouble}()
    api_v_ = Ref{Cdouble}()
    api_w_ = Ref{Cdouble}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElementByCoordinates, gmsh.lib), Cvoid,
          (Cdouble, Cdouble, Cdouble, Ptr{Csize_t}, Ptr{Cint}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cdouble}, Cint, Cint, Ptr{Cint}),
          x, y, z, api_elementTag_, api_elementType_, api_nodeTags_, api_nodeTags_n_, api_u_, api_v_, api_w_, dim, strict, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElementByCoordinates returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    return api_elementTag_[], api_elementType_[], nodeTags, api_u_[], api_v_[], api_w_[]
end

"""
    gmsh.model.mesh.getElementTypes(dim = -1, tag = -1)

Get the types of elements in the entity of dimension `dim` and tag `tag`. If
`tag` < 0, get the types for all entities of dimension `dim`. If `dim` and `tag`
are negative, get all the types in the mesh.

Return `elementTypes`.
"""
function getElementTypes(dim = -1, tag = -1)
    api_elementTypes_ = Ref{Ptr{Cint}}()
    api_elementTypes_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElementTypes, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Cint, Ptr{Cint}),
          api_elementTypes_, api_elementTypes_n_, dim, tag, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElementTypes returned non-zero error code: $(ierr[])")
    elementTypes = unsafe_wrap(Array, api_elementTypes_[], api_elementTypes_n_[], own=true)
    return elementTypes
end

"""
    gmsh.model.mesh.getElementType(familyName, order, serendip = false)

Return an element type given its family name `familyName` ("point", "line",
"triangle", "quadrangle", "tetrahedron", "pyramid", "prism", "hexahedron") and
polynomial order `order`. If `serendip` is true, return the corresponding
serendip element type (element without interior nodes).

Return an integer value.
"""
function getElementType(familyName, order, serendip = false)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelMeshGetElementType, gmsh.lib), Cint,
          (Ptr{Cchar}, Cint, Cint, Ptr{Cint}),
          familyName, order, serendip, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElementType returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.mesh.getElementProperties(elementType)

Get the properties of an element of type `elementType`: its name
(`elementName`), dimension (`dim`), order (`order`), number of nodes
(`numNodes`) and coordinates of the nodes in the reference element (`nodeCoord`
vector, of length `dim` times `numNodes`).

Return `elementName`, `dim`, `order`, `numNodes`, `nodeCoord`.
"""
function getElementProperties(elementType)
    api_elementName_ = Ref{Ptr{Cchar}}()
    api_dim_ = Ref{Cint}()
    api_order_ = Ref{Cint}()
    api_numNodes_ = Ref{Cint}()
    api_nodeCoord_ = Ref{Ptr{Cdouble}}()
    api_nodeCoord_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElementProperties, gmsh.lib), Cvoid,
          (Cint, Ptr{Ptr{Cchar}}, Ptr{Cint}, Ptr{Cint}, Ptr{Cint}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          elementType, api_elementName_, api_dim_, api_order_, api_numNodes_, api_nodeCoord_, api_nodeCoord_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElementProperties returned non-zero error code: $(ierr[])")
    elementName = unsafe_string(api_elementName_[])
    nodeCoord = unsafe_wrap(Array, api_nodeCoord_[], api_nodeCoord_n_[], own=true)
    return elementName, api_dim_[], api_order_[], api_numNodes_[], nodeCoord
end

"""
    gmsh.model.mesh.getElementsByType(elementType, tag = -1, task = 0, numTasks = 1)

Get the elements of type `elementType` classified on the entity of tag `tag`. If
`tag` < 0, get the elements for all entities. `elementTags` is a vector
containing the tags (unique, strictly positive identifiers) of the elements of
the corresponding type. `nodeTags` is a vector of length equal to the number of
elements of the given type times the number N of nodes for this type of element,
that contains the node tags of all the elements of the given type, concatenated:
[e1n1, e1n2, ..., e1nN, e2n1, ...]. If `numTasks` > 1, only compute and return
the part of the data indexed by `task`.

Return `elementTags`, `nodeTags`.
"""
function getElementsByType(elementType, tag = -1, task = 0, numTasks = 1)
    api_elementTags_ = Ref{Ptr{Csize_t}}()
    api_elementTags_n_ = Ref{Csize_t}()
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElementsByType, gmsh.lib), Cvoid,
          (Cint, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Csize_t, Csize_t, Ptr{Cint}),
          elementType, api_elementTags_, api_elementTags_n_, api_nodeTags_, api_nodeTags_n_, tag, task, numTasks, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElementsByType returned non-zero error code: $(ierr[])")
    elementTags = unsafe_wrap(Array, api_elementTags_[], api_elementTags_n_[], own=true)
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    return elementTags, nodeTags
end

"""
    gmsh.model.mesh.addElements(dim, tag, elementTypes, elementTags, nodeTags)

Add elements classified on the entity of dimension `dim` and tag `tag`. `types`
contains the MSH types of the elements (e.g. `2` for 3-node triangles: see the
Gmsh reference manual). `elementTags` is a vector of the same length as `types`;
each entry is a vector containing the tags (unique, strictly positive
identifiers) of the elements of the corresponding type. `nodeTags` is also a
vector of the same length as `types`; each entry is a vector of length equal to
the number of elements of the given type times the number N of nodes per
element, that contains the node tags of all the elements of the given type,
concatenated: [e1n1, e1n2, ..., e1nN, e2n1, ...].
"""
function addElements(dim, tag, elementTypes, elementTags, nodeTags)
    api_elementTags_n_ = [ length(elementTags[i]) for i in 1:length(elementTags) ]
    api_nodeTags_n_ = [ length(nodeTags[i]) for i in 1:length(nodeTags) ]
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshAddElements, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}, Csize_t, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Csize_t, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Csize_t, Ptr{Cint}),
          dim, tag, convert(Vector{Cint}, elementTypes), length(elementTypes), convert(Vector{Vector{Csize_t}},elementTags), api_elementTags_n_, length(elementTags), convert(Vector{Vector{Csize_t}},nodeTags), api_nodeTags_n_, length(nodeTags), ierr)
    ierr[] != 0 && error("gmshModelMeshAddElements returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.addElementsByType(tag, elementType, elementTags, nodeTags)

Add elements of type `elementType` classified on the entity of tag `tag`.
`elementTags` contains the tags (unique, strictly positive identifiers) of the
elements of the corresponding type. `nodeTags` is a vector of length equal to
the number of elements times the number N of nodes per element, that contains
the node tags of all the elements, concatenated: [e1n1, e1n2, ..., e1nN, e2n1,
...]. If the `elementTag` vector is empty, new tags are automatically assigned
to the elements.
"""
function addElementsByType(tag, elementType, elementTags, nodeTags)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshAddElementsByType, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Csize_t}, Csize_t, Ptr{Csize_t}, Csize_t, Ptr{Cint}),
          tag, elementType, convert(Vector{Csize_t}, elementTags), length(elementTags), convert(Vector{Csize_t}, nodeTags), length(nodeTags), ierr)
    ierr[] != 0 && error("gmshModelMeshAddElementsByType returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.getIntegrationPoints(elementType, integrationType)

Get the numerical quadrature information for the given element type
`elementType` and integration rule `integrationType` (e.g. "Gauss4" for a Gauss
quadrature suited for integrating 4th order polynomials). `integrationPoints`
contains the u, v, w coordinates of the G integration points in the reference
element: [g1u, g1v, g1w, ..., gGu, gGv, gGw]. `integrationWeigths` contains the
associated weights: [g1q, ..., gGq].

Return `integrationPoints`, `integrationWeights`.
"""
function getIntegrationPoints(elementType, integrationType)
    api_integrationPoints_ = Ref{Ptr{Cdouble}}()
    api_integrationPoints_n_ = Ref{Csize_t}()
    api_integrationWeights_ = Ref{Ptr{Cdouble}}()
    api_integrationWeights_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetIntegrationPoints, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          elementType, integrationType, api_integrationPoints_, api_integrationPoints_n_, api_integrationWeights_, api_integrationWeights_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetIntegrationPoints returned non-zero error code: $(ierr[])")
    integrationPoints = unsafe_wrap(Array, api_integrationPoints_[], api_integrationPoints_n_[], own=true)
    integrationWeights = unsafe_wrap(Array, api_integrationWeights_[], api_integrationWeights_n_[], own=true)
    return integrationPoints, integrationWeights
end

"""
    gmsh.model.mesh.getJacobians(elementType, integrationPoints, tag = -1, task = 0, numTasks = 1)

Get the Jacobians of all the elements of type `elementType` classified on the
entity of tag `tag`, at the G integration points `integrationPoints` given as
concatenated triplets of coordinates in the reference element [g1u, g1v, g1w,
..., gGu, gGv, gGw]. Data is returned by element, with elements in the same
order as in `getElements` and `getElementsByType`. `jacobians` contains for each
element the 9 entries of the 3x3 Jacobian matrix at each integration point. The
matrix is returned by column: [e1g1Jxu, e1g1Jyu, e1g1Jzu, e1g1Jxv, ..., e1g1Jzw,
e1g2Jxu, ..., e1gGJzw, e2g1Jxu, ...], with Jxu=dx/du, Jyu=dy/du, etc.
`determinants` contains for each element the determinant of the Jacobian matrix
at each integration point: [e1g1, e1g2, ... e1gG, e2g1, ...]. `points` contains
for each element the x, y, z coordinates of the integration points. If `tag` <
0, get the Jacobian data for all entities. If `numTasks` > 1, only compute and
return the part of the data indexed by `task`.

Return `jacobians`, `determinants`, `points`.
"""
function getJacobians(elementType, integrationPoints, tag = -1, task = 0, numTasks = 1)
    api_jacobians_ = Ref{Ptr{Cdouble}}()
    api_jacobians_n_ = Ref{Csize_t}()
    api_determinants_ = Ref{Ptr{Cdouble}}()
    api_determinants_n_ = Ref{Csize_t}()
    api_points_ = Ref{Ptr{Cdouble}}()
    api_points_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetJacobians, gmsh.lib), Cvoid,
          (Cint, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Cint, Csize_t, Csize_t, Ptr{Cint}),
          elementType, convert(Vector{Cdouble}, integrationPoints), length(integrationPoints), api_jacobians_, api_jacobians_n_, api_determinants_, api_determinants_n_, api_points_, api_points_n_, tag, task, numTasks, ierr)
    ierr[] != 0 && error("gmshModelMeshGetJacobians returned non-zero error code: $(ierr[])")
    jacobians = unsafe_wrap(Array, api_jacobians_[], api_jacobians_n_[], own=true)
    determinants = unsafe_wrap(Array, api_determinants_[], api_determinants_n_[], own=true)
    points = unsafe_wrap(Array, api_points_[], api_points_n_[], own=true)
    return jacobians, determinants, points
end

"""
    gmsh.model.mesh.getBasisFunctions(elementType, integrationPoints, functionSpaceType)

Get the basis functions of the element of type `elementType` at the integration
points `integrationPoints` (given as concatenated triplets of coordinates in the
reference element [g1u, g1v, g1w, ..., gGu, gGv, gGw]), for the function space
`functionSpaceType` (e.g. "Lagrange" or "GradLagrange" for Lagrange basis
functions or their gradient, in the u, v, w coordinates of the reference
element). `numComponents` returns the number C of components of a basis
function. `basisFunctions` returns the value of the N basis functions at the
integration points, i.e. [g1f1, g1f2, ..., g1fN, g2f1, ...] when C == 1 or
[g1f1u, g1f1v, g1f1w, g1f2u, ..., g1fNw, g2f1u, ...] when C == 3.

Return `numComponents`, `basisFunctions`.
"""
function getBasisFunctions(elementType, integrationPoints, functionSpaceType)
    api_numComponents_ = Ref{Cint}()
    api_basisFunctions_ = Ref{Ptr{Cdouble}}()
    api_basisFunctions_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetBasisFunctions, gmsh.lib), Cvoid,
          (Cint, Ptr{Cdouble}, Csize_t, Ptr{Cchar}, Ptr{Cint}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          elementType, convert(Vector{Cdouble}, integrationPoints), length(integrationPoints), functionSpaceType, api_numComponents_, api_basisFunctions_, api_basisFunctions_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetBasisFunctions returned non-zero error code: $(ierr[])")
    basisFunctions = unsafe_wrap(Array, api_basisFunctions_[], api_basisFunctions_n_[], own=true)
    return api_numComponents_[], basisFunctions
end

"""
    gmsh.model.mesh.getBasisFunctionsForElements(elementType, integrationPoints, functionSpaceType, tag = -1)

Get the element-dependent basis functions of the elements of type `elementType`
in the entity of tag `tag`at the integration points `integrationPoints` (given
as concatenated triplets of coordinates in the reference element [g1u, g1v, g1w,
..., gGu, gGv, gGw]), for the function space `functionSpaceType` (e.g.
"H1Legendre3" or "GradH1Legendre3" for 3rd order hierarchical H1 Legendre
functions or their gradient, in the u, v, w coordinates of the reference
elements). `numComponents` returns the number C of components of a basis
function. `numBasisFunctions` returns the number N of basis functions per
element. `basisFunctions` returns the value of the basis functions at the
integration points for each element: [e1g1f1,..., e1g1fN, e1g2f1,..., e2g1f1,
...] when C == 1 or [e1g1f1u, e1g1f1v,..., e1g1fNw, e1g2f1u,..., e2g1f1u, ...].
Warning: this is an experimental feature and will probably change in a future
release.

Return `numComponents`, `numFunctionsPerElements`, `basisFunctions`.
"""
function getBasisFunctionsForElements(elementType, integrationPoints, functionSpaceType, tag = -1)
    api_numComponents_ = Ref{Cint}()
    api_numFunctionsPerElements_ = Ref{Cint}()
    api_basisFunctions_ = Ref{Ptr{Cdouble}}()
    api_basisFunctions_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetBasisFunctionsForElements, gmsh.lib), Cvoid,
          (Cint, Ptr{Cdouble}, Csize_t, Ptr{Cchar}, Ptr{Cint}, Ptr{Cint}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          elementType, convert(Vector{Cdouble}, integrationPoints), length(integrationPoints), functionSpaceType, api_numComponents_, api_numFunctionsPerElements_, api_basisFunctions_, api_basisFunctions_n_, tag, ierr)
    ierr[] != 0 && error("gmshModelMeshGetBasisFunctionsForElements returned non-zero error code: $(ierr[])")
    basisFunctions = unsafe_wrap(Array, api_basisFunctions_[], api_basisFunctions_n_[], own=true)
    return api_numComponents_[], api_numFunctionsPerElements_[], basisFunctions
end

"""
    gmsh.model.mesh.getKeysForElements(elementType, functionSpaceType, tag = -1, returnCoord = true)

Generate the `keys` for the elements of type `elementType` in the entity of tag
`tag`, for the `functionSpaceType` function space. Each key uniquely identifies
a basis function in the function space. If `returnCoord` is set, the `coord`
vector contains the x, y, z coordinates locating basis functions for sorting
purposes. Warning: this is an experimental feature and will probably change in a
future release.

Return `keys`, `coord`.
"""
function getKeysForElements(elementType, functionSpaceType, tag = -1, returnCoord = true)
    api_keys_ = Ref{Ptr{Cint}}()
    api_keys_n_ = Ref{Csize_t}()
    api_coord_ = Ref{Ptr{Cdouble}}()
    api_coord_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetKeysForElements, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Cint, Cint, Ptr{Cint}),
          elementType, functionSpaceType, api_keys_, api_keys_n_, api_coord_, api_coord_n_, tag, returnCoord, ierr)
    ierr[] != 0 && error("gmshModelMeshGetKeysForElements returned non-zero error code: $(ierr[])")
    tmp_api_keys_ = unsafe_wrap(Array, api_keys_[], api_keys_n_[], own=true)
    keys = [ (tmp_api_keys_[i], tmp_api_keys_[i+1]) for i in 1:2:length(tmp_api_keys_) ]
    coord = unsafe_wrap(Array, api_coord_[], api_coord_n_[], own=true)
    return keys, coord
end

"""
    gmsh.model.mesh.getInformationForElements(keys, elementType, functionSpaceType)

Get information about the `keys`. `infoKeys` returns information about the
functions associated with the `keys`. `infoKeys[0].first` describes the type of
function (0 for  vertex function, 1 for edge function, 2 for face function and 3
for bubble function). `infoKeys[0].second` gives the order of the function
associated with the key. Warning: this is an experimental feature and will
probably change in a future release.

Return `infoKeys`.
"""
function getInformationForElements(keys, elementType, functionSpaceType)
    api_infoKeys_ = Ref{Ptr{Cint}}()
    api_infoKeys_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetInformationForElements, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cchar}, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(keys))), 2 * length(keys), elementType, functionSpaceType, api_infoKeys_, api_infoKeys_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetInformationForElements returned non-zero error code: $(ierr[])")
    tmp_api_infoKeys_ = unsafe_wrap(Array, api_infoKeys_[], api_infoKeys_n_[], own=true)
    infoKeys = [ (tmp_api_infoKeys_[i], tmp_api_infoKeys_[i+1]) for i in 1:2:length(tmp_api_infoKeys_) ]
    return infoKeys
end

"""
    gmsh.model.mesh.precomputeBasisFunctions(elementType)

Precomputes the basis functions corresponding to `elementType`.
"""
function precomputeBasisFunctions(elementType)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshPrecomputeBasisFunctions, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          elementType, ierr)
    ierr[] != 0 && error("gmshModelMeshPrecomputeBasisFunctions returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.getBarycenters(elementType, tag, fast, primary, task = 0, numTasks = 1)

Get the barycenters of all elements of type `elementType` classified on the
entity of tag `tag`. If `primary` is set, only the primary nodes of the elements
are taken into account for the barycenter calculation. If `fast` is set, the
function returns the sum of the primary node coordinates (without normalizing by
the number of nodes). If `tag` < 0, get the barycenters for all entities. If
`numTasks` > 1, only compute and return the part of the data indexed by `task`.

Return `barycenters`.
"""
function getBarycenters(elementType, tag, fast, primary, task = 0, numTasks = 1)
    api_barycenters_ = Ref{Ptr{Cdouble}}()
    api_barycenters_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetBarycenters, gmsh.lib), Cvoid,
          (Cint, Cint, Cint, Cint, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Csize_t, Csize_t, Ptr{Cint}),
          elementType, tag, fast, primary, api_barycenters_, api_barycenters_n_, task, numTasks, ierr)
    ierr[] != 0 && error("gmshModelMeshGetBarycenters returned non-zero error code: $(ierr[])")
    barycenters = unsafe_wrap(Array, api_barycenters_[], api_barycenters_n_[], own=true)
    return barycenters
end

"""
    gmsh.model.mesh.getElementEdgeNodes(elementType, tag = -1, primary = false, task = 0, numTasks = 1)

Get the nodes on the edges of all elements of type `elementType` classified on
the entity of tag `tag`. `nodeTags` contains the node tags of the edges for all
the elements: [e1a1n1, e1a1n2, e1a2n1, ...]. Data is returned by element, with
elements in the same order as in `getElements` and `getElementsByType`. If
`primary` is set, only the primary (begin/end) nodes of the edges are returned.
If `tag` < 0, get the edge nodes for all entities. If `numTasks` > 1, only
compute and return the part of the data indexed by `task`.

Return `nodeTags`.
"""
function getElementEdgeNodes(elementType, tag = -1, primary = false, task = 0, numTasks = 1)
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElementEdgeNodes, gmsh.lib), Cvoid,
          (Cint, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Cint, Csize_t, Csize_t, Ptr{Cint}),
          elementType, api_nodeTags_, api_nodeTags_n_, tag, primary, task, numTasks, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElementEdgeNodes returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    return nodeTags
end

"""
    gmsh.model.mesh.getElementFaceNodes(elementType, faceType, tag = -1, primary = false, task = 0, numTasks = 1)

Get the nodes on the faces of type `faceType` (3 for triangular faces, 4 for
quadrangular faces) of all elements of type `elementType` classified on the
entity of tag `tag`. `nodeTags` contains the node tags of the faces for all
elements: [e1f1n1, ..., e1f1nFaceType, e1f2n1, ...]. Data is returned by
element, with elements in the same order as in `getElements` and
`getElementsByType`. If `primary` is set, only the primary (corner) nodes of the
faces are returned. If `tag` < 0, get the face nodes for all entities. If
`numTasks` > 1, only compute and return the part of the data indexed by `task`.

Return `nodeTags`.
"""
function getElementFaceNodes(elementType, faceType, tag = -1, primary = false, task = 0, numTasks = 1)
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetElementFaceNodes, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Cint, Csize_t, Csize_t, Ptr{Cint}),
          elementType, faceType, api_nodeTags_, api_nodeTags_n_, tag, primary, task, numTasks, ierr)
    ierr[] != 0 && error("gmshModelMeshGetElementFaceNodes returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    return nodeTags
end

"""
    gmsh.model.mesh.getGhostElements(dim, tag)

Get the ghost elements `elementTags` and their associated `partitions` stored in
the ghost entity of dimension `dim` and tag `tag`.

Return `elementTags`, `partitions`.
"""
function getGhostElements(dim, tag)
    api_elementTags_ = Ref{Ptr{Csize_t}}()
    api_elementTags_n_ = Ref{Csize_t}()
    api_partitions_ = Ref{Ptr{Cint}}()
    api_partitions_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetGhostElements, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, api_elementTags_, api_elementTags_n_, api_partitions_, api_partitions_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetGhostElements returned non-zero error code: $(ierr[])")
    elementTags = unsafe_wrap(Array, api_elementTags_[], api_elementTags_n_[], own=true)
    partitions = unsafe_wrap(Array, api_partitions_[], api_partitions_n_[], own=true)
    return elementTags, partitions
end

"""
    gmsh.model.mesh.setSize(dimTags, size)

Set a mesh size constraint on the model entities `dimTags`. Currently only
entities of dimension 0 (points) are handled.
"""
function setSize(dimTags, size)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetSize, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), size, ierr)
    ierr[] != 0 && error("gmshModelMeshSetSize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setTransfiniteCurve(tag, numNodes, meshType = "Progression", coef = 1.)

Set a transfinite meshing constraint on the curve `tag`, with `numNodes` nodes
distributed according to `meshType` and `coef`. Currently supported types are
"Progression" (geometrical progression with power `coef`) and "Bump" (refinement
toward both extremities of the curve).
"""
function setTransfiniteCurve(tag, numNodes, meshType = "Progression", coef = 1.)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetTransfiniteCurve, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cchar}, Cdouble, Ptr{Cint}),
          tag, numNodes, meshType, coef, ierr)
    ierr[] != 0 && error("gmshModelMeshSetTransfiniteCurve returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setTransfiniteSurface(tag, arrangement = "Left", cornerTags = Cint[])

Set a transfinite meshing constraint on the surface `tag`. `arrangement`
describes the arrangement of the triangles when the surface is not flagged as
recombined: currently supported values are "Left", "Right", "AlternateLeft" and
"AlternateRight". `cornerTags` can be used to specify the (3 or 4) corners of
the transfinite interpolation explicitly; specifying the corners explicitly is
mandatory if the surface has more that 3 or 4 points on its boundary.
"""
function setTransfiniteSurface(tag, arrangement = "Left", cornerTags = Cint[])
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetTransfiniteSurface, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Ptr{Cint}, Csize_t, Ptr{Cint}),
          tag, arrangement, convert(Vector{Cint}, cornerTags), length(cornerTags), ierr)
    ierr[] != 0 && error("gmshModelMeshSetTransfiniteSurface returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setTransfiniteVolume(tag, cornerTags = Cint[])

Set a transfinite meshing constraint on the surface `tag`. `cornerTags` can be
used to specify the (6 or 8) corners of the transfinite interpolation
explicitly.
"""
function setTransfiniteVolume(tag, cornerTags = Cint[])
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetTransfiniteVolume, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}, Csize_t, Ptr{Cint}),
          tag, convert(Vector{Cint}, cornerTags), length(cornerTags), ierr)
    ierr[] != 0 && error("gmshModelMeshSetTransfiniteVolume returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setRecombine(dim, tag)

Set a recombination meshing constraint on the model entity of dimension `dim`
and tag `tag`. Currently only entities of dimension 2 (to recombine triangles
into quadrangles) are supported.
"""
function setRecombine(dim, tag)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetRecombine, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}),
          dim, tag, ierr)
    ierr[] != 0 && error("gmshModelMeshSetRecombine returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setSmoothing(dim, tag, val)

Set a smoothing meshing constraint on the model entity of dimension `dim` and
tag `tag`. `val` iterations of a Laplace smoother are applied.
"""
function setSmoothing(dim, tag, val)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetSmoothing, gmsh.lib), Cvoid,
          (Cint, Cint, Cint, Ptr{Cint}),
          dim, tag, val, ierr)
    ierr[] != 0 && error("gmshModelMeshSetSmoothing returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setReverse(dim, tag, val = true)

Set a reverse meshing constraint on the model entity of dimension `dim` and tag
`tag`. If `val` is true, the mesh orientation will be reversed with respect to
the natural mesh orientation (i.e. the orientation consistent with the
orientation of the geometry). If `val` is false, the mesh is left as-is.
"""
function setReverse(dim, tag, val = true)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetReverse, gmsh.lib), Cvoid,
          (Cint, Cint, Cint, Ptr{Cint}),
          dim, tag, val, ierr)
    ierr[] != 0 && error("gmshModelMeshSetReverse returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setOutwardOrientation(tag)

Set meshing constraints on the bounding surfaces of the volume of tag `tag` so
that all surfaces are oriented with outward pointing normals. Currently only
available with the OpenCASCADE kernel, as it relies on the STL triangulation.
"""
function setOutwardOrientation(tag)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetOutwardOrientation, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          tag, ierr)
    ierr[] != 0 && error("gmshModelMeshSetOutwardOrientation returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.embed(dim, tags, inDim, inTag)

Embed the model entities of dimension `dim` and tags `tags` in the (`inDim`,
`inTag`) model entity. The dimension `dim` can 0, 1 or 2 and must be strictly
smaller than `inDim`, which must be either 2 or 3. The embedded entities should
not be part of the boundary of the entity `inTag`, whose mesh will conform to
the mesh of the embedded entities.
"""
function embed(dim, tags, inDim, inTag)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshEmbed, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}, Csize_t, Cint, Cint, Ptr{Cint}),
          dim, convert(Vector{Cint}, tags), length(tags), inDim, inTag, ierr)
    ierr[] != 0 && error("gmshModelMeshEmbed returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.removeEmbedded(dimTags, dim = -1)

Remove embedded entities from the model entities `dimTags`. if `dim` is >= 0,
only remove embedded entities of the given dimension (e.g. embedded points if
`dim` == 0).
"""
function removeEmbedded(dimTags, dim = -1)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRemoveEmbedded, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), dim, ierr)
    ierr[] != 0 && error("gmshModelMeshRemoveEmbedded returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.reorderElements(elementType, tag, ordering)

Reorder the elements of type `elementType` classified on the entity of tag `tag`
according to `ordering`.
"""
function reorderElements(elementType, tag, ordering)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshReorderElements, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Csize_t}, Csize_t, Ptr{Cint}),
          elementType, tag, convert(Vector{Csize_t}, ordering), length(ordering), ierr)
    ierr[] != 0 && error("gmshModelMeshReorderElements returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.renumberNodes()

Renumber the node tags in a continuous sequence.
"""
function renumberNodes()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRenumberNodes, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshRenumberNodes returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.renumberElements()

Renumber the element tags in a continuous sequence.
"""
function renumberElements()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRenumberElements, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshRenumberElements returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.setPeriodic(dim, tags, tagsMaster, affineTransform)

Set the meshes of the entities of dimension `dim` and tag `tags` as periodic
copies of the meshes of entities `tagsMaster`, using the affine transformation
specified in `affineTransformation` (16 entries of a 4x4 matrix, by row).
Currently only available for `dim` == 1 and `dim` == 2.
"""
function setPeriodic(dim, tags, tagsMaster, affineTransform)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSetPeriodic, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Cint}),
          dim, convert(Vector{Cint}, tags), length(tags), convert(Vector{Cint}, tagsMaster), length(tagsMaster), convert(Vector{Cdouble}, affineTransform), length(affineTransform), ierr)
    ierr[] != 0 && error("gmshModelMeshSetPeriodic returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.getPeriodicNodes(dim, tag)

Get the master entity `tagMaster`, the node tags `nodeTags` and their
corresponding master node tags `nodeTagsMaster`, and the affine transform
`affineTransform` for the entity of dimension `dim` and tag `tag`.

Return `tagMaster`, `nodeTags`, `nodeTagsMaster`, `affineTransform`.
"""
function getPeriodicNodes(dim, tag)
    api_tagMaster_ = Ref{Cint}()
    api_nodeTags_ = Ref{Ptr{Csize_t}}()
    api_nodeTags_n_ = Ref{Csize_t}()
    api_nodeTagsMaster_ = Ref{Ptr{Csize_t}}()
    api_nodeTagsMaster_n_ = Ref{Csize_t}()
    api_affineTransform_ = Ref{Ptr{Cdouble}}()
    api_affineTransform_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshGetPeriodicNodes, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, api_tagMaster_, api_nodeTags_, api_nodeTags_n_, api_nodeTagsMaster_, api_nodeTagsMaster_n_, api_affineTransform_, api_affineTransform_n_, ierr)
    ierr[] != 0 && error("gmshModelMeshGetPeriodicNodes returned non-zero error code: $(ierr[])")
    nodeTags = unsafe_wrap(Array, api_nodeTags_[], api_nodeTags_n_[], own=true)
    nodeTagsMaster = unsafe_wrap(Array, api_nodeTagsMaster_[], api_nodeTagsMaster_n_[], own=true)
    affineTransform = unsafe_wrap(Array, api_affineTransform_[], api_affineTransform_n_[], own=true)
    return api_tagMaster_[], nodeTags, nodeTagsMaster, affineTransform
end

"""
    gmsh.model.mesh.removeDuplicateNodes()

Remove duplicate nodes in the mesh of the current model.
"""
function removeDuplicateNodes()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshRemoveDuplicateNodes, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshRemoveDuplicateNodes returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.splitQuadrangles(quality = 1., tag = -1)

Split (into two triangles) all quadrangles in surface `tag` whose quality is
lower than `quality`. If `tag` < 0, split quadrangles in all surfaces.
"""
function splitQuadrangles(quality = 1., tag = -1)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshSplitQuadrangles, gmsh.lib), Cvoid,
          (Cdouble, Cint, Ptr{Cint}),
          quality, tag, ierr)
    ierr[] != 0 && error("gmshModelMeshSplitQuadrangles returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.classifySurfaces(angle, boundary = true, forReparametrization = false)

Classify ("color") the surface mesh based on the angle threshold `angle` (in
radians), and create new discrete surfaces, curves and points accordingly. If
`boundary` is set, also create discrete curves on the boundary if the surface is
open. If `forReparametrization` is set, create edges and surfaces that can be
reparametrized using a single map.
"""
function classifySurfaces(angle, boundary = true, forReparametrization = false)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshClassifySurfaces, gmsh.lib), Cvoid,
          (Cdouble, Cint, Cint, Ptr{Cint}),
          angle, boundary, forReparametrization, ierr)
    ierr[] != 0 && error("gmshModelMeshClassifySurfaces returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.createGeometry()

Create a parametrization for discrete curves and surfaces (i.e. curves and
surfaces represented solely by a mesh, without an underlying CAD description),
assuming that each can be parametrized with a single map.
"""
function createGeometry()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshCreateGeometry, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshCreateGeometry returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.createTopology()

Create a boundary representation from the mesh if the model does not have one
(e.g. when imported from mesh file formats with no BRep representation of the
underlying model).
"""
function createTopology()
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshCreateTopology, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelMeshCreateTopology returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.computeHomology(domainTags = Cint[], subdomainTags = Cint[], dims = Cint[])

Compute a basis representation for homology spaces after a mesh has been
generated. The computation domain is given in a list of physical group tags
`domainTags`; if empty, the whole mesh is the domain. The computation subdomain
for relative homology computation is given in a list of physical group tags
`subdomainTags`; if empty, absolute homology is computed. The dimensions
homology bases to be computed are given in the list `dim`; if empty, all bases
are computed. Resulting basis representation chains are stored as physical
groups in the mesh.
"""
function computeHomology(domainTags = Cint[], subdomainTags = Cint[], dims = Cint[])
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshComputeHomology, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cint}),
          convert(Vector{Cint}, domainTags), length(domainTags), convert(Vector{Cint}, subdomainTags), length(subdomainTags), convert(Vector{Cint}, dims), length(dims), ierr)
    ierr[] != 0 && error("gmshModelMeshComputeHomology returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.computeCohomology(domainTags = Cint[], subdomainTags = Cint[], dims = Cint[])

Compute a basis representation for cohomology spaces after a mesh has been
generated. The computation domain is given in a list of physical group tags
`domainTags`; if empty, the whole mesh is the domain. The computation subdomain
for relative cohomology computation is given in a list of physical group tags
`subdomainTags`; if empty, absolute cohomology is computed. The dimensions
homology bases to be computed are given in the list `dim`; if empty, all bases
are computed. Resulting basis representation cochains are stored as physical
groups in the mesh.
"""
function computeCohomology(domainTags = Cint[], subdomainTags = Cint[], dims = Cint[])
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshComputeCohomology, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cint}),
          convert(Vector{Cint}, domainTags), length(domainTags), convert(Vector{Cint}, subdomainTags), length(subdomainTags), convert(Vector{Cint}, dims), length(dims), ierr)
    ierr[] != 0 && error("gmshModelMeshComputeCohomology returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    module gmsh.model.mesh.field

Mesh size field functions
"""
module field

import ....gmsh

"""
    gmsh.model.mesh.field.add(fieldType, tag = -1)

Add a new mesh size field of type `fieldType`. If `tag` is positive, assign the
tag explicitly; otherwise a new tag is assigned automatically. Return the field
tag.

Return an integer value.
"""
function add(fieldType, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelMeshFieldAdd, gmsh.lib), Cint,
          (Ptr{Cchar}, Cint, Ptr{Cint}),
          fieldType, tag, ierr)
    ierr[] != 0 && error("gmshModelMeshFieldAdd returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.mesh.field.remove(tag)

Remove the field with tag `tag`.
"""
function remove(tag)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshFieldRemove, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          tag, ierr)
    ierr[] != 0 && error("gmshModelMeshFieldRemove returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.field.setNumber(tag, option, value)

Set the numerical option `option` to value `value` for field `tag`.
"""
function setNumber(tag, option, value)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshFieldSetNumber, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Cdouble, Ptr{Cint}),
          tag, option, value, ierr)
    ierr[] != 0 && error("gmshModelMeshFieldSetNumber returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.field.setString(tag, option, value)

Set the string option `option` to value `value` for field `tag`.
"""
function setString(tag, option, value)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshFieldSetString, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Ptr{Cchar}, Ptr{Cint}),
          tag, option, value, ierr)
    ierr[] != 0 && error("gmshModelMeshFieldSetString returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.field.setNumbers(tag, option, value)

Set the numerical list option `option` to value `value` for field `tag`.
"""
function setNumbers(tag, option, value)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshFieldSetNumbers, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Ptr{Cdouble}, Csize_t, Ptr{Cint}),
          tag, option, convert(Vector{Cdouble}, value), length(value), ierr)
    ierr[] != 0 && error("gmshModelMeshFieldSetNumbers returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.field.setAsBackgroundMesh(tag)

Set the field `tag` as the background mesh size field.
"""
function setAsBackgroundMesh(tag)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshFieldSetAsBackgroundMesh, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          tag, ierr)
    ierr[] != 0 && error("gmshModelMeshFieldSetAsBackgroundMesh returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.mesh.field.setAsBoundaryLayer(tag)

Set the field `tag` as a boundary layer size field.
"""
function setAsBoundaryLayer(tag)
    ierr = Ref{Cint}()
    ccall((:gmshModelMeshFieldSetAsBoundaryLayer, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          tag, ierr)
    ierr[] != 0 && error("gmshModelMeshFieldSetAsBoundaryLayer returned non-zero error code: $(ierr[])")
    return nothing
end

end # end of module field

end # end of module mesh

"""
    module gmsh.model.geo

Built-in CAD kernel functions
"""
module geo

import ...gmsh

"""
    gmsh.model.geo.addPoint(x, y, z, meshSize = 0., tag = -1)

Add a geometrical point in the built-in CAD representation, at coordinates (`x`,
`y`, `z`). If `meshSize` is > 0, add a meshing constraint at that point. If
`tag` is positive, set the tag explicitly; otherwise a new tag is selected
automatically. Return the tag of the point. (Note that the point will be added
in the current model only after `synchronize` is called. This behavior holds for
all the entities added in the geo module.)

Return an integer value.
"""
function addPoint(x, y, z, meshSize = 0., tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddPoint, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cint, Ptr{Cint}),
          x, y, z, meshSize, tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddPoint returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addLine(startTag, endTag, tag = -1)

Add a straight line segment between the two points with tags `startTag` and
`endTag`. If `tag` is positive, set the tag explicitly; otherwise a new tag is
selected automatically. Return the tag of the line.

Return an integer value.
"""
function addLine(startTag, endTag, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddLine, gmsh.lib), Cint,
          (Cint, Cint, Cint, Ptr{Cint}),
          startTag, endTag, tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddLine returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addCircleArc(startTag, centerTag, endTag, tag = -1, nx = 0., ny = 0., nz = 0.)

Add a circle arc (strictly smaller than Pi) between the two points with tags
`startTag` and `endTag`, with center `centertag`. If `tag` is positive, set the
tag explicitly; otherwise a new tag is selected automatically. If (`nx`, `ny`,
`nz`) != (0,0,0), explicitly set the plane of the circle arc. Return the tag of
the circle arc.

Return an integer value.
"""
function addCircleArc(startTag, centerTag, endTag, tag = -1, nx = 0., ny = 0., nz = 0.)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddCircleArc, gmsh.lib), Cint,
          (Cint, Cint, Cint, Cint, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          startTag, centerTag, endTag, tag, nx, ny, nz, ierr)
    ierr[] != 0 && error("gmshModelGeoAddCircleArc returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addEllipseArc(startTag, centerTag, majorTag, endTag, tag = -1, nx = 0., ny = 0., nz = 0.)

Add an ellipse arc (strictly smaller than Pi) between the two points `startTag`
and `endTag`, with center `centerTag` and major axis point `majorTag`. If `tag`
is positive, set the tag explicitly; otherwise a new tag is selected
automatically. If (`nx`, `ny`, `nz`) != (0,0,0), explicitly set the plane of the
circle arc. Return the tag of the ellipse arc.

Return an integer value.
"""
function addEllipseArc(startTag, centerTag, majorTag, endTag, tag = -1, nx = 0., ny = 0., nz = 0.)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddEllipseArc, gmsh.lib), Cint,
          (Cint, Cint, Cint, Cint, Cint, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          startTag, centerTag, majorTag, endTag, tag, nx, ny, nz, ierr)
    ierr[] != 0 && error("gmshModelGeoAddEllipseArc returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addSpline(pointTags, tag = -1)

Add a spline (Catmull-Rom) curve going through the points `pointTags`. If `tag`
is positive, set the tag explicitly; otherwise a new tag is selected
automatically. Create a periodic curve if the first and last points are the
same. Return the tag of the spline curve.

Return an integer value.
"""
function addSpline(pointTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddSpline, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, pointTags), length(pointTags), tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddSpline returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addBSpline(pointTags, tag = -1)

Add a cubic b-spline curve with `pointTags` control points. If `tag` is
positive, set the tag explicitly; otherwise a new tag is selected automatically.
Creates a periodic curve if the first and last points are the same. Return the
tag of the b-spline curve.

Return an integer value.
"""
function addBSpline(pointTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddBSpline, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, pointTags), length(pointTags), tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddBSpline returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addBezier(pointTags, tag = -1)

Add a Bezier curve with `pointTags` control points. If `tag` is positive, set
the tag explicitly; otherwise a new tag is selected automatically.  Return the
tag of the Bezier curve.

Return an integer value.
"""
function addBezier(pointTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddBezier, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, pointTags), length(pointTags), tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddBezier returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addCurveLoop(curveTags, tag = -1)

Add a curve loop (a closed wire) formed by the curves `curveTags`. `curveTags`
should contain (signed) tags of model enties of dimension 1 forming a closed
loop: a negative tag signifies that the underlying curve is considered with
reversed orientation. If `tag` is positive, set the tag explicitly; otherwise a
new tag is selected automatically. Return the tag of the curve loop.

Return an integer value.
"""
function addCurveLoop(curveTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddCurveLoop, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, curveTags), length(curveTags), tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddCurveLoop returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addPlaneSurface(wireTags, tag = -1)

Add a plane surface defined by one or more curve loops `wireTags`. The first
curve loop defines the exterior contour; additional curve loop define holes. If
`tag` is positive, set the tag explicitly; otherwise a new tag is selected
automatically. Return the tag of the surface.

Return an integer value.
"""
function addPlaneSurface(wireTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddPlaneSurface, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, wireTags), length(wireTags), tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddPlaneSurface returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addSurfaceFilling(wireTags, tag = -1, sphereCenterTag = -1)

Add a surface filling the curve loops in `wireTags`. Currently only a single
curve loop is supported; this curve loop should be composed by 3 or 4 curves
only. If `tag` is positive, set the tag explicitly; otherwise a new tag is
selected automatically. Return the tag of the surface.

Return an integer value.
"""
function addSurfaceFilling(wireTags, tag = -1, sphereCenterTag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddSurfaceFilling, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, wireTags), length(wireTags), tag, sphereCenterTag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddSurfaceFilling returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addSurfaceLoop(surfaceTags, tag = -1)

Add a surface loop (a closed shell) formed by `surfaceTags`.  If `tag` is
positive, set the tag explicitly; otherwise a new tag is selected automatically.
Return the tag of the shell.

Return an integer value.
"""
function addSurfaceLoop(surfaceTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddSurfaceLoop, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, surfaceTags), length(surfaceTags), tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddSurfaceLoop returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.addVolume(shellTags, tag = -1)

Add a volume (a region) defined by one or more shells `shellTags`. The first
surface loop defines the exterior boundary; additional surface loop define
holes. If `tag` is positive, set the tag explicitly; otherwise a new tag is
selected automatically. Return the tag of the volume.

Return an integer value.
"""
function addVolume(shellTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelGeoAddVolume, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, shellTags), length(shellTags), tag, ierr)
    ierr[] != 0 && error("gmshModelGeoAddVolume returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.geo.extrude(dimTags, dx, dy, dz, numElements = Cint[], heights = Cdouble[], recombine = false)

Extrude the model entities `dimTags` by translation along (`dx`, `dy`, `dz`).
Return extruded entities in `outDimTags`. If `numElements` is not empty, also
extrude the mesh: the entries in `numElements` give the number of elements in
each layer. If `height` is not empty, it provides the (cumulative) height of the
different layers, normalized to 1. If `dx` == `dy` == `dz` == 0, the entities
are extruded along their normal.

Return `outDimTags`.
"""
function extrude(dimTags, dx, dy, dz, numElements = Cint[], heights = Cdouble[], recombine = false)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoExtrude, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), dx, dy, dz, api_outDimTags_, api_outDimTags_n_, convert(Vector{Cint}, numElements), length(numElements), convert(Vector{Cdouble}, heights), length(heights), recombine, ierr)
    ierr[] != 0 && error("gmshModelGeoExtrude returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.geo.revolve(dimTags, x, y, z, ax, ay, az, angle, numElements = Cint[], heights = Cdouble[], recombine = false)

Extrude the model entities `dimTags` by rotation of `angle` radians around the
axis of revolution defined by the point (`x`, `y`, `z`) and the direction (`ax`,
`ay`, `az`). The angle should be strictly smaller than Pi. Return extruded
entities in `outDimTags`. If `numElements` is not empty, also extrude the mesh:
the entries in `numElements` give the number of elements in each layer. If
`height` is not empty, it provides the (cumulative) height of the different
layers, normalized to 1.

Return `outDimTags`.
"""
function revolve(dimTags, x, y, z, ax, ay, az, angle, numElements = Cint[], heights = Cdouble[], recombine = false)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoRevolve, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), x, y, z, ax, ay, az, angle, api_outDimTags_, api_outDimTags_n_, convert(Vector{Cint}, numElements), length(numElements), convert(Vector{Cdouble}, heights), length(heights), recombine, ierr)
    ierr[] != 0 && error("gmshModelGeoRevolve returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.geo.twist(dimTags, x, y, z, dx, dy, dz, ax, ay, az, angle, numElements = Cint[], heights = Cdouble[], recombine = false)

Extrude the model entities `dimTags` by a combined translation and rotation of
`angle` radians, along (`dx`, `dy`, `dz`) and around the axis of revolution
defined by the point (`x`, `y`, `z`) and the direction (`ax`, `ay`, `az`). The
angle should be strictly smaller than Pi. Return extruded entities in
`outDimTags`. If `numElements` is not empty, also extrude the mesh: the entries
in `numElements` give the number of elements in each layer. If `height` is not
empty, it provides the (cumulative) height of the different layers, normalized
to 1.

Return `outDimTags`.
"""
function twist(dimTags, x, y, z, dx, dy, dz, ax, ay, az, angle, numElements = Cint[], heights = Cdouble[], recombine = false)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoTwist, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), x, y, z, dx, dy, dz, ax, ay, az, angle, api_outDimTags_, api_outDimTags_n_, convert(Vector{Cint}, numElements), length(numElements), convert(Vector{Cdouble}, heights), length(heights), recombine, ierr)
    ierr[] != 0 && error("gmshModelGeoTwist returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.geo.translate(dimTags, dx, dy, dz)

Translate the model entities `dimTags` along (`dx`, `dy`, `dz`).
"""
function translate(dimTags, dx, dy, dz)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoTranslate, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), dx, dy, dz, ierr)
    ierr[] != 0 && error("gmshModelGeoTranslate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.rotate(dimTags, x, y, z, ax, ay, az, angle)

Rotate the model entities `dimTags` of `angle` radians around the axis of
revolution defined by the point (`x`, `y`, `z`) and the direction (`ax`, `ay`,
`az`).
"""
function rotate(dimTags, x, y, z, ax, ay, az, angle)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoRotate, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), x, y, z, ax, ay, az, angle, ierr)
    ierr[] != 0 && error("gmshModelGeoRotate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.dilate(dimTags, x, y, z, a, b, c)

Scale the model entities `dimTag` by factors `a`, `b` and `c` along the three
coordinate axes; use (`x`, `y`, `z`) as the center of the homothetic
transformation.
"""
function dilate(dimTags, x, y, z, a, b, c)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoDilate, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), x, y, z, a, b, c, ierr)
    ierr[] != 0 && error("gmshModelGeoDilate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.symmetrize(dimTags, a, b, c, d)

Apply a symmetry transformation to the model entities `dimTag`, with respect to
the plane of equation `a` * x + `b` * y + `c` * z + `d` = 0.
"""
function symmetrize(dimTags, a, b, c, d)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoSymmetrize, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), a, b, c, d, ierr)
    ierr[] != 0 && error("gmshModelGeoSymmetrize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.copy(dimTags)

Copy the entities `dimTags`; the new entities are returned in `outDimTags`.

Return `outDimTags`.
"""
function copy(dimTags)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoCopy, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), api_outDimTags_, api_outDimTags_n_, ierr)
    ierr[] != 0 && error("gmshModelGeoCopy returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.geo.remove(dimTags, recursive = false)

Remove the entities `dimTags`. If `recursive` is true, remove all the entities
on their boundaries, down to dimension 0.
"""
function remove(dimTags, recursive = false)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoRemove, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), recursive, ierr)
    ierr[] != 0 && error("gmshModelGeoRemove returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.removeAllDuplicates()

Remove all duplicate entities (different entities at the same geometrical
location).
"""
function removeAllDuplicates()
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoRemoveAllDuplicates, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelGeoRemoveAllDuplicates returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.synchronize()

Synchronize the built-in CAD representation with the current Gmsh model. This
can be called at any time, but since it involves a non trivial amount of
processing, the number of synchronization points should normally be minimized.
"""
function synchronize()
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoSynchronize, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelGeoSynchronize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    module gmsh.model.geo.mesh

Built-in CAD kernel meshing constraints
"""
module mesh

import ....gmsh

"""
    gmsh.model.geo.mesh.setSize(dimTags, size)

Set a mesh size constraint on the model entities `dimTags`. Currently only
entities of dimension 0 (points) are handled.
"""
function setSize(dimTags, size)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoMeshSetSize, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), size, ierr)
    ierr[] != 0 && error("gmshModelGeoMeshSetSize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.mesh.setTransfiniteCurve(tag, nPoints, meshType = "Progression", coef = 1.)

Set a transfinite meshing constraint on the curve `tag`, with `numNodes` nodes
distributed according to `meshType` and `coef`. Currently supported types are
"Progression" (geometrical progression with power `coef`) and "Bump" (refinement
toward both extremities of the curve).
"""
function setTransfiniteCurve(tag, nPoints, meshType = "Progression", coef = 1.)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoMeshSetTransfiniteCurve, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cchar}, Cdouble, Ptr{Cint}),
          tag, nPoints, meshType, coef, ierr)
    ierr[] != 0 && error("gmshModelGeoMeshSetTransfiniteCurve returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.mesh.setTransfiniteSurface(tag, arrangement = "Left", cornerTags = Cint[])

Set a transfinite meshing constraint on the surface `tag`. `arrangement`
describes the arrangement of the triangles when the surface is not flagged as
recombined: currently supported values are "Left", "Right", "AlternateLeft" and
"AlternateRight". `cornerTags` can be used to specify the (3 or 4) corners of
the transfinite interpolation explicitly; specifying the corners explicitly is
mandatory if the surface has more that 3 or 4 points on its boundary.
"""
function setTransfiniteSurface(tag, arrangement = "Left", cornerTags = Cint[])
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoMeshSetTransfiniteSurface, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Ptr{Cint}, Csize_t, Ptr{Cint}),
          tag, arrangement, convert(Vector{Cint}, cornerTags), length(cornerTags), ierr)
    ierr[] != 0 && error("gmshModelGeoMeshSetTransfiniteSurface returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.mesh.setTransfiniteVolume(tag, cornerTags = Cint[])

Set a transfinite meshing constraint on the surface `tag`. `cornerTags` can be
used to specify the (6 or 8) corners of the transfinite interpolation
explicitly.
"""
function setTransfiniteVolume(tag, cornerTags = Cint[])
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoMeshSetTransfiniteVolume, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}, Csize_t, Ptr{Cint}),
          tag, convert(Vector{Cint}, cornerTags), length(cornerTags), ierr)
    ierr[] != 0 && error("gmshModelGeoMeshSetTransfiniteVolume returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.mesh.setRecombine(dim, tag, angle = 45.)

Set a recombination meshing constraint on the model entity of dimension `dim`
and tag `tag`. Currently only entities of dimension 2 (to recombine triangles
into quadrangles) are supported.
"""
function setRecombine(dim, tag, angle = 45.)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoMeshSetRecombine, gmsh.lib), Cvoid,
          (Cint, Cint, Cdouble, Ptr{Cint}),
          dim, tag, angle, ierr)
    ierr[] != 0 && error("gmshModelGeoMeshSetRecombine returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.mesh.setSmoothing(dim, tag, val)

Set a smoothing meshing constraint on the model entity of dimension `dim` and
tag `tag`. `val` iterations of a Laplace smoother are applied.
"""
function setSmoothing(dim, tag, val)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoMeshSetSmoothing, gmsh.lib), Cvoid,
          (Cint, Cint, Cint, Ptr{Cint}),
          dim, tag, val, ierr)
    ierr[] != 0 && error("gmshModelGeoMeshSetSmoothing returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.geo.mesh.setReverse(dim, tag, val = true)

Set a reverse meshing constraint on the model entity of dimension `dim` and tag
`tag`. If `val` is true, the mesh orientation will be reversed with respect to
the natural mesh orientation (i.e. the orientation consistent with the
orientation of the geometry). If `val` is false, the mesh is left as-is.
"""
function setReverse(dim, tag, val = true)
    ierr = Ref{Cint}()
    ccall((:gmshModelGeoMeshSetReverse, gmsh.lib), Cvoid,
          (Cint, Cint, Cint, Ptr{Cint}),
          dim, tag, val, ierr)
    ierr[] != 0 && error("gmshModelGeoMeshSetReverse returned non-zero error code: $(ierr[])")
    return nothing
end

end # end of module mesh

end # end of module geo

"""
    module gmsh.model.occ

OpenCASCADE CAD kernel functions
"""
module occ

import ...gmsh

"""
    gmsh.model.occ.addPoint(x, y, z, meshSize = 0., tag = -1)

Add a geometrical point in the OpenCASCADE CAD representation, at coordinates
(`x`, `y`, `z`). If `meshSize` is > 0, add a meshing constraint at that point.
If `tag` is positive, set the tag explicitly; otherwise a new tag is selected
automatically. Return the tag of the point. (Note that the point will be added
in the current model only after `synchronize` is called. This behavior holds for
all the entities added in the occ module.)

Return an integer value.
"""
function addPoint(x, y, z, meshSize = 0., tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddPoint, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cint, Ptr{Cint}),
          x, y, z, meshSize, tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddPoint returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addLine(startTag, endTag, tag = -1)

Add a straight line segment between the two points with tags `startTag` and
`endTag`. If `tag` is positive, set the tag explicitly; otherwise a new tag is
selected automatically. Return the tag of the line.

Return an integer value.
"""
function addLine(startTag, endTag, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddLine, gmsh.lib), Cint,
          (Cint, Cint, Cint, Ptr{Cint}),
          startTag, endTag, tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddLine returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addCircleArc(startTag, centerTag, endTag, tag = -1)

Add a circle arc between the two points with tags `startTag` and `endTag`, with
center `centerTag`. If `tag` is positive, set the tag explicitly; otherwise a
new tag is selected automatically. Return the tag of the circle arc.

Return an integer value.
"""
function addCircleArc(startTag, centerTag, endTag, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddCircleArc, gmsh.lib), Cint,
          (Cint, Cint, Cint, Cint, Ptr{Cint}),
          startTag, centerTag, endTag, tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddCircleArc returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addCircle(x, y, z, r, tag = -1, angle1 = 0., angle2 = 2*pi)

Add a circle of center (`x`, `y`, `z`) and radius `r`. If `tag` is positive, set
the tag explicitly; otherwise a new tag is selected automatically. If `angle1`
and `angle2` are specified, create a circle arc between the two angles. Return
the tag of the circle.

Return an integer value.
"""
function addCircle(x, y, z, r, tag = -1, angle1 = 0., angle2 = 2*pi)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddCircle, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Cdouble, Ptr{Cint}),
          x, y, z, r, tag, angle1, angle2, ierr)
    ierr[] != 0 && error("gmshModelOccAddCircle returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addEllipseArc(startTag, centerTag, majorTag, endTag, tag = -1)

Add an ellipse arc between the two points `startTag` and `endTag`, with center
`centerTag` and major axis point `majorTag`. If `tag` is positive, set the tag
explicitly; otherwise a new tag is selected automatically. Return the tag of the
ellipse arc. Note that OpenCASCADE does not allow creating ellipse arcs with the
major radius smaller than the minor radius.

Return an integer value.
"""
function addEllipseArc(startTag, centerTag, majorTag, endTag, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddEllipseArc, gmsh.lib), Cint,
          (Cint, Cint, Cint, Cint, Cint, Ptr{Cint}),
          startTag, centerTag, majorTag, endTag, tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddEllipseArc returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addEllipse(x, y, z, r1, r2, tag = -1, angle1 = 0., angle2 = 2*pi)

Add an ellipse of center (`x`, `y`, `z`) and radii `r1` and `r2` along the x-
and y-axes respectively. If `tag` is positive, set the tag explicitly; otherwise
a new tag is selected automatically. If `angle1` and `angle2` are specified,
create an ellipse arc between the two angles. Return the tag of the ellipse.
Note that OpenCASCADE does not allow creating ellipses with the major radius
(along the x-axis) smaller than or equal to the minor radius (along the y-axis):
rotate the shape or use `addCircle` in such cases.

Return an integer value.
"""
function addEllipse(x, y, z, r1, r2, tag = -1, angle1 = 0., angle2 = 2*pi)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddEllipse, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Cdouble, Ptr{Cint}),
          x, y, z, r1, r2, tag, angle1, angle2, ierr)
    ierr[] != 0 && error("gmshModelOccAddEllipse returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addSpline(pointTags, tag = -1)

Add a spline (C2 b-spline) curve going through the points `pointTags`. If `tag`
is positive, set the tag explicitly; otherwise a new tag is selected
automatically. Create a periodic curve if the first and last points are the
same. Return the tag of the spline curve.

Return an integer value.
"""
function addSpline(pointTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddSpline, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, pointTags), length(pointTags), tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddSpline returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addBSpline(pointTags, tag = -1, degree = 3, weights = Cdouble[], knots = Cdouble[], multiplicities = Cint[])

Add a b-spline curve of degree `degree` with `pointTags` control points. If
`weights`, `knots` or `multiplicities` are not provided, default parameters are
computed automatically. If `tag` is positive, set the tag explicitly; otherwise
a new tag is selected automatically. Create a periodic curve if the first and
last points are the same. Return the tag of the b-spline curve.

Return an integer value.
"""
function addBSpline(pointTags, tag = -1, degree = 3, weights = Cdouble[], knots = Cdouble[], multiplicities = Cint[])
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddBSpline, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Cint, Ptr{Cdouble}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cint}),
          convert(Vector{Cint}, pointTags), length(pointTags), tag, degree, convert(Vector{Cdouble}, weights), length(weights), convert(Vector{Cdouble}, knots), length(knots), convert(Vector{Cint}, multiplicities), length(multiplicities), ierr)
    ierr[] != 0 && error("gmshModelOccAddBSpline returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addBezier(pointTags, tag = -1)

Add a Bezier curve with `pointTags` control points. If `tag` is positive, set
the tag explicitly; otherwise a new tag is selected automatically. Return the
tag of the Bezier curve.

Return an integer value.
"""
function addBezier(pointTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddBezier, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, pointTags), length(pointTags), tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddBezier returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addWire(curveTags, tag = -1, checkClosed = false)

Add a wire (open or closed) formed by the curves `curveTags`. Note that an
OpenCASCADE wire can be made of curves that share geometrically identical (but
topologically different) points. If `tag` is positive, set the tag explicitly;
otherwise a new tag is selected automatically. Return the tag of the wire.

Return an integer value.
"""
function addWire(curveTags, tag = -1, checkClosed = false)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddWire, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, curveTags), length(curveTags), tag, checkClosed, ierr)
    ierr[] != 0 && error("gmshModelOccAddWire returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addCurveLoop(curveTags, tag = -1)

Add a curve loop (a closed wire) formed by the curves `curveTags`. `curveTags`
should contain tags of curves forming a closed loop. Note that an OpenCASCADE
curve loop can be made of curves that share geometrically identical (but
topologically different) points. If `tag` is positive, set the tag explicitly;
otherwise a new tag is selected automatically. Return the tag of the curve loop.

Return an integer value.
"""
function addCurveLoop(curveTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddCurveLoop, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, curveTags), length(curveTags), tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddCurveLoop returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addRectangle(x, y, z, dx, dy, tag = -1, roundedRadius = 0.)

Add a rectangle with lower left corner at (`x`, `y`, `z`) and upper right corner
at (`x` + `dx`, `y` + `dy`, `z`). If `tag` is positive, set the tag explicitly;
otherwise a new tag is selected automatically. Round the corners if
`roundedRadius` is nonzero. Return the tag of the rectangle.

Return an integer value.
"""
function addRectangle(x, y, z, dx, dy, tag = -1, roundedRadius = 0.)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddRectangle, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Ptr{Cint}),
          x, y, z, dx, dy, tag, roundedRadius, ierr)
    ierr[] != 0 && error("gmshModelOccAddRectangle returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addDisk(xc, yc, zc, rx, ry, tag = -1)

Add a disk with center (`xc`, `yc`, `zc`) and radius `rx` along the x-axis and
`ry` along the y-axis. If `tag` is positive, set the tag explicitly; otherwise a
new tag is selected automatically. Return the tag of the disk.

Return an integer value.
"""
function addDisk(xc, yc, zc, rx, ry, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddDisk, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Ptr{Cint}),
          xc, yc, zc, rx, ry, tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddDisk returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addPlaneSurface(wireTags, tag = -1)

Add a plane surface defined by one or more curve loops (or closed wires)
`wireTags`. The first curve loop defines the exterior contour; additional curve
loop define holes. If `tag` is positive, set the tag explicitly; otherwise a new
tag is selected automatically. Return the tag of the surface.

Return an integer value.
"""
function addPlaneSurface(wireTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddPlaneSurface, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, wireTags), length(wireTags), tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddPlaneSurface returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addSurfaceFilling(wireTag, tag = -1, pointTags = Cint[])

Add a surface filling the curve loops in `wireTags`. If `tag` is positive, set
the tag explicitly; otherwise a new tag is selected automatically. Return the
tag of the surface. If `pointTags` are provided, force the surface to pass
through the given points.

Return an integer value.
"""
function addSurfaceFilling(wireTag, tag = -1, pointTags = Cint[])
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddSurfaceFilling, gmsh.lib), Cint,
          (Cint, Cint, Ptr{Cint}, Csize_t, Ptr{Cint}),
          wireTag, tag, convert(Vector{Cint}, pointTags), length(pointTags), ierr)
    ierr[] != 0 && error("gmshModelOccAddSurfaceFilling returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addSurfaceLoop(surfaceTags, tag = -1, sewing = false)

Add a surface loop (a closed shell) formed by `surfaceTags`.  If `tag` is
positive, set the tag explicitly; otherwise a new tag is selected automatically.
Return the tag of the surface loop. Setting `sewing` allows to build a shell
made of surfaces that share geometrically identical (but topologically
different) curves.

Return an integer value.
"""
function addSurfaceLoop(surfaceTags, tag = -1, sewing = false)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddSurfaceLoop, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, surfaceTags), length(surfaceTags), tag, sewing, ierr)
    ierr[] != 0 && error("gmshModelOccAddSurfaceLoop returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addVolume(shellTags, tag = -1)

Add a volume (a region) defined by one or more surface loops `shellTags`. The
first surface loop defines the exterior boundary; additional surface loop define
holes. If `tag` is positive, set the tag explicitly; otherwise a new tag is
selected automatically. Return the tag of the volume.

Return an integer value.
"""
function addVolume(shellTags, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddVolume, gmsh.lib), Cint,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, shellTags), length(shellTags), tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddVolume returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addSphere(xc, yc, zc, radius, tag = -1, angle1 = -pi/2, angle2 = pi/2, angle3 = 2*pi)

Add a sphere of center (`xc`, `yc`, `zc`) and radius `r`. The optional `angle1`
and `angle2` arguments define the polar angle opening (from -Pi/2 to Pi/2). The
optional `angle3` argument defines the azimuthal opening (from 0 to 2*Pi). If
`tag` is positive, set the tag explicitly; otherwise a new tag is selected
automatically. Return the tag of the sphere.

Return an integer value.
"""
function addSphere(xc, yc, zc, radius, tag = -1, angle1 = -pi/2, angle2 = pi/2, angle3 = 2*pi)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddSphere, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          xc, yc, zc, radius, tag, angle1, angle2, angle3, ierr)
    ierr[] != 0 && error("gmshModelOccAddSphere returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addBox(x, y, z, dx, dy, dz, tag = -1)

Add a parallelepipedic box defined by a point (`x`, `y`, `z`) and the extents
along the x-, y- and z-axes. If `tag` is positive, set the tag explicitly;
otherwise a new tag is selected automatically. Return the tag of the box.

Return an integer value.
"""
function addBox(x, y, z, dx, dy, dz, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddBox, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Ptr{Cint}),
          x, y, z, dx, dy, dz, tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddBox returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addCylinder(x, y, z, dx, dy, dz, r, tag = -1, angle = 2*pi)

Add a cylinder, defined by the center (`x`, `y`, `z`) of its first circular
face, the 3 components (`dx`, `dy`, `dz`) of the vector defining its axis and
its radius `r`. The optional `angle` argument defines the angular opening (from
0 to 2*Pi). If `tag` is positive, set the tag explicitly; otherwise a new tag is
selected automatically. Return the tag of the cylinder.

Return an integer value.
"""
function addCylinder(x, y, z, dx, dy, dz, r, tag = -1, angle = 2*pi)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddCylinder, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Ptr{Cint}),
          x, y, z, dx, dy, dz, r, tag, angle, ierr)
    ierr[] != 0 && error("gmshModelOccAddCylinder returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addCone(x, y, z, dx, dy, dz, r1, r2, tag = -1, angle = 2*pi)

Add a cone, defined by the center (`x`, `y`, `z`) of its first circular face,
the 3 components of the vector (`dx`, `dy`, `dz`) defining its axis and the two
radii `r1` and `r2` of the faces (these radii can be zero). If `tag` is
positive, set the tag explicitly; otherwise a new tag is selected automatically.
`angle` defines the optional angular opening (from 0 to 2*Pi). Return the tag of
the cone.

Return an integer value.
"""
function addCone(x, y, z, dx, dy, dz, r1, r2, tag = -1, angle = 2*pi)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddCone, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Ptr{Cint}),
          x, y, z, dx, dy, dz, r1, r2, tag, angle, ierr)
    ierr[] != 0 && error("gmshModelOccAddCone returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addWedge(x, y, z, dx, dy, dz, tag = -1, ltx = 0.)

Add a right angular wedge, defined by the right-angle point (`x`, `y`, `z`) and
the 3 extends along the x-, y- and z-axes (`dx`, `dy`, `dz`). If `tag` is
positive, set the tag explicitly; otherwise a new tag is selected automatically.
The optional argument `ltx` defines the top extent along the x-axis. Return the
tag of the wedge.

Return an integer value.
"""
function addWedge(x, y, z, dx, dy, dz, tag = -1, ltx = 0.)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddWedge, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Ptr{Cint}),
          x, y, z, dx, dy, dz, tag, ltx, ierr)
    ierr[] != 0 && error("gmshModelOccAddWedge returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addTorus(x, y, z, r1, r2, tag = -1, angle = 2*pi)

Add a torus, defined by its center (`x`, `y`, `z`) and its 2 radii `r` and `r2`.
If `tag` is positive, set the tag explicitly; otherwise a new tag is selected
automatically. The optional argument `angle` defines the angular opening (from 0
to 2*Pi). Return the tag of the wedge.

Return an integer value.
"""
function addTorus(x, y, z, r1, r2, tag = -1, angle = 2*pi)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshModelOccAddTorus, gmsh.lib), Cint,
          (Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cint, Cdouble, Ptr{Cint}),
          x, y, z, r1, r2, tag, angle, ierr)
    ierr[] != 0 && error("gmshModelOccAddTorus returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.model.occ.addThruSections(wireTags, tag = -1, makeSolid = true, makeRuled = false)

Add a volume (if the optional argument `makeSolid` is set) or surfaces defined
through the open or closed wires `wireTags`. If `tag` is positive, set the tag
explicitly; otherwise a new tag is selected automatically. The new entities are
returned in `outDimTags`. If the optional argument `makeRuled` is set, the
surfaces created on the boundary are forced to be ruled surfaces.

Return `outDimTags`.
"""
function addThruSections(wireTags, tag = -1, makeSolid = true, makeRuled = false)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccAddThruSections, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, wireTags), length(wireTags), api_outDimTags_, api_outDimTags_n_, tag, makeSolid, makeRuled, ierr)
    ierr[] != 0 && error("gmshModelOccAddThruSections returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.addThickSolid(volumeTag, excludeSurfaceTags, offset, tag = -1)

Add a hollowed volume built from an initial volume `volumeTag` and a set of
faces from this volume `excludeSurfaceTags`, which are to be removed. The
remaining faces of the volume become the walls of the hollowed solid, with
thickness `offset`. If `tag` is positive, set the tag explicitly; otherwise a
new tag is selected automatically.

Return `outDimTags`.
"""
function addThickSolid(volumeTag, excludeSurfaceTags, offset, tag = -1)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccAddThickSolid, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}, Csize_t, Cdouble, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          volumeTag, convert(Vector{Cint}, excludeSurfaceTags), length(excludeSurfaceTags), offset, api_outDimTags_, api_outDimTags_n_, tag, ierr)
    ierr[] != 0 && error("gmshModelOccAddThickSolid returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.extrude(dimTags, dx, dy, dz, numElements = Cint[], heights = Cdouble[], recombine = false)

Extrude the model entities `dimTags` by translation along (`dx`, `dy`, `dz`).
Return extruded entities in `outDimTags`. If `numElements` is not empty, also
extrude the mesh: the entries in `numElements` give the number of elements in
each layer. If `height` is not empty, it provides the (cumulative) height of the
different layers, normalized to 1.

Return `outDimTags`.
"""
function extrude(dimTags, dx, dy, dz, numElements = Cint[], heights = Cdouble[], recombine = false)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccExtrude, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), dx, dy, dz, api_outDimTags_, api_outDimTags_n_, convert(Vector{Cint}, numElements), length(numElements), convert(Vector{Cdouble}, heights), length(heights), recombine, ierr)
    ierr[] != 0 && error("gmshModelOccExtrude returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.revolve(dimTags, x, y, z, ax, ay, az, angle, numElements = Cint[], heights = Cdouble[], recombine = false)

Extrude the model entities `dimTags` by rotation of `angle` radians around the
axis of revolution defined by the point (`x`, `y`, `z`) and the direction (`ax`,
`ay`, `az`). Return extruded entities in `outDimTags`. If `numElements` is not
empty, also extrude the mesh: the entries in `numElements` give the number of
elements in each layer. If `height` is not empty, it provides the (cumulative)
height of the different layers, normalized to 1. When the mesh is extruded the
angle should be strictly smaller than 2*Pi.

Return `outDimTags`.
"""
function revolve(dimTags, x, y, z, ax, ay, az, angle, numElements = Cint[], heights = Cdouble[], recombine = false)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccRevolve, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), x, y, z, ax, ay, az, angle, api_outDimTags_, api_outDimTags_n_, convert(Vector{Cint}, numElements), length(numElements), convert(Vector{Cdouble}, heights), length(heights), recombine, ierr)
    ierr[] != 0 && error("gmshModelOccRevolve returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.addPipe(dimTags, wireTag)

Add a pipe by extruding the entities `dimTags` along the wire `wireTag`. Return
the pipe in `outDimTags`.

Return `outDimTags`.
"""
function addPipe(dimTags, wireTag)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccAddPipe, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), wireTag, api_outDimTags_, api_outDimTags_n_, ierr)
    ierr[] != 0 && error("gmshModelOccAddPipe returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.fillet(volumeTags, curveTags, radii, removeVolume = true)

Fillet the volumes `volumeTags` on the curves `curveTags` with radii `radii`.
The `radii` vector can either contain a single radius, as many radii as
`curveTags`, or twice as many as `curveTags` (in which case different radii are
provided for the begin and end points of the curves). Return the filleted
entities in `outDimTags`. Remove the original volume if `removeVolume` is set.

Return `outDimTags`.
"""
function fillet(volumeTags, curveTags, radii, removeVolume = true)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccFillet, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          convert(Vector{Cint}, volumeTags), length(volumeTags), convert(Vector{Cint}, curveTags), length(curveTags), convert(Vector{Cdouble}, radii), length(radii), api_outDimTags_, api_outDimTags_n_, removeVolume, ierr)
    ierr[] != 0 && error("gmshModelOccFillet returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.chamfer(volumeTags, curveTags, surfaceTags, distances, removeVolume = true)

Chamfer the volumes `volumeTags` on the curves `curveTags` with distances
`distances` measured on surfaces `surfaceTags`. The `distances` vector can
either contain a single distance, as many distances as `curveTags` and
`surfaceTags`, or twice as many as `curveTags` and `surfaceTags` (in which case
the first in each pair is measured on the corresponding surface in
`surfaceTags`, the other on the other adjacent surface). Return the chamfered
entities in `outDimTags`. Remove the original volume if `removeVolume` is set.

Return `outDimTags`.
"""
function chamfer(volumeTags, curveTags, surfaceTags, distances, removeVolume = true)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccChamfer, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          convert(Vector{Cint}, volumeTags), length(volumeTags), convert(Vector{Cint}, curveTags), length(curveTags), convert(Vector{Cint}, surfaceTags), length(surfaceTags), convert(Vector{Cdouble}, distances), length(distances), api_outDimTags_, api_outDimTags_n_, removeVolume, ierr)
    ierr[] != 0 && error("gmshModelOccChamfer returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.fuse(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)

Compute the boolean union (the fusion) of the entities `objectDimTags` and
`toolDimTags`. Return the resulting entities in `outDimTags`. If `tag` is
positive, try to set the tag explicitly (only valid if the boolean operation
results in a single entity). Remove the object if `removeObject` is set. Remove
the tool if `removeTool` is set.

Return `outDimTags`, `outDimTagsMap`.
"""
function fuse(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    api_outDimTagsMap_ = Ref{Ptr{Ptr{Cint}}}()
    api_outDimTagsMap_n_ = Ref{Ptr{Csize_t}}()
    api_outDimTagsMap_nn_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccFuse, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Cint}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(objectDimTags))), 2 * length(objectDimTags), convert(Vector{Cint}, collect(Cint, Iterators.flatten(toolDimTags))), 2 * length(toolDimTags), api_outDimTags_, api_outDimTags_n_, api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_, tag, removeObject, removeTool, ierr)
    ierr[] != 0 && error("gmshModelOccFuse returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    tmp_api_outDimTagsMap_ = unsafe_wrap(Array, api_outDimTagsMap_[], api_outDimTagsMap_nn_[], own=true)
    tmp_api_outDimTagsMap_n_ = unsafe_wrap(Array, api_outDimTagsMap_n_[], api_outDimTagsMap_nn_[], own=true)
    outDimTagsMap = Vector{Tuple{Cint,Cint}}[]
    resize!(outDimTagsMap, api_outDimTagsMap_nn_[])
    for i in 1:api_outDimTagsMap_nn_[]
        tmp = unsafe_wrap(Array, tmp_api_outDimTagsMap_[i], tmp_api_outDimTagsMap_n_[i], own=true)
        outDimTagsMap[i] = [(tmp[i], tmp[i+1]) for i in 1:2:length(tmp)]
    end
    return outDimTags, outDimTagsMap
end

"""
    gmsh.model.occ.intersect(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)

Compute the boolean intersection (the common parts) of the entities
`objectDimTags` and `toolDimTags`. Return the resulting entities in
`outDimTags`. If `tag` is positive, try to set the tag explicitly (only valid if
the boolean operation results in a single entity). Remove the object if
`removeObject` is set. Remove the tool if `removeTool` is set.

Return `outDimTags`, `outDimTagsMap`.
"""
function intersect(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    api_outDimTagsMap_ = Ref{Ptr{Ptr{Cint}}}()
    api_outDimTagsMap_n_ = Ref{Ptr{Csize_t}}()
    api_outDimTagsMap_nn_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccIntersect, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Cint}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(objectDimTags))), 2 * length(objectDimTags), convert(Vector{Cint}, collect(Cint, Iterators.flatten(toolDimTags))), 2 * length(toolDimTags), api_outDimTags_, api_outDimTags_n_, api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_, tag, removeObject, removeTool, ierr)
    ierr[] != 0 && error("gmshModelOccIntersect returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    tmp_api_outDimTagsMap_ = unsafe_wrap(Array, api_outDimTagsMap_[], api_outDimTagsMap_nn_[], own=true)
    tmp_api_outDimTagsMap_n_ = unsafe_wrap(Array, api_outDimTagsMap_n_[], api_outDimTagsMap_nn_[], own=true)
    outDimTagsMap = Vector{Tuple{Cint,Cint}}[]
    resize!(outDimTagsMap, api_outDimTagsMap_nn_[])
    for i in 1:api_outDimTagsMap_nn_[]
        tmp = unsafe_wrap(Array, tmp_api_outDimTagsMap_[i], tmp_api_outDimTagsMap_n_[i], own=true)
        outDimTagsMap[i] = [(tmp[i], tmp[i+1]) for i in 1:2:length(tmp)]
    end
    return outDimTags, outDimTagsMap
end

"""
    gmsh.model.occ.cut(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)

Compute the boolean difference between the entities `objectDimTags` and
`toolDimTags`. Return the resulting entities in `outDimTags`. If `tag` is
positive, try to set the tag explicitly (only valid if the boolean operation
results in a single entity). Remove the object if `removeObject` is set. Remove
the tool if `removeTool` is set.

Return `outDimTags`, `outDimTagsMap`.
"""
function cut(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    api_outDimTagsMap_ = Ref{Ptr{Ptr{Cint}}}()
    api_outDimTagsMap_n_ = Ref{Ptr{Csize_t}}()
    api_outDimTagsMap_nn_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccCut, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Cint}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(objectDimTags))), 2 * length(objectDimTags), convert(Vector{Cint}, collect(Cint, Iterators.flatten(toolDimTags))), 2 * length(toolDimTags), api_outDimTags_, api_outDimTags_n_, api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_, tag, removeObject, removeTool, ierr)
    ierr[] != 0 && error("gmshModelOccCut returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    tmp_api_outDimTagsMap_ = unsafe_wrap(Array, api_outDimTagsMap_[], api_outDimTagsMap_nn_[], own=true)
    tmp_api_outDimTagsMap_n_ = unsafe_wrap(Array, api_outDimTagsMap_n_[], api_outDimTagsMap_nn_[], own=true)
    outDimTagsMap = Vector{Tuple{Cint,Cint}}[]
    resize!(outDimTagsMap, api_outDimTagsMap_nn_[])
    for i in 1:api_outDimTagsMap_nn_[]
        tmp = unsafe_wrap(Array, tmp_api_outDimTagsMap_[i], tmp_api_outDimTagsMap_n_[i], own=true)
        outDimTagsMap[i] = [(tmp[i], tmp[i+1]) for i in 1:2:length(tmp)]
    end
    return outDimTags, outDimTagsMap
end

"""
    gmsh.model.occ.fragment(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)

Compute the boolean fragments (general fuse) of the entities `objectDimTags` and
`toolDimTags`. Return the resulting entities in `outDimTags`. If `tag` is
positive, try to set the tag explicitly (only valid if the boolean operation
results in a single entity). Remove the object if `removeObject` is set. Remove
the tool if `removeTool` is set.

Return `outDimTags`, `outDimTagsMap`.
"""
function fragment(objectDimTags, toolDimTags, tag = -1, removeObject = true, removeTool = true)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    api_outDimTagsMap_ = Ref{Ptr{Ptr{Cint}}}()
    api_outDimTagsMap_n_ = Ref{Ptr{Csize_t}}()
    api_outDimTagsMap_nn_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccFragment, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Cint}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Cint, Cint, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(objectDimTags))), 2 * length(objectDimTags), convert(Vector{Cint}, collect(Cint, Iterators.flatten(toolDimTags))), 2 * length(toolDimTags), api_outDimTags_, api_outDimTags_n_, api_outDimTagsMap_, api_outDimTagsMap_n_, api_outDimTagsMap_nn_, tag, removeObject, removeTool, ierr)
    ierr[] != 0 && error("gmshModelOccFragment returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    tmp_api_outDimTagsMap_ = unsafe_wrap(Array, api_outDimTagsMap_[], api_outDimTagsMap_nn_[], own=true)
    tmp_api_outDimTagsMap_n_ = unsafe_wrap(Array, api_outDimTagsMap_n_[], api_outDimTagsMap_nn_[], own=true)
    outDimTagsMap = Vector{Tuple{Cint,Cint}}[]
    resize!(outDimTagsMap, api_outDimTagsMap_nn_[])
    for i in 1:api_outDimTagsMap_nn_[]
        tmp = unsafe_wrap(Array, tmp_api_outDimTagsMap_[i], tmp_api_outDimTagsMap_n_[i], own=true)
        outDimTagsMap[i] = [(tmp[i], tmp[i+1]) for i in 1:2:length(tmp)]
    end
    return outDimTags, outDimTagsMap
end

"""
    gmsh.model.occ.translate(dimTags, dx, dy, dz)

Translate the model entities `dimTags` along (`dx`, `dy`, `dz`).
"""
function translate(dimTags, dx, dy, dz)
    ierr = Ref{Cint}()
    ccall((:gmshModelOccTranslate, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), dx, dy, dz, ierr)
    ierr[] != 0 && error("gmshModelOccTranslate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.rotate(dimTags, x, y, z, ax, ay, az, angle)

Rotate the model entities `dimTags` of `angle` radians around the axis of
revolution defined by the point (`x`, `y`, `z`) and the direction (`ax`, `ay`,
`az`).
"""
function rotate(dimTags, x, y, z, ax, ay, az, angle)
    ierr = Ref{Cint}()
    ccall((:gmshModelOccRotate, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), x, y, z, ax, ay, az, angle, ierr)
    ierr[] != 0 && error("gmshModelOccRotate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.dilate(dimTags, x, y, z, a, b, c)

Scale the model entities `dimTag` by factors `a`, `b` and `c` along the three
coordinate axes; use (`x`, `y`, `z`) as the center of the homothetic
transformation.
"""
function dilate(dimTags, x, y, z, a, b, c)
    ierr = Ref{Cint}()
    ccall((:gmshModelOccDilate, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), x, y, z, a, b, c, ierr)
    ierr[] != 0 && error("gmshModelOccDilate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.symmetrize(dimTags, a, b, c, d)

Apply a symmetry transformation to the model entities `dimTag`, with respect to
the plane of equation `a` * x + `b` * y + `c` * z + `d` = 0.
"""
function symmetrize(dimTags, a, b, c, d)
    ierr = Ref{Cint}()
    ccall((:gmshModelOccSymmetrize, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Cdouble, Cdouble, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), a, b, c, d, ierr)
    ierr[] != 0 && error("gmshModelOccSymmetrize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.affineTransform(dimTags, a)

Apply a general affine transformation matrix `a` (16 entries of a 4x4 matrix, by
row; only the 12 first can be provided for convenience) to the model entities
`dimTag`.
"""
function affineTransform(dimTags, a)
    ierr = Ref{Cint}()
    ccall((:gmshModelOccAffineTransform, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), convert(Vector{Cdouble}, a), length(a), ierr)
    ierr[] != 0 && error("gmshModelOccAffineTransform returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.copy(dimTags)

Copy the entities `dimTags`; the new entities are returned in `outDimTags`.

Return `outDimTags`.
"""
function copy(dimTags)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccCopy, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), api_outDimTags_, api_outDimTags_n_, ierr)
    ierr[] != 0 && error("gmshModelOccCopy returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.remove(dimTags, recursive = false)

Remove the entities `dimTags`. If `recursive` is true, remove all the entities
on their boundaries, down to dimension 0.
"""
function remove(dimTags, recursive = false)
    ierr = Ref{Cint}()
    ccall((:gmshModelOccRemove, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cint, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), recursive, ierr)
    ierr[] != 0 && error("gmshModelOccRemove returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.removeAllDuplicates()

Remove all duplicate entities (different entities at the same geometrical
location) after intersecting (using boolean fragments) all highest dimensional
entities.
"""
function removeAllDuplicates()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccRemoveAllDuplicates, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelOccRemoveAllDuplicates returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.healShapes(dimTags = Tuple{Cint,Cint}[], tolerance = 1e-8, fixDegenerated = true, fixSmallEdges = true, fixSmallFaces = true, sewFaces = true)

Apply various healing procedures to the entities `dimTags` (or to all the
entities in the model if `dimTags` is empty). Return the healed entities in
`outDimTags`. Available healing options are listed in the Gmsh reference manual.

Return `outDimTags`.
"""
function healShapes(dimTags = Tuple{Cint,Cint}[], tolerance = 1e-8, fixDegenerated = true, fixSmallEdges = true, fixSmallFaces = true, sewFaces = true)
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccHealShapes, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}, Csize_t, Cdouble, Cint, Cint, Cint, Cint, Ptr{Cint}),
          api_outDimTags_, api_outDimTags_n_, convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), tolerance, fixDegenerated, fixSmallEdges, fixSmallFaces, sewFaces, ierr)
    ierr[] != 0 && error("gmshModelOccHealShapes returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.importShapes(fileName, highestDimOnly = true, format = "")

Import BREP, STEP or IGES shapes from the file `fileName`. The imported entities
are returned in `outDimTags`. If the optional argument `highestDimOnly` is set,
only import the highest dimensional entities in the file. The optional argument
`format` can be used to force the format of the file (currently "brep", "step"
or "iges").

Return `outDimTags`.
"""
function importShapes(fileName, highestDimOnly = true, format = "")
    api_outDimTags_ = Ref{Ptr{Cint}}()
    api_outDimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccImportShapes, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cchar}, Ptr{Cint}),
          fileName, api_outDimTags_, api_outDimTags_n_, highestDimOnly, format, ierr)
    ierr[] != 0 && error("gmshModelOccImportShapes returned non-zero error code: $(ierr[])")
    tmp_api_outDimTags_ = unsafe_wrap(Array, api_outDimTags_[], api_outDimTags_n_[], own=true)
    outDimTags = [ (tmp_api_outDimTags_[i], tmp_api_outDimTags_[i+1]) for i in 1:2:length(tmp_api_outDimTags_) ]
    return outDimTags
end

"""
    gmsh.model.occ.setMeshSize(dimTags, size)

Set a mesh size constraint on the model entities `dimTags`. Currently only
entities of dimension 0 (points) are handled.
"""
function setMeshSize(dimTags, size)
    ierr = Ref{Cint}()
    ccall((:gmshModelOccSetMeshSize, gmsh.lib), Cvoid,
          (Ptr{Cint}, Csize_t, Cdouble, Ptr{Cint}),
          convert(Vector{Cint}, collect(Cint, Iterators.flatten(dimTags))), 2 * length(dimTags), size, ierr)
    ierr[] != 0 && error("gmshModelOccSetMeshSize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.model.occ.getMass(dim, tag)

Get the mass of the model entity of dimension `dim` and tag `tag`.

Return `mass`.
"""
function getMass(dim, tag)
    api_mass_ = Ref{Cdouble}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccGetMass, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cdouble}, Ptr{Cint}),
          dim, tag, api_mass_, ierr)
    ierr[] != 0 && error("gmshModelOccGetMass returned non-zero error code: $(ierr[])")
    return api_mass_[]
end

"""
    gmsh.model.occ.getCenterOfMass(dim, tag)

Get the center of mass of the model entity of dimension `dim` and tag `tag`.

Return `x`, `y`, `z`.
"""
function getCenterOfMass(dim, tag)
    api_x_ = Ref{Cdouble}()
    api_y_ = Ref{Cdouble}()
    api_z_ = Ref{Cdouble}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccGetCenterOfMass, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cdouble}, Ptr{Cint}),
          dim, tag, api_x_, api_y_, api_z_, ierr)
    ierr[] != 0 && error("gmshModelOccGetCenterOfMass returned non-zero error code: $(ierr[])")
    return api_x_[], api_y_[], api_z_[]
end

"""
    gmsh.model.occ.getMatrixOfInertia(dim, tag)

Get the matrix of inertia (by row) of the model entity of dimension `dim` and
tag `tag`.

Return `mat`.
"""
function getMatrixOfInertia(dim, tag)
    api_mat_ = Ref{Ptr{Cdouble}}()
    api_mat_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccGetMatrixOfInertia, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          dim, tag, api_mat_, api_mat_n_, ierr)
    ierr[] != 0 && error("gmshModelOccGetMatrixOfInertia returned non-zero error code: $(ierr[])")
    mat = unsafe_wrap(Array, api_mat_[], api_mat_n_[], own=true)
    return mat
end

"""
    gmsh.model.occ.synchronize()

Synchronize the OpenCASCADE CAD representation with the current Gmsh model. This
can be called at any time, but since it involves a non trivial amount of
processing, the number of synchronization points should normally be minimized.
"""
function synchronize()
    ierr = Ref{Cint}()
    ccall((:gmshModelOccSynchronize, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshModelOccSynchronize returned non-zero error code: $(ierr[])")
    return nothing
end

end # end of module occ

end # end of module model

"""
    module gmsh.view

Post-processing view functions
"""
module view

import ..gmsh

"""
    gmsh.view.add(name, tag = -1)

Add a new post-processing view, with name `name`. If `tag` is positive use it
(and remove the view with that tag if it already exists), otherwise associate a
new tag. Return the view tag.

Return an integer value.
"""
function add(name, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshViewAdd, gmsh.lib), Cint,
          (Ptr{Cchar}, Cint, Ptr{Cint}),
          name, tag, ierr)
    ierr[] != 0 && error("gmshViewAdd returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.view.remove(tag)

Remove the view with tag `tag`.
"""
function remove(tag)
    ierr = Ref{Cint}()
    ccall((:gmshViewRemove, gmsh.lib), Cvoid,
          (Cint, Ptr{Cint}),
          tag, ierr)
    ierr[] != 0 && error("gmshViewRemove returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.view.getIndex(tag)

Get the index of the view with tag `tag` in the list of currently loaded views.
This dynamic index (it can change when views are removed) is used to access view
options.

Return an integer value.
"""
function getIndex(tag)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshViewGetIndex, gmsh.lib), Cint,
          (Cint, Ptr{Cint}),
          tag, ierr)
    ierr[] != 0 && error("gmshViewGetIndex returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.view.getTags()

Get the tags of all views.

Return `tags`.
"""
function getTags()
    api_tags_ = Ref{Ptr{Cint}}()
    api_tags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshViewGetTags, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          api_tags_, api_tags_n_, ierr)
    ierr[] != 0 && error("gmshViewGetTags returned non-zero error code: $(ierr[])")
    tags = unsafe_wrap(Array, api_tags_[], api_tags_n_[], own=true)
    return tags
end

"""
    gmsh.view.addModelData(tag, step, modelName, dataType, tags, data, time = 0., numComponents = -1, partition = 0)

Add model-based post-processing data to the view with tag `tag`. `modelName`
identifies the model the data is attached to. `dataType` specifies the type of
data, currently either "NodeData", "ElementData" or "ElementNodeData". `step`
specifies the identifier (>= 0) of the data in a sequence. `tags` gives the tags
of the nodes or elements in the mesh to which the data is associated. `data` is
a vector of the same length as `tags`: each entry is the vector of double
precision numbers representing the data associated with the corresponding tag.
The optional `time` argument associate a time value with the data.
`numComponents` gives the number of data components (1 for scalar data, 3 for
vector data, etc.) per entity; if negative, it is automatically inferred (when
possible) from the input data. `partition` allows to specify data in several
sub-sets.
"""
function addModelData(tag, step, modelName, dataType, tags, data, time = 0., numComponents = -1, partition = 0)
    api_data_n_ = [ length(data[i]) for i in 1:length(data) ]
    ierr = Ref{Cint}()
    ccall((:gmshViewAddModelData, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cchar}, Ptr{Cchar}, Ptr{Csize_t}, Csize_t, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Csize_t, Cdouble, Cint, Cint, Ptr{Cint}),
          tag, step, modelName, dataType, convert(Vector{Csize_t}, tags), length(tags), convert(Vector{Vector{Cdouble}},data), api_data_n_, length(data), time, numComponents, partition, ierr)
    ierr[] != 0 && error("gmshViewAddModelData returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.view.getModelData(tag, step)

Get model-based post-processing data from the view with tag `tag` at step
`step`. Return the `data` associated to the nodes or the elements with tags
`tags`, as well as the `dataType` and the number of components `numComponents`.

Return `dataType`, `tags`, `data`, `time`, `numComponents`.
"""
function getModelData(tag, step)
    api_dataType_ = Ref{Ptr{Cchar}}()
    api_tags_ = Ref{Ptr{Csize_t}}()
    api_tags_n_ = Ref{Csize_t}()
    api_data_ = Ref{Ptr{Ptr{Cdouble}}}()
    api_data_n_ = Ref{Ptr{Csize_t}}()
    api_data_nn_ = Ref{Csize_t}()
    api_time_ = Ref{Cdouble}()
    api_numComponents_ = Ref{Cint}()
    ierr = Ref{Cint}()
    ccall((:gmshViewGetModelData, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Ptr{Cchar}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Cdouble}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Cdouble}, Ptr{Cint}, Ptr{Cint}),
          tag, step, api_dataType_, api_tags_, api_tags_n_, api_data_, api_data_n_, api_data_nn_, api_time_, api_numComponents_, ierr)
    ierr[] != 0 && error("gmshViewGetModelData returned non-zero error code: $(ierr[])")
    dataType = unsafe_string(api_dataType_[])
    tags = unsafe_wrap(Array, api_tags_[], api_tags_n_[], own=true)
    tmp_api_data_ = unsafe_wrap(Array, api_data_[], api_data_nn_[], own=true)
    tmp_api_data_n_ = unsafe_wrap(Array, api_data_n_[], api_data_nn_[], own=true)
    data = [ unsafe_wrap(Array, tmp_api_data_[i], tmp_api_data_n_[i], own=true) for i in 1:api_data_nn_[] ]
    return dataType, tags, data, api_time_[], api_numComponents_[]
end

"""
    gmsh.view.addListData(tag, dataType, numEle, data)

Add list-based post-processing data to the view with tag `tag`. `dataType`
identifies the data: "SP" for scalar points, "VP", for vector points, etc.
`numEle` gives the number of elements in the data. `data` contains the data for
the `numEle` elements.
"""
function addListData(tag, dataType, numEle, data)
    ierr = Ref{Cint}()
    ccall((:gmshViewAddListData, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Cint, Ptr{Cdouble}, Csize_t, Ptr{Cint}),
          tag, dataType, numEle, convert(Vector{Cdouble}, data), length(data), ierr)
    ierr[] != 0 && error("gmshViewAddListData returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.view.getListData(tag)

Get list-based post-processing data from the view with tag `tag`. Return the
types `dataTypes`, the number of elements `numElements` for each data type and
the `data` for each data type.

Return `dataType`, `numElements`, `data`.
"""
function getListData(tag)
    api_dataType_ = Ref{Ptr{Ptr{Cchar}}}()
    api_dataType_n_ = Ref{Csize_t}()
    api_numElements_ = Ref{Ptr{Cint}}()
    api_numElements_n_ = Ref{Csize_t}()
    api_data_ = Ref{Ptr{Ptr{Cdouble}}}()
    api_data_n_ = Ref{Ptr{Csize_t}}()
    api_data_nn_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshViewGetListData, gmsh.lib), Cvoid,
          (Cint, Ptr{Ptr{Ptr{Cchar}}}, Ptr{Csize_t}, Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Ptr{Ptr{Cdouble}}}, Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Cint}),
          tag, api_dataType_, api_dataType_n_, api_numElements_, api_numElements_n_, api_data_, api_data_n_, api_data_nn_, ierr)
    ierr[] != 0 && error("gmshViewGetListData returned non-zero error code: $(ierr[])")
    tmp_api_dataType_ = unsafe_wrap(Array, api_dataType_[], api_dataType_n_[], own=true)
    dataType = [unsafe_string(tmp_api_dataType_[i]) for i in 1:length(tmp_api_dataType_) ]
    numElements = unsafe_wrap(Array, api_numElements_[], api_numElements_n_[], own=true)
    tmp_api_data_ = unsafe_wrap(Array, api_data_[], api_data_nn_[], own=true)
    tmp_api_data_n_ = unsafe_wrap(Array, api_data_n_[], api_data_nn_[], own=true)
    data = [ unsafe_wrap(Array, tmp_api_data_[i], tmp_api_data_n_[i], own=true) for i in 1:api_data_nn_[] ]
    return dataType, numElements, data
end

"""
    gmsh.view.addAlias(refTag, copyOptions = false, tag = -1)

Add a post-processing view as an `alias` of the reference view with tag
`refTag`. If `copyOptions` is set, copy the options of the reference view. If
`tag` is positive use it (and remove the view with that tag if it already
exists), otherwise associate a new tag. Return the view tag.

Return an integer value.
"""
function addAlias(refTag, copyOptions = false, tag = -1)
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshViewAddAlias, gmsh.lib), Cint,
          (Cint, Cint, Cint, Ptr{Cint}),
          refTag, copyOptions, tag, ierr)
    ierr[] != 0 && error("gmshViewAddAlias returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.view.copyOptions(refTag, tag)

Copy the options from the view with tag `refTag` to the view with tag `tag`.
"""
function copyOptions(refTag, tag)
    ierr = Ref{Cint}()
    ccall((:gmshViewCopyOptions, gmsh.lib), Cvoid,
          (Cint, Cint, Ptr{Cint}),
          refTag, tag, ierr)
    ierr[] != 0 && error("gmshViewCopyOptions returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.view.combine(what, how, remove = false)

Combine elements (if `what` == "elements") or steps (if `what` == "steps") of
all views (`how` == "all"), all visible views (`how` == "visible") or all views
having the same name (`how` == "name"). Remove original views if `remove` is
set.
"""
function combine(what, how, remove = false)
    ierr = Ref{Cint}()
    ccall((:gmshViewCombine, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cchar}, Cint, Ptr{Cint}),
          what, how, remove, ierr)
    ierr[] != 0 && error("gmshViewCombine returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.view.probe(tag, x, y, z, step = -1, numComp = -1, gradient = false, tolerance = 0., xElemCoord = Cdouble[], yElemCoord = Cdouble[], zElemCoord = Cdouble[])

Probe the view `tag` for its `value` at point (`x`, `y`, `z`). Return only the
value at step `step` is `step` is positive. Return only values with `numComp` if
`numComp` is positive. Return the gradient of the `value` if `gradient` is set.
Probes with a geometrical tolerance (in the reference unit cube) of `tolerance`
if `tolerance` is not zero. Return the result from the element described by its
coordinates if `xElementCoord`, `yElementCoord` and `zElementCoord` are
provided.

Return `value`.
"""
function probe(tag, x, y, z, step = -1, numComp = -1, gradient = false, tolerance = 0., xElemCoord = Cdouble[], yElemCoord = Cdouble[], zElemCoord = Cdouble[])
    api_value_ = Ref{Ptr{Cdouble}}()
    api_value_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshViewProbe, gmsh.lib), Cvoid,
          (Cint, Cdouble, Cdouble, Cdouble, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Cint, Cint, Cint, Cdouble, Ptr{Cdouble}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Cdouble}, Csize_t, Ptr{Cint}),
          tag, x, y, z, api_value_, api_value_n_, step, numComp, gradient, tolerance, convert(Vector{Cdouble}, xElemCoord), length(xElemCoord), convert(Vector{Cdouble}, yElemCoord), length(yElemCoord), convert(Vector{Cdouble}, zElemCoord), length(zElemCoord), ierr)
    ierr[] != 0 && error("gmshViewProbe returned non-zero error code: $(ierr[])")
    value = unsafe_wrap(Array, api_value_[], api_value_n_[], own=true)
    return value
end

"""
    gmsh.view.write(tag, fileName, append = false)

Write the view to a file `fileName`. The export format is determined by the file
extension. Append to the file if `append` is set.
"""
function write(tag, fileName, append = false)
    ierr = Ref{Cint}()
    ccall((:gmshViewWrite, gmsh.lib), Cvoid,
          (Cint, Ptr{Cchar}, Cint, Ptr{Cint}),
          tag, fileName, append, ierr)
    ierr[] != 0 && error("gmshViewWrite returned non-zero error code: $(ierr[])")
    return nothing
end

end # end of module view

"""
    module gmsh.plugin

Plugin functions
"""
module plugin

import ..gmsh

"""
    gmsh.plugin.setNumber(name, option, value)

Set the numerical option `option` to the value `value` for plugin `name`.
"""
function setNumber(name, option, value)
    ierr = Ref{Cint}()
    ccall((:gmshPluginSetNumber, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cchar}, Cdouble, Ptr{Cint}),
          name, option, value, ierr)
    ierr[] != 0 && error("gmshPluginSetNumber returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.plugin.setString(name, option, value)

Set the string option `option` to the value `value` for plugin `name`.
"""
function setString(name, option, value)
    ierr = Ref{Cint}()
    ccall((:gmshPluginSetString, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cchar}, Ptr{Cchar}, Ptr{Cint}),
          name, option, value, ierr)
    ierr[] != 0 && error("gmshPluginSetString returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.plugin.run(name)

Run the plugin `name`.
"""
function run(name)
    ierr = Ref{Cint}()
    ccall((:gmshPluginRun, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          name, ierr)
    ierr[] != 0 && error("gmshPluginRun returned non-zero error code: $(ierr[])")
    return nothing
end

end # end of module plugin

"""
    module gmsh.graphics

Graphics functions
"""
module graphics

import ..gmsh

"""
    gmsh.graphics.draw()

Draw all the OpenGL scenes.
"""
function draw()
    ierr = Ref{Cint}()
    ccall((:gmshGraphicsDraw, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshGraphicsDraw returned non-zero error code: $(ierr[])")
    return nothing
end

end # end of module graphics

"""
    module gmsh.fltk

FLTK graphical user interface functions
"""
module fltk

import ..gmsh

"""
    gmsh.fltk.initialize()

Create the FLTK graphical user interface. Can only be called in the main thread.
"""
function initialize()
    ierr = Ref{Cint}()
    ccall((:gmshFltkInitialize, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshFltkInitialize returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.fltk.wait(time = -1.)

Wait at most `time` seconds for user interface events and return. If `time` < 0,
wait indefinitely. First automatically create the user interface if it has not
yet been initialized. Can only be called in the main thread.
"""
function wait(time = -1.)
    ierr = Ref{Cint}()
    ccall((:gmshFltkWait, gmsh.lib), Cvoid,
          (Cdouble, Ptr{Cint}),
          time, ierr)
    ierr[] != 0 && error("gmshFltkWait returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.fltk.update()

Update the user interface (potentially creating new widgets and windows). First
automatically create the user interface if it has not yet been initialized. Can
only be called in the main thread: use `awake("update")` to trigger an update of
the user interface from another thread.
"""
function update()
    ierr = Ref{Cint}()
    ccall((:gmshFltkUpdate, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshFltkUpdate returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.fltk.awake(action = "")

Awake the main user interface thread and process pending events, and optionally
perform an action (currently the only `action` allowed is "update").
"""
function awake(action = "")
    ierr = Ref{Cint}()
    ccall((:gmshFltkAwake, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          action, ierr)
    ierr[] != 0 && error("gmshFltkAwake returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.fltk.lock()

Block the current thread until it can safely modify the user interface.
"""
function lock()
    ierr = Ref{Cint}()
    ccall((:gmshFltkLock, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshFltkLock returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.fltk.unlock()

Release the lock that was set using lock.
"""
function unlock()
    ierr = Ref{Cint}()
    ccall((:gmshFltkUnlock, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshFltkUnlock returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.fltk.run()

Run the event loop of the graphical user interface, i.e. repeatedly call
`wait()`. First automatically create the user interface if it has not yet been
initialized. Can only be called in the main thread.
"""
function run()
    ierr = Ref{Cint}()
    ccall((:gmshFltkRun, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshFltkRun returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.fltk.selectEntities(dim = -1)

Select entities in the user interface. If `dim` is >= 0, return only the
entities of the specified dimension (e.g. points if `dim` == 0).

Return an integer value, `dimTags`.
"""
function selectEntities(dim = -1)
    api_dimTags_ = Ref{Ptr{Cint}}()
    api_dimTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshFltkSelectEntities, gmsh.lib), Cint,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Cint, Ptr{Cint}),
          api_dimTags_, api_dimTags_n_, dim, ierr)
    ierr[] != 0 && error("gmshFltkSelectEntities returned non-zero error code: $(ierr[])")
    tmp_api_dimTags_ = unsafe_wrap(Array, api_dimTags_[], api_dimTags_n_[], own=true)
    dimTags = [ (tmp_api_dimTags_[i], tmp_api_dimTags_[i+1]) for i in 1:2:length(tmp_api_dimTags_) ]
    return api__result__, dimTags
end

"""
    gmsh.fltk.selectElements()

Select elements in the user interface.

Return an integer value, `elementTags`.
"""
function selectElements()
    api_elementTags_ = Ref{Ptr{Csize_t}}()
    api_elementTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshFltkSelectElements, gmsh.lib), Cint,
          (Ptr{Ptr{Csize_t}}, Ptr{Csize_t}, Ptr{Cint}),
          api_elementTags_, api_elementTags_n_, ierr)
    ierr[] != 0 && error("gmshFltkSelectElements returned non-zero error code: $(ierr[])")
    elementTags = unsafe_wrap(Array, api_elementTags_[], api_elementTags_n_[], own=true)
    return api__result__, elementTags
end

"""
    gmsh.fltk.selectViews()

Select views in the user interface.

Return an integer value, `viewTags`.
"""
function selectViews()
    api_viewTags_ = Ref{Ptr{Cint}}()
    api_viewTags_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshFltkSelectViews, gmsh.lib), Cint,
          (Ptr{Ptr{Cint}}, Ptr{Csize_t}, Ptr{Cint}),
          api_viewTags_, api_viewTags_n_, ierr)
    ierr[] != 0 && error("gmshFltkSelectViews returned non-zero error code: $(ierr[])")
    viewTags = unsafe_wrap(Array, api_viewTags_[], api_viewTags_n_[], own=true)
    return api__result__, viewTags
end

end # end of module fltk

"""
    module gmsh.onelab

ONELAB server functions
"""
module onelab

import ..gmsh

"""
    gmsh.onelab.set(data, format = "json")

Set one or more parameters in the ONELAB database, encoded in `format`.
"""
function set(data, format = "json")
    ierr = Ref{Cint}()
    ccall((:gmshOnelabSet, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cchar}, Ptr{Cint}),
          data, format, ierr)
    ierr[] != 0 && error("gmshOnelabSet returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.onelab.get(name = "", format = "json")

Get all the parameters (or a single one if `name` is specified) from the ONELAB
database, encoded in `format`.

Return `data`.
"""
function get(name = "", format = "json")
    api_data_ = Ref{Ptr{Cchar}}()
    ierr = Ref{Cint}()
    ccall((:gmshOnelabGet, gmsh.lib), Cvoid,
          (Ptr{Ptr{Cchar}}, Ptr{Cchar}, Ptr{Cchar}, Ptr{Cint}),
          api_data_, name, format, ierr)
    ierr[] != 0 && error("gmshOnelabGet returned non-zero error code: $(ierr[])")
    data = unsafe_string(api_data_[])
    return data
end

"""
    gmsh.onelab.setNumber(name, value)

Set the value of the number parameter `name` in the ONELAB database. Create the
parameter if it does not exist; update the value if the parameter exists.
"""
function setNumber(name, value)
    ierr = Ref{Cint}()
    ccall((:gmshOnelabSetNumber, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cdouble}, Csize_t, Ptr{Cint}),
          name, convert(Vector{Cdouble}, value), length(value), ierr)
    ierr[] != 0 && error("gmshOnelabSetNumber returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.onelab.setString(name, value)

Set the value of the string parameter `name` in the ONELAB database. Create the
parameter if it does not exist; update the value if the parameter exists.
"""
function setString(name, value)
    ierr = Ref{Cint}()
    ccall((:gmshOnelabSetString, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Ptr{Cchar}}, Csize_t, Ptr{Cint}),
          name, value, length(value), ierr)
    ierr[] != 0 && error("gmshOnelabSetString returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.onelab.getNumber(name)

Get the value of the number parameter `name` from the ONELAB database. Return an
empty vector if the parameter does not exist.

Return `value`.
"""
function getNumber(name)
    api_value_ = Ref{Ptr{Cdouble}}()
    api_value_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshOnelabGetNumber, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Ptr{Cdouble}}, Ptr{Csize_t}, Ptr{Cint}),
          name, api_value_, api_value_n_, ierr)
    ierr[] != 0 && error("gmshOnelabGetNumber returned non-zero error code: $(ierr[])")
    value = unsafe_wrap(Array, api_value_[], api_value_n_[], own=true)
    return value
end

"""
    gmsh.onelab.getString(name)

Get the value of the string parameter `name` from the ONELAB database. Return an
empty vector if the parameter does not exist.

Return `value`.
"""
function getString(name)
    api_value_ = Ref{Ptr{Ptr{Cchar}}}()
    api_value_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshOnelabGetString, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Ptr{Ptr{Cchar}}}, Ptr{Csize_t}, Ptr{Cint}),
          name, api_value_, api_value_n_, ierr)
    ierr[] != 0 && error("gmshOnelabGetString returned non-zero error code: $(ierr[])")
    tmp_api_value_ = unsafe_wrap(Array, api_value_[], api_value_n_[], own=true)
    value = [unsafe_string(tmp_api_value_[i]) for i in 1:length(tmp_api_value_) ]
    return value
end

"""
    gmsh.onelab.clear(name = "")

Clear the ONELAB database, or remove a single parameter if `name` is given.
"""
function clear(name = "")
    ierr = Ref{Cint}()
    ccall((:gmshOnelabClear, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cint}),
          name, ierr)
    ierr[] != 0 && error("gmshOnelabClear returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.onelab.run(name = "", command = "")

Run a ONELAB client. If `name` is provided, create a new ONELAB client with name
`name` and executes `command`. If not, try to run a client that might be linked
to the processed input files.
"""
function run(name = "", command = "")
    ierr = Ref{Cint}()
    ccall((:gmshOnelabRun, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cchar}, Ptr{Cint}),
          name, command, ierr)
    ierr[] != 0 && error("gmshOnelabRun returned non-zero error code: $(ierr[])")
    return nothing
end

end # end of module onelab

"""
    module gmsh.logger

Information logging functions
"""
module logger

import ..gmsh

"""
    gmsh.logger.write(message, level = "info")

Write a `message`. `level` can be "info", "warning" or "error".
"""
function write(message, level = "info")
    ierr = Ref{Cint}()
    ccall((:gmshLoggerWrite, gmsh.lib), Cvoid,
          (Ptr{Cchar}, Ptr{Cchar}, Ptr{Cint}),
          message, level, ierr)
    ierr[] != 0 && error("gmshLoggerWrite returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.logger.start()

Start logging messages.
"""
function start()
    ierr = Ref{Cint}()
    ccall((:gmshLoggerStart, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshLoggerStart returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.logger.get()

Get logged messages.

Return `log`.
"""
function get()
    api_log_ = Ref{Ptr{Ptr{Cchar}}}()
    api_log_n_ = Ref{Csize_t}()
    ierr = Ref{Cint}()
    ccall((:gmshLoggerGet, gmsh.lib), Cvoid,
          (Ptr{Ptr{Ptr{Cchar}}}, Ptr{Csize_t}, Ptr{Cint}),
          api_log_, api_log_n_, ierr)
    ierr[] != 0 && error("gmshLoggerGet returned non-zero error code: $(ierr[])")
    tmp_api_log_ = unsafe_wrap(Array, api_log_[], api_log_n_[], own=true)
    log = [unsafe_string(tmp_api_log_[i]) for i in 1:length(tmp_api_log_) ]
    return log
end

"""
    gmsh.logger.stop()

Stop logging messages.
"""
function stop()
    ierr = Ref{Cint}()
    ccall((:gmshLoggerStop, gmsh.lib), Cvoid,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshLoggerStop returned non-zero error code: $(ierr[])")
    return nothing
end

"""
    gmsh.logger.time()

Return wall clock time.

Return a floating point value.
"""
function time()
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshLoggerTime, gmsh.lib), Cdouble,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshLoggerTime returned non-zero error code: $(ierr[])")
    return api__result__
end

"""
    gmsh.logger.cputime()

Return CPU time.

Return a floating point value.
"""
function cputime()
    ierr = Ref{Cint}()
    api__result__ = ccall((:gmshLoggerCputime, gmsh.lib), Cdouble,
          (Ptr{Cint},),
          ierr)
    ierr[] != 0 && error("gmshLoggerCputime returned non-zero error code: $(ierr[])")
    return api__result__
end

end # end of module logger

end # end of module gmsh
