/**
 * File name: audio_output.c
 * Project: Geonkick (A kick synthesizer)
 *
 * Copyright (C) 2018 Iurie Nistor <http://iuriepage.wordpress.com>
 *
 * This file is part of Geonkick.
 *
 * GeonKick is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "audio_output.h"

enum geonkick_error
gkick_audio_output_create(struct gkick_audio_output **audio_output, int sample_rate)
{
        if (audio_output == NULL) {
                gkick_log_error("wrong arguments");
                return GEONKICK_ERROR;
        }

        *audio_output = (struct gkick_audio_output*)calloc(1, sizeof(struct gkick_audio_output));
        if (*audio_output == NULL) {
                gkick_log_error("can't allocate memory");
                return GEONKICK_ERROR;
        }
        (*audio_output)->decay   = -1;
        (*audio_output)->play    = false;
	(*audio_output)->enabled = true;
        (*audio_output)->muted   = false;
        (*audio_output)->solo    = false;
        (*audio_output)->channel = 0;
        (*audio_output)->sample_rate = sample_rate;

        gkick_buffer_new((struct gkick_buffer**)&(*audio_output)->updated_buffer,
                         (*audio_output)->sample_rate * GEONKICK_MAX_LENGTH);
        if ((*audio_output)->updated_buffer == NULL) {
                gkick_log_error("can't create updated buffer");
                gkick_audio_output_free(audio_output);
                return GEONKICK_ERROR;
        }
        gkick_buffer_set_size((struct gkick_buffer*)(*audio_output)->updated_buffer, 0);

        gkick_buffer_new((struct gkick_buffer**)&(*audio_output)->playing_buffer,
                         (*audio_output)->sample_rate * GEONKICK_MAX_LENGTH);
        if ((*audio_output)->playing_buffer == NULL) {
                gkick_log_error("can't create playing buffer");
                gkick_audio_output_free(audio_output);
                return GEONKICK_ERROR;
        }
        gkick_buffer_set_size((struct gkick_buffer*)(*audio_output)->playing_buffer, 0);

        if (pthread_mutex_init(&(*audio_output)->lock, NULL) != 0) {
                gkick_log_error("error on init mutex");
                gkick_audio_output_free(audio_output);
                return GEONKICK_ERROR;
	}

        return GEONKICK_OK;
}

void gkick_audio_output_free(struct gkick_audio_output **audio_output)
{
        if (audio_output != NULL && *audio_output != NULL) {
                struct gkick_buffer *p = (struct gkick_buffer*)((*audio_output)->playing_buffer);
                gkick_buffer_free(&p);
                p = (struct gkick_buffer*)((*audio_output)->updated_buffer);
                gkick_buffer_free(&p);
                pthread_mutex_destroy(&(*audio_output)->lock);
                free(*audio_output);
                *audio_output = NULL;
        }
}

enum geonkick_error
gkick_audio_output_key_pressed(struct gkick_audio_output *audio_output,
                               struct gkick_note_info *key)
{
        if (key->state == GKICK_KEY_STATE_PRESSED) {
                audio_output->key = *key;
                audio_output->is_play = true;
                gkick_audio_output_swap_buffers(audio_output);
        } else {
                audio_output->decay = GEKICK_KEY_RELESE_DECAY_TIME;
                audio_output->key.state = key->state;
        }

        return GEONKICK_OK;
}

enum geonkick_error
gkick_audio_output_play(struct gkick_audio_output *audio_output)
{
        audio_output->play = true;
        return GEONKICK_OK;
}

gkick_real
gkick_audio_output_tune_factor(int note_number)
{
        return exp2f((gkick_real)(note_number - 69) / 12.0f);
}

enum geonkick_error
gkick_audio_output_get_frame(struct gkick_audio_output *audio_output,
                             gkick_real *val)
{
        int release_time = GEKICK_KEY_RELESE_DECAY_TIME;
        gkick_real decay_val;

        if (audio_output->play) {
                struct gkick_note_info key;
                key.channel     = 1;
                key.note_number = 69;
                key.velocity    = 127;
                key.state       = GKICK_KEY_STATE_PRESSED;
                gkick_audio_output_key_pressed(audio_output, &key);
                audio_output->play = false;
        }

        *val = 0;
        if (audio_output->is_play) {
                if (gkick_buffer_is_end((struct gkick_buffer*)audio_output->playing_buffer)) {
                        audio_output->is_play = false;
                } else {
                        struct gkick_buffer *buff = (struct gkick_buffer*)audio_output->playing_buffer;
                        gkick_real factor = gkick_audio_output_tune_factor(audio_output->key.note_number);
                        if (audio_output->tune)
                                *val = gkick_buffer_stretch_get_next(buff, factor);
                        else
                                *val = gkick_buffer_get_next(buff);

                        if (gkick_buffer_size(buff) - gkick_buffer_index(buff) == GEKICK_KEY_RELESE_DECAY_TIME) {
                                audio_output->decay     = GEKICK_KEY_RELESE_DECAY_TIME;
                                audio_output->key.state = GKICK_KEY_STATE_RELEASED;
                        }

                        if (audio_output->key.state == GKICK_KEY_STATE_RELEASED)
                                decay_val = - 1.0f * ((gkick_real)(release_time - audio_output->decay) / release_time) + 1.0;
                        else
                                decay_val = 1.0f;
                        *val *= decay_val * ((gkick_real)audio_output->key.velocity / 127);

                        if (audio_output->key.state == GKICK_KEY_STATE_RELEASED) {
                                audio_output->decay--;
                                if (audio_output->decay < 0)
                                        audio_output->is_play = false;
                        }
                }
        }

        *val *= (gkick_real)audio_output->limiter / 1000000;

        return GEONKICK_OK;
}

void gkick_audio_output_lock(struct gkick_audio_output *audio_output)
{
        if (audio_output != NULL)
                pthread_mutex_lock(&audio_output->lock);
}

void gkick_audio_output_unlock(struct gkick_audio_output *audio_output)
{
        if (audio_output != NULL)
                pthread_mutex_unlock(&audio_output->lock);
}

struct gkick_buffer*
gkick_audio_output_get_buffer(struct gkick_audio_output  *audio_output)
{
        return (struct gkick_buffer*)audio_output->playing_buffer;
}

void gkick_audio_output_swap_buffers(struct gkick_audio_output *audio_output)
{
        gkick_buffer_reset((struct gkick_buffer*)audio_output->playing_buffer);

        /**
         * Try lock. If succesfull, swap buffers. If not, continue with
         * the current one to play until the next press of the key.
         */
        if (pthread_mutex_trylock(&audio_output->lock) == 0) {
                /* Test if the updated buffer is full. Otherwise it means that it was not updated. */
                if (gkick_buffer_size((struct gkick_buffer*)audio_output->updated_buffer) > 0
                    && gkick_buffer_is_end((struct gkick_buffer*)audio_output->updated_buffer)) {
                        char *buff = audio_output->updated_buffer;
                        audio_output->updated_buffer = audio_output->playing_buffer;
                        audio_output->playing_buffer = buff;
                }
                gkick_buffer_reset((struct gkick_buffer*)audio_output->playing_buffer);
                gkick_audio_output_unlock(audio_output);
        }
}

enum geonkick_error
gkick_audio_output_set_playing_key(struct gkick_audio_output *audio_output, signed char key)
{
        if (key < 21 || key > 108)
                key = GEONKICK_ANY_KEY;
        audio_output->playing_key = key;
        return GEONKICK_OK;
}

enum geonkick_error
gkick_audio_output_get_playing_key(struct gkick_audio_output *audio_output, signed char *key)
{
        *key = audio_output->playing_key;
        return GEONKICK_OK;
}


void gkick_audio_output_tune_output(struct gkick_audio_output *audio_output, bool tune)
{
        audio_output->tune = tune;
}

bool gkick_audio_output_is_tune_output(struct gkick_audio_output *audio_output)
{
        return audio_output->tune;
}

enum geonkick_error
gkick_audio_output_set_channel(struct gkick_audio_output *audio_output,
                               size_t channel)
{
        audio_output->channel = channel;
        return GEONKICK_OK;
}

enum geonkick_error
gkick_audio_output_get_channel(struct gkick_audio_output *audio_output,
                               size_t *channel)
{
        *channel = audio_output->channel;
        return GEONKICK_OK;
}
