""" Python module for Prandtl Meyer functions with Mach angles

Assumptions:
1) Perfect gas with constant specific heats and molecular weights
2) Isentropic flow

"""
"""
 * Copyright (C) 2006 Varun Hiremath.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * Authors: Varun Hiremath, Venkattraman A
 * Version 0.2

"""
from scipy import optimize
from math import *

class PrandtlMeyer:
    def __init__(self, g=1.4):
        """ g is the value of gamma (ratio of specific heats), default = 1.4
        """
        self.g = g

    def get_Prandtl_Func_from_M(self, M):
        """ Returns Prandtl Meyer function (degrees) corresponding to the given
        Mach No. """
        return (sqrt((self.g+1)/(self.g-1))* atan(sqrt((self.g-1)/(self.g+1)*(M**2 - 1))) - atan(sqrt(M**2 -1)))*180.0/pi

    def get_Mach_Angle_from_M(self, M):
        """ Returns Mach Angle (degrees) for a given Mach No. M """
        return asin(1.0/M)*180.0/pi

    def get_M_from_Prandtl_Func(self, omega):
        """ Returns Mach No. corresponding to the given Prandtl
        function value omega (degrees) """
        return optimize.fsolve(lambda M: self.get_Prandtl_Func_from_M(M) - omega, 1.01)

    def get_M_from_Mach_Angle(self, alpha):
        """ Returns Mach No. corresponding to the given Mach Angle
        alpha (degrees)"""
        alpha = alpha*pi/180.0
        return 1.0/sin(alpha)

