/* Public domain. */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <errno.h>
#include <unistd.h>
#include "uint32.h"
#include "allreadwrite.h"
#include "bytestr.h"
#include "djbunix.h"
#include "error.h"
#include "cdb.h"

void cdb_free (struct cdb *c)
{
  if (c->map)
  {
    munmap(c->map, c->size) ;
    c->map = 0 ;
  }
}

void cdb_findstart (struct cdb *c)
{
  c->loop = 0 ;
}

int cdb_init (struct cdb *c, int fd)
{
  struct stat st ;

  cdb_free(c) ;
  cdb_findstart(c) ;
  c->fd = fd ;

  if (fstat(fd, &st) == 0)
  {
    char *x = mmap(0, st.st_size, PROT_READ, MAP_SHARED, fd, 0) ;
    if (x != MAP_FAILED)
    {
      c->size = st.st_size ;
      c->map = x ;
      return 0 ;
    }
  }
  return -1 ;
}

int cdb_read (struct cdb *c, char *buf, unsigned int len, uint32 pos)
{
  if (c->map)
  {
    if ((pos > c->size) || (c->size - pos < len))
      return (errno = EPROTO, -1) ;
    byte_copy(buf, len, c->map + pos) ;
    return 0 ;
  }
  else if (seek_set(c->fd, pos) == -1) return -1 ;
  else
  {
    register int r = allread(c->fd, buf, len) ;
    return (r == -1) ? -1 : ((unsigned int)r < len) ? (errno = EPROTO, -1) : 0 ;
  }
}

static int match (struct cdb *c, char const *key, unsigned int len, uint32 pos)
{
  char buf[1024] ;
  while (len > 0)
  {
    register unsigned int n = 1024 ;
    if (n > len) n = len ;
    if (cdb_read(c, buf, n, pos) == -1) return -1 ;
    if (byte_diff(buf, n, key)) return 0 ;
    pos += n ;
    key += n ;
    len -= n ;
  }
  return 1 ;
}

int cdb_findnext (struct cdb *c, char const *key, unsigned int len)
{
  char buf[8] ;
  uint32 pos ;
  uint32 u ;

  if (!c->loop)
  {
    u = cdb_hash(key, len) ;
    if (cdb_read(c, buf, 8, (u << 3) & 2047) == -1) return -1 ;
    uint32_unpack(buf + 4, &c->hslots) ;
    if (!c->hslots) return 0 ;
    uint32_unpack(buf, &c->hpos) ;
    c->khash = u ;
    u >>= 8 ;
    u %= c->hslots ;
    u <<= 3 ;
    c->kpos = c->hpos + u ;
  }

  while (c->loop < c->hslots)
  {
    if (cdb_read(c, buf, 8, c->kpos) == -1) return -1 ;
    uint32_unpack(buf + 4, &pos) ;
    if (!pos) return 0 ;
    c->loop++ ;
    c->kpos += 8 ;
    if (c->kpos == c->hpos + (c->hslots << 3)) c->kpos = c->hpos ;
    uint32_unpack(buf, &u) ;
    if (u == c->khash)
    {
      if (cdb_read(c, buf, 8, pos) == -1) return -1 ;
      uint32_unpack(buf, &u) ;
      if (u == len)
	switch (match(c, key, len, pos + 8))
        {
	  case -1:
	    return -1 ;
	  case 1:
	    uint32_unpack(buf + 4, &c->dlen) ;
	    c->dpos = pos + 8 + len ;
	    return 1 ;
	}
    }
  }

  return 0 ;
}

int cdb_find (struct cdb *c, char const *key, unsigned int len)
{
  cdb_findstart(c) ;
  return cdb_findnext(c, key, len) ;
}
