/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment interactive test using PgmText.
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * FIXME: add tests for line spacing and outline colour.
 */

#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>
#include "interactive_tests.h"

const gchar *lorem_ipsum =
"<big>Lorem Ipsum</big>\n"
"Lorem ipsum dolor sit amet, consectetur adipisicing elit, sed do eiusmod "
"tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, "
"quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo "
"consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse "
"cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non "
"proident, sunt in culpa qui officia deserunt mollit anim id est laborum.";

/* Test that a simple text is displayed correctly */
PGM_START_INTERACTIVE_TEST (test_text_basic)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *text = NULL;

  canvas = pgm_canvas_new ();

  text = pgm_text_new ("This is some text.");

  pgm_drawable_set_size (text, 3.5f, 0.25f);
  pgm_drawable_set_position (text, 0.25f, 2.75f, 0.0f);
  pgm_drawable_set_fg_color (text, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (text, 20, 20, 20, 255);
  pgm_drawable_show (text);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, text);
}
PGM_END_INTERACTIVE_TEST;

/* Test attributes */
PGM_START_INTERACTIVE_TEST (test_text_attributes)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *text = NULL;
  const gchar *description =
    "This is normal.\n"
    "<tt>This is monospace </tt>\n"
    "<b>This is bold.</b>\n"
    "<u>This is underlined.</u>\n"
    "<i>This is italic.</i>\n"
    "<big>This is big</big>\n"
    "<s>This is striken</s>\n"
    "\n<b><i>This is bold italic.</i></b>\n"
    "<u><b>This is bold underlined.</b></u>\n"
    "<i><u>This is italic underlined.</u></i>\n\n"
    "<i><u><b>This is the ultimate combo.</b></u></i>\n"
    "\nSome utf-8: éàüßčřñ\n"
    "\nI like maths: a<sub><small>n</small></sub><sup><small>2</small></sup>\n"
    "Colors: <span foreground=\"red\">red</span> "
            "<span foreground=\"green\">green</span> "
            "<span foreground=\"blue\">blue</span> "
            "<span foreground=\"yellow\">yellow</span> ";

  canvas = pgm_canvas_new ();

  text = pgm_text_new (NULL);

  pgm_text_set_font_height (PGM_TEXT (text), 1.0f/20.0f);
  pgm_text_set_markup (PGM_TEXT (text), description);
  pgm_drawable_set_size (text, 1.75f, 2.5f);
  pgm_drawable_set_position (text, 1.0f, 0.25f, 0.0f);
  pgm_drawable_set_fg_color (text, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (text, 20, 20, 20, 255);
  pgm_drawable_show (text);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, text);
}
PGM_END_INTERACTIVE_TEST;

static PgmDrawable *
generate_lorem_ipsum_pgm_text(guint div, guint pos)
{
  PgmDrawable *text = pgm_text_new (NULL);
  fail_unless (NULL != text, "text allocation failed");
  gfloat width = 3.75f / div, space = 0.25f / (div+1);
  gfloat font_height = 0.1f;

  /* these are the values for which the text fits in the drawable in these
   * various cases */
  switch (div)
    {
    case 2:
      font_height = 1.0f/20.0f;
      break;
    case 3:
      font_height = 1.0f/25.0f;
      break;
    case 4:
      font_height = 1.0f/28.0f;
      break;
    }

  pgm_text_set_font_height (PGM_TEXT (text), font_height);
  pgm_text_set_markup (PGM_TEXT (text), lorem_ipsum);
  pgm_drawable_set_size (text, width, 3 - space * 2.0f);
  pgm_drawable_set_position (text,
                             space + (width + space) * pos,
                             space,
                             0.0f);
  pgm_drawable_set_fg_color (text, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (text, 20, 20, 20, 255);

  return text;
}

/* test alignment */
PGM_START_INTERACTIVE_TEST (test_text_alignment)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL, *center = NULL;

  canvas = pgm_canvas_new ();

  left = generate_lorem_ipsum_pgm_text (3, 0);
  pgm_text_set_alignment (PGM_TEXT (left), PGM_TEXT_ALIGN_LEFT);
  pgm_drawable_show (left);

  center = generate_lorem_ipsum_pgm_text (3, 1);
  pgm_text_set_alignment (PGM_TEXT (center), PGM_TEXT_ALIGN_CENTER);
  pgm_drawable_show (center);

  right = generate_lorem_ipsum_pgm_text (3, 2);
  pgm_text_set_alignment (PGM_TEXT (right), PGM_TEXT_ALIGN_RIGHT);
  pgm_drawable_show (right);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, center);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST;

/* test some font properties (family and height) */
PGM_START_INTERACTIVE_TEST (test_text_font)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL;

  canvas = pgm_canvas_new ();

  left = generate_lorem_ipsum_pgm_text (2, 0);
  pgm_text_set_font_family (PGM_TEXT (left), "times");
  pgm_drawable_show (left);

  right = generate_lorem_ipsum_pgm_text (2, 1);
  pgm_drawable_show (right);


  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_text_ellipsize)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL, *middle = NULL;
  static char *longish = "This is a bit longish isn't it?";

  canvas = pgm_canvas_new ();

  left = pgm_text_new (longish);
  pgm_text_set_ellipsize (PGM_TEXT (left), PGM_TEXT_ELLIPSIZE_END);
  pgm_drawable_set_position (left, 0.25f, 0.25f, 0.0f);
  pgm_drawable_set_size (left, 1.75f, 0.25f);

  pgm_drawable_set_fg_color (left, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (left, 20, 20, 20, 255);
  pgm_drawable_show (left);

  middle = pgm_text_new (longish);
  pgm_text_set_ellipsize (PGM_TEXT (middle), PGM_TEXT_ELLIPSIZE_MIDDLE);
  pgm_drawable_set_position (middle, 1.25f, 1.25f, 0.0f);
  pgm_drawable_set_size (middle, 1.75f, 0.25f);

  pgm_drawable_set_fg_color (middle, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (middle, 20, 20, 20, 255);
  pgm_drawable_show (middle);

  right = pgm_text_new (longish);
  pgm_text_set_ellipsize (PGM_TEXT (right), PGM_TEXT_ELLIPSIZE_START);
  pgm_drawable_set_position (right, 2.2f, 2.25f, 0.0f);
  pgm_drawable_set_size (right, 1.75f, 0.25f);

  pgm_drawable_set_fg_color (right, 240, 240, 240, 255);
  pgm_drawable_set_bg_color (right, 20, 20, 20, 255);
  pgm_drawable_show (right);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, middle);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_text_justify)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL;

  canvas = pgm_canvas_new ();

  left = generate_lorem_ipsum_pgm_text (2, 0);
  pgm_drawable_show (left);

  right = generate_lorem_ipsum_pgm_text (2, 1);
  pgm_text_set_justify (PGM_TEXT (right), TRUE);
  pgm_drawable_show (right);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_text_wrap)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL, *middle = NULL;
  const gchar *lorem_ipsum_longword =
"<big>Lorem Ipsum</big>\n"
"Lorem ipsum dolor sit amet, consectetur adipisicing elit, sed do eiusmod "
"tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, "
"quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo "
"consequat. Duis aute irure dolor in "
"<i>reprehenderitverylongwordthatmightbetoolong</i> in voluptate velit esse "
"cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non "
"proident, sunt in culpa qui officia deserunt mollit anim id est laborum.";

  canvas = pgm_canvas_new ();

  left = generate_lorem_ipsum_pgm_text (3, 0);
  pgm_text_set_wrap (PGM_TEXT (left), PGM_TEXT_WRAP_WORD);
  pgm_text_set_markup (PGM_TEXT (left), lorem_ipsum_longword);
  pgm_drawable_show (left);

  middle = generate_lorem_ipsum_pgm_text (3, 1);
  pgm_text_set_wrap (PGM_TEXT (middle), PGM_TEXT_WRAP_WORD_CHAR);
  pgm_text_set_markup (PGM_TEXT (middle), lorem_ipsum_longword);
  pgm_drawable_show (middle);

  right = generate_lorem_ipsum_pgm_text (3, 2);
  pgm_text_set_wrap (PGM_TEXT (right), PGM_TEXT_WRAP_CHAR);
  pgm_text_set_markup (PGM_TEXT (right), lorem_ipsum_longword);
  pgm_drawable_show (right);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, middle);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_text_gravity)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *one = NULL, *two = NULL, *three = NULL, *four = NULL;

  canvas = pgm_canvas_new ();

  one = generate_lorem_ipsum_pgm_text (4, 0);
  pgm_text_set_gravity (PGM_TEXT (one), PGM_TEXT_GRAVITY_SOUTH);
  pgm_drawable_show (one);

  two = generate_lorem_ipsum_pgm_text (4, 1);
  pgm_text_set_gravity (PGM_TEXT (two), PGM_TEXT_GRAVITY_EAST);
  pgm_drawable_show (two);

  three = generate_lorem_ipsum_pgm_text (4, 2);
  pgm_text_set_gravity (PGM_TEXT (three), PGM_TEXT_GRAVITY_NORTH);
  pgm_drawable_show (three);

  four = generate_lorem_ipsum_pgm_text (4, 3);
  pgm_text_set_gravity (PGM_TEXT (four), PGM_TEXT_GRAVITY_WEST);
  pgm_drawable_show (four);


  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, one);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, two);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, three);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, four);
}
PGM_END_INTERACTIVE_TEST;

PGM_START_INTERACTIVE_TEST (test_text_style)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL, *center = NULL;

  canvas = pgm_canvas_new ();

  left = generate_lorem_ipsum_pgm_text (3, 0);
  pgm_text_set_font_family (PGM_TEXT (left), "times");
  pgm_text_set_style (PGM_TEXT (left), PGM_TEXT_STYLE_NORMAL);
  pgm_drawable_show (left);

  center = generate_lorem_ipsum_pgm_text (3, 1);
  pgm_text_set_font_family (PGM_TEXT (center), "times");
  pgm_text_set_style (PGM_TEXT (center), PGM_TEXT_STYLE_OBLIQUE);
  pgm_drawable_show (center);

  right = generate_lorem_ipsum_pgm_text (3, 2);
  pgm_text_set_font_family (PGM_TEXT (right), "times");
  pgm_text_set_style (PGM_TEXT (right), PGM_TEXT_STYLE_ITALIC);
  pgm_drawable_show (right);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, center);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST;

PGM_START_INTERACTIVE_TEST (test_text_variant)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL;

  canvas = pgm_canvas_new ();

  left = generate_lorem_ipsum_pgm_text (2, 0);
  pgm_text_set_variant (PGM_TEXT (left), PGM_TEXT_VARIANT_NORMAL);
  pgm_drawable_show (left);

  right = generate_lorem_ipsum_pgm_text (2, 1);
  pgm_text_set_variant (PGM_TEXT (right), PGM_TEXT_VARIANT_SMALL_CAPS);
  pgm_drawable_show (right);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_text_weight)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *left = NULL, *right = NULL, *center = NULL;

  canvas = pgm_canvas_new ();

  left = generate_lorem_ipsum_pgm_text (3, 0);
  pgm_text_set_weight (PGM_TEXT (left), PGM_TEXT_WEIGHT_LIGHT);
  pgm_drawable_show (left);

  center = generate_lorem_ipsum_pgm_text (3, 1);
  pgm_text_set_weight (PGM_TEXT (center), PGM_TEXT_WEIGHT_NORMAL);
  pgm_drawable_show (center);

  right = generate_lorem_ipsum_pgm_text (3, 2);
  pgm_text_set_weight (PGM_TEXT (right), PGM_TEXT_WEIGHT_BOLD);
  pgm_drawable_show (right);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, left);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, center);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, right);
}
PGM_END_INTERACTIVE_TEST;

PGM_START_INTERACTIVE_TEST (test_text_alpha_render)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *text[4] = { NULL };
  guint i = 0;

  canvas = pgm_canvas_new ();

  for (i=0; i<4; i++)
    {
      text[i] = generate_lorem_ipsum_pgm_text (4, i%4);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, text[i]);
      pgm_drawable_set_opacity (text[i], i?(64 * i - 1):0);
      pgm_drawable_show (text[i]);
    }


  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

/* This checks that the stuff introduced in r1069 is working */
PGM_START_INTERACTIVE_TEST (test_text_alpha_render_noblend)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *text[4] = { NULL };
  guint i = 0;

  pgm_viewport_set_alpha_blending (interactive_test_viewport, FALSE);
  canvas = pgm_canvas_new ();

  for (i=0; i<4; i++)
    {
      text[i] = generate_lorem_ipsum_pgm_text (4, i%4);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, text[i]);
      pgm_drawable_set_opacity (text[i], i?(64 * i - 1):0);
      pgm_drawable_show (text[i]);
    }


  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

Suite *
pgm_text_suite (void)
{
  Suite *s = suite_create ("PgmText");
  TCase *tc_chain = tcase_create ("pgmtext tests");

  PGM_INTERACTIVE_TESTS_SET_FIXTURES (tc_chain);

  tcase_set_timeout (tc_chain, 500);

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_text_basic);
  tcase_add_test (tc_chain, test_text_attributes);
  tcase_add_test (tc_chain, test_text_alignment);
  tcase_add_test (tc_chain, test_text_font);
  tcase_add_test (tc_chain, test_text_ellipsize);
  tcase_add_test (tc_chain, test_text_justify);
  tcase_add_test (tc_chain, test_text_wrap);
  tcase_add_test (tc_chain, test_text_gravity);
  tcase_add_test (tc_chain, test_text_style);
  tcase_add_test (tc_chain, test_text_alpha_render);
  tcase_add_test (tc_chain, test_text_alpha_render_noblend);
  /* These are deactivated for now since the functions they test do not seem to
   * do much. */
  /*
  tcase_add_test (tc_chain, test_text_variant);
  tcase_add_test (tc_chain, test_text_weight);
  */

  return s;
}

