/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>

/* test drawable floating reference on canvas_add/del_* with OpenGL plugin */
PGM_START_TEST (test_opengl_add_del_drawable)
{
  PgmViewport *viewport;
  PgmViewportFactory *factory;
  PgmError ret;
  gboolean success;
  PgmCanvas *canvas;
  PgmDrawable *drawable;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  factory = pgm_viewport_factory_new ("opengl");
  ret = pgm_viewport_factory_create (factory, &viewport);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");

  canvas = pgm_canvas_new ();
  pgm_viewport_set_canvas (viewport, canvas);

  /* create a new floating drawable with refcount 1 */
  drawable = pgm_image_new ();
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);
  fail_unless (GST_OBJECT_IS_FLOATING (drawable),
               "drawable object is not floating at creation");

  /* ref it for ourselves */
  gst_object_ref (GST_OBJECT (drawable));
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 2);

  /* adding it in the canvas sinks the drawable, plugin takes a ref though */
  ret = pgm_canvas_add (canvas, PGM_DRAWABLE_NEAR, drawable);
  fail_if (ret != PGM_ERROR_OK, "canvas_add error");

  /* wait half a sec to be sure the flushing has been completed since it's
   * asynchronous */
  g_usleep (500000);
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 3);
  fail_if (GST_OBJECT_IS_FLOATING (drawable),
           "drawable object still floating after being added to the canvas");

  /* removing it reduces the refcount */
  ret = pgm_canvas_remove (canvas, drawable);
  fail_if (ret != PGM_ERROR_OK, "canvas_del error");

  /* wait half a sec to be sure the flushing has been completed since it's
   * asynchronous */
  g_usleep (500000);
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);

  /* clean up our own reference */
  gst_object_unref (GST_OBJECT_CAST (drawable));
  gst_object_unref (GST_OBJECT_CAST (canvas));
  gst_object_unref (GST_OBJECT_CAST (viewport));
  gst_object_unref (GST_OBJECT_CAST (factory));
  pgm_deinit ();
}
PGM_END_TEST

/* test drawable floating reference on canvas unref with OpenGL plugin */
PGM_START_TEST (test_opengl_add_drawable_unref_object)
{
  PgmViewport *viewport;
  PgmViewportFactory *factory;
  PgmError ret;
  gboolean success;
  PgmCanvas *canvas;
  PgmDrawable *drawable;

  success = pgm_init_check (NULL, NULL);
  fail_unless (success, "init_check returned FALSE");

  factory = pgm_viewport_factory_new ("opengl");
  ret = pgm_viewport_factory_create (factory, &viewport);
  fail_if (ret != PGM_ERROR_OK, "viewport_factory_create error");

  canvas = pgm_canvas_new ();
  pgm_viewport_set_canvas (viewport, canvas);

  /* create a new floating drawable with refcount 1 */
  drawable = pgm_image_new ();
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);
  fail_unless (GST_OBJECT_IS_FLOATING (drawable),
               "drawable object is not floating at creation");

  /* ref it for ourselves */
  gst_object_ref (GST_OBJECT (drawable));
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 2);

  /* adding it in the canvas sinks the drawable, plugin takes a ref though */
  ret = pgm_canvas_add (canvas, PGM_DRAWABLE_NEAR, drawable);
  fail_if (ret != PGM_ERROR_OK, "canvas_add_drawable error");

  /* wait half a sec to be sure the flushing has been completed since it's
   * asynchronous */
  g_usleep (500000);
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 3);
  fail_if (GST_OBJECT_IS_FLOATING (drawable),
           "drawable object still floating after being added to the canvas");

  /* unattaching the canvas must decrease the refcount by 1 */
  pgm_viewport_set_canvas (viewport, NULL);
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 2);

  /* unreffing the canvas must decrease the refcount by 1 */
  gst_object_unref (GST_OBJECT_CAST (canvas));
  ASSERT_OBJECT_REFCOUNT (drawable, "drawable", 1);

  /* clean up our own reference */
  gst_object_unref (GST_OBJECT_CAST (drawable));
  pgm_deinit ();
}
PGM_END_TEST

Suite *
pgm_opengl_suite (void)
{
  Suite *s = suite_create ("GL+PgmDrawable References");
  TCase *tc_chain = tcase_create ("drawable basic ref tests");

  tcase_set_timeout(tc_chain, 10);

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_opengl_add_del_drawable);
  tcase_add_test (tc_chain, test_opengl_add_drawable_unref_object);

  return s;
}

GST_CHECK_MAIN (pgm_opengl);
