///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file OrientationDataChannel.h
 * \brief Contains the definition of the AtomViz::OrientationDataChannel class.
 */

#ifndef __ORIENTATION_DATA_CHANNEL_H
#define __ORIENTATION_DATA_CHANNEL_H

#include <core/Core.h>
#include <core/scene/animation/controller/Controller.h>
#include "DataChannel.h"

namespace AtomViz {

/**
 * \brief The data channel stores the atomic orientations.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT OrientationDataChannel : public DataChannel
{
	Q_OBJECT

public:

	/// \brief Serialization constructor that creates an empty data channel.
	///
	/// \note This constructor is only used when a data channel is loaded from a scene file.
	///       It must not be used to create a new data channel.
	OrientationDataChannel(bool isLoading = false);

	/// \brief Constructor that creates a standard data channel.
	/// \param which Specifies which standard data channel should be created.
	///              This should be DataChannelIdentifier::OrientationChannel.
	///
	/// Data type, component count and name are automatically chosen by this constructor.
	OrientationDataChannel(DataChannelIdentifier which);

	/// \brief Renders the channel' contents in a viewport.
	/// \param time The current animation time.
	/// \param vp The viewport into which the channel should be rendered.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param contextNode The scene object the AtomsObject belongs to.
	virtual void render(TimeTicks time, Viewport* vp, AtomsObject* atoms, ObjectNode* contextNode);

	/// \brief Renders the channel's contents in high-quality mode to an offscreen buffer.
	/// \param time The current animation time.
	/// \param atoms The AtomsObject to which this DataChannel belongs to.
	/// \param view Describes the projection parameters.
	/// \param contextNode The scene object the AtomsObject belongs to.
	/// \param imageWidth The width of the image buffer in pixels.
	/// \param imageHeight The height of the image buffer in pixels.
	/// \param glcontext The OpenGL rendering context.
	virtual void renderHQ(TimeTicks time, AtomsObject* atoms, const CameraViewDescription& view, ObjectNode* contextNode, int imageWidth, int imageHeight, Window3D* glcontext);

private:

	DECLARE_SERIALIZABLE_PLUGIN_CLASS(OrientationDataChannel)
};

/**
 * \brief A properties editor for the OrientationDataChannel class.
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT OrientationDataChannelEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(OrientationDataChannelEditor)
};

};	// End of namespace AtomViz

#endif // __ORIENTATION_DATA_CHANNEL_H
