%feature("docstring") OT::TrendFactory
R"RAW(Trend estimator.

Refer to :ref:`trend_transform`.

Parameters
----------
basisSequenceFactory : :class:`~openturns.BasisSequenceFactory`
    The  regression strategy that provides the estimation of the  coefficients associated to the best model among the  basis functions.

    Default is the *least angle regression* (LARS) method for the choice of sparse models: :class:`~openturns.LARS`.
fittingAlgorithm : :class:`~openturns.FittingAlgorithm`, optional
    The fitting algorithm that estimates the empirical error on each sub-basis.

    Default is the *leave one out* strategy: :class:`~openturns.CorrectedLeaveOneOut`.

Notes
-----
A multivariate stochastic  process :math:`X: \Omega \times \cD \rightarrow \Rset^d` of dimension *d* where :math:`\cD \in \Rset^n` can write as the sum of a trend function :math:`f_{trend}: \Rset^n \rightarrow \Rset^d` and a stationary multivariate stochastic process :math:`X_{stat}: \Omega \times \cD \rightarrow \Rset^d` of dimension *d* as follows:

.. math::

   X(\omega,\vect{t}) = X_{stat}(\omega,\vect{t}) + f_{trend}(\vect{t})

The  :class:`~openturns.TrendFactory` enables to identify the trend  function :math:`f_{trend}` from a given field of the process *X* and then to remove this last one from the initial field. The resulting field is a realization of the process :math:`X_{stat}`.

We consider the functional basis :math:`\cB = (f_1, f_2, \ldots, f_K)` with :math:`f_j : \Rset^n \longrightarrow \Rset^d`. The trend function :math:`f_{trend}` writes:

.. math::

    f_{trend}(\vect{t}) = \sum_{j=1}^{K} \alpha_j f_j(\vect{t})

The coefficients :math:`\alpha_j \in \Rset` have to be computed. In the case where the number of available data is of the same order as *K*, the least square system is ill-posed and a  more complex algorithm should be used. Some algorithms combine cross validation techniques and advanced regression strategies, in order to provide the estimation of the  coefficients associated to the best model among the  basis functions (sparse model). For example, we can use the *least angle regression* (LARS) method for the choice of sparse models.  Then, some fitting algorithms like the *leave one out*, coupled to the regression strategy, assess the error on the prediction and enable the selection of the best sparse model.

)RAW"



// ---------------------------------------------------------------------
%feature("docstring") OT::TrendFactory::build
R"RAW(Estimate the trend of a process.

Available usages:
    build(*field, basis*)

Parameters
----------
field : :class:`~openturns.Field`
    One realization of the process.
basis : :class:`~openturns.Basis`
    A collection of functions composing the functional basis.

Returns
-------
myTrendTransform : :class:`~openturns.TrendTransform`
    The estimated trend function.

Examples
--------
Define a scalar temporal Gaussian process on a mesh of dimension 1:

>>> import openturns as ot
>>> myGrid = ot.RegularGrid(0.0, 1.0, 100)
>>> amplitude=[5.0]
>>> scale=[0.2]
>>> myCovModel=ot.ExponentialModel(scale, amplitude)
>>> myXProcess=ot.GaussianProcess(myCovModel, myGrid)

Create a trend function: :math:`f_{trend} : \Rset \mapsto \Rset` where :math:`f_{trend}(t)=1+2t+t^2`:

>>> fTrend = ot.SymbolicFunction(['t'], ['1+2*t+t^2'])
>>> fTemp = ot.TrendTransform(fTrend, myGrid)


Add the trend to the initial process and get a field:

>>> myYProcess = ot.CompositeProcess(fTemp, myXProcess)
>>> myYField = myYProcess.getRealization()

Estimate the trend function from the field:

>>> myBasisSequenceFactory = ot.LARS()
>>> myFittingAlgorithm = ot.KFold()
>>> func1 = ot.SymbolicFunction(['t'], ['1'])
>>> func2 = ot.SymbolicFunction(['t'], ['t'])
>>> func3 = ot.SymbolicFunction(['t'], ['t^2'])
>>> myBasis = ot.Basis([func1, func2, func3])

>>> myTrendFactory = ot.TrendFactory(myBasisSequenceFactory, myFittingAlgorithm)
>>> myTrendTransform =  myTrendFactory.build(myYField, myBasis)

>>> graph = myTrendTransform.getTrendFunction().draw(0.0, 10)
>>> graph.add(fTrend.draw(0.0, 10))
>>> graph.add(ot.Cloud(myYField.getMesh().getVertices(), myYField.getValues()))
>>> graph.setLegends(['estimated trend', 'actual trend', 'sample'])
>>> graph.setLegendPosition('upper left')
>>> graph.setTitle('Trend estimation from a field')
>>> graph.setYTitle('values')

)RAW"
// ---------------------------------------------------------------------
%feature("docstring") OT::TrendFactory::getBasisSequenceFactory
R"RAW(Accessor to functional basis.

Returns
-------
myBasisSequenceFactory : :class:`~openturns.BasisSequenceFactory`
    The functional basis :math:`\cB`.

)RAW"
// ---------------------------------------------------------------------
%feature("docstring") OT::TrendFactory::getFittingAlgorithm
"Accessor to fitting algorithm  basis.

Returns
-------
myFittingAlgorithm : :class:`~openturns.FittingAlgorithm`
    The fitting algorithm that estimates the empirical error on each sub-basis.

"


// ---------------------------------------------------------------------
%feature("docstring") OT::TrendFactory::setBasisSequenceFactory
R"RAW(Accessor to functional basis.

Parameters
----------
myBasisSequenceFactory : :class:`~openturns.BasisSequenceFactory`
    The functional basis :math:`\cB`.

)RAW"
// ---------------------------------------------------------------------
%feature("docstring") OT::TrendFactory::setFittingAlgorithm
"Accessor to fitting algorithm  basis.

Parameters
----------
myFittingAlgorithm : :class:`~openturns.FittingAlgorithm`
    The fitting algorithm that estimates the empirical error on each sub-basis.

"

