/*
 *  $Id: deconvolve.c 23556 2021-04-21 08:00:51Z yeti-dn $
 *  Copyright (C) 2018-2019 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/gwyprocesstypes.h>
#include <libprocess/arithmetic.h>
#include <libprocess/filters.h>
#include <libprocess/inttrans.h>
#include <libprocess/stats.h>
#include <libprocess/simplefft.h>
#include <libgwydgets/gwydgetutils.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwycheckboxes.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "preview.h"

#define DECONV_RUN_MODES GWY_RUN_INTERACTIVE

#define field_convolve_default(field, kernel) \
    gwy_data_field_area_ext_convolve((field), \
                                     0, 0, \
                                     gwy_data_field_get_xres(field), \
                                     gwy_data_field_get_yres(field), \
                                     (field), (kernel), \
                                     GWY_EXTERIOR_FIXED_VALUE, 0.0, TRUE) //GWY_EXTERIOR_BORDER_EXTEND

typedef enum {
    DECONV_DISPLAY_DATA        = 0,
    DECONV_DISPLAY_DECONVOLVED = 1,
    DECONV_DISPLAY_DIFFERENCE  = 2
} DeconvDisplayType;

typedef enum {
    DECONV_LCURVE_DIFFERENCE  = 0,
    DECONV_LCURVE_RMS         = 1,
    DECONV_LCURVE_CURVATURE   = 2,
    DECONV_LCURVE_LCURVE      = 3
} LcurveDisplayType;


typedef enum {
    DECONV_OUTPUT_DECONVOLVED = (1 << 0),
    DECONV_OUTPUT_DIFFERECE   = (1 << 1),
    DECONV_OUTPUTS_MASK       = (1 << 2) - 1,
} DeconvOutputType;

typedef struct {
    gdouble sigma;             /* Log10 */
    gdouble sigma_range;             /* Log10 */
    GwyAppDataId op1;
    GwyAppDataId op2;
    DeconvDisplayType display;
    LcurveDisplayType lcurve;
    DeconvOutputType output_type;
//    GwyWindowingType windowing;
    gboolean as_integral;
} DeconvArgs;

typedef struct {
    DeconvArgs *args;
    GtkObject *sigma;
    GtkObject *sigma_range;
    GtkWidget *chooser_op2;
    GtkWidget *display;
    GtkWidget *lcurve;
    GtkWidget *graph;
    GtkWidget *windowing;
    GtkWidget *actuallabel;
    GtkWidget *resultlabel;
    gdouble *sigmadata;
    gdouble *diffdata;
    gdouble *rmsdata;
    gdouble *lrmsdata;
    gdouble *ldiffdata;
    gdouble *curvaturedata;
    gdouble best_sigma;
    gboolean best_sigma_computed;
    GtkWidget *best_sigma_button;
    gint nsteps;
    GSList *output_type;
    GtkWidget *as_integral;
    GwyDataField *result;
    GwyDataField *dfield;
    GtkWidget *view;
    GwyContainer *mydata;
    GtkWidget *dialog;
    GwyGraphModel *gmodel;
    GwySelection *selection;
    gboolean in_update;
} DeconvControls;

static gboolean module_register              (void);
static void     deconvolve                   (GwyContainer *data,
                                              GwyRunType run);
static gboolean deconv_dialog                (DeconvArgs *args,
                                              GwyContainer *data);
static void     deconv_load_args             (GwyContainer *container,
                                              DeconvArgs *args);
static void     deconv_save_args             (GwyContainer *container,
                                              DeconvArgs *args);
static void     deconv_sanitize_args         (DeconvArgs *args);
static void     sigma_changed                (GtkAdjustment *adj,
                                              DeconvControls *controls);
static void     sigma_range_changed          (GtkAdjustment *adj,
                                              DeconvControls *controls);
//static void     windowing_changed            (GtkComboBox *combo,
//                                              DeconvControls *controls);
static void     deconv_data_changed          (GwyDataChooser *chooser,
                                              DeconvControls *controls);
static gboolean deconv_data_filter           (GwyContainer *data,
                                              gint id,
                                              gpointer user_data);
static void     preview                      (DeconvControls *controls,
                                              DeconvArgs *args);
static void     display_changed              (GtkComboBox *combo,
                                              DeconvControls *controls);
static void     lcurve_changed               (GtkComboBox *combo,
                                              DeconvControls *controls);
static void     as_integral_changed          (GtkToggleButton *toggle,
                                              DeconvControls *controls);
static void     output_type_changed          (GtkToggleButton *toggle,
                                              DeconvControls *controls);
static void     update_sensitivity           (DeconvControls *controls);
static void     deconvolve_with_tf           (GwyDataField *measured,
                                              GwyDataField *tf,
                                              GwyDataField *deconv,
                                              DeconvArgs *args);
static void     deconvolve_with_tf_sigma     (GwyDataField *measured,
                                              GwyDataField *tf,
                                              GwyDataField *deconv,
                                              gdouble sigma);
static void     adjust_deconv_to_non_integral(GwyDataField *psf);
static gint     create_output_field          (GwyDataField *dfield,
                                              GwyAppDataId dataid,
                                              const gchar *name);

static void     lcurve_run                   (DeconvControls *controls,
                                              DeconvArgs *args);
static void     lcurve_replot                (DeconvControls *controls,
                                              DeconvArgs *args);


static GwyAppDataId op2_id = GWY_APP_DATA_ID_NONE;

static const DeconvArgs deconv_defaults = {
    1.0, 1.0,
    GWY_APP_DATA_ID_NONE, GWY_APP_DATA_ID_NONE,
    DECONV_DISPLAY_DECONVOLVED, DECONV_LCURVE_CURVATURE, DECONV_OUTPUT_DECONVOLVED,
    /*GWY_WINDOWING_WELCH,*/ TRUE,
};

static const GwyEnum output_types[] = {
    { N_("Deconvolved"), DECONV_OUTPUT_DECONVOLVED, },
    { N_("Difference"),  DECONV_OUTPUT_DIFFERECE,   },
};

enum {
    OUTPUT_NTYPES = G_N_ELEMENTS(output_types)
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Regularized image deconvolution."),
    "Yeti <yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti)",
    "2018",
};

GWY_MODULE_QUERY2(module_info, deconvolve)

static gboolean
module_register(void)
{
    gwy_process_func_register("deconvolve",
                              (GwyProcessFunc)&deconvolve,
                              N_("/M_ultidata/_Deconvolve..."),
                              GWY_STOCK_DECONVOLVE,
                              DECONV_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Deconvolve image"));

    return TRUE;
}

static void
graph_selected(GwySelection* selection,
               G_GNUC_UNUSED gint i,
               DeconvControls *controls)
{
    gint nofselection;
    gdouble seldata;

    nofselection = gwy_selection_get_data(selection, &seldata);
    if (nofselection == 0)
        return;

    if (!controls->in_update)
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->sigma), seldata);
}

static void
update_graph_selection(DeconvControls *controls, DeconvArgs *args)
{
    GwyGraphArea *area;
    area = GWY_GRAPH_AREA(gwy_graph_get_area(GWY_GRAPH(controls->graph)));

    if (args->lcurve == DECONV_LCURVE_LCURVE) {
        gwy_graph_area_set_selection_editable(area, FALSE);
    }
    else {
        gwy_graph_area_set_selection_editable(area, TRUE);
    }
}

static void
update_resultlabel(DeconvControls *controls, DeconvArgs *args)
{
    gchar *s;

    s = g_strdup_printf(_("Actual sigma: %g"), pow10(args->sigma));
    gtk_label_set_text(GTK_LABEL(controls->actuallabel), s);
    g_free(s);

    if (controls->best_sigma_computed) {
        s = g_strdup_printf(_("Best estimate: %g (log10: %g)"),
                            pow10(controls->best_sigma), controls->best_sigma);
    }
    else
        s = g_strdup(_("Best estimate: unknown"));

    gtk_label_set_text(GTK_LABEL(controls->resultlabel), s);
    gtk_widget_set_sensitive(controls->best_sigma_button,
                             controls->best_sigma_computed);
    g_free(s);
}

static void
best_sigma_button_cb(DeconvControls *controls)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->sigma),
                             controls->best_sigma);
}

static void
deconvolve(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield1, *dfield2, *deconv, *difference;
    DeconvArgs args;
    gboolean ok;
    GQuark quark;
    GwyContainer *mydata;
    gint newid = -1;

    g_return_if_fail(run & DECONV_RUN_MODES);

    deconv_load_args(gwy_app_settings_get(), &args);

    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD_ID, &args.op1.id,
                                     GWY_APP_CONTAINER_ID, &args.op1.datano,
                                     0);

    mydata = gwy_app_data_browser_get(args.op1.datano);
    quark = gwy_app_get_data_key_for_id(args.op1.id);
    dfield1 = GWY_DATA_FIELD(gwy_container_get_object(mydata, quark));

    ok = deconv_dialog(&args, data);
    deconv_save_args(gwy_app_settings_get(), &args);
    if (!ok)
        return;

    mydata = gwy_app_data_browser_get(args.op2.datano);

    if (!mydata) return;

    quark = gwy_app_get_data_key_for_id(args.op2.id);
    dfield2 = GWY_DATA_FIELD(gwy_container_get_object(mydata, quark));

    if (!dfield2) return;

    deconv = gwy_data_field_new_alike(dfield1, TRUE);
    deconvolve_with_tf(dfield1, dfield2, deconv, &args);
    if (args.output_type & DECONV_OUTPUT_DECONVOLVED)
        newid = create_output_field(deconv, args.op1, _("Deconvolved"));
    if (args.output_type & DECONV_OUTPUT_DIFFERECE) {
        difference = gwy_data_field_duplicate(deconv);
        field_convolve_default(difference, dfield2);

        gwy_data_field_subtract_fields(difference, dfield1, difference);
        //gwy_data_field_add(difference, -gwy_data_field_get_avg(dfield1)); //FIXME: this is to handle the absolute value in the input data

        create_output_field(difference, args.op1, _("Difference"));
        g_object_unref(difference);
    }

    g_object_unref(deconv);


    /* Change the normalisation to the discrete (i.e. wrong) one after all
     * calculations are done. */
    if (!args.as_integral && newid != -1) {
        data = gwy_app_data_browser_get(args.op1.datano);
        quark = gwy_app_get_data_key_for_id(newid);
        deconv = gwy_container_get_object(data, quark);
        adjust_deconv_to_non_integral(deconv);
    }
}

static gboolean
deconv_dialog(DeconvArgs *args, GwyContainer *data)
{
    static const GwyEnum deconv_displays[] = {
        { N_("Data"),        DECONV_DISPLAY_DATA,        },
        { N_("Deconvolved"), DECONV_DISPLAY_DECONVOLVED, },
        { N_("Difference"),  DECONV_DISPLAY_DIFFERENCE,  },
    };

    static const GwyEnum deconv_lcurves[] = {
        { N_("Difference"), DECONV_LCURVE_DIFFERENCE,  },
        { N_("RMS"),        DECONV_LCURVE_RMS,         },
        { N_("Curvature"),  DECONV_LCURVE_CURVATURE,   },
        { N_("L-curve"),    DECONV_LCURVE_LCURVE,      },
    };

    GtkWidget *dialog, *table, *hbox, *label, *align;
    GwyDataChooser *chooser;
    DeconvControls controls;
    gint response, row;
    GQuark quark;
    GwyContainer *mydata;
    GwyGraphArea *area;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.in_update = TRUE;

    dialog = gtk_dialog_new_with_buttons(_("Deconvolve"), NULL, 0, NULL);

    gtk_dialog_add_action_widget(GTK_DIALOG(dialog),
                                 gwy_stock_like_button_new(_("_Update L-curve"),
                                                           GTK_STOCK_EXECUTE),
                                 RESPONSE_PREVIEW);

    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), GTK_WIDGET(hbox),
                       FALSE, FALSE, 4);

    mydata = gwy_app_data_browser_get(args->op1.datano);
    quark = gwy_app_get_data_key_for_id(args->op1.id);
    controls.dfield = GWY_DATA_FIELD(gwy_container_get_object(mydata, quark));
    controls.result = gwy_data_field_new_alike(controls.dfield, TRUE);

    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data",
                                     controls.result);
    gwy_app_sync_data_items(data, controls.mydata, args->op1.id, 0, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = gwy_create_preview(controls.mydata, 0, PREVIEW_SIZE, FALSE);
    align = gtk_alignment_new(0.5, 0.0, 0, 0);
    gtk_container_add(GTK_CONTAINER(align), controls.view);
    gtk_box_pack_start(GTK_BOX(hbox), align, FALSE, FALSE, 4);

    table = gtk_table_new(6 + G_N_ELEMENTS(deconv_displays), 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 4);
    row = 0;


    controls.gmodel = gwy_graph_model_new();
    controls.graph = gwy_graph_new(controls.gmodel);
    gtk_widget_set_size_request(controls.graph, -1, PREVIEW_HALF_SIZE);
    gwy_graph_set_status(GWY_GRAPH(controls.graph), GWY_GRAPH_STATUS_XLINES);
    gwy_graph_enable_user_input(GWY_GRAPH(controls.graph), TRUE);
    gtk_table_attach(GTK_TABLE(table), controls.graph,
                     0, 3, row, row+1, GTK_FILL, 0, 0, 0);

    area = GWY_GRAPH_AREA(gwy_graph_get_area(GWY_GRAPH(controls.graph)));
    controls.selection = gwy_graph_area_get_selection(area, GWY_GRAPH_STATUS_XLINES);

    gwy_selection_set_max_objects(controls.selection, 1);
    g_signal_connect(controls.selection, "changed",
                     G_CALLBACK(graph_selected), &controls);

    update_graph_selection(&controls, args);

    row++;


    controls.chooser_op2 = gwy_data_chooser_new_channels();
    chooser = GWY_DATA_CHOOSER(controls.chooser_op2);
    gwy_data_chooser_set_active_id(chooser, &args->op2);
    gwy_data_chooser_set_filter(chooser, deconv_data_filter, &args->op1, NULL);
    g_signal_connect(chooser, "changed",
                     G_CALLBACK(deconv_data_changed), &controls);
    gwy_table_attach_adjbar(table, row, _("Convolution _kernel:"), NULL,
                            GTK_OBJECT(chooser), GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    /*
    controls.windowing
        = gwy_enum_combo_box_new(gwy_windowing_type_get_enum(), -1,
                                 G_CALLBACK(windowing_changed), &controls,
                                 args->windowing, TRUE);
    gwy_table_attach_adjbar(GTK_WIDGET(table), row, _("_Windowing type:"), NULL,
                            GTK_OBJECT(controls.windowing),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    row++;
    */


    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.sigma = gtk_adjustment_new(args->sigma, -8.0, 8.0, 0.001, 1.0, 0);
    gwy_table_attach_adjbar(table, row, _("_Sigma:"), "log<sub>10</sub>",
                            controls.sigma, GWY_HSCALE_LINEAR);
    g_signal_connect(controls.sigma, "value-changed",
                     G_CALLBACK(sigma_changed), &controls);
    row++;

    controls.actuallabel = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(controls.actuallabel), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), controls.actuallabel,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);

    row++;

    controls.resultlabel = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(controls.resultlabel), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), controls.resultlabel,
                     0, 1, row, row+1, GTK_FILL, 0, 0, 0);


    controls.best_sigma_button = gtk_button_new_with_mnemonic(_("_Use"));
    g_signal_connect_swapped(controls.best_sigma_button, "clicked",
                             G_CALLBACK(best_sigma_button_cb), &controls);

    gtk_table_attach(GTK_TABLE(table), controls.best_sigma_button,
                     1, 2, row, row+1, GTK_FILL, 0, 0, 0);

    row++;



    controls.lcurve
        = gwy_enum_combo_box_new(deconv_lcurves, G_N_ELEMENTS(deconv_lcurves),
                                 G_CALLBACK(lcurve_changed),
                                 &controls, args->lcurve, TRUE);
    gwy_table_attach_adjbar(table, row, _("_L-curve display:"), NULL,
                            GTK_OBJECT(controls.lcurve),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;


    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.sigma_range = gtk_adjustment_new(args->sigma_range,
                                              -8.0, 8.0, 0.001, 1.0, 0);
    gwy_table_attach_adjbar(table, row, _("_L-curve sigma range:"),
                            "log<sub>10</sub>",
                            controls.sigma_range, GWY_HSCALE_LINEAR);
    g_signal_connect(controls.sigma_range, "value-changed",
                     G_CALLBACK(sigma_range_changed), &controls);
    row++;


    controls.display
        = gwy_enum_combo_box_new(deconv_displays, G_N_ELEMENTS(deconv_displays),
                                 G_CALLBACK(display_changed),
                                 &controls, args->display, TRUE);
    gwy_table_attach_adjbar(table, row, _("_Display:"), NULL,
                            GTK_OBJECT(controls.display),
                            GWY_HSCALE_WIDGET_NO_EXPAND);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    label = gtk_label_new(_("Output type:"));
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), label, 0, 2, row, row+1,
                     GTK_FILL, 0, 0, 0);
    row++;

    controls.output_type
        = gwy_check_boxes_create(output_types, OUTPUT_NTYPES,
                                 G_CALLBACK(output_type_changed), &controls,
                                 args->output_type);
    row = gwy_check_boxes_attach_to_table(controls.output_type,
                                          GTK_TABLE(table), 2, row);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls.as_integral
        = gtk_check_button_new_with_mnemonic(_("Normalize as _integral"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.as_integral),
                                 args->as_integral);
    g_signal_connect(controls.as_integral, "toggled",
                     G_CALLBACK(as_integral_changed), &controls);
    gtk_table_attach(GTK_TABLE(table), controls.as_integral,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls.in_update = FALSE;
    update_sensitivity(&controls);
    update_resultlabel(&controls, args);
    deconv_data_changed(chooser, &controls);

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            g_object_unref(controls.mydata);
            case GTK_RESPONSE_NONE:
            return FALSE;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_PREVIEW:
            lcurve_run(&controls, args);
            lcurve_replot(&controls, args);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);
    g_object_unref(controls.mydata);

    return TRUE;
}

static void
display_changed(GtkComboBox *combo, DeconvControls *controls)
{
    controls->args->display = gwy_enum_combo_box_get_active(combo);
    preview(controls, controls->args);
}

static void
lcurve_changed(GtkComboBox *combo, DeconvControls *controls)
{
    controls->args->lcurve = gwy_enum_combo_box_get_active(combo);
    lcurve_replot(controls, controls->args);
}

static void
as_integral_changed(GtkToggleButton *toggle, DeconvControls *controls)
{
    controls->args->as_integral = gtk_toggle_button_get_active(toggle);
    preview(controls, controls->args);
}

static void
sigma_changed(GtkAdjustment *adj, DeconvControls *controls)
{
    gint nofselection;
    gdouble seldata;

    controls->args->sigma = gtk_adjustment_get_value(adj);
    preview(controls, controls->args);
    update_resultlabel(controls, controls->args);


    if (controls->best_sigma_computed) {
        controls->in_update = TRUE;
        //printf("sigma changed\n");

        nofselection = gwy_selection_get_data(controls->selection, &seldata);
        if (nofselection == 0) {
            //printf("no sel\n");
            controls->in_update = FALSE;
            return;
        }

        seldata = controls->args->sigma;
        gwy_selection_set_data(controls->selection, 1, &seldata);

        controls->in_update = FALSE;
    }
}

static void
sigma_range_changed(GtkAdjustment *adj, DeconvControls *controls)
{
    controls->args->sigma_range = gtk_adjustment_get_value(adj);
    gwy_graph_model_remove_all_curves(controls->gmodel);
}

/*
static void
windowing_changed(GtkComboBox *combo, DeconvControls *controls)
{
    controls->args->windowing = gwy_enum_combo_box_get_active(combo);
    preview(controls, controls->args);
}
*/

static void
output_type_changed(G_GNUC_UNUSED GtkToggleButton *toggle,
                    DeconvControls *controls)
{
    DeconvArgs *args = controls->args;

    args->output_type = gwy_check_boxes_get_selected(controls->output_type);
    update_sensitivity(controls);
}

static void
update_sensitivity(DeconvControls *controls)
{
    DeconvArgs *args = controls->args;
    gboolean have_tf = args->op2.datano && args->output_type;
    gboolean any_output = args->output_type;

    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialog),
                                      GTK_RESPONSE_OK,
                                      have_tf && any_output);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialog),
                                      RESPONSE_ESTIMATE, have_tf);
}

static gint
create_output_field(GwyDataField *dfield,
                    GwyAppDataId dataid,
                    const gchar *name)
{
    GwyContainer *data;
    gint newid;

    data = gwy_app_data_browser_get(dataid.datano);
    newid = gwy_app_data_browser_add_data_field(dfield, data, TRUE);

    gwy_app_sync_data_items(data, data, dataid.id, newid, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_MASK_COLOR,
                            0);
    gwy_app_set_data_field_title(data, newid, name);
    gwy_app_channel_log_add_proc(data, dataid.id, newid);

    return newid;
}

static void
preview(DeconvControls *controls, DeconvArgs *args)
{
    GwyDataField *dfield1, *dfield2, *deconv, *difference;
    GQuark quark;
    GwyContainer *mydata;

    mydata = gwy_app_data_browser_get(args->op1.datano);
    quark = gwy_app_get_data_key_for_id(args->op1.id);
    dfield1 = GWY_DATA_FIELD(gwy_container_get_object(mydata, quark));

    if (args->op2.datano < 0 || args->op2.id < 0) {
        if (args->display == DECONV_DISPLAY_DATA)
            gwy_data_field_copy(dfield1, controls->result, TRUE);
        else
            gwy_data_field_clear(controls->result);

        gwy_data_field_data_changed(controls->result);
        return;
    }

    mydata = gwy_app_data_browser_get(args->op2.datano);

    if (!mydata)
        return;

    quark = gwy_app_get_data_key_for_id(args->op2.id);
    dfield2 = GWY_DATA_FIELD(gwy_container_get_object(mydata, quark));

    if (!dfield2)
        return;

    deconv = gwy_data_field_new_alike(dfield1, TRUE);

    deconvolve_with_tf(dfield1, dfield2, deconv, args);

    if (args->display == DECONV_DISPLAY_DATA) {
        gwy_data_field_assign(controls->result, dfield1);
        gwy_data_field_data_changed(controls->result);
    }
    else if (args->display == DECONV_DISPLAY_DECONVOLVED) {
        gwy_data_field_assign(controls->result, deconv);
        gwy_data_field_data_changed(controls->result);
    }
    else {
        difference = gwy_data_field_duplicate(deconv);
        field_convolve_default(difference, dfield2);

        //gwy_data_field_add(difference, gwy_data_field_get_avg(dfield1));

        gwy_data_field_subtract_fields(difference, dfield1, difference);
        gwy_data_field_assign(controls->result, difference);
        gwy_data_field_data_changed(controls->result);
        gwy_object_unref(difference);
    }

    /* Change the normalisation to the discrete (i.e. wrong) one after all
     * calculations are done. */
    if (!args->as_integral)
        adjust_deconv_to_non_integral(deconv);

    g_object_unref(deconv);
}

static void
deconv_data_changed(GwyDataChooser *chooser, DeconvControls *controls)
{
    GwyAppDataId *object = &controls->args->op2;

    gwy_data_chooser_get_active_id(chooser, object);
    gwy_debug("data: %d %d", object->datano, object->id);
    update_sensitivity(controls);
    preview(controls, controls->args);
}

static gboolean
deconv_data_filter(GwyContainer *data, gint id, gpointer user_data)
{

    GwyAppDataId *object = (GwyAppDataId*)user_data;
    GwyDataField *op1, *op2;
    gint xresimg, yresimg, xresop, yresop;
    GQuark quark;

    quark = gwy_app_get_data_key_for_id(id);
    op1 = GWY_DATA_FIELD(gwy_container_get_object(data, quark));

    data = gwy_app_data_browser_get(object->datano);
    quark = gwy_app_get_data_key_for_id(object->id);
    op2 = GWY_DATA_FIELD(gwy_container_get_object(data, quark));

    if (op1 == op2)
        return FALSE;

    xresop = gwy_data_field_get_xres(op1);
    yresop = gwy_data_field_get_yres(op1);
    xresimg = gwy_data_field_get_xres(op2);
    yresimg = gwy_data_field_get_yres(op2);
    if (xresop > xresimg || yresop > yresimg)
        return FALSE;

    return !gwy_data_field_check_compatibility(op1, op2,
                                               GWY_DATA_COMPATIBILITY_MEASURE
                                               | GWY_DATA_COMPATIBILITY_LATERAL);
}

static void
deconvolve_with_tf(GwyDataField *measured, GwyDataField *tf,
                   GwyDataField *deconv, DeconvArgs *args)
{
    GwyDataField *xm, *xtf;
    gdouble sigma = pow10(args->sigma);
    gint xres, yres, extx, exty, txres, tyres;

    xres = gwy_data_field_get_xres(measured);
    yres = gwy_data_field_get_yres(measured);

    txres = gwy_data_field_get_xres(tf);
    tyres = gwy_data_field_get_yres(tf);


    extx = txres/2 + 1;
    exty = tyres/2 + 1;
    xm = gwy_data_field_extend(measured, extx, extx, exty, exty,
                               GWY_EXTERIOR_MIRROR_EXTEND, 0.0, FALSE);

    xtf = gwy_data_field_new_alike(xm, TRUE);
    gwy_data_field_copy_units(tf, xtf);
    gwy_data_field_area_copy(tf, xtf,
                             0, 0, txres, tyres,
                             xres/2 + extx - txres/2, yres/2 + exty - tyres/2);

    // gwy_data_field_add(xm, -gwy_data_field_get_avg(xm));

    gwy_data_field_deconvolve_regularized(xm, xtf, deconv, sigma);
    g_object_unref(xtf);
    g_object_unref(xm);

    gwy_data_field_resize(deconv, extx, exty, xres + extx, yres + exty);

 }

static void
deconvolve_with_tf_sigma(GwyDataField *measured, GwyDataField *tf,
                   GwyDataField *deconv, gdouble sigma)
{
    GwyDataField *xm, *xtf;
    gint xres, yres, extx, exty, txres, tyres;

    xres = gwy_data_field_get_xres(measured);
    yres = gwy_data_field_get_yres(measured);
    txres = gwy_data_field_get_xres(tf);
    tyres = gwy_data_field_get_yres(tf);
    extx = txres/2 + 1;
    exty = tyres/2 + 1;
    xm = gwy_data_field_extend(measured, extx, extx, exty, exty,
                               GWY_EXTERIOR_MIRROR_EXTEND, 0.0, FALSE);

    xtf = gwy_data_field_new_alike(xm, TRUE);
    gwy_data_field_copy_units(tf, xtf);
    gwy_data_field_area_copy(tf, xtf,
                             0, 0, txres, tyres,
                             xres/2 + extx - txres/2, yres/2 + exty - tyres/2);
    gwy_data_field_deconvolve_regularized(xm, xtf, deconv, sigma);
    g_object_unref(xtf);
    g_object_unref(xm);
    gwy_data_field_resize(deconv, extx, exty, xres + extx, yres + exty);
}

/*
static void
deconvolve_sigma(GwyDataField *measured, GwyDataField *tf,
                   GwyDataField *deconv, gdouble sigma)
{
    gwy_data_field_deconvolve_regularized(measured, tf, deconv, sigma);
}
*/

static void
get_curvatures(gdouble *xdata, gdouble *ydata, gdouble *curvaturedata, gint nsteps)
{
    gint i;
    gdouble xd, yd, xdd, ydd;

    for (i = 0; i < nsteps; i++) {
        if (i == 0 || i == nsteps-1) {
            xd = yd = xdd = ydd = 0;
            curvaturedata[i] = 0;
        }
        else {
            xd = (xdata[i+1] - xdata[i-1])/2;
            yd = (ydata[i+1] - ydata[i-1])/2;

            xdd = (xdata[i+1] + xdata[i-1] - 2*xdata[i])/4;
            ydd = (ydata[i+1] + ydata[i-1] - 2*ydata[i])/4;
            if (xd*xd + yd*yd)
                curvaturedata[i] = (xd*ydd - yd*xdd)/pow(xd*xd + yd*yd, 1.5);
            else
                curvaturedata[i] = 0;
        }
    }
}

static void
lcurve_replot(DeconvControls *controls, DeconvArgs *args)
{
    GwyGraphCurveModel *gcmodel;

    gwy_graph_model_remove_all_curves(controls->gmodel);

    if (controls->nsteps == 0)
        return;

    gcmodel = gwy_graph_curve_model_new();
    if (args->lcurve == DECONV_LCURVE_DIFFERENCE) {
       gwy_graph_curve_model_set_data(gcmodel,
                                      controls->sigmadata, controls->diffdata,
                                      controls->nsteps);
       g_object_set(gcmodel,
                    "mode", GWY_GRAPH_CURVE_LINE,
                    "description", _("Difference"),
                    NULL);
    }
    else if (args->lcurve == DECONV_LCURVE_RMS) {
       gwy_graph_curve_model_set_data(gcmodel,
                                      controls->sigmadata, controls->rmsdata,
                                      controls->nsteps);
       g_object_set(gcmodel,
                    "mode", GWY_GRAPH_CURVE_LINE,
                    "description", _("RMS"),
                    NULL);
    }
    else if (args->lcurve == DECONV_LCURVE_CURVATURE) {
        gwy_graph_curve_model_set_data(gcmodel,
                                       controls->sigmadata+1,
                                       controls->curvaturedata+1,
                                       controls->nsteps-2);
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "description", _("Curvature"),
                     NULL);
    }
    else {
        gwy_graph_curve_model_set_data(gcmodel,
                                       controls->lrmsdata, controls->ldiffdata,
                                       controls->nsteps);
        g_object_set(gcmodel,
                     "mode", GWY_GRAPH_CURVE_LINE,
                     "description", _("L-curve"),
                     NULL);
    }

    if (args->lcurve == DECONV_LCURVE_LCURVE) {
        g_object_set(controls->gmodel,
                     "axis-label-bottom", "log ||G-FH||",
                     "axis-label-left", "log ||F||",
                     NULL);
    }
    else {
        g_object_set(controls->gmodel,
                     "axis-label-bottom", "log(sigma)",
                     "axis-label-left", "",
                     NULL);
    }

    gwy_graph_model_add_curve(controls->gmodel, gcmodel);
    update_graph_selection(controls, args);
}


static void
lcurve_run(DeconvControls *controls, DeconvArgs *args)
{
    gdouble lsigma, sigma, max, maxsigma;
    gint i, maxsigmapos, nsteps = 20;
    GwyDataField *dfield1, *ldfield1, *dfield2,
                 *deconv, *difference;
    GQuark quark;
    GwyContainer *mydata;

    mydata = gwy_app_data_browser_get(args->op1.datano);
    quark = gwy_app_get_data_key_for_id(args->op1.id);
    dfield1 = GWY_DATA_FIELD(gwy_container_get_object(mydata, quark));

    mydata = gwy_app_data_browser_get(args->op2.datano);

    if (!mydata)
        return;

    quark = gwy_app_get_data_key_for_id(args->op2.id);
    dfield2 = GWY_DATA_FIELD(gwy_container_get_object(mydata, quark));

    if (!dfield2)
        return;

    gwy_app_wait_start(GTK_WINDOW(controls->dialog),
                       _("Computing L-curve data..."));

    deconv = gwy_data_field_new_alike(dfield1, TRUE);
    ldfield1 = gwy_data_field_new_alike(dfield1, TRUE);
    difference = gwy_data_field_new_alike(dfield1, TRUE);

    g_free(controls->diffdata);
    g_free(controls->rmsdata);
    g_free(controls->curvaturedata);
    g_free(controls->sigmadata);

    controls->diffdata = g_new(gdouble, nsteps);
    controls->rmsdata = g_new(gdouble, nsteps);
    controls->ldiffdata = g_new(gdouble, nsteps);
    controls->lrmsdata = g_new(gdouble, nsteps);
    controls->curvaturedata = g_new(gdouble, nsteps);
    controls->sigmadata = g_new(gdouble, nsteps);
    controls->nsteps = nsteps;

    gwy_data_field_copy(dfield1, ldfield1, TRUE);

    gwy_data_field_add(ldfield1, -gwy_data_field_get_avg(dfield1));

    for (i = 0; i < nsteps; i++) {
        if (!gwy_app_wait_set_fraction((gdouble)i/(gdouble)nsteps)) {
            controls->nsteps = i;
            break;
        }

        lsigma = args->sigma - args->sigma_range/2 + i*args->sigma_range/(nsteps-1);
        sigma = pow10(lsigma);

        controls->sigmadata[i] = lsigma;
        gwy_data_field_fill(deconv, 0);

        deconvolve_with_tf_sigma(ldfield1, dfield2, deconv, sigma);

        gwy_data_field_copy(deconv, difference, FALSE);
//        difference = gwy_data_field_duplicate(deconv);

        field_convolve_default(difference, dfield2);

        gwy_data_field_subtract_fields(difference, ldfield1, difference);

        controls->diffdata[i] = sqrt(gwy_data_field_get_mean_square(difference));
        controls->rmsdata[i] = gwy_data_field_get_rms(deconv);
        controls->ldiffdata[i] = log(controls->diffdata[i]);
        controls->lrmsdata[i] = log(controls->rmsdata[i]);


    }

    get_curvatures(controls->ldiffdata, controls->lrmsdata,
                   controls->curvaturedata, nsteps);

    gwy_app_wait_finish();


    max = -G_MAXDOUBLE;
    maxsigmapos = 0;
    for (i = 1; i < nsteps-1; i++) {
         if (controls->curvaturedata[i]>max) {
             max = controls->curvaturedata[i];
             maxsigmapos = i;
             maxsigma = controls->sigmadata[i];
         }
    }
    if (maxsigmapos > 0) {
        controls->best_sigma_computed = TRUE;
        controls->best_sigma = maxsigma;

        update_resultlabel(controls, args);
    }

    gwy_object_unref(deconv);
    gwy_object_unref(ldfield1);
    gwy_object_unref(difference);
}

static void
adjust_deconv_to_non_integral(GwyDataField *deconv)
{
    GwySIUnit *xyunit, *zunit;
    gdouble hxhy;

    xyunit = gwy_data_field_get_si_unit_xy(deconv);
    zunit = gwy_data_field_get_si_unit_z(deconv);
    gwy_si_unit_power_multiply(zunit, 1, xyunit, 2, zunit);

    hxhy = gwy_data_field_get_dx(deconv) * gwy_data_field_get_dy(deconv);
    gwy_data_field_multiply(deconv, hxhy);
    gwy_data_field_data_changed(deconv);
}

static const gchar as_integral_key[] = "/module/deconvolve/as_integral";
static const gchar display_key[]     = "/module/deconvolve/display";
static const gchar lcurve_key[]      = "/module/deconvolve/lcurve";
static const gchar output_type_key[] = "/module/deconvolve/output_type";
static const gchar sigma_key[]       = "/module/deconvolve/sigma";
static const gchar sigma_range_key[] = "/module/deconvolve/sigma_range";
//static const gchar windowing_key[]   = "/module/deconvolve/windowing";

static void
deconv_sanitize_args(DeconvArgs *args)
{
    gwy_app_data_id_verify_channel(&args->op2);
    args->output_type &= DECONV_OUTPUTS_MASK;
    args->sigma = CLAMP(args->sigma, -8.0, 8.0);
    args->sigma_range = CLAMP(args->sigma_range, -8.0, 8.0);
    args->as_integral = !!args->as_integral;
    args->display = CLAMP(args->display, 0, DECONV_DISPLAY_DIFFERENCE);
    args->lcurve = CLAMP(args->lcurve, 0, DECONV_LCURVE_CURVATURE);
//    args->windowing = gwy_enum_sanitize_value(args->windowing,
//                                              GWY_TYPE_WINDOWING_TYPE);
}

static void
deconv_load_args(GwyContainer *container, DeconvArgs *args)
{
    *args = deconv_defaults;

    gwy_container_gis_double_by_name(container, sigma_key, &args->sigma);
    gwy_container_gis_double_by_name(container, sigma_range_key,
                                     &args->sigma_range);
    gwy_container_gis_enum_by_name(container, display_key, &args->display);
    gwy_container_gis_enum_by_name(container, lcurve_key, &args->lcurve);
    gwy_container_gis_enum_by_name(container, output_type_key,
                                   &args->output_type);
//    gwy_container_gis_enum_by_name(container, windowing_key, &args->windowing);
    gwy_container_gis_boolean_by_name(container, as_integral_key,
                                      &args->as_integral);
    args->op2 = op2_id;

    deconv_sanitize_args(args);
}

static void
deconv_save_args(GwyContainer *container, DeconvArgs *args)
{
    op2_id = args->op2;

    gwy_container_set_double_by_name(container, sigma_key, args->sigma);
    gwy_container_set_double_by_name(container, sigma_range_key,
                                     args->sigma_range);
    gwy_container_set_enum_by_name(container, display_key, args->display);
    gwy_container_set_enum_by_name(container, lcurve_key, args->lcurve);
    gwy_container_set_enum_by_name(container, output_type_key,
                                   args->output_type);
//    gwy_container_set_enum_by_name(container, windowing_key, args->windowing);
    gwy_container_set_boolean_by_name(container, as_integral_key,
                                      args->as_integral);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
