/*
 *  $Id: gaussian_step.c 22097 2019-05-27 10:07:49Z yeti-dn $
 *  Copyright (C) 2003-2019 David Necas (Yeti), Petr Klapetek.
 *  E-mail: yeti@gwyddion.net, klapetek@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <string.h>
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libprocess/gwyprocess.h>
#include <libgwydgets/gwyradiobuttons.h>
#include <libgwydgets/gwystock.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwyapp.h>
#include "preview.h"

#define GAUSSIAN_STEP_RUN_MODES (GWY_RUN_IMMEDIATE | GWY_RUN_INTERACTIVE)

#define FWHM2SIGMA (1.0/(2.0*sqrt(2*G_LN2)))

typedef struct {
    gdouble gaussian_fwhm;
    gboolean update;
} GaussianStepArgs;

typedef struct {
    GaussianStepArgs *args;
    GtkWidget *dialog;
    GtkWidget *view;
    GtkObject *gaussian_fwhm;
    GtkWidget *update;
    GwyContainer *mydata;
    GwyDataField *dfield;
    gboolean computed;
    gboolean in_init;
} GaussianStepControls;

static gboolean module_register      (void);
static void     gaussian_step        (GwyContainer *data,
                                      GwyRunType run);
static void     gaussian_step_dialog (GaussianStepArgs *args,
                                      GwyContainer *data,
                                      GwyDataField *dfield,
                                      gint id,
                                      GQuark squark);
static void     gaussian_fwhm_changed(GtkAdjustment *adj,
                                      GaussianStepControls *controls);
static void     update_changed       (GtkToggleButton *check,
                                      GaussianStepControls *controls);
static void     invalidate           (GaussianStepControls *controls);
static void     update_controls      (GaussianStepControls *controls,
                                      GaussianStepArgs *args);
static void     preview              (GaussianStepControls *controls,
                                      GaussianStepArgs *args);
static void     gaussian_step_run    (const GaussianStepArgs *args,
                                      GwyContainer *data,
                                      GwyDataField *dfield,
                                      GQuark squark);
static void     load_args            (GwyContainer *container,
                                      GaussianStepArgs *args);
static void     save_args            (GwyContainer *container,
                                      GaussianStepArgs *args);

static const GaussianStepArgs gaussian_step_defaults = {
    3.5,
    TRUE,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Edge detection using a Gaussian step filter."),
    "Yeti <Yeti@gwyddion.net>",
    "1.0",
    "David Nečas (Yeti) & Petr Klapetek",
    "2019",
};

GWY_MODULE_QUERY2(module_info, gaussian_step)

static gboolean
module_register(void)
{
    gwy_process_func_register("gaussian_step",
                              (GwyProcessFunc)&gaussian_step,
                              N_("/_Presentation/_Edge Detection/_Gaussian Step..."),
                              NULL,
                              GAUSSIAN_STEP_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Detect edges using Gaussian step filter"));

    return TRUE;
}

static void
gaussian_step(GwyContainer *data, GwyRunType run)
{
    GaussianStepArgs args;
    GwyDataField *dfield;
    GQuark squark;
    gint id;

    g_return_if_fail(run & GAUSSIAN_STEP_RUN_MODES);
    load_args(gwy_app_settings_get(), &args);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     GWY_APP_SHOW_FIELD_KEY, &squark,
                                     0);
    g_return_if_fail(dfield && squark);

    if (run == GWY_RUN_IMMEDIATE)
        gaussian_step_run(&args, data, dfield, squark);
    else {
        gaussian_step_dialog(&args, data, dfield, id, squark);
        save_args(gwy_app_settings_get(), &args);
    }
}

static void
gaussian_step_dialog(GaussianStepArgs *args,
                     GwyContainer *data,
                     GwyDataField *dfield,
                     gint id,
                     GQuark squark)
{
    GtkWidget *dialog, *table, *hbox;
    GwyDataField *otherfield;
    GtkObject *adj;
    GaussianStepControls controls;
    gint response;
    GwyDataField *sfield;
    gint row;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.in_init = TRUE;
    controls.dfield = dfield;

    dialog = gtk_dialog_new_with_buttons(_("Gaussian Step Detection"),
                                         NULL, 0, NULL);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialog),
                                 gwy_stock_like_button_new(_("_Update"),
                                                           GTK_STOCK_EXECUTE),
                                 RESPONSE_PREVIEW);
    gtk_dialog_add_button(GTK_DIALOG(dialog), _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(GTK_DIALOG(dialog),
                          GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(GTK_DIALOG(dialog), GWY_HELP_DEFAULT);
    controls.dialog = dialog;

    hbox = gtk_hbox_new(FALSE, 2);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox,
                       FALSE, FALSE, 4);

    controls.mydata = gwy_container_new();
    otherfield = gwy_data_field_new_alike(dfield, TRUE);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(otherfield), NULL);
    gwy_container_set_object_by_name(controls.mydata, "/0/data", otherfield);
    gwy_app_sync_data_items(data, controls.mydata, id, 0, FALSE,
                            GWY_DATA_ITEM_PALETTE,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, FALSE);
    gtk_box_pack_start(GTK_BOX(hbox), controls.view, FALSE, FALSE, 4);

    table = gtk_table_new(3, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 4);
    row = 0;

    adj = gtk_adjustment_new(args->gaussian_fwhm, 0.0, 20.0, 0.1, 1.0, 0);
    controls.gaussian_fwhm = adj;
    gwy_table_attach_adjbar(table, row, _("_Gaussian FWHM:"), _("px"),
                            adj, GWY_HSCALE_SQRT);
    g_signal_connect(adj, "value-changed",
                     G_CALLBACK(gaussian_fwhm_changed),
                     &controls);
    row++;

    controls.update = gtk_check_button_new_with_mnemonic(_("I_nstant updates"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls.update),
                                 args->update);
    gtk_table_attach(GTK_TABLE(table), controls.update,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls.update, "toggled",
                     G_CALLBACK(update_changed), &controls);

    /* finished initializing, allow instant updates */
    controls.in_init = FALSE;

    /* show initial preview if instant updates are on */
    if (args->update) {
        gtk_dialog_set_response_sensitive(GTK_DIALOG(controls.dialog),
                                          RESPONSE_PREVIEW, FALSE);
        preview(&controls, args);
    }

    gtk_widget_show_all(dialog);
    do {
        response = gtk_dialog_run(GTK_DIALOG(dialog));
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(dialog);
            case GTK_RESPONSE_NONE:
            g_object_unref(controls.mydata);
            return;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            args->gaussian_fwhm = gaussian_step_defaults.gaussian_fwhm;
            controls.in_init = TRUE;
            update_controls(&controls, args);
            controls.in_init = FALSE;
            preview(&controls, args);
            break;

            case RESPONSE_PREVIEW:
            preview(&controls, args);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    gtk_widget_destroy(dialog);

    if (controls.computed) {
        sfield = gwy_container_get_object_by_name(controls.mydata, "/0/data");
        gwy_app_undo_qcheckpointv(data, 1, &squark);
        gwy_container_set_object(data, squark, sfield);
        g_object_unref(controls.mydata);
    }
    else {
        g_object_unref(controls.mydata);
        gaussian_step_run(args, data, dfield, squark);
    }
}

static void
gaussian_fwhm_changed(GtkAdjustment *adj, GaussianStepControls *controls)
{
    controls->args->gaussian_fwhm = gtk_adjustment_get_value(adj);
    invalidate(controls);
}

static void
update_changed(GtkToggleButton *check, GaussianStepControls *controls)
{
    controls->args->update = gtk_toggle_button_get_active(check);
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialog),
                                      RESPONSE_PREVIEW,
                                      !controls->args->update);
    if (controls->args->update)
        preview(controls, controls->args);
}

static void
invalidate(GaussianStepControls *controls)
{
    controls->computed = FALSE;
    if (controls->args->update && !controls->in_init)
        preview(controls, controls->args);
}

static void
update_controls(GaussianStepControls *controls, GaussianStepArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->gaussian_fwhm),
                             args->gaussian_fwhm);
}

static void
preview(GaussianStepControls *controls, GaussianStepArgs *args)
{
    GwyDataField *edges;

    if (controls->computed)
        return;

    edges = gwy_container_get_object_by_name(controls->mydata, "/0/data");
    gwy_data_field_copy(controls->dfield, edges, FALSE);
    gwy_data_field_filter_gauss_step(edges, args->gaussian_fwhm*FWHM2SIGMA);
    gwy_data_field_normalize(edges);
    gwy_data_field_data_changed(edges);
    controls->computed = TRUE;
}

static void
gaussian_step_run(const GaussianStepArgs *args,
                  GwyContainer *data,
                  GwyDataField *dfield,
                  GQuark squark)
{
    GwyDataField *edges;

    gwy_app_undo_qcheckpointv(data, 1, &squark);
    edges = gwy_data_field_duplicate(dfield);
    gwy_si_unit_set_from_string(gwy_data_field_get_si_unit_z(edges), NULL);
    gwy_data_field_filter_gauss_step(edges, args->gaussian_fwhm*FWHM2SIGMA);
    gwy_data_field_normalize(edges);
    gwy_container_set_object(data, squark, edges);
    g_object_unref(edges);
}

static const gchar gaussian_fwhm_key[] = "/module/gaussian_step/gaussian-fwhm";
static const gchar update_key[]        = "/module/gaussian_step/update";

static void
sanitize_args(GaussianStepArgs *args)
{
    args->gaussian_fwhm = CLAMP(args->gaussian_fwhm, 0.0, 20.0);
    args->update = !!args->update;
}

static void
load_args(GwyContainer *container, GaussianStepArgs *args)
{
    *args = gaussian_step_defaults;

    gwy_container_gis_double_by_name(container, gaussian_fwhm_key,
                                     &args->gaussian_fwhm);
    gwy_container_gis_boolean_by_name(container, update_key,
                                      &args->update);

    sanitize_args(args);
}

static void
save_args(GwyContainer *container, GaussianStepArgs *args)
{
    gwy_container_set_double_by_name(container, gaussian_fwhm_key,
                                     args->gaussian_fwhm);
    gwy_container_set_boolean_by_name(container, update_key,
                                      args->update);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
