#!/usr/bin/env python3
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

from fenrirscreenreader.core import debug


def get_prev_char(curr_x, curr_y, curr_text):
    line_break = False
    end_of_screen = False
    if curr_text == "":
        return -1, -1, "", end_of_screen, line_break
    wrapped_lines = curr_text.split("\n")
    x = curr_x
    y = curr_y
    if x - 1 < 0:
        if y - 1 >= 0:
            y -= 1
            x = len(wrapped_lines[y]) - 1
            line_break = True
        else:
            line_break = False
            end_of_screen = True
    else:
        x -= 1
    curr_char = ""
    if not end_of_screen:
        curr_char = wrapped_lines[y][x]
    return x, y, curr_char, end_of_screen, line_break


def get_current_char(curr_x, curr_y, curr_text):
    if curr_text == "":
        return -1, -1, ""
    wrapped_lines = curr_text.split("\n")
    curr_char = wrapped_lines[curr_y][curr_x]
    return curr_x, curr_y, curr_char


def get_up_char(curr_x, curr_y, curr_text):
    end_of_screen = False
    if curr_text == "":
        return -1, -1, "", end_of_screen
    wrapped_lines = curr_text.split("\n")
    curr_y -= 1
    if curr_y < 0:
        curr_y = 0
    else:
        end_of_screen = True
    curr_char = ""
    if not end_of_screen:
        curr_char = wrapped_lines[curr_y][curr_x]
    return curr_x, curr_y, curr_char, end_of_screen


def get_down_char(curr_x, curr_y, curr_text):
    end_of_screen = False
    if curr_text == "":
        return -1, -1, "", end_of_screen
    wrapped_lines = curr_text.split("\n")
    curr_y += 1
    if curr_y >= len(wrapped_lines):
        curr_y = len(wrapped_lines) - 1
    else:
        end_of_screen = True
    curr_char = ""
    if not end_of_screen:
        curr_char = wrapped_lines[curr_y][curr_x]
    return curr_x, curr_y, curr_char, end_of_screen


def get_last_char_in_line(curr_y, curr_text):
    end_of_screen = False
    if curr_text == "":
        return -1, -1, ""
    wrapped_lines = curr_text.split("\n")
    curr_x = len(wrapped_lines[curr_y].rstrip()) - 1
    if curr_x < 0:
        curr_x = 0
    curr_char = wrapped_lines[curr_y][curr_x]
    return curr_x, curr_y, curr_char


def get_next_char(curr_x, curr_y, curr_text):
    line_break = False
    end_of_screen = False
    if curr_text == "":
        return -1, -1, "", end_of_screen, line_break
    wrapped_lines = curr_text.split("\n")
    x = curr_x
    y = curr_y
    if x + 1 == len(wrapped_lines[y]):
        if y + 1 <= len(wrapped_lines) - 1:
            y += 1
            x = 0
            line_break = True
        else:
            line_break = False
            end_of_screen = True
    else:
        x += 1
    curr_char = ""
    if not end_of_screen:
        curr_char = wrapped_lines[y][x]

    return x, y, curr_char, end_of_screen, line_break


def get_phonetic(curr_char):
    if len(curr_char) != 1:
        return curr_char
    phonetics_dict = {
        "A": "alpha",
        "B": "bravo",
        "C": "charlie",
        "D": "delta",
        "E": "echo",
        "F": "foxtrot",
        "G": "golf",
        "H": "hotel",
        "I": "india",
        "J": "juliet",
        "K": "kilo",
        "L": "lima",
        "M": "mike",
        "N": "november",
        "O": "oscar",
        "P": "papa",
        "Q": "quebec",
        "R": "romeo",
        "S": "sierra",
        "T": "tango",
        "U": "uniform",
        "V": "victor",
        "W": "whisky",
        "X": "x ray",
        "Y": "yankee",
        "Z": "zulu",
    }
    try:
        phon_char = phonetics_dict[curr_char.upper()]
        if curr_char.isupper():
            phon_char = phon_char[0].upper() + phon_char[1:]
        return phon_char
    except Exception as e:
        # Utility function, no env access - return fallback
        return curr_char


def present_char_for_review(
    env, char, interrupt=True, announce_capital=True, flush=False
):
    """Present a character for explicit review commands only"""
    if char == " ":
        if " " in env["punctuation"]["PUNCTDICT"]:
            announce_char = env["punctuation"]["PUNCTDICT"][" "]
        else:
            announce_char = "space"
        env["runtime"]["OutputManager"].present_text(
            announce_char, interrupt=interrupt, flush=flush
        )
    else:
        env["runtime"]["OutputManager"].present_text(
            char,
            interrupt=interrupt,
            ignore_punctuation=True,
            announce_capital=announce_capital,
            flush=flush,
        )
