//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2009-2012 Mikhail Fedotov <anyremote@mail.ru>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <unistd.h>

#include "common.h"
#include "lib_wrapper.h"
#include "conf.h"
#include "dispatcher.h"
#include "executor.h"
#include "utils.h"
#include "queue.h"

//
// Frontend support (ganyremote/kanyremote)
//


extern char tmp[MAXMAXLEN];

extern boolean_t stillRun;

static int isFirst = 1;

char frontEndBuf[MAXLEN];

int  sockfd_fe = 0;

int connectFrontEnd(int portno)
{
    int flags;
    sprintf(tmp, "connect to frontend >%d<",portno );
    logger(L_DBG, tmp);

    struct sockaddr_in serv_addr;
    struct hostent     *server;

    if (portno <= 0) {
        portno = 5050;
    }
    sockfd_fe = socket(AF_INET, SOCK_STREAM|SOCK_CLOEXEC, 0);
    if (sockfd_fe < 0) {
        logger(L_ERR, "opening socket for frontend");
        return EXIT_NOK;
    }

    if (-1 == (flags = fcntl(sockfd_fe, F_GETFL, 0))) {
        flags = 0;
    }
    fcntl(sockfd_fe, F_SETFL, flags | O_NONBLOCK);

    server = gethostbyname("localhost");
    if (server == NULL) {
        return EXIT_NOK;
    }

    memset((void *) &serv_addr, 0, sizeof(serv_addr));
    serv_addr.sin_family = AF_INET;
    bcopy((char *)server->h_addr, (char *)&serv_addr.sin_addr.s_addr, server->h_length);
    serv_addr.sin_port = htons(portno);

    int ret = connect(sockfd_fe,(const struct sockaddr *) &serv_addr, sizeof(serv_addr));
    if (ret != 0 && errno != EINPROGRESS) {
        sprintf(tmp, "connect socket for frontend (%d) errno = %d", ret, errno);
        logger(L_ERR, tmp);
        return EXIT_NOK;
    }

    return EXIT_OK;
}

void disconnectFrontEnd()
{
    if (sockfd_fe) {
        close(sockfd_fe);
        sockfd_fe = 0;
    }
    return;
}

static void writeToFrontEnd(const char *buf)
{
    int n = write(sockfd_fe, buf,strlen(buf));
    if (n < 0) {
        logger(L_ERR, "ERROR writing to frontend");
    }
    return;
}

void freeFMessage(void *buf)
{
    free(buf);
}

void sendToFrontEnd(const char *buf)
{
    if (queueExists(Q_FE) != RC_OK || sockfd_fe <= 0 || buf == NULL) {
        return;
    }

    sprintf(tmp, "send to frontend >%s<", buf);
    logger(L_DBG, tmp);

    queuePush(Q_FE, strdup(buf));

    return;
}

int readFromFrontEnd(char *buf, int size)
{
    //logger(L_INF, "readFromFrontEnd");
    if (buf == NULL) {
        return EXIT_OK;
    }

    bzero(buf,size);
    int n = read(sockfd_fe,buf,size-1);
    if (n < 0) {
        //logger(L_ERR, "ERROR reading from socket");
        return EXIT_NOK;
    } else {
        *(buf+n) = '\0';
        if (n > 0) {
            sprintf(tmp, "Got from frontend (%d) >%s<", n, buf);
            logger(L_DBG, tmp);

            int i = 0;
            int k = 0;
            for (i=0; i<n; i++) {
                if (*(buf+i) != '\0') {
                    if( i > k) {
                        *(buf+k) = *(buf+i);
                    }
                    k++;
                }
            }
            *(buf+k) = '\0';
            sprintf(tmp, "Got from frontend (%d) >%s<", n, buf);
            logger(L_DBG, tmp);
        }
    }

    return EXIT_OK;
}

boolean_t frontendRoutine(pointer_t data)
{
    if (sockfd_fe == 0) {	// try to connect
        int fePort = getFrontEnd();

        int ret = EXIT_NOK;
        if (fePort > 0 && queueExists(Q_FE) == RC_OK) {
            ret = connectFrontEnd(fePort);
        } else {
            if (isFirst) {
                isFirst = 0;
                logger(L_INF,"Do not start front end thread");
            }
            return BOOL_NO;
        }

        if  (ret == EXIT_OK) {
            logger(L_INF,"Start listening to frontend");
        } else {
            return BOOL_YES;
        }
    }

    // Verify commands from front-end
    int ret = readFromFrontEnd(frontEndBuf, MAXLEN);
    if (ret == EXIT_OK) {
        if (frontEndBuf[0] != '\0') {
            DEBUG2("readFromFrontEnd() >%s<", frontEndBuf);

            eMessage* em = (eMessage*) malloc(sizeof(eMessage));
            em->type  = EM_STRING;
            em->value = strdup(frontEndBuf);

            sendToExecutor(em);
        }
    }

    // Verify commands to front-end
    char *msgIn = (char*) queuePop(Q_FE);
    if (msgIn != NULL) {
        writeToFrontEnd(msgIn);
        free(msgIn);
    }
    return BOOL_YES;
}

