/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * HourFunction.java
 * ------------
 */
package org.pentaho.reporting.libraries.formula.function.datetime;

import java.math.BigDecimal;

import org.pentaho.reporting.libraries.formula.EvaluationException;
import org.pentaho.reporting.libraries.formula.FormulaContext;
import org.pentaho.reporting.libraries.formula.LibFormulaErrorValue;
import org.pentaho.reporting.libraries.formula.function.Function;
import org.pentaho.reporting.libraries.formula.function.ParameterCallback;
import org.pentaho.reporting.libraries.formula.lvalues.TypeValuePair;
import org.pentaho.reporting.libraries.formula.typing.TypeRegistry;
import org.pentaho.reporting.libraries.formula.typing.coretypes.NumberType;
import org.pentaho.reporting.libraries.formula.util.NumberUtil;

/**
 * This function extracts the hour (0 through 23) from a time.
 * 
 * @author Cedric Pronzato
 */
public class HourFunction implements Function
{
  private static final BigDecimal HOUR_24 = new BigDecimal(24);
  private static final long serialVersionUID = 1877256236005061937L;

  public HourFunction()
  {
  }

  public String getCanonicalName()
  {
    return "HOUR";
  }

  public TypeValuePair evaluate(final FormulaContext context,
      final ParameterCallback parameters) throws EvaluationException
  {
    if (parameters.getParameterCount() != 1)
    {
      throw new EvaluationException(LibFormulaErrorValue.ERROR_ARGUMENTS_VALUE);
    }

    final TypeRegistry typeRegistry = context.getTypeRegistry();
    final Number n = typeRegistry.convertToNumber(parameters.getType(0), parameters.getValue(0));

    if (n == null)
    {
      throw new EvaluationException(LibFormulaErrorValue.ERROR_INVALID_ARGUMENT_VALUE);
    }

    final BigDecimal bd = NumberUtil.getAsBigDecimal(n);
    final BigDecimal day = new BigDecimal(NumberUtil.performIntRounding(bd).intValue());
    final BigDecimal dayFraction = bd.subtract(day);

    final BigDecimal hourAndMinutesVal = dayFraction.multiply(HOUR_24);
    final BigDecimal hours = NumberUtil.performIntRounding(hourAndMinutesVal);
    return new TypeValuePair(NumberType.GENERIC_NUMBER, hours);
  }
}
