/*************************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * Copyright 2008 by Sun Microsystems, Inc.
 *
 * OpenOffice.org - a multi-platform office productivity suite
 *
 * $RCSfile: SlsHighlightObject.cxx,v $
 *
 * $Revision: 1.3 $
 *
 * This file is part of OpenOffice.org.
 *
 * OpenOffice.org is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * OpenOffice.org is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with OpenOffice.org.  If not, see
 * <http://www.openoffice.org/license.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/

#include "precompiled_sd.hxx"

#include "model/SlsPageDescriptor.hxx"
#include "view/SlsHighlightObject.hxx"
#include "view/SlsPageObjectViewObjectContact.hxx"
#include "controller/SlideSorterController.hxx"
#include "controller/SlsAnimator.hxx"
#include "controller/SlsProperties.hxx"
#include "controller/SlsCurrentSlideManager.hxx"
#include "Window.hxx"
#include <svx/sdr/contact/displayinfo.hxx>
#include <svx/svdpage.hxx>
#include <vcl/outdev.hxx>

namespace sd { namespace slidesorter { namespace view {

namespace {
    class HighlightScroller
    {
    public:
        HighlightScroller (
            HighlightObject& rObject,
            const ::boost::shared_ptr<sd::Window>& rpWindow,
            const Rectangle& rStart,
            const Rectangle& rEnd);
        void operator() (const double nValue);
    private:
        HighlightObject& mrObject;
        Rectangle maStart;
        Rectangle maEnd;
        ::boost::shared_ptr<sd::Window> mpWindow;
    };
}




//===== HighlightObject =======================================================

HighlightObject::HighlightObject (SlideSorter& rSlideSorter)
    : mpDescriptor(),
      mrSlideSorter(rSlideSorter),
      maBoundingBox(),
      mnOpacity(0.5)
{
    SetSlide(mrSlideSorter.GetController().GetCurrentSlideManager()->GetCurrentSlide());
}




HighlightObject::~HighlightObject (void)
{
}




sdr::contact::ViewContact* HighlightObject::CreateObjectSpecificViewContact (void)
{
    return new HighlightViewContact(*this);
}




void HighlightObject::SetSlide (const model::SharedPageDescriptor& rpDescriptor)
{
    model::SharedPageDescriptor pOldDescriptor = mpDescriptor;
    mpDescriptor = rpDescriptor;

    ::boost::shared_ptr<sd::Window> pDevice = mrSlideSorter.GetContentWindow();
    if (pDevice.get() == NULL)
        return;

    if ( ! mrSlideSorter.GetController().GetProperties()->IsHighlightCurrentSlide())
        return;
    
    if (mpDescriptor.get() == NULL && pOldDescriptor.get() == NULL)
        return;

    view::PageObjectViewObjectContact* pOldVOC
        = (pOldDescriptor.get() == NULL) ? NULL : pOldDescriptor->GetViewObjectContact();
    view::PageObjectViewObjectContact* pNewVOC
        = (mpDescriptor.get() == NULL) ? NULL : mpDescriptor->GetViewObjectContact();

    Rectangle aStart;
    if (pOldVOC != NULL)
        aStart = pDevice->PixelToLogic(pOldVOC->GetBoundingBox(*pDevice,
                view::PageObjectViewObjectContact::PageObjectBoundingBox,
                view::PageObjectViewObjectContact::PixelCoordinateSystem));
    Rectangle aEnd;
    if (pNewVOC != NULL)
        aEnd = pDevice->PixelToLogic(pNewVOC->GetBoundingBox(*pDevice,
                view::PageObjectViewObjectContact::PageObjectBoundingBox,
                view::PageObjectViewObjectContact::PixelCoordinateSystem));
    if ( ! aStart.IsEmpty() || ! aEnd.IsEmpty())
    {
        mrSlideSorter.GetController().GetAnimator()->AddAnimation(
            HighlightScroller(
                *this,
                pDevice,
                aStart,
                aEnd),
            1000);
    }
}




model::SharedPageDescriptor HighlightObject::GetSlide (void) const
{
    return mpDescriptor;
}




void HighlightObject::SetOpacity (const double nOpacity)
{
    mnOpacity = nOpacity;
}




double HighlightObject::GetOpacity (void) const
{
    return mnOpacity;
}




void HighlightObject::SetBoundingBox (const Rectangle& rBox)
{
    ::boost::shared_ptr<sd::Window> pDevice = mrSlideSorter.GetContentWindow();
    if (pDevice.get() != NULL)
        pDevice->Invalidate(maBoundingBox);
    
    maBoundingBox = rBox;

    if (pDevice.get() != NULL)
        pDevice->Invalidate(maBoundingBox);

    SetChanged();
}




const Rectangle& HighlightObject::GetCurrentBoundRect (void) const
{
    return maBoundingBox;
}



void HighlightObject::UpdatePosition (void)
{
    if (mpDescriptor.get() != NULL)
    {
        if (maBoundingBox.IsEmpty())
            SetSlide(mpDescriptor);
    }
}




//===== HighlightViewContact ==================================================

HighlightViewContact::HighlightViewContact (HighlightObject& rHighlightObject)
    : ViewContactOfSdrObj(rHighlightObject),
      mrHighlightObject(rHighlightObject)
{
}




HighlightViewContact::~HighlightViewContact (void)
{
}




sdr::contact::ViewObjectContact& 
    HighlightViewContact::CreateObjectSpecificViewObjectContact(
        sdr::contact::ObjectContact& rObjectContact)
{
    return *new HighlightViewObjectContact(rObjectContact, *this, &mrHighlightObject);
}




void HighlightViewContact::CalcPaintRectangle (void)
{
    maPaintRectangle = mrHighlightObject.GetCurrentBoundRect();
}




void HighlightViewContact::ActionChanged (void)
{
    sdr::contact::ViewContact::ActionChanged();
}




void HighlightViewContact::PrepareDelete (void)
{
    sdr::contact::ViewContact::PrepareDelete();
}





//===== HighlightViewObjectContact ============================================

HighlightViewObjectContact::HighlightViewObjectContact (
    sdr::contact::ObjectContact& rObjectContact, 
    sdr::contact::ViewContact& rViewContact,
    HighlightObject* pHighlightObject)
    : sdr::contact::ViewObjectContact (rObjectContact, rViewContact),
      mpHighlightObject(pHighlightObject)
{
}





HighlightViewObjectContact::~HighlightViewObjectContact (void)
{
}




void HighlightViewObjectContact::PaintObject (sdr::contact::DisplayInfo& rDisplayInfo)
{
    OutputDevice* pDevice = rDisplayInfo.GetOutputDevice();
    if (pDevice == NULL)
        return;

    if (mpHighlightObject == NULL)
        return;

    OSL_TRACE("ordnum of highlight object is %d", mpHighlightObject->GetOrdNum());

    model::SharedPageDescriptor pDescriptor = mpHighlightObject->GetSlide();
    if (pDescriptor.get() == NULL)
        return;
    
    view::PageObjectViewObjectContact* pVOC = pDescriptor->GetViewObjectContact();
    if (pVOC == NULL)
        return;
    
    Rectangle aOuterBox (pDevice->LogicToPixel(mpHighlightObject->GetCurrentBoundRect()));

    const Color aOldFillColor (pDevice->GetFillColor());
    const Color aOldLineColor (pDevice->GetLineColor());
    const sal_Bool bWasEnabled (pDevice->IsMapModeEnabled());

    pDevice->SetLineColor();
    pDevice->SetFillColor(
        pVOC->GetColor(
            *pDevice,
            view::PageObjectViewObjectContact::CS_BACKGROUND,
            mpHighlightObject->GetOpacity()));
    pDevice->EnableMapMode(sal_False);
 
    // Paint the background.
    pDevice->DrawRect(aOuterBox);
    
    // Draw the frame around the background.
    pDevice->SetLineColor(
        pVOC->GetColor(*pDevice, view::PageObjectViewObjectContact::CS_SELECTION));
    pDevice->SetFillColor();
    pDevice->DrawRect(aOuterBox);
 
    // Erase the corner pixel to have somewhat rounded corners.
    const Color aCornerColor (
        pVOC->GetColor(*pDevice, view::PageObjectViewObjectContact::CS_WINDOW));
    Point aCorner (aOuterBox.TopLeft());
    pDevice->DrawPixel (aOuterBox.TopLeft(), aCornerColor);
    aCorner = aOuterBox.TopRight();
    pDevice->DrawPixel (aCorner, aCornerColor);
    aCorner = aOuterBox.BottomLeft();
    pDevice->DrawPixel (aCorner, aCornerColor);
    aCorner = aOuterBox.BottomRight();
    pDevice->DrawPixel (aCorner, aCornerColor);

    // Restore old output device state.
    pDevice->SetFillColor(aOldFillColor);
    pDevice->SetLineColor(aOldLineColor);
    pDevice->EnableMapMode(bWasEnabled);
}




void HighlightViewObjectContact::PrepareDelete (void)
{
    sdr::contact::ViewObjectContact::PrepareDelete();
}




//===== HighlightScroller =====================================================

namespace {

HighlightScroller::HighlightScroller (
    HighlightObject& rObject,
    const ::boost::shared_ptr<sd::Window>& rpWindow,
    const Rectangle& rStart,
    const Rectangle& rEnd)
    : mrObject(rObject),
      maStart(rStart),
      maEnd(rEnd),
      mpWindow(rpWindow)
{
}




void HighlightScroller::operator() (const double nValue)
{
    if (mpWindow.get() == NULL)
        return;

    Rectangle aBox;
    if (!maStart.IsEmpty() && !maEnd.IsEmpty())
    {
        // Move object from one place to another.
        aBox.Left() = sal_Int32(maStart.Left()*(1-nValue) + maEnd.Left()*nValue + 0.5);
        aBox.Top() = sal_Int32(maStart.Top()*(1-nValue) + maEnd.Top()*nValue + 0.5);
        aBox.Right() = sal_Int32(maStart.Right()*(1-nValue) + maEnd.Right()*nValue + 0.5);
        aBox.Bottom() = sal_Int32(maStart.Bottom()*(1-nValue) + maEnd.Bottom()*nValue + 0.5);
    }
    else if ( ! maStart.IsEmpty())
    {
        // Fade out object.
        aBox = maStart;
        mrObject.SetOpacity((1.0-nValue) / 2.0);
    }
    else if ( ! maEnd.IsEmpty())
    {
        // Fade in object.
        aBox = maEnd;
        mrObject.SetOpacity(nValue / 2.0);
    }

    mrObject.SetBoundingBox(aBox);
}

}

} } } // end of namespace ::sd::slidesorter::view
