/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: slide.cxx,v $
 *
 *  $Revision: 1.17 $
 *
 *  last change: $Author: kz $ $Date: 2006/12/13 16:57:04 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_slideshow.hxx"

#include <canvas/debug.hxx>
#include <canvas/canvastools.hxx>
#include <cppcanvas/basegfxfactory.hxx>

#include <basegfx/matrix/b2dhommatrix.hxx>
#include <basegfx/point/b2dpoint.hxx>
#include <basegfx/polygon/b2dpolygon.hxx>
#include <basegfx/polygon/b2dpolygontools.hxx>
#include <basegfx/numeric/ftools.hxx>

#include <com/sun/star/awt/SystemPointer.hpp>
#include <com/sun/star/container/XIndexAccess.hpp>
#include <com/sun/star/drawing/XMasterPageTarget.hpp>
#include <com/sun/star/beans/XPropertySet.hpp>
#include <com/sun/star/container/XEnumerationAccess.hpp>
#include <com/sun/star/awt/Rectangle.hpp>
#include <com/sun/star/presentation/ParagraphTarget.hpp>
#include <com/sun/star/presentation/EffectNodeType.hpp>
#include <com/sun/star/animations/XAnimationNodeSupplier.hpp>
#include <com/sun/star/animations/XTargetPropertiesCreator.hpp>
#include <com/sun/star/drawing/TextAnimationKind.hpp>

#include <animations/animationnodehelper.hxx>

#include <cppuhelper/exc_hlp.hxx>
#include <comphelper/anytostring.hxx>

#include "slide.hxx"
#include "intrinsicanimation.hxx"
#include "doctreenode.hxx"
#include "backgroundshape.hxx"
#include "shapeimporter.hxx"
#include "slideshowexceptions.hxx"
#include "eventqueue.hxx"
#include "activitiesqueue.hxx"
#include "usereventqueue.hxx"
#include "event.hxx"
#include "tools.hxx"
#include "drawshape.hxx"

#include <boost/bind.hpp>
#include <iterator>
#include <algorithm>
#include <functional>


using namespace ::com::sun::star;

// -----------------------------------------------------------------------------

namespace slideshow
{
    namespace internal
    {
        SlideSharedPtr Slide::create( const uno::Reference< drawing::XDrawPage >&			xDrawPage,
                                      const uno::Reference< animations::XAnimationNode >& 	xRootNode,
                                      EventQueue&											rEventQueue,
                                      ActivitiesQueue&										rActivitiesQueue,
                                      EventMultiplexer&										rEventMultiplexer,
                                      UserEventQueue&										rUserEventQueue,
                                      const uno::Reference< uno::XComponentContext >& 		xComponentContext,
                                      const UnoViewContainer&                               rViewContainer)
        {
            SlideSharedPtr pRet( new Slide( xDrawPage, xRootNode, rEventQueue, 
                                            rActivitiesQueue, rEventMultiplexer, 
                                            rUserEventQueue, xComponentContext, 
                                            rViewContainer ));

            rEventMultiplexer.addViewHandler( pRet );

            return pRet;
        }

        Slide::Slide( const uno::Reference< drawing::XDrawPage >&			xDrawPage,
					  const uno::Reference< animations::XAnimationNode >& 	xRootNode,
                      EventQueue&											rEventQueue,
                      ActivitiesQueue&										rActivitiesQueue,
                      EventMultiplexer&										rEventMultiplexer,
                      UserEventQueue&										rUserEventQueue,
                      const uno::Reference< uno::XComponentContext >& 		xComponentContext,
                      const UnoViewContainer&                               rViewContainer) :
            mxDrawPage( xDrawPage ),
			mxRootNode( xRootNode ),
            maContext( LayerManagerSharedPtr(
                           new LayerManager( getSlideRect() ) ),
                       rEventQueue, 
                       rEventMultiplexer,
                       rActivitiesQueue, 
                       rUserEventQueue,
                       rViewContainer,
                       xComponentContext ),
            mpEventBroadcaster( new ShapeEventBroadcaster( rEventMultiplexer ) ),
            maAnimations( maContext ),
            maUserPaintColor(),
            mpPaintOverlay(),
            maSlideBitmaps(),
            mnNextSlideTimeout( 1.0 ),
            meAnimationState( CONSTRUCTING_STATE ),
            mbImageAnimationsAllowed( true ),
            mbShapesLoaded( false ),
            mbShowLoaded( false ),
            mbHaveAnimations( false ),
            mbMainSequenceFound( false ),
            mbHasAutomaticNextSlide( false )
        {
            ENSURE_AND_THROW( maContext.mpLayerManager,
                              "Slide::Slide(): Could not create layer manager" );
            ENSURE_AND_THROW( mpEventBroadcaster,
                              "Slide::Slide(): Could not create event broadcaster" );

            // retrieve slide change parameters from XDrawPage
            // ===============================================

            uno::Reference< beans::XPropertySet > xPropSet( xDrawPage,
                                                            uno::UNO_QUERY );
            
            sal_Int32 nChange(0);
            if( !xPropSet.is() ||
                !getPropertyValue( nChange,
                                   xPropSet,
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("Change"))) )
            {
                OSL_TRACE(
                    "Slide::Slide(): "
                    "Could not extract slide change mode from XDrawPage - assuming <none>\n" );
            }
            
            mbHasAutomaticNextSlide = nChange == 1;
            
            if( !xPropSet.is() ||
                !getPropertyValue( mnNextSlideTimeout,
                                   xPropSet,
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("Duration"))) )
            {
                OSL_TRACE(
                    "Slide::Slide(): "
                    "Could not extract slide transition timeout from "
                    "XDrawPage - assuming 1 sec\n" );
            }

            // init slide bitmap slots (one per view)
            std::for_each( rViewContainer.begin(),
                           rViewContainer.end(),
                           boost::bind( &Slide::viewAdded,
                                        this,
                                        _1 ));
        }

        Slide::~Slide()
        {
            try
            {
                mpEventBroadcaster->dispose();
            }
            catch (uno::Exception &) {
                OSL_ENSURE( false, rtl::OUStringToOString(
                                comphelper::anyToString(
                                    cppu::getCaughtException() ),
                                RTL_TEXTENCODING_UTF8 ).getStr() );
            }
        }
        
        void Slide::viewAdded( const UnoViewSharedPtr& rView )
        {
            maSlideBitmaps.push_back( 
                std::make_pair( rView, 
                                VectorOfSlideBitmaps(SlideAnimationState_NUM_ENTRIES) ));

            // xxx TODO: let LayerManager register itself at EventMultiplexer
            if( maContext.mpLayerManager )
                maContext.mpLayerManager->addView( rView );
        }

        void Slide::viewRemoved( const UnoViewSharedPtr& rView )
        {
            // xxx TODO: let LayerManager register itself at EventMultiplexer
            if( maContext.mpLayerManager )
                maContext.mpLayerManager->removeView( rView );

            const VectorOfVectorOfSlideBitmaps::iterator aEnd( maSlideBitmaps.end() );
            maSlideBitmaps.erase(
                std::remove_if( maSlideBitmaps.begin(),
                                aEnd,
                                boost::bind(
                                    std::equal_to<UnoViewSharedPtr>(),
                                    rView,
                                    // select view:    
                                    boost::bind( 
                                        std::select1st<VectorOfVectorOfSlideBitmaps::value_type>(),
                                        _1 ))), 
                aEnd );
        }

        void Slide::viewChanged( const UnoViewSharedPtr& rView )
        {
            if( maContext.mpLayerManager )
                maContext.mpLayerManager->viewChanged( rView );
        }

        void Slide::addShapeEventListener( const uno::Reference< presentation::XShapeEventListener >& xListener, 
                                           const uno::Reference< drawing::XShape >& 				  xShape )
        {
            if( !prefetchShapes() )
                ENSURE_AND_THROW( false,
                                  "Slide::addShapeEventListener(): Could not prefetch shapes" );

            ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

            if( pShape )
                mpEventBroadcaster->addShapeEventListener( xListener, pShape );
        }

        void Slide::removeShapeEventListener( const uno::Reference< presentation::XShapeEventListener >& 	xListener, 
                                              const uno::Reference< drawing::XShape >&                      xShape )
        {
            if( !prefetchShapes() )
                ENSURE_AND_THROW( false,
                                  "Slide::removeShapeEventListener(): Could not prefetch shapes" );

            ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

            if( pShape )
                mpEventBroadcaster->removeShapeEventListener( xListener, pShape );
        }

        void Slide::setImageAnimationsAllowed( bool bImageAnimationsAllowed )
        {
            if( mbImageAnimationsAllowed != bImageAnimationsAllowed )
            {
                mbImageAnimationsAllowed = bImageAnimationsAllowed;

                // if animations switched off - immediately disable
                // them. If switched on - immediately enable them
                if( bImageAnimationsAllowed == false )
                    endIntrinsicAnimations();
                else
                    startIntrinsicAnimations();
            }
        }

        void Slide::setShapeCursor( const uno::Reference< drawing::XShape >&	xShape, 
                                    sal_Int16 									nPointerShape )
        {
            if( !prefetchShapes() )
                ENSURE_AND_THROW( false,
                                  "Slide::Slide::setShapeCursor(): Could not prefetch shapes" );

            ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

            if( pShape )
                mpEventBroadcaster->setShapeCursor( pShape, nPointerShape );
        }

        bool Slide::paint( const UnoViewSharedPtr& /*rView*/ )
        {
            ENSURE_AND_RETURN( maContext.mpLayerManager, 
                               "Slide::paint(): Invalid layer manager" );

            if( !prefetchShapes() )
                return false;

            // TODO(P3): don't update all view, only the modified one!
            if( !maContext.mpLayerManager->render() )
                return false;

            return true;
        }

        bool Slide::prefetchShow()
        {
            return applyInitialShapeAttributes( mxRootNode );
        }

        namespace
        {
            class SlideRenderer
            {
            public:
                SlideRenderer( Slide& rSlide ) :
                    mrSlide( rSlide )
                {
                }

                void operator()( const UnoViewSharedPtr& rView )
                {
                    SlideBitmapSharedPtr 		 pBitmap( mrSlide.getCurrentSlideBitmap( rView ) );
                    ::cppcanvas::CanvasSharedPtr pCanvas( rView->getCanvas() );

                    const ::basegfx::B2DHomMatrix 	aViewTransform( pCanvas->getTransformation() );
                    const ::basegfx::B2DPoint 		aOutPosPixel( aViewTransform * ::basegfx::B2DPoint() );

                    // setup a canvas with device coordinate space,
                    // the slide bitmap already has the correct
                    // dimension.
                    ::cppcanvas::CanvasSharedPtr pDevicePixelCanvas( pCanvas->clone() );
                    pDevicePixelCanvas->setTransformation( ::basegfx::B2DHomMatrix() );

                    // clear background before (to generate a clean
                    // slate, e.g. after resizes)
                    rView->clear();

                    // render at given output position
                    pBitmap->move( aOutPosPixel );
                    pBitmap->clip( ::basegfx::B2DPolyPolygon() ); // clear
                                                                  // clip
                                                                  // (might
                                                                  // have
                                                                  // been
                                                                  // changed,
                                                                  // e.g. from
                                                                  // comb
                                                                  // transition)
                    pBitmap->draw( pDevicePixelCanvas );
                }

            private:
                Slide& mrSlide;
            };
        }

        bool Slide::show()
        {
            ENSURE_AND_RETURN( maContext.mpLayerManager, 
                               "Slide::show(): Invalid layer manager" );

            if( maContext.mrViewContainer.empty() )
                return false; // no views

            // set initial shape attributes (e.g. hide 'appear' effect
            // shapes)
            if( !applyInitialShapeAttributes( mxRootNode ) )
                return false;

            // tell the ActivitiesQueue to send us repaints. This call
            // overwrites any previously set layer managers, especially
            // those set by previous slides.
            maContext.mrEventMultiplexer.setLayerManager( 
                maContext.mpLayerManager );

            const bool bIsAnimated( isAnimated() );

            if( bIsAnimated )
            {
                // feed events into queues.
                maAnimations.start();
            }

            // NOTE: this looks slightly weird, but is indeed correct:
            // as isAnimated() might return false, _although_ there is
            // a main sequence (because the animation nodes don't
            // contain any executable effects), we gotta check both
            // conditions here.
            if( !bIsAnimated || !mbMainSequenceFound )
            {
                // manually trigger a slide animation end event (we
                // don't have animations at all, or we don't have a
                // main animation sequence, but if we had, it'd end
                // now). Note that having animations alone does not
                // matter here, as only main sequence animations
                // prevents showing the next slide on nextEvent().
                maContext.mrEventMultiplexer.notifySlideAnimationsEnd();
            }

            // render slide to screen (might not have been done, if
            // e.g. no slide transition took place; or slide
            // transition left ugly remnants). Don't render from first
            // principles, but try to use buffered bitmap
            ::std::for_each( maContext.mrViewContainer.begin(),
                             maContext.mrViewContainer.end(),
                             SlideRenderer( *this ) );
            maContext.mpLayerManager->clearPendingUpdates();
            maContext.mrEventMultiplexer.updateScreenContent( true );

            // enable shape-intrinsic animations (drawing layer
            // animations or GIF animations)
            if( mbImageAnimationsAllowed )
                startIntrinsicAnimations();

            // TODO(F3): Enable/disable mpEventBroadcaster,
            // too. Otherwise, previous and next slides might produce
            // ghost events and ghost shape cursor changes.

            // enable paint overlay, if maUserPaintColor is valid
            enablePaintOverlay();

            // enable event broadcasting for shapes of this slide:
            mpEventBroadcaster->enable();
            
            // from now on, animations might be showing
            meAnimationState = SHOWING_STATE;

            return true;
        }

        void Slide::end()
        {
            // from now on, all animations are stopped
            meAnimationState = FINAL_STATE;

            // disable event broadcasting for shapes of this slide:
            mpEventBroadcaster->disable();
            
            // disable user paint overlay under all circumstances,
            // this slide now ceases to be active.
            disablePaintOverlay();

            // remove layer manager from the ActivitiesQueue, this
            // slide has ended its active life. For the duration of
            // the slide transition, the Presentation class takes over
            // control of the View canvas (note that in theory, the
            // layer manager can remain active, since no shape on the
            // slide is supposed to change. Unfortunately,
            // maAnimations.end() below might actually remove various
            // held effects).
            maContext.mrEventMultiplexer.setLayerManager( LayerManagerSharedPtr() );

			// simply forward to animations
            maAnimations.end();

            // switch off all shape-intrinsic animations. Do that
            // unconditionally (i.e. without checking
            // mbImageAnimationsAllowed), since somebody might have
            // disabled image animations _after_ the slide was
            // started.
            endIntrinsicAnimations();
        }

        bool Slide::isShowing() const
        {
            return meAnimationState == SHOWING_STATE;
        }

        bool Slide::isAnimated()
        {
            // prefetch, but don't apply initial shape attributes
            if( !implPrefetchShow() )
                return false;

            return mbHaveAnimations && maAnimations.isAnimated();
        }

        namespace
        {
            /** Little wrapper around a BitmapCanvas, to render into a slide bitmap
             */
            class BitmapView : public View
            {
            public:
                BitmapView( const ::cppcanvas::BitmapCanvasSharedPtr& 	rCanvas,
                            const ::basegfx::B2ISize&					rSize ) :
                    mpCanvas( rCanvas ),
                    maSize( rSize )
                {
                }

                virtual ViewLayerSharedPtr createViewLayer() const
                {
                    // TODO(Q2): Put that into a separate class
                    return ViewLayerSharedPtr( new BitmapView( mpCanvas,
                                                               maSize ) );
                }

                virtual bool updateScreen() const
                {
                    return true; // NOOP on BitmapCanvas
                }

                virtual void clear() const
                {
                    initSlideBackground( mpCanvas,
                                         maSize );
                }

                virtual bool isContentDestroyed() const
                {
                    return false;
                }

                virtual ::basegfx::B2DHomMatrix getTransformation() const
                {
                    OSL_ENSURE( false, "BitmapView::createSprite(): This method is not supposed to be called!" );
                    return ::basegfx::B2DHomMatrix();
                }

                virtual void setViewSize( const ::basegfx::B2DSize& /*rSize*/ )
                {
                    // noop
                    OSL_ENSURE( false, "BitmapView::setViewSize(): This method is not supposed to be called!" );
                }

                virtual void setClip( const ::basegfx::B2DPolyPolygon& /*rClip*/ )
                {
                    // TODO(F1): Implement clipping
                }

                virtual void setMouseCursor( sal_Int16 /*nPointerShape*/ )
                {
                    // noop, ignore here
                }

                virtual ::cppcanvas::CustomSpriteSharedPtr createSprite( const ::basegfx::B2DSize& /*rSpriteSizePixel*/ ) const          
                {
                    OSL_ENSURE( false, "BitmapView::createSprite(): This method is not supposed to be called!" );
                    return ::cppcanvas::CustomSpriteSharedPtr();
                }

                virtual double getSpritePriority( double nSpritePrio ) const
                {
                    return nSpritePrio;
                }

                virtual ::cppcanvas::CanvasSharedPtr getCanvas() const
                {
                    return mpCanvas;
                }

                virtual void setPriority( double /*nPrio*/ )
                {
                    // noop, ignore here
                    OSL_TRACE( "BitmapView::setPriority(): Called, but ignored\n" );
                }

            private:
                ::cppcanvas::BitmapCanvasSharedPtr	mpCanvas;
                const ::basegfx::B2ISize			maSize;
            };
        }

        SlideBitmapSharedPtr Slide::getCurrentSlideBitmap( const UnoViewSharedPtr& rView )
        {
            // search corresponding entry in maSlideBitmaps (which
            // contains the views as the key)
            VectorOfVectorOfSlideBitmaps::iterator       aIter;
            const VectorOfVectorOfSlideBitmaps::iterator aEnd( maSlideBitmaps.end() );
            if( (aIter=std::find_if( maSlideBitmaps.begin(),
                                     aEnd,
                                     boost::bind(
                                         std::equal_to<UnoViewSharedPtr>(),
                                         rView,
                                         // select view:    
                                         boost::bind( 
                                             std::select1st<VectorOfVectorOfSlideBitmaps::value_type>(),
                                             _1 )))) == aEnd )
            {
                // corresponding view not found - maybe view was not
                // added to Slide?
                ENSURE_AND_THROW( false,
                                  "Slide::getInitialSlideBitmap(): view does not "
                                  "match any of the added ones" );
            }

            // ensure that the show is loaded
            if( !mbShowLoaded )
            {
                // only prefetch and init shapes when not done already
                // (otherwise, at least applyInitialShapeAttributes()
                // will be called twice for initial slide
                // rendering). Furthermore,
                // applyInitialShapeAttributes() _always_ performs
                // initializations, which would be highly unwanted
                // during a running show. OTOH, a slide whose
                // mbShowLoaded is false is guaranteed not be running
                // a show.

                // set initial shape attributes (e.g. hide 'appear' effect
                // shapes)
                if( !applyInitialShapeAttributes( mxRootNode ) )
                    ENSURE_AND_THROW(false,"Slide::getCurrentSlideBitmap(): Cannot apply initial attributes");
            }

            SlideBitmapSharedPtr&     rBitmap( aIter->second.at( meAnimationState ));
			const ::basegfx::B2ISize& rSlideSize( getSlideSizePixel( rView ));

			// is the bitmap valid (actually existent, and of correct
			// size)?
            if( !rBitmap || rBitmap->getSize() != rSlideSize )
            {
                // no bitmap there yet, or wrong size - create one
                rBitmap = createCurrentSlideBitmap(rView, rSlideSize);
            }

            return rBitmap;
        }

        bool Slide::hasAutomaticNextSlide() const
        {
            return mbHasAutomaticNextSlide;
        }

        double Slide::getAutomaticNextSlideTimeout() const
        {
            return mnNextSlideTimeout;
        }

        void Slide::setUserPaintColor( boost::optional<RGBColor> const& rColor )
        {
            maUserPaintColor = rColor;

            if( isShowing() && maUserPaintColor )
            {
                enablePaintOverlay();
            }
            else
            {
                disablePaintOverlay();
            }
        }

        // private methods
        //--------------------------------------------------------------------------------------------------------------


        ::basegfx::B2ISize Slide::getSlideSizePixel(
            const UnoViewSharedPtr& rView ) const
        {
            const ::basegfx::B2ISize slideSize( getSlideSize() );

            ::cppcanvas::CanvasSharedPtr pCanvas( rView->getCanvas() );

            // determine transformed page bounds
            const ::basegfx::B2DRectangle aRect( 0,0,
                                                 slideSize.getX(),
                                                 slideSize.getY() );
            ::basegfx::B2DRectangle aTmpRect;
            ::canvas::tools::calcTransformedRectBounds( aTmpRect, aRect, pCanvas->getTransformation() );

            // #i42440# Returned slide size is one pixel too small, as
            // rendering happens one pixel to the right and below the
            // actual bound rect.
            return ::basegfx::B2ISize( ::basegfx::fround( aTmpRect.getRange().getX() ) + 1,
									   ::basegfx::fround( aTmpRect.getRange().getY() ) + 1 );
		}

        SlideBitmapSharedPtr Slide::createCurrentSlideBitmap(
            const UnoViewSharedPtr& rView, const ::basegfx::B2ISize& rBmpSize )
        {
            ENSURE_AND_THROW( rView && rView->getCanvas(), 
                              "Slide::createCurrentSlideBitmap(): Invalid view" );
            ENSURE_AND_THROW( maContext.mpLayerManager, 
                              "Slide::createCurrentSlideBitmap(): Invalid layer manager" );
            ENSURE_AND_THROW( mbShowLoaded, 
                              "Slide::createCurrentSlideBitmap(): No show loaded" );

            ::cppcanvas::CanvasSharedPtr pCanvas( rView->getCanvas() );

            // create a bitmap of appropriate size
            ::cppcanvas::BitmapSharedPtr pBitmap( 
                ::cppcanvas::BaseGfxFactory::getInstance().createBitmap( 
                    pCanvas, 
                    rBmpSize ) );

            ENSURE_AND_THROW( pBitmap,
                              "Slide::createCurrentSlideBitmap(): Cannot create page bitmap" );

            ::cppcanvas::BitmapCanvasSharedPtr pBitmapCanvas( pBitmap->getBitmapCanvas() );

            ENSURE_AND_THROW( pBitmapCanvas,
                              "Slide::createCurrentSlideBitmap(): Cannot create page bitmap canvas" );

            // apply linear part of destination canvas transformation (linear means in this context: 
            // transformation without any translational components)
            ::basegfx::B2DHomMatrix aLinearTransform( pCanvas->getTransformation() );
            aLinearTransform.set( 0, 2, 0.0 );
            aLinearTransform.set( 1, 2, 0.0 );
            pBitmapCanvas->setTransformation( aLinearTransform );

            ViewSharedPtr pView( new BitmapView( pBitmapCanvas,
                                                 rBmpSize ) );
            maContext.mpLayerManager->addView( pView ); // addView implicitely draws everything
            maContext.mpLayerManager->removeView( pView );

            return SlideBitmapSharedPtr( new SlideBitmap( pBitmap ) );
        }

        namespace
        {
            class MainSequenceSearcher
            {
            public:
                MainSequenceSearcher()
                {
                    maSearchKey.Name = ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "node-type" ) );
                    maSearchKey.Value <<= presentation::EffectNodeType::MAIN_SEQUENCE;
                }

                void operator()( const uno::Reference< animations::XAnimationNode >& xChildNode )
                {
                    uno::Sequence< beans::NamedValue > aUserData( xChildNode->getUserData() );
                    
                    if( findNamedValue( aUserData, maSearchKey ) )
                    {
                        maMainSequence = xChildNode;
                    }
                }

                uno::Reference< animations::XAnimationNode > getMainSequence() const
                {
                    return maMainSequence;
                }

            private:
                beans::NamedValue 								maSearchKey;
                uno::Reference< animations::XAnimationNode >	maMainSequence;
            };
        }

        bool Slide::implPrefetchShow()
        {
            if( mbShowLoaded )
                return true;

            ENSURE_AND_RETURN( mxDrawPage.is(), 
                               "Slide::implPrefetchShow(): Invalid draw page" );
            ENSURE_AND_RETURN( maContext.mpLayerManager, 
                               "Slide::implPrefetchShow(): Invalid layer manager" );

            // fetch desired page content
            // ==========================

            if( !prefetchShapes() )
                return false;

            // New animations framework: import the shape effect info
            // ======================================================

            try
            {
                if( mxRootNode.is() )
                {
                    if( !maAnimations.importAnimations( mxRootNode ) )
                    {
                        OSL_ENSURE( false,
                                    "Slide::implPrefetchShow(): have animation nodes, "
                                    "but import animations failed." );

                        // could not import animation framework,
                        // _although_ some animation nodes are there -
                        // this is an error (not finding animations at
                        // all is okay - might be a static slide)
                        return false; 
                    }

                    // now check whether we've got a main sequence (if
                    // not, we must manually call
                    // EventMultiplexer::notifySlideAnimationsEnd()
                    // above, as e.g. interactive sequences alone
                    // don't block nextEvent() from issuing the next
                    // slide)
                    MainSequenceSearcher aSearcher;
                    if( ::anim::for_each_childNode( mxRootNode, aSearcher ) )
                        mbMainSequenceFound = aSearcher.getMainSequence().is();

                    // import successfully done
                    mbHaveAnimations = true;
                }
            }
            catch( uno::Exception& )
            {
                OSL_ENSURE(
                    false,
                    rtl::OUStringToOString(
                        comphelper::anyToString(cppu::getCaughtException()),
                        RTL_TEXTENCODING_UTF8 ) );
                // TODO(E2): Error handling. For now, bail out
            }

            mbShowLoaded = true;

            return true;
        }

        void Slide::enablePaintOverlay()
        {
            if( maUserPaintColor )
                mpPaintOverlay = UserPaintOverlay::create( *maUserPaintColor,
                                                           2.0,
                                                           maContext );
        }

        void Slide::disablePaintOverlay()
        {
            mpPaintOverlay.reset();
        }

        basegfx::B2ISize Slide::getSlideSize() const
        {
            uno::Reference< beans::XPropertySet > xPropSet(
                mxDrawPage, uno::UNO_QUERY_THROW );
            
            sal_Int32 nDocWidth = 0;
            sal_Int32 nDocHeight = 0;
            xPropSet->getPropertyValue( ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM("Width") ) ) >>= nDocWidth;
            xPropSet->getPropertyValue( ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM("Height") ) ) >>= nDocHeight;

            return basegfx::B2ISize( nDocWidth, nDocHeight );
        }

        ::basegfx::B2DRectangle Slide::getSlideRect() const
        {
            const basegfx::B2ISize slideSize( getSlideSize() );
            return ::basegfx::B2DRectangle(0.0,0.0,
                                           slideSize.getX(),
                                           slideSize.getY());
        }

        namespace
        {
            /** Adapt ShapeSharedPtr to the startIntrinsicAnimation()
                method.
             */
            class StartIntrinsicAnimationAdaptor
            {
            public:
                StartIntrinsicAnimationAdaptor( const SlideShowContext& rContext ) :
                    maContext( rContext )
                {
                }

                void operator()( ShapeSharedPtr const& rShape ) const
                {
                    if( rShape &&
                        rShape->hasIntrinsicAnimation() )
                    {
                        IntrinsicAnimationSharedPtr pIntrinsicAnimation(
                            ::boost::dynamic_pointer_cast< IntrinsicAnimation >( rShape ) );
                        if( pIntrinsicAnimation)
                            pIntrinsicAnimation->startIntrinsicAnimation( maContext,
                                                                          rShape );
                    }
                }

            private:
                SlideShowContext maContext;
            };

            class EndIntrinsicAnimationAdaptor
            {
            public:
                EndIntrinsicAnimationAdaptor()
                {
                }

                void operator()( ShapeSharedPtr const& rShape ) const
                {
                    if( rShape &&
                        rShape->hasIntrinsicAnimation() )
                    {
                        IntrinsicAnimationSharedPtr pIntrinsicAnimation(
                            ::boost::dynamic_pointer_cast< IntrinsicAnimation >( rShape ) );

                        if( pIntrinsicAnimation )
                            pIntrinsicAnimation->endIntrinsicAnimation();
                    }
                }
            };
        }

        void Slide::endIntrinsicAnimations()
        {
            if( maContext.mpLayerManager )
            {
                maContext.mpLayerManager->forEachShape( 
                    EndIntrinsicAnimationAdaptor() );
            }
        }

        void Slide::startIntrinsicAnimations()
        {
            if( maContext.mpLayerManager )
            {
                maContext.mpLayerManager->forEachShape( 
                    StartIntrinsicAnimationAdaptor( maContext ) );
            }
        }
    
        struct Slide::ShapesIterationFunc {
            Slide *const that;
            explicit ShapesIterationFunc( Slide * that_ )
                : that(that_) {}
            void operator()( ShapeSharedPtr const& pShape ) const
            {
                if (pShape) {
                    if (pShape->hasHyperlinks())
                        that->mpEventBroadcaster->addHyperlinkShape(pShape);
                    if (pShape->hasIntrinsicAnimation()) {
                        DrawShapeSharedPtr const pDrawShape(
                            boost::shared_dynamic_cast<DrawShape>(pShape) );
                        if (pDrawShape) {
                            pDrawShape->initIntrinsicAnimation(
                                that->maContext, pShape );
                        }
                    }
                }
            }
        };
    
        bool Slide::applyInitialShapeAttributes( const uno::Reference< animations::XAnimationNode >& xRootAnimationNode )
        {
            if( !implPrefetchShow() )
                return false;
            
            // init all intrinsic animations of shapes, so that those present
            // a correct initial state:
            maContext.mpLayerManager->forEachShape( ShapesIterationFunc(this) );
            
            if( !xRootAnimationNode.is() )
            {
                meAnimationState = INITIAL_STATE;

                return true; // no animations - no attributes to apply
                             // - succeeded
            }

            uno::Reference< animations::XTargetPropertiesCreator > xPropsCreator;

            try
            {
                ENSURE_AND_RETURN( maContext.mxComponentContext.is(),
                                   "Slide::applyInitialShapeAttributes(): Invalid component context" );

                uno::Reference<lang::XMultiComponentFactory> xFac( 
                    maContext.mxComponentContext->getServiceManager() );

                xPropsCreator.set(
                    xFac->createInstanceWithContext(
                        ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM(
                                             "com.sun.star.animations.TargetPropertiesCreator") ),
                        maContext.mxComponentContext ), 
                    uno::UNO_QUERY_THROW );
            }
            catch( uno::Exception& ) 
            {
                OSL_ENSURE(
                    false,
                    rtl::OUStringToOString(
                        comphelper::anyToString(cppu::getCaughtException()),
                        RTL_TEXTENCODING_UTF8 ) );
                
                // could not determine initial shape attributes - this
                // is an error, as some effects might then be plainly
                // invisible
                ENSURE_AND_RETURN( false,
                                   "Slide::applyInitialShapeAttributes(): "
                                   "couldn't create TargetPropertiesCreator." );
            }

            uno::Sequence< animations::TargetProperties > aProps(
                xPropsCreator->createInitialTargetProperties( xRootAnimationNode ) );

            // apply extracted values to our shapes
            const ::std::size_t nSize( aProps.getLength() );
            for( ::std::size_t i=0; i<nSize; ++i )
            {
                sal_Int16 							nParaIndex( -1 );
                uno::Reference< drawing::XShape >	xShape( aProps[i].Target,
                                                            uno::UNO_QUERY );

                if( !xShape.is() ) 
                {
                    // not a shape target. Maybe a ParagraphTarget?
                    presentation::ParagraphTarget aParaTarget;
                    
                    if( (aProps[i].Target >>= aParaTarget) )
                    {
                        // yep, ParagraphTarget found - extract shape
                        // and index
                        xShape = aParaTarget.Shape;
                        nParaIndex = aParaTarget.Paragraph;
                    }
                }

                if( xShape.is() )
                {
                    ShapeSharedPtr pShape( maContext.mpLayerManager->lookupShape( xShape ) );

                    if( !pShape )
                    {
                        OSL_ENSURE( false,
                                    "Slide::applyInitialShapeAttributes(): no shape found for given target" );
                        continue;
                    }

                    AttributableShapeSharedPtr pAttrShape( 
                        ::boost::dynamic_pointer_cast< AttributableShape >( pShape ) );

                    if( !pAttrShape )
                    {
                        OSL_ENSURE( false,
                                    "Slide::applyInitialShapeAttributes(): shape found does not "
                                    "implement AttributableShape interface" );
                        continue;
                    }

                    if( nParaIndex != -1 )
                    {
                        // our target is a paragraph subset, thus look
                        // this up first.
                        const DocTreeNodeSupplier& rNodeSupplier( pAttrShape->getTreeNodeSupplier() );

                        pAttrShape = pAttrShape->getSubset( 
                            rNodeSupplier.getTreeNode(
                                nParaIndex,
                                DocTreeNode::NODETYPE_LOGICAL_PARAGRAPH ) );

                        if( !pAttrShape )
                        {
                            OSL_ENSURE( false,
                                        "Slide::applyInitialShapeAttributes(): shape found does not "
                                        "provide a subset for requested paragraph index" );
                            continue;
                        }
                    }

                    const uno::Sequence< beans::NamedValue >& rShapeProps( aProps[i].Properties );
                    const ::std::size_t nShapePropSize( rShapeProps.getLength() );
                    for( ::std::size_t j=0; j<nShapePropSize; ++j )
                    {
                        bool bVisible;
                        if( rShapeProps[j].Name.equalsIgnoreAsciiCaseAscii("visibility") &&
                            extractValue( bVisible,
                                          rShapeProps[j].Value,
                                          pShape,
                                          maContext.mpLayerManager ) )
                        {
                            pAttrShape->setVisibility( bVisible );
                        }
                        else
                        {
                            OSL_ENSURE( false,
                                        "Slide::applyInitialShapeAttributes(): Unexpected "
                                        "(and unimplemented) property encountered" );
                        }
                    }
                }
            }

            meAnimationState = INITIAL_STATE;

            return true;
        }

        bool Slide::prefetchShapes()
        {
            if( mbShapesLoaded )
                return true;

            ENSURE_AND_RETURN( mxDrawPage.is(), 
                               "Slide::prefetchShapes(): Invalid draw page" );
            ENSURE_AND_RETURN( maContext.mpLayerManager, 
                               "Slide::prefetchShapes(): Invalid layer manager" );

            // fetch desired page content
            // ==========================

            // also take master page content
            uno::Reference< drawing::XDrawPage > xMasterPage;
            uno::Reference< drawing::XShapes >   xMasterPageShapes;
            sal_Int32                            nCurrCount(0);

            uno::Reference< drawing::XMasterPageTarget > xMasterPageTarget( mxDrawPage, 
                                                                            uno::UNO_QUERY );
            if( xMasterPageTarget.is() )
            {
                xMasterPage = xMasterPageTarget->getMasterPage();
                xMasterPageShapes.set( xMasterPage, 
                                       uno::UNO_QUERY );

                if( xMasterPage.is() && xMasterPageShapes.is() )
                {
                    // TODO(P2): maybe cache master pages here (or treat the
                    // masterpage as a single metafile. At least currently,
                    // masterpages do not contain animation effects)
                    try
                    {
                        // add the background shape
                        // -------------------------------------------------------------------------
                        maContext.mpLayerManager->addShape( 
                            ShapeSharedPtr(
                                new BackgroundShape( mxDrawPage, 
                                                     xMasterPage,
                                                     maContext )));


                        // load the masterpage shapes
                        // -------------------------------------------------------------------------
                        ShapeImporter aMPShapesFunctor( xMasterPage, 
                                                        mxDrawPage,
                                                        maContext,
                                                        1 /* background shape has 0 */,
                                                        true );

                        while( !aMPShapesFunctor.isImportDone() )
                            maContext.mpLayerManager->addShape( aMPShapesFunctor.importShape() );

                        nCurrCount = xMasterPageShapes->getCount() + 1;
                    }
                    catch( ShapeLoadFailedException& )
                    {
                        // TODO(E2): Error handling. For now, bail out
                        OSL_ENSURE( false,
                                    "Slide::prefetchShapes(): caught ShapeLoadFailedException" );
                        return false;

                    }
                    catch( uno::Exception& )
                    {
                        // TODO(E2): Error handling. For now, bail out
                        OSL_ENSURE( false,
                                    "Slide::prefetchShapes(): caught uno::Exception" );
                        return false;
                    }
                }
            }

            try
            {
                // load the normal page shapes
                // -------------------------------------------------------------------------

                ShapeImporter aShapesFunctor( mxDrawPage, 
                                              mxDrawPage,
                                              maContext,
                                              nCurrCount, 
                                              false );

                while( !aShapesFunctor.isImportDone() )
                    maContext.mpLayerManager->addShape( aShapesFunctor.importShape() );
            }
            catch( ShapeLoadFailedException& )
            {
                // TODO(E2): Error handling. For now, bail out
                OSL_ENSURE( false,
                            "Slide::prefetchShapes(): caught ShapeLoadFailedException" );
                return false;
            }
            catch( uno::Exception& )
            {
                // TODO(E2): Error handling. For now, bail out
                OSL_ENSURE( false,
                            "Slide::prefetchShapes(): caught uno::Exception" );
                return false;
            }

            mbShapesLoaded = true;

            return true;
        }

    }
}

