/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: ainfoscrolltext.cxx,v $
 *
 *  $Revision: 1.9.68.1 $
 *
 *  last change: $Author: kz $ $Date: 2007/01/24 13:33:44 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_svx.hxx"

#ifndef _SDR_ANIMATION_AINFOSCROLLTEXT_HXX
#include <svx/sdr/animation/ainfoscrolltext.hxx>
#endif

#ifndef _SDR_MIXER_BASICMIXER_HXX
#include <svx/sdr/mixer/basicmixer.hxx>
#endif

#ifndef _SVDOTEXT_HXX
#include <svdotext.hxx>
#endif

#ifndef _SDR_PROPERTIES_PROPERTIES_HXX
#include <svx/sdr/properties/properties.hxx>
#endif

#ifndef _SDR_CONTACT_DISPLAYINFO_HXX
#include <svx/sdr/contact/displayinfo.hxx>
#endif

#ifndef _SV_SALBTYPE_HXX
#include <vcl/salbtype.hxx>
#endif

#ifndef _SFXITEMSET_HXX
#include <svtools/itemset.hxx>
#endif

//////////////////////////////////////////////////////////////////////////////

namespace sdr
{
	namespace mixer
	{
		// #i38135# Reorganisation of AInfoScrollText
		class TextScrollMixer : public BasicMixer
		{
			// the associated animationinfo
			sdr::animation::AInfoScrollText&		mrAInfoScrollText;

			// the remembered MixerState
			double									mfMixerState;

		protected:
			// setup associated object for paint at given state
			virtual void SetupObject(double fMixerState, sdr::contact::DisplayInfo& rDisplayInfo);

			// restore associated object to original state
			virtual void RestoreObject(sdr::contact::DisplayInfo& rDisplayInfo);

		public:
			// basic constructor.
			TextScrollMixer(sdr::animation::AInfoScrollText& rAIScrollText);

			// destructor
			virtual ~TextScrollMixer();
		};
	} // end of namespace mixer
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////

namespace sdr
{
	namespace animation
	{
		ScrollTextAnimNode::ScrollTextAnimNode(sal_uInt32 nDuration, sal_uInt32 nRepeat, double fStart, double fStop,
			sal_uInt32 nFrequency, bool bAlternate)
		:	mnDuration(nDuration),
			mnRepeat(nRepeat),
			mfStart(fStart),
			mfStop(fStop),
			mnFrequency(nFrequency),
			mbAlternate(bAlternate)
		{
			// #i71035# A duration of 0L is not allowed
			DBG_ASSERT(mnDuration, "ScrollTextAnimNode: duration == 0L is not allowed (!)");

			// #i71035# prevent duration of 0L and set to some minimum in ms which will not
			// block user inputs
			if(mnDuration < 25L)
			{
				mnDuration = 25L;
			}
		}

		// #i38135# Reorganisation of AInfoScrollText
		double ScrollTextAnimNode::GetStateAtRelativeTime(sal_uInt32 nRelativeTime) const
		{
			if(mnRepeat)
			{
				// ending
				const sal_uInt32 nRepeatCount(nRelativeTime / mnDuration);
				sal_uInt32 nFrameTime(nRelativeTime - (nRepeatCount * mnDuration));

				if(DoAlternate() && (nRepeatCount + 1L) % 2L)
				{
					nFrameTime = mnDuration - nFrameTime;
				}

				return mfStart + ((mfStop - mfStart) * ((double)nFrameTime / (double)mnDuration));
			}
			else
			{
				// endless
				sal_uInt32 nFrameTime(nRelativeTime % mnDuration);

				if(DoAlternate())
				{
					const sal_uInt32 nRepeatCount(nRelativeTime / mnDuration);

					if((nRepeatCount + 1L) % 2L)
					{
						nFrameTime = mnDuration - nFrameTime;
					}
				}

				return mfStart + ((mfStop - mfStart) * ((double)nFrameTime / (double)mnDuration));
			}
		}
	} // end of namespace animation
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////

namespace sdr
{
	namespace mixer
	{
		// setup associated object for paint at given state
		void TextScrollMixer::SetupObject(double fMixerState, sdr::contact::DisplayInfo& rDisplayInfo)
		{
			// setup internal scaling values
			mrAInfoScrollText.InitScaling(rDisplayInfo);

			// Hide text for normal object paint
			SdrTextObj& rTextObj = (SdrTextObj&)mrAInfoScrollText.GetSdrObject();
			rTextObj.NbcSetTextHidden(sal_True);

			// remember MixerState for RestoreObject actions
			mfMixerState = fMixerState;
		}

		// restore associated object to original state
		void TextScrollMixer::RestoreObject(sdr::contact::DisplayInfo& rDisplayInfo)
		{
			// Switch back to show text again
			SdrTextObj& rTextObj = (SdrTextObj&)mrAInfoScrollText.GetSdrObject();
			rTextObj.NbcSetTextHidden(sal_False);

			// paint the animation MetaFile
			GDIMetaFile* pMetaFile = mrAInfoScrollText.GetTextMetaFile();

			if(pMetaFile)
			{
				OutputDevice* pOut = rDisplayInfo.GetOutputDevice();
				sal_Int32 nRotation(rTextObj.GetGeoStat().nDrehWink);

				// rescue ClipRegion
				sal_Bool bClipSet(pOut->IsClipRegion());
				Region aClipSet(pOut->GetClipRegion());

				// calculate offset for scrolling
				const Rectangle& rScrollRectangleLogic = mrAInfoScrollText.GetScrollRectangleLogic();
				const Rectangle& rPaintRectangleLogic = mrAInfoScrollText.GetPaintRectangleLogic();
				Point aOffsetScrolling;

				if(mrAInfoScrollText.ScrollHorizontal())
				{
					const double fZeroEquiv(rScrollRectangleLogic.Left() - rPaintRectangleLogic.GetWidth());
					const double fOneEquiv(rScrollRectangleLogic.Right());
					const double fPositionInState(fZeroEquiv + (mfMixerState * (fOneEquiv - fZeroEquiv)));
					const double fOriginalPosEquiv(rPaintRectangleLogic.Left());
					aOffsetScrolling = Point(FRound(fPositionInState - fOriginalPosEquiv) , 0L);
				}
				else
				{
					const double fZeroEquiv(rScrollRectangleLogic.Top() - rPaintRectangleLogic.GetHeight());
					const double fOneEquiv(rScrollRectangleLogic.Bottom());
					const double fPositionInState(fZeroEquiv + (mfMixerState * (fOneEquiv - fZeroEquiv)));
					const double fOriginalPosEquiv(rPaintRectangleLogic.Top());
					aOffsetScrolling = Point(0L, FRound(fPositionInState - fOriginalPosEquiv));
				}

				// remember old pixel offset
				Size aOldPixelOffset(pOut->GetPixelOffset());

				// different handling for rotated or not
				if(nRotation)
				{
					// set ClipRegion. Use union with object rect to expand accordingly.
					const Rectangle& rScrollRectangle = mrAInfoScrollText.GetScrollRectangleLogic();
					Polygon aClipPolygon(rScrollRectangle);
					Rectangle aUnrotatedRect;
					rTextObj.SdrTextObj::TakeUnrotatedSnapRect(aUnrotatedRect);
					RotatePoly(aClipPolygon, aUnrotatedRect.TopLeft(), rTextObj.GetGeoStat().nSin, rTextObj.GetGeoStat().nCos);
					pOut->SetClipRegion(Region(aClipPolygon.GetBoundRect()));

					// add scrolling offset. Use SetPixelOffset to avoid rounding errors in the
					// MetaFile painting
					RotatePoint(aOffsetScrolling, Point(), rTextObj.GetGeoStat().nSin, rTextObj.GetGeoStat().nCos);
					Size aPixelOffsetScrolling(
						aOldPixelOffset.Width() + mrAInfoScrollText.ToPixel(aOffsetScrolling.X()),
						aOldPixelOffset.Height() + mrAInfoScrollText.ToPixel(aOffsetScrolling.Y()));
					pOut->SetPixelOffset(aPixelOffsetScrolling);
				}
				else
				{
					// set ClipRegion
					const Rectangle& rScrollRectangle = mrAInfoScrollText.GetScrollRectangleLogic();
					pOut->SetClipRegion(Region(rScrollRectangle));

					// copy MetaFile
					GDIMetaFile aMetaFile(*pMetaFile);

					// add scrolling offset. Use SetPixelOffset to avoid rounding errors in the
					// MetaFile painting
					Size aPixelOffsetScrolling(
						aOldPixelOffset.Width() + mrAInfoScrollText.ToPixel(aOffsetScrolling.X()),
						aOldPixelOffset.Height() + mrAInfoScrollText.ToPixel(aOffsetScrolling.Y()));
					pOut->SetPixelOffset(aPixelOffsetScrolling);
				}

				// paint MetaFile
				pMetaFile->WindStart();
				pMetaFile->Play(pOut);

				// restore pixel offset
				pOut->SetPixelOffset(aOldPixelOffset);

				// restore ClipRegion
				if(bClipSet)
				{
					pOut->SetClipRegion(aClipSet);
				}
				else
				{
					pOut->SetClipRegion();
				}
			}
		}

		TextScrollMixer::TextScrollMixer(sdr::animation::AInfoScrollText& rAIScrollText)
		:	mrAInfoScrollText(rAIScrollText)
		{
		}

		TextScrollMixer::~TextScrollMixer()
		{
		}
	} // end of namespace mixer
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////

namespace sdr
{
	namespace animation
	{
		// Access the BasicMixer, plus creation on demand.
		sdr::mixer::BasicMixer* AInfoScrollText::CreateBasicMixer()
		{
			return new sdr::mixer::TextScrollMixer((AInfoScrollText&)(*this));
		}

		void AInfoScrollText::ImpForceScrollTextAnimNodes()
		{
			if(IsScalingInitialized())
			{
				if(!maVector.size())
				{
					// prepare values
					sal_uInt32 nLoopTime;
					double fZeroLogic, fOneLogic, fInitLogic, fDistanceLogic;
					double fZeroLogicAlternate(0.0), fOneLogicAlternate(1.0);
					double fZeroRelative, fOneRelative, fInitRelative, fDistanceRelative;

					if(ScrollHorizontal())
					{
						if(DoAlternate())
						{
							if(maPaintRectangleLogic.GetWidth() > maScrollRectangleLogic.GetWidth())
							{
								fZeroLogicAlternate = maScrollRectangleLogic.Right() - maPaintRectangleLogic.GetWidth();
								fOneLogicAlternate = maScrollRectangleLogic.Left();
							}
							else
							{
								fZeroLogicAlternate = maScrollRectangleLogic.Left();
								fOneLogicAlternate = maScrollRectangleLogic.Right() - maPaintRectangleLogic.GetWidth();
							}
						}
						
						fZeroLogic = maScrollRectangleLogic.Left() - maPaintRectangleLogic.GetWidth();
						fOneLogic = maScrollRectangleLogic.Right();
						fInitLogic = maPaintRectangleLogic.Left();
					}
					else
					{
						if(DoAlternate())
						{
							if(maPaintRectangleLogic.GetHeight() > maScrollRectangleLogic.GetHeight())
							{
								fZeroLogicAlternate = maScrollRectangleLogic.Bottom() - maPaintRectangleLogic.GetHeight();
								fOneLogicAlternate = maScrollRectangleLogic.Top();
							}
							else
							{
								fZeroLogicAlternate = maScrollRectangleLogic.Top();
								fOneLogicAlternate = maScrollRectangleLogic.Bottom() - maPaintRectangleLogic.GetHeight();
							}
						}
						
						fZeroLogic = maScrollRectangleLogic.Top() - maPaintRectangleLogic.GetHeight();
						fOneLogic = maScrollRectangleLogic.Bottom();
						fInitLogic = maPaintRectangleLogic.Top();
					}

					fDistanceLogic = fOneLogic - fZeroLogic;
					fInitRelative = (fInitLogic - fZeroLogic) / fDistanceLogic;

					if(DoAlternate())
					{
						fZeroRelative = (fZeroLogicAlternate - fZeroLogic) / fDistanceLogic;
						fOneRelative = (fOneLogicAlternate - fZeroLogic) / fDistanceLogic;
						fDistanceRelative = fOneRelative - fZeroRelative;
					}
					else
					{
						fZeroRelative = 0.0;
						fOneRelative = 1.0;
						fDistanceRelative = 1.0;
					}

					if(mnStartTime)
					{
						// Start time loop
						ScrollTextAnimNode aStartNode(mnStartTime, 1L, 0.0, 0.0, mnStartTime, false);
						maVector.push_back(aStartNode);
					}

					if(IsVisibleWhenStarted())
					{
						double fRelativeStartValue, fRelativeEndValue, fRelativeDistance;

						if(DoScrollForward())
						{
							fRelativeStartValue = fInitRelative;
							fRelativeEndValue = fOneRelative;
							fRelativeDistance = fRelativeEndValue - fRelativeStartValue;
						}
						else
						{
							fRelativeStartValue = fInitRelative;
							fRelativeEndValue = fZeroRelative;
							fRelativeDistance = fRelativeStartValue - fRelativeEndValue;
						}
						
						const double fNumberSteps = (fRelativeDistance * fDistanceLogic) / GetStepWidthLogic(100L);
						nLoopTime = FRound(fNumberSteps * mnFrequency);

						// #i71035# A duration of 0L is not allowed
						if(nLoopTime)
						{
							// init loop
							ScrollTextAnimNode aInitNode(nLoopTime, 1L, fRelativeStartValue, fRelativeEndValue, mnFrequency, false);
							maVector.push_back(aInitNode);
						}
					}

					// prepare main loop values
					{
						double fRelativeStartValue, fRelativeEndValue, fRelativeDistance;

						if(DoScrollForward())
						{
							fRelativeStartValue = fZeroRelative;
							fRelativeEndValue = fOneRelative;
							fRelativeDistance = fRelativeEndValue - fRelativeStartValue;
						}
						else
						{
							fRelativeStartValue = fOneRelative;
							fRelativeEndValue = fZeroRelative;
							fRelativeDistance = fRelativeStartValue - fRelativeEndValue;
						}

						const double fNumberSteps = (fRelativeDistance * fDistanceLogic) / GetStepWidthLogic(100L);
						nLoopTime = FRound(fNumberSteps * mnFrequency);
						
						// #i71035# A duration of 0L is not allowed
						if(nLoopTime)
						{
							if(0L == mnRepeat)
							{
								if(!DoScrollIn())
								{
									// endless main loop
									ScrollTextAnimNode aMainNode(nLoopTime, 0L, fRelativeStartValue, fRelativeEndValue, mnFrequency, DoAlternate());
									maVector.push_back(aMainNode);
								}
							}
							else
							{
								sal_uInt32 nNumRepeat(mnRepeat);

								if(DoAlternate() && (nNumRepeat + 1L) % 2L)
								{
									nNumRepeat += 1L;
								}

								// ending main loop
								ScrollTextAnimNode aMainNode(nLoopTime, nNumRepeat, fRelativeStartValue, fRelativeEndValue, mnFrequency, DoAlternate());
								maVector.push_back(aMainNode);
							}
						}
					}

					if(IsVisibleWhenStopped())
					{
						double fRelativeStartValue, fRelativeEndValue, fRelativeDistance;

						if(DoScrollForward())
						{
							fRelativeStartValue = fZeroRelative;
							fRelativeEndValue = fInitRelative;
							fRelativeDistance = fRelativeEndValue - fRelativeStartValue;
						}
						else
						{
							fRelativeStartValue = fOneRelative;
							fRelativeEndValue = fInitRelative;
							fRelativeDistance = fRelativeStartValue - fRelativeEndValue;
						}
						
						const double fNumberSteps = (fRelativeDistance * fDistanceLogic) / GetStepWidthLogic(100L);
						nLoopTime = FRound(fNumberSteps * mnFrequency);

						// #i71035# A duration of 0L is not allowed
						if(nLoopTime)
						{
							// exit loop
							ScrollTextAnimNode aExitNode(nLoopTime, 1L, fRelativeStartValue, fRelativeEndValue, mnFrequency, false);
							maVector.push_back(aExitNode);
						}
					}
				}
			}
			else
			{
				maVector.clear();
			}
		}

		ScrollTextAnimNode* AInfoScrollText::ImpGetScrollTextAnimNode(sal_uInt32 nTime, sal_uInt32& rRelativeTime)
		{
			ScrollTextAnimNode* pRetval = 0L;
			ImpForceScrollTextAnimNodes();

			if(maVector.size())
			{
				rRelativeTime = nTime;

				for(sal_uInt32 a(0L); !pRetval && a < maVector.size(); a++)
				{
					if(!maVector[a].GetRepeat())
					{
						// endless loop, use it
						pRetval = &maVector[a];
					}
					else if(maVector[a].GetFullTime() > rRelativeTime)
					{
						// ending node
						pRetval = &maVector[a];
					}
					else
					{
						// look at next
						rRelativeTime -= maVector[a].GetFullTime();
					}
				}
			}

			return pRetval;
		}

		double AInfoScrollText::ImpGetScrollTextMixerState(sal_uInt32 nTime)
		{
			double fRetval(0.0);
			ImpForceScrollTextAnimNodes();

			if(maVector.size())
			{
				sal_uInt32 nRelativeTime;
				ScrollTextAnimNode* pNode = ImpGetScrollTextAnimNode(nTime, nRelativeTime);

				if(pNode)
				{
					// use node
					fRetval = pNode->GetStateAtRelativeTime(nRelativeTime);
				}
				else
				{
					// end of animation, take last entry's end
					fRetval = maVector[maVector.size() - 1L].GetStop();
				}
			}

			return fRetval;
		}

		sal_uInt32 AInfoScrollText::ImpRegisterAgainScrollTextMixerState(sal_uInt32 nTime)
		{
			sal_uInt32 nRetval(0L);
			ImpForceScrollTextAnimNodes();

			if(maVector.size())
			{
				sal_uInt32 nRelativeTime;
				ScrollTextAnimNode* pNode = ImpGetScrollTextAnimNode(nTime, nRelativeTime);

				if(pNode)
				{
					// take register time
					nRetval = pNode->GetFrequency();
				}
			}
			else
			{
				// #i38135# not initialized, return default
				nRetval = mnFrequency;
			}

			return nRetval;
		}

		// calculate the MixerState value for given time
		double AInfoScrollText::GetMixerState(sal_uInt32 nTime, sdr::contact::DisplayInfo& rDisplayInfo) const
		{
			// setup internal scaling values
			InitScaling(rDisplayInfo);

			return ((AInfoScrollText*)this)->ImpGetScrollTextMixerState(nTime);
		}

		// local initializations
		void AInfoScrollText::ImplInit()
		{
			SdrTextObj& rTextObj = (SdrTextObj&)GetSdrObject();
			const SdrTextAniKind eAniKind(rTextObj.GetTextAniKind());
			const SfxItemSet& rSet = rTextObj.GetProperties().GetObjectItemSet();
			
			mbScrollIn = (SDRTEXTANI_SLIDE == eAniKind);
			mbAlternate = (SDRTEXTANI_ALTERNATE == eAniKind);

			// If it is a simple mbScrollIn, reset some parameters
			if(DoScrollIn())
			{
				// most parameters are set correctly form the dialog logic, but
				// e.g. VisisbleWhenStopped is grayed out and needs to be corrected here.
				mbVisisbleWhenStopped = sal_True;
				mbVisisbleWhenStarted = sal_False;
				mnRepeat = 0L;
			}

			// Get animation direction
			meDirection = ((SdrTextAniDirectionItem&)(rSet.Get(SDRATTR_TEXT_ANIDIRECTION))).GetValue();

			// Get step width. Negative means pixel, positive logical units
			mnStepWidth = (sal_Int32)((SdrTextAniAmountItem&)rSet.Get(SDRATTR_TEXT_ANIAMOUNT)).GetValue();

			// #i71337# memory leak: when fetching a new metafile, the old one needs to be deleted
			if(mpMetaFile)
			{
				delete mpMetaFile;
			}

			// Get Text Scroll MetaFile And Rectangle
			mpMetaFile = rTextObj.GetTextScrollMetaFileAndRectangle(maScrollRectangleLogic, maPaintRectangleLogic);
			DBG_ASSERT(mpMetaFile, "AInfoScrollText::ImplInit: Got no Text MetaFile (!)");

			// test if object is rotated. If so, adapt rotation to the MetaFile.
			sal_Int32 nRotation(rTextObj.GetGeoStat().nDrehWink);

			if(nRotation)
			{
				Rectangle aUnrotatedRect;
				rTextObj.SdrTextObj::TakeUnrotatedSnapRect(aUnrotatedRect);

				mpMetaFile->Move(-aUnrotatedRect.Left(), -aUnrotatedRect.Top());
				mpMetaFile->Rotate(nRotation / 10);
				mpMetaFile->Move(aUnrotatedRect.Left(), aUnrotatedRect.Top());
			}

			// init vector
			maVector.clear();
		}

		// basic constructor.
		AInfoScrollText::AInfoScrollText(SdrObject& rObject, sal_uInt32 nDefaultFrequency)
		:	AInfoBlinkText(rObject, nDefaultFrequency),
			mpMetaFile(0L),
			mfPixelToLogic(1.0),
			mnStartTime(0L),
			mbAlternate(sal_False),
			mbPixelToLogicIsSet(sal_False),
			mbScrollIn(sal_False)
		{
			// local initialization
			ImplInit();
		}

		// destructor
		AInfoScrollText::~AInfoScrollText()
		{
			// get rid of MetaFile
			if(mpMetaFile)
			{
				delete mpMetaFile;
			}
		}

		// test if to register at ObjectAnimator again after given time.
		// Fill in the new time value and return accordingly.
		sal_Bool AInfoScrollText::DoRegisterAgain(sal_uInt32 nTime, sal_uInt32& rNewTime, const AnimationState& /*rAssociatedAS*/) const
		{
			sal_uInt32 nFrequency( ((AInfoScrollText*)this)->ImpRegisterAgainScrollTextMixerState(nTime) );

			if(nFrequency)
			{
				rNewTime = nTime + nFrequency;
				return true;
			}

			return false;
		}

		// React on changes of the object of this AnimationInfo.
		void AInfoScrollText::ActionChanged()
		{
			// call parent
			AInfoBlinkText::ActionChanged();

			// work on own changes
			ImplInit();
		}

		// access to rectangles
		const Rectangle& AInfoScrollText::GetScrollRectangleLogic() const
		{
			return maScrollRectangleLogic;
		}

		const Rectangle& AInfoScrollText::GetPaintRectangleLogic() const
		{
			return maPaintRectangleLogic;
		}

		// access to MetaFile
		GDIMetaFile* AInfoScrollText::GetTextMetaFile() const
		{
			return mpMetaFile;
		}

		// scroll horizontal?
		sal_Bool AInfoScrollText::ScrollHorizontal() const
		{
			return (SDRTEXTANI_LEFT == meDirection || SDRTEXTANI_RIGHT == meDirection);
		}

		// Access to StepWidth in logical units
		// #i71337# changing interface to allow handover of minimum return value
		sal_uInt32 AInfoScrollText::GetStepWidthLogic(sal_uInt32 nMinimum) const
		{
			sal_uInt32 nRetval(0L);

			if(mnStepWidth < 0L)
			{
				// is in pixels, convert to logical units
				nRetval = ToLogic(-mnStepWidth);
			}
			else if(mnStepWidth > 0L)
			{
				// is in logical units
				nRetval = mnStepWidth;
			}

			if(0L == nRetval)
			{
				if(IsScalingInitialized())
				{
					// step 1 pixel, use calculated value
					nRetval = FRound(mfPixelToLogic);
					
					// #128389#
					// with very high DPIs like in PDF export, this can still get zero.
					// for that cases, set a default, too.
					if(nRetval < nMinimum)
					{
						nRetval = nMinimum;
					}
				}
				else
				{
					nRetval = nMinimum;
				}
			}

			return nRetval;
		}

		// is the animation direction opposite?
		sal_Bool AInfoScrollText::DoScrollForward() const
		{
			return (SDRTEXTANI_RIGHT == meDirection || SDRTEXTANI_DOWN == meDirection);
		}

		// do alternate text directions?
		sal_Bool AInfoScrollText::DoAlternate() const
		{
			return mbAlternate;
		}

		// do scroll in?
		sal_Bool AInfoScrollText::DoScrollIn() const
		{
			return mbScrollIn;
		}

		// set the scaling value
		void AInfoScrollText::InitScaling(sdr::contact::DisplayInfo& rDisplayInfo) const
		{
			OutputDevice* pOut = rDisplayInfo.GetOutputDevice();

			if(pOut)
			{
				Size aSize(500, 0);
				aSize = pOut->PixelToLogic(aSize);
				const double fNewValue((double)aSize.Width() / 500.0);

				// #i38135# React on zoom changes, too
				if(fNewValue != mfPixelToLogic)
				{
					AInfoScrollText* pNonConstThis = (AInfoScrollText*)this;
					pNonConstThis->mfPixelToLogic = fNewValue;
					pNonConstThis->mbPixelToLogicIsSet = sal_True;

					// re-init vector
					pNonConstThis->maVector.clear();
				}
			}
		}

		// is scale value set?
		sal_Bool AInfoScrollText::IsScalingInitialized() const
		{
			return mbPixelToLogicIsSet;
		}

		// convert Logic->Pixel
		sal_Int32 AInfoScrollText::ToPixel(sal_Int32 nValue) const
		{
			return FRound((double)nValue / mfPixelToLogic);
		}

		// convert Pixel->Logic
		sal_Int32 AInfoScrollText::ToLogic(sal_Int32 nValue) const
		{
			return FRound((double)nValue * mfPixelToLogic);
		}

		// #i71337#
		sal_Bool AInfoScrollText::IsAnimationAllowed(const sdr::contact::ViewObjectContact& rVOContact) const
		{
			sal_Bool bRetval(AInfoBlinkText::IsAnimationAllowed(rVOContact));

			if(bRetval && DoAlternate())
			{
				// #i71337#
				// since IsAnimationAllowed is triggered before ActionChanged() is applied to this
				// AInfoScrollText, get rectangles exclusively
				Rectangle aScroll, aPaint;
				SdrTextObj& rTextObj = (SdrTextObj&)GetSdrObject();
				GDIMetaFile* pTempMetaFile = rTextObj.GetTextScrollMetaFileAndRectangle(aScroll, aPaint);
				delete pTempMetaFile;

				// check if scroll rectangle is too small. If yes, scrolling back and forth makes no sense
				const sal_uInt32 nStepWidthLogic(GetStepWidthLogic(10L));
				sal_Int32 nDeltaX(aScroll.getWidth() - aPaint.getWidth());
				sal_Int32 nDeltaY(aScroll.getHeight() - aPaint.getHeight());

				if(nDeltaX < 0)
				{
					nDeltaX = -nDeltaX;
				}
				
				if(nDeltaY < 0)
				{
					nDeltaY = -nDeltaY;
				}

				const sal_uInt32 nMaxXY(nDeltaX > nDeltaY ? (sal_uInt32)nDeltaX : (sal_uInt32)nDeltaY);

				if(nMaxXY < (nStepWidthLogic * 3))
				{
					// if movement is less than three logic step widths, do not move
					bRetval = sal_False;
				}
			}

			return bRetval;
		}
	} // end of namespace animation
} // end of namespace sdr

//////////////////////////////////////////////////////////////////////////////
// eof
