/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: layer.hxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: kz $ $Date: 2006/12/13 15:59:06 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _SLIDESHOW_LAYER_HXX
#define _SLIDESHOW_LAYER_HXX

#include <basegfx/range/b2dmultirange.hxx>

#include <boost/shared_ptr.hpp>
#include <boost/utility.hpp>

#include <vector>
#include <set>

#include <viewlayer.hxx>
#include <animatableshape.hxx>


namespace slideshow
{
    namespace internal
    {

		/* Definition of Layer class */

        /** This class represents one layer of output on a Slide.

        	Layers group shapes for a certain depth region of a slide.

        	Since slides have a notion of depth, i.e. shapes on it
        	have a certain order in which they lie upon each other,
        	this layering must be modeled. A prime example for this
        	necessity are animations of shapes lying behind other
        	shapes. Then, everything behind the animated shape will be
        	in a background layer, the shape itself will be in an
        	animation layer, and everything before it will be in a
        	foreground layer (these layers are most preferrably
        	modeled as XSprite objects internally).

            @attention All methods of this class are only supposed to
            be called from the LayerManager. Normally, it shouldn't be
            possible to get hold of an instance of this class at all.
         */
        class Layer : private boost::noncopyable
        {
        public:
            Layer();

            /** Add a new view layer.

            	This method adds a new view layer, this layer shall
            	show itself on. All Shapes of this Layer are then
            	rendered on the ViewLayer, without the need for
            	explicit update() calls
             */
            void addViewLayer( const ViewLayerSharedPtr& rNewLayer );

            /** Remove a view layer

            	This method removes the layer from the given view
            	layer.

                @return true, if the layer was successfully removed,
                false otherwise (e.g. if the view layer was not added
                in the first place).
             */
            bool removeViewLayer( const ViewLayerSharedPtr& rNewLayer );

            /** Notify that given ViewLayer has changed
             */
            void viewLayerChanged( const ViewLayerSharedPtr& rChangedLayer );

            /** Add the shape to this layer.

            	@param rShape
                The shape to add

            	@param nPrio
                Priority of the shape, must be in the range [0,1]. The
                priority determines the z position of the shape
                relative to this layer. The higher the priority, the
                further in front of this layer the shape resides.
	
             */
            void addShape( const ShapeSharedPtr& 	rShape ); 
            
            /** Remove the shape from this layer.

            	@return true, if the shape was successfully removed,
            	false otherwise (e.g. if the shape was not added to
            	this layer, in the first place).
             */
            bool removeShape( const ShapeSharedPtr& rShape );

            /** Notify the Layer that the given Shape starts an
                animation now.

				This method enters animation mode for the Shape on all
				registered views.
             */
            void enterAnimationMode( const AnimatableShapeSharedPtr& rShape );

            /** Notify the Layer that the given Shape is no
                longer animated.

				This methods ends animation mode for the given Shape
				on all registered views.
             */
            void leaveAnimationMode( const AnimatableShapeSharedPtr& rShape );

            /** Render the Layer.

				This method renders all shapes on the Layer on all
				registered view layers unconditionally, regardless of
				whether their attributes have changed or not.

	            @return whether the rendering finished successfully.
            */
            bool render() const;

            /** Update the given area on the Layer.

				This method renders all shapes on the Layer on all
				registered view layers, iff they intersect with the
				given update area.

	            @return whether the update finished successfully. If
	            one or more of the shapes failed to update, this
	            method will return false.
            */
            bool update( const ::basegfx::B2DMultiRange& rArea ) const;

        private:
            typedef ::std::set< ShapeSharedPtr, Shape::lessThanShape > 	ShapeSet;
            typedef ::std::vector< ViewLayerSharedPtr > 				ViewLayerVector;

            ShapeSet			maShapeSet;            
            ViewLayerVector		maViewLayers;
        };

        typedef ::boost::shared_ptr< Layer > LayerSharedPtr;

    }
}

#endif /* _SLIDESHOW_LAYER_HXX */
