/*************************************************************************
 *
 *  $RCSfile: drives.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 17:03:05 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#define INCL_DOSPROCESS
#define INCL_DOSDEVIOCTL;
#define INCL_DOSERRORS
#define INCL_DOSDEVICES
#include <svpm.h>
#include <ctype.h>
#include <string.h>

#ifndef _FOLDER_HXX
#include "folder.hxx"
#endif

#define DRIVE_A   0x00000001
#define DRIVE_B   0x00000002
#define DRIVE_C   0x00000004
#define DRIVE_D   0x00000008
#define DRIVE_E   0x00000010
#define DRIVE_F   0x00000020
#define DRIVE_G   0x00000040
#define DRIVE_H   0x00000080
#define DRIVE_I   0x00000100
#define DRIVE_J   0x00000200
#define DRIVE_K   0x00000400
#define DRIVE_L   0x00000800
#define DRIVE_M   0x00001000
#define DRIVE_N   0x00002000
#define DRIVE_O   0x00004000
#define DRIVE_P   0x00008000
#define DRIVE_Q   0x00010000
#define DRIVE_R   0x00020000
#define DRIVE_S   0x00040000
#define DRIVE_T   0x00080000
#define DRIVE_U   0x00100000
#define DRIVE_V   0x00200000
#define DRIVE_W   0x00400000
#define DRIVE_X   0x00800000
#define DRIVE_Y   0x01000000
#define DRIVE_Z   0x02000000
#define DRIVE_ALL 0x03FFFFFF

class DrivesCacheImpl
{
    struct DriveInfo {
        BOOL       bIsRemovable;
        VolumeKind aVolumeKind;
        String     aFileSystemName;
        String     aVolumeName;
        String     aMappingName;
    };

    DriveInfo m_aCache[26];

    ULONG m_ulDriveMap;
    ULONG m_ulCurrentDrive;
//    ULONG m_ulDiskChanged;

public:
    DrivesCacheImpl();

    void   update();
    UINT32 buildCRC();

//    inline BOOL diskChanged (ULONG nMask)  { return m_ulDiskChanged & nMask; };

    inline VolumeKind& getVolumeKind (BYTE n) { return m_aCache[n].aVolumeKind; };

    inline BOOL isRemovable (BYTE n) { return m_aCache[n].bIsRemovable; };
    inline BOOL isValidDrive(BYTE n) { return (m_ulDriveMap >> n) & 1;};

    inline String& getFileSystemName (BYTE n) { return m_aCache[n].aFileSystemName; };
    inline String& getVolumeName     (BYTE n) { return m_aCache[n].aVolumeName; };
    inline String& getMappingName    (BYTE n) { return m_aCache[n].aMappingName; };
};


enum {
    DEVTYPE_48TPI_DISKETTE = 0,
    DEVTYPE_96TPI_DISKETTE = 1,
    DEVTYPE_35INCH_1MB_DISKETTE = 2,
    DEVTYPE_8INCH_SD_DISKETTE = 3,
    DEVTYPE_8INCH_DD_DISKETTE = 4,
    DEVTYPE_FIXED_DISK = 5,
    DEVTYPE_TAPE_DRIVE = 6,
    DEVTYPE_OTHER = 7,
    DEVTYPE_OPTICAL = 8,
    DEVTYPE_35INCH_4MB_DISKETTE = 9,
    DEVTYPE_35INCH_2MB_DISKETTE = 256
};


struct FSInfo {
    FSQBUFFER2 fsq;
    CHAR       achBuffer[CCHMAXPATHCOMP * 3];
};

struct DTInfo
{
    BOOL  bRemovable;
//    BOOL  bDiskChanged;
    ULONG ulDevType;
};

struct CDInfo {
    USHORT usCount;
    USHORT usFirst;
};

static const char cFirstDriveLetter = 'a';
static const char cLastDriveLetter  = 'z';


/*------------------------------------------------------------------------------------
 * GetDTInfo()
 *------------------------------------------------------------------------------------
 */

inline BOOL GetDTInfo (UINT uiDrive, DTInfo *pDTInfo)
{
    APIRET rc;
    BYTE   abParm[2] = { 0, (BYTE)uiDrive };
    ULONG  ulParmLen = 2;
    ULONG  ulDataLen = 0;
    BIOSPARAMETERBLOCK bpb;

    memset(&bpb, 0, sizeof(bpb));

    // Assume Fixed Disk - Novell netware driver fails with filling in the structure so a default is needed
    // Other LAN drivers doesn't seem to set this either. Nether does Karl Olsen's RAMFS
    bpb.bDeviceType = DEVTYPE_FIXED_DISK;

    rc = DosDevIOCtl((HFILE)-1, IOCTL_DISK, DSK_GETDEVICEPARAMS,
                     (PVOID) &abParm[0], sizeof(abParm), &ulParmLen,
                     (PVOID) &bpb, sizeof(bpb), &ulDataLen);

    if((rc != NO_ERROR && rc != ERROR_NOT_SUPPORTED) || ulDataLen < 36)
    {
        return FALSE;
    }

    pDTInfo->ulDevType = bpb.bDeviceType;
    pDTInfo->bRemovable = (BOOL)((bpb.fsDeviceAttr & 0x01) ? FALSE : TRUE);
    // changeline flag. it is set, if the device driver determines that the media has
    // changed since the last I/O operation. This information is useful only if the driver
    // could be asked for a disk change itself.
    //  pDTInfo->bChanged   = (BOOL)((bpb.fsDeviceAttr & 0x02) ? TRUE  : FALSE);

    if((pDTInfo->ulDevType == DEVTYPE_OTHER) && pDTInfo->bRemovable) {
        switch(bpb.usBytesPerSector * bpb.cSectors) {
        case 512 * 1440:
            pDTInfo->ulDevType =  DEVTYPE_35INCH_1MB_DISKETTE;
            break;
        case 512 * 2880:
            pDTInfo->ulDevType =  DEVTYPE_35INCH_2MB_DISKETTE;
            break;
        case 512 * 5760:
            pDTInfo->ulDevType =  DEVTYPE_35INCH_4MB_DISKETTE;
            break;
        default:
            break;
        }
    }
    return TRUE;
}

/*------------------------------------------------------------------------------------
 * GetFSInfo()
 *------------------------------------------------------------------------------------
 */

inline BOOL GetFSInfo(UINT uiDrive, FSInfo *pFSInfo)
{
    CHAR  szDrive[3] = { (CHAR)('A' + uiDrive), ':', '\0' };
    ULONG ulBufLen = sizeof(FSInfo);
    memset( pFSInfo, 0, (size_t)ulBufLen );
    return NO_ERROR == DosQueryFSAttach( szDrive, 0, FSAIL_QUERYNAME, &pFSInfo->fsq, &ulBufLen);
}

/*------------------------------------------------------------------------------------
 * GetCDInfo()
 *------------------------------------------------------------------------------------
 */

inline BOOL GetCDInfo( CDInfo * pCDInfo )
{
    HFILE hFileCD;
    ULONG ulAction;
    
    if( NO_ERROR == DosOpen("\\DEV\\CD-ROM2$",
                            &hFileCD,
                            &ulAction,
                            0,
                            FILE_NORMAL,
                            OPEN_ACTION_OPEN_IF_EXISTS,
                            OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY,
                            NULL ))
    {
        ULONG  ulDataSize = sizeof(CDInfo);
        APIRET rc = DosDevIOCtl( hFileCD,
                                 0x82,
                                 0x60,
                                 NULL,
                                 0,
                                 NULL,
                                 (PVOID)pCDInfo,
                                 ulDataSize,
                                 &ulDataSize);
        DosClose(hFileCD);
        if(rc == NO_ERROR) {
            return TRUE;
        }
    }
    pCDInfo->usFirst = 0;
    pCDInfo->usCount = 0;
    return FALSE;
}

/*------------------------------------------------------------------------------------
 * DriveIsCDROM()
 *------------------------------------------------------------------------------------
 */

inline BOOL DriveIsCDROM(UINT uiDrive, CDInfo *pCDInfo)
{
    return (uiDrive >= pCDInfo->usFirst)
        && (uiDrive < (pCDInfo->usFirst + pCDInfo->usCount));
}

/***************************************************************************************
 *
 * DrivesCacheImpl - Implementation
 *
 ***************************************************************************************
 */

/*------------------------------------------------------------------------------------
 * Constructor
 *------------------------------------------------------------------------------------
 */

DrivesCacheImpl::DrivesCacheImpl()
{
    update();
}

/*------------------------------------------------------------------------------------
 * update()
 *------------------------------------------------------------------------------------
 */

void DrivesCacheImpl::update()
{
    // reset disk change indicator
//    m_ulDiskChanged = 0;

    // update drive map
    m_ulCurrentDrive = 0;
    m_ulDriveMap     = 0;
    DosQueryCurrentDisk (&m_ulCurrentDrive, &m_ulDriveMap);

    // reset current drive 
    m_ulCurrentDrive = 0;

    // query system for CD drives
    CDInfo cdInfo;
    GetCDInfo(&cdInfo);

    // query number of floppies
    BYTE nFloppy = 0;
    DosDevConfig((PVOID) &nFloppy, DEVINFO_FLOPPY);

    // remove phantom floppy
    if(nFloppy < 2)
        m_ulDriveMap &= (ULONG) (nFloppy-4);

    // refill cache with actual information
    for(UINT uiDrive = 0; uiDrive < 26; uiDrive++)
        if((m_ulDriveMap >> uiDrive) & 1)
        {
            DTInfo dtInfo;

            // reset cache values
            m_aCache[uiDrive].bIsRemovable = FALSE;
            m_aCache[uiDrive].aFileSystemName.Erase();
            m_aCache[uiDrive].aVolumeName.Erase();
            m_aCache[uiDrive].aMappingName.Erase();
            
            if(GetDTInfo(uiDrive, &dtInfo))
            {
                // query if it is a floppy device
                if(uiDrive < nFloppy)
                {
                    m_aCache[uiDrive].aFileSystemName = "FAT";
                    m_aCache[uiDrive].bIsRemovable    = TRUE;
                }
                
                // query if drive is a CDROM
                if(DriveIsCDROM(uiDrive, &cdInfo))
                {
                    m_aCache[uiDrive].aVolumeKind  = VOLUME_KIND_CDROM;
                    m_aCache[uiDrive].bIsRemovable = TRUE;
                }

                else
                {
                    switch (dtInfo.ulDevType)
                    {
                    case DEVTYPE_35INCH_1MB_DISKETTE:
                    case DEVTYPE_35INCH_4MB_DISKETTE:
                    case DEVTYPE_35INCH_2MB_DISKETTE:
                        m_aCache[uiDrive].aVolumeKind = VOLUME_KIND_FLOPPY_35;
                        
                        break;
                    case DEVTYPE_48TPI_DISKETTE:
                    case DEVTYPE_96TPI_DISKETTE:
                    case DEVTYPE_8INCH_SD_DISKETTE:
                    case DEVTYPE_8INCH_DD_DISKETTE:
                        m_aCache[uiDrive].aVolumeKind = VOLUME_KIND_FLOPPY_525;
                        
                        break;
                    case DEVTYPE_FIXED_DISK:
                        m_aCache[uiDrive].aVolumeKind = VOLUME_KIND_FIXED;

                        {
                            FSInfo fsInfo;

                            // query filesystem information
                            if(GetFSInfo(uiDrive, &fsInfo))
                            {
                                // save filesystem name
                                m_aCache[uiDrive].aFileSystemName = (const char *) &fsInfo.fsq.szFSDName[fsInfo.fsq.cbName];

                                // determine if local or remote drive
                                if(fsInfo.fsq.iType == FSAT_REMOTEDRV)
                                {
                                    if(m_aCache[uiDrive].aFileSystemName == "RAMFS")
                                    {
                                        m_aCache[uiDrive].aVolumeKind = VOLUME_KIND_RAM;
                                    }
                                    else
                                    {
                                        m_aCache[uiDrive].aVolumeKind = VOLUME_KIND_REMOTE;
                                        m_aCache[uiDrive].aMappingName = (const char *) &fsInfo.fsq.rgFSAData[fsInfo.fsq.cbName + fsInfo.fsq.cbFSDName];
                                    }
                                }
                            }
                        }

                        break;
                    default:
                    case DEVTYPE_TAPE_DRIVE:
                    case DEVTYPE_OPTICAL:
                        
                        if(dtInfo.bRemovable)
                        {
                            m_aCache[uiDrive].bIsRemovable = TRUE;
                            m_aCache[uiDrive].aVolumeKind  = VOLUME_KIND_REMOVABLE;
                        }
                        else
                        {
                            m_aCache[uiDrive].aVolumeKind  = VOLUME_KIND_FIXED;
                        }

                        break;
                    case DEVTYPE_OTHER:
                        if(dtInfo.bRemovable)
                        {
                            m_aCache[uiDrive].bIsRemovable = TRUE;
                            m_aCache[uiDrive].aVolumeKind  = VOLUME_KIND_REMOVABLE;
                        }
                        else
                        {
                            m_aCache[uiDrive].aVolumeKind  = VOLUME_KIND_RAM;
                        }

                        break;
                    }
                }

                // just query if disk has changed when removable drive
                if(m_aCache[uiDrive].bIsRemovable)
                {
#if 0
                    if(dtInfo.bChanged)
                    {
                        m_ulDiskChanged |= (1 << uiDrive);
                    }
#endif
                }
                else
                {
                    FSInfo fsInfo;
                    FSINFO VolumeInfo;

                    // query filesystem information if not already there
                    if(dtInfo.ulDevType != DEVTYPE_FIXED_DISK && GetFSInfo(uiDrive, &fsInfo))
                    {
                        // save filesystem name
                        m_aCache[uiDrive].aFileSystemName = (const char *) &fsInfo.fsq.szFSDName[fsInfo.fsq.cbName];
                    }

                    // query volume name for local drives
                    if(NO_ERROR == DosQueryFSInfo( uiDrive + 1, FSIL_VOLSER,
                                                   &VolumeInfo, sizeof (VolumeInfo)))
                    {
                        m_aCache[uiDrive].aVolumeName = (const char *) &VolumeInfo.vol.szVolLabel;
                    }
                }
            }
        }
}

/*------------------------------------------------------------------------------------
 * buildCRC()
 *------------------------------------------------------------------------------------
 */

UINT32 DrivesCacheImpl::buildCRC()
{
    ULONG  ulCurrentDrive = 0;
    ULONG  ulDriveMap     = 0;
    BYTE   nFloppy        = 0;
    UINT32 nCRC;

    // query actual drive map
    DosQueryCurrentDisk (&m_ulCurrentDrive, &m_ulDriveMap);

    // query number of floppies
    DosDevConfig((PVOID) &nFloppy, DEVINFO_FLOPPY);

    // remove phantom floppy
    if(nFloppy < 2)
        ulDriveMap &= (ULONG) (nFloppy-4);

    // initialize CRC with drive map
    nCRC = ulDriveMap;

    // iterate over drives
    for(UINT uiDrive = 0; uiDrive < 26; uiDrive++)
        if((ulDriveMap >> uiDrive) & 1)
        {
            DTInfo dtInfo;

            if(GetDTInfo(uiDrive, &dtInfo))
            {
                // query filesystem and volume label for fixed and network drives.
                // use cached information here, so there are certain cases where
                // removable drives could be touched when hotplugged.
                if(!m_aCache[uiDrive].bIsRemovable)
                {
                    FSInfo fsInfo;

                    // query filesystem information
                    if(GetFSInfo(uiDrive, &fsInfo))
                    {
                        UINT32 *pInt32 = (UINT32 *) &fsInfo.fsq.szFSDName[fsInfo.fsq.cbName];
                        UINT32 *pLast  = (UINT32 *) ((UCHAR *) pInt32 + fsInfo.fsq.cbFSDName - 4);

                        // build a crc over filesystem name
                        for(; pInt32 < pLast; pInt32++)
                            nCRC ^= *pInt32;

                        // don't forget last item
                        nCRC ^= *pLast;

                        // for local drives build crc over volume names
                        if(fsInfo.fsq.iType == FSAT_LOCALDRV ||
                           strcmp((const char *) &fsInfo.fsq.szFSDName[fsInfo.fsq.cbName], "RAMFS") == 0)
                        {
                            FSINFO VolumeInfo = {0};

                            if(NO_ERROR == DosQueryFSInfo( uiDrive + 1, FSIL_VOLSER,
                                                           &VolumeInfo, sizeof (VolumeInfo)))
                            {
                                pInt32 = (UINT32 *) &VolumeInfo.vol.szVolLabel;
                                pLast  = (UINT32 *) ((UCHAR *) pInt32 + VolumeInfo.vol.cch - 4);
                            }
                        }
                        // for remote drives, build crc over mapping names
                        else
                        {
                            pInt32 = (UINT32 *) ((UCHAR *) pInt32 + 1);
                            pLast  = (UINT32 *) ((UCHAR *) pInt32 + fsInfo.fsq.cbFSAData - 4);
                        }

                        // build the crc
                        for(; pInt32 < pLast; pInt32++)
                            nCRC ^= *pInt32;

                        // don't forget last item
                        nCRC ^= *pLast;
                    }
                }
            }
        }

    return nCRC;
}

/***************************************************************************************
 *
 * DrivesFolderImpl - Implementation
 *
 ***************************************************************************************
 */

/*------------------------------------------------------------------------------------
 * Constructor
 *------------------------------------------------------------------------------------
 */

DrivesFolderImpl::DrivesFolderImpl(const ItemIDPath& rPath) : FolderImpl(rPath)
{
    m_nCurrentDrive = 0;

    if(!m_pCache)
    {
        m_pCache = new DrivesCacheImpl();
        DosExitList(EXLST_ADD | 0x00004200, (PFNEXITLIST) DrivesFolderImpl::DestroyCache);
    }
}

/*------------------------------------------------------------------------------------
 * DestroyCache()
 *------------------------------------------------------------------------------------
 */

DrivesCacheImpl* DrivesFolderImpl::m_pCache = 0;

void DrivesFolderImpl::EnsureCache()
{
    if(!m_pCache)
    {
        m_pCache = new DrivesCacheImpl();
        DosExitList(EXLST_ADD | 0x00004200, (PFNEXITLIST) DrivesFolderImpl::DestroyCache);
    }
}

void DrivesFolderImpl::DestroyCache()
{
    if(m_pCache)
    {
        delete m_pCache;
        m_pCache = 0;
    }
}

/*------------------------------------------------------------------------------------
 * GetNextItem()
 *------------------------------------------------------------------------------------
 */

BYTE DrivesFolderImpl::GetNextItem( ItemIDPathData& rData )
{
    BYTE nRet = FOLDER_NO_MORE_ITEMS;

    // check if called the first time
    if(m_nCurrentDrive == 0)
        m_pCache->update();
        
    // enumerat drives only
    for( ; m_nCurrentDrive < 26; m_nCurrentDrive++ )
        if( m_pCache->isValidDrive(m_nCurrentDrive))
        {
            rData.sPath = (char) (cFirstDriveLetter + m_nCurrentDrive++);
            rData.sPath += ':';
            rData.nImplFlags |= ITEM_IMPL_FLAG_VOL_INFO;

            break;
        }

    if(rData.nImplFlags & ITEM_IMPL_FLAG_VOL_INFO)
        nRet = FOLDER_OK;

    return nRet;
}

/*------------------------------------------------------------------------------------
 * RestartEnum()
 *------------------------------------------------------------------------------------
 */

BOOL DrivesFolderImpl::RestartEnum( const ULONG aFlags )
{
    m_nCurrentDrive = 0;

    return TRUE;
}

/*------------------------------------------------------------------------------------
 * BuildCRC()
 *------------------------------------------------------------------------------------
 */

UINT32 DrivesFolderImpl::BuildCRC( const ULONG aFlags, BOOL bBreak )
{
    return m_pCache->buildCRC();
}

/*------------------------------------------------------------------------------------
 * run()
 *------------------------------------------------------------------------------------
 */

void DrivesFolderImpl::run()
{
    static TimeValue aSleepTime = { 5, 0 };
    UINT32 n = m_pCache->buildCRC();

    wait(aSleepTime);

    while(schedule())
    {
        UINT32 nTmp = m_pCache->buildCRC();

        // check CRC and disk change indicator
//        if(nTmp != n || m_pCache->diskChanged(DRIVE_ALL))
        if(nTmp != n )
        {
            n = nTmp;
            getLink().Call(NULL);
        }

        wait(aSleepTime);
    }
}

/*------------------------------------------------------------------------------------
 * GetItemIDInfo()
 *------------------------------------------------------------------------------------
 */

BOOL DrivesFolderImpl::GetItemIDInfo(const ItemIDPath& rPath, FastItemInfo& rInfo)
{
    ItemIDPathData *pData = (PItemIDPathData) rPath.GetDataPtr();

    if(pData == NULL || pData->sPath.GetChar(1) != ':')
        return FALSE;

    if(!(pData->nImplFlags & ITEM_IMPL_FLAG_VOL_INFO))
        pData->nImplFlags |= ITEM_IMPL_FLAG_VOL_INFO;

    // set info values
    rInfo.aItemKind = ITEM_KIND_VOLUME;
    rInfo.aAttributes =
        ITEM_FLAG_ISFOLDER	|
        ITEM_FLAG_HASSUBFOLDERS |
        ITEM_FLAG_ISFILESYSTEM |
        ITEM_FLAG_HASITEMS;

    return TRUE;
}

/*------------------------------------------------------------------------------------
 * GetVolumeInfo()
 *------------------------------------------------------------------------------------
 */

BOOL DrivesFolderImpl::GetVolumeInfo(const String& aPath, VolumeInfo& rInfo)
{
    UCHAR szDrive[4] = "a:\\";
    UINT  uiDrive;

    // make sure cache exists
    EnsureCache();

    // extract drive letter
    szDrive[0] = tolower(aPath.GetChar(0));
    uiDrive    = (UINT) (szDrive[0] - cFirstDriveLetter);

    // make sure drive letter is valid
    if(szDrive[0] >= cFirstDriveLetter &&
       szDrive[0] <= cLastDriveLetter &&
       m_pCache->isValidDrive(uiDrive))
    {

        rInfo.aDeviceName     = (const char *) szDrive;
        rInfo.aFileSystemName = m_pCache->getFileSystemName(uiDrive);
        rInfo.aVolumeKind     = m_pCache->getVolumeKind(uiDrive);
        rInfo.aVolumeName     = m_pCache->getVolumeName(uiDrive);
        rInfo.aMappingName    = m_pCache->getMappingName(uiDrive);

        rInfo.aDeviceName.ToUpper();

        if( rInfo.aFileSystemName.Compare( "FAT" ) == 0 )
        {
            rInfo.nMaxFileNameLength = 12;
            rInfo.nMaxFilePathLength = 66;
        }
        else
        {
            rInfo.nMaxFileNameLength = CCHMAXPATHCOMP;
            rInfo.nMaxFilePathLength = CCHMAXPATHCOMP;
        }

        return TRUE;
    }

    return FALSE;
}

/*------------------------------------------------------------------------------------
 * GetVolumeKind()
 *------------------------------------------------------------------------------------
 */

BOOL DrivesFolderImpl::GetVolumeKind(const String& rPath, VolumeKind& rVolumeKind)
{
    BOOL bRet = FALSE;

    // make sure cache exists
    EnsureCache();

    // check if UNC notation
    if(rPath.GetChar(0) == '\\' && rPath.GetChar(1) == '\\' && rPath.GetTokenCount('\\') >= 4)
    {
        rVolumeKind = VOLUME_KIND_REMOTE;
        bRet = TRUE;
    }
    else if (rPath.GetChar(1) == ':' &&
            (rPath.GetChar(2) == '\\' || rPath.GetChar(2) == '\0'))
    {
        // extract drive letter
        char c = tolower(rPath.GetChar(0));
        UINT uiDrive = (UINT) (c - cFirstDriveLetter);

        // make sure drive letter is valid
        if(c >= cFirstDriveLetter &&
           c <= cLastDriveLetter  &&
           m_pCache->isValidDrive(uiDrive))
        {
            rVolumeKind = m_pCache->getVolumeKind(uiDrive);
            bRet = TRUE;
        }
    }

    return bRet;
}

/*------------------------------------------------------------------------------------
 * GetFileSystemName()
 *------------------------------------------------------------------------------------
 */

BOOL DrivesFolderImpl::GetFileSystemName(const String& rPath, String& rFileSystemName)
{
    BOOL bRet = FALSE;

    // make sure cache exists
    EnsureCache();

    // check if valid UNC path
    if(rPath.GetChar(0) == '\\' && rPath.GetChar(1) == '\\' && rPath.GetTokenCount('\\') >= 4)
    {
        FSInfo fsInfo;
        ULONG  ulBufLen = sizeof(FSInfo);
        USHORT nIndex = 0;

        for(BYTE n = 0; n < 4; n++)
        {
            nIndex = rPath.Search('\\', nIndex);
        }

        String aTmp = rPath.Copy(0, nIndex);

        // query filesystem information
        memset(&fsInfo, 0, (size_t)ulBufLen );
        if(NO_ERROR == DosQueryFSAttach((const char *) aTmp , 0, FSAIL_QUERYNAME, &fsInfo.fsq, &ulBufLen))
        {
            rFileSystemName = (const char *) &fsInfo.fsq.szFSDName[fsInfo.fsq.cbName];
            bRet = TRUE;
        }
    }

    // check if valid drive information
    else if(rPath.GetChar(1) == ':' && rPath.GetChar(2) == '\\')
    {
        // extract drive letter
        char c = tolower(rPath.GetChar(0));
        UINT uiDrive = (UINT) (c - cFirstDriveLetter);

        // make sure drive letter is valid
        if(c >= cFirstDriveLetter &&
           c <= cLastDriveLetter  &&
           m_pCache->isValidDrive(uiDrive))
        {
            rFileSystemName = m_pCache->getFileSystemName(uiDrive);
            bRet = TRUE;
        }
    }

    return bRet;
}
