/*************************************************************************
 *
 *  $RCSfile: ico2xpm.c,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: svesik $ $Date: 2002/08/28 12:13:27 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdlib.h>
#include <stdio.h>

#include <windows.h>

/************************************************************************
 * Usage
 ************************************************************************/

void Usage() 
{
	printf( "Usage: ico2xpm [-o <outpath>] icofile" );
	exit(-1);
}

/************************************************************************
 * LoadIconFromFile
 ************************************************************************/

int LoadIconFromFile( const char * pIcoFile, int width, int height, PICONINFO pInfo )
{
	HANDLE hImage = NULL;
	HDC hdc = GetDC( GetDesktopWindow() );

	/* load image from icon file */
	hImage = LoadImage(NULL, pIcoFile, IMAGE_ICON, width, height, LR_LOADFROMFILE | LR_VGACOLOR);

	/* file could not be opened */
	if(hImage == NULL)
	{
		PVOID * lpMsgBuf = NULL;;

		FormatMessage( 
			FORMAT_MESSAGE_ALLOCATE_BUFFER |           /* allocated buffer */
			FORMAT_MESSAGE_FROM_SYSTEM |               /* use system error messages */
			FORMAT_MESSAGE_IGNORE_INSERTS,             /* ignore argument array */
			NULL,                                      /* ignored anyway */
			GetLastError(),                            /* error value of LoadLibrary */
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* Default language */
			(LPTSTR) &lpMsgBuf,                        /* store pointer to the allocated buffer here */
			0,                                         /* minimum size of buffer */
			NULL                                       /* argument list - ignored */
			);

		fprintf( stderr, "ico2xpm: %s: %s", pIcoFile, lpMsgBuf);
		LocalFree(lpMsgBuf);

		exit(-1);
	}

	/* examine the returned icon */
	if( GetIconInfo( hImage, pInfo ) )
	{
		BITMAPINFO aInfo;
		int  nBitmaps = 0;

		/* initialize info structure */
		memset( &aInfo, 0, sizeof(BITMAPINFO) );
		aInfo.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);

		/* ensure mask bitmap has the requested size */
		if( GetDIBits( hdc, pInfo->hbmMask, 0, 0, NULL, &aInfo, 0 ) )
		{
			if( aInfo.bmiHeader.biWidth == width && aInfo.bmiHeader.biHeight == height )
				nBitmaps++;
		}

		/* re-initialize info structure */
		memset( &aInfo, 0, sizeof(BITMAPINFO) );
		aInfo.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);

		/* ensure color bitmap has the requested size */
		if( GetDIBits( hdc, pInfo->hbmColor, 0, 0, NULL, &aInfo, 0 ) )
		{
			if( aInfo.bmiHeader.biWidth == width && aInfo.bmiHeader.biHeight== height  )
				nBitmaps++;
		}

		/* free icon resources */
		DestroyIcon( hImage );

		/* check if conditions were met */
		if( nBitmaps == 2 )
		{
			printf("%d x %d icon with 16 colors extracted.\n", 
				width, height, aInfo.bmiHeader.biBitCount);
			return 1;
		}
	}

	return 0;
}

/************************************************************************
 * FileName
 ************************************************************************/

const char * FileExtension( int width, int height )
{
	/* check if width equals height */
	if( width == height )
	{
		/* follow cde naming conventions */
		switch( width )
		{
			case 48:
				return ".l.xpm";
			case 32:
				return ".m.xpm";
			case 24:
				return ".s.xpm";
			case 16:
				return ".t.xpm";
			default:
				;
		}
	}

	return ".xpm";
}


/************************************************************************
 * SaveXPMFile
 ************************************************************************/

int SaveXPMFile( const char * pOutFile, int width, int height, PICONINFO pIconInfo )
{
	PBITMAPINFO pColorInfo;
	PBYTE pColorData;
	HDC hDC;

	int nColorInfoSize = sizeof(BITMAPINFO) + 15 * sizeof(RGBQUAD);
	int ret = 0;

	/* further calls to GetDIBits() need a valid device context */
	hDC = GetDC(GetDesktopWindow());

	/* allocate and initialize info structure for color bitmap */
	pColorInfo = (PBITMAPINFO) calloc(1, nColorInfoSize);
	pColorInfo->bmiHeader.biSize = nColorInfoSize;
	pColorInfo->bmiHeader.biWidth = width;
	pColorInfo->bmiHeader.biHeight = height;
	pColorInfo->bmiHeader.biPlanes = 1;
	pColorInfo->bmiHeader.biBitCount = 4;
	pColorInfo->bmiHeader.biCompression = BI_RGB;

	pColorData = (PBYTE) calloc(width / 2, height );

	if( GetDIBits( hDC, pIconInfo->hbmColor, 0, height, pColorData, pColorInfo, DIB_RGB_COLORS ) )
	{
		PBITMAPINFO pMaskInfo;
		PBYTE pMaskData;

		/* scan lines must be DWORD alligned */
		int nScanLineWidth = ( ( (width - 1) / 8 ) & ~0x03 ) + 4;
		int nMaskInfoSize  = sizeof(BITMAPINFO) + sizeof(RGBQUAD);

		/* allocate and initialize info structure for mask bitmap */
		pMaskInfo = (PBITMAPINFO) calloc(1, nMaskInfoSize);
		pMaskInfo->bmiHeader.biSize = nMaskInfoSize;
		pMaskInfo->bmiHeader.biWidth = width;
		pMaskInfo->bmiHeader.biHeight = height;
		pMaskInfo->bmiHeader.biPlanes = 1;
		pMaskInfo->bmiHeader.biBitCount = 1;
		pMaskInfo->bmiHeader.biCompression = BI_RGB;

		pMaskData = (PBYTE) calloc( nScanLineWidth, height );

		if( GetDIBits( hDC, pIconInfo->hbmMask, 0, height, pMaskData, pMaskInfo, DIB_RGB_COLORS ) )
		{
			const char * pc;
            char * pIcon;
			int i, j;
			FILE * fp;

			/* extract file name from pOutFile */
			if( NULL == ( pc = strrchr(pOutFile, '\\') ) )
				pc = strrchr(pOutFile, '/');

            /* maybe only filename given */
            if( NULL == pc )
                pc = pOutFile;
            else
                ++pc;

            /* set leading '_' if 1st character is a number */
            if( ! isalpha( *pc ) )
            {
                pIcon = (char *) malloc( strlen(pc) + 2 );
                pIcon[0] = '_';
                strcpy( pIcon + 1, pc );
            }
            else
                pIcon = strdup( pc );

            /* replace '.' with '_' */
            for( i = 0; pIcon[i]; i++ )
                if( '.' == pIcon[i] )
                    pIcon[i] = '_';

			/* open output file for writing */
			fp = fopen(pOutFile, "w");

			if( fp )
			{
				fprintf(fp, "/* XPM */\012");
				fprintf(fp, "static char * %s[] = {\012", pIcon );
				fprintf(fp, "/* width height ncolors chars_per_pixel */\012");
				fprintf(fp, "\"%d %d 17 1\",\012", width, height);
				fprintf(fp, "/* colors */\012");
				fprintf(fp, "\"  c None\",\012");

				/* iterate over colors */
				for(i = 0; i < 16; i++)
				{
					fprintf(fp, "\"%c c #%2.2X%2.2X%2.2X\",\012", 
						'a' + i, 
						pColorInfo->bmiColors[i].rgbRed,
						pColorInfo->bmiColors[i].rgbGreen,
						pColorInfo->bmiColors[i].rgbBlue
					);
				}

				fprintf(fp, "/* pixels */\012");

				/* iterate over scan lines */
				for(i = height - 1; i >= 0; i--)
				{
					BYTE mask = 0x80;
						fprintf(fp, "\"");

					/* iterate over pixel in current line */
					for(j = 0; j < width; j++)
					{
						DWORD pixel = i * width + j;
						BYTE  color = pColorData[ pixel / 2 ];
					
						if( pixel % 2 )
							color &= 0x0F;
						else
							color = (color & 0xF0) >> 4;


						fprintf(fp, "%c", pMaskData[i * nScanLineWidth + j / 8] & mask ? ' ' : 'a' + color );

						mask >>= 1;

						if(mask == 0)
							mask = 0x80;
					}

					fprintf(fp, "\",\012");
				}

				fprintf(fp, "};\012");
				fclose(fp);
			}
			else
				fprintf( stderr, "ico2xpm: unable to create output file %s: %s\n", pOutFile, strerror( errno ));

			free(pMaskInfo);
			free(pMaskData);
            free(pIcon);
		}
	}

	free(pColorInfo);
	free(pColorData);

	return ret;
}

/************************************************************************
 * main
 ************************************************************************/

int main( int argc, char * argv[] )
{
	const char * pIcoFile = NULL;
	const char * pOutPath = NULL;
	char * pOutFileExt = NULL;
	char * pOutFile = NULL;

	int IconSizes[] = { 16, 24, 32, 48 };
	int i;

	ICONINFO aInfo;

	/* default behavior without parameter */
	if( argc == 1 )
		Usage();

	/* iterate over command line arguments */
	for( i = 1; i < argc; i++ )
	{
		/* check if outpath is specified */
		if (0 == strncmp(argv[i], "-o", 2))
		{
			if(argc - i >= 2)
				pOutPath = argv[++i];
			else
				Usage();
		}

		/* use last parameter as file name */
		else if (pIcoFile == NULL)
			pIcoFile = argv[i];
		else
			Usage();
	}

	/* create stub for output file names */
	if( pOutPath )
	{
		/* extract file name from path */
		const char * pc = strrchr(pIcoFile, '\\');

		/* also try a '/' if no '\\' found */
		if( pc == NULL )
			pc = strrchr(pIcoFile, '/');

		/* if still no delimiter found use full file name */
		if( pc == NULL )
			pc = pIcoFile;
		
		/* concat outpath and file name - leave space for extension */
		pOutFile = (char *) malloc( strlen(pOutPath) + strlen(pc) + 8 );
		strcpy(pOutFile, pOutPath);
		strcat(pOutFile, "\\");
		strcat(pOutFile, pc);
	}
	else
	{
		/* alloc enough space for extension */
		pOutFile = (char *) malloc( strlen(pIcoFile) + 7 );
		strcpy(pOutFile, pIcoFile);
	}

	/* cut off .ico extension when found */
	{
		char * pc = pOutFile + strlen(pOutFile) - 4;

		if( strnicmp( pc, ".ico", 4) == 0 )
			*pc = '\0';
	}

	/* remember where to place extension */
	pOutFileExt = pOutFile + strlen(pOutFile);

	/* extract all icon sizes */
	for( i = 0; i < 4; i++)
	{
		if(LoadIconFromFile( pIcoFile, IconSizes[i], IconSizes[i], &aInfo ))
		{
			strcpy(pOutFileExt, FileExtension( IconSizes[i], IconSizes[i] ));
			SaveXPMFile(pOutFile, IconSizes[i], IconSizes[i], &aInfo);
		}
	}
	
	free( pOutFile );
	return 0;
}
