/*************************************************************************
 *
 *  $RCSfile: argumentanalyzer.cxx,v $
 *
 *  $Revision: 1.33 $
 *
 *  last change: $Author: vg $ $Date: 2003/04/17 16:14:16 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

//_________________________________________________________________________________________________________________
//	my own includes
//_________________________________________________________________________________________________________________

#ifndef __FRAMEWORK_CLASSES_ARGUMENTANALYZER_HXX_
#include <classes/argumentanalyzer.hxx>
#endif

#ifndef __FRAMEWORK_CLASSES_FILTERCACHE_HXX_
#include <classes/filtercache.hxx>
#endif

#ifndef __FRAMEWORK_TARGETS_H_
#include <targets.h>
#endif

//_________________________________________________________________________________________________________________
//	interface includes
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	other includes
//_________________________________________________________________________________________________________________

#ifndef _URLOBJ_HXX
#include <tools/urlobj.hxx>
#endif

//_________________________________________________________________________________________________________________
//	const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	namespace
//_________________________________________________________________________________________________________________

namespace framework{

//_________________________________________________________________________________________________________________
//	non exported const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	non exported declarations
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	definitions
//_________________________________________________________________________________________________________________

/*-************************************************************************************************************//**
    @short      set argument list for analyzing
    @descr      It set the list of arguments internaly and analyze the structure.
                Of course it's possible to set another list later using setArguments() or get it back using getArguments..().

    @seealso    methods setArguments()
    @seealso    enum EArgument

    @param      rArguments
                    the list of arguments, which should be referenced

    @param      bReadOnly
                    setArgument() calls will be rejected later
*//*-*************************************************************************************************************/
ArgumentAnalyzer::ArgumentAnalyzer()
{
    // It initialize this analyzer with an empty list and set it to writeable.
    // So the outside code can use it. But he starts from scratch!
    setArguments(css::uno::Sequence< css::beans::PropertyValue >(), sal_False);
}

//*****************************************************************************************************************
ArgumentAnalyzer::ArgumentAnalyzer( const css::uno::Sequence< css::beans::PropertyValue >& rArguments ,
                                          sal_Bool                                         bReadOnly  )
{
    setArguments(rArguments,bReadOnly);
}

/*-************************************************************************************************************//**
    @short      free all used ressource
    @descr      Here we forget all internaly used ressources and die ...
*//*-*************************************************************************************************************/
ArgumentAnalyzer::~ArgumentAnalyzer()
{
    m_aAnalyzed.forgetIt();
    m_lArgs     = css::uno::Sequence< css::beans::PropertyValue >();
    m_bReadOnly = sal_False;
    m_bValid    = sal_False;
}

/*-************************************************************************************************************//**
    @short      set argument list to analyze
    @descr      Normaly you give us a argument list at creation time using the ctor ...
                but sometimes it's neccessary to set a new list during runtime.
                Then you can use this set methods.

    @seealso    constructor

    @param      rArguments
                    the list of arguments, which should be referenced

    @param      bReadOnly
                    setArgument() calls will be rejected later
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::setArguments( const css::uno::Sequence< css::beans::PropertyValue >& rArguments ,
                                           sal_Bool                                         bReadOnly  )
{
    m_aAnalyzed.forgetIt();

    m_lArgs     = rArguments;
    m_bReadOnly = bReadOnly ;
    m_bValid    = sal_True  ;

    // Before we analyze the given list - we have to correct
    // some inconsistent (deprecated) arguments! It works on our internal saved
    // list (available on member m_pArguments) and change it. So it must be valid here ...
    impl_correctOldFormat();
    // The we can analyze it and update our internal structures
    impl_analyzeIt();
}

/*-************************************************************************************************************//**
    @short      returns a const reference to the internal saved argument list
    @descr      This reference can be used as direct parameter of methods, which requires a const reference only.
                It will not change the internal structure and we can work on this list later too without parsing it again.

    @return     const css::uno::Sequence< css::beans::PropertyValue >&
                    a const reference to the internal saved argument list
*//*-*************************************************************************************************************/
const css::uno::Sequence< css::beans::PropertyValue >& ArgumentAnalyzer::getArgumentsConst() const
{
    return m_lArgs;
}

/*-************************************************************************************************************//**
    @short      returns a copy of the internal saved argument list
    @descr      It returns a copy of our internal saved argument list and clear all internal structures.
                This instance of the ArgumentAnalyzer can work later only, if you set a new list.
                This mode has to be used to call methods, which requires a writable argument list!

    @return     css::uno::Sequence< css::beans::PropertyValue >
                    a copy of the internal saved argument list
*//*-*************************************************************************************************************/
css::uno::Sequence< css::beans::PropertyValue > ArgumentAnalyzer::getArgumentsAndReset()
{
    css::uno::Sequence< css::beans::PropertyValue > lCopy(m_lArgs);

    m_aAnalyzed.forgetIt();
    m_lArgs     = css::uno::Sequence< css::beans::PropertyValue >();
    m_bReadOnly = sal_False;
    m_bValid    = sal_False;

    return lCopy;
}

/*-************************************************************************************************************//**
    @short      here we analyze the given list realy
    @descr      We step over our internal save argument list m_lArgs and update our index values inside
                m_aAnalyzed member. So all these index values points to an item inside the original sequence.
                We don't use pointer - we use index values. So we can move items easiear!

    @seealso    setArguments()
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::impl_analyzeIt()
{
    m_aAnalyzed.forgetIt();

    sal_uInt32                        nCount = m_lArgs.getLength();
    const css::beans::PropertyValue*  pItem  = m_lArgs.getConstArray() ;
    for( sal_uInt32 nItem=0; nItem<nCount; ++nItem, ++pItem )
	{
		//---------------------------------------------------------------------------------------------------------
        if( pItem->Name == ARGUMENTNAME_CHARACTERSET )
            m_aAnalyzed.psCharacterSet = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_MEDIATYPE )
            m_aAnalyzed.psMediaType = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_DETECTSERVICE )
            m_aAnalyzed.psDetectService = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_EXTENSION )
            m_aAnalyzed.psExtension = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_URL )
            m_aAnalyzed.psURL = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_FILTERNAME )
            m_aAnalyzed.psFilterName = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_TYPENAME )
            m_aAnalyzed.psTypeName = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_FORMAT )
            m_aAnalyzed.psFormat = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_FRAMENAME )
            m_aAnalyzed.psFrameName = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_PATTERN )
            m_aAnalyzed.psPattern = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_POSTDATA )
            m_aAnalyzed.pxPostData = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_POSTSTRING )
            m_aAnalyzed.psPostString = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_REFERRER )
            m_aAnalyzed.psReferrer = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_TEMPLATENAME )
            m_aAnalyzed.psTemplateName = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_TEMPLATEREGIONNAME )
            m_aAnalyzed.psTemplateRegionName = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_VERSION )
            m_aAnalyzed.pnVersion = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_VIEWID )
            m_aAnalyzed.pnViewId = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_FILTEROPTIONS )
            m_aAnalyzed.pnFilterOptions = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_ASTEMPLATE )
            m_aAnalyzed.pbAsTemplate = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_HIDDEN )
            m_aAnalyzed.pbHidden = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_OPENNEWVIEW )
            m_aAnalyzed.pbOpenNewView = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_READONLY )
            m_aAnalyzed.pbReadOnly = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_PREVIEW )
            m_aAnalyzed.pbPreview = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_SILENT )
            m_aAnalyzed.pbSilent = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_MINIMIZED )
            m_aAnalyzed.pbMinimized = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_DEEPDETECTION )
            m_aAnalyzed.pbDeepDetection = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_POSSIZE )
            m_aAnalyzed.paPosSize = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_INPUTSTREAM )
            m_aAnalyzed.pxInputStream = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_OUTPUTSTREAM )
            m_aAnalyzed.pxOutputStream = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_STATUSINDICATOR )
            m_aAnalyzed.pxStatusIndicator = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_INTERACTIONHANDLER )
            m_aAnalyzed.pxInteractionHandler = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_JUMPMARK )
            m_aAnalyzed.psJumpMark = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_MACROEXECUTIONMODE )
            m_aAnalyzed.pnMacroExecutionMode = nItem;
        else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_UPDATEDOCMODE )
            m_aAnalyzed.pnUpdateDocMode = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_REPAIRPACKAGE )
            m_aAnalyzed.pbRepairPackage = nItem;
		else
		//---------------------------------------------------------------------------------------------------------
        if ( pItem->Name == ARGUMENTNAME_DOCUMENTTITLE )
            m_aAnalyzed.psDocumentTitle = nItem;

    }

	// We must analyze a possible set URL again. May the user forgot to
	// set the JumpMark property. Then we do it here ...
    // But suppress it for readonly mode !
    if (
        ( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST ) &&
        ( ! m_bReadOnly                           )
       )
	{
        // Don't set the jumpmark here .. because extraction of it is part
        // of functionality to set the URL too! So we have to set the URL again to get the jumpmark.
        ::rtl::OUString sURL;
        if (m_lArgs[m_aAnalyzed.psURL].Value >>= sURL)
            impl_setArgument(E_URL,(void*)&sURL);
	}
}

/*-************************************************************************************************************//**
    @short      get valid state and value of an argument
    @descr      After calling setArguments() you can call this method to get information about the content
                of set list. You give us a pointer to a valid memory position (of course the type behind such pointer
                must match the value type of the property you requested) and we fill it with the value
                if search item exist. We return the valid state too. If we return FALSE your given pointer wasn't used!

    @param      eArgument
                    specify, which argument is searched

    @param      pValue
                    reference to user memory, which should be filled with right value, if argument exist

    @return     <TRUE/>, if argument could be found and was set on the given pointer
                <FALSE/> otherwise.
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::impl_getArgument( EArgument eArgument, void* pValue ) const
{
    if (pValue==NULL)
        return sal_False;

    sal_Bool bValid = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   if( m_aAnalyzed.psCharacterSet != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psCharacterSet].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   if( m_aAnalyzed.psMediaType != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psMediaType].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   if( m_aAnalyzed.psDetectService != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psDetectService].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   if( m_aAnalyzed.psExtension != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psExtension].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   if( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psURL].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   if( m_aAnalyzed.psFilterName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psFilterName].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   if( m_aAnalyzed.psFormat != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psFormat].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   if( m_aAnalyzed.psFrameName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psFrameName].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   if( m_aAnalyzed.psPattern != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psPattern].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   if( m_aAnalyzed.pxPostData != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.pxPostData].Value >>= *((css::uno::Reference< css::io::XInputStream >*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   if( m_aAnalyzed.psPostString != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psPostString].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   if( m_aAnalyzed.psReferrer != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psReferrer].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   if( m_aAnalyzed.psTemplateName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psTemplateName].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   if( m_aAnalyzed.psTemplateRegionName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psTemplateRegionName].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   if( m_aAnalyzed.psJumpMark != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psJumpMark].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   if( m_aAnalyzed.psTypeName != ARGUMENT_NOT_EXIST )
										{
                                            bValid = ( m_lArgs[m_aAnalyzed.psTypeName].Value >>= *((::rtl::OUString*)pValue) );
										}
										break;
        //_________________________________________________________________________________________________________
        case E_VERSION              :   if( m_aAnalyzed.pnVersion != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pnVersion].Value >>= *((sal_Int16*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_VIEWID               :   if( m_aAnalyzed.pnViewId != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pnViewId].Value >>= *((sal_Int16*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_MACROEXECUTIONMODE   :   if( m_aAnalyzed.pnMacroExecutionMode != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pnMacroExecutionMode].Value >>= *((sal_Int16*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_UPDATEDOCMODE        :   if( m_aAnalyzed.pnUpdateDocMode != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pnUpdateDocMode].Value >>= *((sal_Int16*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_FILTEROPTIONS        :   if( m_aAnalyzed.pnFilterOptions != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pnFilterOptions].Value >>= *((::rtl::OUString*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_ASTEMPLATE           :   if( m_aAnalyzed.pbAsTemplate != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbAsTemplate].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_HIDDEN               :   if( m_aAnalyzed.pbHidden != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbHidden].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_OPENNEWVIEW          :   if( m_aAnalyzed.pbOpenNewView != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbOpenNewView].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_READONLY             :   if( m_aAnalyzed.pbReadOnly != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbReadOnly].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_PREVIEW              :   if( m_aAnalyzed.pbPreview != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbPreview].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_SILENT               :   if( m_aAnalyzed.pbSilent != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbSilent].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_MINIMIZED            :   if( m_aAnalyzed.pbMinimized != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbMinimized].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_DEEPDETECTION        :   if( m_aAnalyzed.pbDeepDetection != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbDeepDetection].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_POSSIZE              :   if( m_aAnalyzed.paPosSize != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.paPosSize].Value >>= *((css::awt::Rectangle*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_INPUTSTREAM          :   if( m_aAnalyzed.pxInputStream != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pxInputStream].Value >>= *((css::uno::Reference< css::io::XInputStream >*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM         :   if( m_aAnalyzed.pxOutputStream != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pxOutputStream].Value >>= *((css::uno::Reference< css::io::XOutputStream >*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_STATUSINDICATOR      :   if( m_aAnalyzed.pxStatusIndicator != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pxStatusIndicator].Value >>= *((css::uno::Reference< css::task::XStatusIndicator >*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER   :   if( m_aAnalyzed.pxInteractionHandler != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pxInteractionHandler].Value >>= *((css::uno::Reference< css::task::XInteractionHandler >*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_REPAIRPACKAGE        :   if( m_aAnalyzed.pbRepairPackage != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.pbRepairPackage].Value >>= *((sal_Bool*)pValue) );
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_DOCUMENTTITLE        :   if( m_aAnalyzed.psDocumentTitle != ARGUMENT_NOT_EXIST )
                                        {
                                            bValid = ( m_lArgs[m_aAnalyzed.psDocumentTitle].Value >>= *((::rtl::OUString*)pValue) );
                                        }
                                        break;

	}

	return bValid;
}

/*-************************************************************************************************************//**
    @short      specialized way to get the URL in structured util.URL notation
    @descr      Normaly the URL inside the argument list is set as string. But sometimes
                our user need it in the special structured notation. Then we convert it automaticly here.

    @param      eArgument
                    must be E_URL (all others will be ignored and return <FALSE/>!)

    @param      aValue
                    reference to user memory, which should be filled with right value, if argument exist

    @return     <TRUE/>, if argument could be found and was set on the given pointer
                <FALSE/> otherwise.
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::util::URL& aValue )
{
    if ( eArgument != E_URL )
        return sal_False;

	sal_Bool bValid = sal_False;

    if( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST )
    {
        ::rtl::OUString sURL;
        m_lArgs[m_aAnalyzed.psURL].Value >>= sURL;

        // It's not neccessary to use the Jumpmark property of the argument list here too.
        // Because it's defined, that the URL must include it!
        INetURLObject aParser(sURL);

        aValue.Protocol   = INetURLObject::GetScheme( aParser.GetProtocol() );
        aValue.User       = aParser.GetUser   ( INetURLObject::DECODE_WITH_CHARSET );
        aValue.Password   = aParser.GetPass   ( INetURLObject::DECODE_WITH_CHARSET );
        aValue.Server     = aParser.GetHost   ( INetURLObject::DECODE_WITH_CHARSET );
        aValue.Port       = (sal_Int16)aParser.GetPort();
        aValue.Path       = aParser.GetURLPath( INetURLObject::NO_DECODE           );
        aValue.Arguments  = aParser.GetParam  ( INetURLObject::NO_DECODE           );
        aValue.Mark       = aParser.GetMark   ( INetURLObject::DECODE_WITH_CHARSET );

        // Jumpmark was already set inside the structure
        // But the main part must be set without the optional things -like mark or arguments
        // So we have to delete it here.
        aParser.SetMark ( ::rtl::OUString() );
        aParser.SetParam( ::rtl::OUString() );

        aValue.Main       = aParser.GetMainURL( INetURLObject::NO_DECODE           );

        bValid = sal_True;
    }

	return bValid;
}

/*-************************************************************************************************************//**
    @short      get/set method with the possibility to check for right type
    @descr      We should prevent the internal MediaDescriptor to be invalid. So we have to check
                the right type for every set/get request and reject wrong ones.
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, ::rtl::OUString& sValue )
{
    if (
        eArgument==E_CHARACTERSET       ||
        eArgument==E_DETECTSERVICE      ||
        eArgument==E_EXTENSION          ||
        eArgument==E_FILTERNAME         ||
        eArgument==E_FILTEROPTIONS      ||
        eArgument==E_FORMAT             ||
        eArgument==E_FRAMENAME          ||
        eArgument==E_JUMPMARK           ||
        eArgument==E_MEDIATYPE          ||
        eArgument==E_PATTERN            ||
        eArgument==E_POSTSTRING         ||
        eArgument==E_REFERRER           ||
        eArgument==E_TEMPLATENAME       ||
        eArgument==E_TEMPLATEREGIONNAME ||
        eArgument==E_TYPENAME           ||
        eArgument==E_URL                ||
        eArgument==E_DOCUMENTTITLE
       )
    {
        return impl_getArgument(eArgument,(void*)&sValue);
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(string)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, sal_Int16& nValue )
{
    if (
        eArgument==E_MACROEXECUTIONMODE ||
        eArgument==E_UPDATEDOCMODE      ||
        eArgument==E_VERSION            ||
        eArgument==E_VIEWID
       )
    {
        return impl_getArgument(eArgument,(void*)&nValue);
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(int16)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, sal_Int32& nValue )
{
    LOG_WARNING("ArgumentAnalyzer::getArgument(int32)","Currently there is no property which requires long type for value! Please check it ...")
    return sal_False;
}

sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, sal_Bool& bValue )
{
    if (
        eArgument==E_ASTEMPLATE     ||
        eArgument==E_DEEPDETECTION  ||
        eArgument==E_HIDDEN         ||
        eArgument==E_MINIMIZED      ||
        eArgument==E_OPENNEWVIEW    ||
        eArgument==E_PREVIEW        ||
        eArgument==E_READONLY       ||
        eArgument==E_SILENT			||
        eArgument==E_REPAIRPACKAGE
       )
    {
        return impl_getArgument(eArgument,(void*)&bValue);
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(bool)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::awt::Rectangle& aValue )
{
    if (eArgument==E_POSSIZE)
    {
        return impl_getArgument(eArgument,(void*)&aValue);
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(rectangle)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::io::XInputStream >& xValue )
{
    if (
        eArgument==E_POSTDATA    ||
        eArgument==E_INPUTSTREAM
       )
    {
        return impl_getArgument(eArgument,(void*)&xValue);
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(xinputstream)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::io::XOutputStream >& xValue )
{
    if (eArgument==E_OUTPUTSTREAM)
    {
        return impl_getArgument(eArgument,(void*)&xValue);
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(xoutputstream)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::task::XStatusIndicator >& xValue )
{
    if (eArgument==E_STATUSINDICATOR)
    {
        return impl_getArgument(eArgument,(void*)&xValue);
        LOG_WARNING("ArgumentAnalyzer::getArgument(xstatusindicator)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(xstatusindicator)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
sal_Bool ArgumentAnalyzer::getArgument( EArgument eArgument, css::uno::Reference< css::task::XInteractionHandler >& xValue )
{
    if (eArgument==E_INTERACTIONHANDLER)
    {
        return impl_getArgument(eArgument,(void*)&xValue);
    }
    else
    {
        LOG_WARNING("ArgumentAnalyzer::getArgument(xinteractionhandler)","There s a mismatch between your required property and the given value type!")
        return sal_False;
    }
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const ::rtl::OUString& sValue )
{
    if (
        eArgument==E_CHARACTERSET       ||
        eArgument==E_DETECTSERVICE      ||
        eArgument==E_EXTENSION          ||
        eArgument==E_FILTERNAME         ||
        eArgument==E_FILTEROPTIONS      ||
        eArgument==E_FORMAT             ||
        eArgument==E_FRAMENAME          ||
        eArgument==E_JUMPMARK           ||
        eArgument==E_MEDIATYPE          ||
        eArgument==E_PATTERN            ||
        eArgument==E_REFERRER           ||
        eArgument==E_TEMPLATENAME       ||
        eArgument==E_TEMPLATEREGIONNAME ||
        eArgument==E_TYPENAME           ||
        eArgument==E_URL                ||
        eArgument==E_DOCUMENTTITLE
       )
    {
        impl_setArgument(eArgument,(void*)&sValue);
    }
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(string)","There s a mismatch between your required property and the given value type!")
    #endif
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, sal_Int16 nValue )
{
    if (
        eArgument==E_MACROEXECUTIONMODE ||
        eArgument==E_UPDATEDOCMODE      ||
        eArgument==E_VERSION            ||
        eArgument==E_VIEWID
       )
    {
        impl_setArgument(eArgument,(void*)&nValue);
    }
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(int16)","There s a mismatch between your required property and the given value type!")
    #endif
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, sal_Int32 nValue )
{
    LOG_WARNING("ArgumentAnalyzer::setArgument(int32)","Currently there is no property which requires long type for value! Please check it ...")
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, sal_Bool bValue )
{
    if (
        eArgument==E_ASTEMPLATE     ||
        eArgument==E_DEEPDETECTION  ||
        eArgument==E_HIDDEN         ||
        eArgument==E_MINIMIZED      ||
        eArgument==E_OPENNEWVIEW    ||
        eArgument==E_PREVIEW        ||
        eArgument==E_READONLY       ||
        eArgument==E_SILENT			||
        eArgument==E_REPAIRPACKAGE
       )
    {
        impl_setArgument(eArgument,(void*)&bValue);
    }
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(bool)","There s a mismatch between your required property and the given value type!")
    #endif
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::awt::Rectangle& aValue )
{
    if (eArgument==E_POSSIZE)
        impl_setArgument(eArgument,(void*)&aValue);
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(rectangle)","There s a mismatch between your required property and the given value type!")
    #endif
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::io::XInputStream >& xValue )
{
    if (
        eArgument==E_POSTDATA     ||
        eArgument==E_INPUTSTREAM
       )
    {
        impl_setArgument(eArgument,(void*)&xValue);
    }
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(xinputstream)","There s a mismatch between your required property and the given value type!")
    #endif
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::io::XOutputStream >& xValue )
{
    if (eArgument==E_OUTPUTSTREAM)
        impl_setArgument(eArgument,(void*)&xValue);
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(xoutputstream)","There s a mismatch between your required property and the given value type!")
    #endif
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::task::XStatusIndicator >& xValue )
{
    if (eArgument==E_STATUSINDICATOR)
        impl_setArgument(eArgument,(void*)&xValue);
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(xstatusindicator)","There s a mismatch between your required property and the given value type!")
    #endif
}

//**************************************************************************************************
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::uno::Reference< css::task::XInteractionHandler >& xValue )
{
    if (eArgument==E_INTERACTIONHANDLER)
        impl_setArgument(eArgument,(void*)&xValue);
    #ifdef ENABLE_WARNINGS
    else
        LOG_WARNING("ArgumentAnalyzer::setArgument(xinteractionhandler)","There s a mismatch between your required property and the given value type!")
    #endif
}

/*-************************************************************************************************************//**
    @short      change or add a argument to current list
    @descr      Use these methods to add a new argument to current list or set a new value for it.
                You give us a pointer to the memory from where we can copy it. Of course the value
                type must match the required one for given argument.

    @seealso    method getArgument()

    @param      eArgument
                    specify, which argument should be changed or added

    @param      pValue
                    pointer to the new value for this argument
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::impl_setArgument( EArgument eArgument, const void* pValue )
{
    if (m_bReadOnly)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument()", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    LOG_ASSERT( pValue!=NULL, "ArgumentAnalyzer::setArgument(void*)\ninvalid argument, NULL pointer are not allowed here!" )
    if (pValue == NULL)
        return;

    // ppArgument points into the structure m_aAnalyzed to a member, which represent an index into the
    // argument list m_pArguments. So we update an index into the sequence - and of course the sequence itself.
    sal_Int32* ppArgument = NULL;

    // It includes the new value for this argument, which will be set inside the
    // sequence< PropertyValue > we use internaly as m_pArguments
    css::uno::Any aPackedValue;

    // the name of the property, which should be added
    // In case - we change it - this information isn't realy used ...
    // In case we must add a new entry - this will be the name of the new PropertyValue struct.
    ::rtl::OUString sName;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property CharacterSet!" )
                                            ppArgument     = &(m_aAnalyzed.psCharacterSet);
                                            sName          = ARGUMENTNAME_CHARACTERSET;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property MediaType!" )
                                            ppArgument     = &(m_aAnalyzed.psMediaType);
                                            sName          = ARGUMENTNAME_MEDIATYPE;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property DetectService!" )
                                            ppArgument     = &(m_aAnalyzed.psDetectService);
                                            sName          = ARGUMENTNAME_DETECTSERVICE;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property Extension!" )
                                            ppArgument     = &(m_aAnalyzed.psExtension);
                                            sName          = ARGUMENTNAME_EXTENSION;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property URL!" )
                                            ppArgument     = &(m_aAnalyzed.psURL);
                                            sName          = ARGUMENTNAME_URL;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);

                                            // special mode for URLs
                                            // We must actualize the Jumpmark property if the argument list too!
                                            INetURLObject aParser(*((::rtl::OUString*)pValue));
                                            if (
                                                aParser.GetProtocol() != INET_PROT_NOT_VALID    &&
                                                aParser.HasMark()
                                               )
                                            {
                                                sal_Int32*    ppJumpMark     = &(m_aAnalyzed.psJumpMark);
                                                css::uno::Any aPackedJumpMark;
                                                aPackedJumpMark <<= ::rtl::OUString(aParser.GetMark( INetURLObject::DECODE_WITH_CHARSET ));
                                                impl_addOrChangeArgument( ppJumpMark, ARGUMENTNAME_JUMPMARK, aPackedJumpMark );
                                            }
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0            , "You try to set an empty string value for property FilterName!" )
                                            LOG_ASSERT( FilterCache().existsFilter(*((::rtl::OUString*)pValue)), "You try to set an unknown filter for property FilterName!"     )
                                            ppArgument     = &(m_aAnalyzed.psFilterName);
                                            sName          = ARGUMENTNAME_FILTERNAME;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property Format!" )
                                            ppArgument     = &(m_aAnalyzed.psFormat);
                                            sName          = ARGUMENTNAME_FORMAT;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0            , "You try to set an empty string value for property FrameName!"     )
                                            LOG_ASSERT( TargetCheck::isValidTarget(*((::rtl::OUString*)pValue)), "You try to set an unsupported target name for property FrameName!")
                                            ppArgument     = &(m_aAnalyzed.psFrameName);
                                            sName          = ARGUMENTNAME_FRAMENAME;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property Pattern!" )
                                            ppArgument     = &(m_aAnalyzed.psPattern);
                                            sName          = ARGUMENTNAME_PATTERN;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   {
                                            LOG_ASSERT( ((css::uno::Reference< css::io::XInputStream >*)pValue)->is(), "You try to set an empty string value for property PostData!" )
                                            ppArgument     = &(m_aAnalyzed.pxPostData);
                                            sName          = ARGUMENTNAME_POSTDATA;
                                            aPackedValue <<= *((css::uno::Reference< css::io::XInputStream >*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property PostString!" )
                                            ppArgument     = &(m_aAnalyzed.psPostString);
                                            sName          = ARGUMENTNAME_POSTSTRING;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property Referer!" )
                                            ppArgument     = &(m_aAnalyzed.psReferrer);
                                            sName          = ARGUMENTNAME_REFERRER;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property TemplateName!" )
                                            ppArgument     = &(m_aAnalyzed.psTemplateName);
                                            sName          = ARGUMENTNAME_TEMPLATENAME;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property TemplateRegionName!" )
                                            ppArgument     = &(m_aAnalyzed.psTemplateRegionName);
                                            sName          = ARGUMENTNAME_TEMPLATEREGIONNAME;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property JumpMark!" )
                                            ppArgument     = &(m_aAnalyzed.psJumpMark);
                                            sName          = ARGUMENTNAME_JUMPMARK;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);

                                            // if we change the jumpmark - we have to change the complete URL too!
                                            // If no URL item currently exist inside this descriptor ... do nothing
                                            if ( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST )
                                            {
                                                ::rtl::OUString sOldURL;
                                                if ( m_lArgs[m_aAnalyzed.psURL].Value >>= sOldURL )
                                                {
                                                    INetURLObject aAssembler(sOldURL);
                                                    if (aAssembler.GetProtocol() != INET_PROT_NOT_VALID)
                                                    {
                                                        aAssembler.SetMark(*((::rtl::OUString*)pValue));
                                                        css::uno::Any aPackedURL;
                                                        aPackedURL <<= ::rtl::OUString(aAssembler.GetMainURL(INetURLObject::NO_DECODE));
                                                        sal_Int32* ppURL = &(m_aAnalyzed.psURL);
                                                        impl_addOrChangeArgument( ppURL, ARGUMENTNAME_URL, aPackedURL );
                                                    }
                                                }
                                            }
                                        }
										break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0          , "You try to set an empty string value for property TypeName!" )
                                            LOG_ASSERT( FilterCache().existsType(*((::rtl::OUString*)pValue)), "You try to set an unknown type for property TypeName!"       )
                                            ppArgument     = &(m_aAnalyzed.psTypeName);
                                            sName          = ARGUMENTNAME_TYPENAME;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;
        //_________________________________________________________________________________________________________
        case E_VERSION              :   {
                                            ppArgument     = &(m_aAnalyzed.pnVersion);
                                            sName          = ARGUMENTNAME_VERSION;
                                            aPackedValue <<= *((sal_Int16*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_VIEWID               :   {
                                            ppArgument     = &(m_aAnalyzed.pnViewId);
                                            sName          = ARGUMENTNAME_VIEWID;
                                            aPackedValue <<= *((sal_Int16*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_MACROEXECUTIONMODE   :   {
                                            ppArgument     = &(m_aAnalyzed.pnMacroExecutionMode);
                                            sName          = ARGUMENTNAME_MACROEXECUTIONMODE ;
                                            aPackedValue <<= *((sal_Int16*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_UPDATEDOCMODE        :   {
                                            ppArgument     = &(m_aAnalyzed.pnUpdateDocMode);
                                            sName          = ARGUMENTNAME_UPDATEDOCMODE;
                                            aPackedValue <<= *((sal_Int16*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_FILTEROPTIONS        :   {
                                            LOG_ASSERT( ((::rtl::OUString*)pValue)->getLength() > 0, "You try to set an empty string value for property FilterOptions!" )
                                            ppArgument     = &(m_aAnalyzed.pnFilterOptions);
                                            sName          = ARGUMENTNAME_FILTEROPTIONS;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_ASTEMPLATE           :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property AsTemplate!" )
                                            ppArgument     = &(m_aAnalyzed.pbAsTemplate);
                                            sName          = ARGUMENTNAME_ASTEMPLATE;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_HIDDEN               :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property Hidden!" )
                                            ppArgument     = &(m_aAnalyzed.pbHidden);
                                            sName          = ARGUMENTNAME_HIDDEN;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_OPENNEWVIEW          :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property OpenNewView!" )
                                            ppArgument     = &(m_aAnalyzed.pbOpenNewView);
                                            sName          = ARGUMENTNAME_OPENNEWVIEW;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_READONLY             :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property ReadOnly!" )
                                            ppArgument     = &(m_aAnalyzed.pbReadOnly);
                                            sName          = ARGUMENTNAME_READONLY;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_PREVIEW              :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property Preview!" )
                                            ppArgument     = &(m_aAnalyzed.pbPreview);
                                            sName          = ARGUMENTNAME_PREVIEW;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_SILENT               :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property Silent!" )
                                            ppArgument     = &(m_aAnalyzed.pbSilent);
                                            sName          = ARGUMENTNAME_SILENT;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_MINIMIZED            :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property Silent!" )
                                            ppArgument     = &(m_aAnalyzed.pbMinimized);
                                            sName          = ARGUMENTNAME_MINIMIZED;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_DEEPDETECTION        :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property DeepDetection!" )
                                            ppArgument     = &(m_aAnalyzed.pbDeepDetection);
                                            sName          = ARGUMENTNAME_DEEPDETECTION;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_POSSIZE              :   {
                                            ppArgument     = &(m_aAnalyzed.paPosSize);
                                            sName          = ARGUMENTNAME_POSSIZE;
                                            aPackedValue <<= *((css::awt::Rectangle*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_INPUTSTREAM          :   {
                                            LOG_ASSERT( ((css::uno::Reference< css::io::XInputStream >*)pValue)->is(), "You try to set an empty reference for property InputStream!" )
                                            ppArgument     = &(m_aAnalyzed.pxInputStream);
                                            sName          = ARGUMENTNAME_INPUTSTREAM;
                                            aPackedValue <<= *((css::uno::Reference< css::io::XInputStream >*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM         :   {
                                            LOG_ASSERT( ((css::uno::Reference< css::io::XOutputStream >*)pValue)->is(), "You try to set an empty reference for property OutputStream!" )
                                            ppArgument     = &(m_aAnalyzed.pxOutputStream);
                                            sName          = ARGUMENTNAME_OUTPUTSTREAM;
                                            aPackedValue <<= *((css::uno::Reference< css::io::XOutputStream >*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_STATUSINDICATOR      :   {
                                            LOG_ASSERT( ((css::uno::Reference< css::task::XStatusIndicator >*)pValue)->is(), "You try to set an empty reference for property StatusIndicator!" )
                                            ppArgument     = &(m_aAnalyzed.pxStatusIndicator);
                                            sName          = ARGUMENTNAME_STATUSINDICATOR;
                                            aPackedValue <<= *((css::uno::Reference< css::task::XStatusIndicator >*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER   :   {
                                            LOG_ASSERT( ((css::uno::Reference< css::task::XInteractionHandler >*)pValue)->is(), "You try to set an empty reference for property InteractionHandler!" )
                                            ppArgument     = &(m_aAnalyzed.pxInteractionHandler);
                                            sName          = ARGUMENTNAME_INTERACTIONHANDLER;
                                            aPackedValue <<= *((css::uno::Reference< css::task::XInteractionHandler >*)pValue);
                                        }
                                        break;
        //_________________________________________________________________________________________________________
        case E_REPAIRPACKAGE          :   {
                                            LOG_ASSERT( *((sal_Bool*)pValue)==sal_True || *((sal_Bool*)pValue)==sal_False, "It seams you try to set a non bool value for property RepairPackage!" )
                                            ppArgument     = &(m_aAnalyzed.pbRepairPackage);
                                            sName          = ARGUMENTNAME_REPAIRPACKAGE;
                                            aPackedValue <<= *((sal_Bool*)pValue);
                                        }
                                        break;
		//_________________________________________________________________________________________________________
        case E_DOCUMENTTITLE             :   {
                                            ppArgument     = &(m_aAnalyzed.psDocumentTitle);
                                            sName          = ARGUMENTNAME_DOCUMENTTITLE;
                                            aPackedValue <<= *((::rtl::OUString*)pValue);
                                        }
										break;

	}

    impl_addOrChangeArgument( ppArgument, sName, aPackedValue );
}

/*-************************************************************************************************************//**
    @short      specialized way to set an URL in structured util.URL notation
    @descr      Normaly the URL inside the argument list is set as string. But sometimes
                our user need it in the special structured notation. Then we convert it automaticly here.

    @param      eArgument
                    must be E_URL (all others will be ignored and return <FALSE/>!)

    @param      aValue
                    reference to user memory, which should be used as new value for setting
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::setArgument( EArgument eArgument, const css::util::URL& aValue )
{
    if (m_bReadOnly)
    {
        LOG_WARNING( "ArgumentAnalyzer::setArgument(util::URL)", "You try to set a value on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    if (eArgument != E_URL)
        return;

    // We can use setArgument( E_URL, string ) .. because
    // it analyze the complete URL too and split it into URL and JumpMark automaticly!
    impl_setArgument( E_URL, (void*)&(aValue.Complete) );
}

/*-************************************************************************************************************//**
    @short      delete the required item inside argument list
    @descr      It doesn't work for readonly mode!

    @param      eArgument
                    specify, which item must be deleted
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::deleteArgument( EArgument eArgument )
{
    if (m_bReadOnly)
    {
        LOG_WARNING( "ArgumentAnalyzer::deleteArgument()", "You try to work on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    sal_Int32* ppArgument     = NULL;
    sal_Int32* ppLastArgument = m_aAnalyzed.getLastReference();

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   ppArgument = &(m_aAnalyzed.psCharacterSet);
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   ppArgument = &(m_aAnalyzed.psMediaType);
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   ppArgument = &(m_aAnalyzed.psDetectService);
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   ppArgument = &(m_aAnalyzed.psExtension);
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   ppArgument = &(m_aAnalyzed.psURL);
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   ppArgument = &(m_aAnalyzed.psFilterName);
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   ppArgument = &(m_aAnalyzed.psFormat);
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   ppArgument = &(m_aAnalyzed.psFrameName);
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   ppArgument = &(m_aAnalyzed.psPattern);
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   ppArgument = &(m_aAnalyzed.pxPostData);
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   ppArgument = &(m_aAnalyzed.psPostString);
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   ppArgument = &(m_aAnalyzed.psReferrer);
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   ppArgument = &(m_aAnalyzed.psTemplateName);
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   ppArgument = &(m_aAnalyzed.psTemplateRegionName);
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   ppArgument = &(m_aAnalyzed.psJumpMark);
										break;
		//_________________________________________________________________________________________________________
        case E_VERSION              :   ppArgument = &(m_aAnalyzed.pnVersion);
										break;
		//_________________________________________________________________________________________________________
        case E_VIEWID               :   ppArgument = &(m_aAnalyzed.pnViewId);
										break;
		//_________________________________________________________________________________________________________
        case E_FILTEROPTIONS        :   ppArgument = &(m_aAnalyzed.pnFilterOptions);
										break;
		//_________________________________________________________________________________________________________
        case E_ASTEMPLATE           :   ppArgument = &(m_aAnalyzed.pbAsTemplate);
										break;
		//_________________________________________________________________________________________________________
        case E_HIDDEN               :   ppArgument = &(m_aAnalyzed.pbHidden);
										break;
		//_________________________________________________________________________________________________________
        case E_OPENNEWVIEW          :   ppArgument = &(m_aAnalyzed.pbOpenNewView);
										break;
		//_________________________________________________________________________________________________________
        case E_READONLY             :   ppArgument = &(m_aAnalyzed.pbReadOnly);
										break;
		//_________________________________________________________________________________________________________
        case E_PREVIEW              :   ppArgument = &(m_aAnalyzed.pbPreview);
										break;
		//_________________________________________________________________________________________________________
        case E_SILENT               :   ppArgument = &(m_aAnalyzed.pbSilent);
										break;
		//_________________________________________________________________________________________________________
        case E_MINIMIZED            :   ppArgument = &(m_aAnalyzed.pbMinimized);
										break;
		//_________________________________________________________________________________________________________
        case E_DEEPDETECTION        :   ppArgument = &(m_aAnalyzed.pbDeepDetection);
										break;
		//_________________________________________________________________________________________________________
        case E_POSSIZE              :   ppArgument = &(m_aAnalyzed.paPosSize);
										break;
		//_________________________________________________________________________________________________________
        case E_INPUTSTREAM          :   ppArgument = &(m_aAnalyzed.pxInputStream);
                                        break;
		//_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM         :   ppArgument = &(m_aAnalyzed.pxOutputStream);
                                        break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   ppArgument = &(m_aAnalyzed.psTypeName);
                                        break;
		//_________________________________________________________________________________________________________
        case E_STATUSINDICATOR      :   ppArgument = &(m_aAnalyzed.pxStatusIndicator);
                                        break;
		//_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER   :   ppArgument = &(m_aAnalyzed.pxInteractionHandler);
                                        break;
		//_________________________________________________________________________________________________________
        case E_MACROEXECUTIONMODE   :   ppArgument = &(m_aAnalyzed.pnMacroExecutionMode);
                                        break;
		//_________________________________________________________________________________________________________
        case E_UPDATEDOCMODE        :   ppArgument = &(m_aAnalyzed.pnUpdateDocMode);
                                        break;
		//_________________________________________________________________________________________________________
        case E_REPAIRPACKAGE        :   ppArgument = &(m_aAnalyzed.pbRepairPackage);
										break;
		//_________________________________________________________________________________________________________
        case E_DOCUMENTTITLE        :   ppArgument = &(m_aAnalyzed.psDocumentTitle);
                                        break;

	}

    if(
        ( ppArgument     != NULL               )   &&
        ( ppLastArgument != NULL               )   &&
        ( *ppArgument    != ARGUMENT_NOT_EXIST )
      )
    {
        impl_deleteArgument( ppArgument, ppLastArgument );
    }
}

/*-************************************************************************************************************//**
    @short      clear the whole internaly saved sequence and reset this instance of the ArgumentAnalyzer
    @descr      Of course - it doesn't work for readonly mode.
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::deleteAll()
{
    if (m_bReadOnly)
    {
        LOG_WARNING( "ArgumentAnalyzer::deleteAll()", "You try to work on a readonly descriptor! Call will be ignored ..." )
        return;
    }

    m_lArgs.realloc(0);
    m_aAnalyzed.forgetIt();
}

/*-************************************************************************************************************//**
    @short      check for already existing arguments in sequence
    @descr      -

    @seealso    method getArgument()

    @param      eArgument
                    specify, which item is searched

    @return     <TRUE/>, if it exists,
                <FALSE/> otherwise
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::existArgument( EArgument eArgument ) const
{
    sal_Bool bExist = sal_False;

	switch( eArgument )
	{
		//_________________________________________________________________________________________________________
        case E_CHARACTERSET         :   bExist = ( m_aAnalyzed.psCharacterSet != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_MEDIATYPE            :   bExist = ( m_aAnalyzed.psMediaType != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_DETECTSERVICE        :   bExist = ( m_aAnalyzed.psDetectService != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_EXTENSION            :   bExist = ( m_aAnalyzed.psExtension != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_URL                  :   bExist = ( m_aAnalyzed.psURL != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FILTERNAME           :   bExist = ( m_aAnalyzed.psFilterName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FORMAT               :   bExist = ( m_aAnalyzed.psFormat != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FRAMENAME            :   bExist = ( m_aAnalyzed.psFrameName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_PATTERN              :   bExist = ( m_aAnalyzed.psPattern != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_POSTDATA             :   bExist = ( m_aAnalyzed.pxPostData != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_POSTSTRING           :   bExist = ( m_aAnalyzed.psPostString != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_REFERRER             :   bExist = ( m_aAnalyzed.psReferrer != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATENAME         :   bExist = ( m_aAnalyzed.psTemplateName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_TEMPLATEREGIONNAME   :   bExist = ( m_aAnalyzed.psTemplateRegionName != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_JUMPMARK             :   bExist = ( m_aAnalyzed.psJumpMark != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_VERSION              :   bExist = ( m_aAnalyzed.pnVersion != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_VIEWID               :   bExist = ( m_aAnalyzed.pnViewId != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_FILTEROPTIONS        :   bExist = ( m_aAnalyzed.pnFilterOptions != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_ASTEMPLATE           :   bExist = ( m_aAnalyzed.pbAsTemplate != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_HIDDEN               :   bExist = ( m_aAnalyzed.pbHidden != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_OPENNEWVIEW          :   bExist = ( m_aAnalyzed.pbOpenNewView != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_READONLY             :   bExist = ( m_aAnalyzed.pbReadOnly != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_PREVIEW              :   bExist = ( m_aAnalyzed.pbPreview != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_SILENT               :   bExist = ( m_aAnalyzed.pbSilent != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_MINIMIZED            :   bExist = ( m_aAnalyzed.pbMinimized != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_DEEPDETECTION        :   bExist = ( m_aAnalyzed.pbDeepDetection != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_POSSIZE              :   bExist = ( m_aAnalyzed.paPosSize != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_INPUTSTREAM          :   bExist = ( m_aAnalyzed.pxInputStream != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_OUTPUTSTREAM         :   bExist = ( m_aAnalyzed.pxOutputStream != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_TYPENAME             :   bExist = ( m_aAnalyzed.psTypeName != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_STATUSINDICATOR      :   bExist = ( m_aAnalyzed.pxStatusIndicator != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_INTERACTIONHANDLER   :   bExist = ( m_aAnalyzed.pxInteractionHandler != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_MACROEXECUTIONMODE   :   bExist = ( m_aAnalyzed.pnMacroExecutionMode != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_UPDATEDOCMODE        :   bExist = ( m_aAnalyzed.pnUpdateDocMode != ARGUMENT_NOT_EXIST );
                                        break;
		//_________________________________________________________________________________________________________
        case E_REPAIRPACKAGE        :   bExist = ( m_aAnalyzed.pbRepairPackage != ARGUMENT_NOT_EXIST );
										break;
		//_________________________________________________________________________________________________________
        case E_DOCUMENTTITLE        :   bExist = ( m_aAnalyzed.psDocumentTitle != ARGUMENT_NOT_EXIST );
                                        break;
	}

    return bExist;
}

/*-************************************************************************************************************//**
    @short      checks if the given descriptor is used for a still load request
    @descr      If this descriptor is used for loading a document in hidden or preview mode,
                we can name it "still request". Because the outside code don't whish to see any
                dialogs nor message boxes then.

    @param      -
    @return     <TRUE/> if "Hidden=true" or "Preview=true" are already part of the descriptor and set;
                <FALSE/> otherwhise
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::isStillRequest() const
{
    sal_Bool bHidden  = sal_False;
    sal_Bool bPreview = sal_False;

    if (m_aAnalyzed.pbHidden != ARGUMENT_NOT_EXIST)
        m_lArgs[m_aAnalyzed.pbHidden].Value >>= bHidden;

    if (m_aAnalyzed.pbPreview != ARGUMENT_NOT_EXIST)
        m_lArgs[m_aAnalyzed.pbPreview].Value >>= bPreview;

    return (bHidden || bPreview);
}

/*-************************************************************************************************************//**
    @short      checks if the internal set list seams to be valid
    @descr      There are some checks which can be done:
                    - url with jumpmark but no jumpmark argument
                    - deprecated values
                    - empty values

    @param      -
    @return     <TRUE/> if "Hidden=true" or "Preview=true" are already part of the descriptor and set;
                <FALSE/> otherwhise
*//*-*************************************************************************************************************/
sal_Bool ArgumentAnalyzer::isValid() const
{
    return m_bValid;
}

/*-************************************************************************************************************//**
    @short      correct the internal saved MediaDescriptor
    @descr      There exist some inconsitent properties inside this structure.
                Sometimes we must handle two names for the same meaning ...
                Then we prefer the new notation. We delete the old (mostly deprecated
                item) and replace it by the new one.

    @attention  This method should be called before setArguments() analyze the given descriptor!
                Or they have to be called again to actualize the internal saved informations.
                The reason: we change the descriptor!
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::impl_correctOldFormat()
{
    // analyze it
    // We safe indices to the positions of searched members
    // Such informations can be used to reorganize the descriptor later!
    sal_Int32  nURL          = -1;
    sal_Int32  nFileName     = -1;
    sal_Int32  nMediaType    = -1;
    sal_Int32  nContentType  = -1;
    sal_Int32  nFilterFlags  = -1;
    sal_Int32  nFilterOptions= -1;
    sal_Int32  nFilterName   = -1;
    sal_Int32  nCount        = m_lArgs.getLength();
    sal_Int32  nPackedLength = nCount;

    for( sal_Int32 nItem=0; nItem<nCount; ++nItem )
    {
        if( m_lArgs[nItem].Name == DECLARE_ASCII("URL") )
            nURL = nItem;
        else
        if( m_lArgs[nItem].Name == DECLARE_ASCII("FileName") )
            nFileName = nItem;
        else
        if( m_lArgs[nItem].Name == DECLARE_ASCII("MediaType") )
            nMediaType = nItem;
        else
        if( m_lArgs[nItem].Name == DECLARE_ASCII("ContentType") )
            nContentType = nItem;
        else
        if( m_lArgs[nItem].Name == DECLARE_ASCII("FilterOptions") )
            nFilterOptions = nItem;
        else
        if( m_lArgs[nItem].Name == DECLARE_ASCII("FilterFlags") )
            nFilterFlags = nItem;
        else
        if( m_lArgs[nItem].Name == DECLARE_ASCII("FilterName") )
            nFilterName = nItem;
    }

#ifdef ENABLE_WARNINGS
    if (nFilterName!=-1)
    {
        ::rtl::OUString sFilterName;
        m_lArgs[nFilterName].Value >>= sFilterName;

        if (sFilterName.getLength() < 1)
            LOG_WARNING("ArgumentAnalyzer::impl_correctOldFormat()", "Empty FilterName detected!")
        else if ( ! FilterCache().existsFilter(sFilterName))
            LOG_WARNING("ArgumentAnalyzer::impl_correctOldFormat()", "FilterName is unknown!")
    }
#endif // ENABLE_WARNINGS

    // reorganize it
    // a) if "FileName" but not "URL" exist -> rename it to "URL"
    if (
        nURL      == -1  &&
        nFileName != -1
       )
    {
        m_lArgs[nFileName].Name = DECLARE_ASCII("URL");
    }

    // b) if "ContentType" but not "MediaType" exist -> rename it to "MediaType"
    if (
        nMediaType   == -1  &&
        nContentType != -1
       )
    {
        m_lArgs[nContentType].Name = DECLARE_ASCII("MediaType");
    }

    // c) if "FilterFlags" but not "FilterOptions" exist -> rename it to "FilterOptions"
    if (
        nFilterOptions == -1  &&
        nFilterFlags   != -1
       )
    {
        m_lArgs[nFilterFlags].Name = DECLARE_ASCII("FilterOptions");
    }

    // d) if "FileName" and "URL" exist -> prefer "URL" and delete "FileName"
    //    Deletion is done by copying the last item of the descriptor to this position.
    //    But you have to look, if it's neccessary or possible to do so!
    if (
        nURL      != -1  &&
        nFileName != -1
       )
    {
        --nPackedLength;
        // It's not neccessary to copy it - if it is already the last item in list.
        // Then the operation --nPackedLength will bring it "out of bound" and the
        // following realloc() at the end of this method will delete it realy!
        if (nFileName!=nPackedLength)
            m_lArgs[nFileName] = m_lArgs[nPackedLength];

		// d1) check first, if copy action (for URL and FileName before) has copied
		//     one of the required items for the next action! Correct the index values then.
		//     Last item was moved to nFileName - so the possible new value for the moved
		//     item is nFileName.
		if (nPackedLength == nMediaType)
			nMediaType = nFileName;
		else
		if (nPackedLength == nContentType)
			nContentType = nFileName;
		else
		if (nPackedLength == nFilterOptions)
			nFilterOptions = nContentType;
		else
		if (nPackedLength == nFilterFlags)
			nFilterFlags = nContentType;
    }

    // e) if "ContentType" and "MediaType" exist -> prefer "MediaType" and delete "ContentType"
    if (
        nMediaType   != -1  &&
        nContentType != -1
       )
    {
        --nPackedLength;
        // It's not neccessary to copy it - if it is already the last item in list.
        // Then the operation --nPackedLength will bring it "out of bound" and the
        // following realloc() at the end of this method will delete it realy!
        if (nContentType!=nPackedLength)
            m_lArgs[nContentType] = m_lArgs[nPackedLength];

		// e1) check first, if copy action (for MediaType and ContentType before) has copied
		//     one of the required items for the next action! Correct the index values then.
		//     Last item was moved to nContentType - so the possible new value for the moved
		//     item is nContentType.
		if (nPackedLength == nFilterOptions)
			nFilterOptions = nContentType;
		else
		if (nPackedLength == nFilterFlags)
			nFilterFlags = nContentType;
    }

    // f) if "FilterFlags" and "FilterOptions" exist -> prefer "FilterOptions" and delete "FilterFlags"
    if (
        nFilterOptions != -1  &&
        nFilterFlags   != -1
       )
    {
        --nPackedLength;
        // It's not neccessary to copy it - if it is already the last item in list.
        // Then the operation --nPackedLength will bring it "out of bound" and the
        // following realloc() at the end of this method will delete it realy!
        if (nFilterFlags!=nPackedLength)
            m_lArgs[nFilterFlags] = m_lArgs[nPackedLength];
    }

    // now it's time to pack the descriptor
    // reallocate it to the new size
    // Last (one, two or three !?) items should be superflous and can be deleted.
	if (nPackedLength != nCount)
        m_lArgs.realloc(nPackedLength);
}

/*-************************************************************************************************************//**
    @short      add/delete a property or change a already existing one
    @descr      If a user set or delete a value on this class we must add/delete these property to/from our internal
                sequences(!) - because if user get this list; all set properties must exist with right values!

    @seealso    methods set...()
    @seealso    methods delete...()

    @param      pPosition
                    pointer to an index value of the member m_aAnalyzed to change the index value for this property

    @param      pLast
                    Deletion of an element will be done by copying the last item of the argument list to
                    the position of te item, which should be deleted. Of course we have to update the index value
                    of the structure m_aAnalyzed then for this copied item!
                    => index_last_item=index_deleted_item

    @param      sName
                    the name of the property, if it must be added to the descriptor
                    Then it's used as Name of the PropertyValue struct.

    @param      aValue
                    the new value for this property.
*//*-*************************************************************************************************************/
void ArgumentAnalyzer::impl_addOrChangeArgument(       sal_Int32*       pPosition   ,
                                                 const ::rtl::OUString& sName       ,
                                                 const css::uno::Any&   aValue      )
{
	if (pPosition == NULL)
		return;

    // Element doesnt already exist.
    // Resize sequence and add new item at the end of it.
    // Don't forget to set *ppArgument! This index value of m_aAnalyzed
    // must point to right item in m_pArguments!
    if (*pPosition == ARGUMENT_NOT_EXIST)
    {
        *pPosition = m_lArgs.getLength();
        m_lArgs.realloc( (*pPosition)+1 );
    }

    // It doesn't matter, if element should be changed or added here!
    // *ppArguments must point to right place in m_pArguments always ...
    // and name and value are well known. So we can set it here directly for
    // both cases.
    m_lArgs[*pPosition].Name  = sName ;
    m_lArgs[*pPosition].Value = aValue;
}

//*****************************************************************************************************************
void ArgumentAnalyzer::impl_deleteArgument( sal_Int32* pPosition ,
                                            sal_Int32* pLast     )
{
	if (pPosition == NULL || pLast == NULL)
		return;

	// Algorithm:
    //  a)  copy last item of list to position of item, which should be deleted
    //      If last item should be deleted, do nothing! It will be automticly done by "realloc()" call of b) !
    //      Otherwise dont forget to actualize index of copied last item ...
    //  b)  reallocate list of arguments. Last Item will be destroyed then ...
    //  c)  forget pointer to deleted argument!!!

    // a)
    if( *pPosition != *pLast )
    {
        m_lArgs[*pPosition] = m_lArgs[*pLast];
        *pLast              = *pPosition;
    }

    // b)
    m_lArgs.realloc( m_lArgs.getLength()-1 );

    // c)
    *pPosition = ARGUMENT_NOT_EXIST;
}

/*-************************************************************************************************************//**
    @short      debug helper
    @descr      write the whole content of the current MediaDescriptor to a log file
                Can be used to get an overwview about the current state of such descriptor
                before and after an operation using this ArgumentAnalyzer.

    @param      pMessage
                    description for this dump, written to the file before all properties are written

    @param      pBuffer
                    the stringbuffer, from outside, which must be used to write the dump
                    The outside code can write to the file then.
*//*-*************************************************************************************************************/
#ifdef ENABLE_COMPONENT_SELF_CHECK

    // by the way - define a log file for following debug operations
    #define ARGUMENTANALYZER_LOGFILE "dump_mediadescriptor.log"

void ArgumentAnalyzer::impldbg_dumpArguments(const char* pMessage, ::rtl::OUStringBuffer* pBuffer)
{
    if (m_lArgs.getLength()<1)
        pBuffer->appendAscii("is empty\n");
    else
    {
        sal_Int32 nCount = m_lArgs.getLength();
        for (sal_Int32 nItem = 0; nItem < nCount ; ++nItem )
        {
            pBuffer->appendAscii("[");
            pBuffer->append     (nItem);
            pBuffer->appendAscii("]\t\"");
            pBuffer->append     (m_lArgs[nItem].Name);
            pBuffer->appendAscii("\"\t=\t");

            if (m_lArgs[nItem].Value.getValueTypeClass()==css::uno::TypeClass_STRING)
            {
                ::rtl::OUString sValue;
                m_lArgs[nItem].Value >>= sValue;
                pBuffer->appendAscii("\"");
                pBuffer->append     (sValue);
                pBuffer->appendAscii("\"");
            }
            else
            if (m_lArgs[nItem].Value.getValueTypeClass()==css::uno::TypeClass_SHORT)
            {
                sal_Int16 nValue;
                m_lArgs[nItem].Value >>= nValue;
                pBuffer->append((sal_Int32)nValue);
            }
            else
            if (m_lArgs[nItem].Value.getValueTypeClass()==css::uno::TypeClass_LONG)
            {
                sal_Int32 nValue;
                m_lArgs[nItem].Value >>= nValue;
                pBuffer->append((sal_Int32)nValue);
            }
            else
            if (m_lArgs[nItem].Value.getValueTypeClass()==css::uno::TypeClass_BOOLEAN)
            {
                sal_Bool bValue;
                m_lArgs[nItem].Value >>= bValue;
                if (bValue)
                    pBuffer->appendAscii("true");
                else
                    pBuffer->appendAscii("false");
            }
            else
            if (m_lArgs[nItem].Value.getValueTypeClass()==css::uno::TypeClass_INTERFACE)
            {
                css::uno::Reference< css::uno::XInterface > xValue;
                m_lArgs[nItem].Value >>= xValue;
                if (xValue.is())
                    pBuffer->appendAscii("valid");
                else
                    pBuffer->appendAscii("null");
            }

            pBuffer->appendAscii("\n");
        }
    }
}
#endif // ENABLE_COMPONENT_SELF_CHECK

/*-************************************************************************************************************//**
    @short      debug helper
    @descr      It searches for an item inside the given sequence and
                returns his value. Of course - we must implement it twice.
                We can't use original function of the normal ArgumentAnalyzer!
                We will check his functionality ... and not live with his bugs :-)

    @param      sName
                    name of the argument, we search

    @param      pString
                    returns the value, if its from type string

    @param      pShort
                    returns the value, if its from type short

    @param      pLong
                    returns the value, if its from type long

    @param      pBool
                    returns the value, if its from type boolean

    @param      pInterface
                    returns the value, if its from type object/interface
*//*-*************************************************************************************************************/
#ifdef ENABLE_COMPONENT_SELF_CHECK
sal_Bool ArgumentAnalyzer::impldbg_getArgument(const ::rtl::OUString&                             sName      ,
                                                     ::rtl::OUString*                             pString    ,
                                                     sal_Int16*                                   pShort     ,
                                                     sal_Int32*                                   pLong      ,
                                                     sal_Bool*                                    pBool      ,
                                                     css::uno::Reference< css::uno::XInterface >* pInterface )
{
    if (m_lArgs.getLength()<1)
        return sal_False;

    css::uno::Any   aValue    ;
    sal_Bool        bState    = sal_False;

    sal_Int32 nCount = m_lArgs.getLength();
    for (sal_Int32 i=0; i<nCount; ++i )
    {
        if (m_lArgs[i].Name == sName)
        {
            aValue = m_lArgs[i].Value;
            bState = sal_True;
            break;
        }
    }

    if (aValue.getValueTypeClass()==css::uno::TypeClass_STRING)
        aValue >>= *pString;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_SHORT)
        aValue >>= *pShort;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_LONG)
        aValue >>= *pLong;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_BOOLEAN)
        aValue >>= *pBool;
    else
    if (aValue.getValueTypeClass()==css::uno::TypeClass_INTERFACE)
        aValue >>= *pInterface;

    return bState;
}
#endif // ENABLE_COMPONENT_SELF_CHECK

/*-************************************************************************************************************//**
    @short      implements a test scenario for this class ArgumentAnalyzer
    @descr      It's a static function, which is builded for debug only.
                It create a MediaDescriptor and work with an instance of an ArgumentAnalyzer
                on top of it. It writes a log file, which holds all collected informations of
                this test. So you can find out, if it was successfully or not.

    @attention  The test is seperated into different parts. Don't mnix it. That means: no test should
                use the descriptor of the tests before. Let it work standalone.
                Otherwhise you can't be shure to check the right things.
*//*-*************************************************************************************************************/
#ifdef ENABLE_COMPONENT_SELF_CHECK
void ArgumentAnalyzer::impldbg_checkIt( const css::uno::Reference< css::lang::XMultiServiceFactory >& xSMGR    ,
                                        const css::uno::Reference< css::uno::XInterface >&            xContext )
{
    ::rtl::OUStringBuffer sOut(10000);

    css::uno::Sequence< css::beans::PropertyValue > lDescriptor ;
    ArgumentAnalyzer                                aAnalyzer   ;
    ::rtl::OUString                                 sProp1      ;
    ::rtl::OUString                                 sProp2      ;
    EArgument                                       eProp1      ;
    EArgument                                       eProp2      ;
    ::rtl::OUString                                 sOrg1       ;
    ::rtl::OUString                                 sOrg2       ;
    ::rtl::OUString                                 sCheck1     ;
    ::rtl::OUString                                 sCheck2     ;
    sal_Bool                                        bApiTest    ;
    sal_Bool                                        bDeepTest   ;
    sal_Int32                                       nTest       ;


    //------------------------------------------------------------
    // test "FileName" and "URL"

    nTest = 1;

    sOut.appendAscii("\n\n**************************************************************************************\n");
    sOut.appendAscii("check "                 );
    sOut.append     (nTest                    );
    sOut.appendAscii(") FileName <=> URL ... ");

    sProp1 = DECLARE_ASCII("FileName"                 );
    sOrg1  = DECLARE_ASCII("c:\test\test1.sxw"        );

    eProp2 = E_URL                                     ;
    sProp2 = DECLARE_ASCII("URL"                      );
    sOrg2  = DECLARE_ASCII("file:///c:/test/test1.sxw");

    lDescriptor.realloc(2);

    lDescriptor[0].Name    = sProp1;
    lDescriptor[0].Value <<= sOrg1 ;

    lDescriptor[1].Name    = sProp2;
    lDescriptor[1].Value <<= sOrg2 ;

    aAnalyzer.setArguments(lDescriptor);

    bApiTest = (
                ( aAnalyzer.existArgument(eProp2)        )   &&
                ( aAnalyzer.getArgument(eProp2,sCheck2 ) )   &&
                ( sOrg2 == sCheck2                       )
               );

    bDeepTest = (
                 ( ! aAnalyzer.impldbg_getArgument(sProp1,&sCheck1,NULL,NULL,NULL,NULL ) )   &&
                 (   aAnalyzer.impldbg_getArgument(sProp2,&sCheck2,NULL,NULL,NULL,NULL ) )   &&
                 (   sOrg2 == sCheck2                                                    )
                );

    if (bApiTest && bDeepTest)
        sOut.appendAscii("OK\n");
    else
    {
        sOut.appendAscii("failed\n");
        if ((bApiTest && ! bDeepTest) || (!bApiTest && bDeepTest))
            sOut.appendAscii("reading seams to be wrong.\n");
        else
        if (!bApiTest && !bDeepTest)
            sOut.appendAscii("writing seams to be wrong.\n");
    }
    aAnalyzer.impldbg_dumpArguments("Dump:\n", &sOut);

    //------------------------------------------------------------
    // test "ContentType" and "MediaType"

    ++nTest;

    sOut.appendAscii("\n\n**************************************************************************************\n");
    sOut.appendAscii("check "                          );
    sOut.append     (nTest                             );
    sOut.appendAscii(") ContentType <=> MediaType ... ");

    sProp1 = DECLARE_ASCII("ContentType"   );
    sOrg1  = DECLARE_ASCII("draw/irgendwas");

    eProp2 = E_MEDIATYPE                    ;
    sProp2 = DECLARE_ASCII("MediaType"     );
    sOrg2  = DECLARE_ASCII("text/plain"    );

    lDescriptor.realloc(2);

    lDescriptor[0].Name    = sProp1;
    lDescriptor[0].Value <<= sOrg1 ;

    lDescriptor[1].Name    = sProp2;
    lDescriptor[1].Value <<= sOrg2 ;

    aAnalyzer.setArguments(lDescriptor);

    bApiTest = (
                ( aAnalyzer.existArgument(eProp2)       )   &&
                ( aAnalyzer.getArgument(eProp2,sCheck2) )   &&
                ( sOrg2 == sCheck2                      )
               );

    bDeepTest = (
                 ( ! aAnalyzer.impldbg_getArgument(sProp1,&sCheck1,NULL,NULL,NULL,NULL ) )   &&
                 (   aAnalyzer.impldbg_getArgument(sProp2,&sCheck2,NULL,NULL,NULL,NULL ) )   &&
                 (   sOrg2 == sCheck2                                                    )
                );

    if (bApiTest && bDeepTest)
        sOut.appendAscii("OK\n");
    else
    {
        sOut.appendAscii("failed\n");
        if ((bApiTest && ! bDeepTest) || (!bApiTest && bDeepTest))
            sOut.appendAscii("reading seams to be wrong.\n");
        else
        if (!bApiTest && !bDeepTest)
            sOut.appendAscii("writing seams to be wrong.\n");
    }
    aAnalyzer.impldbg_dumpArguments("Dump:\n", &sOut);

    //------------------------------------------------------------
    // test "FilterFlags" and "FilterOptions"

    ++nTest;

    sOut.appendAscii("\n\n**************************************************************************************\n");
    sOut.appendAscii("check "                              );
    sOut.append     (nTest                                 );
    sOut.appendAscii(") FilterFlags <=> FilterOptions ... ");

    sProp1 = DECLARE_ASCII("FilterFlags"             );
    sOrg1  = DECLARE_ASCII("filter flags old style"  );

    eProp2 = E_FILTEROPTIONS                          ;
    sProp2 = DECLARE_ASCII("FilterOptions"           );
    sOrg2  = DECLARE_ASCII("filter options new style");

    lDescriptor.realloc(2);

    lDescriptor[0].Name    = sProp1;
    lDescriptor[0].Value <<= sOrg1 ;

    lDescriptor[1].Name    = sProp2;
    lDescriptor[1].Value <<= sOrg2 ;

    aAnalyzer.setArguments(lDescriptor);

    bApiTest = (
                ( aAnalyzer.existArgument(eProp2)       )   &&
                ( aAnalyzer.getArgument(eProp2,sCheck2) )   &&
                ( sOrg2 == sCheck2                      )
               );

    bDeepTest = (
                 ( ! aAnalyzer.impldbg_getArgument(sProp1,&sCheck1,NULL,NULL,NULL,NULL ) )   &&
                 (   aAnalyzer.impldbg_getArgument(sProp2,&sCheck2,NULL,NULL,NULL,NULL ) )   &&
                 (   sOrg2 == sCheck2                                                    )
                );

    if (bApiTest && bDeepTest)
        sOut.appendAscii("OK\n");
    else
    {
        sOut.appendAscii("failed\n");
        if ((bApiTest && ! bDeepTest) || (!bApiTest && bDeepTest))
            sOut.appendAscii("reading seams to be wrong.\n");
        else
        if (!bApiTest && !bDeepTest)
            sOut.appendAscii("writing seams to be wrong.\n");
    }
    aAnalyzer.impldbg_dumpArguments("Dump:\n", &sOut);

    //------------------------------------------------------------
    // test "URL" and "JumpMark"

    ++nTest;

    sOut.appendAscii("\n\n**************************************************************************************\n");
    sOut.appendAscii("check "               );
    sOut.append     (nTest                  );
    sOut.appendAscii(") URL & JumpMark ... ");

    eProp1 = E_JUMPMARK                                 ;
    sProp1 = DECLARE_ASCII("JumpMark"                  );
    sOrg1  = DECLARE_ASCII("jump1"                     );

    eProp2 = E_URL                                      ;
    sProp2 = DECLARE_ASCII("URL"                       );
    sOrg2  = DECLARE_ASCII("file:///c:/test/test1.sxw#");
    sOrg2 += sOrg1;

    lDescriptor.realloc(1);

    lDescriptor[0].Name    = sProp2;
    lDescriptor[0].Value <<= sOrg2 ;

    aAnalyzer.setArguments(lDescriptor);

    bApiTest = (
                ( aAnalyzer.existArgument(eProp1)       )   &&
                ( aAnalyzer.getArgument(eProp1,sCheck1) )   &&
                ( aAnalyzer.existArgument(eProp2)       )   &&
                ( aAnalyzer.getArgument(eProp2,sCheck2) )   &&
                ( sOrg1 == sCheck1                      )   &&
                ( sOrg2 == sCheck2                      )
               );

    bDeepTest = (
                 ( aAnalyzer.impldbg_getArgument(sProp1,&sCheck1,NULL,NULL,NULL,NULL ) )   &&
                 ( aAnalyzer.impldbg_getArgument(sProp2,&sCheck2,NULL,NULL,NULL,NULL ) )   &&
                 ( sOrg1 == sCheck1                                                    )   &&
                 ( sOrg2 == sCheck2                                                    )
                );

    if (bApiTest && bDeepTest)
        sOut.appendAscii("OK\n");
    else
    {
        sOut.appendAscii("failed\n");
        if ((bApiTest && ! bDeepTest) || (!bApiTest && bDeepTest))
            sOut.appendAscii("reading seams to be wrong.\n");
        else
        if (!bApiTest && !bDeepTest)
            sOut.appendAscii("writing seams to be wrong.\n");
    }
    aAnalyzer.impldbg_dumpArguments("Dump:\n", &sOut);

    //------------------------------------------------------------
    // test adding properties to empty list

    ++nTest;

    sOut.appendAscii("\n\n**************************************************************************************\n");
    sOut.appendAscii("check "                                 );
    sOut.append     (nTest                                    );
    sOut.appendAscii(") adding properties to empty list ...\n");

    aAnalyzer.deleteAll();

    ::rtl::OUString                                         sCharacterSet       = DECLARE_ASCII("CharacterSet");
    ::rtl::OUString                                         sDetectService      = DECLARE_ASCII("DetectService");
    ::rtl::OUString                                         sExtension          = DECLARE_ASCII("Extensions");
    ::rtl::OUString                                         sFilterName         = DECLARE_ASCII("StarOffice XML (Writer)");
    ::rtl::OUString                                         sFilterOptions      = DECLARE_ASCII("FilterOptions");
    ::rtl::OUString                                         sFormat             = DECLARE_ASCII("Format");
    ::rtl::OUString                                         sFrameName          = DECLARE_ASCII("FrameName");
    ::rtl::OUString                                         sJumpMark           = DECLARE_ASCII("JumpMark");
    ::rtl::OUString                                         sMediaType          = DECLARE_ASCII("MediaType");
    ::rtl::OUString                                         sPattern            = DECLARE_ASCII("Pattern");
    ::rtl::OUString                                         sPostString         = DECLARE_ASCII("PostString");
    ::rtl::OUString                                         sReferrer           = DECLARE_ASCII("Referrer");
    ::rtl::OUString                                         sTemplateName       = DECLARE_ASCII("TemplateName");
    ::rtl::OUString                                         sTemplateRegionName = DECLARE_ASCII("TemplateRegionName");
    ::rtl::OUString                                         sTypeName           = DECLARE_ASCII("writer_StarOffice_XML_Writer");
    ::rtl::OUString                                         sURL                = DECLARE_ASCII("URL");
    ::rtl::OUString                                         sDocumentTitle      = DECLARE_ASCII("DocumentTitle");

    sal_Int16                                               nMacroExecutionMode = 1;
    sal_Int16                                               nUpdateDocMode      = 1;
    sal_Int16                                               nVersion            = 1111;
    sal_Int16                                               nViewID             = 2222;

    sal_Bool                                                bAsTemplate         = sal_True;
    sal_Bool                                                bDeepDetection      = sal_False;
    sal_Bool                                                bHidden             = sal_True;
    sal_Bool                                                bMinimized          = sal_False;
    sal_Bool                                                bOpenNewView        = sal_True;
    sal_Bool                                                bSilent             = sal_False;
    sal_Bool                                                bPreview            = sal_True;
    sal_Bool                                                bReadOnly           = sal_False;
    sal_Bool                                                bRepairPackage      = sal_True;

    css::uno::Reference< css::io::XInputStream >            xInputStream        = css::uno::Reference< css::io::XInputStream >();
    css::uno::Reference< css::io::XInputStream >            xPostData           = css::uno::Reference< css::io::XInputStream >();
    css::uno::Reference< css::task::XInteractionHandler >   xInteractionHandler = css::uno::Reference< css::task::XInteractionHandler >();
    css::uno::Reference< css::io::XOutputStream >           xOutputStream       = css::uno::Reference< css::io::XOutputStream >();
    css::uno::Reference< css::task::XStatusIndicator >      xStatusIndicator    = css::uno::Reference< css::task::XStatusIndicator >();

    css::awt::Rectangle                                     aPosSize            = css::awt::Rectangle(10,10,10,10);

    aAnalyzer.setArgument(E_ASTEMPLATE        ,bAsTemplate        );
    aAnalyzer.setArgument(E_CHARACTERSET      ,sCharacterSet      );
    aAnalyzer.setArgument(E_DEEPDETECTION     ,bDeepDetection     );
    aAnalyzer.setArgument(E_DETECTSERVICE     ,sDetectService     );
    aAnalyzer.setArgument(E_EXTENSION         ,sExtension         );
    aAnalyzer.setArgument(E_FILTERNAME        ,sFilterName        );
    aAnalyzer.setArgument(E_FILTEROPTIONS     ,sFilterOptions     );
    aAnalyzer.setArgument(E_FORMAT            ,sFormat            );
    aAnalyzer.setArgument(E_FRAMENAME         ,sFrameName         );
    aAnalyzer.setArgument(E_HIDDEN            ,bHidden            );
    aAnalyzer.setArgument(E_INPUTSTREAM       ,xInputStream       );
    aAnalyzer.setArgument(E_INTERACTIONHANDLER,xInteractionHandler);
    aAnalyzer.setArgument(E_JUMPMARK          ,sJumpMark          );
    aAnalyzer.setArgument(E_MACROEXECUTIONMODE,nMacroExecutionMode);
    aAnalyzer.setArgument(E_MEDIATYPE         ,sMediaType         );
    aAnalyzer.setArgument(E_MINIMIZED         ,bMinimized         );
    aAnalyzer.setArgument(E_OPENNEWVIEW       ,bOpenNewView       );
    aAnalyzer.setArgument(E_OUTPUTSTREAM      ,xOutputStream      );
    aAnalyzer.setArgument(E_PATTERN           ,sPattern           );
    aAnalyzer.setArgument(E_POSSIZE           ,aPosSize           );
    aAnalyzer.setArgument(E_POSTDATA          ,xPostData          );
    aAnalyzer.setArgument(E_POSTSTRING        ,sPostString        );
    aAnalyzer.setArgument(E_PREVIEW           ,bPreview           );
    aAnalyzer.setArgument(E_READONLY          ,bReadOnly          );
    aAnalyzer.setArgument(E_REFERRER          ,sReferrer          );
    aAnalyzer.setArgument(E_SILENT            ,bSilent            );
    aAnalyzer.setArgument(E_STATUSINDICATOR   ,xStatusIndicator   );
    aAnalyzer.setArgument(E_TEMPLATENAME      ,sTemplateName      );
    aAnalyzer.setArgument(E_TEMPLATEREGIONNAME,sTemplateRegionName);
    aAnalyzer.setArgument(E_TYPENAME          ,sTypeName          );
    aAnalyzer.setArgument(E_UPDATEDOCMODE     ,nUpdateDocMode     );
    aAnalyzer.setArgument(E_URL               ,sURL               );
    aAnalyzer.setArgument(E_VERSION           ,nVersion           );
    aAnalyzer.setArgument(E_VIEWID            ,nViewID            );
    aAnalyzer.setArgument(E_REPAIRPACKAGE     ,bRepairPackage     );
    aAnalyzer.setArgument(E_DOCUMENTTITLE     ,sDocumentTitle     );

    if ( !aAnalyzer.existArgument(E_ASTEMPLATE        ) ) sOut.appendAscii("failed\tAsTemplate doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_CHARACTERSET      ) ) sOut.appendAscii("failed\tCharacterSet doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_DEEPDETECTION     ) ) sOut.appendAscii("failed\tDeepDetection doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_DETECTSERVICE     ) ) sOut.appendAscii("failed\tDetectService doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_EXTENSION         ) ) sOut.appendAscii("failed\tExtensions doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_FILTERNAME        ) ) sOut.appendAscii("failed\tFilterName doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_FILTEROPTIONS     ) ) sOut.appendAscii("failed\tFilterOptions doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_FORMAT            ) ) sOut.appendAscii("failed\tFormat doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_FRAMENAME         ) ) sOut.appendAscii("failed\tFrameName doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_HIDDEN            ) ) sOut.appendAscii("failed\tHidden doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_INPUTSTREAM       ) ) sOut.appendAscii("failed\tInputStream doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_INTERACTIONHANDLER) ) sOut.appendAscii("failed\tInteractionHandler doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_JUMPMARK          ) ) sOut.appendAscii("failed\tJumpMark doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_MACROEXECUTIONMODE) ) sOut.appendAscii("failed\tMacroExecMode doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_MEDIATYPE         ) ) sOut.appendAscii("failed\tMediaType doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_MINIMIZED         ) ) sOut.appendAscii("failed\tMinimized doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_OPENNEWVIEW       ) ) sOut.appendAscii("failed\tOpenNewView doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_OUTPUTSTREAM      ) ) sOut.appendAscii("failed\tOutputStream doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_PATTERN           ) ) sOut.appendAscii("failed\tPattern doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_POSSIZE           ) ) sOut.appendAscii("failed\tPosSize doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_POSTDATA          ) ) sOut.appendAscii("failed\tPostData doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_POSTSTRING        ) ) sOut.appendAscii("failed\tPostString doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_PREVIEW           ) ) sOut.appendAscii("failed\tPreview doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_READONLY          ) ) sOut.appendAscii("failed\tReadOnly doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_REFERRER          ) ) sOut.appendAscii("failed\tReferer doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_SILENT            ) ) sOut.appendAscii("failed\tSilent doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_STATUSINDICATOR   ) ) sOut.appendAscii("failed\tStatusIndicator doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_TEMPLATENAME      ) ) sOut.appendAscii("failed\tTemplateName doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_TEMPLATEREGIONNAME) ) sOut.appendAscii("failed\tTemplateRegionName doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_TYPENAME          ) ) sOut.appendAscii("failed\tTypeName doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_UPDATEDOCMODE     ) ) sOut.appendAscii("failed\tUpdateDocMode doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_URL               ) ) sOut.appendAscii("failed\tURL doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_VERSION           ) ) sOut.appendAscii("failed\tVersion doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_VIEWID            ) ) sOut.appendAscii("failed\tViewID doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_REPAIRPACKAGE     ) ) sOut.appendAscii("failed\tRepairPackage doesn't exist?\n");
    if ( !aAnalyzer.existArgument(E_DOCUMENTTITLE     ) ) sOut.appendAscii("failed\tDocumentTitle doesn't exist?\n");

    ::rtl::OUString                                         sVal;
    sal_Int16                                               nVal;
    sal_Bool                                                bVal;
    css::awt::Rectangle                                     aVal;
    css::uno::Reference< css::io::XInputStream >            xInp;
    css::uno::Reference< css::io::XOutputStream >           xOut;
    css::uno::Reference< css::task::XInteractionHandler >   xHan;
    css::uno::Reference< css::task::XStatusIndicator >      xSta;

    if ( !aAnalyzer.getArgument(E_ASTEMPLATE        ,bVal) || bVal != bAsTemplate        ) sOut.appendAscii("failed\tAsTemplate seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_CHARACTERSET      ,sVal) || sVal != sCharacterSet      ) sOut.appendAscii("failed\tCharacterSet seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_DEEPDETECTION     ,bVal) || bVal != bDeepDetection     ) sOut.appendAscii("failed\tDeepDetection seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_DETECTSERVICE     ,sVal) || sVal != sDetectService     ) sOut.appendAscii("failed\tDetectService seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_EXTENSION         ,sVal) || sVal != sExtension         ) sOut.appendAscii("failed\tExtensions seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_FILTERNAME        ,sVal) || sVal != sFilterName        ) sOut.appendAscii("failed\tFilterName seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_FILTEROPTIONS     ,sVal) || sVal != sFilterOptions     ) sOut.appendAscii("failed\tFilterOptions seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_FORMAT            ,sVal) || sVal != sFormat            ) sOut.appendAscii("failed\tFormat seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_FRAMENAME         ,sVal) || sVal != sFrameName         ) sOut.appendAscii("failed\tFrameName seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_HIDDEN            ,bVal) || bVal != bHidden            ) sOut.appendAscii("failed\tHidden seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_INPUTSTREAM       ,xInp) || xInp != xInputStream       ) sOut.appendAscii("failed\tInputStream seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_INTERACTIONHANDLER,xHan) || xHan != xInteractionHandler) sOut.appendAscii("failed\tInteractionHandler seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_JUMPMARK          ,sVal) || sVal != sJumpMark          ) sOut.appendAscii("failed\tJumpMark seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_MACROEXECUTIONMODE,nVal) || nVal != nMacroExecutionMode) sOut.appendAscii("failed\tMacroExecMode seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_MEDIATYPE         ,sVal) || sVal != sMediaType         ) sOut.appendAscii("failed\tMediaType seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_MINIMIZED         ,bVal) || bVal != bMinimized         ) sOut.appendAscii("failed\tMinimized seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_OPENNEWVIEW       ,bVal) || bVal != bOpenNewView       ) sOut.appendAscii("failed\tOpenNewView seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_OUTPUTSTREAM      ,xOut) || xOut != xOutputStream      ) sOut.appendAscii("failed\tOutputStream seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_PATTERN           ,sVal) || sVal != sPattern           ) sOut.appendAscii("failed\tPattern seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_POSSIZE           ,aVal) || aVal.X != 10 || aVal.Y != 10 || aVal.Width != 10 || aVal.Height != 10 ) sOut.appendAscii("failed\tPosSize seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_POSTDATA          ,xInp) || xInp != xPostData          ) sOut.appendAscii("failed\tPostData seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_POSTSTRING        ,sVal) || sVal != sPostString        ) sOut.appendAscii("failed\tPostString seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_PREVIEW           ,bVal) || bVal != bPreview           ) sOut.appendAscii("failed\tPreview seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_READONLY          ,bVal) || bVal != bReadOnly          ) sOut.appendAscii("failed\tReadOnly seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_REFERRER          ,sVal) || sVal != sReferrer          ) sOut.appendAscii("failed\tReferer seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_SILENT            ,bVal) || bVal != bSilent            ) sOut.appendAscii("failed\tSilent seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_STATUSINDICATOR   ,xSta) || xSta != xStatusIndicator   ) sOut.appendAscii("failed\tStatusIndicator seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_TEMPLATENAME      ,sVal) || sVal != sTemplateName      ) sOut.appendAscii("failed\tTemplateName seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_TEMPLATEREGIONNAME,sVal) || sVal != sTemplateRegionName) sOut.appendAscii("failed\tTemplateRegionName seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_TYPENAME          ,sVal) || sVal != sTypeName          ) sOut.appendAscii("failed\tTypeName seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_UPDATEDOCMODE     ,nVal) || nVal != nUpdateDocMode     ) sOut.appendAscii("failed\tUpdateDocMode seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_URL               ,sVal) || sVal != sURL               ) sOut.appendAscii("failed\tURL seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_VERSION           ,nVal) || nVal != nVersion           ) sOut.appendAscii("failed\tVersion seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_VIEWID            ,nVal) || nVal != nViewID            ) sOut.appendAscii("failed\tViewID seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_REPAIRPACKAGE     ,bVal) || bVal != bRepairPackage     ) sOut.appendAscii("failed\tRepairPackage seams to have wrong value?\n");
    if ( !aAnalyzer.getArgument(E_DOCUMENTTITLE     ,sVal) || sVal != sDocumentTitle     ) sOut.appendAscii("failed\tDocumentTitle seams to have wrong value?\n");

    sOut.appendAscii("OK\n");

    WRITE_LOGFILE(ARGUMENTANALYZER_LOGFILE, U2B(sOut.makeStringAndClear()) )
}
#endif // ENABLE_COMPONENT_SELF_CHECK

} // namespace framework
