/*************************************************************************
 *
 *  $RCSfile: ppdparser.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 17:05:39 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#include <stdlib.h>
#include <stdio.h>

#include <ppdparser.hxx>
#include <tools/debug.hxx>
#include <strhelper.hxx>

#undef DBG_ASSERT
#if defined DBG_UTIL || defined DEBUG
#include <stdio.h>
#define DBG_ASSERT( x, y ) { if( ! (x) ) fprintf( stderr, (y) ); }
#else
#define DBG_ASSERT( x, y )
#endif

PPDParser::PPDParserList PPDParser::aAllParsers;

static String aEmptyString;

#ifdef UNX
#define PATH_DELIMITER '/'
#else
#define PATH_DELIMITER '\\'
#endif

static String GetPPDFile( const String& rFile )
{
	static const sal_Unicode static_PSExtension[] = { '.', 'P', 'S', 0 };
	static const sal_Unicode static_PPDExtension[] = { '.', 'P', 'P', 'D', 0 };
	static const sal_Unicode static_ppds[] = { 'p', 'p', 'd', 's', 0 };

	String aPPD( rFile );
	SvFileStream aStream( aPPD, STREAM_READ );
	// someone might enter a full qualified name here
	if( ! aStream.IsOpen() )
	{
		aPPD = String( getenv( "XPPATH" ), gsl_getSystemTextEncoding() );
		aPPD += PATH_DELIMITER;
		aPPD += String( static_ppds );
		aPPD += PATH_DELIMITER;
		aPPD += rFile;
		aStream.Open( aPPD, STREAM_READ );
		// append .PS if necessary
		if( ! aStream.IsOpen() )
		{
			aPPD += String( static_PSExtension );
			aStream.Open( aPPD, STREAM_READ );
			if( ! aStream.IsOpen() )
			{
				aPPD.Erase( aPPD.Len()-3 );
				aPPD += String( static_PPDExtension );
				aStream.Open( aPPD, STREAM_READ );
				if( ! aStream.IsOpen() )
					aPPD.Erase(); // give up
			}
		}
	}
	return aPPD;
}

String PPDParser::getPPDPrinterName( const String& rFile )
{
	String aPath = GetPPDFile( rFile );
	String aName;
	
	// read in the file
	SvFileStream aStream( aPath, STREAM_READ );
	if( aStream.IsOpen() )
	{
		String aCurLine;
		while( ! aStream.IsEof() )
		{
			ByteString aByteLine;
			aStream.ReadLine( aByteLine );
			aCurLine = String( aByteLine, RTL_TEXTENCODING_MS_1252 );
			if( aCurLine.CompareIgnoreCaseToAscii( "*include:", 9 ) == COMPARE_EQUAL )
			{
				aCurLine.Erase( 0, 9 );
				aCurLine.EraseLeadingChars( ' ' );
				aCurLine.EraseTrailingChars( ' ' );
				aCurLine.EraseLeadingChars( '\t' );
				aCurLine.EraseTrailingChars( '\t' );
				aCurLine.EraseTrailingChars( '\r' );
				aCurLine.EraseTrailingChars( '\n' );
				aCurLine.EraseLeadingChars( '"' );
				aCurLine.EraseTrailingChars( '"' );
				aStream.Close();
				aStream.Open( GetPPDFile( aCurLine ), STREAM_READ );
				continue;
			}
			if( aCurLine.CompareToAscii( "*ModelName:", 11 ) == COMPARE_EQUAL )
			{
				aName = aCurLine.GetToken( 1, '"' );
				break;
			}
			else if( aCurLine.CompareToAscii( "*NickName:", 10 ) == COMPARE_EQUAL )
				aName = aCurLine.GetToken( 1, '"' );
		}
	}
	return aName;
}

PPDParser* PPDParser::getParser( String aFile )
{

	aFile = GetPPDFile( aFile );
	if( ! aFile.Len() )
		return NULL;
	
	for( int i = 0; i < aAllParsers.Count(); i++ )
		if( aAllParsers.GetObject( i )->maFile == aFile )
			return aAllParsers.GetObject( i );

	PPDParser *pNewParser = new PPDParser( aFile );
	aAllParsers.Insert( pNewParser );
	return pNewParser;
}

void PPDParser::freeAll()
{
	while( aAllParsers.Count() )
		delete aAllParsers.Remove( (ULONG)0 );
}

PPDParser::PPDParser( const String& rFile ) :
		maFile( rFile ),
		mpDefaultImageableArea( NULL ),
		mpImageableAreas( NULL ),
		mpDefaultPaperDimension( NULL ),
		mpPaperDimensions( NULL ),
		mpDefaultInputSlot( NULL ),
		mpInputSlots( NULL ),
		mpDefaultResolution( NULL ),
		mpResolutions( NULL ),
		mpDefaultDuplexType( NULL ),
		mpDuplexTypes( NULL ),
		mpFontList( NULL ),
		mnLanguageLevel( 0 ),
		mbColorDevice( TRUE )
{
	// read in the file
	StringList aLines;
	SvFileStream aStream( maFile, STREAM_READ );
	if( aStream.IsOpen() )
	{
		String aCurLine;
		while( ! aStream.IsEof() )
		{
			ByteString aByteLine;
			aStream.ReadLine( aByteLine );
			aCurLine = String( aByteLine, RTL_TEXTENCODING_MS_1252 );
			if( aCurLine.CompareIgnoreCaseToAscii( "*include:", 9 ) == COMPARE_EQUAL )
			{
				aCurLine.Erase( 0, 9 );
				aCurLine.EraseLeadingChars( ' ' );
				aCurLine.EraseTrailingChars( ' ' );
				aCurLine.EraseLeadingChars( '\t' );
				aCurLine.EraseTrailingChars( '\t' );
				aCurLine.EraseTrailingChars( '\r' );
				aCurLine.EraseTrailingChars( '\n' );
				aCurLine.EraseLeadingChars( '"' );
				aCurLine.EraseTrailingChars( '"' );
				aStream.Close();
				aStream.Open( GetPPDFile( aCurLine ), STREAM_READ );
				continue;
			}
			aLines.Insert( new String( aCurLine ), LIST_APPEND );
		}
	}
	aStream.Close();

	// now get the Values
	parse( aLines );
	// and clean up the line buffer again
	while( aLines.Count() )
		delete aLines.Remove( (ULONG)0 );
#ifdef __DEBUG
	fprintf( stderr, "acquired %d Keys from PPD %s:\n", maKeys.Count(), maFile.GetStr() );
	for( int i = 0; i < maKeys.Count(); i++ )
	{
		PPDKey* pKey = maKeys.GetObject( i );
		char* pSetupType = "<unknown>";
		switch( pKey->meSetupType )
		{
			case PPDKey::ExitServer:		pSetupType = "ExitServer";break;
			case PPDKey::Prolog:			pSetupType = "Prolog";break;
			case PPDKey::DocumentSetup:	pSetupType = "DocumentSetup";break;
			case PPDKey::PageSetup:		pSetupType = "PageSetup";break;
			case PPDKey::JCLSetup:			pSetupType = "JCLSetup";break;
			case PPDKey::AnySetup:			pSetupType = "AnySetup";break;
			default: break;
		};
		fprintf( stderr, "\t\"%s\" (\"%s\") (%d values) OrderDependency: %d %s\n",
				 pKey->getKey().GetStr(),
				 pKey->maUITranslation.GetStr(),
				 pKey->countValues(),
				 pKey->mnOrderDependency,
				 pSetupType );
		for( int j = 0; j < pKey->countValues(); j++ )
		{
			fprintf( stderr, "\t\t" );
			PPDValue* pValue = pKey->getValue( j );
			if( pValue == pKey->mpDefaultValue )
				fprintf( stderr, "(Default:) " );
			char* pVType = "<unknown>";
			switch( pValue->meType )
			{
				case eInvocation:		pVType = "invocation";break;
				case eQuoted:			pVType = "quoted";break;
				case eString:			pVType = "string";break;
				case eSymbol:			pVType = "symbol";break;
				case eNo:				pVType = "no";break;
				default: break;
			};
			fprintf( stderr, "option: \"%s\" (\"%s\"), value: type %s \"%s\" (\"%s\")\n",
					 pValue->maOption.GetStr(),
					 pValue->maOptionTranslation.GetStr(),
					 pVType,
					 pValue->maValue.GetStr(),
					 pValue->maValueTranslation.GetStr() );
		}
	}
	fprintf( stderr, "constraints: (%d found)\n", maConstraints.Count() );
	for( int j = 0; j < maConstraints.Count(); j++ )
	{
		PPDConstraint* pCon = maConstraints.GetObject( j );
		fprintf( stderr, "*\"%s\" \"%s\" *\"%s\" \"%s\"\n",
				 pCon->mpKey1->getKey().GetStr(),
				 pCon->mpOption1 ? pCon->mpOption1->maOption.GetStr() : "<nil>",
				 pCon->mpKey2->getKey().GetStr(),
				 pCon->mpOption2 ? pCon->mpOption2->maOption.GetStr() : "<nil>"
				 );
	}
#endif

	// fill in shortcuts
	PPDKey* pKey;

	mpImageableAreas = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "ImageableArea" ) ) );
	if( mpImageableAreas )
		mpDefaultImageableArea = mpImageableAreas->getDefaultValue();
	DBG_ASSERT( mpImageableAreas, "Warning: no ImageableArea in PPD\n" );
	DBG_ASSERT( mpDefaultImageableArea, "Warning: no DefaultImageableArea in PPD\n" );

	mpPaperDimensions = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "PaperDimension" ) ) );
	if( mpPaperDimensions )
		mpDefaultPaperDimension = mpPaperDimensions->getDefaultValue();
	DBG_ASSERT( mpPaperDimensions, "Warning: no PaperDimension in PPD\n" );
	DBG_ASSERT( mpDefaultPaperDimension, "Warning: no DefaultPaperDimension in PPD\n" );

	mpResolutions = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "Resolution" ) ) );
	if( mpResolutions )
		mpDefaultResolution = mpResolutions->getDefaultValue();
	DBG_ASSERT( mpResolutions, "Warning: no Resolution in PPD\n" );
	DBG_ASSERT( mpDefaultResolution, "Warning: no DefaultResolution in PPD\n" );

	mpInputSlots = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "InputSlot" ) ) );
	if( mpInputSlots )
		mpDefaultInputSlot = mpInputSlots->getDefaultValue();
	DBG_ASSERT( mpPaperDimensions, "Warning: no InputSlot in PPD\n" );
	DBG_ASSERT( mpDefaultPaperDimension, "Warning: no DefaultInputSlot in PPD\n" );

	mpDuplexTypes = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "Duplex" ) ) );
	if( mpDuplexTypes )
		mpDefaultDuplexType = mpDuplexTypes->getDefaultValue();

	mpFontList = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "Font" ) ) );
	DBG_ASSERT( mpFontList, "Warning: no Font in PPD\n" );

	// fill in direct values
	if( pKey = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "ModelName" ) ) ) )
		maPrinterName = pKey->getValue( 0 )->maValue;
	if( pKey = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "NickName" ) ) ) )
		maNickName = pKey->getValue( 0 )->maValue;
	if( pKey = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "ColorDevice" ) ) ) )
		mbColorDevice = pKey->getValue( 0 )->maValue.CompareIgnoreCaseToAscii( "true", 4 ) == COMPARE_EQUAL ? TRUE : FALSE;

	if( pKey = getKey( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "LanguageLevel" ) ) ) )
		mnLanguageLevel = pKey->getValue( 0 )->maValue.ToInt32();
}

PPDParser::~PPDParser()
{
	while( maKeys.Count() )
		delete maKeys.Remove( (ULONG)0 );
	while( maConstraints.Count() )
		delete maConstraints.Remove( (ULONG)0 );
}

PPDKey* PPDParser::getKey( const String& rKey )
{
	for( int i = 0; i < maKeys.Count(); i++ )
	{
		PPDKey* pOption = maKeys.GetObject( i );
		if( pOption->getKey() == rKey )
			return pOption;
	}
	return NULL;
}

BOOL PPDParser::hasKey( PPDKey* pKey )
{
	for( int i = 0; i < maKeys.Count(); i++ )
		if( maKeys.GetObject( i ) == pKey )
			return TRUE;
	return FALSE;
}

void PPDParser::parse( StringList& rLines )
{
	PPDValue*	pValue	= NULL;
	PPDKey*		pKey	= NULL;

	int nCurrentLine = 0;
	while( nCurrentLine < rLines.Count() )
	{
		String* pLine = rLines.GetObject( nCurrentLine++ );
		if( pLine->GetChar(0) != '*' )
			continue;
		if( pLine->GetChar(1) == '%' )
			continue;

		String aKey = GetCommandLineToken( 0, pLine->GetToken( 0, ':' ) );
		int nPos = aKey.Search( '/' );
		if( nPos != STRING_NOTFOUND )
			aKey.Erase( nPos );
		aKey.Erase( 0, 1 ); // remove the '*'

		if( aKey.EqualsAscii( "CloseUI" ) || aKey.EqualsAscii( "OpenGroup" ) || aKey.EqualsAscii( "CloseGroup" ) || aKey.EqualsAscii( "End" ) || aKey.EqualsAscii( "OpenSubGroup" ) || aKey.EqualsAscii( "CloseSubGroup" ) )
			continue;

		if( aKey.EqualsAscii( "OpenUI" ) )
		{
			parseOpenUI( *pLine );
			continue;
		}
		else if( aKey.EqualsAscii( "OrderDependency" ) )
		{
			parseOrderDependency( *pLine );
			continue;
		}
		else if( aKey.EqualsAscii( "UIConstraints" ) || aKey.EqualsAscii( "NonUIConstraints" ) )
			continue; // parsed in pass 2

		// default values are parsed in pass 2
		if( aKey.CompareToAscii( "Default", 7 ) == COMPARE_EQUAL )
			continue;
		BOOL bQuery		= FALSE;
		if( aKey.GetChar( 0 ) == '?' )
		{
			aKey.Erase( 0, 1 );
			bQuery = TRUE;
		}

		pKey = getKey( aKey );
		if( ! pKey )
		{
			pKey = new PPDKey( aKey );
			maKeys.Insert( pKey, LIST_APPEND );
		}
		
		String aOption;
		nPos = pLine->Search( ':' );
		if( nPos != STRING_NOTFOUND )
		{
			aOption = pLine->Copy( 1, nPos-1 );
			aOption = GetCommandLineToken( 1, aOption );
			int nTransPos = aOption.Search( '/' );
			if( nTransPos != STRING_NOTFOUND )
				aOption.Erase( nTransPos );
		}
		pValue = pKey->insertValue( aOption );
		if( ! pValue )
			continue;

		if( bQuery && pKey->mpQueryValue == NULL )
		{
			pKey->mpQueryValue = pValue;
			pKey->maValues.Remove( pValue );
		}

		if( nPos == STRING_NOTFOUND )
		{
			// have a single main keyword
			pValue->meType = eNo;
			continue;
		}
		
		// found a colon, there may be an option
		String aLine = pLine->Copy( 1, nPos-1 );
		aLine = WhitespaceToSpace( aLine );
		int nTransPos = aLine.Search( '/' );
		if( nTransPos != STRING_NOTFOUND )
			pValue->maOptionTranslation = aLine.Copy( nTransPos+1 );

		// read in more lines if necessary for multiline values
		aLine = pLine->Copy( nPos+1 );
		while( ! ( aLine.GetTokenCount( '"' ) & 1 ) &&
			   nCurrentLine < rLines.Count() )
			// while there is an even number of tokens; that means
			// an odd number of doubleqoutes
		{
			// copy the newlines also
			aLine += '\n';
			aLine += *(rLines.GetObject( nCurrentLine++ ) );
		}
		aLine = WhitespaceToSpace( aLine );
		
		// check for invocation or quoted value
		if( aLine.GetChar(0) == '"' )
		{
			aLine.Erase( 0, 1 );
			nTransPos = aLine.Search( '"' );
			pValue->maValue = aLine.Copy( 0, nTransPos );
			// after the second doublequote can follow a / and a translation
			pValue->maValueTranslation = aLine.Copy( nTransPos+2 );
			// check for quoted value
			if( pValue->maOption.Len() &&
				aKey.CompareToAscii( "JCL", 3 ) != COMPARE_EQUAL )
				pValue->meType = eInvocation;
			else
				pValue->meType = eQuoted;
			continue;
		}
		// check for symbol value
		if( aLine.GetChar(0) == '^' )
		{
			aLine.Erase( 0, 1 );
			pValue->maValue = aLine;
			pValue->meType = eSymbol;
			continue;
		}
		else
		{
			// must be a string value then
			// strictly this is false because string values
			// can contain any whitespace which is reduced
			// to one space by now
			// who cares ...
			nTransPos = aLine.Search( '/' );
			if( nTransPos == STRING_NOTFOUND )
				nTransPos = aLine.Len();
			pValue->maValue = aLine.Copy( 0, nTransPos );
			pValue->maValueTranslation = aLine.Copy( nTransPos+1 );
			pValue->meType = eString;
		}
	}

	// second pass: fill in defaults
	nCurrentLine = 0;
	while( nCurrentLine < rLines.Count() )
	{
		String aLine( *rLines.GetObject( nCurrentLine ) );
		if( aLine.CompareToAscii( "*Default", 8 ) == COMPARE_EQUAL )
		{
			String aKey( aLine.Copy( 8 ) );
			USHORT nPos = aKey.Search( ':' );
			if( nPos != STRING_NOTFOUND )
				aKey.Erase( nPos );
			PPDKey* pKey = getKey( aKey );
			if( pKey )
			{
				aKey = WhitespaceToSpace( aLine.Copy( nPos+9 ) );
				PPDValue* pValue = pKey->getValue( aKey );
				if( pKey->mpDefaultValue == NULL )
					pKey->mpDefaultValue = pValue;
			}
		}
		else if( aLine.CompareToAscii( "*UIConstraints", 14 ) == COMPARE_EQUAL  ||
				 aLine.CompareToAscii( "*NonUIConstraints", 17 ) == COMPARE_EQUAL )
			parseConstraint( aLine );

		nCurrentLine++;
	}
}

void PPDParser::parseOpenUI( const String& rLine )
{
	String aTranslation;
	String aKey = rLine;

	int nPos = aKey.Search( ':' );
	if( nPos != STRING_NOTFOUND )
		aKey.Erase( nPos );
	nPos = aKey.Search( '/' );
	if( nPos != STRING_NOTFOUND )
	{
		aTranslation = aKey.Copy( nPos + 1 );
		aKey.Erase( nPos );
	}
	aKey = GetCommandLineToken( 1, aKey );
	aKey.Erase( 0, 1 );
	
	PPDKey* pKey = getKey( aKey );
	if( ! pKey )
	{
		pKey = new PPDKey( aKey );
		maKeys.Insert( pKey, LIST_APPEND );
	}
	pKey->mbUIOption = TRUE;
	pKey->maUITranslation = aTranslation;
	
	String aValue = WhitespaceToSpace( rLine.GetToken( 1, ':' ) );
	if( aValue.CompareIgnoreCaseToAscii( "boolean" ) == COMPARE_EQUAL )
		pKey->meUIType = PPDKey::Boolean;
	else if( aValue.CompareIgnoreCaseToAscii( "pickmany" ) == COMPARE_EQUAL )
		pKey->meUIType = PPDKey::PickMany;
	else
		pKey->meUIType = PPDKey::PickOne;
}

void PPDParser::parseOrderDependency( const String& rLine )
{
	String aLine( rLine );
	int nPos = aLine.Search( ':' );
	if( nPos != STRING_NOTFOUND )
		aLine.Erase( 0, nPos+1 );

	int nOrder = GetCommandLineToken( 0, aLine ).ToInt32();
	String aSetup = GetCommandLineToken( 1, aLine );
	String aKey = GetCommandLineToken( 2, aLine );
	if( aKey.GetChar( 0 ) != '*' )
		return; // invalid order depency
	aKey.Erase( 0, 1 );

	PPDKey* pKey = getKey( aKey );
	if( ! pKey )
	{
		pKey = new PPDKey( aKey );
		maKeys.Insert( pKey, LIST_APPEND );
	}
	pKey->mnOrderDependency = nOrder;
	if( aSetup.EqualsAscii( "ExitServer" ) )
		pKey->meSetupType = PPDKey::ExitServer;
	else if( aSetup.EqualsAscii( "Prolog" ) )
		pKey->meSetupType = PPDKey::Prolog;
	else if( aSetup.EqualsAscii( "DocumentSetup" ) )
		pKey->meSetupType = PPDKey::DocumentSetup;
	else if( aSetup.EqualsAscii( "PageSetup" ) )
		pKey->meSetupType = PPDKey::PageSetup;
	else if( aSetup.EqualsAscii( "JCLSetup" ) )
		pKey->meSetupType = PPDKey::JCLSetup;
	else
		pKey->meSetupType = PPDKey::AnySetup;
}

void PPDParser::parseConstraint( const String& rLine )
{
	BOOL bFailed = FALSE;

	String aLine( rLine );
	aLine.Erase( 0, rLine.Search( ':' )+1 );
	PPDConstraint* pConstraint = new PPDConstraint;
	int nTokens = GetCommandLineTokenCount( aLine );
	for( int i = 0; i < nTokens; i++ )
	{
		String aToken = GetCommandLineToken( i, aLine );
		if( aToken.GetChar( 0 ) == '*' )
		{
			aToken.Erase( 0, 1 );
			if( pConstraint->mpKey1 )
				pConstraint->mpKey2 = getKey( aToken );
			else
				pConstraint->mpKey1 = getKey( aToken );
		}
		else
		{
			if( pConstraint->mpKey2 )
			{
				if( ! ( pConstraint->mpOption2 = pConstraint->mpKey2->getValue( aToken ) ) )
					bFailed = TRUE;
			}
			else if( pConstraint->mpKey1 )
			{
				if( ! ( pConstraint->mpOption1 = pConstraint->mpKey1->getValue( aToken ) ) )
					bFailed = TRUE;
			}
			else
				// constraint for nonexistent keys; this happens
				// e.g. in HP4PLUS3 (#75636#)
				bFailed = TRUE;
		}
	}
	// there must be two keywords
	if( ! pConstraint->mpKey1 || ! pConstraint->mpKey2 || bFailed )
	{
#ifdef __DEBUG
		fprintf( stderr, "Warning: constraint \"%s\" is invalid\n", rLine.GetStr() );
#endif
		delete pConstraint;
	}
	else
		maConstraints.Insert( pConstraint, LIST_APPEND );
}

const String& PPDParser::getDefaultPaperDimension()
{
	if( mpDefaultPaperDimension )
		return mpDefaultPaperDimension->maValue;
	
	return aEmptyString;
}

BOOL PPDParser::getMargins( const String& rPaperName,
							int& rLeft, int& rRight,
							int& rUpper, int& rLower )
{
	if( ! mpImageableAreas || ! mpPaperDimensions )
		return FALSE;

	int nPDim=-1, nImArea=-1, i;
	for( i = 0; i < mpImageableAreas->countValues(); i++ )
		if( rPaperName == mpImageableAreas->getValue( i )->maOption )
			nImArea = i;
	for( i = 0; i < mpPaperDimensions->countValues(); i++ )
		if( rPaperName == mpPaperDimensions->getValue( i )->maOption )
			nPDim = i;
	if( nPDim == -1 || nImArea == -1 )
		return FALSE;

	double ImLLx, ImLLy, ImURx, ImURy;
	double PDWidth, PDHeight;
	String aArea = mpImageableAreas->getValue( nImArea )->maValue;
	ImLLx = StringToDouble( GetCommandLineToken( 0, aArea ) );
	ImLLy = StringToDouble( GetCommandLineToken( 1, aArea ) );
	ImURx = StringToDouble( GetCommandLineToken( 2, aArea ) );
	ImURy = StringToDouble( GetCommandLineToken( 3, aArea ) );
//	sscanf( mpImageableAreas->getValue( nImArea )->maValue.GetStr(),
//			"%lg%lg%lg%lg", &ImLLx, &ImLLy, &ImURx, &ImURy );
	aArea = mpPaperDimensions->getValue( nPDim )->maValue;
	PDWidth		= StringToDouble( GetCommandLineToken( 0, aArea ) );
	PDHeight	= StringToDouble( GetCommandLineToken( 1, aArea ) );
//	sscanf( mpPaperDimensions->getValue( nPDim )->maValue.GetStr(),
//			"%lg%lg", &PDWidth, &PDHeight );
	rLeft  = (int)(ImLLx + 0.5);
	rLower = (int)(ImLLy + 0.5);
	rUpper = (int)(PDHeight - ImURy + 0.5);
	rRight = (int)(PDWidth - ImURx + 0.5);

	return TRUE;
}

BOOL PPDParser::getPaperDimension( const String& rPaperName,
								   int& rWidth, int& rHeight )
{
	if( ! mpPaperDimensions )
		return FALSE;

	int nPDim=-1;
	for( int i = 0; i < mpPaperDimensions->countValues(); i++ )
		if( rPaperName == mpPaperDimensions->getValue( i )->maOption )
			nPDim = i;
	if( nPDim == -1 )
		return FALSE;

	double PDWidth, PDHeight;
	String aArea = mpPaperDimensions->getValue( nPDim )->maValue;
	PDWidth		= StringToDouble( GetCommandLineToken( 0, aArea ) );
	PDHeight	= StringToDouble( GetCommandLineToken( 1, aArea ) );
//	sscanf( mpPaperDimensions->getValue( nPDim )->maValue.GetStr(),
//			"%lg%lg", &PDWidth, &PDHeight );
	rHeight	= (int)(PDHeight + 0.5);
	rWidth	= (int)(PDWidth + 0.5);

	return TRUE;
}

const String& PPDParser::matchPaper( int nWidth, int nHeight )
{
	if( ! mpPaperDimensions )
		return aEmptyString;
	
	int nPDim = -1;
	double PDWidth, PDHeight;
	double fSort = 2e36, fNewSort;

	for( int i = 0; i < mpPaperDimensions->countValues(); i++ )
	{
		String aArea =  mpPaperDimensions->getValue( i )->maValue;		
		PDWidth		= StringToDouble( GetCommandLineToken( 0, aArea ) );
		PDHeight	= StringToDouble( GetCommandLineToken( 1, aArea ) );
//		sscanf( mpPaperDimensions->getValue( i )->maValue.GetStr(),
//				"%lg%lg", &PDWidth, &PDHeight );
		PDWidth		/= (double)nWidth;
		PDHeight	/= (double)nHeight;
		if( PDWidth >= 0.9		&&	PDWidth <= 1.1		&&
			PDHeight >= 0.9		&&	PDHeight <= 1.1			)
		{
			fNewSort =
				(1.0-PDWidth)*(1.0-PDWidth) + (1.0-PDHeight)*(1.0-PDHeight);
			if( fNewSort == 0.0 ) // perfect match
				return mpPaperDimensions->getValue( i )->maOption;

			if( fNewSort < fSort )
			{
				fSort = fNewSort;
				nPDim = i;
			}
		}
	}

	static BOOL bDontSwap = FALSE;
	if( nPDim == -1 && ! bDontSwap )
	{
		// swap protrait/landscape and try again
		bDontSwap = TRUE;
		const String& rRet = matchPaper( nHeight, nWidth );
		bDontSwap = FALSE;
		return rRet;
	}

	return nPDim != -1 ? mpPaperDimensions->getValue( nPDim )->maOption : aEmptyString;
}

const String& PPDParser::getDefaultInputSlot()
{
	if( mpDefaultInputSlot )
		return mpDefaultInputSlot->maValue;
	return aEmptyString;
}

const String& PPDParser::getSlot( int nSlot )
{
	if( ! mpInputSlots )
		return aEmptyString;

	if( nSlot > 0 && nSlot < mpInputSlots->countValues() )
		return mpInputSlots->getValue( nSlot )->maOption;
	else if( mpInputSlots->countValues() > 0 )
		return mpInputSlots->getValue( (ULONG)0 )->maOption;
	
	return aEmptyString;
}

const String& PPDParser::getSlotCommand( int nSlot )
{
	if( ! mpInputSlots )
		return aEmptyString;

	if( nSlot > 0 && nSlot < mpInputSlots->countValues() )
		return mpInputSlots->getValue( nSlot )->maValue;
	else if( mpInputSlots->countValues() > 0 )
		return mpInputSlots->getValue( (ULONG)0 )->maValue;
	
	return aEmptyString;
}

const String& PPDParser::getSlotCommand( const String& rSlot )
{
	if( ! mpInputSlots )
		return aEmptyString;

	for( int i=0; i < mpInputSlots->countValues(); i++ )
	{
		PPDValue* pValue = mpInputSlots->getValue( i );
		if( pValue->maOption == rSlot )
			return pValue->maValue;
	}
	return aEmptyString;
}

const String& PPDParser::getPaperDimension( int nPaperDimension )
{
	if( ! mpPaperDimensions )
		return aEmptyString;

	if( nPaperDimension > 0 && nPaperDimension < mpPaperDimensions->countValues() )
		return mpPaperDimensions->getValue( nPaperDimension )->maOption;
	else if( mpPaperDimensions->countValues() > 0 )
		return mpPaperDimensions->getValue( (ULONG)0 )->maOption;
	
	return aEmptyString;
}

const String& PPDParser::getPaperDimensionCommand( int nPaperDimension )
{
	if( ! mpPaperDimensions )
		return aEmptyString;

	if( nPaperDimension > 0 && nPaperDimension < mpPaperDimensions->countValues() )
		return mpPaperDimensions->getValue( nPaperDimension )->maValue;
	else if( mpPaperDimensions->countValues() > 0 )
		return mpPaperDimensions->getValue( (ULONG)0 )->maValue;
	
	return aEmptyString;
}

const String& PPDParser::getPaperDimensionCommand( const String& rPaperDimension )
{
	if( ! mpPaperDimensions )
		return aEmptyString;

	for( int i=0; i < mpPaperDimensions->countValues(); i++ )
	{
		PPDValue* pValue = mpPaperDimensions->getValue( i );
		if( pValue->maOption == rPaperDimension )
			return pValue->maValue;
	}
	return aEmptyString;
}

void PPDParser::getResolutionFromString( const String& rString,
										 int& rXRes, int& rYRes )
{
	int nPos = 0, nDPIPos;

	nDPIPos = rString.SearchAscii( "dpi" );
	if( ( nPos = rString.Search( 'x' ) ) != STRING_NOTFOUND )
	{
		rXRes = rString.Copy( 0, nPos ).ToInt32();
		rYRes = rString.GetToken( 1, 'x' ).Copy( nPos+1, nDPIPos - nPos - 1 ).ToInt32();
		return;
	}
	rXRes = rYRes = rString.Copy( 0, nDPIPos ).ToInt32();
}

void PPDParser::getDefaultResolution( int& rXRes, int& rYRes )
{
	if( mpDefaultResolution )
	{
		getResolutionFromString( mpDefaultResolution->maValue, rXRes, rYRes );
		return;
	}

	rXRes = 300;
	rYRes = 300;
}

int PPDParser::getResolutions()
{
	if( ( ! mpResolutions || mpResolutions->countValues() == 0 ) &&
		mpDefaultResolution )
		return 1;
	return mpResolutions ? mpResolutions->countValues() : 0;
}

void PPDParser::getResolution( int nNr, int& rXRes, int& rYRes )
{
	if( ( ! mpResolutions || mpResolutions->countValues() == 0 ) && mpDefaultResolution && nNr == 0 )
	{
		getDefaultResolution( rXRes, rYRes );
		return;
	}
	if( ! mpResolutions )
		return;

	getResolutionFromString( mpResolutions->getValue( nNr )->maOption,
							 rXRes, rYRes );
}

const String& PPDParser::getResolutionCommand( int nXRes, int nYRes )
{
	if( ( ! mpResolutions || mpResolutions->countValues() == 0 ) && mpDefaultResolution )
		return mpDefaultResolution->maValue;

	if( ! mpResolutions )
		return aEmptyString;

	int nX, nY;
	for( int i = 0; i < mpResolutions->countValues(); i++ )
	{
		getResolutionFromString( mpResolutions->getValue( i )->maOption,
								 nX, nY );
		if( nX == nXRes && nY == nYRes )
			return mpResolutions->getValue( i )->maValue;
	}
	return aEmptyString;
}

const String& PPDParser::getDefaultDuplexType()
{
	if( mpDefaultDuplexType )
		return mpDefaultDuplexType->maValue;
	return aEmptyString;
}

const String& PPDParser::getDuplex( int nDuplex )
{
	if( ! mpDuplexTypes )
		return aEmptyString;

	if( nDuplex > 0 && nDuplex < mpDuplexTypes->countValues() )
		return mpDuplexTypes->getValue( nDuplex )->maOption;
	else if( mpDuplexTypes->countValues() > 0 )
		return mpDuplexTypes->getValue( (ULONG)0 )->maOption;
	
	return aEmptyString;
}

const String& PPDParser::getDuplexCommand( int nDuplex )
{
	if( ! mpDuplexTypes )
		return aEmptyString;

	if( nDuplex > 0 && nDuplex < mpDuplexTypes->countValues() )
		return mpDuplexTypes->getValue( nDuplex )->maValue;
	else if( mpDuplexTypes->countValues() > 0 )
		return mpDuplexTypes->getValue( (ULONG)0 )->maValue;
	
	return aEmptyString;
}

const String& PPDParser::getDuplexCommand( const String& rDuplex )
{
	if( ! mpDuplexTypes )
		return aEmptyString;

	for( int i=0; i < mpDuplexTypes->countValues(); i++ )
	{
		PPDValue* pValue = mpDuplexTypes->getValue( i );
		if( pValue->maOption == rDuplex )
			return pValue->maValue;
	}
	return aEmptyString;
}

void PPDParser::getFontAttributes( int nFont,
								   String& rEncoding,
								   String& rCharset )
{
	if( mpFontList && nFont >= 0 && nFont < mpFontList->countValues() )
	{
		String aAttribs =
			WhitespaceToSpace( mpFontList->getValue( nFont )->maValue );
		rEncoding	= GetCommandLineToken( 0, aAttribs );
		rCharset	= GetCommandLineToken( 2, aAttribs );
	}
}

void PPDParser::getFontAttributes( const String& rFont,
								   String& rEncoding,
								   String& rCharset )
{
	if( mpFontList )
	{
		for( int i = 0; i < mpFontList->countValues(); i++ )
			if( mpFontList->getValue( i )->maOption == rFont )
				getFontAttributes( i, rEncoding, rCharset );
	}
}

const String& PPDParser::getFont( int nFont )
{
	if( ! mpFontList )
		return aEmptyString;

	if( nFont >=0 && nFont < mpFontList->countValues() )
		return mpFontList->getValue( nFont )->maOption;
	return aEmptyString;
}
extern "C" void ReadPrinterMargin( char* pDriver, char* pPaper,
								   int* pLeft, int* pRight,
								   int* pUpper, int* pLower )
{
	PPDParser* pParser = PPDParser::getParser( String( pDriver, gsl_getSystemTextEncoding() ) );
	if( pParser )
	{
		BOOL bFound =
			pParser->getMargins( String( pPaper, RTL_TEXTENCODING_MS_1252 ), *pLeft, *pRight, *pUpper, *pLower );
		if( ! bFound )
			pParser->getMargins( pParser->getDefaultPaperDimension(),
								 *pLeft, *pRight, *pUpper, *pLower );
		
		// pt -> mm/100
		*pLeft = (int)(((double)*pLeft)*35.277778);
		*pRight = (int)(((double)*pRight)*35.277778);
		*pUpper = (int)(((double)*pUpper)*35.277778);
		*pLower = (int)(((double)*pLower)*35.277778);
	}
}

extern "C" const char* GetDefaultPrinterPaper( char* pDriver )
{
	PPDParser* pParser = PPDParser::getParser( String( pDriver, gsl_getSystemTextEncoding() ) );
	if( pParser )
	{
		static ByteString aDefPaper;
		aDefPaper = ByteString( pParser->getDefaultPaperDimension(), RTL_TEXTENCODING_MS_1252 );
		return aDefPaper.GetBuffer();
	}
	return NULL;
}

extern "C" const char* GetPaperTrayName( int nTray, char* pDriver )
{
	PPDParser* pParser = PPDParser::getParser( String( pDriver, gsl_getSystemTextEncoding() ) );
	if( pParser )
	{
		static ByteString aTrayName;
		aTrayName = ByteString( pParser->getSlot( nTray ), RTL_TEXTENCODING_MS_1252 );
		return aTrayName.GetBuffer();
	}
	return NULL;
}

extern "C" const char* GetPaperNameFromSize( char* pDriver, int nWidth, int nHeight )
{
	PPDParser* pParser = PPDParser::getParser( String( pDriver, gsl_getSystemTextEncoding() ) );
	if( pParser )
	{
		static ByteString aPaper;
		aPaper = ByteString( pParser->matchPaper( nWidth, nHeight ), RTL_TEXTENCODING_MS_1252 );
		return aPaper.GetBuffer();
	}
	return NULL;
}

extern "C" int GetPaperSizeFromName( char* pDriver, char* pPaper, int* pWidth, int* pHeight )
{
	PPDParser* pParser = PPDParser::getParser( String( pDriver, gsl_getSystemTextEncoding() ) );
	if( pParser )
	{
		return pParser->getPaperDimension( String( pPaper, RTL_TEXTENCODING_MS_1252 ), *pWidth, *pHeight );
	}
	return 0;
}

extern "C" int GetLevelFromDriver( char* pDriver )
{
	PPDParser* pParser = PPDParser::getParser( String( pDriver, gsl_getSystemTextEncoding() ) );
	if( pParser )
	{
		return pParser->getLanguageLevel();
	}
	return 1;
}

extern "C" int GetColordeviceFromDriver( char* pDriver )
{
	PPDParser* pParser = PPDParser::getParser( String( pDriver, gsl_getSystemTextEncoding() ) );
	if( pParser )
	{
		return (int)pParser->isColorDevice();
	}
	return 0;
}

/*
 *	PPDKey
 */

PPDKey::PPDKey( const String& rKey ) :
			maKey( rKey ),
			mbUIOption( FALSE ),
			meUIType( PickOne ),
			mpDefaultValue( NULL ),
			mpQueryValue( NULL ),
			mnOrderDependency( 100 ),
			meSetupType( AnySetup )
{
}

// -------------------------------------------------------------------

PPDKey::~PPDKey()
{
	while( maValues.Count() )
		delete maValues.Remove( (ULONG)0 );
	if( mpQueryValue )
		delete mpQueryValue;
}

// -------------------------------------------------------------------

PPDValue* PPDKey::getValue( const String& rOption )
{
	PPDValue* pValue = NULL;
	for( int i = 0; i < maValues.Count(); i++ )
	{
		pValue = maValues.GetObject( i );
		if( pValue->maOption.CompareTo( rOption ) == COMPARE_EQUAL )
			return pValue;
	}
	return NULL;
}

// -------------------------------------------------------------------

PPDValue* PPDKey::insertValue( const String& rOption )
{
	if( getValue( rOption ) )
		return NULL;

	PPDValue* pValue = new PPDValue;
	pValue->maOption = rOption;
	maValues.Insert( pValue, LIST_APPEND );
	return pValue;
}

// -------------------------------------------------------------------

/*
 * PPDContext
 */

PPDContext::PPDContext( PPDParser* pParser ) :
		m_pParser( pParser )
{
}

// -------------------------------------------------------------------

PPDContext::~PPDContext()
{
	while( m_aCurrentValues.Count() )
		delete m_aCurrentValues.Remove( (ULONG)0 );
}

// -------------------------------------------------------------------

void PPDContext::setParser( PPDParser* pParser )
{
	while( m_aCurrentValues.Count() )
		delete m_aCurrentValues.Remove( (ULONG)0 );
	m_pParser = pParser;
}

// -------------------------------------------------------------------

PPDValue* PPDContext::getValue( PPDKey* pKey )
{
	if( ! m_pParser )
		return NULL;

	PPDKeyValue* pContainer = NULL;
	for( int i = 0; i < m_aCurrentValues.Count(); i++ )
	{
		PPDKeyValue* pKV = m_aCurrentValues.GetObject( i );
		if( pKV->m_pKey == pKey )
		{
			pContainer = pKV;
			break;
		}
	}
	if( ! pContainer )
	{
		if( ! m_pParser->hasKey( pKey ) )
			return NULL;
		pContainer = new PPDKeyValue( pKey, pKey->getDefaultValue() );
		if( ! pContainer->m_pCurrentValue && pKey->countValues() )
			pContainer->m_pCurrentValue = pKey->getValue( 0 );
		if( pContainer->m_pCurrentValue )
			m_aCurrentValues.Insert( pContainer, LIST_APPEND );
		else
		{
			delete pContainer;
			pContainer = NULL;
		}
	}
	return pContainer ? pContainer->m_pCurrentValue : NULL;
}

// -------------------------------------------------------------------

PPDValue* PPDContext::setValue( PPDKey* pKey, PPDValue* pValue, BOOL bDontCareForConstraints )
{
	if( ! m_pParser || ! pKey )
		return NULL;

	// pValue can be NULL - it means ignore this option

	PPDKeyValue* pContainer = NULL;
	for( int i = 0; i < m_aCurrentValues.Count(); i++ )
	{
		PPDKeyValue* pKV = m_aCurrentValues.GetObject( i );
		if( pKV->m_pKey == pKey )
		{
			pContainer = pKV;
			break;
		}
	}
	if( ! pContainer )
	{
		if( ! m_pParser->hasKey( pKey ) )
			return NULL;
		
		pContainer = new PPDKeyValue( pKey, pKey->getDefaultValue() );
		m_aCurrentValues.Insert( pContainer, LIST_APPEND );
	}

	// check constraints
	if( pValue )
	{
		if( bDontCareForConstraints )
		{
			pContainer->m_pCurrentValue = pValue;
		}
		else if( checkConstraints( pContainer, pValue ) )
		{
			pContainer->m_pCurrentValue = pValue;

			// after setting this value, check all constraints !
			for( int n = 0; n < m_aCurrentValues.Count(); n++ )
			{
				PPDKeyValue* pCont = m_aCurrentValues.GetObject( n );
				if( pCont == pContainer )
					continue;
				
				if( ! checkConstraints( pCont, pCont->m_pCurrentValue, FALSE ) )
				{
#ifdef __DEBUG	
					fprintf( stderr, "PPDContext::setValue: option %s (%s) is constrained after setting %s to %s\n",
							 pCont->m_pKey->getKey().GetStr(),
							 pCont->m_pCurrentValue->maOption.GetStr(),
							 pKey->getKey().GetStr(),
							 pValue->maOption.GetStr() );
#endif
					resetValue( pCont->m_pKey, TRUE );
					n = -1;
				}
			}
		}
	}
	else
		pContainer->m_pCurrentValue = NULL;

	return pContainer->m_pCurrentValue;
}

// -------------------------------------------------------------------

BOOL PPDContext::checkConstraints( PPDKey* pKey, PPDValue* pValue )
{
	if( ! m_pParser || ! pKey || ! pValue )
		return FALSE;

	// ensure that this key is already in the list if it exists at all
	getValue( pKey );

	for( int i = 0; i < m_aCurrentValues.Count(); i++ )
	{
		PPDKeyValue* pKV = m_aCurrentValues.GetObject( i );
		if( pKV->m_pKey == pKey )
			return checkConstraints( pKV, pValue, FALSE );
	}
	return FALSE;
}

// -------------------------------------------------------------------

BOOL PPDContext::resetValue( PPDKey* pKey, BOOL bDefaultable )
{
	if( ! pKey || ! m_pParser || ! m_pParser->hasKey( pKey ) )
		return FALSE;

#ifdef __DEBUG
	fprintf( stderr, "resetValue( %s, %s ) ", pKey->getKey().GetStr(),
			 bDefaultable ? "TRUE" : "FALSE" );
#endif

	PPDValue* pResetValue = pKey->getValue( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "None" ) ) );
	if( ! pResetValue )
		pResetValue = pKey->getValue( String::CreateFromAscii( RTL_CONSTASCII_STRINGPARAM( "False" ) ) );
	if( ! pResetValue && bDefaultable )
		pResetValue = pKey->getDefaultValue();

	BOOL bRet = pResetValue ? ( setValue( pKey, pResetValue ) == pResetValue ? TRUE : FALSE ) : FALSE;

#ifdef DEBUG
	fprintf( stderr, "%s\n", bRet ? "succeeded" : "failed" );
#endif

	return bRet;
}

// -------------------------------------------------------------------

BOOL PPDContext::checkConstraints( PPDKeyValue* pContainer, PPDValue* pNewValue, BOOL bDoReset )
{
	if( ! pNewValue )
		return TRUE;

	if( ! m_pParser || ! pContainer || ! pContainer->m_pKey )
		return FALSE;

	if( pContainer->m_pKey->getValue( pNewValue->maOption ) != pNewValue )
		return FALSE;

	// None / False and the default can always be set, but be careful !
	// setting them might influence constrained values
	if( pNewValue->maOption.EqualsAscii( "None" ) || pNewValue->maOption.EqualsAscii( "False" ) ||
		pNewValue == pContainer->m_pKey->getDefaultValue() )
		return TRUE;

	const PPDParser::PPDConstraintList& rConstraints = m_pParser->getConstraints();
	PPDKey* pKey = pContainer->m_pKey;
	for( int i = 0; i < rConstraints.Count(); i++ )
	{
		PPDParser::PPDConstraint* pConstraint = rConstraints.GetObject( i );
		PPDKey* pLeft	= pConstraint->mpKey1;
		PPDKey* pRight	= pConstraint->mpKey2;
		if( ! pLeft || ! pRight || ( pKey != pLeft && pKey != pRight ) )
			continue;

		PPDKey* pOtherKey = pKey == pLeft ? pRight : pLeft;
		PPDValue* pOtherKeyOption = pKey == pLeft ? pConstraint->mpOption2 : pConstraint->mpOption1;
		PPDValue* pKeyOption = pKey == pLeft ? pConstraint->mpOption1 : pConstraint->mpOption2;

		// syntax *Key1 option1 *Key2 option2
		if( pKeyOption && pOtherKeyOption )
		{
			if( pNewValue != pKeyOption )
				continue;
			if( pOtherKeyOption == getValue( pOtherKey ) )
			{
				return FALSE;
			}
		}
		// syntax *Key1 option *Key2  or  *Key1 *Key2 option
		else if( pOtherKeyOption || pKeyOption )
		{
			if( pKeyOption )
			{
				if( ! ( pOtherKeyOption = getValue( pOtherKey ) ) )
					continue; // this should not happen, PPD broken
				
				if( pKeyOption == pNewValue &&
					! pOtherKeyOption->maOption.EqualsAscii( "None" ) &&
					! pOtherKeyOption->maOption.EqualsAscii( "False" ) )
				{
					// check if the other value can be reset and
					// do so if possible
					if( bDoReset && resetValue( pOtherKey ) )
						continue;

					return FALSE;
				}
			}
			else if( pOtherKeyOption )
			{
				if( getValue( pOtherKey ) == pOtherKeyOption &&
					! pNewValue->maOption.EqualsAscii( "None" ) &&
					! pNewValue->maOption.EqualsAscii( "False" ) )
					return FALSE;
			}
			else
			{
				// this should not happen, PPD is broken
			}
		}
		// syntax *Key1 *Key2
		else
		{
			PPDValue* pOtherValue = getValue( pOtherKey );
			if( ! pOtherValue->maOption.EqualsAscii( "None" ) 	&&
				! pOtherValue->maOption.EqualsAscii( "False" ) 	&&
				! pNewValue->maOption.EqualsAscii( "None" )		&&
				! pNewValue->maOption.EqualsAscii( "False" ) )
				return FALSE;
		}
	}
	return TRUE;
}

// -------------------------------------------------------------------

void* PPDContext::getStreamableBuffer( ULONG& rBytes )
{
	rBytes = 0;
	int i;
	if( ! m_aCurrentValues.Count() )
		return NULL;

	for( i = 0; i < m_aCurrentValues.Count(); i++ )
	{
		PPDKeyValue* pKV = m_aCurrentValues.GetObject( i );
		ByteString aCopy( pKV->m_pKey->getKey(), RTL_TEXTENCODING_MS_1252 );
		rBytes += aCopy.Len();
		rBytes += 1; // for ':'
		if( pKV->m_pCurrentValue )
		{
			aCopy = ByteString( pKV->m_pCurrentValue->maOption, RTL_TEXTENCODING_MS_1252 );
			rBytes += aCopy.Len();
		}
		else
			rBytes += 4;
		rBytes += 1; // for '\0'
	}
	rBytes += 1;
	void* pBuffer = new char[ rBytes ];
	memset( pBuffer, 0, rBytes );
	char* pRun = (char*)pBuffer;
	for( i = 0; i < m_aCurrentValues.Count(); i++ )
	{
		PPDKeyValue* pKV = m_aCurrentValues.GetObject( i );
		ByteString aCopy( pKV->m_pKey->getKey(), RTL_TEXTENCODING_MS_1252 );
		int nBytes = aCopy.Len();
		memcpy( pRun, aCopy.GetBuffer(), nBytes );
		pRun += nBytes;
		*pRun++ = ':';
		if( pKV->m_pCurrentValue )
			aCopy = ByteString( pKV->m_pCurrentValue->maOption, RTL_TEXTENCODING_MS_1252 );
		else
			aCopy = "*nil";
		nBytes = aCopy.Len();
		memcpy( pRun, aCopy.GetBuffer(), nBytes );
		pRun += nBytes;

		*pRun++ = 0;
	}
	return pBuffer;
}

// -------------------------------------------------------------------

void PPDContext::rebuildFromStreamBuffer( void* pBuffer, ULONG nBytes )
{
	if( ! m_pParser )
		return;

	while( m_aCurrentValues.Count() )
		delete m_aCurrentValues.Remove( (ULONG)0 );

	char* pRun = (char*)pBuffer;
	while( *pRun && nBytes )
	{
		ByteString aLine( pRun );
		int nPos = aLine.Search( ':' );
		if( nPos != STRING_NOTFOUND )
		{
			PPDKey* pKey = m_pParser->getKey( String( aLine.Copy( 0, nPos ), RTL_TEXTENCODING_MS_1252 ) );
			if( pKey )
			{
				PPDKeyValue* pKV = new PPDKeyValue( pKey );
				String aOption( aLine.Copy( nPos+1 ), RTL_TEXTENCODING_MS_1252 );
				if( ! aOption.EqualsAscii( "*nil" ) )
					pKV->m_pCurrentValue = pKey->getValue( aOption );
#ifdef __DEBUG
				fprintf( stderr, "PPDContext::rebuildFromStreamBuffer: read PPDKeyValue { %s, %s }\n", pKV->m_pKey->getKey().GetStr(), pKV->m_pCurrentValue ? pKV->m_pCurrentValue->maOption.GetStr() : "<nil>" );
#endif
				m_aCurrentValues.Insert( pKV, LIST_APPEND );
			}
		}
		nBytes -= aLine.Len()+1;
		pRun += aLine.Len()+1;
	}
}
