/*************************************************************************
 *
 *  $RCSfile: ppdparser.hxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/19 10:23:05 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#ifndef _SV_PPDPARSER_HXX
#define _SV_PPDPARSER_HXX

#include <tools/string.hxx>
#include <tools/stream.hxx>
#include <tools/list.hxx>

class PPDParser;

enum PPDValueType { eInvocation, eQuoted, eSymbol, eString, eNo };

struct PPDValue
{
	PPDValueType	meType;
	String			maOption;
	String			maOptionTranslation;
	String			maValue;
	String			maValueTranslation;
};

DECLARE_LIST( PPDValueList, PPDValue* );

// ----------------------------------------------------------------------

/*
 * PPDKey - a container for the available options (=values) of a PPD keyword
 */

class PPDKey
{
	friend class PPDParser;

	String			maKey;
	PPDValueList	maValues;
	PPDValue*		mpDefaultValue;
	PPDValue*		mpQueryValue;

public:
	enum UIType { PickOne, PickMany, Boolean };
	enum SetupType { ExitServer, Prolog, DocumentSetup, PageSetup, JCLSetup, AnySetup };
private:

	BOOL			mbUIOption;
	String			maUITranslation;
	UIType			meUIType;
	int				mnOrderDependency;
	SetupType		meSetupType;
public:
	PPDKey( const String& rKey );
	~PPDKey();

	PPDValue*			insertValue( const String& rOption );
	int					countValues()
		{ return maValues.Count(); }
	// neither getValue will return the query option
	PPDValue*			getValue( int n )
		{ return maValues.GetObject( n ); }
	PPDValue*			getValue( const String& rOption );
	PPDValue*			getDefaultValue() { return mpDefaultValue; }
	PPDValue*			getQueryValue() { return mpQueryValue; }

	const String&		getKey() { return maKey; }
	BOOL				isUIKey() { return mbUIOption; }
	const String&		getUITranslation() { return maUITranslation; }
	UIType				getUIType() { return meUIType; }
	SetupType			getSetupType() { return meSetupType; }
};

DECLARE_LIST( PPDKeyList, PPDKey* );

// ----------------------------------------------------------------------

/*
 * PPDParser - parses a PPD file and contains all available keys from it
 */

class PPDContext;

class PPDParser
{
	friend class PPDContext;
public:
	struct PPDConstraint
	{
		PPDKey*		mpKey1;
		PPDValue*	mpOption1;
		PPDKey*		mpKey2;
		PPDValue*	mpOption2;

		PPDConstraint() : mpKey1( NULL ), mpOption1( NULL ), mpKey2( NULL ), mpOption2( NULL ) {}
	};

	DECLARE_LIST( PPDConstraintList, PPDConstraint* )
	DECLARE_LIST( PPDParserList, PPDParser* )
	DECLARE_LIST( StringList, String* )

	static PPDParserList			aAllParsers;

	PPDKeyList						maKeys;
	PPDConstraintList				maConstraints;

	// some identifying fields
	String							maPrinterName;
	String							maNickName;
	// the full path of the PPD file
	String							maFile;
	// some basic attributes
	BOOL							mbColorDevice;
	ULONG							mnLanguageLevel;


	// shortcuts to important keys and their default values
	// imageable area
	PPDValue*						mpDefaultImageableArea;
	PPDKey*							mpImageableAreas;
	// paper dimensions
	PPDValue*						mpDefaultPaperDimension;
	PPDKey*							mpPaperDimensions;
	// paper trays
	PPDValue*						mpDefaultInputSlot;
	PPDKey*							mpInputSlots;
	// resolutions
	PPDValue*						mpDefaultResolution;
	PPDKey*							mpResolutions;
	// duplex commands
	PPDValue*						mpDefaultDuplexType;
	PPDKey*							mpDuplexTypes;

	// fonts
	PPDKey*							mpFontList;

	PPDParser( const String& rFile );
	~PPDParser();

	void parseOrderDependency( const String& rLine );
	void parseOpenUI( const String& rLine );
	void parseConstraint( const String& rLine );
	void parse( StringList& rLines );
public:
	static PPDParser* getParser( String aFile );
	static String getPPDPrinterName( const String& rFile );
	static void freeAll();

	const String&	getFilename() { return maFile; }

	PPDKey*			getKey( int n ) { return maKeys.GetObject( n ); }
	PPDKey*			getKey( const String& rKey );
	int				getKeys() { return maKeys.Count(); }
	BOOL			hasKey( PPDKey* );

	const PPDConstraintList& getConstraints() { return maConstraints; }

	const String&	getPrinterName() const
		{ return maPrinterName.Len() ? maPrinterName : maNickName; }
	const String&	getNickName() const
		{ return maNickName.Len() ? maNickName : maPrinterName; }

	BOOL			isColorDevice() { return mbColorDevice; }
	ULONG			getLanguageLevel() { return mnLanguageLevel; }

	const String&	getDefaultPaperDimension();
	const void		getDefaultPaperDimension( int& rWidth, int& rHeight )
		{ getPaperDimension( getDefaultPaperDimension(), rWidth, rHeight ); }
	BOOL getPaperDimension( const String& rPaperName,
							int& rWidth, int& rHeight );
	// width and height in pt
	// returns FALSE if paper not found
	int 			getPaperDimensions()
		{ return mpPaperDimensions ? mpPaperDimensions->countValues() : 0; }
	const String&	getPaperDimension( int );
	const String&	getPaperDimensionCommand( int );
	const String&	getPaperDimensionCommand( const String & );

	// match the best paper for width and height
	const String&	matchPaper( int nWidth, int nHeight );
	
	BOOL getMargins( const String& rPaperName,
					 int &rLeft, int& rRight,
					 int &rUpper, int& rLower );
	// values in pt
	// returns TRUE if paper found
	
	// values int pt
	
	const String&	getDefaultInputSlot();
	int				getInputSlots()
		{ return mpInputSlots ? mpInputSlots->countValues() : 0; }
	const String&	getSlot( int );
	const String&	getSlotCommand( int );
	const String&	getSlotCommand( const String& );

	void			getDefaultResolution( int& rXRes, int& rYRes );
	int				getResolutions();
	void			getResolution( int, int& rXRes, int& rYRes );
	const String&	getResolutionCommand( int nXRes, int nYRes );
	// values in dpi
	void			getResolutionFromString( const String&, int&, int& );
	// helper function

	const String&	getDefaultDuplexType();
	int				getDuplexTypes()
		{ return mpDuplexTypes ? mpDuplexTypes->countValues() : 0; }
	const String&	getDuplex( int );
	const String&	getDuplexCommand( int );
	const String&	getDuplexCommand( const String& );

	int				getFonts()
		{ return mpFontList ? mpFontList->countValues() : 0; }
	void			getFontAttributes( int,
									   String& rEncoding,
									   String& rCharset );
	void			getFontAttributes( const String&,
									   String& rEncoding,
									   String& rCharset );
	const String&	getFont( int );
};

// ----------------------------------------------------------------------

/*
 * PPDContext - a class to manage user definable states based on the
 * contents of a PPDParser.
 */

class PPDContext
{
	PPDParser*			m_pParser;

public:
	struct PPDKeyValue
	{
		PPDKey*			m_pKey;
		PPDValue*		m_pCurrentValue;

		PPDKeyValue( PPDKey* pKey, PPDValue* pValue = NULL ) :
				m_pKey( pKey ), m_pCurrentValue( pValue ) {}
	};
	DECLARE_LIST( PPDKeyValueList, PPDKeyValue* )

	PPDKeyValueList		m_aCurrentValues;

	// returns FALSE: check failed, new value is constrained
	//         TRUE:  check succeded, new value can be set
	BOOL checkConstraints( PPDKeyValue*, PPDValue*, BOOL bDoReset = TRUE );
	BOOL resetValue( PPDKey*, BOOL bDefaultable = FALSE );
public:
	PPDContext( PPDParser* pParser = NULL );
	~PPDContext();

	void setParser( PPDParser* );
	PPDParser* getParser() { return m_pParser; }

	PPDValue* getValue( PPDKey* );
	PPDValue* setValue( PPDKey*, PPDValue*, BOOL bDontCareForConstraints = FALSE );

	int countValuesModified() { return m_aCurrentValues.Count(); }
	PPDKey* getModifiedKey( int n )
		{ return m_aCurrentValues.GetObject( n )->m_pKey; }

	// public wrapper for the private method
	BOOL checkConstraints( PPDKey*, PPDValue* );

	// for printer setup
	void*	getStreamableBuffer( ULONG& rBytes );
	void	rebuildFromStreamBuffer( void* pBuffer, ULONG nBytes );
};

#endif // _SV_PPDPARSER_HXX
