/*************************************************************************
 *
 *  $RCSfile: urlconv.cxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: pl $ $Date: 2000/09/22 13:54:08 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#include <hash_map>
#include <stdio.h>

#include <rtl/textcvt.h>
#include <rtl/string.hxx>
#include <rtl/ustring.hxx>
#include <osl/profile.hxx>
#include <osl/file.hxx>
#include <fsys.hxx>

#define SHORTCUT "InternetShortcut"

#ifdef WIN32
#define LINEEND "\n"
#else
#define LINEEND "\r\n"
#endif

using namespace osl;
using namespace rtl;
using namespace std;

namespace std {
struct hash< rtl::OUString >
{
	size_t operator()(const rtl::OUString& rString) const
		{ return (size_t)rString.hashCode(); }
};

struct hash< rtl::OString >
{
	size_t operator()(const rtl::OString& rString) const
		{ return (size_t)rString.hashCode(); }
};

}

int verbosity = 0;

class UrlConverter : public Profile
{
	OString aOutFile;
public:
	UrlConverter( const OUString& rIn, const OUString& rOut ) :
			Profile( rIn ), aOutFile( OUStringToOString(rOut, gsl_getSystemTextEncoding())) {}
	~UrlConverter() {}

	int convertToUnicode();
	int convertToLocale();
	rtl_TextEncoding getEncoding( const OString& rLang );
};

rtl_TextEncoding UrlConverter::getEncoding( const OString& rLang )
{
	if( rLang == "english" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "german" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "french" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "italian" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "dutch" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "portuguese" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "swedish" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "spanish" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "chinese_simplified" )
		return RTL_TEXTENCODING_MS_936;
	else if( rLang == "danish" )
		return RTL_TEXTENCODING_MS_1252;
	else if( rLang == "russian" )
		return RTL_TEXTENCODING_MS_1251;
	else if( rLang == "polish" )
		return RTL_TEXTENCODING_MS_1250;
	else if( rLang == "greek" )
		return RTL_TEXTENCODING_MS_1253;
	else if( rLang == "turkish" )
		return RTL_TEXTENCODING_MS_1254;
	else if( rLang == "chinese_traditional" )
		return RTL_TEXTENCODING_MS_950;
	else if( rLang == "korean" )
		return RTL_TEXTENCODING_MS_949;
	else if( rLang == "japanese" )
		return RTL_TEXTENCODING_MS_932;

	// default
	return RTL_TEXTENCODING_DONTKNOW;
}

int UrlConverter::convertToLocale()
{
	FILE* fileOut = fopen( aOutFile.getStr(), "w" );
	if( ! fileOut )
		return -1;

	list< OString > aSections = getSections();
	for( list< OString >::iterator sect = aSections.begin();
		 sect != aSections.end(); ++sect )
	{
		OString aSection = *sect;
		list< OString > aKeys = getSectionEntries( aSection );
		if( aSection.indexOf( ".W", aSection.getLength()-2 ) == aSection.getLength()-2 )
		{
			// get language
			OString aLang( aSection.getToken( 1, '-' ) );
			if( aLang.getLength() > 2 )
				aLang = aLang.copy( 0, aLang.getLength()-2 );
			else
				continue;

			rtl_TextEncoding eLang = getEncoding( aLang );
			if( eLang == RTL_TEXTENCODING_DONTKNOW )
			{
				if( aLang.getLength() && verbosity > 0 )
					fprintf( stderr, "warning: did not find encoding for language %s, defaulting to ansi1252\n", aLang.getStr() );
				eLang = RTL_TEXTENCODING_MS_1252;
			}

			OString aNewSection = aSection.copy( 0, aSection.getLength()-2 );
			fprintf( fileOut, "[%s]" LINEEND, aNewSection.getStr() );

			for( list< OString >::iterator key = aKeys.begin();
				 key != aKeys.end(); ++key )
			{
				OUString aUTF16Value = OStringToOUString( readString( *sect, *key, "" ), RTL_TEXTENCODING_UTF7 );
				OString aLocaleValue = OUStringToOString( aUTF16Value, eLang );
				fprintf( fileOut, "%s=%s" LINEEND, (*key).getStr(), aLocaleValue.getStr() );
			}
			fprintf( fileOut, LINEEND );
		}
		else
		{
			fprintf( fileOut, "[%s]" LINEEND, aSection.getStr() );
			for( list< OString >::iterator key = aKeys.begin();
				 key != aKeys.end(); ++key )
			{
				fprintf( fileOut, "%s=%s" LINEEND, (*key).getStr(), readString( *sect, *key, "" ).getStr() );
			}
			fprintf( fileOut, LINEEND );
		}
	}
	fclose( fileOut );
	return 0;
}

int UrlConverter::convertToUnicode()
{
	FILE* fileOut = fopen( aOutFile.getStr(), "w" );
	if( ! fileOut )
		return -1;
	fprintf( fileOut, "%s" LINEEND, readString( SHORTCUT, "URL", "" ).getStr() );
	fprintf( fileOut, "[" SHORTCUT "]" LINEEND );

	list< OString > aKeys = getSectionEntries( SHORTCUT );
	list< OString >::iterator it;

	hash_map< OString, OString, hash< OString > > aKeyValues;

	for( it = aKeys.begin(); it != aKeys.end(); ++it )
	{
		OString aValue( readString( SHORTCUT, *it, "" ) );
		aKeyValues[ *it ] = aValue;
		fprintf( fileOut, "%s=%s" LINEEND,
				 (*it).getStr(),
				 aValue.getStr()
				 );
	}
	fprintf( fileOut, LINEEND );
	fprintf( fileOut, "[" SHORTCUT ".W]" LINEEND );
	for( it = aKeys.begin(); it != aKeys.end(); ++it )
	{
		// defaulting encoding to ansi
		OUString aUTF16Value( OStringToOUString( aKeyValues[ *it ], RTL_TEXTENCODING_MS_1252 ) );
		OString aUTF7Value( OUStringToOString( aUTF16Value, RTL_TEXTENCODING_UTF7 ) );
		fprintf( fileOut, "%s=%s" LINEEND,
				 (*it).getStr(),
				 aUTF7Value.getStr()
				 );
	}
	fprintf( fileOut, LINEEND );

	list< OString > aSections = getSections();
	for( it = aSections.begin(); it != aSections.end(); ++it )
	{
		// language string is found after InternetShortcut-
		if( (*it).getLength() > 16 )
		{
			OString aLanguage = (*it).copy( 17 );
			rtl_TextEncoding aEncoding = getEncoding( aLanguage );
			if( aEncoding != RTL_TEXTENCODING_DONTKNOW )
			{
				fprintf( fileOut, "[" SHORTCUT "-%s.W]" LINEEND, aLanguage.getStr() );
				aKeys = getSectionEntries( *it );
				list< OString >::iterator iter;
				for( iter = aKeys.begin(); iter != aKeys.end(); ++iter )
				{
					OUString aUTF16Value( OStringToOUString( readString( *it, *iter, "" ), aEncoding ) );
					OString aUTF7Value( OUStringToOString( aUTF16Value, RTL_TEXTENCODING_UTF7 ) );
					fprintf( fileOut, "%s=%s" LINEEND,
							 (*iter).getStr(),
							 aUTF7Value.getStr()
							 );
				}
				fprintf( fileOut, LINEEND );
			}
			else
				fprintf( stderr, "Warning: language %s is unknown\n", aLanguage.getStr() );
		}
	}
	fclose( fileOut );
	return 0;
}

int
#ifdef WIN32
__cdecl
#endif
main( int argc, char** argv )
{
	OString aInFile;
	OString aOutFile;

	int bToLocale = 0;

	// parse parameters
	for( int i = 1; i < argc; i++ )
	{
		OString aArg( argv[ i ] );
		if( aArg.getToken( 0, '=' ).equalsIgnoreCase( "infile" ) )
			aInFile = aArg.getToken( 1, '=' );
		else if( aArg.getToken( 0, '=' ).equalsIgnoreCase( "outfile" ) )
			aOutFile = aArg.getToken( 1, '=' );
		else if( aArg == "-v" )
			verbosity++;
		else if( aArg == "-tolocale" )
			bToLocale = 1;
		else if( aArg == "-h" )
		{
			// display usage info
			printf(
				"USAGE:%s [-v] [-tolocale] [infile=<old> outfile=<new>]\n"
				, argv[0] );
			printf( "-tolocale: conversion is done from unicode to locale specific files\n" );
			printf( "if infile and outfile are not present, all *.url files in the\ncurrent working directory are processed and overwritten on success\n" );
			return 0;
		}
	}
	if( aInFile.getLength() && aOutFile.getLength() )
	{
		DirEntry aIn( String( OStringToOUString( aInFile, gsl_getSystemTextEncoding()) ) );
		aIn.ToAbs();
		DirEntry aOut( String( OStringToOUString( aOutFile, gsl_getSystemTextEncoding()) ) );
		aOut.ToAbs();

		OUString aUInFile;
		::osl::FileBase::normalizePath( aIn.GetFull(), aUInFile );
		UrlConverter aConverter( aUInFile, aOut.GetFull() );
		int nResult = bToLocale ? aConverter.convertToLocale() : aConverter.convertToUnicode();
		if( nResult )
		{
			fprintf( stderr, "conversion of %s to %s failed\n",
					 aInFile.getStr(), aOutFile.getStr() );
			return 1;
		}
	}
	else
	{
		Dir aDir( DirEntry( String::CreateFromAscii( "*.url") ) );
		aDir.Update();
		int nEntries = aDir.Count();
		for( int i = 0; i < nEntries; i++ )
		{
			DirEntry aIn( aDir[i] );
			aIn.ToAbs();
			DirEntry aOut( aIn );
			aOut.SetExtension( String::CreateFromAscii( "new" ) );

			OUString aUInFile;
			::osl::FileBase::normalizePath( aIn.GetFull(), aUInFile );
			UrlConverter aConverter( aUInFile, aOut.GetFull() );
			if( ! ( bToLocale ? aConverter.convertToLocale() : aConverter.convertToUnicode() ) )
			{
				aIn.Kill();
				int nErr = aOut.MoveTo( aIn );
				if( nErr )
				{
					fprintf( stderr, "moveto %s -> %s failed for reason %d\n",
							 ByteString( aOut.GetName(), RTL_TEXTENCODING_MS_1252 ).GetBuffer(),
							 ByteString( aIn.GetName(), RTL_TEXTENCODING_MS_1252 ).GetBuffer(),
							 nErr);
				}
			}
		}
	}
	return 0;
}
