/*************************************************************************
 *
 *  $RCSfile: poly3d.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: aw $ $Date: 2001/03/14 13:05:11 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdio.h>

#ifndef _STRING_H
#include <string.h>
#ifndef _STRING_H
#define _STRING_H
#endif
#endif

#ifndef _XPOLY_HXX
#include "xpoly.hxx"
#endif

#ifndef _POLY3D_HXX
#include "poly3d.hxx"
#endif

#ifndef _SVX_MATRIX3D_HXX
#include "matrix3d.hxx"
#endif

#ifndef _SV_POLY_HXX
#include <vcl/poly.hxx>
#endif

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif

#ifndef _B3D_BASE3D_HXX
#include <goodies/base3d.hxx>
#endif

DBG_NAME(Polygon3D);
DBG_NAME(PolyPolygon3D);

/*************************************************************************
|*
|* 3D-PolyPolygon-Implementierung
|*
\************************************************************************/

DECLARE_LIST(Polygon3DList, Polygon3D*);

class ImpPolyPolygon3D
{
public:
	Polygon3DList				aPoly3DList;
	UINT16						nRefCount;

	ImpPolyPolygon3D(UINT16 nInitSize = 4, UINT16 nResize = 4) 
	:	aPoly3DList(1024, nInitSize, nResize) { nRefCount = 1; }
	ImpPolyPolygon3D(const ImpPolyPolygon3D& rImpPolyPoly3D);
	~ImpPolyPolygon3D();

	BOOL operator==(const ImpPolyPolygon3D& rImpPolyPoly3D) const;
	BOOL operator!=(const ImpPolyPolygon3D& rImpPolyPoly3D) const
		{ return !operator==(rImpPolyPoly3D); }
};

/*************************************************************************
|*
|* Standard-Konstruktor
|*
\************************************************************************/

ImpPolygon3D::ImpPolygon3D(UINT16 nInitSize, UINT16 nPolyResize)
{
	pPointAry = NULL;
	bDeleteOldAry = FALSE;
	bClosed = FALSE;
	nSize = 0;
	nResize = nPolyResize;
	nPoints = 0;
	nRefCount = 1;
	Resize(nInitSize);
}

/*************************************************************************
|*
|* Copy-Konstruktor
|*
\************************************************************************/

ImpPolygon3D::ImpPolygon3D(const ImpPolygon3D& rImpPoly3D)
{
	((ImpPolygon3D&)rImpPoly3D).CheckPointDelete();

	pPointAry = NULL;
	bDeleteOldAry = FALSE;
	bClosed = rImpPoly3D.bClosed;
	nSize = 0;
	nResize = rImpPoly3D.nResize;
	nPoints = 0;
	nRefCount = 1;
	Resize(rImpPoly3D.nSize);

	// Kopieren
	nPoints	= rImpPoly3D.nPoints;
	memcpy(pPointAry, rImpPoly3D.pPointAry, nSize*sizeof(Vector3D));
}

/*************************************************************************
|*
|* Destruktor
|*
\************************************************************************/

ImpPolygon3D::~ImpPolygon3D()
{
	delete[] (char*)pPointAry;

	if(bDeleteOldAry)
		delete[] (char*)pOldPointAry;
}

/*************************************************************************
|*
|* Gibt unter Beachtung eines Flags den Speicher fuer das Polygon frei.
|*
\************************************************************************/

void ImpPolygon3D::CheckPointDelete()
{
	if(bDeleteOldAry)
	{
		delete[] (char*)pOldPointAry;
		bDeleteOldAry = FALSE;
	}
}

/*************************************************************************
|*
|*    ImpPolygon3D::Resize()
|*
|*    !!! Polygongroesse aendern - wenn bDeletePoints FALSE, dann den
|*    Punkt-Array nicht loeschen, sondern in pOldPointAry sichern und
|*    das Flag bDeleteOldAry setzen. Beim naechsten Zugriff wird
|*    das Array dann geloescht.
|*    Damit wird verhindert, dass bei Poly3D[n] = Poly3D[0] durch ein
|*    Resize der fuer den rechten Ausdruck verwendete Vector3D-Array
|*    vorzeitig geloescht wird.
|*
\************************************************************************/

void ImpPolygon3D::Resize(UINT16 nNewSize, BOOL bDeletePoints)
{
	if(nNewSize == nSize)
		return;

	UINT16 nOldSize = nSize;

	CheckPointDelete();
	pOldPointAry = pPointAry;

	// Neue Groesse auf vielfaches von nResize runden, sofern Objekt
	// nicht neu angelegt wurde (nSize != 0)
	if(nSize && nNewSize > nSize)
	{
		DBG_ASSERT(nResize, "Resize-Versuch trotz nResize = 0 !");
		nNewSize = nSize + ((nNewSize-nSize-1) / nResize + 1) * nResize;
	}

	// Punkt-Array erzeugen
	nSize = nNewSize;
	pPointAry = (Vector3D*)new char[nSize * sizeof(Vector3D)];
	memset(pPointAry, 0, nSize * sizeof(Vector3D));

	// Eventuell umkopieren
	if(nOldSize)
	{
		if(nOldSize < nSize)
		{
			memcpy(pPointAry, pOldPointAry, nOldSize * sizeof(Vector3D));
		}
		else
		{
			memcpy(pPointAry, pOldPointAry, nSize*sizeof(Vector3D));

			// Anzahl der gueltigen Punkte anpassen
			if(nPoints > nSize)
				nPoints = nSize;
		}
		if(bDeletePoints)
			delete[] (char*)pOldPointAry;
		else					
			bDeleteOldAry = TRUE;
	}
}

/*************************************************************************
|*
|* Vektoren einfuegen
|*
\************************************************************************/

void ImpPolygon3D::InsertSpace(UINT16 nPos, UINT16 nCount)
{
	UINT16 nOldSize = nSize;

	CheckPointDelete();

	if(nPos > nPoints)
		nPos = nPoints;

	// Wenn Polygon zu klein dann groesser machen
	if(nPoints + nCount > nSize)
		Resize(nPoints + nCount);

	// Wenn nicht hinter dem letzten Punkt eingefuegt wurde,
	// den Rest nach hinten schieben
	if(nPos < nPoints)
	{
		memmove(&pPointAry[nPos+nCount], &pPointAry[nPos],
			(nPoints - nPos) * sizeof(Vector3D));
	}
	memset(&pPointAry[nPos], 0, nCount * sizeof(Vector3D));

	nPoints += nCount;
}

/*************************************************************************
|*
|* Vektoren entfernen
|*
\************************************************************************/

void ImpPolygon3D::Remove(UINT16 nPos, UINT16 nCount)
{
	CheckPointDelete();

	if(nPos + nCount <= nPoints)
	{
		UINT16 nMove = nPoints - nPos - nCount;
		if(nMove)
			memmove(&pPointAry[nPos], &pPointAry[nPos+nCount],
					nMove * sizeof(Vector3D));

		nPoints -= nCount;
		memset(&pPointAry[nPoints], 0, nCount * sizeof(Vector3D));
	}
}

/*************************************************************************
|*
|* Standard-Konstruktor
|*
\************************************************************************/

Polygon3D::Polygon3D(UINT16 nSize, UINT16 nResize)
{
	pImpPolygon3D = new ImpPolygon3D(nSize, nResize);
}

/*************************************************************************
|*
|* Copy-Konstruktor
|*
\************************************************************************/

Polygon3D::Polygon3D(const Polygon3D& rPoly3D)
{
	pImpPolygon3D = rPoly3D.pImpPolygon3D;
	pImpPolygon3D->nRefCount++;
}

/*************************************************************************
|*
|* Konstruktor aus Standard-Polygon
|*
\************************************************************************/

Polygon3D::Polygon3D(const Polygon& rPoly, double fScale)
{
	UINT16 nSize(rPoly.GetSize());
	pImpPolygon3D = new ImpPolygon3D(nSize);

	if(fScale != 1.0)
	{
		for(UINT16 a=0; a<nSize; a++)
		{
			pImpPolygon3D->pPointAry[a].X() = rPoly[a].X() * fScale;
			pImpPolygon3D->pPointAry[a].Y() = -rPoly[a].Y() * fScale;
			pImpPolygon3D->pPointAry[a].Z() = 0.0;
		}
	}
	else
	{
		for(UINT16 a=0; a<nSize; a++)
		{
			pImpPolygon3D->pPointAry[a].X() = rPoly[a].X();
			pImpPolygon3D->pPointAry[a].Y() = -rPoly[a].Y();
			pImpPolygon3D->pPointAry[a].Z() = 0.0;
		}
	}

	pImpPolygon3D->nPoints = nSize;

	CheckClosed();
}

/*************************************************************************
|*
|* Konstruktor aus XPolygon - Achtung! Es werden nur die Punkte
|* uebernommen, ohne Konvertierung
|*
\************************************************************************/

Polygon3D::Polygon3D(const XPolygon& rXPoly, double fScale)
{
	UINT16 nSize(rXPoly.GetPointCount());
	pImpPolygon3D = new ImpPolygon3D(nSize);

	if(fScale != 1.0)
	{
		for(UINT16 a=0; a<nSize; a++)
		{
			pImpPolygon3D->pPointAry[a].X() = rXPoly[a].X() * fScale;
			pImpPolygon3D->pPointAry[a].Y() = -rXPoly[a].Y() * fScale;
			pImpPolygon3D->pPointAry[a].Z() = 0.0;
		}
	}
	else
	{
		for(UINT16 a=0; a<nSize; a++)
		{
			pImpPolygon3D->pPointAry[a].X() = rXPoly[a].X();
			pImpPolygon3D->pPointAry[a].Y() = -rXPoly[a].Y();
			pImpPolygon3D->pPointAry[a].Z() = 0.0;
		}
	}

	pImpPolygon3D->nPoints = nSize;

	CheckClosed();
}

/*************************************************************************
|*
|* Closed-Status der Einzelpolygone korrigieren
|*
\************************************************************************/

void Polygon3D::CheckClosed()
{
	if(pImpPolygon3D->nPoints 
		&& pImpPolygon3D->pPointAry[0] == pImpPolygon3D->pPointAry[pImpPolygon3D->nPoints - 1])
	{
		pImpPolygon3D->bClosed = TRUE;
		pImpPolygon3D->nPoints--;
	}
}

/*************************************************************************
|*
|* Destruktor
|*
\************************************************************************/

Polygon3D::~Polygon3D()
{
	if( pImpPolygon3D->nRefCount > 1 )
		pImpPolygon3D->nRefCount--;
	else
		delete pImpPolygon3D;
}

/*************************************************************************
|*
|* Closed-Zustand des Polygons testen
|*
\************************************************************************/

BOOL Polygon3D::IsClosed() const
{
	return pImpPolygon3D->bClosed;
}

/*************************************************************************
|*
|* Closed-Zustand des Polygons aendern
|*
\************************************************************************/

void Polygon3D::SetClosed(BOOL bNew)
{
	if(bNew != IsClosed())
	{
		CheckReference();
		pImpPolygon3D->bClosed = bNew;
	}
}

/*************************************************************************
|*
|* Referenzzaehler desImpPoly3D pruefen und ggf. von diesem abkoppeln
|*
\************************************************************************/

void Polygon3D::CheckReference()
{
	if(pImpPolygon3D->nRefCount > 1)
	{
		pImpPolygon3D->nRefCount--;
		pImpPolygon3D = new ImpPolygon3D(*pImpPolygon3D);
	}
}

/*************************************************************************
|*
|* neue Groesse setzen
|*
\************************************************************************/

void Polygon3D::SetSize(UINT16 nNewSize)
{
	CheckReference();
	pImpPolygon3D->Resize(nNewSize);
}

/*************************************************************************
|*
|* Groesse zurueckgeben
|*
\************************************************************************/

UINT16 Polygon3D::GetSize() const
{
	pImpPolygon3D->CheckPointDelete();
	return pImpPolygon3D->nSize;
}

/*************************************************************************
|*
|* Laenge der Polygonkante zurueckgeben
|*
\************************************************************************/

double Polygon3D::GetLength() const
{
	double fRetval = 0.0;
	Vector3D *pLast = &pImpPolygon3D->pPointAry[
		IsClosed() ? pImpPolygon3D->nPoints - 1 : 0];

	for(UINT16 a=IsClosed() ? 0 : 1;a<pImpPolygon3D->nPoints;a++)
	{
		Vector3D *pCandidate = &pImpPolygon3D->pPointAry[a];
		Vector3D aVec = *pCandidate - *pLast;
		fRetval += aVec.GetLength();
		pLast = pCandidate;
	}

	return fRetval;
}

/*************************************************************************
|*
|* Anzahl der belegten Punkte setzen
|*
\************************************************************************/

void Polygon3D::SetPointCount(UINT16 nPoints)
{
	pImpPolygon3D->CheckPointDelete();
	CheckReference();

	if(pImpPolygon3D->nSize < nPoints)
		pImpPolygon3D->Resize(nPoints);

	if(nPoints < pImpPolygon3D->nPoints)
	{
		UINT16 nSize = pImpPolygon3D->nPoints - nPoints;
		memset(&pImpPolygon3D->pPointAry[nPoints], 0, nSize * sizeof(Vector3D));
	}
	pImpPolygon3D->nPoints = nPoints;
}

/*************************************************************************
|*
|* Anzahl der belegten Punkte zurueckgeben
|*
\************************************************************************/

UINT16 Polygon3D::GetPointCount() const
{
	pImpPolygon3D->CheckPointDelete();
	return pImpPolygon3D->nPoints;
}

/*************************************************************************
|*
|* Polygonpunkte entfernen
|*
\************************************************************************/

void Polygon3D::Remove(UINT16 nPos, UINT16 nCount)
{
	CheckReference();
	pImpPolygon3D->Remove(nPos, nCount);
}

/*************************************************************************
|*
|* const-Arrayoperator
|*
\************************************************************************/

const Vector3D& Polygon3D::operator[]( UINT16 nPos ) const
{
	DBG_ASSERT(nPos < pImpPolygon3D->nPoints, "Ungueltiger Index bei const-Arrayzugriff auf Polygon3D");
	pImpPolygon3D->CheckPointDelete();
	return pImpPolygon3D->pPointAry[nPos];
}

/*************************************************************************
|*
|* Arrayoperator
|*
\************************************************************************/

Vector3D& Polygon3D::operator[]( UINT16 nPos )
{
	pImpPolygon3D->CheckPointDelete();
	CheckReference();

	if(nPos >= pImpPolygon3D->nSize)
	{
		DBG_ASSERT(pImpPolygon3D->nResize, "Ungueltiger Index bei Arrayzugriff auf Polygon3D");
		pImpPolygon3D->Resize(nPos + 1, FALSE);
	}

	if(nPos >= pImpPolygon3D->nPoints)
		pImpPolygon3D->nPoints = nPos + 1;

	return pImpPolygon3D->pPointAry[nPos];
}

/*************************************************************************
|*
|* Zuweisungsoperator
|*
\************************************************************************/

Polygon3D& Polygon3D::operator=(const Polygon3D& rPoly3D)
{
	pImpPolygon3D->CheckPointDelete();

	rPoly3D.pImpPolygon3D->nRefCount++;

	if(pImpPolygon3D->nRefCount > 1)
		pImpPolygon3D->nRefCount--;
	else
		delete pImpPolygon3D;

	pImpPolygon3D = rPoly3D.pImpPolygon3D;

	return *this;
}

/*************************************************************************
|*
|* Vergleichsoperator
|*
\************************************************************************/

BOOL Polygon3D::operator==(const Polygon3D& rPoly3D) const
{
	pImpPolygon3D->CheckPointDelete();
	
	if(rPoly3D.pImpPolygon3D == pImpPolygon3D)
		return TRUE;

	if(pImpPolygon3D->nPoints != rPoly3D.pImpPolygon3D->nPoints)
		return FALSE;

	// point-to point compare necessary
	for(UINT16 a=0;a<pImpPolygon3D->nPoints;a++)
		if(pImpPolygon3D->pPointAry[a] != rPoly3D.pImpPolygon3D->pPointAry[a])
			return FALSE;

	return TRUE;
}

/*************************************************************************
|*
|* Ungleichoperator
|*
\************************************************************************/

BOOL Polygon3D::operator!=( const Polygon3D& rPoly3D ) const
{
	pImpPolygon3D->CheckPointDelete();

	if(rPoly3D.pImpPolygon3D != pImpPolygon3D)
		return TRUE;

	if(pImpPolygon3D->nPoints != rPoly3D.pImpPolygon3D->nPoints)
		return TRUE;

	// point-to point compare necessary
	for(UINT16 a=0;a<pImpPolygon3D->nPoints;a++)
		if(pImpPolygon3D->pPointAry[a] != rPoly3D.pImpPolygon3D->pPointAry[a])
			return TRUE;

	return FALSE;
}

/*************************************************************************
|*
|* aus einem Stream lesen
|*
*************************************************************************/

SvStream& operator>>(SvStream& rIStream, Polygon3D& rPoly3D)
{
	DBG_CHKOBJ(&rPoly3D, Polygon3D, NULL);
	UINT16 nPntCnt;

	rPoly3D.pImpPolygon3D->CheckPointDelete();

	// Anzahl der Punkte einlesen und Array erzeugen
	rIStream >> nPntCnt;
	rPoly3D.pImpPolygon3D->nPoints = nPntCnt;

	if(rPoly3D.pImpPolygon3D->nRefCount != 1)
	{
		if(rPoly3D.pImpPolygon3D->nRefCount)
			rPoly3D.pImpPolygon3D->nRefCount--;
		rPoly3D.pImpPolygon3D = new ImpPolygon3D(nPntCnt);
	}
	else
		rPoly3D.pImpPolygon3D->Resize(nPntCnt);

	Vector3D* pPoint = rPoly3D.pImpPolygon3D->pPointAry;

	for(UINT16 i = 0; i < nPntCnt; i++)
		rIStream >> *pPoint++;

	// auf geschlossenheit pruefen
	if(*rPoly3D.pImpPolygon3D->pPointAry == *(pPoint-1))
	{
		rPoly3D.pImpPolygon3D->bClosed = TRUE;
		rPoly3D.pImpPolygon3D->nPoints = nPntCnt-1;
	}

	INT16 nTmp;
	rIStream >> nTmp; // war mal nConvexPoint
	rIStream >> nTmp; // alt: rPoly3D.nOuterPoly;

	return rIStream;
}

/*************************************************************************
|*
|* in einem Stream speichern
|*
*************************************************************************/

SvStream& operator<<(SvStream& rOStream, const Polygon3D& rPoly3D)
{
	DBG_CHKOBJ(&rPoly3D, Polygon3D, NULL);

	rPoly3D.pImpPolygon3D->CheckPointDelete();

	Vector3D* pPoint = rPoly3D.pImpPolygon3D->pPointAry;
	UINT16 nPntCnt = rPoly3D.GetPointCount();

	if(rPoly3D.pImpPolygon3D->bClosed)
	{
		nPntCnt++;
		rOStream << nPntCnt;
		nPntCnt--;
	}
	else
	{
		rOStream << nPntCnt;
	}

	for (UINT16 i = 0; i < nPntCnt; i++)
		rOStream << *pPoint++;

	if(rPoly3D.pImpPolygon3D->bClosed)
	{
		// ersten Punkt nochmal, um polygon auf geschlossen zu zwingen
		pPoint = rPoly3D.pImpPolygon3D->pPointAry;
		rOStream << *pPoint;
	}

	rOStream << (INT16)0; // war mal nConvexPoint
	rOStream << (INT16)-1; // alt: rPoly3D.nOuterPoly;

	return rOStream;
}

/*************************************************************************
|*
|* Laufrichtung des Polygons feststellen
|*
\************************************************************************/

BOOL Polygon3D::IsClockwise(const Vector3D &rNormal) const
{
	double fZValue = rNormal.Scalar(GetNormal());
	return (fZValue >= 0.0);
}

/*************************************************************************
|*
|* Eine garantier die Orientierung wiederspiegelnde Ecke des Polygons
|* liefern
|*
\************************************************************************/

UINT16 Polygon3D::GetHighestEdge() const
{
	UINT16 nRetval = 0;
	Vector3D *pHighest = &pImpPolygon3D->pPointAry[nRetval];
	for(UINT16 a=1;a<pImpPolygon3D->nPoints;a++)
	{
		Vector3D *pCandidate = &pImpPolygon3D->pPointAry[a];
		if(pCandidate->X() <= pHighest->X())
		{
			if(pCandidate->X() < pHighest->X())
			{
				pHighest = pCandidate;
				nRetval = a;
			}
			else
			{
				if(pCandidate->Y() <= pHighest->Y())
				{
					if(pCandidate->Y() < pHighest->Y())
					{
						pHighest = pCandidate;
						nRetval = a;
					}
					else
					{
						if(pCandidate->Z() < pHighest->Z())
						{
							pHighest = pCandidate;
							nRetval = a;
						}
					}
				}
			}
		}
	}
	return nRetval;
}

/*************************************************************************
|*
|* Normale des Polygons liefern
|*
|* Dabei eine Ecke waehlen, die die spezifische Orientierung des Polygons
|* besitzt und mit gleichen Punkten rechnen!
|*
\************************************************************************/

Vector3D Polygon3D::GetNormal() const
{
	Vector3D aNormal(0.0, 0.0, -1.0);

	if(pImpPolygon3D->nPoints > 2)
	{
		// HighestEdge bestimmen
		UINT16 nPntCnt = pImpPolygon3D->nPoints;
		UINT16 nHighest = GetHighestEdge();
		const Vector3D &rHighest = (*this)[nHighest];

		// Vorgaenger bestimmen
		UINT16 nPrev(nHighest);
		do {
			nPrev = (nPrev == 0) ? nPntCnt-1 : nPrev-1;
		} while((*this)[nPrev] == rHighest && nPrev != nHighest);
		const Vector3D &rPrev = (*this)[nPrev];

		// Nachfolger bestimmen
		UINT16 nNext(nHighest);
		do {
			nNext = (nNext == nPntCnt-1) ? 0 : nNext+1;
		} while((*this)[nNext] == rHighest && nNext != nHighest);
		const Vector3D &rNext = (*this)[nNext];

		// Fehlerfaelle abfangen
		if(rHighest == rPrev || rHighest == rNext || rPrev == rNext)
			return aNormal;

		// Normale bilden
		aNormal = (rPrev - rHighest)|(rNext - rHighest);
		
		// get length
		double fLen = aNormal.GetLength();
		
		// correct length
		if(fabs(fLen) < SMALL_DVALUE)
			fLen = 0.0;
		
		if(0.0 != fLen)
		{
			// use own normalize here since we already know the vector length
			// with square root applied
			// aNormal.Normalize();
			if(1.0 != fLen)
			{
				// normalize
				double fFac = 1.0 / fLen;
				aNormal = Vector3D(aNormal.X() * fFac, aNormal.Y() * fFac, aNormal.Z() * fFac);
			}
		}
		else
		{
			// points are on one line, use default normal
			aNormal = Vector3D(0.0, 0.0, -1.0);
		}
	}
	return aNormal;
}

/*************************************************************************
|*
|* Mittelpunkt des Polygons (in Bezug auf die Einzelpunkte) berechnen
|*
\************************************************************************/

Vector3D Polygon3D::GetMiddle() const
{
	Vector3D aMiddle;
	UINT16 nPntCnt = pImpPolygon3D->nPoints;

	for(UINT16 i = 0; i < nPntCnt; i++)
		aMiddle += pImpPolygon3D->pPointAry[i];

	aMiddle /= (double)nPntCnt;

	return aMiddle;
}

/*************************************************************************
|*
|* Laufrichtung des Polygons umkehren
|*
\************************************************************************/

void Polygon3D::FlipDirection()
{
	pImpPolygon3D->CheckPointDelete();
	CheckReference();

	UINT16 nPntCnt = pImpPolygon3D->nPoints;
	UINT16 nCnt = nPntCnt / 2;
	Vector3D* pBeg = pImpPolygon3D->pPointAry;
	Vector3D* pEnd = pBeg + nPntCnt - 1;

	for(UINT16 i = 0; i < nCnt; i++)
	{
		Vector3D aTmp = *pBeg;
		*pBeg++ = *pEnd;
		*pEnd-- = aTmp;
	}
}

/*************************************************************************
|*
|* Polygonpunkte mit uebergebener Matrix transformieren
|*
\************************************************************************/

void Polygon3D::Transform(const Matrix4D& rTfMatrix)
{
	pImpPolygon3D->CheckPointDelete();
	CheckReference();

	Vector3D* pPoint = pImpPolygon3D->pPointAry;
	UINT16 nPntCnt = pImpPolygon3D->nPoints;

	for(UINT16 i = 0; i < nPntCnt; i++)
		*pPoint++ *= rTfMatrix;
}

/*************************************************************************
|*
|* doppelte aufeinanderfolgende Polygonpunkte entfernen; ein
|* geschlossenes Polygon wird dabei ggf. "geoeffnet"
|*
\************************************************************************/

void Polygon3D::RemoveDoublePoints()
{
	pImpPolygon3D->CheckPointDelete();
	CheckReference();

	UINT16 nPntCnt = pImpPolygon3D->nPoints;

	if(nPntCnt)
	{
		Vector3D* pPoint = pImpPolygon3D->pPointAry;
		Vector3D aP0 = *pPoint;

		pPoint += nPntCnt;

		while(*--pPoint == aP0 && nPntCnt > 3)
		{
			nPntCnt--;
			pImpPolygon3D->bClosed = TRUE;
		}

		for(UINT16 i = nPntCnt-1; i > 0 && nPntCnt > 3; i--)
		{
			if(*pPoint == *(pPoint-1))
			{
				pImpPolygon3D->Remove(i, 1);
				nPntCnt--;
			}
			pPoint--;
		}

		SetPointCount(nPntCnt);
	}
	else
	{
		DBG_ASSERT(FALSE, "Empty polygon used!");
	}
}

/*************************************************************************
|*
|* Ueberlappen sich das aktuelle und das angegebene Polygon ?
|*
\************************************************************************/

BOOL Polygon3D::DoesBoundVolumeOverlap(const Polygon3D& rOrig, UINT16 nDegreeFlag) const
{
	Volume3D aVolumeOrig = rOrig.GetPolySize();
	Volume3D aVolumeThis = GetPolySize();

	BOOL bOverlapX(TRUE);
	if(nDegreeFlag & DEGREE_FLAG_X)
		bOverlapX = BOOL(aVolumeOrig.MinVec().X() < aVolumeThis.MaxVec().X() && aVolumeOrig.MaxVec().X() > aVolumeThis.MinVec().X());

	BOOL bOverlapY(TRUE);
	if(nDegreeFlag & DEGREE_FLAG_Y)
		bOverlapY = BOOL(aVolumeOrig.MinVec().Y() < aVolumeThis.MaxVec().Y() && aVolumeOrig.MaxVec().Y() > aVolumeThis.MinVec().Y());

	BOOL bOverlapZ(TRUE);
	if(nDegreeFlag & DEGREE_FLAG_Z)
		bOverlapZ = BOOL(aVolumeOrig.MinVec().Z() < aVolumeThis.MaxVec().Z() && aVolumeOrig.MaxVec().Z() > aVolumeThis.MinVec().Z());

	return (bOverlapX && bOverlapY && bOverlapZ);
}

BOOL Polygon3D::DoesOverlap(const Polygon3D& rOrig, UINT16 nDegreeFlag) const
{
	BOOL bRetval = DoesBoundVolumeOverlap(rOrig, nDegreeFlag);

	if(bRetval)
	{
		// Schneiden sich die Polygone auch?
		BOOL bPolyCut = DoesCut(rOrig, nDegreeFlag);

		// Falls Ueberlappung von Einzelpolygonen
		if(bPolyCut)
		{
			bRetval = TRUE;
		}
		else
		{
			// Polygone koennen sich noch umschliessen
			if((IsClosed() && IsInside(rOrig[0]))
				|| (rOrig.IsClosed() && rOrig.IsInside((*this)[0])))
				bRetval = TRUE;
			else
				bRetval = FALSE;
		}
	}

	return bRetval;
}

/*************************************************************************
|*
|* Existiert ein Schnitt zwischen den Polys?
|*
\************************************************************************/

BOOL Polygon3D::DoesCut(const Polygon3D& rOrig, UINT16 nDegreeFlag) const
{
	BOOL bRetval(FALSE);

	UINT16 nEndLoopA = IsClosed() ? pImpPolygon3D->nPoints : pImpPolygon3D->nPoints - 1;
	for(UINT16 a=0;!bRetval && a < nEndLoopA; a++)
	{
		UINT16 nEndLoopB = rOrig.IsClosed() ? rOrig.GetPointCount() : rOrig.GetPointCount() - 1;
		for(UINT16 b=0;!bRetval && b < nEndLoopB; b++)
		{
			UINT16 nCut = FindCut(a, rOrig, b);
			if(nCut)
				bRetval = TRUE;
		}
	}

	return bRetval;
}

/*************************************************************************
|*
|* Crossings Test for point and whole polygon
|*
\************************************************************************/

BOOL Polygon3D::IsInside(const Vector3D& rPnt, BOOL bWithBorder) const
{
	BOOL bInside(FALSE);
	UINT16 nNumPoint(GetPointCount());
	Vector3D* pPoints = pImpPolygon3D->pPointAry;

	for(UINT16 a=0;a<nNumPoint;a++)
	{
		if(bWithBorder
			&& (fabs(pPoints[a].X() - rPnt.X()) < SMALL_DVALUE)
			&& (fabs(pPoints[a].Y() - rPnt.Y()) < SMALL_DVALUE))
			return TRUE;

		UINT16 nPrev((!a) ? nNumPoint-1 : a-1);

		if((pPoints[nPrev].Y() - rPnt.Y() > -SMALL_DVALUE) != (pPoints[a].Y() - rPnt.Y() > -SMALL_DVALUE))
		{
			BOOL bXFlagOld(pPoints[nPrev].X() - rPnt.X() > -SMALL_DVALUE);

			if(bXFlagOld == (pPoints[a].X() - rPnt.X() > -SMALL_DVALUE))
			{
				if(bXFlagOld)
					bInside = !bInside;
			}
			else
			{
				double fCmp = 
					pPoints[a].X() - (pPoints[a].Y() - rPnt.Y()) *
					(pPoints[nPrev].X() - pPoints[a].X()) /
					(pPoints[nPrev].Y() - pPoints[a].Y());

				if((bWithBorder && fCmp > rPnt.X()) || (!bWithBorder && fCmp - rPnt.X() > -SMALL_DVALUE))
					bInside = !bInside;
			}
		}
	}

	return bInside;
}

BOOL Polygon3D::IsInside(const Polygon3D& rPoly, BOOL bWithBorder) const
{
	UINT16 nPnt(rPoly.GetPointCount());

	for(UINT16 a=0;a<nPnt;a++)
		if(!IsInside(rPoly[a], bWithBorder))
			return FALSE;

	return TRUE;
}

/*************************************************************************
|*
|* XPolygon herausgeben
|*
\************************************************************************/

XPolygon Polygon3D::GetXPolygon() const
{
	XPolygon aXPolygon(GetPolygon());
	return(aXPolygon);
}

/*************************************************************************
|*
|* FG: Identisch wie oben, das Polygon herausgeben
|*
\************************************************************************/

Polygon Polygon3D::GetPolygon() const
{
	BOOL bClosed = IsClosed();
	UINT16 nSize = pImpPolygon3D->nPoints;
	if(bClosed)
		nSize++;
	Polygon aPolygon(nSize);
	Vector3D* pVec3D = pImpPolygon3D->pPointAry;

	if (pVec3D)
	{
		UINT16 i;
		for (i = 0; i < pImpPolygon3D->nPoints; i++)
		{
			// X und Y uebernehmen, Z vernachlaessigen
			aPolygon.SetPoint(Point((long)  pVec3D[i].X(),
									(long) -pVec3D[i].Y()), i);
		}
		if(bClosed)
			aPolygon.SetPoint(Point((long)  pVec3D[0].X(),
									(long) -pVec3D[0].Y()), i);
	}
	else
	{
		for (UINT16 i = 0; i < nSize; i++)
		{
			aPolygon.SetPoint(Point(), i);
		}
	}
	aPolygon.SetSize(nSize);
	return(aPolygon);
}

/*************************************************************************
|*
|* Ausdehnung des Polygons ermitteln
|*
\************************************************************************/

Volume3D Polygon3D::GetPolySize() const
{
	Volume3D aRetval;
	UINT16 nPntCnt = pImpPolygon3D->nPoints;

	aRetval.Reset();
	for(UINT16 i = 0; i < nPntCnt; i++)
		aRetval.Union((*this)[i]);

	return aRetval;
}

/*************************************************************************
|*
|* Flaeche des Polygons ermitteln
|*
\************************************************************************/

double Polygon3D::GetPolyArea() const
{
	Vector3D aNormal = GetNormal();
	return GetPolyArea(aNormal);
}

double Polygon3D::GetPolyArea(const Vector3D& rNormal) const
{
	double fRetval = 0.0;
	UINT16 nPntCnt = pImpPolygon3D->nPoints;

	if(nPntCnt > 2)
	{
		const Vector3D& rFirst = (*this)[0];
		Vector3D aLastVector = (*this)[1] - rFirst;

		for (UINT16 i = 2; i < nPntCnt; i++)
		{
			const Vector3D& rPoint = (*this)[i];
			Vector3D aNewVec = rPoint - rFirst;
			Vector3D aArea = aLastVector;
			aArea |= aNewVec;
			fRetval += (rNormal.Scalar(aArea)) / 2.0;
		}
	}
	return fabs(fRetval);
}

/*************************************************************************
|*
|* Schnitt zwischen den von den Punkten angegebenen Kanten ermitteln.
|* Dabei ist der Rueckgabewert != 0.0, wenn der Schnitt innerhalb
|* der Parameterbereiche der Kanten liegt und gibt den Wert ]0.0, 1.0]
|* innerhalb der ersten Kante an.
|*
\************************************************************************/

UINT16 Polygon3D::FindCut(UINT16 nEdge1, UINT16 nEdge2, 
	UINT16 nCutFlags, double* pCut1, double* pCut2) const
{
	UINT16 nRetval(0);
	UINT16 nPntCnt = pImpPolygon3D->nPoints;

	if(nEdge1 < nPntCnt && nEdge2 < nPntCnt && nEdge1 != nEdge2)
	{
		UINT16 nEnd1 = (nEdge1 == nPntCnt-1) ? 0 : nEdge1+1;
		UINT16 nEnd2 = (nEdge2 == nPntCnt-1) ? 0 : nEdge2+1;

		nRetval = FindCut(
			(*this)[nEdge1], (*this)[nEnd1] - (*this)[nEdge1],
			(*this)[nEdge2], (*this)[nEnd2] - (*this)[nEdge2],
			nCutFlags, pCut1, pCut2);
	}

	return nRetval;
}

/*************************************************************************
|*
|* Diese Version arbeitet mit der Kante nEdge1 aus dem lokalen
|* Polygon und nEdge2 aus dem uebergebenen
|*
\************************************************************************/

UINT16 Polygon3D::FindCut(UINT16 nEdge1, const Polygon3D& rPoly3D, UINT16 nEdge2,
	UINT16 nCutFlags, double* pCut1, double* pCut2) const
{
	UINT16 nRetval(0);
	UINT16 nPntCnt1 = pImpPolygon3D->nPoints;

	if(nEdge1 < nPntCnt1)
	{
		UINT16 nPntCnt2 = rPoly3D.GetPointCount();
		if(nEdge2 < nPntCnt2)
		{
			UINT16 nEnd1 = (nEdge1 == nPntCnt1-1) ? 0 : nEdge1+1;
			UINT16 nEnd2 = (nEdge2 == nPntCnt2-1) ? 0 : nEdge2+1;

			nRetval = FindCut(
				(*this)[nEdge1], (*this)[nEnd1] - (*this)[nEdge1],
				rPoly3D[nEdge2], rPoly3D[nEnd2] - rPoly3D[nEdge2],
				nCutFlags, pCut1, pCut2);
		}
	}

	return nRetval;
}

/*************************************************************************
|*
|* test if point is on line in range ]0.0..1.0[ without
|* the points. If so, return TRUE and put the parameter
|* value in pCut (if provided)
|*
\************************************************************************/

BOOL Polygon3D::FindPointInLine(
	const Vector3D& rPoint, 
	const Vector3D& rEdgeStart, 
	const Vector3D& rEdgeDelta, 
	double* pCut)
{
	BOOL bDeltaXIsZero(fabs(rEdgeDelta.X()) < SMALL_DVALUE);
	BOOL bDeltaYIsZero(fabs(rEdgeDelta.Y()) < SMALL_DVALUE);

	if(bDeltaXIsZero && bDeltaYIsZero)
	{
		return FALSE;
	}
	else if(bDeltaXIsZero)
	{
		if(fabs(rPoint.X() - rEdgeStart.X()) < SMALL_DVALUE)
		{
			double fValue = (rPoint.Y() - rEdgeStart.Y()) / rEdgeDelta.Y();
			if(fValue >= SMALL_DVALUE && fValue <= (1.0 - SMALL_DVALUE))
			{
				if(pCut)
					*pCut = fValue;
				return TRUE;
			}
		}
	}
	else if(bDeltaYIsZero)
	{
		if(fabs(rPoint.Y() - rEdgeStart.Y()) < SMALL_DVALUE)
		{
			double fValue = (rPoint.X() - rEdgeStart.X()) / rEdgeDelta.X();
			if(fValue >= SMALL_DVALUE && fValue <= (1.0 - SMALL_DVALUE))
			{
				if(pCut)
					*pCut = fValue;
				return TRUE;
			}
		}
	}
	else
	{
		double fTOne = (rPoint.X() - rEdgeStart.X()) / rEdgeDelta.X();
		double fTTwo = (rPoint.Y() - rEdgeStart.Y()) / rEdgeDelta.Y();
		if(fabs(fTOne - fTTwo) < SMALL_DVALUE)
		{
			// same representation, point is on line
			double fValue = (fTOne + fTTwo) / 2.0;
			if(fValue >= SMALL_DVALUE && fValue <= (1.0 - SMALL_DVALUE))
			{
				// point is inside line bounds, too
				if(pCut)
					*pCut = fValue;
				return TRUE;
			}
		}
	}

	return FALSE;
}

/*************************************************************************
|*
|* Diese Version nimmt die Startpunkte und Vektoren (relative Angabe
|* des Endpunktes) zweier Kanten
|*
\************************************************************************/

UINT16 Polygon3D::FindCut(
	const Vector3D& rEdge1Start, const Vector3D& rEdge1Delta,
	const Vector3D& rEdge2Start, const Vector3D& rEdge2Delta,
	UINT16 nCutFlags,
	double* pCut1, double* pCut2)
{
	UINT16 nRetval(0);
	double fCut1(0.0);
	double fCut2(0.0);
	BOOL bFinished(!((BOOL)(nCutFlags & CUTFLAG_ALL)));

	// test for same points?
	if(!bFinished 
		&& (nCutFlags & (CUTFLAG_START1|CUTFLAG_END1))
		&& (nCutFlags & (CUTFLAG_START2|CUTFLAG_END2)))
	{
		// same startpoint?
		if(!bFinished
			&& ((nCutFlags & (CUTFLAG_START1|CUTFLAG_START2)) == (CUTFLAG_START1|CUTFLAG_START2))
			&& (fabs(rEdge1Start.X() - rEdge2Start.X()) < SMALL_DVALUE)
			&& (fabs(rEdge1Start.Y() - rEdge2Start.Y()) < SMALL_DVALUE))
		{
			bFinished = TRUE;
			nRetval = (CUTFLAG_START1|CUTFLAG_START2);
		}

		// same endpoint?
		if(!bFinished
			&& ((nCutFlags & (CUTFLAG_END1|CUTFLAG_END2)) == (CUTFLAG_END1|CUTFLAG_END2))
			&& (fabs(rEdge1Start.X() + rEdge1Delta.X() - rEdge2Start.X() + rEdge2Delta.X()) < SMALL_DVALUE)
			&& (fabs(rEdge1Start.Y() + rEdge1Delta.Y() - rEdge2Start.Y() + rEdge2Delta.Y()) < SMALL_DVALUE))
		{
			bFinished = TRUE;
			nRetval = (CUTFLAG_END1|CUTFLAG_END2);
			fCut1 = fCut2 = 1.0;
		}

		// startpoint1 == endpoint2?
		if(!bFinished
			&& ((nCutFlags & (CUTFLAG_START1|CUTFLAG_END2)) == (CUTFLAG_START1|CUTFLAG_END2))
			&& (fabs(rEdge1Start.X() - rEdge2Start.X() + rEdge2Delta.X()) < SMALL_DVALUE)
			&& (fabs(rEdge1Start.Y() - rEdge2Start.Y() + rEdge2Delta.Y()) < SMALL_DVALUE))
		{
			bFinished = TRUE;
			nRetval = (CUTFLAG_START1|CUTFLAG_END2);
			fCut1 = 0.0;
			fCut2 = 1.0;
		}

		// startpoint2 == endpoint1?
		if(!bFinished
			&& ((nCutFlags & (CUTFLAG_START2|CUTFLAG_END1)) == (CUTFLAG_START2|CUTFLAG_END1))
			&& (fabs(rEdge1Start.X() + rEdge1Delta.X() - rEdge2Start.X()) < SMALL_DVALUE)
			&& (fabs(rEdge1Start.Y() + rEdge1Delta.Y() - rEdge2Start.Y()) < SMALL_DVALUE))
		{
			bFinished = TRUE;
			nRetval = (CUTFLAG_START2|CUTFLAG_END1);
			fCut1 = 1.0;
			fCut2 = 0.0;
		}
	}

	if(!bFinished 
		&& (nCutFlags & CUTFLAG_LINE))
	{
		if(!bFinished 
			&& (nCutFlags & CUTFLAG_START1))
		{
			// start1 on line 2 ?
			if(FindPointInLine(rEdge1Start, rEdge2Start, rEdge2Delta, &fCut2))
			{
				bFinished = TRUE;
				nRetval = (CUTFLAG_LINE|CUTFLAG_START1);
			}
		}
		
		if(!bFinished 
			&& (nCutFlags & CUTFLAG_START2))
		{
			// start2 on line 1 ?
			if(FindPointInLine(rEdge2Start, rEdge1Start, rEdge1Delta, &fCut1))
			{
				bFinished = TRUE;
				nRetval = (CUTFLAG_LINE|CUTFLAG_START2);
			}
		}

		if(!bFinished 
			&& (nCutFlags & CUTFLAG_END1))
		{
			// end1 on line 2 ?
			if(FindPointInLine(rEdge1Start + rEdge1Delta, rEdge2Start, rEdge2Delta, &fCut2))
			{
				bFinished = TRUE;
				nRetval = (CUTFLAG_LINE|CUTFLAG_END1);
			}
		}
		
		if(!bFinished 
			&& (nCutFlags & CUTFLAG_END2))
		{
			// end2 on line 1 ?
			if(FindPointInLine(rEdge2Start + rEdge2Delta, rEdge1Start, rEdge1Delta, &fCut1))
			{
				bFinished = TRUE;
				nRetval = (CUTFLAG_LINE|CUTFLAG_END2);
			}
		}

		if(!bFinished)
		{
			// cut in line1, line2 ?
			fCut1 = (rEdge1Delta.X() * rEdge2Delta.Y()) - (rEdge1Delta.Y() * rEdge2Delta.X());
			if(fabs(fCut1) > SMALL_DVALUE)
			{
				fCut1 = (rEdge2Delta.Y() * (rEdge2Start.X() - rEdge1Start.X())
					+ rEdge2Delta.X() * (rEdge1Start.Y() - rEdge2Start.Y())) / fCut1;

				// inside parameter range edge1 AND fCut2 ist calcable
				if(fCut1 >= SMALL_DVALUE 
					&& fCut1 <= (1.0 - SMALL_DVALUE)
					&& (fabs(rEdge2Delta.X()) > SMALL_DVALUE || fabs(rEdge2Delta.Y()) > SMALL_DVALUE))
				{
					if(fabs(rEdge2Delta.X()) > fabs(rEdge2Delta.Y()))
					{
						fCut2 = (rEdge1Start.X() + fCut1
							* rEdge1Delta.X() - rEdge2Start.X()) / rEdge2Delta.X();
					}
					else
					{
						fCut2 = (rEdge1Start.Y() + fCut1
							* rEdge1Delta.Y() - rEdge2Start.Y()) / rEdge2Delta.Y();
					}

					// inside parameter range edge2, too
					if(fCut2 >= SMALL_DVALUE && fCut2 <= (1.0 - SMALL_DVALUE))
					{
						bFinished = TRUE;
						nRetval = CUTFLAG_LINE;
					}
				}
			}
		}
	}

	// copy values if wanted
	if(pCut1)
		*pCut1 = fCut1;
	if(pCut2)
		*pCut2 = fCut2;

	return nRetval;
}

/*************************************************************************
|*
|* Orientierung im Punkt nIndex liefern
|*
\************************************************************************/

BOOL Polygon3D::GetPointOrientation(UINT16 nIndex) const
{
	UINT16 nPntCnt = pImpPolygon3D->nPoints;
	BOOL bRetval(TRUE);

	if(nIndex < nPntCnt)
	{
		const Vector3D& rMid = (*this)[nIndex];
		const Vector3D& rPre = (*this)[(nIndex == 0) ? nPntCnt-1 : nIndex-1];
		const Vector3D& rPos = (*this)[(nIndex == nPntCnt-1) ? 0 : nIndex+1];

		Vector3D aNormal = (rPre - rMid)|(rPos - rMid);
		bRetval = (aNormal.Z() > 0.0) ? TRUE : FALSE;
	}

	return bRetval;
}

/*************************************************************************
|*
|* get position on polypos, with clipping to start/end
|*
\************************************************************************/
 
Vector3D Polygon3D::GetPosition(double fPos) const
{
	Vector3D aRetval((*this)[0]);

	if(fPos <= 0.0 || pImpPolygon3D->nPoints < 2)
		return aRetval;

	double fLen = GetLength();

	if(fPos >= fLen)
	{
		aRetval = (*this)[pImpPolygon3D->nPoints - 1];
		return aRetval;
	}

	UINT16 nPos(0);
	Vector3D aPart((*this)[0] - (*this)[1]);
	double fPartLen = aPart.GetLength();

	while(fPos > fPartLen)
	{
		fPos -= fPartLen;
		nPos++;
		aPart = Vector3D((*this)[nPos] - (*this)[nPos+1]);
		fPartLen = aPart.GetLength();
	}

	aRetval.CalcInBetween((*this)[nPos], (*this)[nPos+1], fPos / fPartLen);

	return aRetval;
}

//////////////////////////////////////////////////////////////////////////////
// create a expanded or compresssed poly with exactly nNum Points
//
Polygon3D Polygon3D::GetExpandedPolygon(sal_uInt32 nNum)
{
	if(GetPointCount() && nNum && (sal_uInt32)GetPointCount() != nNum)
	{
		Polygon3D aDestPoly((sal_uInt16)nNum);

		// length of step in dest poly
		double fStep = GetLength() / (double)(IsClosed() ? nNum : nNum - 1);

		// actual positions for run
		double fDestPos = 0.0;
		double fSrcPos = 0.0;

		// actual indices for run
		sal_uInt32 nSrcPos = 0;
		sal_uInt32 nSrcPosNext = (nSrcPos+1 == (sal_uInt32)GetPointCount()) ? 0 : nSrcPos + 1;

		// length of next source step
		double fNextSrcLen = ((*this)[(sal_uInt16)nSrcPos] - (*this)[(sal_uInt16)nSrcPosNext]).GetLength();

		for(sal_uInt32 b = 0; b < nNum; b++)
		{
			// calc fDestPos in source
			while(fSrcPos + fNextSrcLen < fDestPos)
			{
				fSrcPos += fNextSrcLen;
				nSrcPos++;
				nSrcPosNext = (nSrcPos+1 == (sal_uInt32)GetPointCount()) ? 0 : nSrcPos + 1;
				fNextSrcLen = ((*this)[(sal_uInt16)nSrcPos] - (*this)[(sal_uInt16)nSrcPosNext]).GetLength();
			}

			// fDestPos is between fSrcPos and (fSrcPos + fNextSrcLen)
			double fLenA = (fDestPos - fSrcPos) / fNextSrcLen;
			Vector3D aOld1 = (*this)[(sal_uInt16)nSrcPos];
			Vector3D aOld2 = (*this)[(sal_uInt16)nSrcPosNext];
			Vector3D aNewPoint;
			aNewPoint.CalcInBetween(aOld1, aOld2, fLenA);

			aDestPoly[(sal_uInt16)b] = aNewPoint;

			// next step
			fDestPos += fStep;
		}

		if(aDestPoly.GetPointCount() >= 3)
			aDestPoly.SetClosed(IsClosed());

		return aDestPoly;
	}
	else
		return *this;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+--------------- ImpPolyPolygon3D -------------------------------------+
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

/*************************************************************************
|*
|* Copy-Konstruktor
|*
\************************************************************************/

ImpPolyPolygon3D::ImpPolyPolygon3D(const ImpPolyPolygon3D& rImpPolyPoly3D) :
	aPoly3DList(rImpPolyPoly3D.aPoly3DList)
{
	nRefCount = 1;
	// Einzelne Elemente duplizieren
	Polygon3D* pPoly3D = aPoly3DList.First();

	while ( pPoly3D )
	{
		aPoly3DList.Replace(new Polygon3D(*(aPoly3DList.GetCurObject())));
		pPoly3D = aPoly3DList.Next();
	}
}

/*************************************************************************
|*
|* Destruktor
|*
\************************************************************************/

ImpPolyPolygon3D::~ImpPolyPolygon3D()
{
	Polygon3D* pPoly3D = aPoly3DList.First();

	while( pPoly3D )
	{
		delete pPoly3D;
		pPoly3D = aPoly3DList.Next();
	}
}

/*************************************************************************
|*
|* Gleichheitsoperator
|*
\************************************************************************/

BOOL ImpPolyPolygon3D::operator==(const ImpPolyPolygon3D& rImpPolyPoly3D) const
{
	UINT16 nCnt = (UINT16) aPoly3DList.Count();
	const Polygon3DList& rCmpList = rImpPolyPoly3D.aPoly3DList;

	if ( nCnt != (UINT16) rCmpList.Count() )
		return FALSE;

	BOOL bEqual = TRUE;

	for ( UINT16 i = 0; i < nCnt && bEqual; i++ )
		bEqual = ( *aPoly3DList.GetObject(i) == *rCmpList.GetObject(i) );

	return bEqual;
}

//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
//+--------------- PolyPolygon3D ----------------------------------------+
//++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

/*************************************************************************
|*
|* Konstruktor
|*
\************************************************************************/

PolyPolygon3D::PolyPolygon3D(UINT16 nInitSize, UINT16 nResize)
{
	DBG_CTOR(PolyPolygon3D, NULL);
	pImpPolyPolygon3D = new ImpPolyPolygon3D(nInitSize, nResize);
}


/*************************************************************************
|*
|* Konstruktor mit Polygon3D
|*
\************************************************************************/

PolyPolygon3D::PolyPolygon3D(const Polygon3D& rPoly3D)
{
	DBG_CTOR(PolyPolygon3D, NULL);
	pImpPolyPolygon3D = new ImpPolyPolygon3D;
	pImpPolyPolygon3D->aPoly3DList.Insert(new Polygon3D(rPoly3D));
}

/*************************************************************************
|*
|* Konstruktor mit PolyPolygon3D
|*
\************************************************************************/

PolyPolygon3D::PolyPolygon3D(const PolyPolygon3D& rPolyPoly3D)
{
	DBG_CTOR(PolyPolygon3D,NULL);
	pImpPolyPolygon3D = rPolyPoly3D.pImpPolyPolygon3D;
	pImpPolyPolygon3D->nRefCount++;
}

/*************************************************************************
|*
|* Konstruktor mit SV-Polygon
|*
\************************************************************************/

PolyPolygon3D::PolyPolygon3D(const Polygon& rPoly, double fScale)
{
	DBG_CTOR(PolyPolygon3D, NULL);
	pImpPolyPolygon3D = new ImpPolyPolygon3D;
	pImpPolyPolygon3D->aPoly3DList. Insert(new Polygon3D(rPoly, fScale));
}

/*************************************************************************
|*
|* Konstruktor mit SV-PolyPolygon
|*
\************************************************************************/

PolyPolygon3D::PolyPolygon3D(const PolyPolygon& rPolyPoly, double fScale)
{
	DBG_CTOR(PolyPolygon3D, NULL);
	pImpPolyPolygon3D = new ImpPolyPolygon3D;
	UINT16 nCnt = rPolyPoly.Count();

	for ( UINT16 i = 0; i < nCnt; i++ )
		pImpPolyPolygon3D->aPoly3DList.Insert(
			new Polygon3D(rPolyPoly.GetObject(i), fScale), LIST_APPEND);
}

/*************************************************************************
|*
|* Konstruktor mit XPolygon
|*
\************************************************************************/

PolyPolygon3D::PolyPolygon3D(const XPolygon& rXPoly, double fScale)
{
	DBG_CTOR(PolyPolygon3D, NULL);
	pImpPolyPolygon3D = new ImpPolyPolygon3D;
	pImpPolyPolygon3D->aPoly3DList.Insert(new Polygon3D(rXPoly, fScale));
}

/*************************************************************************
|*
|* Konstruktor mit XPolyPolygon
|*
\************************************************************************/

PolyPolygon3D::PolyPolygon3D(const XPolyPolygon& rXPolyPoly, double fScale)
{
	DBG_CTOR(PolyPolygon3D, NULL);
	pImpPolyPolygon3D = new ImpPolyPolygon3D;
	UINT16 nCnt = rXPolyPoly.Count();

	for ( UINT16 i = 0; i < nCnt; i++ )
		pImpPolyPolygon3D->aPoly3DList.Insert(
			new Polygon3D(rXPolyPoly.GetObject(i), fScale), LIST_APPEND);
}

/*************************************************************************
|*
|* Closed-Status der Einzelpolygone korrigieren
|*
\************************************************************************/

void PolyPolygon3D::CheckClosed()
{
	for(UINT16 a=0;a<Count();a++)
		(*this)[a].CheckClosed();
}

/*************************************************************************
|*
|* Destruktor
|*
\************************************************************************/

PolyPolygon3D::~PolyPolygon3D()
{
	DBG_DTOR(PolyPolygon3D, NULL);

	if( pImpPolyPolygon3D->nRefCount > 1 )	pImpPolyPolygon3D->nRefCount--;
	else									delete pImpPolyPolygon3D;
}

/*************************************************************************
|*
|* Feststellen, ob ein Punkt innerhalb liegt
|*
\************************************************************************/

BOOL PolyPolygon3D::IsInside(const Vector3D& rPnt, BOOL bWithBorder) const
{
	BOOL bInside(FALSE);

	for(UINT16 a=0;a<Count();a++)
		if((*this)[a].IsInside(rPnt, bWithBorder))
			bInside = !bInside;
	return bInside;
}

BOOL PolyPolygon3D::IsInside(const Polygon3D& rPoly, BOOL bWithBorder) const
{
	BOOL bInside(FALSE);

	for(UINT16 a=0;a<Count();a++)
		if((*this)[a].IsInside(rPoly, bWithBorder))
			bInside = !bInside;
	return bInside;
}

/*************************************************************************
|*
|* Scnittpunkt angegebene Linie mit Polygon
|*
\************************************************************************/

BOOL PolyPolygon3D::GetCutPoint(Vector3D &rCut, const Vector3D &rLeft,
	const Vector3D &rRight) const
{
	BOOL bCutValid = FALSE;
	const Polygon3D& rPoly3D = (*this)[0];
	UINT16 nPntCnt = rPoly3D.GetPointCount();
	if(nPntCnt > 2)
	{
		// Normale und Skalar der Ebenengleichung ermitteln
		Vector3D aVector = rPoly3D[0] - rPoly3D[1];
		Vector3D aNormal;
		UINT16 a=2;
		do
		{
			const Vector3D &rThree = rPoly3D[a++];
			aNormal = aVector;
			aNormal |= rThree - rPoly3D[1];
			aNormal.Normalize();
		} while(a < nPntCnt && aNormal.GetLength() < 0.000000001);
		double fScalar = -(rPoly3D[1].Scalar(aNormal));

		Vector3D aLineVec = rLeft - rRight;
		double fZwi = aNormal.Scalar(aLineVec);

		if(fabs(fZwi) > 0.000000001)
		{
			fZwi = (-fScalar - (rRight.Scalar(aNormal))) / fZwi;
			if(fZwi > 0.000000001 && fZwi < 1.0 - 0.000000001)
			{
				rCut.X() = rRight.X() + (aLineVec.X() * fZwi);
				rCut.Y() = rRight.Y() + (aLineVec.Y() * fZwi);
				rCut.Z() = rRight.Z() + (aLineVec.Z() * fZwi);

				bCutValid = TRUE;
			}
		}
	}
	return bCutValid;
}

/*************************************************************************
|*
|* Referenzzaehler pruefen und ggf. neuen Container erzeugen
|*
\************************************************************************/

void PolyPolygon3D::CheckReference()
{
	if( pImpPolyPolygon3D->nRefCount > 1 )
	{
		pImpPolyPolygon3D->nRefCount--;
		pImpPolyPolygon3D = new ImpPolyPolygon3D(*pImpPolyPolygon3D);
	}
}

/*************************************************************************
|*
|* Polygon in die Liste einfuegen
|*
\************************************************************************/

void PolyPolygon3D::Insert(const Polygon3D& rPoly3D, UINT16 nPos)
{
	CheckReference();
	pImpPolyPolygon3D->aPoly3DList.Insert(new Polygon3D(rPoly3D), nPos);
}

/*************************************************************************
|*
|* PolyPolygon in die Liste einfuegen
|*
\************************************************************************/

void PolyPolygon3D::Insert(const PolyPolygon3D& rPolyPoly3D, UINT16 nPos)
{
	CheckReference();
	UINT16 nCnt = rPolyPoly3D.Count();

	for ( UINT16 i = 0; i < nCnt; i++)
	{
		Polygon3D* pPoly3D = new Polygon3D(rPolyPoly3D[i]);
		pImpPolyPolygon3D->aPoly3DList. Insert(pPoly3D, nPos);

		if ( nPos != POLYPOLY3D_APPEND )
			nPos++;
	}
}

/*************************************************************************
|*
|* Polygon aus der Liste entfernen
|*
\************************************************************************/

Polygon3D PolyPolygon3D::Remove(UINT16 nPos)
{
	CheckReference();
	Polygon3D* pTmpPoly3D = pImpPolyPolygon3D->aPoly3DList.Remove(nPos);
	Polygon3D aPoly3D(*pTmpPoly3D);
	delete pTmpPoly3D;
	return aPoly3D;
}


/*************************************************************************
|*
|* Polygon in der Liste ersetzen
|*
\************************************************************************/

Polygon3D PolyPolygon3D::Replace(const Polygon3D& rPoly3D, UINT16 nPos)
{
	CheckReference();
	Polygon3D* pPoly3D = new Polygon3D(rPoly3D);
	Polygon3D* pTmpPoly3D = pImpPolyPolygon3D->aPoly3DList.Replace(pPoly3D, nPos);
	Polygon3D  aPoly3D(*pTmpPoly3D);
	delete pTmpPoly3D;
	return aPoly3D;
}

/*************************************************************************
|*
|* Referenz auf Polygon zurueckgeben, ggf. neues Polygon einfuegen
|*
\************************************************************************/

const Polygon3D& PolyPolygon3D::GetObject(UINT16 nPos) const
{
	Polygon3D* pPoly3D = pImpPolyPolygon3D->aPoly3DList.GetObject(nPos);

	if ( pPoly3D == NULL )
	{	// Wenn noch kein Polygon an der Position vorhanden, neu erzeugen
		pPoly3D = new Polygon3D;
		pImpPolyPolygon3D->aPoly3DList.Insert(pPoly3D, nPos);
	}
	return *pPoly3D;
}

/*************************************************************************
|*
|* Liste loeschen
|*
\************************************************************************/

void PolyPolygon3D::Clear()
{
	if ( pImpPolyPolygon3D->nRefCount > 1 )
	{
		pImpPolyPolygon3D->nRefCount--;
		pImpPolyPolygon3D = new ImpPolyPolygon3D();
	}
	else
	{
		Polygon3D* pPoly3D = pImpPolyPolygon3D->aPoly3DList.First();

		while( pPoly3D )
		{
			delete pPoly3D;
			pPoly3D = pImpPolyPolygon3D->aPoly3DList.Next();
		}
		pImpPolyPolygon3D->aPoly3DList.Clear();
	}
}

/*************************************************************************
|*
|* Anzahl der Polygone zurueckgeben
|*
\************************************************************************/

UINT16 PolyPolygon3D::Count() const
{
	return (UINT16)(pImpPolyPolygon3D->aPoly3DList.Count());
}

/*************************************************************************
|*
|* Arrayoperator
|*
\************************************************************************/

Polygon3D& PolyPolygon3D::operator[](UINT16 nPos)
{
	CheckReference();
	Polygon3D* pPoly3D = pImpPolyPolygon3D->aPoly3DList.GetObject(nPos);

	if ( pPoly3D == NULL )
	{	// Wenn noch kein Polygon an der Position vorhanden, neu erzeugen
		pPoly3D = new Polygon3D;
		pImpPolyPolygon3D->aPoly3DList.Insert(pPoly3D, nPos);
	}
	return *pPoly3D;
}

/*************************************************************************
|*
|* Zuweisungsoperator
|*
\************************************************************************/

PolyPolygon3D& PolyPolygon3D::operator=(const PolyPolygon3D& rPolyPoly3D)
{
	rPolyPoly3D.pImpPolyPolygon3D->nRefCount++;

	if( pImpPolyPolygon3D->nRefCount > 1 )	pImpPolyPolygon3D->nRefCount--;
	else									delete pImpPolyPolygon3D;

	pImpPolyPolygon3D = rPolyPoly3D.pImpPolyPolygon3D;
	return *this;
}

/*************************************************************************
|*
|* Gleichheitsoperator
|*
\************************************************************************/

BOOL PolyPolygon3D::operator==(const PolyPolygon3D& rPolyPoly3D) const
{
	if ( pImpPolyPolygon3D == rPolyPoly3D.pImpPolyPolygon3D )
		return TRUE;
	else
		return *pImpPolyPolygon3D == *rPolyPoly3D.pImpPolyPolygon3D;
}

/*************************************************************************
|*
|* Ungleichheitsoperator
|*
\************************************************************************/

BOOL PolyPolygon3D::operator!=(const PolyPolygon3D& rPolyPoly3D) const
{
	if ( pImpPolyPolygon3D == rPolyPoly3D.pImpPolyPolygon3D )
		return FALSE;
	else
		return *pImpPolyPolygon3D != *rPolyPoly3D.pImpPolyPolygon3D;
}

/*************************************************************************
|*
|* alle Polygone mit uebergebener Matrix transformieren
|*
\************************************************************************/

void PolyPolygon3D::Transform(const Matrix4D& rTfMatrix)
{
	CheckReference();
	UINT16 nCnt = Count();

	for ( UINT16 i = 0; i < nCnt; i++ )
		pImpPolyPolygon3D->aPoly3DList.GetObject(i)->Transform(rTfMatrix);
}

/*************************************************************************
|*
|* Die Umlaufrichtungen ineinanderliegender Polygone so anpassen, dass
|* sie wechseln; Ausserdem wird ggf. das Polygon mit der geometrisch
|* betrachtet aeussersten Kontur an den Anfang der Liste verschoben
|* werden, damit Normalenberechnungen immer anhand des ersten Polygons
|* ausgefuehrt werden koennen.
|*
\************************************************************************/

void PolyPolygon3D::SetDirections(const Vector3D& rNormal)
{
	CheckReference();

	UINT16	nCnt = Count();
	UINT16	nPoly;
	short	nFirstPoly = -1;

	if(nCnt)
	{
		for ( nPoly = 0; nPoly < nCnt; nPoly++ )
		{
			Polygon3D& rPoly3D = *pImpPolyPolygon3D->aPoly3DList.GetObject(nPoly);
			BOOL bFlip = !rPoly3D.IsClockwise(rNormal);
			short nDepth = 0;
			const Vector3D& rPos = rPoly3D[0];

			// bestimmen, wie tief das aktuelle Polygon in den anderen
			// verschachtelt ist
			for ( UINT16 i = 0; i < nCnt; i++ )
			{
				if ( i != nPoly &&
					 pImpPolyPolygon3D->aPoly3DList.GetObject(i)->IsInside(rPos) )
						nDepth++;
			}
			// ungerade nDepth-Werte bedeuten: das  Polygon ist ein "Loch"
			// in der aeusseren Kontur
			BOOL bHole = ((nDepth & 0x0001) == 1);

			if ( (bFlip && !bHole) || (!bFlip && bHole) )
				rPoly3D.FlipDirection();

			// Den Index des (geometrisch) aeussersten Polygons merken
			if ( nDepth == 0 && nFirstPoly == -1 )
				nFirstPoly = (short) nPoly;
		}
		// liegt das aeussere Polygon nicht am Anfang, wird es dahin verschoben
		if ( nFirstPoly > 0 )
		{
			Polygon3D* pOuter = pImpPolyPolygon3D->aPoly3DList.Remove(nFirstPoly);
			pImpPolyPolygon3D->aPoly3DList.Insert(pOuter, (ULONG) 0);
		}
	}
}

/*************************************************************************
|*
|* in allen Polygonen doppelte aufeinanderfolgende Polygonpunkte
|* entfernen; ein geschlossenes Polygon wird dabei ggf. "geoeffnet"
|*
\************************************************************************/

void PolyPolygon3D::RemoveDoublePoints()
{
	CheckReference();
	UINT16 nCnt = Count();

	for ( UINT16 i = 0; i < nCnt; i++ )
		pImpPolyPolygon3D->aPoly3DList.GetObject(i)->RemoveDoublePoints();
}

/*************************************************************************
|*
|* Ein mittels GrowPoly() geschrumpftes PolyPolygon in eventuell
|* entstandenen Selbstueberschneidungen in Eckpunkten ohne
|* Punktreduzierung korrigieren
|*
\************************************************************************/

void PolyPolygon3D::CorrectGrownPoly(const PolyPolygon3D& rPolyOrig)
{
	if(Count() == rPolyOrig.Count())
	{
		for(UINT16 a=0;a<Count();a++)
		{
			const Polygon3D& rOrig = rPolyOrig[a];
			Polygon3D& rPoly = (*this)[a];
			UINT16 nPntCnt = rOrig.GetPointCount();

			if(nPntCnt == rPoly.GetPointCount() && nPntCnt > 2)
			{
				UINT16 nNumDiff(0);
				UINT16 nDoneStart(0xffff);

				// Testen auf Anzahl Aenderungen
				UINT16 b;
				for(b=0;b<nPntCnt;b++)
				{
					if(rOrig.GetPointOrientation(b) != rPoly.GetPointOrientation(b))
					{
						nNumDiff++;
					}
					else
					{
						if(nDoneStart == 0xffff)
						{
							// eventuellen Startpunkt auf gleiche Orientierung legen
							nDoneStart = b;
						}
					}
				}

				if(nNumDiff == nPntCnt)
				{
					// Komplett umgedreht, alles auf einen Punkt
					Vector3D aMiddle = rPoly.GetMiddle();

					for(b=0;b<nPntCnt;b++)
					{
						rPoly[b] = aMiddle;
					}
				}
				else if(nNumDiff)
				{
					// es gibt welche, nDoneStart ist gesetzt. Erzeuge (und
					// setze) nDoneEnd
					UINT16 nDoneEnd(nDoneStart);
					UINT16 nStartLoop;
					BOOL bInLoop(FALSE);

					// einen step mehr in der Schleife, um Loops abzuschliessen
					BOOL bFirstStep(TRUE);

					while(nDoneEnd != nDoneStart || bFirstStep)
					{
						bFirstStep = FALSE;

						// nCandidate ist Kandidat fuer Test
						UINT16 nCandidate = (nDoneEnd == nPntCnt-1) ? 0 : nDoneEnd+1;

						if(rOrig.GetPointOrientation(nCandidate) == rPoly.GetPointOrientation(nCandidate))
						{
							// Orientierung ist gleich
							if(bInLoop)
							{
								// Punkte innerhalb bInLoop auf ihr Zentrum setzen
								Vector3D aMiddle;
								UINT16 nCounter(0);
								UINT16 nStart(nStartLoop);

								while(nStart != nCandidate)
								{
									aMiddle += rPoly[nStart];
									nCounter++;
									nStart = (nStart == nPntCnt-1) ? 0 : nStart+1;
								}

								// Mittelwert bilden
								aMiddle /= (double)nCounter;

								// Punkte umsetzen
								nStart = nStartLoop;
								while(nStart != nCandidate)
								{
									rPoly[nStart] = aMiddle;
									nStart = (nStart == nPntCnt-1) ? 0 : nStart+1;
								}

								// Loop beenden
								bInLoop = FALSE;
							}
						}
						else
						{
							// Orientierung unterschiedlich
							if(!bInLoop)
							{
								// Start eines Loop mit geaenderter Orientierung
								nStartLoop = nCandidate;
								bInLoop = TRUE;
							}
						}

						// Weitergehen
						nDoneEnd = nCandidate;
					}
				}
			}
		}
	}
}

/*************************************************************************
|*
|* Ueberlappen sich das aktuelle und das angegebene PolyPolygon (in X/Y) ?
|*
\************************************************************************/

BOOL PolyPolygon3D::DoesBoundVolumeOverlap(const PolyPolygon3D& rOrig, UINT16 nDegreeFlag) const
{
	Volume3D aVolumeOrig = rOrig.GetPolySize();
	Volume3D aVolumeThis = GetPolySize();

	BOOL bOverlapX(TRUE);
	if(nDegreeFlag & DEGREE_FLAG_X)
		bOverlapX = BOOL(aVolumeOrig.MinVec().X() < aVolumeThis.MaxVec().X() && aVolumeOrig.MaxVec().X() > aVolumeThis.MinVec().X());

	BOOL bOverlapY(TRUE);
	if(nDegreeFlag & DEGREE_FLAG_Y)
		bOverlapY = BOOL(aVolumeOrig.MinVec().Y() < aVolumeThis.MaxVec().Y() && aVolumeOrig.MaxVec().Y() > aVolumeThis.MinVec().Y());

	BOOL bOverlapZ(TRUE);
	if(nDegreeFlag & DEGREE_FLAG_Z)
		bOverlapZ = BOOL(aVolumeOrig.MinVec().Z() < aVolumeThis.MaxVec().Z() && aVolumeOrig.MaxVec().Z() > aVolumeThis.MinVec().Z());

	return (bOverlapX && bOverlapY && bOverlapZ);
}

BOOL PolyPolygon3D::DoesOverlap(const PolyPolygon3D& rOrig, UINT16 nDegreeFlag) const
{
	BOOL bRetval = DoesBoundVolumeOverlap(rOrig, nDegreeFlag);

	if(bRetval)
	{
		// ueberlappen auch einzelen Polygone?
		BOOL bSinglePolyOverlap(FALSE);

		for(UINT16 a=0;!bSinglePolyOverlap && a<Count();a++)
		{
			const Polygon3D& rPoly = (*this)[a];

			for(UINT16 b=0;!bSinglePolyOverlap && b<rOrig.Count();b++)
			{
				const Polygon3D& rOrigPoly = rOrig[b];

				bSinglePolyOverlap = rPoly.DoesOverlap(rOrigPoly, nDegreeFlag);
			}
		}

		// Falls keine Ueberlappung von Einzelpolygonen
		if(!bSinglePolyOverlap)
			bRetval = FALSE;
	}

	return bRetval;
}

/*************************************************************************
|*
|* Remove all completely overlapping polygons
|*
\************************************************************************/
 
UINT16 PolyPolygon3D::RemoveContainedPolygons(BOOL bRemoveHoles, BOOL bWithBorder)
{
	UINT16 nRetval(0);
	UINT16 nCount(Count());

	if(nCount > 1)
	{
		for(UINT16 a=0;a+1<nCount;)
		{
			BOOL bIncA(TRUE);

			for(UINT16 b=a+1;b<nCount;)
			{
				BOOL bIncB(TRUE);

				if(GetObject(b).IsInside(GetObject(a), bWithBorder)
					&& (bRemoveHoles || GetObject(a).IsClockwise()))
				{
					// remove a
					Remove(a);
					nRetval++;
					nCount--;
					bIncA = FALSE;
					bIncB = FALSE;
					b = nCount;
				}
				else if(GetObject(a).IsInside(GetObject(b), bWithBorder)
					&& (bRemoveHoles || GetObject(b).IsClockwise()))
				{
					// remove b
					Remove(b);
					nRetval++;
					nCount--;
					bIncB = FALSE;
				}

				if(bIncB)
					b++;
			}

			if(bIncA)
				a++;
		}
	}

	return nRetval;
}

/*************************************************************************
|*
|* aus Stream laden
|*
\************************************************************************/

SvStream& operator>>(SvStream& rIStream, PolyPolygon3D& rPolyPoly3D)
{
	DBG_CHKOBJ(&rPolyPoly3D, PolyPolygon3D, NULL);

	Polygon3D* pPoly3D;

	// Anzahl der Polygone einlesen
	UINT16 nPolyCount;
	rIStream >> nPolyCount;

	BOOL bTruncated = FALSE;
	// Gesamtanzahl der Punkte mitzaehlen
	ULONG nAllPointCount = 0;

	if ( rPolyPoly3D.pImpPolyPolygon3D->nRefCount > 1 )
		rPolyPoly3D.pImpPolyPolygon3D->nRefCount--;
	else
		delete rPolyPoly3D.pImpPolyPolygon3D;

	rPolyPoly3D.pImpPolyPolygon3D = new ImpPolyPolygon3D(nPolyCount);

	while ( nPolyCount > 0 )
	{
		pPoly3D = new Polygon3D;
		rIStream >> *pPoly3D;
		nAllPointCount += pPoly3D->GetPointCount();

		if ( !bTruncated )
		{
			if ( nAllPointCount > POLY3D_MAXPOINTS )
			{
				UINT16 nDel = (UINT16)(nAllPointCount - POLY3D_MAXPOINTS);
				UINT16 nPos = pPoly3D->GetPointCount() - nDel;
				pPoly3D->Remove(nPos, nDel);
				// alle nachfolgenden Polygone ignorieren
				bTruncated = TRUE;
			}
			rPolyPoly3D.pImpPolyPolygon3D->aPoly3DList.Insert(pPoly3D, LIST_APPEND);
		}
		else
			delete pPoly3D;

		nPolyCount--;
	}

	return rIStream;
}

/*************************************************************************
|*
|* in Stream speichern
|*
\************************************************************************/

SvStream& operator<<(SvStream& rOStream, const PolyPolygon3D& rPolyPoly3D)
{
	DBG_CHKOBJ(&rPolyPoly3D, PolyPolygon3D, NULL);

	// Anzahl der Polygone rausschreiben
	rOStream << rPolyPoly3D.Count();

	// Die einzelnen Polygone ausgeben
	Polygon3D* pPoly3D = rPolyPoly3D.pImpPolyPolygon3D->aPoly3DList.First();

	while( pPoly3D )
	{
		rOStream << *pPoly3D;
		pPoly3D = rPolyPoly3D.pImpPolyPolygon3D->aPoly3DList.Next();
	}

	return rOStream;
}

BOOL PolyPolygon3D::IsClockwise(UINT16 nInd) const
{
	if(pImpPolyPolygon3D->aPoly3DList.Count() > nInd)
	{
		Vector3D aNormal = (*this)[0].GetNormal();
		return (*this)[nInd].IsClockwise(aNormal);
	}
	return FALSE;
}

Vector3D PolyPolygon3D::GetNormal() const
{
	if(pImpPolyPolygon3D->aPoly3DList.Count())
		return (*this)[0].GetNormal();
	return Vector3D(0.0, 0.0, -1.0);
}

/*************************************************************************
|*
|* die Umlaufrichtung des ersten Polygons umkehren
|*
\************************************************************************/

void PolyPolygon3D::FlipDirection(UINT16 nInd)
{
	CheckReference();
	if(pImpPolyPolygon3D->aPoly3DList.Count() > nInd)
		pImpPolyPolygon3D->aPoly3DList.GetObject(nInd)->FlipDirection();
}

void PolyPolygon3D::FlipDirections()
{
	CheckReference();
	UINT16 nCnt = Count();

	for ( UINT16 i = 0; i < nCnt; i++ )
		pImpPolyPolygon3D->aPoly3DList.GetObject(i)->FlipDirection();
}

Vector3D PolyPolygon3D::GetMiddle() const
{
	if(pImpPolyPolygon3D->aPoly3DList.Count())
		return (*this)[0].GetMiddle();
	return Vector3D();
}

BOOL PolyPolygon3D::IsClosed() const
{
	BOOL bClosed = TRUE;
	UINT16 nCnt = Count();

	for ( UINT16 i = 0; i < nCnt; i++ )
		if(!pImpPolyPolygon3D->aPoly3DList.GetObject(i)->IsClosed())
			bClosed = FALSE;
	return bClosed;
}

/*************************************************************************
|*
|* XPolyPolygon herausgeben
|*
\************************************************************************/

XPolyPolygon PolyPolygon3D::GetXPolyPolygon() const
{
	XPolyPolygon aXPolyPolygon;
	UINT16 nCnt = Count();

	for ( UINT16 i = 0; i < nCnt; i++ )
		aXPolyPolygon.Insert((*this)[i].GetXPolygon());

	return(aXPolyPolygon);
}

/*************************************************************************
|*
|* export PolyPolygon
|*
\************************************************************************/

PolyPolygon PolyPolygon3D::GetPolyPolygon() const
{
	PolyPolygon aPolyPolygon;
	UINT16 nCnt = Count();

	for(UINT16 i=0; i<nCnt;i++)
		aPolyPolygon.Insert((*this)[i].GetPolygon());

	return(aPolyPolygon);
}

/*************************************************************************
|*
|* Ausdehnung ermitteln
|*
\************************************************************************/

Volume3D PolyPolygon3D::GetPolySize() const
{
	UINT16 nCnt = Count();
	Volume3D aRetval;
	Volume3D aSubVolume;

	aRetval.Reset();
	for ( UINT16 i = 0; i < nCnt; i++ )
	{
		aSubVolume = (*this)[i].GetPolySize();
		aRetval.Union(aSubVolume);
	}

	return aRetval;
}

/*************************************************************************
|*
|* Flaeche des Polygons ermitteln
|*
\************************************************************************/

double PolyPolygon3D::GetPolyArea() const
{
	UINT16 nCnt = Count();
	double fRetval = 0.0;

	// Einzelflaechen aufsummieren
	Vector3D aNormal = GetNormal();
	for ( UINT16 i = 0; i < nCnt; i++ )
	{
		if((*this)[i].IsClockwise(aNormal))
		{
			fRetval += (*this)[i].GetPolyArea(aNormal);
		}
		else
		{
			fRetval -= (*this)[i].GetPolyArea(aNormal);
		}
	}
	return fabs(fRetval);
}

/*************************************************************************
|*
|* Laenge des Polygons ermitteln
|*
\************************************************************************/

double PolyPolygon3D::GetLength() const
{
	UINT16 nCnt = Count();
	double fRetval = 0.0;

	for ( UINT16 i = 0; i < nCnt; i++ )
		fRetval += (*this)[i].GetLength();

	return fRetval;
}

/*************************************************************************
|*
|* Umschliessenden Kugelradius feststellen
|*
\************************************************************************/

double PolyPolygon3D::GetEnclosingRadius() const
{
	Volume3D aSize = GetPolySize();
	double fBiggestDistance = 0.0;
	if(aSize.MinVec().X() != DBL_MAX && aSize.MaxVec().X() != DBL_MIN)
		if(aSize.MaxVec().X() - aSize.MinVec().X() > fBiggestDistance)
			fBiggestDistance = aSize.MaxVec().X() - aSize.MinVec().X();
	if(aSize.MinVec().Y() != DBL_MAX && aSize.MaxVec().Y() != DBL_MIN)
		if(aSize.MaxVec().Y() - aSize.MinVec().Y() > fBiggestDistance)
			fBiggestDistance = aSize.MaxVec().Y() - aSize.MinVec().Y();
	if(aSize.MinVec().Z() != DBL_MAX && aSize.MaxVec().Z() != DBL_MIN)
		if(aSize.MaxVec().Z() - aSize.MinVec().Z() > fBiggestDistance)
			fBiggestDistance = aSize.MaxVec().Z() - aSize.MinVec().Z();
	return (fBiggestDistance / 2.0);
}

///////////////////////////////////////////////////////////////////////////////

class ImpPolyNode
{
	Vector3D					maPosition;
	ImpPolyNode*				mpPrev;
	ImpPolyNode*				mpNext;

	ImpPolyNode*				mpListPrev;
	ImpPolyNode*				mpListNext;

public:
	ImpPolyNode(const Vector3D& rPos, ImpPolyNode* pPrev);
	~ImpPolyNode();

	ImpPolyNode* GetPrev() const { return mpPrev; }
	ImpPolyNode* GetNext() const { return mpNext; }
	const Vector3D& GetPos() const { return maPosition; }
	
	void CalcMinMaxX(double& fMaxAX, double& fMinAX);
	void CalcMinMaxY(double& fMaxAY, double& fMinAY);
	
	void SwapPrevNext() { ImpPolyNode* pZwi = mpPrev; mpPrev = mpNext; mpNext = pZwi; }
	void SwapNextPointers(ImpPolyNode* pCand);
	
	void AddToList(ImpPolyNode*& rpList);
	void RemFromList(ImpPolyNode*& rpList);

	BOOL GetOrientation();
	void SwapOrientation();

	void GetVolume(Volume3D& rVolume);

	BOOL IsInside(const Vector3D& rPnt, BOOL bWithBorder = TRUE);
	BOOL IsPolyInside(ImpPolyNode* pPoly);
};

ImpPolyNode::ImpPolyNode(const Vector3D& rPos, ImpPolyNode* pPrev)
:	maPosition(rPos),
	mpListPrev(this),
	mpListNext(this)
{
	if(pPrev)
	{
		mpNext = pPrev->GetNext();
		mpPrev = pPrev;
		mpNext->mpPrev = this;
		mpPrev->mpNext = this;
	}
	else
		mpPrev = mpNext = this;
}

ImpPolyNode::~ImpPolyNode()
{
	if(mpNext != this)
	{
		mpPrev->mpNext = mpNext;
		mpNext->mpPrev = mpPrev;
	}
}

BOOL ImpPolyNode::IsPolyInside(ImpPolyNode* pPoly)
{
	ImpPolyNode* pTest = pPoly;
	BOOL bAllAInside(TRUE);

	do {
		bAllAInside = IsInside(pTest->GetPos());
		pTest = pTest->GetNext();
	} while(bAllAInside && pTest != pPoly);

	return bAllAInside;
}

BOOL ImpPolyNode::IsInside(const Vector3D& rPnt, BOOL bWithBorder)
{
	BOOL bInside(FALSE);
	ImpPolyNode* pAct = this;

	do
	{
		if(bWithBorder
			&& (fabs(pAct->GetPos().X() - rPnt.X()) < SMALL_DVALUE)
			&& (fabs(pAct->GetPos().Y() - rPnt.Y()) < SMALL_DVALUE))
			return TRUE;

		ImpPolyNode* pNext = pAct->GetNext();
		if((pAct->GetPos().Y() - rPnt.Y() > -SMALL_DVALUE) != (pNext->GetPos().Y() - rPnt.Y() > -SMALL_DVALUE))
		{
			BOOL bXFlagOld(pAct->GetPos().X() - rPnt.X() > -SMALL_DVALUE);

			if(bXFlagOld == (pNext->GetPos().X() - rPnt.X() > -SMALL_DVALUE))
			{
				if(bXFlagOld)
					bInside = !bInside;
			}
			else
			{
				double fCmp = 
					pNext->GetPos().X() - (pNext->GetPos().Y() - rPnt.Y()) *
					(pAct->GetPos().X() - pNext->GetPos().X()) /
					(pAct->GetPos().Y() - pNext->GetPos().Y());

				if((bWithBorder && fCmp > rPnt.X()) || (!bWithBorder && fCmp - rPnt.X() > -SMALL_DVALUE))
					bInside = !bInside;
			}
		}

		// next edge
		pAct = pNext; 

	} while(pAct != this);

	return bInside;
}

void ImpPolyNode::GetVolume(Volume3D& rVolume)
{
	ImpPolyNode* pAct = this;
	do {
		rVolume.Union(pAct->GetPos());
		pAct = pAct->GetPrev();
	} while(pAct != this);
}

void ImpPolyNode::SwapOrientation()
{
	ImpPolyNode* pAct = this;
	do {
		pAct->SwapPrevNext();
		pAct = pAct->GetPrev();
	} while(pAct != this);
}

BOOL ImpPolyNode::GetOrientation()
{
	ImpPolyNode* pOutmost = this;
	ImpPolyNode* pAct = this->GetNext();

	while(pAct != this)
	{
		if(pOutmost->GetPos().X() - pAct->GetPos().X() > -SMALL_DVALUE)
		{
			if(pAct->GetPos().X() < pOutmost->GetPos().X())
			{
				pOutmost = pAct;
			}
			else
			{
				if(pAct->GetPos().Y() < pOutmost->GetPos().Y())
				{
					pOutmost = pAct;
				}
			}
		}

		// next node
		pAct = pAct->GetNext();
	}

	Vector3D aVec1 = pOutmost->GetPrev()->GetPos() - pOutmost->GetPos();
	Vector3D aVec2 = pOutmost->GetNext()->GetPos() - pOutmost->GetPos();
	return (BOOL)((aVec1.X() * aVec2.Y() - aVec1.Y() * aVec2.X()) > -SMALL_DVALUE);
}

void ImpPolyNode::SwapNextPointers(ImpPolyNode* pCand)
{
	ImpPolyNode* pZwi = mpNext;
	mpNext = pCand->mpNext;
	pCand->mpNext = pZwi;
	mpNext->mpPrev = this;
	pCand->mpNext->mpPrev = pCand;
}

void ImpPolyNode::AddToList(ImpPolyNode*& rpList)
{
	if(rpList)
	{
		mpListNext = rpList->mpListNext;
		rpList->mpListNext = this;
		mpListPrev = rpList;
		mpListNext->mpListPrev = this;
	}
	else
		rpList = this;
}

void ImpPolyNode::RemFromList(ImpPolyNode*& rpList)
{
	if(mpListNext != this)
	{
		if(rpList == this)
			rpList = mpListPrev;
		mpListPrev->mpListNext = mpListNext;
		mpListNext->mpListPrev = mpListPrev;
		mpListNext = mpListPrev = this;
	}
	else
	{
		if(rpList == this)
			rpList = NULL;
	}
}

void ImpPolyNode::CalcMinMaxX(double& fMaxAX, double& fMinAX)
{
	if(maPosition.X() > mpNext->maPosition.X())
	{
		fMaxAX = maPosition.X();
		fMinAX = mpNext->maPosition.X();
	}
	else
	{
		fMaxAX = mpNext->maPosition.X();
		fMinAX = maPosition.X();
	}
}

void ImpPolyNode::CalcMinMaxY(double& fMaxAY, double& fMinAY)
{
	if(maPosition.Y() > mpNext->maPosition.Y())
	{
		fMaxAY = maPosition.Y();
		fMinAY = mpNext->maPosition.Y();
	}
	else
	{
		fMaxAY = mpNext->maPosition.Y();
		fMinAY = maPosition.Y();
	}
}

///////////////////////////////////////////////////////////////////////////////

DECLARE_LIST(ImpPolyNodeList, ImpPolyNode*);

///////////////////////////////////////////////////////////////////////////////

class ImpSimpleCut
{
	ImpPolyNode*				mpLeft;
	ImpPolyNode*				mpRight;

	BOOL						mbCorrectOrientation;
	BOOL						mbOrientation;

public:
	ImpSimpleCut(ImpPolyNode* pL, ImpPolyNode* pR, BOOL bCoOr = FALSE, BOOL bOr = TRUE)
	:	mpLeft(pL),
		mpRight(pR),
		mbCorrectOrientation(bCoOr),
		mbOrientation(bOr)
	{
	}

	void Solve();
	ImpPolyNode* GetLeft() const { return mpLeft; }
	ImpPolyNode* GetRight() const { return mpRight; }

	BOOL IsSameCut(ImpPolyNode* pA, ImpPolyNode* pB) const
		{ return ((pA == mpLeft && pB == mpRight) || (pB == mpLeft && pA == mpRight)); }
};

void ImpSimpleCut::Solve() 
{ 
	mpLeft->SwapNextPointers(mpRight); 

	if(mbCorrectOrientation)
	{
		if(mpLeft->GetOrientation() != mbOrientation)
			mpLeft->SwapOrientation();

		if(mpRight->GetOrientation() != mbOrientation)
			mpRight->SwapOrientation();
	}
}

///////////////////////////////////////////////////////////////////////////////

DECLARE_LIST(ImpSimpleCutList, ImpSimpleCut*);

///////////////////////////////////////////////////////////////////////////////

class ImpExtraPolyInfo
{
	Volume3D					maVolume;
	INT16						mnDepth;
	BOOL						mbOrientation;

public:
	ImpExtraPolyInfo() {}

	void Init(ImpPolyNode* pNew);
	const Volume3D& GetVolume() const { return maVolume; }
	BOOL GetOrientation() const { return mbOrientation; }

	INT16 GetDepth() const { return mnDepth; }
	void ChangeDepth(BOOL bOrient) { if(bOrient) mnDepth++; else mnDepth--; }
};

void ImpExtraPolyInfo::Init(ImpPolyNode* pNew) 
{ 
	pNew->GetVolume(maVolume); 
	mbOrientation = pNew->GetOrientation(); 
	mnDepth = (mbOrientation) ? 0 : -1; 
}

///////////////////////////////////////////////////////////////////////////////

class ImpPolygonCutter
{
	// list of polys
	ImpPolyNodeList			aPolyList;
	PolyPolygon3D			aNotClosedPolys;

	// help routines
	BOOL IsSamePos(const Vector3D& rPntA, const Vector3D& rPntB)
		{ return (fabs(rPntA.X() - rPntB.X()) < SMALL_DVALUE && fabs(rPntA.Y() - rPntB.Y()) < SMALL_DVALUE); }
	ImpSimpleCut* GetExistingCut(ImpSimpleCutList& rTmpCuts, ImpPolyNode* pA, ImpPolyNode* pB);
	ImpPolyNode* ExtractNextPoly(ImpPolyNode*& rpList);
	BOOL IsCrossover(ImpPolyNode* pA, ImpPolyNode* pB);
	BOOL IsCrossover(ImpSimpleCut* pEnter, ImpSimpleCut* pLeave);
	BOOL IsNextSamePos(ImpPolyNode* pA, ImpPolyNode* pB)
		{ return IsSamePos(pA->GetNext()->GetPos(), pB->GetNext()->GetPos()); }
	BOOL IsPrevSamePos(ImpPolyNode* pA, ImpPolyNode* pB)
		{ return IsSamePos(pA->GetPrev()->GetPos(), pB->GetPrev()->GetPos()); }
	void AddAllNodes(ImpPolyNode* pPoly, ImpPolyNode*& rpList);
	ImpPolyNode* CreateNewPoly(const Polygon3D& rPoly);
	void DeletePoly(ImpPolyNode* pCand);
	void PolysToList(ImpPolyNode*& rpList);
	void ListToPolys(ImpPolyNode*& rpList);
	BOOL DoVolumesIntersect(const Volume3D& rVol1, const Volume3D& rVol2) const;
	BOOL DoVolumesInclude(const Volume3D& rVol1, const Volume3D& rVol2) const;
	void SolveAllCuts(ImpSimpleCutList& rCuts);

public:
	ImpPolygonCutter() {}
	~ImpPolygonCutter();

	// put/get poly
	void AddPolyPolygon3D(PolyPolygon3D& rPoly, BOOL bForceClockwise = FALSE);
	void GetPolyPolygon3D(PolyPolygon3D& rPoly);

	// transformations
	void RemoveSelfIntersections();
	void RemoveDoubleIntersections();

	// remove included
	void RemoveIncludedPolygons(BOOL bUseOr = TRUE);
};

ImpPolygonCutter::~ImpPolygonCutter()
{
	while(aPolyList.Count())
		delete aPolyList.Remove((UINT32)0L);
}

void ImpPolygonCutter::RemoveIncludedPolygons(BOOL bUseOr)
{
	ImpExtraPolyInfo* pInfos = new ImpExtraPolyInfo[aPolyList.Count()];
	UINT32 aCount(aPolyList.Count());
	UINT32 a, b;

	// fill infos
	for(a=0;a<aCount;a++)
		pInfos[a].Init(aPolyList.GetObject(a));

	// get all includes
	for(a=0;a<aCount;a++)
	{
		ImpExtraPolyInfo& rInfoA = pInfos[a];
		for(b=0;b<aCount;b++)
		{
			ImpExtraPolyInfo& rInfoB = pInfos[b];
			if(a!=b && DoVolumesInclude(rInfoA.GetVolume(), rInfoB.GetVolume()))
			{
				// volume B in A, test pA, pB for inclusion
				if(aPolyList.GetObject(a)->IsPolyInside(aPolyList.GetObject(b)))
				{
					// pB is inside pA
					rInfoB.ChangeDepth(rInfoA.GetOrientation());
				}
			}
		}
	}

	// delete removable
	for(a=0,b=0;a<aCount;a++)
	{
		ImpExtraPolyInfo& rInfo = pInfos[a];

		if((bUseOr && rInfo.GetDepth() != 0) || (!bUseOr && rInfo.GetDepth() < 1))
			DeletePoly(aPolyList.Remove((UINT32)b));
		else
			b++;
	}

	// delete infos
	delete pInfos;
}

void ImpPolygonCutter::SolveAllCuts(ImpSimpleCutList& rCuts)
{
	ImpPolyNode* pNewList = NULL;

	// add all nodes of polys to list
	PolysToList(pNewList);

	// solve cuts
	while(rCuts.Count())
	{
		ImpSimpleCut* pCut = rCuts.Remove((UINT32)0L);
		pCut->Solve();
		delete pCut;
	}

	// extract polys
	ListToPolys(pNewList);
}

BOOL ImpPolygonCutter::DoVolumesIntersect(const Volume3D& rVol1, const Volume3D& rVol2) const
{
	if(rVol2.MaxVec().X() - rVol1.MinVec().X() > -SMALL_DVALUE && rVol1.MaxVec().X() - rVol2.MinVec().X() > -SMALL_DVALUE)
		if(rVol2.MaxVec().Y() - rVol1.MinVec().Y() > -SMALL_DVALUE && rVol1.MaxVec().Y() - rVol2.MinVec().Y() > -SMALL_DVALUE)
			return TRUE;
	return FALSE;
}

BOOL ImpPolygonCutter::DoVolumesInclude(const Volume3D& rVolA, const Volume3D& rVolB) const
{
	return ((rVolB.MinVec().X() - rVolA.MinVec().X() > -SMALL_DVALUE)
		&& (rVolA.MaxVec().X() - rVolB.MaxVec().X() > -SMALL_DVALUE)
		&& (rVolB.MinVec().Y() - rVolA.MinVec().Y() > -SMALL_DVALUE)
		&& (rVolA.MaxVec().Y() - rVolB.MaxVec().Y() > -SMALL_DVALUE));
}

void ImpPolygonCutter::PolysToList(ImpPolyNode*& rpList)
{
	while(aPolyList.Count())
		AddAllNodes(aPolyList.Remove((UINT32)0L), rpList);
}

void ImpPolygonCutter::ListToPolys(ImpPolyNode*& rpList)
{
	while(rpList)
	{
		// get one
		ImpPolyNode* pNew = ExtractNextPoly(rpList);
		if(pNew)
			aPolyList.Insert(pNew, LIST_APPEND);
	}
}

ImpPolyNode* ImpPolygonCutter::CreateNewPoly(const Polygon3D& rPoly)
{
	ImpPolyNode* pRetval = NULL;

	for(UINT16 a=0;a<rPoly.GetPointCount();a++)
		pRetval = new ImpPolyNode(rPoly[a], pRetval);

	return pRetval;
}

void ImpPolygonCutter::DeletePoly(ImpPolyNode* pCand)
{
	ImpPolyNode* pPoly = pCand;

	while(pPoly)
	{
		ImpPolyNode* pNext = pPoly->GetNext();
		if(pNext == pPoly)
			pNext = NULL;
		delete pPoly;
		pPoly = pNext;
	}
}

void ImpPolygonCutter::AddAllNodes(ImpPolyNode* pPoly, ImpPolyNode*& rpList)
{
	ImpPolyNode* pAct = pPoly;
	do {
		pAct->AddToList(rpList);
		pAct = pAct->GetNext();
	} while(pAct != pPoly);
}

void ImpPolygonCutter::AddPolyPolygon3D(PolyPolygon3D& rPoly, BOOL bForceClockwise)
{
	for(UINT16 a=0;a<rPoly.Count();a++)
	{
		Polygon3D& rCand = rPoly[a];
		rCand.RemoveDoublePoints();

		if(!rCand.IsClosed() || rCand.GetPointCount() < 3)
		{
			aNotClosedPolys.Insert(rCand);
		}
		else
		{
			if(bForceClockwise && !rCand.IsClockwise())
				rCand.FlipDirection();

			ImpPolyNode* pNew = CreateNewPoly(rCand); // new ImpSimplePoly(rCand);
			aPolyList.Insert(pNew, LIST_APPEND);
		}
	}
}

void ImpPolygonCutter::GetPolyPolygon3D(PolyPolygon3D& rPoly)
{
	while(aPolyList.Count())
	{
		ImpPolyNode* pCand = aPolyList.Remove((UINT32)0L);
		ImpPolyNode* pAct = pCand;
		UINT16 nCount(0);
		
		do {
			nCount++;
			pAct = pAct->GetNext();
		} while(pAct != pCand);
		
		if(nCount > 2)
		{
			Polygon3D aNewPoly(nCount);
			nCount = 0;

			do {
				aNewPoly[nCount++] = pAct->GetPos();
				pAct = pAct->GetNext();
			} while(pAct != pCand);

			aNewPoly.SetClosed(TRUE);
			rPoly.Insert(aNewPoly);
		}
		
		DeletePoly(pCand);
	}

	while(aNotClosedPolys.Count())
		rPoly.Insert(aNotClosedPolys.Remove(0L));
}

ImpSimpleCut* ImpPolygonCutter::GetExistingCut(ImpSimpleCutList& rTmpCuts, ImpPolyNode* pA, ImpPolyNode* pB)
{
	for(UINT32 a=0;a<rTmpCuts.Count();a++)
	{
		ImpSimpleCut* pCand = rTmpCuts.GetObject(a);
		if(pCand->IsSameCut(pA, pB))
			return pCand;
	}

	return NULL;
}

ImpPolyNode* ImpPolygonCutter::ExtractNextPoly(ImpPolyNode*& rpList)
{
	ImpPolyNode* pStart = rpList;

	// remove all nodes of this poly from list
	ImpPolyNode* pAct = pStart;
	UINT32 nNumNodes(0L);
	do {
		pAct->RemFromList(rpList);
		pAct = pAct->GetNext();
		nNumNodes++;
	} while(pAct != pStart);

	if(nNumNodes < 3)
	{
		DeletePoly(pStart);
		return NULL;
	}
	else
		return pStart;
}

void ImpPolygonCutter::RemoveSelfIntersections()
{
	ImpSimpleCutList aCuts(256, 256);
	ImpSimpleCutList aNewCuts(64, 64);
	ImpPolyNode* pCand;
	ImpPolyNode* pA;
	ImpPolyNode* pB;
	double fMaxAX, fMinAX, fMaxAY, fMinAY;
	double fMaxBX, fMinBX, fMaxBY, fMinBY;
	double fCut;

	// first job: Find all cuts and add points there
	for(UINT32 a=0;a<aPolyList.Count();a++)
	{
		pCand = aPolyList.GetObject(a);
		pA = pCand;

		// one run to find same start positions (so there is no need to
		// search for existing cuts in main loop)
		do {
			pB = pA->GetNext();
			do {
				if(IsSamePos(pA->GetPos(), pB->GetPos()))
					aNewCuts.Insert(new ImpSimpleCut(pA, pB, TRUE, pCand->GetOrientation()), LIST_APPEND);

				// next B
				pB = pB->GetNext();
			} while(pB != pCand);
			
			// next A
			pA = pA->GetNext();
		} while(pA->GetNext() != pCand);

		// second run to find real cuts
		pA = pCand;
		do {
			// get bounds for this edge in poly
			pA->CalcMinMaxX(fMaxAX, fMinAX);
			pA->CalcMinMaxY(fMaxAY, fMinAY);
			
			pB = pA->GetNext();
			do {
				pB->CalcMinMaxX(fMaxBX, fMinBX);
				if(fMaxBX - fMinAX > -SMALL_DVALUE && fMaxAX - fMinBX > -SMALL_DVALUE)
				{
					pB->CalcMinMaxY(fMaxBY, fMinBY);
					if(fMaxBY - fMinAY > -SMALL_DVALUE && fMaxAY - fMinBY > -SMALL_DVALUE)
					{
						if(!IsSamePos(pA->GetPos(), pB->GetPos()))
						{
							if(Polygon3D::FindCut(pA->GetPos(), pA->GetNext()->GetPos() - pA->GetPos(), pB->GetPos(), pB->GetNext()->GetPos() - pB->GetPos(), CUTFLAG_LINE, &fCut))
							{
								// crossover, two new points
								Vector3D aNewPos;
								aNewPos.CalcInBetween(pA->GetPos(), pA->GetNext()->GetPos(), fCut);
								ImpPolyNode* pCutLo = new ImpPolyNode(aNewPos, pA);
								ImpPolyNode* pCutHi = new ImpPolyNode(aNewPos, pB);
								aNewCuts.Insert(new ImpSimpleCut(pCutLo, pCutHi, TRUE, pCand->GetOrientation()), LIST_APPEND);
								pA->CalcMinMaxX(fMaxAX, fMinAX);
								pA->CalcMinMaxY(fMaxAY, fMinAY);
							}
							else
							{
								if(Polygon3D::FindPointInLine(pA->GetPos(), pB->GetPos(), pB->GetNext()->GetPos() - pB->GetPos(), &fCut))
								{
									// startpoint A at edge B, one new point
									ImpPolyNode* pCutHi = new ImpPolyNode(pA->GetPos(), pB);
									aNewCuts.Insert(new ImpSimpleCut(pA, pCutHi, TRUE, pCand->GetOrientation()), LIST_APPEND);
								}
								else if(Polygon3D::FindPointInLine(pB->GetPos(), pA->GetPos(), pA->GetNext()->GetPos() - pA->GetPos(), &fCut))
								{
									// startpoint B at edge A, one new point
									ImpPolyNode* pCutLo = new ImpPolyNode(pB->GetPos(), pA);
									aNewCuts.Insert(new ImpSimpleCut(pCutLo, pB, TRUE, pCand->GetOrientation()), LIST_APPEND);
									pA->CalcMinMaxX(fMaxAX, fMinAX);
									pA->CalcMinMaxY(fMaxAY, fMinAY);
								}
							}
						}
					}
				}

				// next B
				pB = pB->GetNext();
			} while(pB != pCand);
			
			// next A
			pA = pA->GetNext();
		} while(pA->GetNext() != pCand);
		
		// copy new cuts to cuts
		while(aNewCuts.Count())
			aCuts.Insert(aNewCuts.Remove((UINT32)0L));
	}

	// second job: if there were cuts, split polys
	if(aCuts.Count())
		SolveAllCuts(aCuts);
}

BOOL ImpPolygonCutter::IsCrossover(ImpPolyNode* pA, ImpPolyNode* pB)
{
	// build entering vectors
	Vector3D aVecA = pA->GetPrev()->GetPos() - pA->GetPos();
	Vector3D aVecB = pB->GetPrev()->GetPos() - pA->GetPos();
	aVecA.Normalize();
	aVecB.Normalize();
	double fDegreeA2 = atan2(aVecA.Y(), aVecA.X());
	double fDegreeB2 = atan2(aVecB.Y(), aVecB.X());

	// build leaving vectors
	aVecA = pA->GetNext()->GetPos() - pA->GetPos();
	aVecB = pB->GetNext()->GetPos() - pA->GetPos();
	aVecA.Normalize();
	aVecB.Normalize();
	double fDegreeA1 = atan2(aVecA.Y(), aVecA.X());
	double fDegreeB1 = atan2(aVecB.Y(), aVecB.X());

	// compare
	if(fDegreeA1 > fDegreeA2)
	{
		double fZwi = fDegreeA2;
		fDegreeA2 = fDegreeA1;
		fDegreeA1 = fZwi;
	}

	BOOL bB1Inside = (fDegreeB1 - fDegreeA1 > SMALL_DVALUE && fDegreeA2 - fDegreeB1 > SMALL_DVALUE);
	BOOL bB2Inside = (fDegreeB2 - fDegreeA1 > SMALL_DVALUE && fDegreeA2 - fDegreeB2 > SMALL_DVALUE);

	if(bB1Inside && bB2Inside)
		return FALSE;

	BOOL bB1Outside = (fDegreeA1 - fDegreeB1 > SMALL_DVALUE || fDegreeB1 - fDegreeA2 > SMALL_DVALUE);
	BOOL bB2Outside = (fDegreeA1 - fDegreeB2 > SMALL_DVALUE || fDegreeB2 - fDegreeA2 > SMALL_DVALUE);

	return !(bB1Outside && bB2Outside);
}

BOOL ImpPolygonCutter::IsCrossover(ImpSimpleCut* pEnter, ImpSimpleCut* pLeave)
{
	// build entering vectors
	Vector3D aVecJ = pEnter->GetLeft()->GetNext()->GetPos() - pEnter->GetLeft()->GetPos();
	Vector3D aVecA = pEnter->GetLeft()->GetPrev()->GetPos() - pEnter->GetLeft()->GetPos();
	Vector3D aVecB = pEnter->GetRight()->GetPrev()->GetPos() - pEnter->GetLeft()->GetPos();
	aVecJ.Normalize();
	aVecA.Normalize();
	aVecB.Normalize();
	double fDegreeJo = atan2(aVecJ.Y(), aVecJ.X());
	double fDegreeA2 = atan2(aVecA.Y(), aVecA.X()) - fDegreeJo;
	double fDegreeB2 = atan2(aVecB.Y(), aVecB.X()) - fDegreeJo;

	// move to range [0..2PI[
	while(fDegreeA2 < 0) 
		fDegreeA2 += (2.0 * F_PI);
	while(fDegreeA2 >= (2.0 * F_PI)) 
		fDegreeA2 -= (2.0 * F_PI);
	
	// move to range [0..2PI[
	while(fDegreeB2 < 0) 
		fDegreeB2 += (2.0 * F_PI);
	while(fDegreeB2 >= (2.0 * F_PI)) 
		fDegreeB2 -= (2.0 * F_PI);

	BOOL bA2BiggerB2 = (fDegreeA2 - fDegreeB2 > -SMALL_DVALUE);
	
	// build leaving vectors
	aVecJ = pLeave->GetLeft()->GetPrev()->GetPos() - pLeave->GetLeft()->GetPos();
	aVecA = pLeave->GetLeft()->GetNext()->GetPos() - pLeave->GetLeft()->GetPos();
	aVecB = pLeave->GetRight()->GetNext()->GetPos() - pLeave->GetLeft()->GetPos();
	aVecJ.Normalize();
	aVecA.Normalize();
	aVecB.Normalize();
	fDegreeJo = atan2(aVecJ.Y(), aVecJ.X());
	double fDegreeA1 = atan2(aVecA.Y(), aVecA.X()) - fDegreeJo;
	double fDegreeB1 = atan2(aVecB.Y(), aVecB.X()) - fDegreeJo;

	// move to range [0..2PI[
	while(fDegreeA1 < 0) 
		fDegreeA1 += (2.0 * F_PI);
	while(fDegreeA1 >= (2.0 * F_PI)) 
		fDegreeA1 -= (2.0 * F_PI);
	
	// move to range [0..2PI[
	while(fDegreeB1 < 0) 
		fDegreeB1 += (2.0 * F_PI);
	while(fDegreeB1 >= (2.0 * F_PI)) 
		fDegreeB1 -= (2.0 * F_PI);

	BOOL bA1BiggerB1 = (fDegreeA1 - fDegreeB1 > -SMALL_DVALUE);

	// compare
	return (bA1BiggerB1 == bA2BiggerB2);
}

void ImpPolygonCutter::RemoveDoubleIntersections()
{
	double fMaxAX, fMinAX, fMaxAY, fMinAY;
	double fMaxBX, fMinBX, fMaxBY, fMinBY;
	double fCut;
	ImpSimpleCutList aCuts(256, 256);
	ImpSimpleCutList aTmpCuts(64, 64);
	ImpSimpleCutList aNewCuts(64, 64);
	ImpPolyNode* pCandA;
	ImpPolyNode* pCandB;
	ImpPolyNode* pA;
	ImpPolyNode* pB;
	UINT32 a;

	// create volume list for all polys for faster compares
	Volume3D* pVolumes = new Volume3D[aPolyList.Count()];

	for(a=0L;a<aPolyList.Count();a++)
		aPolyList.GetObject(a)->GetVolume(pVolumes[a]);

	// register cuts (and add points for them) between pCandA and pCandB
	for(a=0L;a+1<aPolyList.Count();a++)
	{
		pCandA = aPolyList.GetObject(a);
		for(UINT32 b=a+1;b<aPolyList.Count();b++)
		{
			if(DoVolumesIntersect(pVolumes[a], pVolumes[b]))
			{
				pCandB = aPolyList.GetObject(b);
				pA = pCandA;

				// one run to find same start positions (so there is no need to
				// search for existing cuts in main loop)
				do {
					pB = pCandB;
					do {
						if(IsSamePos(pA->GetPos(), pB->GetPos()))
							aTmpCuts.Insert(new ImpSimpleCut(pA, pB), LIST_APPEND);

						// next B
						pB = pB->GetNext();
					} while(pB != pCandB);

					// next A
					pA = pA->GetNext();
				} while(pA != pCandA);

				// second run to find real cuts
				pA = pCandA;
				do {
					// get bounds for this edge in poly
					pA->CalcMinMaxX(fMaxAX, fMinAX);
					pA->CalcMinMaxY(fMaxAY, fMinAY);
		
					pB = pCandB;
					do {
						pB->CalcMinMaxX(fMaxBX, fMinBX);
						if(fMaxBX - fMinAX > -SMALL_DVALUE && fMaxAX - fMinBX > -SMALL_DVALUE)
						{
							pB->CalcMinMaxY(fMaxBY, fMinBY);
							if(fMaxBY - fMinAY > -SMALL_DVALUE && fMaxAY - fMinBY > -SMALL_DVALUE)
							{
								if(!IsSamePos(pA->GetPos(), pB->GetPos()))
								{
									if(Polygon3D::FindCut(pA->GetPos(), pA->GetNext()->GetPos() - pA->GetPos(), pB->GetPos(), pB->GetNext()->GetPos() - pB->GetPos(), CUTFLAG_LINE, &fCut))
									{
										// crossover, two new points, use as cutpoint
										Vector3D aNewPos;
										
										aNewPos.CalcInBetween(pA->GetPos(), pA->GetNext()->GetPos(), fCut);
										ImpPolyNode* pCutLo = new ImpPolyNode(aNewPos, pA);
										ImpPolyNode* pCutHi = new ImpPolyNode(aNewPos, pB);
										aNewCuts.Insert(new ImpSimpleCut(pCutLo, pCutHi), LIST_APPEND);
										pA->CalcMinMaxX(fMaxAX, fMinAX);
										pA->CalcMinMaxY(fMaxAY, fMinAY);
									}
									else
									{
										if(Polygon3D::FindPointInLine(pA->GetPos(), pB->GetPos(), pB->GetNext()->GetPos() - pB->GetPos(), &fCut))
										{
											// startpoint A at edge B, one new point
											// leaves or enters common section
											ImpPolyNode* pCutHi = new ImpPolyNode(pA->GetPos(), pB);
											aTmpCuts.Insert(new ImpSimpleCut(pA, pCutHi), LIST_APPEND);
										}
										else if(Polygon3D::FindPointInLine(pB->GetPos(), pA->GetPos(), pA->GetNext()->GetPos() - pA->GetPos(), &fCut))
										{
											// startpoint B at edge A, one new point
											// leaves or enters common section
											ImpPolyNode* pCutLo = new ImpPolyNode(pB->GetPos(), pA);
											aTmpCuts.Insert(new ImpSimpleCut(pCutLo, pB), LIST_APPEND);
											pA->CalcMinMaxX(fMaxAX, fMinAX);
											pA->CalcMinMaxY(fMaxAY, fMinAY);
										}
									}
								}
							}
						}

						// next B
						pB = pB->GetNext();
					} while(pB != pCandB);

					// next A
					pA = pA->GetNext();
				} while(pA != pCandA);

				// test all temporary cuts for simple criteria
				for(UINT32 c=0;c<aTmpCuts.Count();)
				{
					ImpSimpleCut* pCand = aTmpCuts.GetObject(c);
					BOOL bPrevSamePos(IsPrevSamePos(pCand->GetLeft(), pCand->GetRight()));
					BOOL bNextSamePos(IsNextSamePos(pCand->GetLeft(), pCand->GetRight()));
					BOOL bDelete(FALSE);
					BOOL bIncC(TRUE);

					if(bPrevSamePos && bNextSamePos)
					{
						// single point inside continued same direction section
						bDelete = TRUE;
					}
					else if(!bPrevSamePos && !bNextSamePos)
					{
						// this is no same direction section, test for real cut
						if(IsCrossover(pCand->GetLeft(), pCand->GetRight()))
						{
							// real cut, move to real cutlist
							aNewCuts.Insert(aTmpCuts.Remove(c), LIST_APPEND);
							bIncC = FALSE;
						}
						else
						{
							// no cut, just a touch in one point
							bDelete = TRUE;
						}
					}

					// delete if wanted
					if(bDelete)
					{
						delete aTmpCuts.Remove(c);
						bIncC = FALSE;
					}

					// next candidate
					if(bIncC)
						c++;
				}

				// are there entering/leaving same direction sections?
				while(aTmpCuts.Count())
				{
					// this cuts enter/leave a common same-direction section between
					// polygons pCandA, pCandB. If it is a real crossover, a cutpoint
					// for it is needed, else it can be ignored.
					ImpSimpleCut* pCutA = aTmpCuts.Remove((UINT32)0L);
					ImpPolyNode* pActA = pCutA->GetLeft();
					ImpPolyNode* pActB = pCutA->GetRight();
					BOOL bPrevSamePos(IsPrevSamePos(pActA, pActB));
					BOOL bNextSamePos(IsNextSamePos(pActA, pActB));

					if(aTmpCuts.Count())
					{
						ImpSimpleCut* pCutB = NULL;

						if(IsNextSamePos(pCutA->GetLeft(), pCutA->GetRight()))
						{
							// this is a start node
							ImpPolyNode* pActA = pCutA->GetLeft()->GetNext();
							ImpPolyNode* pActB = pCutA->GetRight()->GetNext();

							while(!pCutB && pActA != pCutA->GetLeft())
							{
								if(!IsNextSamePos(pActA, pActB))
									pCutB = GetExistingCut(aTmpCuts, pActA, pActB);
								
								pActA = pActA->GetNext();
								pActB = pActB->GetNext();
							}
							
							if(pCutB)
							{
								aTmpCuts.Remove(pCutB);

								if(IsCrossover(pCutA, pCutB))
									aNewCuts.Insert(pCutB, LIST_APPEND);
								else
									delete pCutB;
							}
						}
						else
						{
							// this is a end node
							ImpPolyNode* pActA = pCutA->GetLeft()->GetPrev();
							ImpPolyNode* pActB = pCutA->GetRight()->GetPrev();

							while(!pCutB && pActA != pCutA->GetLeft())
							{
								if(!IsPrevSamePos(pActA, pActB))
									pCutB = GetExistingCut(aTmpCuts, pActA, pActB);
								
								pActA = pActA->GetPrev();
								pActB = pActB->GetPrev();
							}

							if(pCutB)
							{
								aTmpCuts.Remove(pCutB);

								if(IsCrossover(pCutB, pCutA))
									aNewCuts.Insert(pCutB, LIST_APPEND);
								else
									delete pCutB;
							}
						}
					}

					// delete cut in EVERY case
					delete pCutA;
				}

				// copy new cuts to all cuts
				while(aNewCuts.Count())
					aCuts.Insert(aNewCuts.Remove((UINT32)0L), LIST_APPEND);
			}
		}
	}

	// delete volume list again
	delete pVolumes;

	// are there cuts to solve? Solve them all in one run
	if(aCuts.Count())
		SolveAllCuts(aCuts);
}

///////////////////////////////////////////////////////////////////////////////

void PolyPolygon3D::Merge(BOOL bForceClockwise, BOOL bInvertRemove)
{
	ImpPolygonCutter aCutter;

	aCutter.AddPolyPolygon3D(*this, bForceClockwise);
	aCutter.RemoveSelfIntersections();
	aCutter.RemoveDoubleIntersections();
	aCutter.RemoveIncludedPolygons(!bInvertRemove);
	Clear();
	aCutter.GetPolyPolygon3D(*this);
}



