/*************************************************************************
 *
 *  $RCSfile: decltor.cxx,v $
 *
 *  $Revision: 1.32.2.2 $
 *
 *  last change: $Author: mh $ $Date: 2002/10/31 20:47:02 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "fields.hxx"
#include "decltor.hxx"
#include "compiler.hxx" // fr Fehlerausgabe
#include "instdb.hxx"
#include "script.hxx"
#include "environ.hxx"

#include "sihelp.hxx"

// RTTI -----------------------------------------------------------

TYPEINIT0(SiDeclarator)

TYPEINIT1(SiModule,SiDeclarator)
TYPEINIT1(SiModulesSet,SiDeclarator)
TYPEINIT1(SiModuleIds,SiDeclarator)
TYPEINIT1(SiInstallation,SiDeclarator);

const char NSEP[] = "//"; // Separator fuer Teile der NaturalID

///////////////////////////////////////////////////////////////////////////////
//
//		SiDeclarator
//

SiDeclarator::SiDeclarator( SiIdentifier const& ID, SiCompiler* pCompiler, BOOL bSystemObject )
{
	m_xCompiler		= pCompiler;
	m_ID			= ID.AsByteString();
	m_bSystemObject = bSystemObject;

	m_nLanguage		= LANG_DEFAULT;
	m_pLangParent	= NULL;
    m_nRefCount     = 0;
}

SiDeclarator::~SiDeclarator()
{
	for( USHORT i = 0; i < m_aLangRefs.Count(); ++i )
		delete m_aLangRefs.GetObject(i);
}

ByteString SiDeclarator::GetID() const
{
	if( !IsLangRef() )
		return m_ID;
	ByteString aLangID( m_ID );
	aLangID += "__";
	aLangID += ByteString::CreateFromInt32( m_nLanguage );
	return aLangID;
}

void SiDeclarator::Error(ByteString const& aErrorMessage)
{
	m_xCompiler->SymanticError(aErrorMessage);
}

void SiDeclarator::Warning(ByteString const& aMessage)
{
	m_xCompiler->Warning(aMessage);
}

void SiDeclarator::Missing(char const* aFieldname)
{
	ByteString aMsg("Object <");
	aMsg += GetID();
	aMsg += "> is missing ";
	aMsg += aFieldname;
	Error( aMsg );
}

void SiDeclarator::Obsolete(char const* aFieldname)
{
	ByteString aMsg(aFieldname);
	aMsg += " is obsolete in Object <";
	aMsg += GetID();
	aMsg += ">";

	m_xCompiler->Warning( aMsg );
}

void SiDeclarator::OsWarning(char const* aFieldname)
{
	ByteString aMsg(aFieldname);
	aMsg += " in Object <";
	aMsg += GetID();
	aMsg += "> is illegal on this operating system";
	m_xCompiler->Warning( aMsg );
}

BOOL SiDeclarator::SetProperty(ByteString const& aProperty, ByteString const&)
{
	ByteString aMsg(aProperty);
	aMsg += " is not a (string-) member of ";
	aMsg += m_ID;
	Error( aMsg );
	return FALSE;
}

BOOL SiDeclarator::SetProperty(ByteString const& aProperty, long aValue)
{
	ByteString aMsg(aProperty);
	aMsg += " is not a (integer-) member of ";
	aMsg += m_ID;
	Error( aMsg );
	return FALSE;
}

BOOL SiDeclarator::SetProperty( ByteString const& aProperty, sal_uInt32 nValue )
{
	ByteString aMsg(aProperty);
	aMsg += " is not a (integer-) member of ";
	aMsg += m_ID;
	Error( aMsg );
	return FALSE;
}

BOOL SiDeclarator::SetProperty(ByteString const& aProperty, SiDeclarator* aValue)
{
	ByteString aMsg(aProperty);
	aMsg += " is not a (id-) member of ";
	aMsg += m_ID;
	Error( aMsg );
	return FALSE;
}

BOOL SiDeclarator::Check()
{
	return TRUE;
}

BOOL SiDeclarator::CheckField(ByteString const& aField, char const* aFieldname)
{
	if (aField.Len() == 0)
	{
		Missing(aFieldname);
		return FALSE;
	}
	else
		return TRUE;
}
BOOL SiDeclarator::CheckField(UniString const& _suField, char const* _sFieldname)
{
	return CheckField(ByteString(_suField, osl_getThreadTextEncoding()), _sFieldname);
}

BOOL SiDeclarator::CheckField(SiDeclarator const* pField, char const* aFieldname)
{
	if (pField == NULL)
	{
		Missing(aFieldname);
		return FALSE;
	}
	else
		return TRUE;
}

BOOL SiDeclarator::CheckFieldEmpty(ByteString const& aField, char const* aFieldname)
{
	if (aField.Len() != 0)
	{
		Obsolete(aFieldname);
		return FALSE;
	}
	else
		return TRUE;
}

BOOL SiDeclarator::CheckFieldEmpty(SiDeclarator const* pField, char const* aFieldname)
{
	if (pField != NULL)
	{
		Obsolete(aFieldname);
		return FALSE;
	}
	else
		return TRUE;
}

BOOL SiDeclarator::SetBool(BOOL& aProperty, ByteString const& aValue)
{
	if (aValue == VALUE_YES)
	{
		aProperty = TRUE;
		return TRUE;
	}

	if (aValue == VALUE_NO)
	{
		aProperty = FALSE;
		return TRUE;
	}

	Error("expected YES or NO");
	return FALSE;
}

BOOL SiDeclarator::SetDate(Date& aProperty, ByteString const& aValue)
{
	if (aValue.Len() != 8)
	{
		Error("expected date i.e. \"30041997\"");
		return FALSE;
	}

	long l = aValue.ToInt32();
	aProperty = Date(USHORT(l/1000000L),USHORT(l/10000),USHORT(l%10000));
	return TRUE;
}

BOOL SiDeclarator::SetTime(Time& aProperty, ByteString const& aValue)
{
	if (aValue.Len() != 4)
	{
		Error("expected time i.e. \"0400\"");
		return FALSE;
	}

	long l = aValue.ToInt32();
	aProperty = Time(USHORT(l/100),USHORT(l%100));
	return TRUE;
}

BOOL SiDeclarator::IsWin() const
{
	return m_xCompiler->GetOsType() == com::sun::star::setup::OSType_WIN;
}

BOOL SiDeclarator::IsOs2() const
{
	return m_xCompiler->GetOsType() == com::sun::star::setup::OSType_OS2;
}

BOOL SiDeclarator::IsMac() const
{
	return m_xCompiler->GetOsType() == com::sun::star::setup::OSType_MAC;
}

BOOL SiDeclarator::IsUnix() const
{
	return	m_xCompiler->GetOsType() == com::sun::star::setup::OSType_UNIX_SOLS ||
			m_xCompiler->GetOsType() == com::sun::star::setup::OSType_UNIX_SOLI ||
			m_xCompiler->GetOsType() == com::sun::star::setup::OSType_UNIX_SOLSG ||
			m_xCompiler->GetOsType() == com::sun::star::setup::OSType_UNIX_SOLIG ||
			m_xCompiler->GetOsType() == com::sun::star::setup::OSType_UNIX_LINUX ||
			m_xCompiler->GetOsType() == com::sun::star::setup::OSType_UNIX_HP ||
			m_xCompiler->GetOsType() == com::sun::star::setup::OSType_UNIX_SCO;
}

SiDeclarator* SiDeclarator::GetLangRef(USHORT nLanguage)
{
	SiInstallation* pInst = m_xCompiler->GetCScript()->GetInstallation();
	if( !pInst )
		return NULL;

	if( nLanguage == LANG_DEFAULT ||
		nLanguage == (USHORT) pInst->GetDefLanguage().ToInt32() )
		if( m_pLangParent )
			return m_pLangParent;
		else
			return this;

	for( USHORT nIdx = 0; nIdx < m_aLangRefs.Count(); ++nIdx )
	{
		SiDeclarator* pDecl = m_aLangRefs.GetObject(nIdx);
		if( pDecl && pDecl->GetLanguage() == nLanguage )
			return pDecl;
	}

	return NULL;
}

SiDeclarator* SiDeclarator::GetInternalLangRef(USHORT nLanguage)
{
	if( nLanguage == m_nLanguage )
		return this;

	SiDeclarator* pDecl = GetLangRef(nLanguage);
	if( pDecl )
		return pDecl;

	if( (pDecl = PTR_CAST(SiFile,this)) != NULL )
	{
		pDecl = new SiFile(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiModule,this)) != NULL )
	{
		pDecl = new SiModule(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiRegistryItem,this)) != NULL )
	{
		pDecl = new SiRegistryItem(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiProfileItem,this)) != NULL )
	{
		pDecl = new SiProfileItem(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiDirectory,this)) != NULL )
	{
		pDecl = new SiDirectory(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiFolder,this)) != NULL )
	{
		pDecl = new SiFolder(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiFolderItem,this)) != NULL )
	{
		pDecl = new SiFolderItem(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiProfile,this)) != NULL )
	{
		pDecl = new SiProfile(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiCustom,this)) != NULL )
	{
		pDecl = new SiCustom(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiProcedure,this)) != NULL )
	{
		pDecl = new SiProcedure(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiStarRegistry,this)) != NULL )
	{
		pDecl = new SiStarRegistry(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiStarRegistryItem,this)) != NULL )
	{
		pDecl = new SiStarRegistryItem(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiSlide,this)) != NULL )
	{
		pDecl = new SiSlide(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiRegistryArea,this)) != NULL )
	{
		pDecl = new SiRegistryArea(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiOs2Class,this)) != NULL )
	{
		pDecl = new SiOs2Class(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiOs2Creator,this)) != NULL )
	{
		pDecl = new SiOs2Creator(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiOs2Template,this)) != NULL )
	{
		pDecl = new SiOs2Template(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiShortcut,this)) != NULL )
	{
		pDecl = new SiShortcut(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiInstallation,this)) != NULL )
	{
		pDecl = new SiInstallation(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiHelpText,this)) != NULL )
	{
		pDecl = new SiHelpText(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiDataCarrier,this)) != NULL )
	{
		pDecl = new SiDataCarrier(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiScpAction,this)) != NULL )
	{
		pDecl = new SiScpAction(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiConfigurationItem,this)) != NULL )
	{
		pDecl = new SiConfigurationItem(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiModuleIds,this)) != NULL )
	{
		pDecl = new SiModuleIds(GetID(), GetCompiler());
	}
	else if( (pDecl = PTR_CAST(SiModulesSet,this)) != NULL )
	{
		pDecl = new SiModulesSet(GetID(), GetCompiler());
	}

	if( pDecl )
	{
		pDecl->m_pLangParent = this;
		pDecl->SetLanguage( nLanguage );
		m_aLangRefs.Insert( pDecl );
	}
	return pDecl;
}

///////////////////////////////////////////////////////////////////////////////
//
//		SiModule
//

SiModule::SiModule(SiIdentifier const& ID, SiCompiler *pCompiler)
: SiDeclarator(ID,pCompiler)
{
	INIT_MULTI_LANG_MEMBER(	pParent				);
	INIT_MULTI_LANG_MEMBER(	Name 				);
	INIT_MULTI_LANG_MEMBER(	Description 		);
	INIT_MULTI_LANG_MEMBER(	OnSelect			);
	INIT_MULTI_LANG_MEMBER(	OnDeselect			);
	INIT_MULTI_LANG_MEMBER(	nMaxSelect			);
	INIT_MULTI_LANG_MEMBER(	bMinimal			);
	INIT_MULTI_LANG_MEMBER(	bDefault			);
	INIT_MULTI_LANG_MEMBER(	bHiddenRoot			);
	INIT_MULTI_LANG_MEMBER(	bHiddenRootRecursive );
	INIT_MULTI_LANG_MEMBER(	Files				);
	INIT_MULTI_LANG_MEMBER(	Dirs				);
	INIT_MULTI_LANG_MEMBER(	Procedures			);
	INIT_MULTI_LANG_MEMBER(	Customs				);
	INIT_MULTI_LANG_MEMBER(	Modules				);
	INIT_MULTI_LANG_MEMBER(	Profiles			);
	INIT_MULTI_LANG_MEMBER(	ProfileItems		);
	INIT_MULTI_LANG_MEMBER(	StarRegistryItems	);
	INIT_MULTI_LANG_MEMBER(	RegistryItems		);
	INIT_MULTI_LANG_MEMBER(	RegistryAreas		);
	INIT_MULTI_LANG_MEMBER(	FolderItems			);
	INIT_MULTI_LANG_MEMBER(	Os2Classes			);
	INIT_MULTI_LANG_MEMBER(	Os2Templates		);
	INIT_MULTI_LANG_MEMBER(	nEditions			);
	INIT_MULTI_LANG_MEMBER(	bInstalled			);
	INIT_MULTI_LANG_MEMBER(	bSelected			);
	INIT_MULTI_LANG_MEMBER(	bDontSelectByUser	);
	INIT_MULTI_LANG_MEMBER(	bHasChangeUninstall	);
	INIT_MULTI_LANG_MEMBER(	bHasChangeInstall	);
	INIT_MULTI_LANG_MEMBER(	eExecType			);
	INIT_MULTI_LANG_MEMBER(	eExecCommand		);
	INIT_MULTI_LANG_MEMBER(	eFollowExecType		);
	INIT_MULTI_LANG_MEMBER(	eFollowExecCommand	);

	m_pParent    			= NULL;
	m_nEditions				= 0UL;
	m_bMinimal   			= FALSE;
	m_bDefault  	   		= FALSE;
	m_bInstalled			= FALSE;
	m_bHiddenRoot			= FALSE;
	m_bHiddenRootRecursive	= FALSE;
	m_bHasChangeUninstall 	= FALSE;
	m_bHasChangeInstall		= FALSE;
	m_bSelected 			= FALSE;
	m_bDontSelectByUser		= FALSE;
	m_nMaxSelect			= 0;

	m_pDoneFileList			= new SiDoneModList( 2 );
	m_pDoneDirList         	= new SiDoneModList( 2 );
}

// ----------------------------------------------------------------------------

SiModule::~SiModule()
{
	delete m_pDoneFileList;
	delete m_pDoneDirList;
}

// ----------------------------------------------------------------------------

void SiModule::Add(SiFile *pFile)
{
	if( m_pDoneFileList->Find(pFile->GetID()) )
		return;
	m_pDoneFileList->Insert(pFile->GetID(), 1);

	m_IsFiles = TRUE;
	m_Files.Insert(pFile,LIST_APPEND);
	pFile->AddRefCount();
}

void SiModule::Add(SiDirectory *pDir)
{
	if( m_pDoneDirList->Find(pDir->GetID()) )
		return;
	m_pDoneDirList->Insert(pDir->GetID(), 1);

	m_IsDirs = TRUE;
	m_Dirs.Insert(pDir,LIST_APPEND);
	pDir->AddRefCount();
}

// ----------------------------------------------------------------------------

void SiModule::JoinWithParent()
{
	if( !IsLangRef() )
		return ;

	SiModule* pParent = (SiModule*) m_pLangParent;

	JOIN_MULTI_LANG_MEMBER(	pParent				);
	JOIN_MULTI_LANG_MEMBER(	Name 				);
	JOIN_MULTI_LANG_MEMBER(	Description 		);
	JOIN_MULTI_LANG_MEMBER(	OnSelect			);
	JOIN_MULTI_LANG_MEMBER(	OnDeselect			);
	JOIN_MULTI_LANG_MEMBER(	nMaxSelect			);
	JOIN_MULTI_LANG_MEMBER(	bMinimal			);
	JOIN_MULTI_LANG_MEMBER(	bDefault			);
	JOIN_MULTI_LANG_MEMBER(	bHiddenRoot			);
	JOIN_MULTI_LANG_MEMBER(	bHiddenRootRecursive );
	JOIN_MULTI_LANG_MEMBER(	Files				);
	JOIN_MULTI_LANG_MEMBER(	Dirs				);
	JOIN_MULTI_LANG_MEMBER(	Procedures			);
	JOIN_MULTI_LANG_MEMBER(	Customs				);
	JOIN_MULTI_LANG_MEMBER(	Modules				);
	JOIN_MULTI_LANG_MEMBER(	Profiles			);
	JOIN_MULTI_LANG_MEMBER(	ProfileItems		);
	JOIN_MULTI_LANG_MEMBER(	StarRegistryItems	);
	JOIN_MULTI_LANG_MEMBER(	RegistryItems		);
	JOIN_MULTI_LANG_MEMBER(	RegistryAreas		);
	JOIN_MULTI_LANG_MEMBER(	FolderItems			);
	JOIN_MULTI_LANG_MEMBER(	Os2Classes			);
	JOIN_MULTI_LANG_MEMBER(	Os2Templates		);
	JOIN_MULTI_LANG_MEMBER(	nEditions			);
	JOIN_MULTI_LANG_MEMBER(	bInstalled			);
	JOIN_MULTI_LANG_MEMBER(	bSelected			);
	JOIN_MULTI_LANG_MEMBER(	bDontSelectByUser	);
	JOIN_MULTI_LANG_MEMBER(	bHasChangeUninstall	);
	JOIN_MULTI_LANG_MEMBER(	bHasChangeInstall	);
	JOIN_MULTI_LANG_MEMBER(	eExecType			);
	JOIN_MULTI_LANG_MEMBER(	eExecCommand		);
	JOIN_MULTI_LANG_MEMBER(	eFollowExecType		);
	JOIN_MULTI_LANG_MEMBER(	eFollowExecCommand	);
}

// ----------------------------------------------------------------------------

BOOL SiModule::SetProperty(ByteString const& aProperty, ByteString const& aValue)
{
	if (aProperty == PROPERTY_NAME)
	{
		SET_MULTI_LANG_MEMBER( Name, aValue );
		return TRUE;
	}
	if (aProperty == PROPERTY_DESCRIPTION)
	{
		SET_MULTI_LANG_MEMBER( Description, aValue );
		return TRUE;
	}
	if (aProperty == PROPERTY_ONSELECT)
	{
		SET_MULTI_LANG_MEMBER( OnSelect, aValue );
		return TRUE;
	}
	if (aProperty == PROPERTY_ONDESELECT)
	{
		SET_MULTI_LANG_MEMBER( OnDeselect, aValue );
		return TRUE;
	}
	if (aProperty == PROPERTY_MINIMAL)
	{
		BOOL bBool;
		BOOL bRet = SetBool( bBool, aValue );
		SET_MULTI_LANG_MEMBER( bMinimal, bBool );
		return bRet;
	}
	if (aProperty == PROPERTY_DEFAULT)
	{
		BOOL bBool;
		BOOL bRet = SetBool( bBool, aValue );
		SET_MULTI_LANG_MEMBER( bDefault, bBool );
		return bRet;
	}
	if (aProperty == PROPERTY_INSTALLED)
	{
		BOOL bBool;
		BOOL bRet = SetBool( bBool, aValue );
		SET_MULTI_LANG_MEMBER( bInstalled, bBool );
		return bRet;
	}

	if (aProperty == PROPERTY_FLAGS)
	{
		if (aValue == VALUE_HIDDENROOT)
		{
			SET_MULTI_LANG_MEMBER( bHiddenRoot, TRUE );
			return TRUE;
		}
		if (aValue == VALUE_HIDDENROOT_RECURSIVE)
		{
			SET_MULTI_LANG_MEMBER( bHiddenRootRecursive, TRUE );
			return TRUE;
		}
		if (aValue == VALUE_DONT_SELECT_BY_USER)
		{
			SET_MULTI_LANG_MEMBER( bDontSelectByUser, TRUE );
			return TRUE;
		}
	}

	if( aProperty == PROPERTY_EXECCOMMAND )
	{
		SET_MULTI_LANG_MEMBER( eExecCommand, aValue );
		return TRUE;
	}
	if( aProperty == PROPERTY_FOLLOW_EXECCOMMAND )
	{
		SET_MULTI_LANG_MEMBER( eFollowExecCommand, aValue );
		return TRUE;
	}

	if( aProperty == PROPERTY_EXECTYPE )
	{
		if( aValue == VALUE_EXEC_SCRIPT )
		{
			SET_MULTI_LANG_MEMBER( eExecType, EXEC_SCRIPT );
			return TRUE;
		}
		if( aValue == VALUE_EXEC_APP )
		{
			SET_MULTI_LANG_MEMBER( eExecType, EXEC_APP );
			return TRUE;
		}
		if( aValue == VALUE_EXEC_SHELL )
		{
			SET_MULTI_LANG_MEMBER( eExecType, EXEC_SHELL );
			return TRUE;
		}
		return FALSE;
	}

	if( aProperty == PROPERTY_FOLLOW_EXECTYPE )
	{
		if( aValue == VALUE_EXEC_APP )
		{
			SET_MULTI_LANG_MEMBER( eExecType, EXEC_APP );
			return TRUE;
		}
		if( aValue == VALUE_EXEC_SHELL )
		{
			SET_MULTI_LANG_MEMBER( eExecType, EXEC_SHELL );
			return TRUE;
		}
		return FALSE;
	}

	return SiDeclarator::SetProperty(aProperty,aValue);
}

BOOL SiModule::SetProperty(ByteString const& aProperty, SiDeclarator* aValue)
{
	if (aProperty == PROPERTY_PARENTID)
	{
		if (aValue == NULL)
		{
			m_pParent = NULL;
			return TRUE;
		}
		m_pParent = PTR_CAST(SiModule,aValue);

		if (m_pParent==NULL)
		{
			Error("parent is not a module");
			return FALSE;
		}
		m_IspParent = TRUE;
		return TRUE;
	}

	if (aProperty == PROPERTY_FILES)
	{
		SiFile *pFile = PTR_CAST(SiFile,aValue);

		if (pFile == NULL)
		{
			ByteString aMsg(aValue->GetID());
			aMsg += " is not a file";
			Error( aMsg );
			return FALSE;
		}

		if (pFile->GetPartOf() != NULL)
		{
			ByteString aMsg(aValue->GetID());
			aMsg += " is a filepart and must not be linked to a module";
			Error( aMsg );
			return FALSE;
		}

		Add(pFile);
		return TRUE;
	}

	if (aProperty == PROPERTY_DIRS)
	{
		SiDirectory *pDir = PTR_CAST(SiDirectory,aValue);

		if (pDir == NULL)
		{
			ByteString aMsg(aValue->GetID());
			aMsg += " is not a directory";
			Error( aMsg );
			return FALSE;
		}

		Add(pDir);
		return TRUE;
	}

	if (aProperty == PROPERTY_PROCEDURES)
	{
		SiProcedure *pProc = PTR_CAST(SiProcedure,aValue);
		if (pProc == NULL)
		{
			ByteString aMsg(aValue->GetID());
			aMsg += " is not a procedure";
			Error( aMsg );
			return FALSE;
		}
		Add(pProc);
		return TRUE;
	}

	if( aProperty == PROPERTY_CUSTOMS )
	{
		SiCustom *pCustom = PTR_CAST(SiCustom,aValue);
		if(pCustom == NULL)
		{
			ByteString aMsg(aValue->GetID());
			aMsg += " is not a custom object";
			Error( aMsg );
			return FALSE;
		}
		Add(pCustom);
		return TRUE;
	}

	return SiDeclarator::SetProperty(aProperty,aValue);
}

BOOL SiModule::SetProperty(ByteString const& aProperty, long aValue)
{
	if( aProperty == PROPERTY_MAXSELECT )
	{
		SET_MULTI_LANG_MEMBER( nMaxSelect, (USHORT)aValue );
		return TRUE;
	}
	return SiDeclarator::SetProperty(aProperty,aValue);
}

void SiModule::Add(SiModule* pModule)
{
	m_Modules.Insert( pModule, LIST_APPEND );
	m_IsModules = TRUE;
}

BOOL SiModule::Check()
{
	if( GetCompiler() && !GetCompiler()->IsApplication() )
		return TRUE; // commandline second level parse

	BOOL bOk = TRUE;
	bOk = bOk && CheckField(m_Name, PROPERTY_NAME);
	return bOk ? SiDeclarator::Check() : FALSE;
}

BOOL SiModule::WriteTo(SiDatabase& aStream) const
{
	if( !IsLangRef() )
		aStream.BeginDeclaration("Module", this);

	// PROPERTY_PARENTID
	WRITE_MULTI_LANG_MEMBER( PROPERTY_PARENTID, pParent );

	// PROPERTY_NAME
	WRITE_MULTI_LANG_MEMBER( PROPERTY_NAME, Name );

	// PROPERTY_DESCRIPTION
	WRITE_MULTI_LANG_MEMBER( PROPERTY_DESCRIPTION, Description );

	// PROPERTY_ONSELECT
	WRITE_MULTI_LANG_MEMBER( PROPERTY_ONSELECT,	OnSelect );

	// PROPERTY_ONDESELECT
	WRITE_MULTI_LANG_MEMBER( PROPERTY_ONDESELECT, OnDeselect );

	// PROPERTY_MINIMAL
	WRITE_MULTI_LANG_MEMBER( PROPERTY_MINIMAL, bMinimal );

	// PROPERTY_DEFAULT
	WRITE_MULTI_LANG_MEMBER( PROPERTY_DEFAULT, bDefault );

	// PROPERTY_INSTALLED
	WRITE_MULTI_LANG_MEMBER( PROPERTY_INSTALLED, bInstalled );

	// PROPERTY_MAXSELECT
	if( m_nMaxSelect != 0 )
		WRITE_MULTI_LANG_MEMBER( PROPERTY_MAXSELECT, nMaxSelect );

	// PROPERTY_FILES
	if( m_Files.Count() != 0 )
	{
		aStream.BeginProperty( PROPERTY_FILES, m_nLanguage );
		aStream.BeginList();
		for( USHORT i=0; i < m_Files.Count(); i++ )
			aStream.AddListValue( m_Files.GetObject(i) );
		aStream.EndList();
		aStream.EndProperty();
	}

	// PROPERTY_DIRS
	if( m_Dirs.Count() != 0 )
	{
		aStream.BeginProperty( PROPERTY_DIRS, m_nLanguage );
		aStream.BeginList();
		for( USHORT i=0; i < m_Dirs.Count(); i++ )
		{
			SiDirectory* pDir = m_Dirs.GetObject(i);
			if( pDir->GetSingleName().CompareTo("PREDEFINED_", 11) != COMPARE_EQUAL )
				aStream.AddListValue( pDir );
		}
		aStream.EndList();
		aStream.EndProperty();
	}

	// PROPERTY_PROCEDURES
	if( m_Procedures.Count() != 0 )
	{
		aStream.BeginProperty( PROPERTY_PROCEDURES, m_nLanguage );
		aStream.BeginList();
		for( USHORT i=0; i < m_Procedures.Count(); i++ )
			aStream.AddListValue( m_Procedures.GetObject(i) );
		aStream.EndList();
		aStream.EndProperty();
	}

	// PROPERTY_CUSTOMS
	if( m_Customs.Count() != 0 )
	{
		aStream.BeginProperty(PROPERTY_CUSTOMS, m_nLanguage );
		aStream.BeginList();
		for( USHORT i=0; i < m_Customs.Count(); i++ )
			aStream.AddListValue( m_Customs.GetObject(i) );
		aStream.EndList();
		aStream.EndProperty();
	}

	// PROPERTY_FLAGS
	if( m_IsbHiddenRoot || m_IsbHiddenRootRecursive || m_IsbDontSelectByUser )
	{
		aStream.BeginProperty( PROPERTY_FLAGS, m_nLanguage );
		aStream.BeginList();

		if( m_IsbHiddenRoot )
			aStream.AddListValue( SiIdentifier(VALUE_HIDDENROOT) );
		if( m_IsbHiddenRootRecursive )
			aStream.AddListValue( SiIdentifier(VALUE_HIDDENROOT_RECURSIVE) );
		if( m_IsbDontSelectByUser )
			aStream.AddListValue( SiIdentifier(VALUE_DONT_SELECT_BY_USER) );
		aStream.EndList();
		aStream.EndProperty();
	}

	for( USHORT nIdx = 0; nIdx < m_aLangRefs.Count(); ++nIdx )
	{
		SiModule* pMod = (SiModule*) m_aLangRefs.GetObject(nIdx);
		pMod->WriteTo( aStream );
	}

	if( !IsLangRef() )
		aStream.EndDeclaration();

	return TRUE;
}

ByteString SiModule::GetNaturalID() const
{
	ByteString aNatID(m_Name);
	if( IsLangRef() ) {
		aNatID += "__";
        aNatID += ByteString::CreateFromInt32( m_nLanguage );
	}
	return aNatID;
}

BOOL SiModule::IsHiddenRecursive()
{
	SiModule* pMod = this;

	while( !pMod->IsHiddenRecursiveFlag() && pMod->GetParent() )
		pMod = pMod->GetParent();

	if( pMod->IsHiddenRecursiveFlag() )
		return TRUE;
	return FALSE;
}

void SiModule::Select (Selection eModSel)
{
	if( !(eModSel == ALL_UNSEL && m_bDontSelectByUser) )
	{
		if( eModSel == ALL_UNSEL && m_bInstalled )
			m_bSelected = TRUE;
		else
			m_bSelected =	eModSel == THIS_SEL		||
							eModSel == ALL_SEL		||
					  		(eModSel == ALL_DEFAULT   && (m_bDefault || m_pParent == NULL))	||
					  		(eModSel == ALL_MINIMAL   && (m_bMinimal || m_pParent == NULL))	||
					  		(eModSel == ALL_INSTALLED && m_bInstalled);
	}

	if( eModSel != THIS_SEL && eModSel != THIS_UNSEL )
		for (USHORT i=0; i<m_Modules.Count(); i++)
			m_Modules.GetObject(i)->Select(eModSel);
}

#if defined(OS2)
#define _EA_SPACE_OS2
#endif

ULONG SiModule::_CalculateSize( SiFile* pFile, ULONG lThisSize, MType eType, ULONG lClusterSize, BOOL bSystem,
	BOOL bWorkstation, BOOL bIsFATFileSystem ) const
{
	if(lClusterSize==0)
		return 0;

	ULONG lSize = 0;
	if( eType != TEMP )
	{
		if( pFile->IsArchive() )
		{
			ULONG nMin 		= pFile->GetArchiveFiles() * lClusterSize;
			ULONG nArchSz	= (ULONG) (pFile->GetArchiveSize() / lClusterSize);
			lSize = Max( ((nArchSz+1) * lClusterSize), nMin );

			#ifdef _EA_SPACE_OS2
			if( bIsFATFileSystem )
				lSize += nMin;
			#endif
		}
		else
		{
			lSize = (ULONG) (pFile->GetSize() / lClusterSize);
			lSize = (lSize+1) * lClusterSize;

			#ifdef _EA_SPACE_OS2
			if( bIsFATFileSystem )
				lSize += lClusterSize;
			#endif
		}

		ULONG lSize2 = pFile->GetSubfileList().Count() * lClusterSize;

		lSize = Max(lSize,lSize2);
		lSize = Max(lSize,lClusterSize);

		if( bSystem )
		{
			if( pFile->IsSystem() )
				return lSize;
		}
		else
		{
			if( bWorkstation )
			{
				if( pFile->InstallOnWorkstation() )
				{
					if( pFile->IsSoftLinkPrefered() )
						return lClusterSize;
					else
						return lSize;
				}
			}
			else
			{
				return lSize;
			}
		}
	}
	else
	{	// TEMP
		if( pFile->IsHelpFile() && ((bWorkstation && pFile->InstallOnWorkstation()) || !bWorkstation) )
			if( lThisSize < (ULONG)pFile->GetSize() )
				return pFile->GetSize();
	}
	return 0;
}

ULONG SiModule::CalculateSize (
	SiEnvironment &rEnv,
	MType	eType,
	ULONG	lClusterSize,		// Des Ziellaufwerkes
	BOOL	bSystem,			// = FALSE
	BOOL	bWorkstation,		// = FALSE
	BOOL 	bIsFATFileSystem	// = FALSE
) const
{
	// #89644#, #91921#
	// 6th Sep. 2001
	// LLA: average, we need 10mb more memory for dynamic grown files at startup.
	// e.g. applicat.rdb, configuration, ...

	ULONG nSize = CalculateSize_impl(rEnv, eType, lClusterSize, bSystem, bWorkstation, bIsFATFileSystem);
	if (eType == SiModule::DEFAULT ||
		eType == SiModule::MINIMAL ||
		eType == SiModule::INSTALL)
	{
		if (!bWorkstation)
		{
			nSize += 10 * 1024 * 1024;
		}
	}
	return nSize;
}

ULONG SiModule::CalculateSize_impl(
	SiEnvironment &rEnv,
	MType	eType,
	ULONG	lClusterSize,		// Des Ziellaufwerkes
	BOOL	bSystem,			// = FALSE
	BOOL	bWorkstation,		// = FALSE
	BOOL 	bIsFATFileSystem	// = FALSE
) const
{
	ULONG lThisSize = 0;

	for( USHORT i = 0; i < m_Files.Count(); i++ )
	{
		SiFile* pFile = m_Files.GetObject(i);
		if( pFile->GetName().CompareIgnoreCaseToAscii(README_ZIP) == COMPARE_EQUAL )
			continue;

		if( pFile->HasLangRef() )
		{
			SiLangCtxList& rLst = rEnv.GetUILanguageContext();
			for( USHORT xx = 0; xx < rLst.Count(); ++xx )
			{
				LanguageContext* pCtx = rLst.GetObject(xx);
				if( pCtx && pCtx->isProg )
				{
					SiFile* pLangRef = NULL;
					if( pCtx->nLanguage == LANG_DEFAULT )
						pLangRef = pFile;
					else
						pLangRef = (SiFile*) pFile->GetLangRef( pCtx->nLanguage );

					SiFile* pCalcFile = pFile;
					if( pLangRef )
					{
						pLangRef->JoinWithParent();
						pCalcFile = pLangRef;
					}

					lThisSize += _CalculateSize( pCalcFile, lThisSize, eType, lClusterSize,
									 bSystem, bWorkstation, bIsFATFileSystem );
				}
			}
		}
		else
		{
			lThisSize += _CalculateSize( pFile, lThisSize, eType, lClusterSize,
										 bSystem, bWorkstation, bIsFATFileSystem );
		}
	}

	if( eType == ONLYTHIS )
		return lThisSize;

	ULONG lAllSize = 0;

	// #89644#
	if ( eType == ALL
	||  (eType == MINIMAL   && (m_bMinimal || m_pParent == NULL))
	||  (eType == DEFAULT   && (m_bDefault || m_pParent == NULL))
	||  ((eType == INSTALL || eType == INSTALL_SYSTEM)   && m_bSelected && !m_bInstalled)
	||  (eType == UNINSTALL && m_bSelected &&  m_bInstalled)
	||	(eType == TEMP 		&& m_bSelected) )
		lAllSize += lThisSize;

	// rekursiv runter
	for( USHORT x = 0; x < m_Modules.Count(); x++ )
		lAllSize += m_Modules.GetObject(x)->CalculateSize_impl( rEnv, eType, lClusterSize, bSystem, bWorkstation );

	return lAllSize;
}

///////////////////////////////////////////////////////////////////////////////
//
BOOL SiModule::HasObjects() const
{
	return m_Files.Count()			!= 0
		|| m_Customs.Count()  		!= 0
		|| m_FolderItems.Count()	!= 0
		|| m_Profiles.Count()		!= 0
		|| m_ProfileItems.Count()	!= 0
		|| m_RegistryItems.Count()	!= 0
		|| m_RegistryAreas.Count()	!= 0
		|| m_Os2Classes.Count()		!= 0
		|| m_Os2Templates.Count()	!= 0;
}

///////////////////////////////////////////////////////////////////////////////
//
BOOL SiModule::HasUISubModules() const
{
	for( USHORT i = 0; i < m_Modules.Count(); i++ )
		if( ! m_Modules.GetObject(i)->IsHidden() )
			return TRUE;
	return FALSE;
}

///////////////////////////////////////////////////////////////////////////////
//
BOOL SiModule::HasLangRef() const
{
	if( HasObjects() )
	{
		USHORT i;
		for( i = 0; i < m_Files.Count(); ++i )
			if( (m_Files.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_Customs.Count(); ++i )
			if( (m_Customs.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_FolderItems.Count(); ++i )
			if( (m_FolderItems.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_Customs.Count(); ++i )
			if( (m_Customs.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_Profiles.Count(); ++i )
			if( (m_Profiles.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_ProfileItems.Count(); ++i )
			if( (m_ProfileItems.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_RegistryItems.Count(); ++i )
			if( (m_RegistryItems.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_RegistryAreas.Count(); ++i )
			if( (m_RegistryAreas.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_Os2Classes.Count(); ++i )
			if( (m_Os2Classes.GetObject(i))->HasLangRef() )
				return TRUE;

		for( i = 0; i < m_Os2Templates.Count(); ++i )
			if( (m_Os2Templates.GetObject(i))->HasLangRef() )
				return TRUE;
	}

	if( m_IsName || m_IsDescription || m_IsOnSelect || m_IsOnDeselect )
		return TRUE;

	return FALSE;
}

// -------------------------------------------------------------------
// SiModuleSet Hilfsklasse
// -------------------------------------------------------------------

void SiModuleSet::Select()
{
	for( USHORT i = 0; i < aModLst.Count(); ++i )
		aModLst.GetObject(i)->Select(SiModule::THIS_SEL);
}

void SiModuleSet::DeSelect()
{
	for( USHORT i = 0; i < aModLst.Count(); ++i )
		aModLst.GetObject(i)->Select(SiModule::THIS_UNSEL);
}


///////////////////////////////////////////////////////////////////////////////
//
//		SiModuleIds
//

SiModuleIds::SiModuleIds( SiIdentifier const& ID,
                          SiCompiler* pCompiler )
: SiDeclarator(ID,pCompiler)
{
	INIT_MULTI_LANG_MEMBER(	Name	);
    INIT_MULTI_LANG_MEMBER( Modules );
    INIT_MULTI_LANG_MEMBER( bRecursive );

	m_bRecursive          = FALSE;
}

// ----------------------------------------------------------------------------

SiModuleIds::~SiModuleIds()
{
}

// ----------------------------------------------------------------------------

void SiModuleIds::JoinWithParent()
{
	if( !IsLangRef() )
		return ;

	SiModuleIds* pParent = (SiModuleIds*) m_pLangParent;

	JOIN_MULTI_LANG_MEMBER(	Name );
    JOIN_MULTI_LANG_MEMBER( Modules );
    JOIN_MULTI_LANG_MEMBER( bRecursive );
}

// ----------------------------------------------------------------------------

BOOL SiModuleIds::SetProperty( ByteString const& aProperty,
                               ByteString const& aValue)
{
	if (aProperty == PROPERTY_NAME)
	{
		SET_MULTI_LANG_MEMBER( Name, aValue );
		return TRUE;
	}
	if( aProperty == PROPERTY_FLAGS )
	{
		if( aValue == VALUE_RECURSIVE )
		{
			SET_MULTI_LANG_MEMBER(bRecursive, TRUE);
			return TRUE;
		}
		ByteString aMsg( "unknown value " );
		aMsg += aValue;
		Error( aMsg );

		return FALSE;
	}

	return SiDeclarator::SetProperty(aProperty,aValue);
}

// ----------------------------------------------------------------------------

BOOL SiModuleIds::SetProperty( ByteString const& aProperty,
                               SiDeclarator* aValue )
{
	if ( aProperty == PROPERTY_MODULEIDS )
	{
		SiModule *pModule = PTR_CAST( SiModule, aValue );

		if ( pModule == NULL )
		{
			ByteString aMsg( aValue->GetID() );
			aMsg += " is not a module";
			Error( aMsg );
			return FALSE;
		}

		Add( pModule );
		return TRUE;
	}

    return FALSE;
}

// ----------------------------------------------------------------------------

BOOL SiModuleIds::Check()
{
	if( GetCompiler() && !GetCompiler()->IsApplication() )
		return TRUE; // commandline second level parse

	if ( m_Modules.Count() == 0)
	{
//		Missing( PROPERTY_MODULEIDS );
		return FALSE;
	}

    return TRUE;
}

// ----------------------------------------------------------------------------

BOOL SiModuleIds::WriteTo( SiDatabase& aStream ) const
{
    if( !IsLangRef() )
        aStream.BeginDeclaration("ModuleList", this);

	// PROPERTY_NAME
	WRITE_MULTI_LANG_MEMBER( PROPERTY_NAME, Name );

    // PROPERTY_MODULEIDS
    if( m_Modules.Count() != 0 )
    {
        aStream.BeginProperty( PROPERTY_MODULEIDS, m_nLanguage );
        aStream.BeginList();
        for( USHORT i=0; i < m_Modules.Count(); i++ )
            aStream.AddListValue( m_Modules.GetObject(i) );
        aStream.EndList();
        aStream.EndProperty();
    }

    // PROPERTY_FLAGS
    if( m_IsbRecursive )
    {
        aStream.BeginProperty( PROPERTY_FLAGS, m_nLanguage );
        aStream.BeginList();

        if( m_IsbRecursive )
            aStream.AddListValue( SiIdentifier( VALUE_RECURSIVE ) );

        aStream.EndList();
        aStream.EndProperty();
    }

	for( USHORT nIdx = 0; nIdx < m_aLangRefs.Count(); ++nIdx )
	{
		SiModuleIds* pIDs = (SiModuleIds*) m_aLangRefs.GetObject(nIdx);
		pIDs->WriteTo( aStream );
	}


    if( !IsLangRef() )
        aStream.EndDeclaration();

    return TRUE;
}

// ----------------------------------------------------------------------------

ByteString SiModuleIds::GetNaturalID() const
{
	ByteString aNatID(m_Name);
	if( IsLangRef() ) {
		aNatID += "__";
		aNatID += ByteString::CreateFromInt32( m_nLanguage );
	}
	return aNatID;
}

// ----------------------------------------------------------------------------

void SiModuleIds::Add( SiModule* pModule )
{
	m_Modules.Insert( pModule, LIST_APPEND );
	m_IsModules = TRUE;
}


///////////////////////////////////////////////////////////////////////////////
//
//		SiModulesSet
//

SiModulesSet::SiModulesSet( SiIdentifier const& ID,
                            SiCompiler* pCompiler )
: SiDeclarator( ID, pCompiler )
{
	INIT_MULTI_LANG_MEMBER(	Name );
	INIT_MULTI_LANG_MEMBER(	Description );
    INIT_MULTI_LANG_MEMBER( ModulesList );
}

// ----------------------------------------------------------------------------

SiModulesSet::~SiModulesSet()
{
}

// ----------------------------------------------------------------------------

void SiModulesSet::JoinWithParent()
{
	if( !IsLangRef() )
		return ;

	SiModulesSet* pParent = (SiModulesSet*) m_pLangParent;

	JOIN_MULTI_LANG_MEMBER(	Name );
	JOIN_MULTI_LANG_MEMBER(	Description );
    JOIN_MULTI_LANG_MEMBER( ModulesList );
}

// ----------------------------------------------------------------------------

BOOL SiModulesSet::SetProperty( ByteString const& aProperty,
                                ByteString const& aValue)
{
	if ( aProperty == PROPERTY_NAME )
	{
		SET_MULTI_LANG_MEMBER( Name, aValue );
		return TRUE;
	}
	if (aProperty == PROPERTY_DESCRIPTION)
	{
		SET_MULTI_LANG_MEMBER( Description, aValue );
		return TRUE;
	}

    return FALSE;
}

// ----------------------------------------------------------------------------

BOOL SiModulesSet::SetProperty( ByteString const& aProperty,
                                SiDeclarator* aValue)
{
	if ( aProperty == PROPERTY_MODULELISTS )
	{
		SiModuleIds *pModuleIds = PTR_CAST( SiModuleIds, aValue );

		if ( pModuleIds == NULL )
		{
			ByteString aMsg( aValue->GetID() );
			aMsg += " is not a modules list";
			Error( aMsg );
			return FALSE;
		}

		Add( pModuleIds );
		return TRUE;
	}

    return FALSE;
}

// ----------------------------------------------------------------------------

BOOL SiModulesSet::Check()
{
	if( GetCompiler() && !GetCompiler()->IsApplication() )
		return TRUE; // commandline second level parse

	BOOL bOk = CheckField( m_Name, PROPERTY_NAME );

    if ( bOk && ( m_ModulesList.Count() == 0 ) )
	{
//		Missing( PROPERTY_MODULELISTS );
        bOk = FALSE;
    }

    return bOk ? SiDeclarator::Check() : FALSE;
}

// ----------------------------------------------------------------------------

BOOL SiModulesSet::WriteTo( SiDatabase& aStream ) const
{
    if( !IsLangRef() )
        aStream.BeginDeclaration("ModuleSet", this);

	// PROPERTY_NAME
	WRITE_MULTI_LANG_MEMBER( PROPERTY_NAME, Name );

	// PROPERTY_DESCRIPTION
	WRITE_MULTI_LANG_MEMBER( PROPERTY_DESCRIPTION, Description );

    // PROPERTY_MODULELISTS
    if( m_ModulesList.Count() != 0 )
    {
        aStream.BeginProperty( PROPERTY_MODULELISTS, m_nLanguage );
        aStream.BeginList();
        for( USHORT i=0; i < m_ModulesList.Count(); i++ )
            aStream.AddListValue( m_ModulesList.GetObject( i ) );
        aStream.EndList();
        aStream.EndProperty();
    }

	for( USHORT nIdx = 0; nIdx < m_aLangRefs.Count(); ++nIdx )
	{
		SiModulesSet* pSet = (SiModulesSet*) m_aLangRefs.GetObject(nIdx);
		pSet->WriteTo( aStream );
	}

    if( !IsLangRef() )
        aStream.EndDeclaration();

    return TRUE;
}

// ----------------------------------------------------------------------------

ByteString SiModulesSet::GetNaturalID() const
{
	ByteString aNatID(m_Name);
	if( IsLangRef() ) {
		aNatID += "__";
		aNatID += ByteString::CreateFromInt32( m_nLanguage );
	}
	return aNatID;
}


// ----------------------------------------------------------------------------

void SiModulesSet::Add( SiModuleIds* pModuleList )
{
	m_ModulesList.Insert( pModuleList, LIST_APPEND );
	m_IsModulesList = TRUE;
}


///////////////////////////////////////////////////////////////////////////////
//
//		SiInstallation
//

SiInstallation::SiInstallation(SiIdentifier const& ID, SiCompiler *pCompiler)
: SiDeclarator(ID,pCompiler, FALSE /* bSystemObject */)
{
	m_eInstallMode	  			= IM_INVALID;
	m_bInstallFromNet 			= FALSE;
	m_nScriptVersion  			= 100; // 1.00
	m_bForceOverwrite 			= FALSE;
	m_bKeepOldVersion 			= FALSE;
	m_bOldVersionRequired 		= FALSE;
	m_bRequiresShare  			= FALSE;
	m_bModifyPath	  			= FALSE;
	m_bRegistrationRequired 	= FALSE;
	m_nUpdateSize				= 0L;
	m_bOUpdateDone				= FALSE;
	m_bHideStandard				= FALSE;
	m_bOnlyCustom				= FALSE;
	m_bNeedCfgServer			= FALSE;
	m_nPatchLevel				= 0;
	m_bPatched					= FALSE;
}

// ----------------------------------------------------------------------------

void SiInstallation::JoinWithParent()
{
	if( !IsLangRef() )
	{
		DBG_ASSERT( !this, "JoinWhithParent am Parent gerufen?" );
		return ;
	}
}

// ----------------------------------------------------------------------------

BOOL SiInstallation::SetProperty(ByteString const& aProperty, ByteString const& aValue)
{
	if (aProperty == PROPERTY_MODE)
	{
		if (aValue == VALUE_STANDALONE)
		{
			m_eInstallMode = IM_STANDALONE;
			return TRUE;
		}

		if (aValue == VALUE_NETWORK)
		{
			m_eInstallMode = IM_NETWORK;
			return TRUE;
		}

		if (aValue == VALUE_WORKSTATION)
		{
			m_eInstallMode = IM_WORKSTATION;
			return TRUE;
		}

		if (aValue == VALUE_APPSERVER)
		{
			m_eInstallMode = IM_APPSERVER;
			return TRUE;
		}

		if (aValue == VALUE_PATCH)
		{
			m_eInstallMode = IM_PATCH;
			return TRUE;
		}

		ByteString aMsg( aValue );
		aMsg += " is an illegal value";
		Error( aMsg );

		return FALSE;
	}

	if (aProperty == PROPERTY_SUITENAME)
	{
		m_aSuiteName = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_VENDORNAME)
	{
		m_aVendorName = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_VENDORVERSION)
	{
		m_aVendorVersion = aValue;
		return TRUE;
	}

    if (aProperty == PROPERTY_PRODUCTNAME)
	{
		m_aProductName = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_PRODUCTVERSION)
	{
		m_aProductVersion = aValue;
		return TRUE;
	}
	if (aProperty == PROPERTY_INTERNALPRODUCTVERSION)
	{
		m_aInternalProductVersion = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_TEMPLATEFILE)
	{
		m_aTemplateFilename = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_FOLLOWAPP)
	{
		m_aFollowApp = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_VENDORBITMAP)
	{
		m_aVendorBMP = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_BITMAP)
	{
		m_aProductBMP = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_COMPANYNAME)
	{
		m_aCompanyName = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_INSTALLFROMNET)
		return SetBool(m_bInstallFromNet,aValue);

	if (aProperty == PROPERTY_SOURCEPATH)
	{
		m_aSourcePath = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_DEFAULTDESTPATH)
	{
		m_aDefaultDestPath = aValue;
		if( !GetCompiler()->GetCScript()->IsSecondLevel() && GetCompiler()->IsApplication() )
		{
// #91677#
// #ifdef WNT
//			if (GetCompiler()->GetEnvironment().IsLocal())
//			{
//				m_aDefaultDestPath.SearchAndReplace("<winprogpath>",
//				ByteString(WinOS::SHGetUserProgramFilesFolder(),
//						   osl_getThreadTextEncoding()));
//			}
// #endif

		}
		return TRUE;
	}

	if (aProperty == PROPERTY_DESTPATH)
	{
		m_aDestPath = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_UPDATEFOR)
	{
		m_anUpdateFor = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_UPGRADEFOR)
	{
		m_anUpgradeFor = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_FORCEOVERWRITE)
		return SetBool(m_bForceOverwrite,aValue);

	if( aProperty == PROPERTY_UIPROC )
	{
		m_aUiProcName = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_FLAGS)
	{
		if( aValue == VALUE_NEEDCONFIG_SERVER )
		{
			m_bNeedCfgServer = TRUE;
			return TRUE;
		}

		if( aValue == VALUE_ONLYCUSTOM )
		{
			m_bOnlyCustom = TRUE;
			return TRUE;
		}

		if (aValue == VALUE_HIDESTANDARD)
		{
			m_bHideStandard = TRUE;
			return TRUE;
		}

		if (IsWin()
		&&	aValue == VALUE_WIN_REQUIRES_SHARE)
		{
			m_bRequiresShare = TRUE;
			return TRUE;
		}

		if (IsOs2()
		&&  aValue == VALUE_OS2_MODIFY_PATH)
		{
			m_bModifyPath = TRUE;
			return TRUE;
		}

		if (aValue == VALUE_REBOOT) { return TRUE; /* dead */ }

		if (aValue == VALUE_KEEP_OLD_VERSION)
		{
			m_bKeepOldVersion = TRUE;
			return TRUE;
		}

		if (aValue == VALUE_OLD_VERSION_REQUIRED)
		{
			m_bOldVersionRequired = TRUE;
			return TRUE;
		}

		if (aValue == VALUE_REGISTRATION_REQUIRED)
		{
			m_bRegistrationRequired = TRUE;
			return TRUE;
		}

		if( aValue == VALUE_PATCHED )
		{
			m_bPatched = TRUE;
			return TRUE;
		}

		ByteString aMsg( aValue );
		aMsg += " is an illegal value";
		Error( aMsg );

		return FALSE;
	}

	if (aProperty == PROPERTY_SPECIAL_VERSION)
	{
		m_aSpecialVersion = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_DELETE_KEY)
	{
		m_aDeleteKey = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_UPDATENAME)
	{
		m_aUpdateName = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_TRANSMITTER)
	{
		m_aTransmitter = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_DEFLANGUAGE)
	{
		m_aDefLanguage = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_LANGUAGES)
	{
		m_aLanguages = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_INST_LANGUAGES)
	{
		m_aInstLanguages = aValue;
		return TRUE;
	}

	return SiDeclarator::SetProperty(aProperty,aValue);
}

BOOL SiInstallation::SetProperty(ByteString const& aProperty, long aValue)
{
	if (aProperty == PROPERTY_SCRIPTVERSION)
	{
		m_nScriptVersion = int(aValue);
		return TRUE;
	}

	if (aProperty == PROPERTY_PATCH_LEVEL)
	{
		m_nPatchLevel = (USHORT)aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_UPDATESIZE)
	{
		m_nUpdateSize = aValue;
		return TRUE;
	}

	if (aProperty == PROPERTY_OUPDATEDONE )
	{
		if( aValue != 0 )
			m_bOUpdateDone = TRUE;
		else
			m_bOUpdateDone = FALSE;
		return TRUE;
	}

	return SiDeclarator::SetProperty(aProperty,aValue);
}

BOOL SiInstallation::Check()
{
	BOOL bOk = TRUE;

	if (m_bForceOverwrite
	||  m_bKeepOldVersion
	||  m_bOldVersionRequired)
		bOk = bOk && CheckField(m_anUpdateFor, PROPERTY_UPDATEFOR);

	if (m_anUpgradeFor.Len() != 0)
		bOk = bOk && CheckFieldEmpty(m_anUpdateFor, PROPERTY_UPDATEFOR);

	if (m_anUpdateFor.Len() != 0)
		bOk = bOk && CheckFieldEmpty(m_anUpgradeFor, PROPERTY_UPGRADEFOR);

	return bOk ? SiDeclarator::Check() : FALSE;
}

BOOL SiInstallation::WriteTo(SiDatabase& aStream) const
{
	BOOL bIsSecondLevel = GetCompiler()->GetCScript()->IsSecondLevel();
	aStream.BeginDeclaration("Installation", this);

	if( m_aProductName.Len() )
		aStream.WriteProperty( PROPERTY_PRODUCTNAME, m_aProductName );

	if( m_aSuiteName.Len() )
		aStream.WriteProperty( PROPERTY_SUITENAME, m_aSuiteName );

	if( m_aProductVersion.Len() )
		aStream.WriteProperty( PROPERTY_PRODUCTVERSION, m_aProductVersion );

	if( m_aInternalProductVersion.Len() )
		aStream.WriteProperty( PROPERTY_INTERNALPRODUCTVERSION, m_aInternalProductVersion );

	if( m_aVendorName.Len() )
		aStream.WriteProperty( PROPERTY_VENDORNAME, m_aVendorName );
	if( m_aVendorVersion.Len() )
		aStream.WriteProperty( PROPERTY_VENDORVERSION, m_aVendorVersion );

    if ( m_aCompanyName.Len() )
		aStream.WriteProperty( PROPERTY_COMPANYNAME, m_aCompanyName );

	aStream.WriteProperty(PROPERTY_DEFAULTDESTPATH, m_aDefaultDestPath);
	if( m_aDestPath.Len() )
		aStream.WriteProperty(PROPERTY_DESTPATH, m_aDestPath);

	// Fix zu #56789#; man achte auf die ID (5er Pasch)
	if( m_aSourcePath.Len() )
	{
		ByteString aSourcePath(m_aSourcePath);
#if defined(WNT) || defined(OS2)
		SiDirEntry aEntry( aSourcePath );
		if( aEntry.Level() == 1 && aSourcePath.GetChar(aSourcePath.Len()-1) == '\\' )
			aSourcePath.Erase(aSourcePath.Len()-1);
#endif
		aStream.WriteProperty(PROPERTY_SOURCEPATH, aSourcePath);
	}

	switch( m_eInstallMode )
	{
		case IM_STANDALONE:
			aStream.WriteProperty(PROPERTY_MODE, SiIdentifier(VALUE_STANDALONE));
			break;
		case IM_NETWORK:
			aStream.WriteProperty(PROPERTY_MODE, SiIdentifier(VALUE_NETWORK));
			break;
		case IM_WORKSTATION:
			aStream.WriteProperty(PROPERTY_MODE, SiIdentifier(VALUE_WORKSTATION));
			break;
		case IM_PATCH:
			aStream.WriteProperty(PROPERTY_MODE, SiIdentifier(VALUE_PATCH));
			break;
		case IM_APPSERVER:
			aStream.WriteProperty(PROPERTY_MODE, SiIdentifier(VALUE_APPSERVER));
			break;
	}

	if( m_nPatchLevel )
		aStream.WriteProperty(PROPERTY_PATCH_LEVEL, m_nPatchLevel);

	if( !bIsSecondLevel )
		aStream.WriteProperty(PROPERTY_INSTALLFROMNET,	m_bInstallFromNet);

	aStream.WriteProperty( PROPERTY_SCRIPTVERSION, m_nScriptVersion );

	if( m_anUpdateFor.Len() )
		aStream.WriteProperty( PROPERTY_UPDATEFOR, m_anUpdateFor );

	if( m_anUpgradeFor.Len() )
		aStream.WriteProperty( PROPERTY_UPGRADEFOR, m_anUpgradeFor );

	if ( m_aTemplateFilename.Len() )
		aStream.WriteProperty( PROPERTY_TEMPLATEFILE, m_aTemplateFilename );

	if ( m_aFollowApp.Len() )
		aStream.WriteProperty( PROPERTY_FOLLOWAPP, m_aFollowApp );

	if ( m_aProductBMP.Len() )
		aStream.WriteProperty( PROPERTY_BITMAP, m_aProductBMP );

	if ( m_aVendorBMP.Len() )
		aStream.WriteProperty( PROPERTY_VENDORBITMAP, m_aVendorBMP );

	if (m_bForceOverwrite)
		aStream.WriteProperty( PROPERTY_FORCEOVERWRITE,	m_bForceOverwrite );

	if( m_bOUpdateDone )
		aStream.WriteProperty( PROPERTY_OUPDATEDONE, 1L );

	if( m_aUiProcName.Len() )
		aStream.WriteProperty( PROPERTY_UIPROC, m_aUiProcName );

	if( m_aDefLanguage.Len() )
		aStream.WriteProperty( PROPERTY_DEFLANGUAGE, m_aDefLanguage );

	if( m_aLanguages.Len() )
		aStream.WriteProperty( PROPERTY_LANGUAGES, m_aLanguages );

	if( m_aInstLanguages.Len() )
		aStream.WriteProperty( PROPERTY_INST_LANGUAGES, m_aInstLanguages );

	if( m_bRequiresShare || m_bModifyPath || m_bKeepOldVersion || m_bOldVersionRequired ||
		m_bRegistrationRequired || m_bOnlyCustom || m_bReboot || m_bNeedCfgServer || m_bPatched )
	{
		aStream.BeginProperty(PROPERTY_FLAGS);
		aStream.BeginList();

		if (m_bHideStandard)			aStream.AddListValue(SiIdentifier(VALUE_HIDESTANDARD));
		if (m_bRequiresShare)			aStream.AddListValue(SiIdentifier(VALUE_WIN_REQUIRES_SHARE));
		if (m_bModifyPath)	 			aStream.AddListValue(SiIdentifier(VALUE_OS2_MODIFY_PATH));
		if (m_bKeepOldVersion)			aStream.AddListValue(SiIdentifier(VALUE_KEEP_OLD_VERSION));
		if (m_bOldVersionRequired)		aStream.AddListValue(SiIdentifier(VALUE_OLD_VERSION_REQUIRED));
		if (m_bRegistrationRequired)	aStream.AddListValue(SiIdentifier(VALUE_REGISTRATION_REQUIRED));
		if (m_bOnlyCustom) 				aStream.AddListValue(SiIdentifier(VALUE_ONLYCUSTOM));
		if (m_bNeedCfgServer)			aStream.AddListValue(SiIdentifier(VALUE_NEEDCONFIG_SERVER));
		if (m_bPatched)					aStream.AddListValue(SiIdentifier(VALUE_PATCHED));

		aStream.EndList();
		aStream.EndProperty();
	}

	if( m_aSpecialVersion.Len() )
		aStream.WriteProperty(PROPERTY_SPECIAL_VERSION,	m_aSpecialVersion);

	if( m_aDeleteKey.Len() )
		aStream.WriteProperty(PROPERTY_DELETE_KEY, m_aDeleteKey);

	aStream.EndDeclaration();
	return TRUE;
}

ByteString SiInstallation::GetNaturalID() const
{
	ByteString aNatID("SiInstallation");
	if( IsLangRef() ) {
		aNatID += "__";
		aNatID += ByteString::CreateFromInt32( m_nLanguage );
	}
	return aNatID;
}

void SiInstallation::SetInstalledLanguages(SiEnvironment* pEnv)
{
	m_aInstLanguages = "";
	SiLangCtxList& rLst = pEnv->GetLanguageContext();
	for( USHORT i = 0; i < rLst.Count(); ++i )
	{
		LanguageContext* pAct = rLst.GetObject(i);
		if( pAct->isProg || pAct->isDoc )
		{
			m_aInstLanguages += pAct->nLanguage == LANG_DEFAULT?
								m_aDefLanguage : ByteString::CreateFromInt32(pAct->nLanguage);
			m_aInstLanguages += ":";
			m_aInstLanguages += pAct->isProg? "1" : "0";
			m_aInstLanguages += ":";
			m_aInstLanguages += pAct->isDoc? "1" : "0";

			if( i != rLst.Count()-1 )
				m_aInstLanguages += ",";
		}
	}
}

BOOL SiInstallation::IsLanguageInstalled(USHORT nLang, BOOL& rProg, BOOL& rDoc)
{
	if( !m_aInstLanguages.Len() )
		return FALSE;

	USHORT nDelimTok = 0;
	USHORT nTokCount = m_aInstLanguages.GetTokenCount(',');
	for( USHORT x = 0; x < nTokCount; ++x )
	{
		ByteString a( m_aInstLanguages.GetToken(0, ',', nDelimTok) );
		USHORT nD = 0;
		USHORT nT = a.GetTokenCount(':');

		USHORT nLanguage = (USHORT) a.GetToken( 0, ':', nD ).ToInt32();
		if( nLanguage == nLang )
		{
			rProg	= a.GetToken(0, ':', nD) == "1"? TRUE : FALSE;
			rDoc	= a.GetToken(0, ':', nD) == "1"? TRUE : FALSE;
			return TRUE;
		}
	}
	return FALSE;
}

