/*************************************************************************
 *
 *  $RCSfile: thread.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: jl $ $Date: 2001/03/14 09:48:11 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef __THREAD_HXX
#include "thread.hxx"
#endif

#ifndef __SEMAPHORE_HXX
#include "semaphor.hxx"
#endif

#ifndef __SCHEDULE_HXX
#include "schedule.hxx"
#endif

#ifndef __SOCKET_HXX
#include "socket.hxx"
#endif

#ifndef __SWITCH_H
#include "switch.h"
#endif

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>

#include <winsock.h>

#ifndef _OSL_TIME_H_
#include <osl/time.h>
#endif


void SALThread::registerCallback(TaskType eType, HTASK hTask)
{
	meTaskType = eType;
	mhTask = hTask;
	SCHEDULER->readyQueue()->remove(this);
	SCHEDULER->callbackQueue()->append(this);
}

void SALThread::unregisterCallback()
{
	meTaskType = NoTaskType;
	mhTask = NULL;
	SCHEDULER->callbackQueue()->remove(this);
	SCHEDULER->readyQueue()->append(this);
}

//----------------------------------------------------------------------------
//	SALThreadQueue::~SALThreadQueue
//----------------------------------------------------------------------------

SALThreadQueue::~SALThreadQueue()
{
    while (firstItem())
    {
        SALItem *pItem = removeItem(firstItem());
        if (SCHEDULER && SCHEDULER->readyQueue())
            SCHEDULER->readyQueue()->appendItem(pItem);
    }
}

//----------------------------------------------------------------------------
//	SALThreadQueue::removeItem
//----------------------------------------------------------------------------

SALItem *SALThreadQueue::removeItem(SALItem *pItem)
{
    pItem = SALReadyQueue::removeItem(pItem);
    return pItem;
}


#define FP_OFF(p)   ((unsigned short)(((unsigned long)(void __far *)(p)) & 0x0000FFFFuL))
#define FP_SEG(p)   ((unsigned short)((((unsigned long)(void __far *)(p)) & 0xFFFF0000uL) >> 16))
#define MK_FP(s,o)	((void __far *)((((unsigned long)(((unsigned short)(s)) & 0xFFFF)) << 16) | ((unsigned long)(((unsigned short)(o)) & 0xFFFF))))

#ifdef __WATCOMC__
extern "C" unsigned short _nullarea[8];
#endif


//----------------------------------------------------------------------------
//	SALThread::SALThread
//----------------------------------------------------------------------------

SALThread::SALThread(SALThreadEntryFunc func, void *pParam, long nPriority,
                     long nLevel,unsigned short nStackSize) :
    SALItem(),
	mhStackBase(NULL),
    mpParam(pParam),
    mnPriorityLevel(nLevel),
    mnPriorityBase(nPriority),
    mnPriorityCounter(nPriority),
    mnDelay(0),
    mpSemaphore(new SALSemaphore(0)),
    mpMutexQueue(new SALQueue),
    mnWSAError(WSAENOERROR),
    meTaskType(NoTaskType),
	mhTask(NULL),
    mbTerminate(FALSE),
	mbValid(TRUE)
{
    // BR: the following code initialize the holder-array (see below)
#define __br__initialSize 8  // must not be lesser than 2
#define __br__growSize 8     // must not be lesser than 1
    nKeyCount = __br__initialSize; // initialize ThreadKeyHolder-Array....
    pTlsDataKey = (tThreadKeyHolder *) malloc(__br__initialSize * sizeof(tThreadKeyHolder));
    for (int i = __br__initialSize - 2; i >= 0; i--) pTlsDataKey[i].key = i + 1;
    pTlsDataKey[__br__initialSize - 1].key = 0;
    // the single linked list now looks like: [0].key=1, [1].key=2 ... [7].key=0

    mbValid = mbValid && mpSemaphore && mpMutexQueue;

	void *pStackBase;

  	pStackBase = allocStack(nStackSize);

    if (pStackBase)
    {
        void *pFramePtr = create_context(pStackBase, func, pParam, nStackSize);
        maStackFrame.sptr = FP_OFF(pFramePtr);
        maStackFrame.sseg = FP_SEG(pFramePtr);
    }

	mbValid = mbValid && mhStackBase;
}

//----------------------------------------------------------------------------
//	SALThread::SALThread
//----------------------------------------------------------------------------

SALThread::SALThread() :
    SALItem(),
	mhStackBase(NULL),
    mpParam(NULL),
    mnPriorityLevel(TPL_NORMAL),
    mnPriorityBase(TPD_NORMAL),
    mnPriorityCounter(TPD_NORMAL),
    mnDelay(0),
    mpSemaphore(new SALSemaphore(0)),
    mpMutexQueue(new SALQueue),
    mnWSAError(WSAENOERROR),
    meTaskType(NoTaskType),
	mhTask(NULL),
	mbTerminate(FALSE),
	mbValid(TRUE)
{
	mbValid = mbValid && mpSemaphore && mpMutexQueue;
}

//----------------------------------------------------------------------------
//	SALThread::~SALThread
//----------------------------------------------------------------------------

SALThread::~SALThread()
{
	kill();
    freeStack();
    free(pTlsDataKey/*, nKeyCount * sizeof(tThreadKeyHolder)*/);
}

//----------------------------------------------------------------------------
//	SALThread::allocStack
//----------------------------------------------------------------------------

void *SALThread::allocStack(size_t size)
{
	mhStackBase = GlobalAlloc(GPTR, size);

	if (mhStackBase)
		return GlobalLock(mhStackBase);
	else
		return NULL;
}

//----------------------------------------------------------------------------
//	SALThread::freeStack
//----------------------------------------------------------------------------

void SALThread::freeStack()
{
	if (mhStackBase)
	{
		GlobalUnlock(mhStackBase);
		GlobalFree(mhStackBase);
		mhStackBase = NULL;
	}
}

//----------------------------------------------------------------------------
//	SALThread::join
//----------------------------------------------------------------------------

void SALThread::join()
{
    if (mpSemaphore)
        mpSemaphore->acquire();
}

//----------------------------------------------------------------------------
//	SALThread::schedule
//----------------------------------------------------------------------------

int SALThread::schedule()
{
	if (!mbTerminate && SCHEDULER->activeThread() == this)
		SCHEDULER->dispatch();
	return !mbTerminate;
}

//----------------------------------------------------------------------------
//	SALThread::terminate
//----------------------------------------------------------------------------

void SALThread::terminate()
{
	if (!mbTerminate)
		mbTerminate = TRUE;
}

//----------------------------------------------------------------------------
//	SALThread::kill
//----------------------------------------------------------------------------

void SALThread::kill(void *pExitParam)
{
    if (mpSemaphore)  // Der Thread laeuft noch
    {
        mpParam = pExitParam;
        while (mpMutexQueue->firstItem())
        {
            SALMutex *pMutex = (SALMutex *)mpMutexQueue->firstItem();

            mpMutexQueue->removeItem(pMutex);
            pMutex->setOwner(pMutex->first());
        }
        delete mpSemaphore;
        mpSemaphore = NULL;
        
        if (queue())
            queue()->remove(this);

		switch (meTaskType)
		{
			case WSATaskType:
				WSACancelAsyncRequest(mhTask);
				break;
			default:
				break;
		}

		unregisterCallback();
                
        SCHEDULER->readyQueue()->remove(this);
        if (SCHEDULER->activeThread() == this && SCHEDULER->mainThread() != this)
		{
			SCHEDULER->terminatedQueue()->append(this);
			SCHEDULER->dispatch();
		}
		else
			freeStack();
    }
}

//----------------------------------------------------------------------------
//	SALThread::getExitParam
//----------------------------------------------------------------------------

void *SALThread::getExitParam() const
{
    if (mpSemaphore == NULL)
        return mpParam;
    else
        return NULL;
}

//----------------------------------------------------------------------------
//	SALThread::resume
//----------------------------------------------------------------------------

void SALThread::resume()
{
    if (isRunning())
        SCHEDULER->readyQueue()->append(this);
}

//----------------------------------------------------------------------------
//	SALThread::suspend
//----------------------------------------------------------------------------

void SALThread::suspend()
{
    SCHEDULER->readyQueue()->remove(this);
    if (SCHEDULER->activeThread() == this && SCHEDULER->mainThread() != this)
        SCHEDULER->dispatch();
}

//----------------------------------------------------------------------------
//	SALThread::delay
//----------------------------------------------------------------------------

void SALThread::delay(long milliseconds)
{
    mnDelay = milliseconds;
    
    SCHEDULER->readyQueue()->remove(this);
    SCHEDULER->delayQueue()->append(this);
    if (SCHEDULER->activeThread() == this)
        SCHEDULER->dispatch();
}

//----------------------------------------------------------------------------
//	SALThread::wakeup
//----------------------------------------------------------------------------

void SALThread::wakeup()
{
    mnDelay = 0;
    SCHEDULER->dispatch();
}

//============================================================================
//
//	OSL-Interface
//
//============================================================================

#include <osl/thread.h>

extern "C" {

struct ThreadWrapper
{
    oslWorkerFunction	mpWorker;
    void            	*mpParam;
};

static void * __cdecl WorkerFunctionWrapper(void *param)
{
    ThreadWrapper 		*pWrapper = (ThreadWrapper *)param;
	oslWorkerFunction	pWorker = pWrapper->mpWorker;
	void				*pThreadParam = pWrapper->mpParam;

	delete pWrapper;
    pWorker(pThreadParam);
    return NULL;    
}


/** Create the thread, using the function-ptr pWorker as
	its main (worker) function. This functions receives in
	its void* parameter the value supplied by pThreadData.
	The thread will be created, but it won't start running.
	To wake-up the thread, use resume().
	@return 0 if creation failed, otherwise a handle to the thread
*/
oslThread SAL_CALL osl_createSuspendedThread(oslWorkerFunction pWorker, void *pParam)
{
    ThreadWrapper	*pWrapper = new ThreadWrapper;
	SALThread		*pThread = NULL;

    if (pWrapper)
    {
        pWrapper->mpWorker = pWorker;
        pWrapper->mpParam = pParam;
        pThread = new SALThread(WorkerFunctionWrapper, pWrapper);
		
		if (pThread && !pThread->isValid())
		{
			delete pThread;
			pThread = NULL;
		}

		if (!pThread)
			delete pWrapper;
    }
		
    return (oslThread)pThread;
}


/** Create the thread, using the function-ptr pWorker as
	its main (worker) function. This functions receives in
	its void* parameter the value supplied by pThreadData.
	Once the OS-structures are initialized,the thread starts
	running.
	@return 0 if creation failed, otherwise a handle to the thread
*/
oslThread SAL_CALL osl_createThread(oslWorkerFunction pWorker, void *pParam)
{
    oslThread hThread = osl_createSuspendedThread(pWorker, pParam);

    if (hThread)
        osl_resumeThread(hThread);

    return hThread;
}


/** Get the identifier for the specified thread or if parameter
    Thread is NULL of the current active thread.
	@return identifier of the thread
*/
oslThreadIdentifier SAL_CALL osl_getThreadIdentifier(oslThread Thread)
{
	return (oslThreadIdentifier)Thread;
}

/** Forcefully abort the thread, if it is still running.
	Then release the OS-structures and our thread data-structure.
	If Thread is NULL, the function won't do anything.
*/
void SAL_CALL osl_destroyThread(oslThread hThread)
{
	osl_freeThreadHandle(hThread);
}


/** Release our thread data-structure.
	If Thread is NULL, the function won't do anything.
	Note that we do not interfere with the actual running of
	the thread, we just free up the memory needed by the handle.
*/
void SAL_CALL osl_freeThreadHandle(oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

	if (pThread)
		delete pThread;
}


/** Wake-up a thread that was suspended with suspend() or
	createSuspended(). The oslThread must be valid!
*/
void SAL_CALL osl_resumeThread(oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread)
        pThread->resume();
}


/** Suspend the execution of the thread. If you want the thread
	to continue, call resume(). The oslThread must be valid!
*/
void SAL_CALL osl_suspendThread(oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread)
        pThread->suspend();
}



/** Changes the threads priority.
	The oslThread must be valid!
*/
void SAL_CALL osl_setThreadPriority(oslThread hThread, oslThreadPriority ePriority)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread)
    {
        long    nPriority;
        
        switch (ePriority)
        {
            case osl_Thread_PriorityHighest:
                nPriority = TPD_HIGHEST;
                break;
            case osl_Thread_PriorityAboveNormal:
                nPriority = TPD_HIGH;
                break;
            case osl_Thread_PriorityNormal:
                nPriority = TPD_NORMAL;
                break;
            case osl_Thread_PriorityBelowNormal:
                nPriority = TPD_LOW;
                break;
            case osl_Thread_PriorityLowest:
                nPriority = TPD_LOWEST;
                break;
            default:
                return;
        }

        pThread->setPriority(nPriority);
    }
}



/** Retrieves the threads priority.
	Returns oslThreadPriorityUnknown for invalid Thread-argument or
	terminated thread. (I.e.: The oslThread might be invalid.)
*/
oslThreadPriority SAL_CALL osl_getThreadPriority(const oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;
    oslThreadPriority   ePriority = osl_Thread_PriorityUnknown;

    if (pThread)
    {
        switch (pThread->getPriority())
        {
            case TPD_HIGHEST:
                ePriority = osl_Thread_PriorityHighest;
                break;
            case TPD_HIGH:
                ePriority = osl_Thread_PriorityAboveNormal;
                break;
            case TPD_NORMAL:
                ePriority = osl_Thread_PriorityNormal;
                break;
            case TPD_LOW:
                ePriority = osl_Thread_PriorityBelowNormal;
                break;
            case TPD_LOWEST:
                ePriority = osl_Thread_PriorityLowest;
                break;
        }
    }

    return ePriority;
}



/** Returns True if the thread was created and has not terminated yet.
	Note that according to this definition a "running" thread might be
	suspended! Also returns False is Thread is NULL.
*/
sal_Bool SAL_CALL osl_isThreadRunning(const oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread)
        return (sal_Bool)pThread->isRunning();
    else
        return sal_False;
}


/** Returns True is both Handles represent the same
	thread-instance. Returns also False if at least one of
	the handles is invalid.
*/
sal_Bool osl_isEqualThread(const oslThread hThread1, const oslThread hThread2)
{
    SALThread *pThread1 = (SALThread *)hThread1;
    SALThread *pThread2 = (SALThread *)hThread2;
    
    return (sal_Bool)(pThread1 && pThread2 && pThread1 == pThread2);
}



/** Blocks the calling thread until Thread has terminated.
	Returns immediately if Thread is NULL.
*/
void SAL_CALL osl_joinWithThread(oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread)
        pThread->join();
}


/** Blocks the calling thread at least for the given number of
	of time. Returns False if the sleep is aborted by a	call
	to awakeThread.
*/
oslThreadSleep SAL_CALL osl_sleepThread(oslThread hThread, const TimeValue *pTimeVal)
{
    unsigned long milliseconds = MILLISECONDS(pTimeVal);

    SALThread *pThread = (SALThread *)hThread;

    if (pThread)
    {
        pThread->delay(milliseconds);
        return osl_Thread_SleepNormal;
    }
    else
        return osl_Thread_SleepError;
}


/** Awake a sleeping thread. Returns False if at least one of
	the handles is invalid or the thread is not sleeping.
*/
sal_Bool SAL_CALL osl_awakeThread(oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread && pThread->queue() == SCHEDULER->delayQueue())
    {
        pThread->wakeup();
        return sal_True;
    }
    else
        return sal_False;
}



/** Blocks the calling thread at least for the given number
    of time.
*/
void SAL_CALL osl_waitThread(const TimeValue *pTimeVal)
{
    unsigned long milliseconds = MILLISECONDS(pTimeVal);

    if (SCHEDULER->activeThread())
        SCHEDULER->activeThread()->delay(milliseconds);
}

/** The requested thread will get terminate the next time
	scheduleThread() is called.
*/
void SAL_CALL osl_terminateThread(oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread)
        pThread->terminate();
}


/** Offers the rest of the threads time-slice to the OS.
	scheduleThread() should be called in the working loop
	of the thread, so any other thread could also get the
	processor. Returns False if the thread should terminate, so
	the thread could free any allocated resources.
*/
sal_Bool SAL_CALL osl_scheduleThread(oslThread hThread)
{
    SALThread *pThread = (SALThread *)hThread;

    if (pThread && pThread->isRunning() && !pThread->isTerminating())
    {
        pThread->schedule();
        return sal_True;        
    }
    else
        return sal_False;
}

/** Offers the rest of the threads time-slice to the OS.
	Under POSIX you _need_ to yield(), otherwise, since the
	threads are not preempted during execution, NO other thread
	(even with higher priority) gets the processor. Control is
	only given to another thread if the current thread blocks
	or uses yield().
*/
void SAL_CALL osl_yieldThread()
{
    SCHEDULER->dispatch();
}

/** BR: The ThreadKeys reside in an array of tThreadKeyHolder. The available
 entries are linked in a simple list. The listhead is at index 0. This
 Entry will never be used for holding data. If the head points to itself
 the array will grow in size. Therefore the result of the function will
 always be greater than zero.
 */

oslThreadKey SAL_CALL osl_createThreadKey(void)
{
    SALScheduler *pScheduler = SALScheduler::SchedulerInstance();
    if (!pScheduler) return 0;
    SALThread *pCurrentThread = pScheduler->activeThread();
    if (!pCurrentThread) return 0;

    tThreadKeyHolder *pData = pCurrentThread->pTlsDataKey;
    int threadKeyCount = pCurrentThread->nKeyCount;
    if (!pData[0].key)
    { // the listhead points to itself --> grow array
        int i, newCount = threadKeyCount + __br__growSize; // the arrry gets more elements
        pData = (tThreadKeyHolder *) realloc(pData, newCount * sizeof(tThreadKeyHolder));
        pCurrentThread->pTlsDataKey = pData;
        for (i = newCount - 2; i >= threadKeyCount; i--) // link together
            pData[i].key = i + 1;                        // the new holder
        pData[newCount - 1].key = pData[0].key; // = 0, because the listhead points to itself
        pData[0].key = threadKeyCount;          // let the listhead points to the first of the new entries
        pCurrentThread->nKeyCount = newCount;   // remember the new size
    }
    oslThreadKey index = pData[0].key; // get the first entry of the list
    pData[0].key = pData[index].key;   // remove this entry from the list
    return index;
}

void SAL_CALL osl_destroyThreadKey(oslThreadKey Key)
{
    SALScheduler *pScheduler = SALScheduler::SchedulerInstance();
    if (!pScheduler) return;
    SALThread *pCurrentThread = pScheduler->activeThread();
    if (!pCurrentThread) return;

    tThreadKeyHolder *pData = pCurrentThread->pTlsDataKey;
    pData[Key].key = pData[0].key; // link the free holder
    pData[0].key = Key;            // in the list
}

void* SAL_CALL osl_getThreadKeyData(oslThreadKey Key)
{
    SALScheduler *pScheduler = SALScheduler::SchedulerInstance();
    if (!pScheduler) return 0;
    SALThread *pCurrentThread = pScheduler->activeThread();
    if (!pCurrentThread) return 0;

    return pCurrentThread->pTlsDataKey[Key].data;
}

sal_Bool SAL_CALL osl_setThreadKeyData(oslThreadKey Key, void *pData)
{
    SALScheduler *pScheduler = SALScheduler::SchedulerInstance();
    if (!pScheduler) return sal_False;
    SALThread *pCurrentThread = pScheduler->activeThread();
    if (!pCurrentThread) return sal_False;

    pCurrentThread->pTlsDataKey[Key].data = pData;
    return sal_True;
}

} // extern "C"

	

