/*************************************************************************
 *
 *  $RCSfile: sendclnt.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mhu $ $Date: 2001/03/19 11:06:59 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INET_SENDCLNT_CXX "$Revision: 1.2 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif
#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif

#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif

#ifndef _INETCORESTRM_HXX
#include <inetstrm.hxx>
#endif
#ifndef _INET_SENDCLNT_HXX
#include <sendclnt.hxx>
#endif

using namespace rtl;

using namespace inet;
using namespace inet::mail;
using namespace inet::smtp;

/*========================================================================
 *
 * SendClient_Impl internals.
 *
 *======================================================================*/
namespace inet
{
namespace smtp
{

/*========================================================================
 *
 * SendContext_Impl.
 *
 *======================================================================*/
class SendContext_Impl : public NAMESPACE_VOS(OReference)
{
public:
	/** Representation.
	 */
	typedef INetCoreMailerCallback callback_type;

	callback_type *m_pfnCB;
	void          *m_pDataCB;

	/** Construction.
	 */
	SendContext_Impl (callback_type *pfnCB, void *pDataCB)
		: m_pfnCB (pfnCB), m_pDataCB (pDataCB)
	{}

protected:
	/** Destruction.
	 */
	virtual ~SendContext_Impl (void);

private:
	/** Not implemented.
	 */
	SendContext_Impl (const SendContext_Impl&);
	SendContext_Impl& operator= (const SendContext_Impl&);
};

/*
 * ~SendContext_Impl.
 */
SendContext_Impl::~SendContext_Impl (void)
{
}

/*========================================================================
 *
 * MessageContext_Impl.
 *
 *======================================================================*/
class MessageContext_Impl : public SendContext_Impl
{
	/** Representation.
	 */
	INetCoreMIMEMessageStream m_aEncoder;

	UniString                 m_aQuotes;
	UniString                 m_aRcptList;
	sal_uInt32                m_nRcptCount;
	sal_uInt32                m_nRcptIndex;

public:
	/** Representation.
	 */
	sal_Bool                  m_bRcptList;

	/** Construction.
	 */
	MessageContext_Impl (INetCoreMailerCallback *pfnCB, void *pDataCB)
		: SendContext_Impl (pfnCB, pDataCB),
		  m_aQuotes    ("\"\"()", RTL_TEXTENCODING_ASCII_US),
		  m_bRcptList  (sal_False),
		  m_nRcptCount (0),
		  m_nRcptIndex (0)
	{}

	/** Initialization.
	 */
	sal_Bool initialize (
		SendClient_Impl::ClientManager &rManager,
		INetCoreMIMEMessage            &rMessage)
	{
		// Check Recipient.
		if (!rMessage.GetTo().Len())
			return sal_False;

		// Prepare message.
		if (!rManager.prepareMessage (rMessage))
			return sal_False;

		// Prepare recipient list.
		m_aRcptList = rMessage.GetTo();
		if (rMessage.GetCC().Len())
		{
			m_aRcptList += ',';
			m_aRcptList += rMessage.GetCC();
		}
		if (rMessage.GetBCC().Len())
		{
			m_aRcptList += ',';
			m_aRcptList += rMessage.GetBCC();
			rMessage.SetBCC (String());
		}

		m_nRcptCount = m_aRcptList.GetQuotedTokenCount (m_aQuotes, ',');
		m_nRcptIndex = 0;

		// Check document stream.
		SvStream *pDocStrm = rMessage.GetDocumentStream();
		if (pDocStrm)
		{
			// Rewind document stream.
			pDocStrm->Seek (STREAM_SEEK_TO_BEGIN);
		}

		// Done.
		m_aEncoder.SetSourceMessage (&rMessage);
		return sal_True;
	}

	/** nextRecipient.
	 */
	UniString nextRecipient (void)
	{
		if (m_nRcptIndex < m_nRcptCount)
			return m_aRcptList.GetQuotedToken (m_nRcptIndex++, m_aQuotes, ',');
		else
			return UniString();
	}

	/** Operation.
	 */
	INetCoreMessageIStream& operator() (void) { return m_aEncoder; }
};

} // smtp
} // inet

/*========================================================================
 *
 * SendClient_Impl implementation.
 *
 *======================================================================*/
/*
 * SendClient_Impl.
 */
SendClient_Impl::SendClient_Impl (ClientManager &rManager)
	: m_rManager (rManager),
	  m_eState   (STATE_NONE)
{
}

/*
 * ~SendClient_Impl.
 */
SendClient_Impl::~SendClient_Impl (void)
{
}

/*
 * getMutex_Impl.
 */
inline NAMESPACE_VOS(IMutex)& SendClient_Impl::getMutex_Impl (void) const
{
	return ((SAL_CONST_CAST(SendClient_Impl*, this))->m_aMutex);
}

/*
 * setState_Impl.
 */
inline void SendClient_Impl::setState_Impl (State eState)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	m_eState = eState;
}

/*
 * changeState_Impl.
 */
inline sal_Bool SendClient_Impl::changeState_Impl (State ePrev, State eNext)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	if (m_eState == ePrev)
	{
		m_eState = eNext;
		return sal_True;
	}
	return sal_False;
}

/*
 * createConnection_Impl.
 */
inline sal_Bool SendClient_Impl::createConnection_Impl (
	NAMESPACE_VOS(ORef)<connection_type> &rxConnection)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	if (!m_xConnection.isValid())
	{
		m_xConnection = new connection_type();
		m_xConnection->SetTransferCallback (onConnectionEvent, this);
		m_xConnection->SetTerminateCallback (onConnectionEvent, this);
	}
	rxConnection = m_xConnection;
	return (rxConnection.isValid());
}

/*
 * getConnection_Impl.
 */
inline sal_Bool SendClient_Impl::getConnection_Impl (
	NAMESPACE_VOS(ORef)<connection_type> &rxConnection) const
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	rxConnection = m_xConnection;
	return (rxConnection.isValid());
}

/*
 * onConnectionEvent.
 */
sal_Bool SendClient_Impl::onConnectionEvent (
	connection_type *pConnection,
	sal_Int32 nReplyCode, const sal_Char *pReplyText, void *pData)
{
	SendClient_Impl *pThis = SAL_REINTERPRET_CAST(SendClient_Impl*, pData);
	if (pThis)
	{
		NAMESPACE_VOS(ORef)<SendClient_Impl> xThis (pThis);
		xThis->handleConnectionEvent (pConnection, nReplyCode, pReplyText);
	}
	return sal_True;
}

/*
 * handleConnectionEvent.
 */
void SendClient_Impl::handleConnectionEvent (
	connection_type *pConnection,
	sal_Int32 nReplyCode, const sal_Char *pReplyText)
{
	if (changeState_Impl (STATE_DATA, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<SendContext_Impl> xContext (m_xContext);
		if (xContext.isValid())
		{
			if (nReplyCode == INETCORESMTP_REPLY_TRANSFER_WAIT)
			{
				// Wait for next event.
				changeState_Impl (STATE_EVENT, STATE_DATA);
			}
			else
			{
				// Finished.
				setState_Impl (STATE_NONE);
			}

			// Notify caller.
			if (xContext->m_pfnCB) (xContext->m_pfnCB) (
				m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
		}
	}
	else if (changeState_Impl (STATE_TO, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<MessageContext_Impl> xContext (
			SAL_REINTERPRET_CAST (MessageContext_Impl*, &*m_xContext));
		if (xContext.isValid())
		{
			if ((nReplyCode == INETCORESMTP_REPLY_ACTION_OK ) ||
				(nReplyCode == INETCORESMTP_REPLY_FORWARDING)    )
			{
				// RCPT TO command ok.
				xContext->m_bRcptList |= sal_True;
			}
			else
			{
				// RCPT TO command failure.
				xContext->m_bRcptList |= sal_False;
			}

			// Recipient notification #68758#
			if (xContext->m_pfnCB) (xContext->m_pfnCB) (
				m_rManager, -(VOS_ABS(nReplyCode)),
				pReplyText, xContext->m_pDataCB);
		}

		xContext = SAL_REINTERPRET_CAST (MessageContext_Impl*, &*m_xContext);
		if (xContext.isValid())
		{
			UniString aRecipient (xContext->nextRecipient());
			if (aRecipient.Len())
			{
				// Send RCPT TO command.
				if (!mailTo_Impl (aRecipient))
				{
					nReplyCode = INETCORESMTP_REPLY_ACTION_ABORTED;
					setState_Impl (STATE_NONE);

					if (xContext->m_pfnCB) (xContext->m_pfnCB) (
						m_rManager, nReplyCode, NULL, xContext->m_pDataCB);
				}
			}
			else if (xContext->m_bRcptList)
			{
				// Send DATA command.
				if (!transferData_Impl ((*xContext)()))
				{
					nReplyCode = INETCORESMTP_REPLY_ACTION_ABORTED;
					setState_Impl (STATE_NONE);

					if (xContext->m_pfnCB) (xContext->m_pfnCB) (
						m_rManager, nReplyCode, NULL, xContext->m_pDataCB);
				}
			}
			else
			{
				// No recipient -> No DATA command.
				nReplyCode = INETCORESMTP_REPLY_ACTION_ABORTED;
				setState_Impl (STATE_NONE);

				if (xContext->m_pfnCB) (xContext->m_pfnCB) (
					m_rManager, nReplyCode, NULL, xContext->m_pDataCB);
			}
		}
	}
	else if (changeState_Impl (STATE_FROM, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<MessageContext_Impl> xContext (
			SAL_REINTERPRET_CAST (MessageContext_Impl*, &*m_xContext));
		if (xContext.isValid())
		{
			if (nReplyCode == INETCORESMTP_REPLY_ACTION_OK)
			{
				// MAIL FROM command ok, send RCPT TO command.
				UniString aRecipient (xContext->nextRecipient());
				if (!mailTo_Impl (aRecipient))
				{
					nReplyCode = INETCORESMTP_REPLY_ACTION_ABORTED;
					setState_Impl (STATE_NONE);

					if (xContext->m_pfnCB) (xContext->m_pfnCB) (
						m_rManager, nReplyCode, NULL, xContext->m_pDataCB);
				}
			}
			else
			{
				// MAIL FROM command failure.
				setState_Impl (STATE_NONE);

				// Notify caller.
				if (xContext->m_pfnCB) (xContext->m_pfnCB) (
					m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
			}
		}
	}
	else if (changeState_Impl (STATE_OPEN, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<SendContext_Impl> xContext (m_xContext);
		if (xContext.isValid())
		{
			if ((nReplyCode == INETCORESMTP_REPLY_RESOLVER_WAIT) ||
				(nReplyCode == INETCORESMTP_REPLY_CONNECT_WAIT )    )
			{
				// Wait for next event.
				changeState_Impl (STATE_EVENT, STATE_OPEN);
			}
			else
			{
				// Finished.
				setState_Impl (STATE_NONE);

				// Notify caller.
				if (xContext->m_pfnCB) (xContext->m_pfnCB) (
					m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
			}
		}
	}
	else if (changeState_Impl (STATE_CLOSE, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<SendContext_Impl> xContext (m_xContext);
		if (xContext.isValid())
		{
			// Finished.
			setState_Impl (STATE_NONE);

			// Notify caller.
			if (xContext->m_pfnCB) (xContext->m_pfnCB) (
				m_rManager, nReplyCode, pReplyText, xContext->m_pDataCB);
		}
	}
	else
	{
		VOS_ASSERT(m_eState == STATE_NONE);
		if (nReplyCode == INETCORESMTP_REPLY_NETWORK_ERROR)
		{
			// Notify manager.
			m_rManager.terminated (this);
		}
	}
}

/*
 * openConnection.
 */
sal_Bool SendClient_Impl::openConnection (
	const OUString &rHost, sal_uInt16 nPort,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_OPEN))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!createConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext = new SendContext_Impl (pfnCallback, pData);
		if (!xConnection->Open (rHost, nPort, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_OPEN, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * isOpen.
 */
sal_Bool SendClient_Impl::isOpen (void) const
{
	NAMESPACE_VOS(ORef)<connection_type> xConnection;
	if (getConnection_Impl (xConnection))
		return xConnection->IsOpen();
	else
		return sal_False;
}

/*
 * closeConnection.
 */
sal_Bool SendClient_Impl::closeConnection (
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_CLOSE))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext = new SendContext_Impl (pfnCallback, pData);
		if (!xConnection->Close (onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_CLOSE, STATE_NONE))
			{
				// Failure.
				m_xContext.unbind();
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * abortConnection.
 */
void SendClient_Impl::abortConnection (void)
{
	NAMESPACE_VOS(OGuard) aGuard (getMutex_Impl());
	if (m_xConnection.isValid())
	{
		m_xConnection->Destroy();
		m_xConnection.unbind();
	}
	if (m_xContext.isValid())
	{
		m_xContext->m_pfnCB = 0;
		m_xContext.unbind();
	}
	m_eState = STATE_NONE;
}

/*
 * sendMail.
 */
sal_Bool SendClient_Impl::sendMail (
	INetCoreMIMEMessage    &rMessage,
	INetCoreMailerCallback *pfnCallback, void *pData)
{
	if (changeState_Impl (STATE_NONE, STATE_EVENT))
	{
		NAMESPACE_VOS(ORef)<MessageContext_Impl> xContext (
			new MessageContext_Impl (pfnCallback, pData));
		if (!xContext->initialize (m_rManager, rMessage))
		{
			setState_Impl (STATE_NONE);
			return sal_False;
		}

		m_xContext.bind (&*xContext);
		if (!mailFrom_Impl (rMessage.GetFrom()))
		{
			m_xContext.unbind();
			setState_Impl (STATE_NONE);
			return sal_False;
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * getTransferCount.
 */
sal_uInt32 SendClient_Impl::getTransferCount (void) const
{
	NAMESPACE_VOS(ORef)<connection_type> xConnection;
	if (getConnection_Impl (xConnection))
		return xConnection->GetTransferCount();
	else
		return 0;
}

/*
 * mailFrom_Impl.
 */
sal_Bool SendClient_Impl::mailFrom_Impl (const OUString &rFrom)
{
	if (changeState_Impl (STATE_EVENT, STATE_FROM))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_EVENT);
			return sal_False;
		}
		if (!xConnection->MailFrom (rFrom, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_FROM, STATE_EVENT))
			{
				// Failure.
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * mailTo_Impl.
 */
sal_Bool SendClient_Impl::mailTo_Impl (const OUString &rRcpt)
{
	if (changeState_Impl (STATE_EVENT, STATE_TO))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_EVENT);
			return sal_False;
		}
		if (!xConnection->MailTo (rRcpt, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_TO, STATE_EVENT))
			{
				// Failure.
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

/*
 * transferData_Impl.
 */
sal_Bool SendClient_Impl::transferData_Impl (INetCoreMessageIStream &rStrm)
{
	if (changeState_Impl (STATE_EVENT, STATE_DATA))
	{
		NAMESPACE_VOS(ORef)<connection_type> xConnection;
		if (!getConnection_Impl (xConnection))
		{
			setState_Impl (STATE_EVENT);
			return sal_False;
		}
		if (!xConnection->TransferData (rStrm, onConnectionEvent, this))
		{
			if (changeState_Impl (STATE_DATA, STATE_EVENT))
			{
				// Failure.
				return sal_False;
			}
		}
		return sal_True;
	}
	return sal_False;
}

