/*************************************************************************
 *
 *  $RCSfile: imapscan.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:31:16 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef INET_IMAPCHAR_HXX
#include <imapchar.hxx>
#endif
#ifndef INET_IMAPSCAN_HXX
#include <imapscan.hxx>
#endif

//============================================================================
//
//  class INetIMAPScanner
//
//============================================================================

inline bool INetIMAPScanner::isAtomChar(sal_Char cChar) const
{
	if (m_eMode <= MODE_ATOM_QUOTED_LITERAL)
		return INetIMAPASCII::isAtomChar(sal_uChar(cChar));
	else if (m_eMode <= MODE_RESTRICTED_QUOTED_LITERAL)
		return INetMIME::isAlphanumeric(sal_uChar(cChar)) || cChar == '-';
	else
		return cChar != '\x0A' && cChar != '\x0D'; // LF, CR
}

//============================================================================
inline void INetIMAPScanner::addToPrefix(void const * pBuffer,
										 sal_uInt32 & rStart, sal_uInt32 nEnd)
{
	DBG_ASSERT(nEnd - rStart <= STRING_MAXLEN,
			   "INetIMAPScanner::addToPrefix(): Too long");
	m_aPrefix.Append(static_cast< sal_Char const * >(pBuffer) + rStart,
					 xub_StrLen(nEnd - rStart));
	rStart = nEnd;
}

//============================================================================
void INetIMAPScanner::clearBuffers()
{
	while (m_pBuffers)
	{
		Buffer * pNext = m_pBuffers->m_pNext;
		delete[] m_pBuffers->m_pData;
		delete m_pBuffers;
		m_pBuffers = pNext;
	}
}

//============================================================================
// virtual
ErrCode INetIMAPScanner::WriteAt(ULONG nPos, void const * pBuffer,
								 ULONG nCount, ULONG * pWritten)
{
	if (nCount == 0)
	{
		if (pWritten)
			*pWritten = 0;
		return ERRCODE_NONE;
	}

	if (!pBuffer)
		return ERRCODE_IO_INVALIDPARAMETER;

	if (nPos != m_nPosition || m_nPosition > 0xFFFFFFFF - nCount)
		return ERRCODE_IO_CANTWRITE;
	m_nPosition += nCount;
	if (pWritten)
		*pWritten = nCount;

	if (m_bAborted)
		return ERRCODE_NONE;

	if (m_bActive)
	{
		Buffer ** p = &m_pBuffers;
		while (*p)
			p = &(*p)->m_pNext;
		*p = new Buffer;
		(*p)->m_pNext = 0;
		(*p)->m_pData = new sal_Char[nCount];
		rtl_copyMemory((*p)->m_pData, pBuffer, nCount);
		(*p)->m_nSize = nCount;
		return ERRCODE_NONE;
	}

	SvRefBaseRef xRef = this;
	m_bActive = true;

	enum { NO_START = 0xFFFFFFFF };
	sal_uInt32 nStart;
	switch (m_eState)
	{
		case STATE_ATOM:
		case STATE_QUOTE:
		case STATE_QUOTE_ESCAPE:
		case STATE_LITERAL_TEXT:
			nStart = 0;
			break;

		default:
			nStart = NO_START;
			break;
	}

	sal_uInt32 nIndex = 0;
	sal_Char cChar;
	INetIMAPScannerToken * pToken = 0;

	for (bool bRead = true;;)
	{
		if (pToken)
		{
			m_eMode = m_pCallback(*pToken, m_pCallbackData);
			delete pToken;
			pToken = 0;
			if (m_eMode == MODE_ABORTED)
				m_bAborted = true;
			if (m_bAborted)
			{
				clearBuffers();
				m_bActive = false;
				return ERRCODE_NONE;
			}
		}

		if (bRead)
		{
			if (nIndex >= nCount)
			{
				if (nStart != NO_START)
				{
					addToPrefix(pBuffer, nStart, nCount);
					nStart = 0;
				}

				if (m_pBuffers && pBuffer == m_pBuffers->m_pData)
				{
					Buffer * pNext = m_pBuffers->m_pNext;
					delete[] m_pBuffers->m_pData;
					delete m_pBuffers;
					m_pBuffers = pNext;
				}

				if (!m_pBuffers)
				{
					m_bActive = false;
					return ERRCODE_NONE;
				}

				pBuffer = m_pBuffers->m_pData;
				nCount = m_pBuffers->m_nSize;
				nIndex = 0;
			}

			cChar = static_cast< sal_Char const * >(pBuffer)[nIndex++];
		}

		switch (m_eState)
		{
			case STATE_INITIAL:
				DBG_ASSERT(nStart == NO_START,
						   "INetIMAPScanner::WriteAt(): Invariant");

				switch (cChar)
				{
					case '\x0A': // LF
						pToken = new INetIMAPScannerToken
						              (INetIMAPScannerToken::TYPE_NEWLINE);
						bRead = true;
						break;

					case '\x0D': // CR
						m_eState = STATE_CR;
						bRead = true;
						break;

					case ' ':
						if (m_eMode == MODE_TEXT)
						{
							nStart = nIndex - 1;
							m_eState = STATE_ATOM;
						}
						else
							m_eState = STATE_SPACE;
						bRead = true;
						break;

					case '"':
						if (m_eMode == MODE_TEXT)
						{
							nStart = nIndex - 1;
							m_eState = STATE_ATOM;
						}
						else if (m_eMode & MODE_ATOM_QUOTED)
						{
							nStart = nIndex;
							m_eState = STATE_QUOTE;
						}
						else
							pToken
								= new INetIMAPScannerToken(
									      INetIMAPScannerToken::TYPE_SPECIAL,
										  cChar);
						bRead = true;
						break;

					case '{':
						if (m_eMode == MODE_TEXT)
						{
							nStart = nIndex - 1;
							m_eState = STATE_ATOM;
						}
						else if (m_eMode & MODE_ATOM_LITERAL)
						{
							m_nLiteralLength = 0;
							m_eState = STATE_LITERAL_LENGTH;
						}
						else
							pToken
								= new INetIMAPScannerToken(
									      INetIMAPScannerToken::TYPE_SPECIAL,
										  cChar);
						bRead = true;
						break;

					default:
						if (isAtomChar(cChar))
						{
							nStart = nIndex - 1;
							m_eState = STATE_ATOM;
						}
						else
							pToken
								= new INetIMAPScannerToken(
									      INetIMAPScannerToken::TYPE_SPECIAL,
										  cChar);
						bRead = true;
						break;
				}
				break;

			case STATE_SPACE:
				DBG_ASSERT(nStart == NO_START,
						   "INetIMAPScanner::WriteAt(): Invariant");

				if (cChar == ' ')
					bRead = true;
				else
				{
					pToken = new INetIMAPScannerToken
					              (INetIMAPScannerToken::TYPE_SPACE);
					m_eState = STATE_INITIAL;
					bRead = false;
				}
				break;

			case STATE_ATOM:
				DBG_ASSERT(nStart < nIndex,
						   "INetIMAPScanner::WriteAt(): Invariant");

				if (isAtomChar(cChar))
					if (nIndex - nStart <= STRING_MAXLEN - m_aPrefix.Len())
						bRead = true;
					else
					{
						addToPrefix(pBuffer, nStart, nIndex - 1);
						pToken
						 = new INetIMAPScannerToken
						        (INetIMAPScannerToken::TYPE_ATOM_TOO_LONG,
								 m_aPrefix);
						m_aPrefix.Erase();
						bRead = false;
					}
				else
				{
					addToPrefix(pBuffer, nStart, nIndex - 1);
					nStart = NO_START;
					pToken
					 = new INetIMAPScannerToken
					        (INetIMAPScannerToken::TYPE_ATOM, m_aPrefix);
					m_aPrefix.Erase();
					m_eState = STATE_INITIAL;
					bRead = false;
				}
				break;

			case STATE_QUOTE:
				DBG_ASSERT(nStart < nIndex,
						   "INetIMAPScanner::WriteAt(): Invariant");

				switch (cChar)
				{
					case '\x0A': // LF
					case '\x0D': // CR
						addToPrefix(pBuffer, nStart, nIndex - 1);
						nStart = NO_START;
						pToken
						 = new INetIMAPScannerToken
						        (INetIMAPScannerToken::TYPE_QUOTED_BROKEN,
								 m_aPrefix);
						m_aPrefix.Erase();
						m_eState = STATE_INITIAL;
						bRead = false;
						break;

					case '"':
						addToPrefix(pBuffer, nStart, nIndex - 1);
						nStart = NO_START;
						pToken
						 = new INetIMAPScannerToken
						        (INetIMAPScannerToken::TYPE_QUOTED,
								 m_aPrefix);
						m_aPrefix.Erase();
						m_eState = STATE_INITIAL;
						bRead = true;
						break;

					case '\\':
						addToPrefix(pBuffer, nStart, nIndex - 1);
						++nStart;
						m_eState = STATE_QUOTE_ESCAPE;
						if (m_aPrefix.Len() >= STRING_MAXLEN)
						{
							pToken
							 = new
							    INetIMAPScannerToken
							     (INetIMAPScannerToken::TYPE_QUOTED_TOO_LONG,
								  m_aPrefix);
							m_aPrefix.Erase();
						}
						bRead = true;
						break;

					default:
						if (nIndex - nStart
							    <= STRING_MAXLEN - m_aPrefix.Len())
							bRead = true;
						else
						{
							addToPrefix(pBuffer, nStart, nIndex - 1);
							pToken
							 = new
							    INetIMAPScannerToken
								 (INetIMAPScannerToken::TYPE_QUOTED_TOO_LONG,
								  m_aPrefix);
							m_aPrefix.Erase();
							bRead = false;
						}
						break;
				}
				break;

			case STATE_QUOTE_ESCAPE:
				DBG_ASSERT(nStart < nIndex,
						   "INetIMAPScanner::WriteAt(): Invariant");

				switch (cChar)
				{
					case '\x0A': // LF
					case '\x0D': // CR
						nStart = NO_START;
						pToken
						 = new INetIMAPScannerToken
						        (INetIMAPScannerToken::TYPE_QUOTED_BROKEN,
								 m_aPrefix);
						m_aPrefix.Erase();
						m_eState = STATE_INITIAL;
						bRead = false;
						break;

					default:
						m_eState = STATE_QUOTE;
						bRead = true;
						break;
				}
				break;

			case STATE_LITERAL_LENGTH:
				DBG_ASSERT(nStart == NO_START,
						   "INetIMAPScanner::WriteAt(): Invariant");

				if (INetMIME::isDigit(sal_uChar(cChar)))
				{
					m_nLiteralLength
						= 10 * m_nLiteralLength
						      + INetMIME::getWeight(sal_uChar(cChar));
					bRead = true;
				}
				else if (cChar == '}')
				{
					m_eState = STATE_LITERAL_CLOSE;
					bRead = true;
				}
				else
				{
					pToken
						= new INetIMAPScannerToken(
							      INetIMAPScannerToken::TYPE_LITERAL_BROKEN,
								  ByteString());
					m_eState = STATE_INITIAL;
					bRead = false;
				}
				break;

			case STATE_LITERAL_CLOSE:
				DBG_ASSERT(nStart == NO_START,
						   "INetIMAPScanner::WriteAt(): Invariant");

				switch (cChar)
				{
					case '\x0A': // LF
						if (m_nLiteralLength)
						{
							nStart = nIndex;
							m_eState = STATE_LITERAL_TEXT;
						}
						else
						{
							pToken
								= new INetIMAPScannerToken(
									      INetIMAPScannerToken::TYPE_LITERAL,
										  ByteString());
							m_eState = STATE_INITIAL;
						}
						bRead = true;
						break;

					case '\x0D': // LF
						m_eState = STATE_LITERAL_CR;
						bRead = true;
						break;

					default:
						pToken
							= new INetIMAPScannerToken(
								    INetIMAPScannerToken::TYPE_LITERAL_BROKEN,
									ByteString());
						m_eState = STATE_INITIAL;
						bRead = false;
						break;
				}
				break;

			case STATE_LITERAL_CR:
				DBG_ASSERT(nStart == NO_START,
						   "INetIMAPScanner::WriteAt(): Invariant");

				if (m_nLiteralLength)
				{
					if (cChar == '\x0A') // LF
					{
						nStart = nIndex;
						bRead = true;
					}
					else
					{
						nStart = nIndex - 1;
						bRead = false;
					}
					m_eState = STATE_LITERAL_TEXT;
				}
				else
				{
					pToken = new INetIMAPScannerToken(
						             INetIMAPScannerToken::TYPE_LITERAL,
									 ByteString());
					m_eState = STATE_INITIAL;
					bRead = cChar == '\x0A'; // LF
				}
				break;

			case STATE_LITERAL_TEXT:
				DBG_ASSERT(nStart < nIndex,
						   "INetIMAPScanner::WriteAt(): Invariant");

				if (nIndex - nStart <= STRING_MAXLEN - m_aPrefix.Len())
				{
					if (--m_nLiteralLength == 0)
					{
						addToPrefix(pBuffer, nStart, nIndex);
						nStart = NO_START;
						pToken = new INetIMAPScannerToken
						              (INetIMAPScannerToken::TYPE_LITERAL,
									   m_aPrefix);
						m_aPrefix.Erase();
						m_eState = STATE_INITIAL;
					}
					bRead = true;
				}
				else
				{
					addToPrefix(pBuffer, nStart, nIndex - 1);
					pToken
					 = new INetIMAPScannerToken
					        (INetIMAPScannerToken::TYPE_QUOTED_TOO_LONG,
							 m_aPrefix);
					m_aPrefix.Erase();
					bRead = false;
				}
				break;

			case STATE_CR:
				DBG_ASSERT(nStart == NO_START,
						   "INetIMAPScanner::WriteAt(): Invariant");

				pToken
				 = new
				    INetIMAPScannerToken(INetIMAPScannerToken::TYPE_NEWLINE);
				m_eState = STATE_INITIAL;
				bRead = cChar == '\x0A'; // LF
				break;
		}
	}
}

