/*************************************************************************
 *
 *  $RCSfile: socket.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: sb $ $Date: 2001/12/04 10:28:19 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INET_SOCKET_CXX "$Revision: 1.5 $"

#ifndef __STL_USE_NEWALLOC
#define __STL_USE_NEWALLOC
#endif

#ifndef __UTILITY__
#include <utility>
#endif
#ifndef __HASH_MAP__
#include <hash_map>
#endif
#ifndef __HASH_SET__
#include <hash_set>
#endif
#include <list>

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif
#ifndef _OSL_SOCKET_H_
#include <osl/socket.h>
#endif

#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif
#ifndef _VOS_CONDITN_HXX_
#include <vos/conditn.hxx>
#endif
#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif
#ifndef _VOS_OBJECT_HXX_
#include <vos/object.hxx>
#endif
#ifndef _VOS_QUEUE_HXX_
#include <vos/queue.hxx>
#endif
#ifndef _VOS_SOCKET_HXX_
#include <vos/socket.hxx>
#endif
#ifndef _VOS_THREAD_HXX_
#include <vos/thread.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif
#ifndef _INET_SOCKS4_H_
#include <inet/socks4.h>
#endif

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

/*========================================================================
 *
 * Internal interfaces.
 *
 *======================================================================*/
static NAMESPACE_VOS(IMutex)& __getGlobalMutex_Impl (void);

typedef NAMESPACE_INET(INetSocket) socket_type;
typedef socket_type*               key_type;

struct key_hash : public std::unary_function<key_type, sal_uInt32>
{
	sal_uInt32 operator() (key_type k) const { return ((sal_uInt32)k); }
};

struct key_cmp : public std::binary_function<key_type, key_type, bool>
{
	bool operator() (key_type k1, key_type k2) const { return (k1 == k2); }
};

typedef std::hash_set<key_type, key_hash, key_cmp> set_type;


#if 1  /* __STL_CONST_CONSTRUCTOR_BUG */
typedef void*    map_key;
#else
typedef key_type map_key;
#endif /* __STL_CONST_CONSTRUCTOR_BUG */

typedef NAMESPACE_INET(INetSocket)::EventHandler handler_type;

struct map_value
{
	handler_type *m_pfnHandler;
	void         *m_pData;

	map_value (handler_type *pfnHandler = 0, void *pData = 0)
		: m_pfnHandler (pfnHandler), m_pData (pData)
	{}
	sal_Bool operator== (const map_value& rOther) const
	{
		return (m_pfnHandler == rOther.m_pfnHandler) && (m_pData == rOther.m_pData);
	}
	sal_Bool operator< (const map_value& rOther) const
	{
		return (!(*this == rOther));
	}
};

typedef std::pair<const map_key, map_value> map_entry;

struct map_hash : public std::unary_function<map_key, sal_uInt32>
{
	sal_uInt32 operator() (map_key k) const { return ((sal_uInt32)k); }
};

struct map_cmp : public std::binary_function<map_key, map_key, bool>
{
	bool operator() (map_key k1, map_key k2) const { return (k1 == k2); }
};

typedef std::hash_multimap<map_key, map_value, map_hash, map_cmp> map_type;


#if 0  /* __SUNPRO_CC_TYPEDEF_BUG */
typedef std::pair
	<map_type::iterator, map_type::iterator> range_type;
typedef std::pair
	<map_type::const_iterator, map_type::const_iterator> const_range_type;
#endif /* __SUNPRO_CC_TYPEDEF_BUG */


#ifdef _USE_NAMESPACE
namespace inet {
#endif

/*========================================================================
 *
 * OSocketSet interface.
 *
 *======================================================================*/
class OSocketSet : public NAMESPACE_VOS(OMutex)
{
	VOS_DECLARE_CLASSINFO (VOS_NAMESPACE (OSocketSet, inet));

public:
	OSocketSet (void);
	virtual ~OSocketSet (void);

	void convert (oslSocketSet hSocketSet);

	sal_Bool isEmpty (void) const;

	sal_Bool insert (key_type k);
	sal_Bool remove (key_type k);
	sal_Bool verify (key_type k);

private:
	/** Representation.
	 */
	set_type m_aSet;

	/** Not implemented.
	 */
	OSocketSet (const OSocketSet&);
	OSocketSet& operator= (const OSocketSet&);
};

inline sal_Bool OSocketSet::isEmpty (void) const
{
	return m_aSet.empty();
}

/*========================================================================
 *
 * OSocketMonitor interface.
 *
 *======================================================================*/
class OSocketMonitor :
	public NAMESPACE_VOS(OReference),
	public NAMESPACE_VOS(OThread)
{
	VOS_DECLARE_CLASSINFO (VOS_NAMESPACE (OSocketMonitor, inet));

public:
	/** Get or create the single instance.
	 */
	static sal_Bool getOrCreate (
		NAMESPACE_VOS(ORef)<OSocketMonitor> &rxMonitor);

	sal_Bool isEmpty (void) const;

	sal_Bool insert (key_type k);
	sal_Bool remove (key_type k);

	/** getEventMask.
	 */
	sal_Int32 getEventMask (key_type k);

	/** setEventMask.
	 */
	void setEventMask (key_type k, sal_Int32 nEventMask);

protected:
	/** Construction, destruction.
	 */
	OSocketMonitor (void);
	virtual ~OSocketMonitor (void);

private:
	/** The single instance.
	 */
	static OSocketMonitor *m_pThis;

	/** Representation.
	 */
	NAMESPACE_VOS(OMutex)     m_aMutex;
	set_type                  m_aSet;

	NAMESPACE_VOS(OCondition) m_aNotEmpty;
	OSocketSet                m_aIncomingSet;
	OSocketSet                m_aOutgoingSet;
	OSocketSet                m_aOutOfBandSet;

	/** OThread.
	 */
	virtual void SAL_CALL run  (void);
	virtual void SAL_CALL kill (void);

	/** Not implemented.
	 */
	OSocketMonitor (const OSocketMonitor&);
	OSocketMonitor& operator= (const OSocketMonitor&);
};

inline sal_Bool OSocketMonitor::isEmpty (void) const
{
	return (m_aIncomingSet.isEmpty() &&
			m_aOutgoingSet.isEmpty() &&
			m_aOutOfBandSet.isEmpty()   );
}

/*========================================================================
 *
 * OSocketDispatcher interface.
 *
 *======================================================================*/
class OSocketDispatcher :
	public NAMESPACE_VOS(OReference),
	public NAMESPACE_VOS(OThread)
{
	VOS_DECLARE_CLASSINFO (VOS_NAMESPACE (OSocketDispatcher, inet));

public:
	/** Obtain an OSocketDispatcher instance.
	 */
	static sal_Bool createInstance (
		NAMESPACE_VOS(ORef)<OSocketDispatcher> &rxDispatcher);

	/** OReference.
	 */
	virtual RefCount SAL_CALL release (void);

	/** Event handler.
	 */
	sal_Bool registerEventHandler (
		key_type k, handler_type *pfnHandler, void *pData);

	sal_Bool deregisterEventHandler (
		key_type k, handler_type *pfnHandler);

	/** handleEvent (call registered handlers).
	 */
	sal_Bool handleEvent (key_type k, sal_Int32 n);

	/** Post event.
	 */
	sal_Bool postEvent (key_type k, sal_Int32 n);

protected:
	/** Construction, destruction.
	 */
	OSocketDispatcher (void);
	virtual ~OSocketDispatcher (void);

private:
	/** Representation.
	 */
	struct Event
	{
		key_type  m_pSocket;
		sal_Int32 m_nEvent;

		Event (key_type k = 0, sal_Int32 n = 0)
			: m_pSocket (k), m_nEvent (n)
		{
			if (m_pSocket)
				m_pSocket->acquire();
		}

		~Event (void)
		{
			if (m_pSocket)
				m_pSocket->release();
		}

		Event (const Event& rOther)
			: m_pSocket (rOther.m_pSocket),
			  m_nEvent  (rOther.m_nEvent)
		{
			if (m_pSocket)
				m_pSocket->acquire();
		}

		Event& operator= (const Event& rOther)
		{
			if (m_pSocket)
				m_pSocket->release();
			m_pSocket = rOther.m_pSocket;
			if (m_pSocket)
				m_pSocket->acquire();
			m_nEvent = rOther.m_nEvent;
			return *this;
		}

		sal_Bool operator== (const Event& rOther) const
		{
			return ((m_pSocket == rOther.m_pSocket) &&
					(m_nEvent  == rOther.m_nEvent )    );
		}

		sal_Bool operator< (const Event& rOther) const
		{
			return (!(*this == rOther));
		}
	};

	NAMESPACE_VOS(OQueue)<Event> m_aQueue;
	NAMESPACE_VOS(OMutex)        m_aMutex;
	map_type                     m_aMap;

	/** OThread.
	 */
	virtual void SAL_CALL run (void);
	virtual void SAL_CALL terminate (void);
	virtual void SAL_CALL onTerminated (void);

	/** Not implemented.
	 */
	OSocketDispatcher (const OSocketDispatcher&);
	OSocketDispatcher& operator= (const OSocketDispatcher&);
};

#if 1  /* __SUNPRO_CC_TYPEDEF_BUG */

typedef std::pair<map_type::iterator, map_type::iterator> range_type;

#endif /* __SUNPRO_CC_TYPEDEF_BUG */

/*========================================================================
 *
 * End internal interfaces.
 *
 *======================================================================*/
#ifdef _USE_NAMESPACE
}
#endif

/*
 * __getGlobalMutex_Impl.
 */
static NAMESPACE_VOS(IMutex)& __getGlobalMutex_Impl (void)
{
	static NAMESPACE_VOS(IMutex) *pMutex = NULL;
	if (!pMutex)
	{
		NAMESPACE_VOS(OGuard) aGuard (NAMESPACE_VOS(OMutex)::getGlobalMutex());
		if (!pMutex)
		{
			static NAMESPACE_VOS(OMutex) aGlobalMutex;
			pMutex = &aGlobalMutex;
		}
	}
	return *pMutex;
}

/*========================================================================
 *
 * OSocketSet implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (OSocketSet, inet),
	VOS_NAMESPACE (OSocketSet, inet),
	VOS_NAMESPACE (OMutex, vos),
	0);

/*
 * OSocketSet.
 */
OSocketSet::OSocketSet (void)
{
}

/*
 * ~OSocketSet.
 */
OSocketSet::~OSocketSet (void)
{
}

/*
 * convert.
 */
void OSocketSet::convert (oslSocketSet hSocketSet)
{
	NAMESPACE_VOS(OGuard) aGuard (*this);

	set_type::const_iterator first = m_aSet.begin();
	set_type::const_iterator last  = m_aSet.end();

	osl_clearSocketSet (hSocketSet);
	while (first != last)
	{
		key_type k = (key_type)(*first++);
		if (k->isValid())
			osl_addToSocketSet (hSocketSet, (oslSocket)(*k));
	}
}

/*
 * insert.
 */
sal_Bool OSocketSet::insert (key_type k)
{
	NAMESPACE_VOS(OGuard) aGuard (*this);
	if (k)
	{
		typedef std::pair<set_type::iterator, bool> result_t;
		result_t result = m_aSet.insert (k);
		if (!result.second)
			k = 0;
	}
	return (!!k);
}

/*
 * remove.
 */
sal_Bool OSocketSet::remove (key_type k)
{
	NAMESPACE_VOS(OGuard) aGuard (*this);

	set_type::iterator it = m_aSet.find (k);
	if (it != m_aSet.end())
	{
		m_aSet.erase (it);
		return sal_True;
	}
	return sal_False;
}

/*
 * verify.
 */
sal_Bool OSocketSet::verify (key_type k)
{
	NAMESPACE_VOS(OGuard) aGuard (*this);

	set_type::const_iterator it = m_aSet.find (k);
	return (it != m_aSet.end());
}

/*========================================================================
 *
 * OSocketMonitor implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (OSocketMonitor, inet),
	VOS_NAMESPACE (OSocketMonitor, inet),
	VOS_NAMESPACE (OThread, vos),
	0);

/*
 * The single instance.
 */
NAMESPACE_INET(OSocketMonitor)*
NAMESPACE_INET(OSocketMonitor)::m_pThis = NULL;

/*
 * OSocketMonitor.
 */
OSocketMonitor::OSocketMonitor (void)
{
	NAMESPACE_VOS(IMutex) &rMutex = __getGlobalMutex_Impl();

	create();

	rMutex.acquire();
	m_pThis = this;
	rMutex.release();
}

/*
 * ~OSocketMonitor.
 */
OSocketMonitor::~OSocketMonitor (void)
{
	NAMESPACE_VOS(IMutex) &rMutex = __getGlobalMutex_Impl();

	rMutex.acquire();
	m_pThis = NULL;
	rMutex.release();

	kill();
}

/*
 * getOrCreate.
 */
sal_Bool OSocketMonitor::getOrCreate (
	NAMESPACE_VOS(ORef)<OSocketMonitor> &rxMonitor)
{
	NAMESPACE_VOS(OGuard) aGuard (__getGlobalMutex_Impl());
	if (!m_pThis)
		new OSocketMonitor();
	rxMonitor = m_pThis;

	return rxMonitor.isValid();
}

/*
 * insert.
 */
sal_Bool OSocketMonitor::insert (key_type k)
{
	NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
	if (k)
	{
		typedef std::pair<set_type::iterator, bool> result_t;
		result_t result = m_aSet.insert (k);
		if (!result.second)
			k = 0;
	}
	return (!!k);
}

/*
 * remove.
 */
sal_Bool OSocketMonitor::remove (key_type k)
{
	NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

	set_type::iterator it = m_aSet.find (k);
	if (it != m_aSet.end())
	{
		m_aSet.erase (it);
		setEventMask (k, 0);
		return sal_True;
	}
	return sal_False;
}

/*
 * getEventMask.
 */
sal_Int32 OSocketMonitor::getEventMask (key_type k)
{
	sal_Int32 nEventMask = 0;
	if (m_aIncomingSet.verify(k))
		nEventMask |= socket_type::EVENT_READ;
	if (m_aOutgoingSet.verify(k))
		nEventMask |= socket_type::EVENT_WRITE;
	if (m_aOutOfBandSet.verify(k))
		nEventMask |= socket_type::EVENT_OOB;
	return nEventMask;
}

/*
 * setEventMask.
 */
void OSocketMonitor::setEventMask (key_type k, sal_Int32 nEventMask)
{
	if (nEventMask & socket_type::EVENT_READ)
		m_aIncomingSet.insert(k);
	else
		m_aIncomingSet.remove(k);

	if (nEventMask & socket_type::EVENT_WRITE)
		m_aOutgoingSet.insert(k);
	else
		m_aOutgoingSet.remove(k);

	if (nEventMask & socket_type::EVENT_OOB)
		m_aOutOfBandSet.insert(k);
	else
		m_aOutOfBandSet.remove(k);

	if (isEmpty())
		m_aNotEmpty.reset();
	else
		m_aNotEmpty.set();
}

/*
 * run.
 */
void SAL_CALL OSocketMonitor::run (void)
{
	setPriority (TPriority_BelowNormal);

	oslSocketSet hIncomingSet  = osl_createSocketSet();
	oslSocketSet hOutgoingSet  = osl_createSocketSet();
	oslSocketSet hOutOfBandSet = osl_createSocketSet();
	if (!(hIncomingSet && hOutgoingSet && hOutOfBandSet))
	{
		// Unable to monitor.
		terminate();
	}

	while (schedule())
	{
		// Check socket sets.
		if (isEmpty())
			m_aNotEmpty.reset();
		m_aNotEmpty.wait();

		// Initialize socket sets.
		m_aIncomingSet.convert (hIncomingSet);
		m_aOutgoingSet.convert (hOutgoingSet);
		m_aOutOfBandSet.convert (hOutOfBandSet);

		// Enter blocking select().
		TimeValue tmo;
		tmo.Seconds = 0;
		tmo.Nanosec = 100 * 1000 * 1000; /* 100 [ms] */

		sal_Int32 nReady = osl_demultiplexSocketEvents (
			hIncomingSet, hOutgoingSet, hOutOfBandSet, &tmo);

		if (nReady <= 0)
		{
			// Failure or Timeout.
			oslSocketError eErrCode = osl_Socket_E_TimedOut;
			if (nReady < 0)
				eErrCode = osl_getLastSocketError (NULL);
			continue;
		}

		// Process ready sockets.
		m_aMutex.acquire();
		set_type aSet (m_aSet);
		m_aMutex.release();

		set_type::const_iterator first = aSet.begin();
		set_type::const_iterator last  = aSet.end();

		while ((nReady > 0) && (first != last))
		{
			key_type k = (key_type)(*first++);
			if (k)
			{
				// Verify and Acquire.
				NAMESPACE_VOS(OGuard) aGuard (m_aMutex);

				set_type::const_iterator it = m_aSet.find(k);
				if ((it != m_aSet.end()) && k->isValid())
					k->acquire();
				else
					k = 0;
			}
			if (k)
			{
				// Process and Release.
				oslSocket socket = (oslSocket)(*k);
				if ((nReady > 0) && osl_isInSocketSet (hIncomingSet, socket))
				{
					m_aIncomingSet.remove(k);
					k->postEvent (socket_type::EVENT_READ);
					nReady--;
				}
				if ((nReady > 0) && osl_isInSocketSet (hOutgoingSet, socket))
				{
					sal_Int32 nEvent = socket_type::EVENT_WRITE;
					m_aOutgoingSet.remove(k);
					if (m_aOutOfBandSet.remove(k))
						nEvent |= socket_type::EVENT_CONNECT;
					k->postEvent (nEvent);
					nReady--;
				}
				if ((nReady > 0) && osl_isInSocketSet (hOutOfBandSet, socket))
				{
					sal_Int32 nEvent = socket_type::EVENT_OOB;
					m_aOutOfBandSet.remove(k);
					if (m_aOutgoingSet.remove(k))
						nEvent |= socket_type::EVENT_CONNECT;
					k->postEvent (nEvent);
					nReady--;
				}
				k->release();
			}
		}
	}

	// Finished.
	osl_destroySocketSet (hIncomingSet);
	osl_destroySocketSet (hOutgoingSet);
	osl_destroySocketSet (hOutOfBandSet);
}

/*
 * kill.
 */
void SAL_CALL OSocketMonitor::kill (void)
{
	terminate();
	if (!(getCurrentIdentifier() == getIdentifier()))
	{
		m_aNotEmpty.set();
		join();
	}
	OThread::kill();
}

/*========================================================================
 *
 * OSocketDispatcher implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (OSocketDispatcher, inet),
	VOS_NAMESPACE (OSocketDispatcher, inet),
	VOS_NAMESPACE (OThread, vos),
	0);

/*
 * OSocketDispatcher.
 */
OSocketDispatcher::OSocketDispatcher (void)
{
	if (createSuspended())
	{
		// Ensure clean destruction.
		acquire();
		resume();
	}
}

/*
 * ~OSocketDispatcher.
 */
OSocketDispatcher::~OSocketDispatcher (void)
{
	while (!m_aQueue.isEmpty())
		m_aQueue.removeHead();
}

/*
 * release.
 */
NAMESPACE_VOS(OReference)::RefCount SAL_CALL OSocketDispatcher::release (void)
{
	if ((referenced() == 2) && isRunning())
	{
		// Initiate clean destruction.
		terminate();
	}
	return OReference::release();
}

/*
 * createInstance.
 */
sal_Bool OSocketDispatcher::createInstance (
	NAMESPACE_VOS(ORef)<OSocketDispatcher> &rxDispatcher)
{
	rxDispatcher = new OSocketDispatcher();
	return rxDispatcher.isValid();
}

/*
 * registerEventHandler.
 */
sal_Bool OSocketDispatcher::registerEventHandler (
	key_type k, handler_type *pfnHandler, void *pData)
{
	NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
	range_type range = m_aMap.equal_range(k);

	map_type::iterator it;
	for (it = range.first; it != range.second; ++it)
	{
		map_value v = (*it).second;
		if ((v.m_pfnHandler == pfnHandler) && (v.m_pData == pData))
			break;
	}
	if (it == range.second)
	{
		m_aMap.insert (map_entry (k, map_value (pfnHandler, pData)));
		return sal_True;
	}
	return sal_False;
}

/*
 * deregisterEventHandler.
 */
sal_Bool OSocketDispatcher::deregisterEventHandler (
	key_type k, handler_type *pfnHandler)
{
	NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
	range_type range = m_aMap.equal_range(k);

	sal_Bool result = sal_False;
	while (range.first != range.second)
	{
		map_type::iterator next = range.first;
		++next;

		map_value v = (*(range.first)).second;
		if (v.m_pfnHandler == pfnHandler)
		{
			m_aMap.erase (range.first);
			result |= sal_True;
		}
		range.first = next;
	}
	return (result);
}

/*
 * handleEvent (call registered handlers).
 */
sal_Bool OSocketDispatcher::handleEvent (key_type k, sal_Int32 n)
{
	if (k)
	{
        typedef std::list< map_value > List;
        List list;
        {
            vos::OGuard aGuard(m_aMutex);
            range_type range(m_aMap.equal_range(k));
            for (map_type::iterator it = range.first; it != range.second;
                 ++it)
                list.push_back(it->second);
        }
        for (List::iterator it = list.begin(); it != list.end(); ++it)
            it->m_pfnHandler(vos::ORef< INetSocket >(k), n, it->m_pData);
	}
	return (!!k);
}

/*
 * postEvent.
 */
sal_Bool OSocketDispatcher::postEvent (key_type k, sal_Int32 n)
{
	if (k && isRunning())
	{
		// Enqueue.
		Event evt(k, n);
		m_aQueue.addTail (evt);
		return sal_True;
	}
	return sal_False;
}

/*
 * run.
 */
void SAL_CALL OSocketDispatcher::run (void)
{
	setPriority (TPriority_BelowNormal);
	while (schedule())
	{
		// Wait for next event.
		Event evt (m_aQueue.getHead());
		if (evt.m_pSocket)
		{
			// Handle event.
			evt.m_pSocket->handleEvent (evt.m_nEvent);
		}
	}
}

/*
 * terminate.
 */
void SAL_CALL OSocketDispatcher::terminate (void)
{
	// Terminate and unblock.
	OThread::terminate();
	m_aQueue.addTail (Event(0, 0));
}

/*
 * onTerminated.
 */
void SAL_CALL OSocketDispatcher::onTerminated (void)
{
	// Release constructor reference.
	release();
}

/*========================================================================
 *
 * INetSocket implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (INetSocket, inet),
	VOS_NAMESPACE (INetSocket, inet),
	VOS_NAMESPACE (OSocket, vos),
	0);

/*
 * INetSocket.
 */
INetSocket::INetSocket (TSocketType eType)
	: OSocket (eType, TFamily_Inet, TProtocol_Ip)
{
	enableNonBlockingMode();

	OSocketDispatcher::createInstance (m_xDispatcher);

	if (OSocketMonitor::getOrCreate (m_xMonitor))
		m_xMonitor->insert (this);
}

/*
 * INetSocket.
 */
INetSocket::INetSocket (oslSocket hSocket)
	: OSocket (hSocket)
{
	enableNonBlockingMode();

	OSocketDispatcher::createInstance (m_xDispatcher);

	if (OSocketMonitor::getOrCreate (m_xMonitor))
		m_xMonitor->insert (this);
}

/*
 * ~INetSocket.
 */
INetSocket::~INetSocket (void)
{
	if (m_xMonitor.isValid())
		m_xMonitor->remove (this);
}

/*
 * enableEvent.
 */
void INetSocket::enableEvent (sal_Int32 nEvent)
{
	if (m_xMonitor.isValid())
	{
		sal_Int32 nMask = m_xMonitor->getEventMask (this);
		m_xMonitor->setEventMask (this, nMask | nEvent);
	}
}

/*
 * registerEventHandler.
 */
sal_Bool INetSocket::registerEventHandler (
	EventHandler *pfnHandler, void *pData)
{
	if (m_xDispatcher.isValid())
		return m_xDispatcher->registerEventHandler (this, pfnHandler, pData);
	else
		return sal_False;
}

/*
 * deregisterEventHandler.
 */
sal_Bool INetSocket::deregisterEventHandler (EventHandler *pfnHandler)
{
	if (m_xDispatcher.isValid())
		return m_xDispatcher->deregisterEventHandler (this, pfnHandler);
	else
		return sal_False;
}

/*
 * handleEvent (call registered handlers).
 */
sal_Bool INetSocket::handleEvent (sal_Int32 nEvent)
{
	if (m_xDispatcher.isValid())
		return m_xDispatcher->handleEvent (this, nEvent);
	else
		return sal_False;
}

/*
 * postEvent.
 */
sal_Bool INetSocket::postEvent (sal_Int32 nEvent)
{
	if (m_xDispatcher.isValid())
		return m_xDispatcher->postEvent (this, nEvent);
	else
		return sal_False;
}

/*
 * getMyAddr.
 */
sal_Bool INetSocket::getMyAddr (NAMESPACE_VOS(OSocketAddr) &rMyAddr) const
{
	// Determine Local address.
	OSocket::getLocalAddr (rMyAddr);

	// Check result.
	oslSocketAddr hAddr = (oslSocketAddr)rMyAddr;
	return (!!hAddr);
}

/*
 * getToAddr.
 */
sal_Bool INetSocket::getToAddr (NAMESPACE_VOS(OSocketAddr) &rToAddr) const
{
	// Determine Peer address.
	OSocket::getPeerAddr (rToAddr);

	// Check result.
	oslSocketAddr hAddr = (oslSocketAddr)rToAddr;
	return (!!hAddr);
}

/*
 * getLastError.
 */
oslSocketError INetSocket::getLastError (void) const
{
	oslSocketError eErrCode = osl_Socket_E_NotSocket;
	if (isValid())
	{
		eErrCode = osl_getLastSocketError (*this);
		switch (eErrCode)
		{
			case osl_Socket_E_NetDown:
			case osl_Socket_E_NetReset:
			case osl_Socket_E_NetUnreachable:
			case osl_Socket_E_HostDown:
			case osl_Socket_E_HostUnreachable:
				eErrCode = osl_Socket_E_NetDown;
				break;

			case osl_Socket_E_Already:
			case osl_Socket_E_InProgress:
			case osl_Socket_E_WouldBlock:
				eErrCode = osl_Socket_E_WouldBlock;
				break;

			default:
				break;
		}
	}
	return eErrCode;
}

/*
 * recv.
 */
sal_Int32 INetSocket::recv (
	void *pData, sal_uInt32 nData, TSocketMsgFlag eFlag)
{
	if (!isValid())
		return (-osl_Socket_E_NotSocket);

	sal_Int32 nRecv = osl_receiveSocket (
		*this, pData, nData, (oslSocketMsgFlag)eFlag);
	if (nRecv < 0)
	{
		oslSocketError eErrCode = getLastError();
		if (eErrCode == osl_Socket_E_WouldBlock)
		{
			enableEvent (EVENT_READ);
		}
		nRecv = -eErrCode;
	}
	if (nRecv > 0)
	{
		enableEvent (EVENT_READ);
	}
	return nRecv;
}

/*
 * send.
 */
sal_Int32 INetSocket::send (
	const void *pData, sal_uInt32 nData, TSocketMsgFlag eFlag)
{
	if (!isValid())
		return (-osl_Socket_E_NotSocket);

	sal_Int32 nSend = osl_sendSocket (
		*this, pData, nData, (oslSocketMsgFlag)eFlag);
	if (nSend < 0)
	{
		oslSocketError eErrCode = getLastError();
		if (eErrCode == osl_Socket_E_WouldBlock)
		{
			enableEvent (EVENT_WRITE);
		}
		nSend = -eErrCode;
	}
	return nSend;
}

/*
 * close.
 */
void SAL_CALL INetSocket::close (void)
{
	if (m_xMonitor.isValid())
	{
		// Disable further event notification.
		m_xMonitor->setEventMask (this, 0);
	}
	if (m_xDispatcher.isValid())
	{
		// Post final close event.
		m_xDispatcher->postEvent (this, EVENT_CLOSE);
	}
	OSocket::close();
}

/*========================================================================
 *
 * INetUDPSocket implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (INetUDPSocket, inet),
	VOS_NAMESPACE (INetUDPSocket, inet),
	VOS_NAMESPACE (INetSocket, inet),
	0);

/*
 * INetUDPSocket.
 */
INetUDPSocket::INetUDPSocket (void)
	: INetSocket (TType_Dgram)
{
}

/*
 * ~INetUDPSocket.
 */
INetUDPSocket::~INetUDPSocket (void)
{
}

/*
 * recvFrom.
 */
sal_Int32 INetUDPSocket::recvFrom (
	NAMESPACE_VOS(OSocketAddr) &rFromAddr,
	void                       *pData,
	sal_uInt32                  nData,
	TSocketMsgFlag              eFlag)
{
	if (!isValid())
		return (-osl_Socket_E_NotSocket);

	sal_Int32 nRecv = osl_receiveFromSocket (
		*this, rFromAddr, pData, nData, (oslSocketMsgFlag)eFlag);
	if (nRecv < 0)
	{
		oslSocketError eErrCode = getLastError();
		if (eErrCode == osl_Socket_E_WouldBlock)
		{
			enableEvent (EVENT_READ);
		}
		nRecv = -eErrCode;
	}
	if (nRecv > 0)
	{
		enableEvent (EVENT_READ);
	}
	return nRecv;
}

/*
 * sendTo.
 */
sal_Int32 INetUDPSocket::sendTo (
	const NAMESPACE_VOS(OSocketAddr) &rToAddr,
	const void                       *pData,
	sal_uInt32                        nData,
	TSocketMsgFlag                    eFlag)
{
	if (!isValid())
		return (-osl_Socket_E_NotSocket);

	sal_Int32 nSend = osl_sendToSocket (
		*this, rToAddr, pData, nData, (oslSocketMsgFlag)eFlag);
	if (nSend < 0)
	{
		oslSocketError eErrCode = getLastError();
		if (eErrCode == osl_Socket_E_WouldBlock)
		{
			enableEvent (EVENT_WRITE);
		}
		nSend = -eErrCode;
	}
	return nSend;
}

/*========================================================================
 *
 * INetTCPSocket implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (INetTCPSocket, inet),
	VOS_NAMESPACE (INetTCPSocket, inet),
	VOS_NAMESPACE (INetSocket, inet),
	0);

/*
 * INetTCPSocket.
 */
INetTCPSocket::INetTCPSocket (void)
	: INetSocket (TType_Stream),
	  m_hContext (NULL)
{
}

/*
 * INetTCPSocket.
 */
INetTCPSocket::INetTCPSocket (oslSocket hSocket)
	: INetSocket (hSocket),
	  m_hContext (NULL)
{
}

/*
 * ~INetTCPSocket.
 */
INetTCPSocket::~INetTCPSocket (void)
{
	__osl_socks_destroyContext (m_hContext);
}

/*
 * setSocksGateway.
 */
sal_Bool INetTCPSocket::setSocksGateway (
	const NAMESPACE_VOS(OSocketAddr) &rGwAddr)
{
	oslSocketAddr hAddr = (oslSocketAddr)rGwAddr;
	if (m_hContext)
	{
		__osl_socks_destroyContext (m_hContext);
		m_hContext = (oslSocksContext)NULL;
	}
	if (isValid())
	{
		// Create Socks context.
		m_hContext = __osl_socks_createContext (*this, hAddr);
	}
	return (!!m_hContext);
}

/*
 * getMyAddr.
 */
sal_Bool INetTCPSocket::getMyAddr (NAMESPACE_VOS(OSocketAddr) &rMyAddr) const
{
	oslSocketAddr hAddr = (oslSocketAddr)NULL;
	if (m_hContext)
	{
		// Determine Bind address.
		__osl_socks_getContext (m_hContext, &hAddr);
		rMyAddr = hAddr;
	}
	else
	{
		// Determine Local address.
		OSocket::getLocalAddr (rMyAddr);
		hAddr = (oslSocketAddr)rMyAddr;
	}
	return (!!hAddr);
}

/*
 * getToAddr.
 */
sal_Bool INetTCPSocket::getToAddr (NAMESPACE_VOS(OSocketAddr) &rToAddr) const
{
	oslSocketAddr hAddr = (oslSocketAddr)m_aToAddr;
	if (!hAddr)
	{
		// Determine Peer address.
		OSocket::getPeerAddr ((NAMESPACE_VOS(OSocketAddr)&)m_aToAddr);
		hAddr = (oslSocketAddr)m_aToAddr;
	}
	rToAddr = m_aToAddr;
	return (!!hAddr);
}

/*
 * setToAddr.
 */
sal_Bool INetTCPSocket::setToAddr (const NAMESPACE_VOS(OSocketAddr) &rToAddr)
{
	oslSocketAddr hAddr = (oslSocketAddr)rToAddr;
	m_aToAddr = rToAddr;
	return (!!hAddr);
}

/*========================================================================
 *
 * INetActiveTCPSocket implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (INetActiveTCPSocket, inet),
	VOS_NAMESPACE (INetActiveTCPSocket, inet),
	VOS_NAMESPACE (INetTCPSocket, inet),
	0);

/*
 * INetActiveTCPSocket.
 */
INetActiveTCPSocket::INetActiveTCPSocket (void)
	: m_bConnected (sal_False)
{
}

/*
 * ~INetActiveTCPSocket.
 */
INetActiveTCPSocket::~INetActiveTCPSocket (void)
{
}

/*
 * connect.
 */
sal_Bool INetActiveTCPSocket::connect (
	const NAMESPACE_VOS(OSocketAddr) &rToAddr)
{
	oslSocketResult result;

	VOS_PRECOND (isValid(), "INetActiveTCPSocket::connect(): not valid");
	if (!isValid())
		return sal_False;

	if (!setToAddr (rToAddr))
		return sal_False;

	oslSocksContext hContext = getSocksContext();
	if (hContext)
	{
		// Initiate gateway connection.
		result = __osl_socks_connectSocketTo (hContext, rToAddr);
	}
	else
	{
		// Initiate direct connection.
		result = osl_connectSocketTo (*this, rToAddr, NULL);
	}

	if (result == osl_Socket_Error)
	{
		oslSocketError error = getLastError();
		if (error == osl_Socket_E_WouldBlock)
		{
			// Nonblocking operation in progress.
			result = osl_Socket_InProgress;
		}
	}

	m_bConnected = (result == osl_Socket_Ok);

	if (result == osl_Socket_InProgress)
	{
		// Enable event notification.
		enableEvent (EVENT_WRITE | EVENT_OOB);
	}

	if (m_bConnected)
		return postEvent (EVENT_CONNECT | EVENT_WRITE);
	else
		return (!(result == osl_Socket_Error));
}

/*
 * close.
 */
void SAL_CALL INetActiveTCPSocket::close (void)
{
	if (isValid())
	{
		// Terminate gracefully.
		osl_shutdownSocket (*this, osl_Socket_DirReadWrite);
	}
	INetTCPSocket::close();
}

/*
 * handleEvent.
 */
sal_Bool INetActiveTCPSocket::handleEvent (sal_Int32 nEvent)
{
	if (!(m_bConnected || (nEvent & EVENT_OOB) || (nEvent & EVENT_CLOSE)))
	{
		// Check for Socks gateway.
		oslSocksContext hContext = getSocksContext();
		if (hContext)
		{
			// Gateway connection.
			oslSocketResult result;

			result = __osl_socks_connectContext (hContext);
			if (!(result == osl_Socket_Ok))
			{
				if (result == osl_Socket_InProgress)
				{
					enableEvent (EVENT_WRITE | EVENT_OOB);
					return sal_True;
				}
				return INetTCPSocket::handleEvent (EVENT_CONNECT | EVENT_OOB);
			}

			result = __osl_socks_sendContext (hContext);
			if (!(result == osl_Socket_Ok))
			{
				if (result == osl_Socket_InProgress)
				{
					enableEvent (EVENT_WRITE);
					return sal_True;
				}
				return INetTCPSocket::handleEvent (EVENT_CONNECT | EVENT_OOB);
			}

			result = __osl_socks_recvContext (hContext);
			if (!(result == osl_Socket_Ok))
			{
				if (result == osl_Socket_InProgress)
				{
					enableEvent (EVENT_READ);
					return sal_True;
				}
				return INetTCPSocket::handleEvent (EVENT_CONNECT | EVENT_OOB);
			}

			result = __osl_socks_getContext (hContext, NULL);
			m_bConnected = (result == osl_Socket_Ok);

			nEvent = EVENT_CONNECT;
			if (m_bConnected)
				nEvent |= EVENT_WRITE;
			else
				nEvent |= EVENT_OOB;
		}
		else
		{
			// Direct connection.
			if (isValid())
			{
				NAMESPACE_VOS(OInetSocketAddr) aAddr;
				getToAddr (aAddr);
				osl_connectSocketTo (*this, aAddr, NULL);

				oslSocketError error = getLastError();
				if ((error == osl_Socket_E_WouldBlock  ) ||
					(error == osl_Socket_E_InvalidError)    )
				{
					// Should happen once, only.
					enableEvent (EVENT_WRITE | EVENT_OOB);
					return sal_True;
				}
				if (error == osl_Socket_E_IsConnected)
				{
					// Should happen always.
					error = osl_Socket_E_None;
				}
				if (error == osl_Socket_E_None)
				{
					// Obtain local address.
					m_bConnected = getMyAddr (aAddr);
				}
			}

			nEvent = EVENT_CONNECT;
			if (m_bConnected)
				nEvent |= EVENT_WRITE;
			else
				nEvent |= EVENT_OOB;
		}
	}
	return INetTCPSocket::handleEvent (nEvent);
}

/*========================================================================
 *
 * INetAcceptedTCPSocket implementation.
 *
 *======================================================================*/
class INetAcceptedTCPSocket : public NAMESPACE_INET(INetTCPSocket)
{
public:
	INetAcceptedTCPSocket (oslSocket hSocket);
	virtual void SAL_CALL close (void);

protected:
	virtual ~INetAcceptedTCPSocket (void);

private:
	INetAcceptedTCPSocket (const INetAcceptedTCPSocket&);
	INetAcceptedTCPSocket& operator= (const INetAcceptedTCPSocket&);
};

/*
 * INetAcceptedSocket.
 */
INetAcceptedTCPSocket::INetAcceptedTCPSocket (oslSocket hSocket)
	: INetTCPSocket (hSocket)
{
}

/*
 * ~INetAcceptedSocket.
 */
INetAcceptedTCPSocket::~INetAcceptedTCPSocket (void)
{
}

/*
 * close.
 */
void SAL_CALL INetAcceptedTCPSocket::close (void)
{
	if (isValid())
	{
		// Terminate gracefully.
		osl_shutdownSocket (*this, osl_Socket_DirReadWrite);
	}
	INetTCPSocket::close();
}

/*========================================================================
 *
 * INetPassiveTCPSocket implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME (INetPassiveTCPSocket, inet),
	VOS_NAMESPACE (INetPassiveTCPSocket, inet),
	VOS_NAMESPACE (INetTCPSocket, inet),
	0);

/*
 * INetPassiveTCPSocket.
 */
INetPassiveTCPSocket::INetPassiveTCPSocket (void)
	: m_bListening (sal_False)
{
}

/*
 * ~INetPassiveTCPSocket.
 */
INetPassiveTCPSocket::~INetPassiveTCPSocket (void)
{
}

/*
 * accept.
 */
sal_Bool INetPassiveTCPSocket::accept (
	NAMESPACE_VOS(ORef)<INetTCPSocket> &rxSocket,
	NAMESPACE_VOS(OSocketAddr)         &rFromAddr)
{
	if (rxSocket.isValid())
	{
		rxSocket->close();
		rxSocket.unbind();
	}

	if (isValid())
	{
		oslSocket     hSocket = NULL;
		oslSocketAddr hAddr   = NULL;

		oslSocksContext hContext = getSocksContext();
		if (hContext)
		{
			// Accept gateway connection.
			hSocket = __osl_socks_acceptConnectionOnSocket (hContext, &hAddr);
		}
		else
		{
			// Accept direct connection.
			hSocket = osl_acceptConnectionOnSocket (*this, &hAddr);
		}
		if (hSocket)
		{
			// Create accepted socket.
			rxSocket  = new INetAcceptedTCPSocket (hSocket);
			rFromAddr = hAddr;
		}
	}
	return rxSocket.isValid();
}

/*
 * listen.
 */
sal_Bool INetPassiveTCPSocket::listen (
	const NAMESPACE_VOS(OSocketAddr) &rLocalAddr,
	const NAMESPACE_VOS(OSocketAddr) *pToAddr)
{
	oslSocketResult result;
	if (!isValid())
		return sal_False;

	oslSocksContext hContext = getSocksContext();
	if (hContext)
	{
		// Remote listen on gateway address.
		if (pToAddr)
			result = __osl_socks_listenOnSocket (hContext, *pToAddr);
		else
			result = osl_Socket_Error;

		if (result == osl_Socket_InProgress)
		{
			// Enable event notification.
			enableEvent (EVENT_WRITE | EVENT_OOB);
		}
	}
	else
	{
		// Direct listen on local address.
		setReuseAddr (1);
		if (!bind (rLocalAddr))
			return sal_False;

		if (osl_listenOnSocket (*this, -1))
			result = osl_Socket_Ok;
		else
			result = osl_Socket_Error;
	}

	m_bListening = (result == osl_Socket_Ok);
	if (m_bListening)
		return postEvent (EVENT_LISTEN);
	else
		return (!(result == osl_Socket_Error));
}

/*
 * handleEvent.
 */
sal_Bool INetPassiveTCPSocket::handleEvent (sal_Int32 nEvent)
{
	if (!m_bListening)
	{
		// Check for Socks gateway.
		oslSocksContext hContext = getSocksContext();
		if (hContext)
		{
			oslSocketResult result;

			result = __osl_socks_connectContext (hContext);
			if (!(result == osl_Socket_Ok))
			{
				if (result == osl_Socket_InProgress)
				{
					enableEvent (EVENT_WRITE | EVENT_OOB);
					return sal_True;
				}
				return INetTCPSocket::handleEvent (EVENT_LISTEN | EVENT_OOB);
			}

			result = __osl_socks_sendContext (hContext);
			if (!(result == osl_Socket_Ok))
			{
				if (result == osl_Socket_InProgress)
				{
					enableEvent (EVENT_WRITE);
					return sal_True;
				}
				return INetTCPSocket::handleEvent (EVENT_LISTEN | EVENT_OOB);
			}

			result = __osl_socks_recvContext (hContext);
			if (!(result == osl_Socket_Ok))
			{
				if (result == osl_Socket_InProgress)
				{
					enableEvent (EVENT_READ);
					return sal_True;
				}
				return INetTCPSocket::handleEvent (EVENT_LISTEN | EVENT_OOB);
			}

			result = __osl_socks_getContext (hContext, NULL);
			m_bListening = (result == osl_Socket_Ok);

			nEvent = EVENT_LISTEN;
			if (m_bListening)
				nEvent |= EVENT_WRITE;
			else
				nEvent |= EVENT_OOB;
		}
	}

	if (nEvent & EVENT_LISTEN)
	{
		// Listen...
		enableEvent (EVENT_READ);
	}
	if (nEvent & EVENT_READ)
	{
		// Accept...
		nEvent = EVENT_ACCEPT;
	}
	return INetTCPSocket::handleEvent (nEvent);
}

