/*************************************************************************
 *
 *  $RCSfile: ldapmsg.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mhu $ $Date: 2001/03/19 12:12:56 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _LDAPMSG_CXX "$Revision: 1.2 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_STRING_HXX_
#include <rtl/string.hxx>
#endif

#ifndef _LDAPBER_LDAPASN_H
#include <ldapber/ldapasn.h>
#endif
#ifndef _LDAPBER_LDAPBER_H
#include <ldapber/ldapber.h>
#endif

#ifndef _LDAPMSG_HXX
#include "ldapmsg.hxx"
#endif

using rtl::OUString;

/*=======================================================================
 *
 * LDAP Message Objects.
 *
 * References:
 *   RFC 1777 - Lightweight Directory Access Protocol (Draft Standard).
 *   RFC 1823 - The LDAP Application Program Interface (Informational).
 *   RFC 1920 - Internet Official Protocol Standards (STD 1).
 *
 *=====================================================================*/
/*
 * TextEncoding.
 */
#define LDAP_TEXTENCODING RTL_TEXTENCODING_UTF8

/*
 * LDAP_STRING_newFromOUString.
 */
static void LDAP_STRING_newFromOUString (
	LDAP_STRING *&ldapString, const OUString &uniString)
{
	LDAP_STRING_free (ldapString);
	ldapString = LDAP_STRING_new();

	rtl_uString *w = uniString.pData;
	rtl::OString value (w->buffer, w->length, LDAP_TEXTENCODING);

	ldapString->length = value.pData->length;
	ldapString->data = (sal_uInt8*)(rtl_allocateMemory (ldapString->length));

	rtl_copyMemory (
		ldapString->data, value.pData->buffer, ldapString->length);
}

/*=======================================================================
 *
 * INetCoreLDAPResult.
 *
 *=====================================================================*/
INetCoreLDAPResult::INetCoreLDAPResult (void)
{
	m_pImp = LDAP_RESULT_new();
}

INetCoreLDAPResult::INetCoreLDAPResult (const LDAP_RESULT *pImp)
{
	m_pImp = LDAP_RESULT_dup ((LDAP_RESULT*)pImp);
}

INetCoreLDAPResult::~INetCoreLDAPResult (void)
{
	LDAP_RESULT_free (m_pImp);
}

sal_Int32 INetCoreLDAPResult::GetResultCode (void) const
{
	sal_Int32 nCode = INETCORELDAP_RESULT_OTHER;
	if (m_pImp)
		ASN1_ENUMERATED_get (m_pImp->resultCode, &nCode);
	return nCode;
}

sal_Bool INetCoreLDAPResult::GetMatchedDN (OUString &rMatchedDN) const
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_DN *p = m_pImp->matchedDN;
	rMatchedDN = OUString (
		(sal_Char*)(p->data), p->length, LDAP_TEXTENCODING);

	return sal_True;
}

sal_Bool INetCoreLDAPResult::GetErrorMessage (OUString &rErrorMessage) const
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_STRING *m = m_pImp->errorMessage;
	rErrorMessage = OUString (
		(sal_Char*)(m->data), m->length, LDAP_TEXTENCODING);

	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPAttributeValuePair.
 *
 *=====================================================================*/
INetCoreLDAPAttributeValuePair::INetCoreLDAPAttributeValuePair (void)
{
	m_pImp = LDAP_ATTRIBUTE_VALUE_PAIR_new();
}

INetCoreLDAPAttributeValuePair::INetCoreLDAPAttributeValuePair (
	const LDAP_ATTRIBUTE_VALUE_PAIR *pImp)
{
	m_pImp = LDAP_ATTRIBUTE_VALUE_PAIR_dup(
		(LDAP_ATTRIBUTE_VALUE_PAIR *)pImp);
}

INetCoreLDAPAttributeValuePair::~INetCoreLDAPAttributeValuePair (void)
{
	LDAP_ATTRIBUTE_VALUE_PAIR_free (m_pImp);
}

sal_Bool INetCoreLDAPAttributeValuePair::GetType (OUString &rType) const
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTE_TYPE *t = m_pImp->attributeType;
	rType = OUString (
		(sal_Char*)(t->data), t->length, LDAP_TEXTENCODING);

	return sal_True;
}

sal_Bool INetCoreLDAPAttributeValuePair::GetValue (OUString &rValue) const
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTE_VALUE *v = m_pImp->attributeValue;
	rValue = OUString (
		(sal_Char*)(v->data), v->length, LDAP_TEXTENCODING);

	return sal_True;
}

sal_Bool INetCoreLDAPAttributeValuePair::SetType (const OUString &rNewType)
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTE_TYPE *&rpType = m_pImp->attributeType;
	LDAP_STRING_newFromOUString (rpType, rNewType);

	return sal_True;
}

sal_Bool INetCoreLDAPAttributeValuePair::SetValue (const OUString &rNewValue)
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTE_VALUE *&rpValue = m_pImp->attributeValue;
	LDAP_STRING_newFromOUString (rpValue, rNewValue);

	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPAttribute.
 *
 *=====================================================================*/
INetCoreLDAPAttribute::INetCoreLDAPAttribute (void)
{
	m_pImp = LDAP_ATTRIBUTES_new();
}

INetCoreLDAPAttribute::INetCoreLDAPAttribute (
	const LDAP_ATTRIBUTES *pImp)
{
	m_pImp = LDAP_ATTRIBUTES_dup ((LDAP_ATTRIBUTES*)pImp);
}

INetCoreLDAPAttribute::~INetCoreLDAPAttribute (void)
{
	LDAP_ATTRIBUTES_free (m_pImp);
}

sal_Bool INetCoreLDAPAttribute::GetType (OUString &rType) const
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTE_TYPE *t = m_pImp->type;
	rType = OUString (
		(sal_Char*)(t->data), t->length, LDAP_TEXTENCODING);

	return sal_True;
}

sal_uInt16 INetCoreLDAPAttribute::GetValueCount (void) const
{
	return (m_pImp ? LDAP_ATTRIBUTES_count (m_pImp) : 0);
}

sal_Bool INetCoreLDAPAttribute::GetValue (
	OUString &rValue, sal_uInt16 nIndex) const
{
	if (m_pImp == NULL)
		return sal_False;

	if (nIndex >= LDAP_ATTRIBUTES_count (m_pImp))
		return sal_False;

	LDAP_ATTRIBUTE_VALUE *value = LDAP_ATTRIBUTES_value (m_pImp, nIndex);
	if (value == NULL)
		return sal_False;

	rValue = OUString (
		(sal_Char*)(value->data), value->length, LDAP_TEXTENCODING);

	return sal_True;
}

sal_Bool INetCoreLDAPAttribute::SetType (const OUString &rNewType)
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTE_TYPE *&rpType = m_pImp->type;
	LDAP_STRING_newFromOUString (rpType, rNewType);

	return sal_True;
}

sal_Bool INetCoreLDAPAttribute::SetValue (
	const OUString &rNewValue, sal_uInt16 nIndex)
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTE_VALUE *value = NULL;
	LDAP_STRING_newFromOUString (value, rNewValue);

	if (nIndex < LDAP_ATTRIBUTES_count (m_pImp))
	{
		LDAP_ATTRIBUTE_VALUE *old =
			(LDAP_ATTRIBUTE_VALUE*)ASN1_SET_remove(m_pImp->values, nIndex);
		LDAP_ATTRIBUTE_VALUE_free (old);
	}

	return (ASN1_SET_insert(m_pImp->values, (char*)value, nIndex) > 0);
}

/*=======================================================================
 *
 * INetCoreLDAPEntry.
 *
 *=====================================================================*/
INetCoreLDAPEntry::INetCoreLDAPEntry (void)
{
	m_pImp = LDAP_ENTRY_new();
}

INetCoreLDAPEntry::INetCoreLDAPEntry (const LDAP_ENTRY *pImp)
{
	m_pImp = LDAP_ENTRY_dup ((LDAP_ENTRY*)pImp);
}

INetCoreLDAPEntry::~INetCoreLDAPEntry (void)
{
	LDAP_ENTRY_free (m_pImp);
}

sal_Bool INetCoreLDAPEntry::GetObjectName (OUString &rObjectDN) const
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_DN *p = m_pImp->objectName;
	rObjectDN = OUString (
		(sal_Char*)(p->data), p->length, LDAP_TEXTENCODING);

	return sal_True;
}

sal_uInt16 INetCoreLDAPEntry::GetAttributeCount (void) const
{
	return (m_pImp ? LDAP_ENTRY_count (m_pImp) : 0);
}

INetCoreLDAPAttribute*
INetCoreLDAPEntry::GetAttribute (sal_uInt16 nIndex) const
{
	if (m_pImp == NULL)
		return NULL;
	if (nIndex >= LDAP_ENTRY_count (m_pImp))
		return NULL;

	LDAP_ATTRIBUTES *attrib = LDAP_ENTRY_value (m_pImp, nIndex);
	return (attrib ? new INetCoreLDAPAttribute (attrib) : NULL);
}

sal_Bool INetCoreLDAPEntry::SetObjectName (const OUString &rNewObjectDN)
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_DN *&rpName = m_pImp->objectName;
	LDAP_STRING_newFromOUString (rpName, rNewObjectDN);

	return sal_True;
}

sal_Bool INetCoreLDAPEntry::SetAttribute (
	const INetCoreLDAPAttribute& rAttrib, sal_uInt16 nIndex)
{
	if (m_pImp == NULL)
		return sal_False;

	LDAP_ATTRIBUTES *attrib = LDAP_ATTRIBUTES_dup(
		(LDAP_ATTRIBUTES*)(rAttrib.GetpImp()));

	if (nIndex < LDAP_ENTRY_count (m_pImp))
	{
		LDAP_ATTRIBUTES *old =
			(LDAP_ATTRIBUTES*)ASN1_SEQUENCE_remove(m_pImp->attributes, nIndex);
		LDAP_ATTRIBUTES_free (old);
	}

	return (ASN1_SEQUENCE_insert(
		m_pImp->attributes, (char *)attrib, nIndex) > 0);
}

/*=======================================================================
 *
 * INetCoreLDAPFilter.
 *
 *=====================================================================*/
INetCoreLDAPFilter::INetCoreLDAPFilter (void)
{
	m_pImp = LDAP_FILTER_new();
}

INetCoreLDAPFilter::INetCoreLDAPFilter (sal_Int32 nFilterType)
{
	m_pImp = LDAP_FILTER_type_new (nFilterType);
}

INetCoreLDAPFilter::~INetCoreLDAPFilter (void)
{
	LDAP_FILTER_free (m_pImp);
}

/*=======================================================================
 *
 * INetCoreLDAPAndFilter.
 *
 *=====================================================================*/
INetCoreLDAPAndFilter::INetCoreLDAPAndFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_AND)
{
}

INetCoreLDAPAndFilter::~INetCoreLDAPAndFilter (void)
{
}

sal_Bool INetCoreLDAPAndFilter::SetFilter (
	const INetCoreLDAPFilter& rFilter, sal_uInt16 nIndex)
{
	if (GetpImp() == NULL) return sal_False;

	LDAP_FILTER *filter = LDAP_FILTER_dup((LDAP_FILTER*)(rFilter.GetpImp()));

	ASN1_SET *andFilter = (ASN1_SET *)(GetpImp()->m_filter.m_and);
	if (nIndex < ASN1_SET_count (andFilter))
	{
		LDAP_FILTER *old = (LDAP_FILTER *)ASN1_SET_remove (andFilter, nIndex);
		LDAP_FILTER_free (old);
	}

	return (ASN1_SET_insert(andFilter, (char *)filter, nIndex) > 0);
}

/*=======================================================================
 *
 * INetCoreLDAPOrFilter.
 *
 *=====================================================================*/
INetCoreLDAPOrFilter::INetCoreLDAPOrFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_OR)
{
}

INetCoreLDAPOrFilter::~INetCoreLDAPOrFilter (void)
{
}

sal_Bool INetCoreLDAPOrFilter::SetFilter (
	const INetCoreLDAPFilter& rFilter, sal_uInt16 nIndex)
{
	if (GetpImp() == NULL) return sal_False;

	LDAP_FILTER *filter = LDAP_FILTER_dup(
		(LDAP_FILTER *)(rFilter.GetpImp ()));

	ASN1_SET *orFilter = (ASN1_SET *)(GetpImp()->m_filter.m_or);
	if (nIndex < ASN1_SET_count (orFilter))
	{
		LDAP_FILTER *old = (LDAP_FILTER *)ASN1_SET_remove (orFilter, nIndex);
		LDAP_FILTER_free (old);
	}

	return (ASN1_SET_insert(orFilter, (char *)filter, nIndex) > 0);
}

/*=======================================================================
 *
 * INetCoreLDAPNotFilter.
 *
 *=====================================================================*/
INetCoreLDAPNotFilter::INetCoreLDAPNotFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_NOT)
{
}

INetCoreLDAPNotFilter::~INetCoreLDAPNotFilter (void)
{
}

sal_Bool INetCoreLDAPNotFilter::SetFilter (const INetCoreLDAPFilter& rFilter)
{
	if (GetpImp() == NULL) return sal_False;

	LDAP_FILTER *filter = (LDAP_FILTER *)GetpImp ();
	LDAP_FILTER_free (filter->m_filter.m_not);

	filter->m_filter.m_not =
		LDAP_FILTER_dup ((LDAP_FILTER *)(rFilter.GetpImp()));
	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPEqualityMatchFilter.
 *
 *=====================================================================*/
INetCoreLDAPEqualityMatchFilter::INetCoreLDAPEqualityMatchFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_EQUALITY_MATCH)
{
}

INetCoreLDAPEqualityMatchFilter::~INetCoreLDAPEqualityMatchFilter (void)
{
}

sal_Bool INetCoreLDAPEqualityMatchFilter::SetAssertion (
	const INetCoreLDAPAttributeValuePair& rAssertion)
{
	if (GetpImp() == NULL) return sal_False;

	LDAP_FILTER *filter = (LDAP_FILTER *)GetpImp ();
	LDAP_ATTRIBUTE_VALUE_PAIR_free (filter->m_filter.m_equalityMatch);

	filter->m_filter.m_equalityMatch = LDAP_ATTRIBUTE_VALUE_PAIR_dup(
		(LDAP_ATTRIBUTE_VALUE_PAIR *)(rAssertion.GetpImp()));
	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPApproxMatchFilter.
 *
 *=====================================================================*/
INetCoreLDAPApproxMatchFilter::INetCoreLDAPApproxMatchFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_APPROX_MATCH)
{
}

INetCoreLDAPApproxMatchFilter::~INetCoreLDAPApproxMatchFilter (void)
{
}

sal_Bool INetCoreLDAPApproxMatchFilter::SetAssertion (
	const INetCoreLDAPAttributeValuePair& rAssertion)
{
	if (GetpImp() == NULL) return sal_False;

	LDAP_FILTER *filter = (LDAP_FILTER *)GetpImp ();
	LDAP_ATTRIBUTE_VALUE_PAIR_free (filter->m_filter.m_approxMatch);

	filter->m_filter.m_approxMatch = LDAP_ATTRIBUTE_VALUE_PAIR_dup(
		(LDAP_ATTRIBUTE_VALUE_PAIR *)(rAssertion.GetpImp()));
	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPGreaterOrEqualFilter.
 *
 *=====================================================================*/
INetCoreLDAPGreaterOrEqualFilter::INetCoreLDAPGreaterOrEqualFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_GREATER_OR_EQUAL)
{
}

INetCoreLDAPGreaterOrEqualFilter::~INetCoreLDAPGreaterOrEqualFilter (void)
{
}

sal_Bool INetCoreLDAPGreaterOrEqualFilter::SetAssertion (
	const INetCoreLDAPAttributeValuePair& rAssertion)
{
	if (GetpImp() == NULL) return sal_False;

	LDAP_FILTER *filter = (LDAP_FILTER *)GetpImp ();
	LDAP_ATTRIBUTE_VALUE_PAIR_free (filter->m_filter.m_greaterOrEqual);

	filter->m_filter.m_greaterOrEqual = LDAP_ATTRIBUTE_VALUE_PAIR_dup(
		(LDAP_ATTRIBUTE_VALUE_PAIR *)(rAssertion.GetpImp()));
	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPLessOrEqualFilter.
 *
 *=====================================================================*/
INetCoreLDAPLessOrEqualFilter::INetCoreLDAPLessOrEqualFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_LESS_OR_EQUAL)
{
}

INetCoreLDAPLessOrEqualFilter::~INetCoreLDAPLessOrEqualFilter (void)
{
}

sal_Bool INetCoreLDAPLessOrEqualFilter::SetAssertion (
	const INetCoreLDAPAttributeValuePair& rAssertion)
{
	if (GetpImp() == NULL) return sal_False;

	LDAP_FILTER *filter = (LDAP_FILTER *)GetpImp ();
	LDAP_ATTRIBUTE_VALUE_PAIR_free (filter->m_filter.m_lessOrEqual);

	filter->m_filter.m_lessOrEqual = LDAP_ATTRIBUTE_VALUE_PAIR_dup(
		(LDAP_ATTRIBUTE_VALUE_PAIR *)(rAssertion.GetpImp()));
	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPPresentFilter.
 *
 *=====================================================================*/
INetCoreLDAPPresentFilter::INetCoreLDAPPresentFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_PRESENT)
{
}

INetCoreLDAPPresentFilter::~INetCoreLDAPPresentFilter (void)
{
}

sal_Bool INetCoreLDAPPresentFilter::SetAttributeType (
	const OUString &rAttribType)
{
	if (GetpImp() == NULL)
		return sal_False;
	if (rAttribType.getLength() == 0)
		return sal_False;

	LDAP_FILTER *filter = (LDAP_FILTER *)GetpImp();
	LDAP_ATTRIBUTE_TYPE *&rpType = filter->m_filter.m_present;
	LDAP_STRING_newFromOUString (rpType, rAttribType);

	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPSubstringFilter.
 *
 *=====================================================================*/
INetCoreLDAPSubstringFilter::INetCoreLDAPSubstringFilter (void)
	: INetCoreLDAPFilter (LDAP_FILTER_TYPE_SUBSTRINGS)
{
}

INetCoreLDAPSubstringFilter::~INetCoreLDAPSubstringFilter (void)
{
}

sal_Bool INetCoreLDAPSubstringFilter::SetAttributeType (
	const OUString &rAttribType)
{
	if (GetpImp() == NULL)
		return sal_False;
	if (rAttribType.getLength() == 0)
		return sal_False;

	LDAP_SUBSTRING_FILTER *subFilter =
		(LDAP_SUBSTRING_FILTER *)(GetpImp()->m_filter.m_substrings);
	LDAP_ATTRIBUTE_TYPE *&rpType = subFilter->type;
	LDAP_STRING_newFromOUString (rpType, rAttribType);

	return sal_True;
}

sal_Bool INetCoreLDAPSubstringFilter::SetSubstring (
	const OUString &rSubstring,
	INetCoreLDAPSubstringPosition ePos, sal_uInt16 nIndex)
{
	if (GetpImp() == NULL)
		return sal_False;
	if (rSubstring.getLength() == 0)
		return sal_False;

	LDAP_SUBSTRING_FILTER *subFilter =
		(LDAP_SUBSTRING_FILTER *)(GetpImp()->m_filter.m_substrings);
	if (nIndex < ASN1_SEQUENCE_count (subFilter->substrings))
	{
		LDAP_SUBSTRING *oldSubStr = (LDAP_SUBSTRING *)(
			ASN1_SEQUENCE_remove (subFilter->substrings, nIndex));
		LDAP_SUBSTRING_free (oldSubStr);
	}

	LDAP_STRING *p = NULL;
	LDAP_STRING_newFromOUString (p, rSubstring);

	LDAP_SUBSTRING *newSubStr = LDAP_SUBSTRING_type_new (ePos);
	newSubStr->substring.ptr  = (sal_uInt8*)p;

	return (ASN1_SEQUENCE_insert(
		subFilter->substrings, (char *)newSubStr, nIndex) > 0);
}

/*=======================================================================
 *
 * INetCoreLDAPModification.
 *
 *=====================================================================*/
INetCoreLDAPModification::INetCoreLDAPModification (void)
{
	m_pImp = LDAP_MODIFICATION_new();
}

INetCoreLDAPModification::~INetCoreLDAPModification (void)
{
	LDAP_MODIFICATION_free (m_pImp);
}

sal_Bool INetCoreLDAPModification::SetOperation (
	INetCoreLDAPModificationOpcode eOpcode)
{
	if (m_pImp == NULL) return sal_False;

	return (ASN1_ENUMERATED_set (m_pImp->operation, eOpcode) > 0);
}

sal_Bool INetCoreLDAPModification::SetAttribute (
	const INetCoreLDAPAttribute& rAttrib)
{
	if (m_pImp == NULL) return sal_False;

	LDAP_ATTRIBUTES_free (m_pImp->modification);
	m_pImp->modification = LDAP_ATTRIBUTES_dup(
		(LDAP_ATTRIBUTES *)(rAttrib.GetpImp()));

	return (m_pImp->modification != NULL);
}

/*=======================================================================
 *
 * INetCoreLDAPMessage.
 *
 *=====================================================================*/
INetCoreLDAPMessage::INetCoreLDAPMessage (sal_uInt8 **ppData, sal_uInt32 nSize)
{
	m_pImp = d2i_LDAP_MESSAGE (NULL, ppData, nSize);
}

INetCoreLDAPMessage::INetCoreLDAPMessage (sal_Int32 nMessageType)
{
	m_pImp = LDAP_MESSAGE_type_new (nMessageType);
}

INetCoreLDAPMessage::INetCoreLDAPMessage (const INetCoreLDAPMessage &rMessage)
{
	m_pImp = LDAP_MESSAGE_dup ((LDAP_MESSAGE*)(rMessage.m_pImp));
}

INetCoreLDAPMessage::~INetCoreLDAPMessage (void)
{
	LDAP_MESSAGE_free (m_pImp);
}

sal_uInt16 INetCoreLDAPMessage::GetMessageID (void) const
{
	if (m_pImp == NULL) return 0;

	sal_Int32 msgid;
	ASN1_INTEGER_get (m_pImp->messageID, &msgid);

	return ((sal_uInt16)(msgid & 0xffff));
}

sal_Bool INetCoreLDAPMessage::SetMessageID (sal_uInt16 nMessageID)
{
	if (m_pImp)
		return (ASN1_INTEGER_set (m_pImp->messageID, nMessageID) > 0);
	else
		return sal_False;
}

sal_uInt32 INetCoreLDAPMessage::GetEncodedLength (void) const
{
	return ((sal_uInt32)i2d_LDAP_MESSAGE (m_pImp, NULL));
}

sal_uInt32 INetCoreLDAPMessage::GetEncodedData (
	sal_uInt8 *pData, sal_uInt32 nSize) const
{
	if ((sal_uInt32)i2d_LDAP_MESSAGE (m_pImp, NULL) <= nSize)
		return ((sal_uInt32)i2d_LDAP_MESSAGE (m_pImp, &pData));
	else
		return 0;
}

sal_Bool INetCoreLDAPMessage::IsResult (void) const
{
	if (m_pImp == NULL) return sal_False;

	sal_Bool bIsResult = sal_False;
	switch (m_pImp->messageType)
	{
		case LDAP_MESSAGE_TYPE_BIND_RESPONSE:
		case LDAP_MESSAGE_TYPE_SEARCH_RESULT:
		case LDAP_MESSAGE_TYPE_MODIFY_RESPONSE:
		case LDAP_MESSAGE_TYPE_ADD_RESPONSE:
		case LDAP_MESSAGE_TYPE_DELETE_RESPONSE:
		case LDAP_MESSAGE_TYPE_MODIFYRDN_RESPONSE:
		case LDAP_MESSAGE_TYPE_COMPARE_RESPONSE:
			bIsResult = sal_True;
			break;

		default:
			bIsResult = sal_False;
			break;
	}
	return bIsResult;
}

sal_Bool INetCoreLDAPMessage::IsEntry (void) const
{
	if (m_pImp)
		return (m_pImp->messageType == LDAP_MESSAGE_TYPE_SEARCH_ENTRY);
	else
		return sal_False;
}

INetCoreLDAPResult* INetCoreLDAPMessage::GetResult (void) const
{
	if (m_pImp == NULL) return NULL;

	LDAP_RESULT *result = NULL;
	switch (m_pImp->messageType)
	{
		case LDAP_MESSAGE_TYPE_BIND_RESPONSE:
			result = m_pImp->protocolOp.bindResponse;
			break;
		case LDAP_MESSAGE_TYPE_SEARCH_RESULT:
			result = m_pImp->protocolOp.searchResult;
			break;

		case LDAP_MESSAGE_TYPE_MODIFY_RESPONSE:
			result = m_pImp->protocolOp.modifyResponse;
			break;

		case LDAP_MESSAGE_TYPE_ADD_RESPONSE:
			result = m_pImp->protocolOp.addResponse;
			break;

		case LDAP_MESSAGE_TYPE_DELETE_RESPONSE:
			result = m_pImp->protocolOp.deleteResponse;
			break;

		case LDAP_MESSAGE_TYPE_MODIFYRDN_RESPONSE:
			result = m_pImp->protocolOp.modifyRDNResponse;
			break;

		case LDAP_MESSAGE_TYPE_COMPARE_RESPONSE:
			result = m_pImp->protocolOp.compareResponse;
			break;

		default:
			result = NULL;
			break;
	}
	return (result ? new INetCoreLDAPResult (result) : NULL);
}

INetCoreLDAPEntry* INetCoreLDAPMessage::GetEntry (void) const
{
	if (m_pImp == NULL) return NULL;

	LDAP_ENTRY *entry = NULL;
	if (m_pImp->messageType == LDAP_MESSAGE_TYPE_SEARCH_ENTRY)
		entry = m_pImp->protocolOp.searchEntry;

	return (entry ? new INetCoreLDAPEntry (entry) : NULL);
}

/*=======================================================================
 *
 * INetCoreLDAPBindRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPBindRequestMessage::INetCoreLDAPBindRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_BIND_REQUEST)
{
}

INetCoreLDAPBindRequestMessage::~INetCoreLDAPBindRequestMessage (void)
{
}

sal_Bool INetCoreLDAPBindRequestMessage::SetLDAPVersion (sal_uInt16 nVersion)
{
	if (!IsValid())
		return sal_False;

	LDAP_BIND_REQUEST *pReq = GetpImp()->protocolOp.bindRequest;
	return (ASN1_INTEGER_set (pReq->version, nVersion) > 0);
}

sal_Bool INetCoreLDAPBindRequestMessage::SetObjectName (
	const OUString &rObjectDN)
{
	if (!IsValid())
		return sal_False;

	LDAP_BIND_REQUEST *pReq = GetpImp()->protocolOp.bindRequest;
	LDAP_DN *&rpName = pReq->name;
	LDAP_STRING_newFromOUString (rpName, rObjectDN);

	return sal_True;
}

sal_Bool INetCoreLDAPBindRequestMessage::SetObjectPassword (
	const OUString &rPassword)
{
	if (!IsValid())
		return sal_False;

	LDAP_BIND_REQUEST *pReq = GetpImp()->protocolOp.bindRequest;
	LDAP_DN *&rpName = pReq->authentication.simple;
	LDAP_STRING_newFromOUString (rpName, rPassword);

	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPUnbindRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPUnbindRequestMessage::INetCoreLDAPUnbindRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_UNBIND_REQUEST)
{
}

INetCoreLDAPUnbindRequestMessage::~INetCoreLDAPUnbindRequestMessage (void)
{
}

/*=======================================================================
 *
 * INetCoreLDAPSearchRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPSearchRequestMessage::INetCoreLDAPSearchRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_SEARCH_REQUEST)
{
}

INetCoreLDAPSearchRequestMessage::INetCoreLDAPSearchRequestMessage (
	const INetCoreLDAPSearchRequestMessage &rMessage)
	: INetCoreLDAPMessage (rMessage)
{
}

INetCoreLDAPSearchRequestMessage::~INetCoreLDAPSearchRequestMessage (void)
{
}

sal_Bool INetCoreLDAPSearchRequestMessage::SetBaseObject (
	const OUString &rBaseObject)
{
	if (!IsValid())
		return sal_False;
	if (rBaseObject.getLength() == 0)
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	LDAP_DN *&rpName = pReq->baseObject;
	LDAP_STRING_newFromOUString (rpName, rBaseObject);

	return sal_True;
}

sal_Bool INetCoreLDAPSearchRequestMessage::SetScope (
	INetCoreLDAPSearchScope eScope)
{
	if (!IsValid())
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	return (ASN1_ENUMERATED_set (pReq->scope, eScope) > 0);
}

sal_Bool INetCoreLDAPSearchRequestMessage::SetDerefAlias (
	INetCoreLDAPSearchDerefAlias eDerefAlias)
{
	if (!IsValid())
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	return (ASN1_ENUMERATED_set (pReq->derefAlias, eDerefAlias) > 0);
}

sal_Bool
INetCoreLDAPSearchRequestMessage::SetSizeLimit (sal_uInt16 nSizeLimit)
{
	if (!IsValid())
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	return (ASN1_INTEGER_set (pReq->sizeLimit, nSizeLimit) > 0);
}

sal_Bool
INetCoreLDAPSearchRequestMessage::SetTimeLimit (sal_uInt16 nTimeLimit)
{
	if (!IsValid())
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	return (ASN1_INTEGER_set (pReq->timeLimit, nTimeLimit) > 0);
}

sal_Bool
INetCoreLDAPSearchRequestMessage::SetAttributesOnly (sal_Bool bAttrsOnly)
{
	if (!IsValid())
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	pReq->attrsOnly->value = bAttrsOnly;
	return sal_True;
}

sal_Bool INetCoreLDAPSearchRequestMessage::SetFilter (
	const INetCoreLDAPFilter& rFilter)
{
	if (!IsValid())
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	LDAP_FILTER_free (pReq->filter);
	pReq->filter = LDAP_FILTER_dup ((LDAP_FILTER *)(rFilter.GetpImp()));
	return sal_True;
}

sal_Bool INetCoreLDAPSearchRequestMessage::SetAttributeType (
	const OUString &rAttribType, sal_uInt16 nIndex)
{
	if (!IsValid())
		return sal_False;

	LDAP_SEARCH_REQUEST *pReq = GetpImp()->protocolOp.searchRequest;
	if (nIndex < ASN1_SEQUENCE_count (pReq->attributes))
	{
		LDAP_ATTRIBUTE_TYPE *old = (LDAP_ATTRIBUTE_TYPE *)(
			ASN1_SEQUENCE_remove (pReq->attributes, nIndex));
		LDAP_ATTRIBUTE_TYPE_free (old);
	}

	LDAP_ATTRIBUTE_TYPE *attr = NULL;
	LDAP_STRING_newFromOUString (attr, rAttribType);

	return (ASN1_SEQUENCE_insert(
		pReq->attributes, (char *)attr, nIndex) > 0);
}

/*=======================================================================
 *
 * INetCoreLDAPModifyRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPModifyRequestMessage::INetCoreLDAPModifyRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_MODIFY_REQUEST)
{
}

INetCoreLDAPModifyRequestMessage::INetCoreLDAPModifyRequestMessage (
	const INetCoreLDAPModifyRequestMessage &rMessage)
	: INetCoreLDAPMessage (rMessage)
{
}

INetCoreLDAPModifyRequestMessage::~INetCoreLDAPModifyRequestMessage (void)
{
}

sal_Bool INetCoreLDAPModifyRequestMessage::SetObjectName (
	const OUString &rObjectDN)
{
	if (!IsValid())
		return sal_False;
	if (rObjectDN.getLength() == 0)
		return sal_False;

	LDAP_MODIFY_REQUEST *pReq = GetpImp()->protocolOp.modifyRequest;
	LDAP_DN *&rpName = pReq->object;
	LDAP_STRING_newFromOUString (rpName, rObjectDN);

	return sal_True;
}

sal_Bool INetCoreLDAPModifyRequestMessage::SetModification (
	const INetCoreLDAPModification& rMod, sal_uInt16 nIndex)
{
	if (!IsValid())
		return sal_False;

	LDAP_MODIFICATION *mod = LDAP_MODIFICATION_dup(
		(LDAP_MODIFICATION *)(rMod.GetpImp()));

	LDAP_MODIFY_REQUEST *pReq = GetpImp()->protocolOp.modifyRequest;
	if (nIndex < ASN1_SEQUENCE_count (pReq->modifications))
	{
		LDAP_MODIFICATION *old = (LDAP_MODIFICATION *)(
			ASN1_SEQUENCE_remove (pReq->modifications, nIndex));
		LDAP_MODIFICATION_free (old);
	}

	return (ASN1_SEQUENCE_insert(
		pReq->modifications, (char *)mod, nIndex) > 0);
}

/*=======================================================================
 *
 * INetCoreLDAPAddRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPAddRequestMessage::INetCoreLDAPAddRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_ADD_REQUEST)
{
}

INetCoreLDAPAddRequestMessage::~INetCoreLDAPAddRequestMessage (void)
{
}

sal_Bool INetCoreLDAPAddRequestMessage::SetEntry (
	const INetCoreLDAPEntry& rEntry)
{
	if (!IsValid())
		return sal_False;

	LDAP_MESSAGE *pMsg = (LDAP_MESSAGE *)GetpImp();

	LDAP_ADD_REQUEST *pReq = pMsg->protocolOp.addRequest;
	LDAP_ADD_REQUEST_free (pReq);

	pReq = LDAP_ENTRY_dup ((LDAP_ENTRY *)(rEntry.GetpImp()));
	pMsg->protocolOp.addRequest = pReq;

	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPDeleteRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPDeleteRequestMessage::INetCoreLDAPDeleteRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_DELETE_REQUEST)
{
}

INetCoreLDAPDeleteRequestMessage::~INetCoreLDAPDeleteRequestMessage (void)
{
}

sal_Bool INetCoreLDAPDeleteRequestMessage::SetObjectName (
	const OUString &rObjectDN)
{
	if (!IsValid())
		return sal_False;
	if (rObjectDN.getLength() == 0)
		return sal_False;

	LDAP_MESSAGE *pMsg = (LDAP_MESSAGE*)GetpImp();
	LDAP_DN *&rpName = ((LDAP_DN*)(pMsg->protocolOp.deleteRequest));
	LDAP_STRING_newFromOUString (rpName, rObjectDN);

	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPModifyRDNRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPModifyRDNRequestMessage::INetCoreLDAPModifyRDNRequestMessage()
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_MODIFYRDN_REQUEST)
{
}

INetCoreLDAPModifyRDNRequestMessage::~INetCoreLDAPModifyRDNRequestMessage()
{
}

sal_Bool INetCoreLDAPModifyRDNRequestMessage::SetObjectName (
	const OUString &rObjectDN)
{
	if (!IsValid())
		return sal_False;
	if (rObjectDN.getLength() == 0)
		return sal_False;

	LDAP_MODIFYRDN_REQUEST *pReq = GetpImp()->protocolOp.modifyRDNRequest;
	LDAP_DN *&rpName = pReq->entry;
	LDAP_STRING_newFromOUString (rpName, rObjectDN);

	return sal_True;
}

sal_Bool INetCoreLDAPModifyRDNRequestMessage::SetNewRDN (
	const OUString &rNewRDN)
{
	if (!IsValid())
		return sal_False;
	if (rNewRDN.getLength() == 0)
		return sal_False;

	LDAP_MODIFYRDN_REQUEST *pReq = GetpImp()->protocolOp.modifyRDNRequest;
	LDAP_DN *&rpName = pReq->newrdn;
	LDAP_STRING_newFromOUString (rpName, rNewRDN);

	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPCompareRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPCompareRequestMessage::INetCoreLDAPCompareRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_COMPARE_REQUEST)
{
}

INetCoreLDAPCompareRequestMessage::~INetCoreLDAPCompareRequestMessage (void)
{
}

sal_Bool INetCoreLDAPCompareRequestMessage::SetObjectName (
	const OUString &rObjectDN)
{
	if (!IsValid())
		return sal_False;
	if (rObjectDN.getLength() == 0)
		return sal_False;

	LDAP_COMPARE_REQUEST *pReq = GetpImp()->protocolOp.compareRequest;
	LDAP_DN *&rpName = pReq->entry;
	LDAP_STRING_newFromOUString (rpName, rObjectDN);

	return sal_True;
}

sal_Bool INetCoreLDAPCompareRequestMessage::SetAssertion (
	const INetCoreLDAPAttributeValuePair& rAssertion)
{
	if (!IsValid())
		return sal_False;

	LDAP_COMPARE_REQUEST *pReq = GetpImp()->protocolOp.compareRequest;
	LDAP_ATTRIBUTE_VALUE_PAIR_free (pReq->ava);

	pReq->ava = LDAP_ATTRIBUTE_VALUE_PAIR_dup(
		(LDAP_ATTRIBUTE_VALUE_PAIR *)(rAssertion.GetpImp()));
	return sal_True;
}

/*=======================================================================
 *
 * INetCoreLDAPAbandonRequestMessage.
 *
 *=====================================================================*/
INetCoreLDAPAbandonRequestMessage::INetCoreLDAPAbandonRequestMessage (void)
	: INetCoreLDAPMessage (LDAP_MESSAGE_TYPE_ABANDON_REQUEST)
{
}

INetCoreLDAPAbandonRequestMessage::~INetCoreLDAPAbandonRequestMessage (void)
{
}

sal_Bool
INetCoreLDAPAbandonRequestMessage::SetAbandonMessageID (sal_uInt16 nMessageID)
{
	if (!IsValid())
		return sal_False;

	LDAP_ABANDON_REQUEST *pReq = GetpImp()->protocolOp.abandonRequest;
	return (ASN1_INTEGER_set (pReq, nMessageID) > 0);
}

