/*************************************************************************
 *
 *  $RCSfile: vfsrootcontent.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:16:54 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <vfsrootcontent.hxx>
#include <vfsfilecontent.hxx>
#include <vfshelper.hxx>

#include <tools/urlobj.hxx>
#include <tools/debug.hxx>

// =======================================================================
// VFSRootContent
// =======================================================================

VFSRootContent::VFSRootContent( const OUString& rIdentifier )
 : VFSFolderContent( rIdentifier, STORE_ATTRIB_ISDIR, this )
{
	OUString aVFSScheme( L"vfs://" );
	OUString aTmp = rIdentifier.copy( aVFSScheme.getLength() );
	String aURL = OUStringToOString( aTmp, ENCODINGHACK ).getStr();
	INetURLObject aURLObj( aURL );
	String maStorageName = aURLObj.PathToFileName();
	sal_Bool bNewStorage = sal_False;
	storeError eError = maStorage.create( maStorageName, store_AccessReadWrite );
	if ( store_E_NotExists == eError )
	{
		eError = maStorage.create( maStorageName, store_AccessReadCreate );
		bNewStorage = sal_True;
	}
	if ( bNewStorage && ( store_E_None == eError ) )
	{
		OStoreDirectory aContentDir;
		aContentDir.create( maStorage, "/", "Cnt", store_AccessReadCreate );
		aContentDir.close();

		OStoreDirectory aAttrDir;
		aAttrDir.create( maStorage, "/", "Attr", store_AccessReadCreate );
		aAttrDir.close();

		InitAttributes();
		StoreExtendedAttributes();
		StoreSimpleAttributes();
	}
	maOpenContents.Insert( this, LIST_APPEND );
}

VFSRootContent::~VFSRootContent()
{
	ClearRoot();
	maStorage.close();
}

VFSContent* VFSRootContent::FindContent( const OUString& rIdentifier )
{
	Guard< Mutex > aGuard( GetMutex() );

	VFSContent* pContent = NULL;
	for ( USHORT n = maOpenContents.Count(); n && !pContent; ) 
	{
		VFSContent* pC = (VFSContent*)maOpenContents.GetObject( --n );
		OUString aID = pC->getIdentifier()->getContentIdentifier();
		if ( aID == rIdentifier )
			pContent = pC;
	}
	return pContent;
}

void VFSRootContent::FindContents( const OUString& rIdentifierStart, list<VFSContent*>& rFound )
{
	Guard< Mutex > aGuard( GetMutex() );

	for ( USHORT n = maOpenContents.Count(); n; ) 
	{
		VFSContent* pC = (VFSContent*)maOpenContents.GetObject( --n );
		OUString aID = pC->getIdentifier()->getContentIdentifier();
		if ( aID.indexOf( rIdentifierStart ) == 0 )
			rFound.push_back( pC );
	}
}

void VFSRootContent::RemoveContent( VFSContent* pContent )
{
	Guard< Mutex > aGuard( GetMutex() );

	maOpenContents.Remove( pContent );
	if ( !maOpenContents.Count() )
		maStorage.flush();
}

sal_Bool VFSRootContent::Exists( const String& rPath, const String& rName )
{
	Guard< Mutex > aGuard( GetMutex() );

 	storeError nError = GetStorage().attrib( rPath.GetStr(), rName.GetStr(), 0, 0 );
	return ( store_E_NotExists != nError );
}

sal_Bool VFSRootContent::RenameChilds( const String& rPath, const String& rFolderName, sal_uInt16 nOldStart, sal_uInt16 nOldLen, const String& rNewFolderName )
{
 	DBG_ASSERT( !Exists( rPath, rFolderName ), "RenameChilds: Folder not found!" );

	sal_Bool bError = sal_False;

	OStoreDirectory aDir;

	String aContentPath( CONTENTDIR );
	aContentPath += rPath;
	if ( !bError && aDir.create( GetRoot()->GetStorage(), aContentPath, rFolderName, store_AccessReadOnly ) == store_E_None )
	{
		String aFullPath( rPath );
		aFullPath += rFolderName;
		aFullPath += '/';

		String aFullContentPath( CONTENTDIR );
		aFullContentPath += rPath;
		aFullContentPath += rFolderName;
		aFullContentPath += '/';

		String aNewContentPath( aFullContentPath );
		sal_uInt16 nEraseCntPos = nOldStart + String( CONTENTDIR ).Len();
		aNewContentPath.Erase( nEraseCntPos, nOldLen );
		aNewContentPath.Insert( rNewFolderName, nEraseCntPos );

		String aFullAttribPath( ATTRIBDIR );
		aFullAttribPath += rPath;
		aFullAttribPath += rFolderName;
		aFullAttribPath += '/';

		String aNewAttribPath( aFullAttribPath );
		sal_uInt16 nEraseAttrPos = nOldStart + String( ATTRIBDIR ).Len();
		aNewAttribPath.Erase( nEraseAttrPos, nOldLen );
		aNewAttribPath.Insert( rNewFolderName, nEraseAttrPos );

		storeFindData aIterator;
		storeError eError = aDir.first( aIterator);
		while ( eError == store_E_None )
		{
			String aName( aIterator.m_pszName );
			if ( aIterator.m_nAttrib & STORE_ATTRIB_ISDIR )
			{
				if ( !RenameChilds( aFullPath, aName, nOldStart, nOldLen, rNewFolderName ) )
					bError = sal_True;
				aName += '/';	// Needed for rename
			}

			GetStorage().rename( aFullContentPath, aName, aNewContentPath, aName );
			RemoveTrailingSlash( aName );
			GetStorage().rename( aFullAttribPath, aName, aNewAttribPath, aName );

			eError = aDir.next( aIterator);
		}
		aDir.close();
	}

	return !bError;
}

sal_Bool VFSRootContent::Rename( const String& rPath, const String& rOldName, const String& rNewName )
{
	Guard< Mutex > aGuard( GetMutex() );

	sal_Bool bDone = sal_False;

	String aContentPath( CONTENTDIR );
	aContentPath += rPath;

	if ( Exists( aContentPath, rOldName ) && !Exists( aContentPath, rNewName ) )
	{
		sal_Bool bCanRename = sal_True;

		OUString aContentIdentifer = CreateContentIdentifier( OStringToOUString( rPath.GetStr(), RTL_TEXTENCODING_UTF8 ), 
																OStringToOUString( rOldName.GetStr(), RTL_TEXTENCODING_UTF8 ) );
		list<VFSContent*> aExistingContents;
		FindContents( aContentIdentifer, aExistingContents );

		for( list<VFSContent*>::iterator it = aExistingContents.begin(); bCanRename && ( it != aExistingContents.end() ); it++ )
		{
			if ( (*it)->IsOpen() )
				bCanRename = sal_False;
		}

		if ( bCanRename )
		{
			OStoreDirectory aTestDir;
			String aDirName( rOldName );
			RemoveTrailingSlash( aDirName );
 			storeError eError = aTestDir.create( GetStorage(), aContentPath.GetStr(), aDirName.GetStr(), store_AccessReadWrite );
			sal_Bool bIsDir = ( store_E_None == eError );
			aTestDir.close();
			if ( bIsDir )
			{
				DBG_ERROR( "Rename folders not possible yet" );
				return sal_False;

				// Childs...
				String aNewDirName( rNewName );
				RemoveTrailingSlash( aNewDirName );
				RenameChilds( rPath, aDirName, rPath.Len(), aDirName.Len(), aNewDirName );
			}

			String aNewAttrOrContentName( rNewName );
			RemoveTrailingSlash( aNewAttrOrContentName );	// Allways without '/', rename from Folders not implemented yet
			eError = GetStorage().rename( aContentPath, rOldName, aContentPath, aNewAttrOrContentName );
			bDone = ( store_E_None == eError );
			String aAttribPath( ATTRIBDIR );
			aAttribPath += rPath;
			String aOldAttrName( rOldName );
			RemoveTrailingSlash( aOldAttrName );
			eError = GetStorage().rename( aAttribPath, aOldAttrName, aAttribPath, aNewAttrOrContentName );

			// Exchange ContentIdentifiers...
			OUString aRoot = CreateContentIdentifier( L"", L"" );
			OUString aReplace = OStringToOUString( rNewName.GetStr(), RTL_TEXTENCODING_UTF8 );
			VFSContent* pRenamer = FindContent( aContentIdentifer );
			for( list<VFSContent*>::iterator it = aExistingContents.begin(); it != aExistingContents.end(); it++ )
			{
				VFSContent* pContent = *it;
				if ( pContent != pRenamer )
				{
					OUString aID = pContent->getIdentifier()->getContentIdentifier();
					aID = aID.replaceAt( aRoot.getLength()+rPath.Len()-1, rOldName.Len(), aReplace );
					pContent->ChangeContentIdentifier( aID );
				}
			}
		}
	}
	return bDone;
}

sal_Bool VFSRootContent::DeleteFile( const String& rFullName, sal_Bool bMarkOnly )
{
	sal_Bool bDone = sal_False;
	String aPath, aName;
	VFSContent::CreatePathAndName( rFullName, aPath, aName, CONTENTDIR );
	DBG_ASSERT( Exists( aPath, aName ), "DeleteFile: File does not exist!" );

	if ( bMarkOnly )
	{
		// !!!!!!!!!
	}
	else
	{
 		storeError eError = GetStorage().remove( aPath, aName );
		if ( store_E_None == eError )
		{
			VFSContent::CreatePathAndName( rFullName, aPath, aName, ATTRIBDIR );
			GetStorage().remove( aPath, aName );

			VFSContent* pContent = FindContent( OStringToOUString( rFullName.GetStr(), RTL_TEXTENCODING_UTF8 ) );
			if ( pContent )
				pContent->SetValid( sal_False );

			bDone = sal_True;
		}
	}
	return bDone;
}

sal_Bool VFSRootContent::DeleteFolder( const String& rFullName, sal_Bool bMarkOnly )
{
	sal_Bool bDone = sal_False;

	String aPath, aName;
	VFSContent::CreatePathAndName( rFullName, aPath, aName, CONTENTDIR );
	DBG_ASSERT( Exists( aPath, aName ), "DeleteFile: File does not exist!" );

	sal_Bool bCanDelete = sal_True;

	OUString aContentIdentifer = OStringToOUString( rFullName.GetStr(), RTL_TEXTENCODING_UTF8 );
	list<VFSContent*> aExistingContents;
	FindContents( aContentIdentifer, aExistingContents );

	for( list<VFSContent*>::iterator it = aExistingContents.begin(); bCanDelete && ( it != aExistingContents.end() ); it++ )
	{
		if ( (*it)->IsOpen() )
			bCanDelete = sal_False;
	}

	if ( bCanDelete )
	{
		if ( bMarkOnly )
		{
			// !!!!!!!!!
		}
		else
		{
			OStoreDirectory aDir;
			String aDirName( aName );
			RemoveTrailingSlash( aDirName );
			if ( aDir.create( GetRoot()->GetStorage(), aPath, aDirName, store_AccessReadOnly ) == store_E_None )
			{
				storeFindData aIterator;
				storeError eError = aDir.first( aIterator);
				while ( eError == store_E_None )
				{
					String aNewFullName( rFullName );
					aNewFullName += aIterator.m_pszName;
					if ( aIterator.m_nAttrib & STORE_ATTRIB_ISDIR )
					{
						aNewFullName += '/';
						DeleteFolder( aNewFullName, bMarkOnly );
					}
					else
					{
						DeleteFile( aNewFullName, bMarkOnly );
					}

					eError = aDir.next( aIterator);
				}
				aDir.close();
			}

 			storeError eError = GetStorage().remove( aPath, aName );
			if ( store_E_None == eError )
			{
				VFSContent::CreatePathAndName( rFullName, aPath, aName, ATTRIBDIR );
				RemoveTrailingSlash( aName );
				GetStorage().remove( aPath, aName );

				VFSContent* pContent = FindContent( OStringToOUString( rFullName.GetStr(), RTL_TEXTENCODING_UTF8 ) );
				if ( pContent )
					pContent->SetValid( sal_False );

				bDone = sal_True;
			}
		}
	}
	return bDone;
}


sal_Bool VFSRootContent::QueryContent( const OUString& rIdentifier, Reference<XContent>& rxNewContent ) throw(RuntimeException)
{
	Guard< Mutex > aGuard( GetMutex() );

	sal_Bool bDone = sal_True;
	rxNewContent = FindContent( rIdentifier );
	if ( !rxNewContent.is() ) 
	{
		bDone = sal_False;
		VFSContent* pNewContent = NULL;

		String aPath, aName;
		VFSContent::CreatePathAndName( rIdentifier, aPath, aName, CONTENTDIR );

		sal_uInt32 nAttrs;
 		storeError eError = GetStorage().attrib( aPath, aName, 0, 0, nAttrs );
		if ( store_E_None == eError )
		{
			if ( aName.Len() && ( aName.GetChar( aName.Len()-1 ) != '/' ) )
			{
				pNewContent = new VFSFileContent( rIdentifier, nAttrs, this );
				maOpenContents.Insert( pNewContent, LIST_APPEND );
			}
			else
			{
				pNewContent = new VFSFolderContent( rIdentifier, nAttrs, this );
				maOpenContents.Insert( pNewContent, LIST_APPEND );
			}
			bDone = TRUE;
			rxNewContent = pNewContent;
		}

	}

	return bDone;
}

sal_Bool VFSRootContent::InsertNewContent( VFSContent* pVFSContent ) throw(RuntimeException)
{
	Guard< Mutex > aGuard( GetMutex() );

	sal_Bool bDone = sal_False;

	OUString aNewContentName = pVFSContent->GetFileName();
	if ( !aNewContentName.getLength() )
		aNewContentName = pVFSContent->GetTitle();

	DBG_ASSERT( aNewContentName.getLength(), "InsertNewContent: FileName ?!" );

	if ( aNewContentName.getLength() )
	{
		sal_Bool bIsFolder = pVFSContent->getContentType() == VFSFolderContent::GetContentType();

		if ( bIsFolder )
			aNewContentName += L"/";

		OUString aID = pVFSContent->getIdentifier()->getContentIdentifier();
		aID += aNewContentName;

		String aPath, aName;
		VFSContent::CreatePathAndName( aID, aPath, aName, CONTENTDIR );
		if ( !Exists( aPath, aName ) )
		{
			// Create File/Folder
			if( bIsFolder )
			{
				OStoreDirectory aFolder;
				RemoveTrailingSlash( aName );
				storeError eError = aFolder.create( GetStorage(), aPath, aName, store_AccessCreate );
				if ( store_E_None == eError )
				{
					aFolder.close();
					GetStorage().flush();
					bDone = sal_True;
				}
			}
			else
			{
				OStoreStream aFile;
				storeError eError = aFile.create( GetStorage(), aPath, aName, store_AccessCreate );
				if ( store_E_None == eError )
				{
					aFile.close();
					GetStorage().flush();
					bDone = sal_True;
				}
			}

			if ( bDone )
			{
				VFSContent::CreatePathAndName( aID, aPath, aName, NULL );	// Path ohne /Cnt/
				OUString aContentIdentifier = CreateContentIdentifier( OStringToOUString( aPath.GetStr(), RTL_TEXTENCODING_UTF8 ), 
																		OStringToOUString( aName.GetStr(), RTL_TEXTENCODING_UTF8 ) );
				pVFSContent->ChangeContentIdentifier( aContentIdentifier );
				pVFSContent->InitAttributes();
				pVFSContent->SetValid( sal_True );
				pVFSContent->StoreSimpleAttributes();
				pVFSContent->StoreExtendedAttributes();

				maOpenContents.Insert( pVFSContent, LIST_APPEND );
			}
		}
	}
	return bDone;
}


