/*************************************************************************
 *
 *  $RCSfile: synchron.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:16:53 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "synccont.hxx"
#include "synccoll.hxx"
#include "synchron.hxx"

#include <tools/tempfile.hxx>

// ----------------
// - Synchronizer -
// ----------------

Synchronizer::Synchronizer( const REF( XMultiServiceFactory )& rxFact ) :
	mxFact			( rxFact ),
	mpSyncBase		( NULL ),
	mbInSynchronize	( sal_False )
{
}

// -----------------------------------------------------------------------------

Synchronizer::~Synchronizer()
{
}

// -----------------------------------------------------------------------------

Any SAL_CALL Synchronizer::queryInterface( const Type & rType ) throw (RuntimeException)
{
	const Any aRet( ::cppu::queryInterface( rType,
											static_cast< XSynchronizer* >( this ) ) );

	return( aRet.hasValue() ? aRet : OWeakObject::queryInterface( rType ) );
}

// -----------------------------------------------------------------------------

void SAL_CALL Synchronizer::acquire() throw( RuntimeException )
{
	OWeakObject::acquire();
}

// -----------------------------------------------------------------------------

void SAL_CALL Synchronizer::release() throw( RuntimeException )
{
	OWeakObject::release();
}

// -----------------------------------------------------------------------------

sal_Bool SAL_CALL Synchronizer::beginSynchronize( const REF( XSyncCollector )& rxClientCollector, 
												  const REF( XSyncCollector )& rxServerCollector,
												  const OUString& rSchemeName, 
												  const FilterData& rFilterData, 
												  sal_uInt32 nSyncMode, 
												  sal_uInt32 nSyncOptions )
{
	DBG_ASSERT( !mbInSynchronize, "already in synchronize" );

	sal_Bool bRet = sal_False;

	if( !mbInSynchronize && rxClientCollector.is() && rxServerCollector.is() )
	{
		mxClientCollector = rxClientCollector;
		mxServerCollector = rxServerCollector;

		SEQ( SyncScheme )	aSyncSchemesServer( mxServerCollector->getRegisteredSyncSchemes() );
		SEQ( SyncScheme )	aSyncSchemesClient( mxClientCollector->getRegisteredSyncSchemes() );
		sal_uInt32			i;
		sal_Bool			bFound;

		// find correct SyncScheme for server
		for( i = 0, bFound = sal_False; ( i < aSyncSchemesServer.getLength() ) && !bFound; i++ )
		{
			if( aSyncSchemesServer[ i ].Scheme == rSchemeName )
			{
				maSyncInfo.ServerScheme = aSyncSchemesServer[ i ];
				bFound = sal_True;
			}
		}

		if( bFound )
		{
			// find correct SyncScheme for client
			for( i = 0, bFound = sal_False; ( i < aSyncSchemesClient.getLength() ) && !bFound; i++ )
			{
				if( aSyncSchemesClient[ i ].Scheme == rSchemeName )
				{
					maSyncInfo.ClientScheme = aSyncSchemesClient[ i ];
					bFound = sal_True;
				}
			}

			if( bFound )
			{
				maSyncInfo.SyncFilter = rFilterData;
				maSyncInfo.SyncMode = nSyncMode;
				maSyncInfo.SyncOptions = nSyncOptions;

				// begin synchronize
				mxServerCollector->beginSynchronize();
				mxClientCollector->beginSynchronize();

				bRet = mbInSynchronize = sal_True;
			}
			else
				maSyncInfo = SyncInfo();
		}
	}

	return bRet;
}

// -----------------------------------------------------------------------------

sal_Bool SAL_CALL Synchronizer::endSynchronize()
{
	DBG_ASSERT( mbInSynchronize, "not in synchronize" );

	if( mbInSynchronize )
	{
		// end synchronize
		mxServerCollector->endSynchronize();
		mxClientCollector->endSynchronize();

		mbInSynchronize = sal_False;
		mxClientCollector = REF( XSyncCollector )();
		mxServerCollector = REF( XSyncCollector )();
		maSyncInfo = SyncInfo();
	}

	return !mbInSynchronize;
}

// -----------------------------------------------------------------------------

sal_Bool SAL_CALL Synchronizer::getSyncList( SEQ( SyncElement )& rSyncSeq )
{
	DBG_ASSERT( mbInSynchronize, "not in synchronize" );

	sal_Bool bRet = sal_False;

	rSyncSeq.realloc( 0 );

	if( mbInSynchronize )
	{
		// init temporary bases
		ImplInitTempBase();

		// prepare bases for synchronization
		if( maSyncInfo.SyncMode & SyncMode::SERVER_TO_CLIENT )
			ImplPrepareSynchronizeServer( mxServerCollector->getSyncSequence( maSyncInfo.ServerScheme.Scheme, maSyncInfo ) );

		if( maSyncInfo.SyncMode & SyncMode::CLIENT_TO_SERVER )
			ImplPrepareSynchronizeClient( mxClientCollector->getSyncSequence( maSyncInfo.ClientScheme.Scheme, maSyncInfo ) );

		// build sync sequence from database
		bRet = ImplCreateSyncSequence( rSyncSeq );

		// release temporary bases
		ImplReleaseTempBase();
	}

	return bRet;
}

// -----------------------------------------------------------------------------

sal_Bool SAL_CALL Synchronizer::performSync( const SEQ( SyncElement )& rSyncSeq )
{
	DBG_ASSERT( mbInSynchronize, "not in synchronize" );

	sal_Bool bRet = sal_False;

	if( mbInSynchronize )
	{
		const OUString& rSchemeName = maSyncInfo.ServerScheme.Scheme;

		for( sal_uInt32 i = 0, nCount = rSyncSeq.getLength(); i < nCount; i++ )
		{
			SyncBaseElement		aActElem( rSyncSeq[ i ] );
			const sal_uInt32	nAction = aActElem.GetAction();
			sal_Bool			bServerNotify = sal_False, bClientNotify = sal_False;

			if( !( SyncAction::CONFLICT_UID & nAction ) && !( SyncAction::CONFLICT_AMBIGUITY & nAction ) )
			{
				if( SyncAction::COPY_TO_SERVER & nAction )
				{
					ImplCopyElement( rSchemeName, aActElem, mxClientCollector );
					bServerNotify = bClientNotify = sal_True;
				}
				else if( SyncAction::COPY_TO_CLIENT & nAction )
				{
					ImplCopyElement( rSchemeName, aActElem, mxServerCollector );
					bServerNotify = bClientNotify = sal_True;
				}
				else
				{
					if( SyncAction::REMOVE_FROM_SERVER & nAction )
					{
						ImplRemoveElement( rSchemeName, aActElem, mxServerCollector );
						bServerNotify = sal_True;
					}

					if( SyncAction::REMOVE_FROM_CLIENT & nAction )
					{
						ImplRemoveElement( rSchemeName, aActElem, mxClientCollector );
						bClientNotify = sal_True;
					}
				}

				aActElem.SetAction( SyncAction::NONE );

				if( bServerNotify )
					mxServerCollector->elementSynchronized( rSchemeName, aActElem, nAction );

				if( bClientNotify )
					mxClientCollector->elementSynchronized( rSchemeName, aActElem, nAction );
			}
		}

		bRet = sal_True;
	}

	return bRet;
}

// -----------------------------------------------------------------------------

void Synchronizer::ImplInitTempBase()
{
	mpSyncBase = new SyncBase( TempFile::CreateTempName(), ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "SYNC" ) ) );
}

// -----------------------------------------------------------------------------

void Synchronizer::ImplReleaseTempBase()
{
	if( mpSyncBase )
	{
		const String aBaseFile( mpSyncBase->GetBaseFile() );

		delete mpSyncBase, mpSyncBase = NULL; 

		INetURLObject aURL; aURL.SetSmartProtocol( INET_PROT_FILE ); aURL.SetSmartURL( aBaseFile );

	    try
		{
			::ucb::Content aCnt( aURL.GetMainURL(), Reference< XCommandEnvironment > () );
			aCnt.executeCommand( rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "delete" ) ), makeAny( sal_Bool( sal_True ) ) );
		}
		catch( ... )
		{
			DBG_ERROR( "file not deleted" );
		}
    }
}

// -----------------------------------------------------------------------------

void Synchronizer::ImplPrepareSynchronizeServer( const SEQ( SyncElement )& rServerSyncSeq )
{
	SyncBaseElement	aElemClient, aElemServer;

	// prepare synchronize for server collector
	for( sal_uInt32 i = 0, nCount = rServerSyncSeq.getLength(); i < nCount; i++ )
	{
		mxClientCollector->getCorrespondingElement( maSyncInfo.ServerScheme, aElemServer = rServerSyncSeq[ i ], aElemClient );
		ImplPrepareSynchronize( aElemClient, aElemServer );
	}
}

// -----------------------------------------------------------------------------

void Synchronizer::ImplPrepareSynchronizeClient( const SEQ( SyncElement )& rClientSyncSeq )
{
	SyncBaseElement	aElemClient, aElemServer;

	// prepare synchronize for client collector
	for( sal_uInt32 i = 0, nCount = rClientSyncSeq.getLength(); i < nCount; i++ )
	{
		mxServerCollector->getCorrespondingElement( maSyncInfo.ClientScheme, aElemClient = rClientSyncSeq[ i ], aElemServer );
		ImplPrepareSynchronize( aElemClient, aElemServer );
	}
}

// -----------------------------------------------------------------------------

void Synchronizer::ImplPrepareSynchronize( SyncBaseElement& rElemClient, SyncBaseElement& rElemServer )
{
	SyncBaseElement aElem;
	sal_uInt32		nAction = SyncAction::NONE;
	const sal_Bool	bServerWins = ( maSyncInfo.SyncOptions & SyncOptions::SERVER_WINS ) == SyncOptions::SERVER_WINS;
	const sal_Bool	bClientWins = ( maSyncInfo.SyncOptions & SyncOptions::CLIENT_WINS ) == SyncOptions::CLIENT_WINS;
	const sal_Bool	bAmbiguous = ( bServerWins == bClientWins );

	if( !rElemClient.IsEmpty() && !rElemServer.IsEmpty() )
	{
		if( bServerWins )
		{
			if( rElemServer.GetEvent() == SyncEvent::DELETED )
				nAction = ( SyncAction::REMOVE_FROM_CLIENT | SyncAction::REMOVE_FROM_SERVER );
			else if( rElemServer.GetEvent() != SyncEvent::NONE )
				nAction = SyncAction::COPY_TO_CLIENT;
		}
		else if( bClientWins )
		{
			if( rElemClient.GetEvent() == SyncEvent::DELETED )
				nAction = ( SyncAction::REMOVE_FROM_CLIENT | SyncAction::REMOVE_FROM_SERVER );
			else if( rElemClient.GetEvent() != SyncEvent::NONE )
				nAction = SyncAction::COPY_TO_SERVER;
		}
		else
		{
			if( rElemClient.GetUID() != rElemServer.GetUID() )
				nAction = SyncAction::CONFLICT_UID;
			else
			{
				// !!! To do
				if( rElemClient.GetEvent() == SyncEvent::DELETED )
					nAction = ( SyncAction::REMOVE_FROM_CLIENT | SyncAction::REMOVE_FROM_SERVER );
				else if( rElemServer.GetEvent() == SyncEvent::DELETED )
					nAction = ( SyncAction::REMOVE_FROM_CLIENT | SyncAction::REMOVE_FROM_SERVER );
				else
					nAction = SyncAction::CONFLICT_AMBIGUITY;
			}
		}
	}
	else if( !rElemClient.IsEmpty() )
	{
		if( rElemClient.GetEvent() != SyncEvent::DELETED )
			nAction = SyncAction::COPY_TO_SERVER;
		else
			nAction = SyncAction::REMOVE_FROM_CLIENT;
	}
	else if( !rElemServer.IsEmpty() )
	{
		if( rElemServer.GetEvent() != SyncEvent::DELETED )
			nAction = SyncAction::COPY_TO_CLIENT;
		else
			nAction = SyncAction::REMOVE_FROM_SERVER;
	}

	// is copying allowed
	if( maSyncInfo.SyncOptions & SyncOptions::DONT_COPY )
	{
		if( SyncAction::COPY_TO_CLIENT & nAction )
			nAction &= ~SyncAction::COPY_TO_CLIENT;

		if( SyncAction::COPY_TO_SERVER & nAction )
			nAction &= ~SyncAction::COPY_TO_SERVER;
	}
	
	// is removing allowed
	if( maSyncInfo.SyncOptions & SyncOptions::DONT_REMOVE )
	{
		if( SyncAction::REMOVE_FROM_CLIENT & nAction )
			nAction &= ~SyncAction::REMOVE_FROM_CLIENT;

		if( SyncAction::REMOVE_FROM_SERVER & nAction )
			nAction &= ~SyncAction::REMOVE_FROM_SERVER;
	}

	// put sync element to base
	if( SyncAction::NONE != nAction )
	{
		if( SyncAction::COPY_TO_SERVER & nAction )
		{
			aElem = rElemClient;
			aElem.SetAction( SyncAction::COPY_TO_SERVER );
		}
		if( SyncAction::COPY_TO_CLIENT & nAction )
		{
			aElem = rElemServer;
			aElem.SetAction( SyncAction::COPY_TO_CLIENT );
		}
		else if( ( SyncAction::REMOVE_FROM_SERVER & nAction ) || ( SyncAction::REMOVE_FROM_CLIENT & nAction ) )
		{
			aElem = rElemClient;
			aElem.SetAction( nAction & ( SyncAction::REMOVE_FROM_SERVER | SyncAction::REMOVE_FROM_CLIENT ) );
		}

		// conflict found?
		if( ( SyncAction::CONFLICT_UID & nAction ) || ( SyncAction::CONFLICT_AMBIGUITY & nAction ) )
		{
			aElem = rElemClient;
			aElem.SetAction( aElem.GetAction() | ( nAction & ( SyncAction::CONFLICT_UID | SyncAction::CONFLICT_AMBIGUITY ) ) );
		}

		if( !aElem.IsEmpty() )
			mpSyncBase->PutElement( aElem );
	}
}

// -----------------------------------------------------------------------------

sal_Bool Synchronizer::ImplCreateSyncSequence( SEQ( SyncElement )& rSyncSeq )
{
	SyncCursor*	pCursor = mpSyncBase->GetCursor();
	sal_Bool	bRet = sal_False;

	if( pCursor )
	{
		static const sal_uInt32	nResize = 1024;
		sal_uInt32				i = 0;
		
		for( const SyncBaseElement* pElem = pCursor->First(); pElem; pElem = pCursor->Next() )
		{
			if( i == rSyncSeq.getLength() )
				rSyncSeq.realloc( rSyncSeq.getLength() + nResize );

			rSyncSeq[ i++ ] = *pElem;
		}

		rSyncSeq.realloc( i );

		bRet = sal_True;
	}

	return bRet;
}

// -----------------------------------------------------------------------------

void Synchronizer::ImplCopyElement( const OUString& rSchemeName, const SyncBaseElement& rElem, 
									REF( XSyncCollector )& rSourceCollector )
{
	REF( XSyncCollector )&	rCollSrc = ( &rSourceCollector == &mxClientCollector ) ? mxClientCollector : mxServerCollector;
	REF( XSyncCollector )&	rCollDst = ( &rSourceCollector == &mxClientCollector ) ? mxServerCollector : mxClientCollector;
	REF( XInterface )		xPipe( mxFact->createInstance( ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "com.sun.star.io.Pipe" ) ) ) );

	if( xPipe.is() )
	{
		OpenCommandArgument aArg;

		aArg.Mode = com::sun::star::chaos::OpenMode::ALL;
		aArg.Priority = 0;
		aArg.Sink = REF( XOutputStream )( xPipe, UNO_QUERY );

		Any aStm; aStm <<= aArg;
		rCollSrc->executeElementCommand( rSchemeName, rElem, ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "open" ) ), aStm );

		aStm <<= REF( XInputStream )( xPipe, UNO_QUERY );
		rCollDst->executeElementCommand( rSchemeName, rElem, ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "insert" ) ), aStm );
	}
}

// -----------------------------------------------------------------------------

void Synchronizer::ImplRemoveElement( const OUString& rSchemeName, const SyncBaseElement& rElem,
									  REF( XSyncCollector )& rSourceCollector )
{
	Any	aAny;
	rSourceCollector->executeElementCommand( rSchemeName, rElem, ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "delete" ) ), aAny );
}
